<?php
// ****************************************************************************
// ----------------------------------------------------------------------------
//			Copyright 2014, Nokia Networks
//			All Rights Reserved
// ----------------------------------------------------------------------------
//
// Title:	LogoutHandler.php
//
// Description:	This file handles an asynchronous logout request from the client.
//		A logout request message is sent to the WebUIAgent C++ process,
//		which disconnects from SMA (if connected) and clears the session.
//
// ****************************************************************************

include_once "../../phplib/MessageHandler.php";
include_once "../../phplib/XmlMessageParser.php";
include_once "../../phplib/messages/LogoutRequest.php";
include_once "../../phplib/messages/LogoutResponse.php";

openlog( "WebUI CGI: Logout", LOG_ODELAY, LOG_USER );
header("Content-type: text/plain; charset=utf-8");
header("Cache-control: no-store");
header("Pragma: no-cache");

// Logout request. Retrieve session cookie from client. Send logout request
// to WebUIAgent.

if( !isset( $_COOKIE[ "FZAPWebUISession" ] ) )
{
	// Unexpected. User should not have been able to access WebUI page with
	// cookies disabled. Log error and send failure indication.
	syslog( LOG_ERR, "No session cookie" );

	// User not logged in; nothing to do. Return success indication
	// (any positive integer) and exit.
	echo 1;

	closelog();
	exit();
}


// Session cookie received from client. Retrieve user session ID.
$sessionId = $_COOKIE[ "FZAPWebUISession" ];


$logoutRequest = new LogoutRequest( $sessionId );
if( is_null( $logoutRequest ) )
{
	syslog( LOG_ERR, "Failed to allocate memory for LogoutRequest message" );

	// The user is logging out anyway. Treat as successful logout for
	// now... Delete session cookie in client.
	setcookie( "FZAPWebUISession", "", time() - 86400, "/", "", true, true );
	echo 2;

	closelog();
	exit();
}

// Send message to the WebUIAgent. The MessageHandler will wait for, and return,
// the response, if any.
$response = MessageHandler::SendAndReceive( $logoutRequest->XmlMsg() );
if( is_null( $response ) )
{
	syslog( LOG_ERR, "Failed to send LogoutRequest message and/or receive response" );

	// Failed to send and/or receive a response. Treat as successful logout...
	// Delete cookie and exit.
	setcookie( "FZAPWebUISession", "", time() - 86400, "/", "", true, true );
	echo 3;

	closelog();
	exit();
}


// Parse the received XML message response.
$msg = XmlMessageParser::ParseMsg( $response );

// Validate response.
if( (is_null( $msg )) || ($msg->OK() !== true) || ($msg->GetMessageType() !== "LogoutResponse") )
{
	// Remove line breaks from response string and print to syslog.
	$msgString = str_replace( "\n", null, $response );
	syslog( LOG_ERR, "Invalid or null LogoutResponse message: " . $msgString );

	// Oh well, proceed with logout...
	setcookie( "FZAPWebUISession", "", time() - 86400, "/", "", true, true );
	echo 4;

	closelog();
	exit();
}


// Logout successful. Delete session cookie in client by setting past due expiration time.
setcookie( "FZAPWebUISession", "", time() - 86400, "/", "", true, true );	// yesterday
	
// Return success indication (any positive integer) and exit.
echo 10;
closelog();
exit();

?>
