// ****************************************************************************
// ----------------------------------------------------------------------------
//			Copyright 2015, Nokia Networks
//			All Rights Reserved
// ----------------------------------------------------------------------------
//
// Title:		BlockSite.js
//
// ****************************************************************************

( function( $ )
{

	var xmlHttp;

	NSNReady( function()
	{

		$( "#blockSiteLink" ).click( function()
		{
			if( $( this ).parent().is( ".disabled" ) )
			{
				return;
			}

			var $blockDialog = $( "#blockDialog" );
			if( $blockDialog.is( ":visible" ) )
			{
				// Dialog window already open.
				return;
			}

			// Initial settings. Progress indication is shown once block/unblock operation
			// is initiated. The close button is initially labeled Cancel to indicate the
			// user can close the dialog window without starting the procedure. The label
			// is changed to Close once the operation is complete.
			var $confirmButton = $("#blockConfirmButton");
			$( "#blockProgressIndicator" ).hide();
			$confirmButton.removeClass( "disabled" ).show();
			$confirmButton.removeClass("call-to-action");

			var $closeButton = $( "#blockCloseButton" );
			$closeButton.removeClass( "disabled" );
			$closeButton.addClass( "call-to-action" );
			$closeButton.find( ".button-content" ).text( "Cancel" );
            
                        var accessRight = $( "#menuPanel" ).find( "input[name='accessRight']" ).val();
                        var accountRole = $( "#menuPanel" ).find( "input[name='accountRole']" ).val();
                        if((accessRight !== "read-write") || (accountRole !== "sysAdmin" &&  accountRole !== "appAdmin"))
                        {
                            $confirmButton.addClass("disabled");
                            $( "#blockProgressIndicator" )
                            .removeClass()
                            .addClass( "step warning" )
                            .find( ".text" ).text( "Invalid AccessRight or AccountRole" ).end()
                            .show();
                        }

			// Open the dialog
			$blockDialog.trigger( "open" );

			// Set focus to confirm button once dialog has opened.
			setTimeout( function()
			{
				$closeButton.focus();
			}, 100 );

			$confirmButton.focus(function()
			{
				if($(this).is(".disabled"))
				{
					return;
				}
                        
				$(this).addClass("call-to-action");
				$closeButton.removeClass("call-to-action");
			});
                        
			$confirmButton.blur(function()
			{
				$(this).removeClass("call-to-action");
				$closeButton.addClass("call-to-action");
			});
        
			$confirmButton.click( function()
			{
				if( $( this ).is( ".disabled" ) )
				{
					return;
				}

				$closeButton.addClass( "disabled" );
				$( this ).addClass( "disabled" );

				updateAndShowProgressIndicator( "current" );
				sendBlockingStateChangeRequest( "blocked" );
			});

			$closeButton.click( function()
			{
				if( $( this ).is( ".disabled" ) )
				{
					return;
				}

				// Close the dialog window.
				$blockDialog.trigger( "close" );
			});
		});


		$( "#blockDialog" ).bind( "close", function()
		{
			// Stop any ongoing request.
			// Note, the dialog window itself is closed by handler in webuiDialogs.js
			if( typeof xmlHttp !== "undefined" )
			{
				xmlHttp.removeEventListener( "readystatechange", processBlockingStateChangeResponse, false );
				xmlHttp.abort();
			}
		});
	});


	var sendBlockingStateChangeRequest = function( blockingState )
	{
		// Create and send XMLHttpRequest to server.
		try
		{
			xmlHttp = new XMLHttpRequest();
			var csrfToken = $( "#rttPanel" ).find( "input[name='csrfToken']" ).val();
			var csrfTokenEncoded = encodeURIComponent( csrfToken );

			// Register event listener for response
			// FZM/FZAP block/unblock have antModId of 0
			xmlHttp.addEventListener( "readystatechange", processBlockingStateChangeResponse, false );
			var postData = "bs=" + blockingState + "&antModId=0" + "&ctkn=" + csrfTokenEncoded;
			var url = "cgi-bin/BlockUnblockHandler.php";


			xmlHttp.open( "POST", url, true );
			// Set encoding used for the request body (HTML form) and length of contents.
			xmlHttp.setRequestHeader( "Content-Type", "application/x-www-form-urlencoded" );
			xmlHttp.setRequestHeader( "Content-length", postData.length );

			xmlHttp.send(postData);
		}
		catch( exception )
		{
			console.log( "Asynchronous blocking state change request failed" );

			updateAndShowProgressIndicator( "error" );
			updateButtonsOnCompletion();
			return;
		}
	}


	var processBlockingStateChangeResponse = function()
	{
		if( (xmlHttp.readyState == 4) && (xmlHttp.status == 200) )
		{
			var blockingStateChangeResponse;

			try
			{
				// A valid response should contain a JSON object.
				blockingStateChangeResponse = JSON.parse( xmlHttp.responseText );
			}
			catch( exception )
			{
				// Parsing failure
				console.log( "JSON parsing error in processBlockingStateChangeResponse():\n" + exception );
				console.log( xmlHttp.responseText );

				updateAndShowProgressIndicator( "error" );
				updateButtonsOnCompletion();
				return;
			}


			if( !("statusInd" in blockingStateChangeResponse) ||
			     (typeof blockingStateChangeResponse.statusInd !== "string") )
			{
				console.log( "Invalid blocking state change response; missing status indicator" );
				updateAndShowProgressIndicator( "error" );
				updateButtonsOnCompletion();
				return;
			}

			if( blockingStateChangeResponse.statusInd === "Success" )
			{
				// Update progress indication.
				updateAndShowProgressIndicator( "success" );
				updateButtonsOnCompletion();
			}
			else
			{
				// Failure has occurred. Read failure reason.
				if( !("_failReason" in blockingStateChangeResponse) ||
				     (typeof blockingStateChangeResponse._failReason !== "string") )
				{
					console.log( "Invalid blocking state change response; missing failure reason" );
					updateAndShowProgressIndicator( "error" );
					updateButtonsOnCompletion();
					return;
				}

				console.log( "Block or unblock failure reason: " + blockingStateChangeResponse._failReason );

				if( (blockingStateChangeResponse._failReason === "invalidSession"))
				{
					updateAndShowProgressIndicator( "warning", "Session expired" );

					// User session has expired.
					window.alert( "Session has expired - Please login again" );

					// Open login page with session expired indication.
					location.assign( "login.php?expired" );
					return;
				}

				if(blockingStateChangeResponse._failReason === "invalidSecurityToken")
				{
					updateAndShowProgressIndicator( "warning", "Security Token expired" );

					// User session has expired.
					window.alert( "Security token has expired - Please login again" );

					// Open login page with session expired indication.
					location.assign( "login.php?expired" );
					return;
				}


				// Procedure failed.
				updateAndShowProgressIndicator( "error" );
				updateButtonsOnCompletion();
			}
		}

		else if( xmlHttp.readyState == 4 )
		{
			// HTTP level failure has occurred.
			console.log( "Failed to receive blocking state change response from server" );

			// Update progress indication.
			updateAndShowProgressIndicator( "error" );
			updateButtonsOnCompletion();
		}
	}


	var updateAndShowProgressIndicator = function( stepName, displayText )
	{
		// displayText argument is "optional". Expected to be explicitly passed
		// by calling function for info and warning step indicators.
		if( typeof displayText === "undefined" )
		{
			switch( stepName )
			{
				case "current":
					displayText = "Performing operation";
					break;

				case "error":
					displayText = "Operation failed";
					break;

				case "success":
					displayText = "Operation complete";
					break;

				default:
					// Explicit display text should have been provided.
					console.log( "Unrecognized step: " + stepName );
					return;
			}
		}

		var classList = "step " + stepName;

		$( "#blockProgressIndicator" )
			.removeClass()
			.addClass( classList )
			.find( ".text" ).text( displayText ).end()
			.show();
	}


	var updateButtonsOnCompletion = function()
	{
		$( "#blockConfirmButton" ).hide();

		var $closeButton = $( "#blockCloseButton" );
		$closeButton.removeClass( "disabled" );
		$closeButton.addClass( "call-to-action" );
		$closeButton.find( ".button-content" ).text( "Close" );
		$closeButton.focus();
	}

})( jQuery )
