// ****************************************************************************
// ----------------------------------------------------------------------------
//			Copyright 2015, Nokia Networks
//			All Rights Reserved
// ----------------------------------------------------------------------------
//
// Title:		DisplayIPSecConfiguration.js
//
// ****************************************************************************

(function( $ )
{
	var NUM_ADD_OR_DELETE_ROWS = 1;
	var MAX_ASSOCIATIONS = 288;
	var MIN_NUM_TABLE_ROWS = 2;

	var xmlHttp;
	var associationsArray = [];
	var $sortColumn;


	// IKEAssociation helper object. Note: IKE = Internet Key Exchange
	var IKEAssociation = function( remoteTunnelEndpoint, peerState )
	{
		this.remoteTunnelEndpoint = initializeStringValue( remoteTunnelEndpoint );
		this.peerState = initializeStringValue( peerState );
	};


	NSNReady( function()
	{
		// Setup associations table.
		var $table = $( "#associationsTable" );
		$table.standard_table( MIN_NUM_TABLE_ROWS, MAX_ASSOCIATIONS );

		$table.find( "thead th.sortable" ).each( function()
		{
			var $headerCell = $( this );

			$headerCell.bind( "sortNeeded", function()
			{
				$sortColumn = $headerCell;
				sortAssociationsTable();
				displayAssociationsTable();
			});
		});


		$( "#displayIpSecLink" ).click( function() {

			if( $( "#ipSecPanel" ).is( ":visible" ) )
			{
				return;
			}
            
            if( $( this ).parent().is( ".disabled" ) )
            {
                return;
            }

			// Hide any open panel(s) in the display area.
			$( ".content-display-panel" ).not( "#ipSecPanel" ).trigger( "closePanel" );

			// Show the IP sec panel in the display area.
			$( "#ipSecPanel" ).show();

			$( "#refreshIpSecButton" ).addClass( "disabled" );
			updateAndShowProgressIndicator( "current" );
			sendIpSecStatusRequest();
		});

		$( "#refreshIpSecButton" ).click( function() {

			var $refreshButton = $( this );
			if( $refreshButton.is( ".disabled" ) )
			{
				return;
			}

			$refreshButton.addClass( "disabled" );
			updateAndShowProgressIndicator( "current" );
			sendIpSecStatusRequest();
		});

		$( "#ipSecPanel" ).bind( "closePanel", function()
		{
			if( typeof xmlHttp !== "undefined" )
			{
				xmlHttp.removeEventListener( "readystatechange", processIpSecStatusResponse, false );
				xmlHttp.abort();
			}

			$( "#ipSecPanel" ).hide();
		});

		// IPSec panel is hidden until opened by user.
		$( "#ipSecPanel" ).hide();
	});


	var sendIpSecStatusRequest = function()
	{
		// Create and send XMLHttpRequest to server
		try
		{
			xmlHttp = new XMLHttpRequest();

			// Register event listener for response
			xmlHttp.addEventListener( "readystatechange", processIpSecStatusResponse, false );

			// GET information asynchronously from server
			xmlHttp.open( "GET", "cgi-bin/IPSecHandler.php", true );
			xmlHttp.send();
		}
		catch( exception )
		{
			console.log( "Asynchronous IPSec connection status request failed" );
			$( "#refreshIpSecButton" ).removeClass( "disabled" );
			updateAndShowProgressIndicator( "error" );
			return;
		}
	}


	var processIpSecStatusResponse = function()
	{
		if( (xmlHttp.readyState == 4) && (xmlHttp.status == 200) )
		{
			$( "#refreshIpSecButton" ).removeClass( "disabled" );
			$( "#ipSecProgressIndicator" ).hide();

			var ipSecConnectionStatusResponse;

			try
			{
				// A successful response should contain a JSON object.
				ipSecConnectionStatusResponse = JSON.parse( xmlHttp.responseText );
			}
			catch( exception )
			{
				// Parsing failure
				console.log( "JSON parsing error in processIpSecStatusResponse():\n" + exception );
				console.log( xmlHttp.responseText );
				updateAndShowProgressIndicator( "error" );
				return;
			}


			if( !("statusInd" in ipSecConnectionStatusResponse) ||
			     (typeof ipSecConnectionStatusResponse.statusInd !== "string") )
			{
				console.log( "Invalid IPSec connection status response; missing status indicator" );
				updateAndShowProgressIndicator( "error" );
				return;
			}

			if( ipSecConnectionStatusResponse.statusInd === "Success" )
			{
				updateIpSecStatus( ipSecConnectionStatusResponse );
			}
			else
			{
				updateAndShowProgressIndicator( "error" );

				// Failure has occurred. Read failure reason.
				if( !("_failReason" in ipSecConnectionStatusResponse) ||
				     (typeof ipSecConnectionStatusResponse._failReason !== "string") )
				{
					console.log( "Invalid IPSec connection status response; missing failure reason" );
					return;
				}

				console.log( "IPSec failure reason: " + ipSecConnectionStatusResponse._failReason );

				if( ipSecConnectionStatusResponse._failReason === "invalidSession" )
				{
					// Configuration failed because the user session has expired.
					window.alert( "Session has expired - Please login again" );

					// Open login page with session expired indication.
					location.assign( "login.php?expired" );
					return;
				}
			}
		}

		else if( xmlHttp.readyState == 4 )
		{
			// HTTP level failure has occurred.
			console.log( "Failed to receive IPSec connection status response from server" );
			$( "#refreshIpSecButton" ).removeClass( "disabled" );
			updateAndShowProgressIndicator( "error" );
		}
	}


	var updateIpSecStatus = function( ipSecConnectionStatusResponse )
	{
		if( ipSecConnectionStatusResponse == null )
		{
			return;
		}

		if( !("ipSecEnabled" in ipSecConnectionStatusResponse) ||
		     (typeof ipSecConnectionStatusResponse.ipSecEnabled !== "string") )
		{
			console.log( "Invalid IPSec connection status response; missing ipSecEnabled flag" );
			updateAndShowProgressIndicator( "error" );
			return;
		}

		displayIpSecEnabledState( ipSecConnectionStatusResponse.ipSecEnabled );


		// Clear out existing pairings array.
		while( associationsArray.length > 0 )
		{
			associationsArray.pop();
		}

		if( "IPSecPeerInfo" in ipSecConnectionStatusResponse )
		{
			if( $.isArray( ipSecConnectionStatusResponse.IPSecPeerInfo ) )
			{
				for( var i = 0; (i < ipSecConnectionStatusResponse.IPSecPeerInfo.length && i < MAX_ASSOCIATIONS); i++ )
				{
					var ikeAssociation = new IKEAssociation(
						ipSecConnectionStatusResponse.IPSecPeerInfo[i].remoteTunnelEndpoint,
						ipSecConnectionStatusResponse.IPSecPeerInfo[i].peerState );

					associationsArray.push( ikeAssociation );
				}

				sortAssociationsTable();
			}

			else
			{
				// Single, non-array type object.
				var ikeAssociation = new IKEAssociation(
					ipSecConnectionStatusResponse.IPSecPeerInfo.remoteTunnelEndpoint,
					ipSecConnectionStatusResponse.IPSecPeerInfo.peerState );

				associationsArray.push( ikeAssociation );
			}
		}

		// Display IKE associations in table; note: empty table displayed if no IPSecPeerInfo
		// instances received in IPSec connection status response.
		displayAssociationsTable();
	}


	var displayIpSecEnabledState = function( ipSecEnabled )
	{
		var uiIpSecEnabled = "Invalid";

		// Format IpSecEnabled flag for UI display
		switch( ipSecEnabled.toLowerCase() )
		{
			case "1":
			case "true":
				uiIpSecEnabled = "True";
				break;

			case "0":
			case "false":
				uiIpSecEnabled = "False";
				break;

			default:
				// UI format remains as Invalid
				break;
		}

		$( "#ipSecPanel" ).find( "#ipSecStatus" ).text( uiIpSecEnabled );
	}


	var sortAssociationsTable = function()
	{
		if( typeof $sortColumn === "undefined" )
		{
			// No sort order selected.
			return;
		}

		if( $sortColumn.hasClass( "first-child" ) )
		{
			// Remote tunnel endpoint (first) column. Use alphabetic sort.
			associationsArray.sort( function( a, b )
			{
				if( $sortColumn.hasClass( "up" ) )
				{
					if( b.remoteTunnelEndpoint < a.remoteTunnelEndpoint )
					{
						return -1;
					}
					else if( b.remoteTunnelEndpoint == a.remoteTunnelEndpoint )
					{
						return 0;
					}
					else
					{
						return 1;
					}
				}

				else
				{
					if( a.remoteTunnelEndpoint < b.remoteTunnelEndpoint )
					{
						return -1;
					}
					else if( a.remoteTunnelEndpoint == b.remoteTunnelEndpoint )
					{
						return 0;
					}
					else
					{
						return 1;
					}
				}
			});
		}

		else
		{
			// Peer state (last) column. Use alphabetic sort.
			associationsArray.sort( function( a, b )
			{
				if( $sortColumn.hasClass( "up" ) )
				{
					if( b.peerState < a.peerState )
					{
						return -1;
					}
					else if( b.peerState == a.peerState )
					{
						return 0;
					}
					else
					{
						return 1;
					}
				}

				else
				{
					if( a.peerState < b.peerState )
					{
						return -1;
					}
					else if( a.peerState == b.peerState )
					{
						return 0;
					}
					else
					{
						return 1;
					}
				}
			});
		}
	}


	var displayAssociationsTable = function()
	{

		$table = $( "#associationsTable" );

		var rowsInTable = $table.find( "tbody > tr" ).length;

		// Clear out display table
		for( var i = 1; i <= rowsInTable; i++ )
		{
			var $row = $table.find( "tbody > tr:nth-of-type(" + i + ")" );
			clear_association_row( $row );
		}

		// Check if we need to add rows.
		while( (associationsArray.length > rowsInTable) && (rowsInTable < MAX_ASSOCIATIONS) )
		{
			// Add groups of rows
			$table.add_rows_to_table( NUM_ADD_OR_DELETE_ROWS, MAX_ASSOCIATIONS );
			rowsInTable = $table.find( "tbody > tr" ).length;
		}

		for( var i = 0; (i < associationsArray.length && i < rowsInTable); i++ )
		{
			var $row = $table.find( "tbody > tr:nth-of-type(" + (i+1) + ")" );
			populate_association_row( $row,
						associationsArray[i].remoteTunnelEndpoint,
						associationsArray[i].peerState );
		}

		var numAssociations = associationsArray.length;
		if( numAssociations < MIN_NUM_TABLE_ROWS )
		{
			numAssociations = MIN_NUM_TABLE_ROWS;
		}

		if( numAssociations > MAX_ASSOCIATIONS )
		{
			numAssociations = MAX_ASSOCIATIONS;
		}
		// Check if we can delete rows
		while( rowsInTable >= (numAssociations + NUM_ADD_OR_DELETE_ROWS) )
		{
			$table.remove_rows_from_table( NUM_ADD_OR_DELETE_ROWS, MIN_NUM_TABLE_ROWS );
			rowsInTable = $table.find( "tbody > tr" ).length;
		}

		// The width of the associations table may have changed, which may impact the
		// panel width. Hide or show the menu panel accordingly.
		$( "#menuPanel" ).trigger( "contentPanelResized" );
	}


	var clear_association_row = function( $row )
	{
		$row.find( "td:not( :last-child )" ).text( "" );
		$row.find( "td:last-child" ).find( ".cell-value" ).text( "" );
	}


	var populate_association_row = function( $row, remoteTunnelEndpoint, peerState )
	{
		var uiPeerState = peerState;

		// Format peer state for UI display.
		switch( peerState.toLowerCase() )
		{
			case "enabled":
				uiPeerState = "Enabled";
				break;

			case "disabled":
				uiPeerState = "Disabled";
				break;

			default:
				// UI format remains unchanged
				break;
		}

		$row.find( "td:first-child" ).text( remoteTunnelEndpoint )
			.next().find( ".cell-value" ).text( uiPeerState );
	}


	var updateAndShowProgressIndicator = function( stepName, displayText )
	{
		// displayText argument is "optional". Expected to be explicitly passed
		// by calling function for info and warning step indicators.
		if( typeof displayText === "undefined" )
		{
			switch( stepName )
			{
				case "current":
					displayText = "";
					break;

				case "error":
					displayText = "Failed to retrieve data from server";
					break;

				case "success":
					displayText = "";
					break;

				default:
					// Explicit display text should have been provided.
					console.log( "Unrecognized step: " + stepName );
					return;
			}
		}

		var classList = "step " + stepName;

		$( "#ipSecProgressIndicator" )
			.removeClass()
			.addClass( classList )
			.find( ".text" ).text( displayText ).end()
			.show();
	}

})( jQuery )
