<?php
// ****************************************************************************
// ----------------------------------------------------------------------------
//			Copyright 2017, Nokia Networks
//			All Rights Reserved
// ----------------------------------------------------------------------------
//
// Title:	MessageHandler.php
//
// Description:	The MessageHandler class sends and receives messages to/from
//		the WebUIAgent persistent C++ process. Messages are sent and
//		received over a UDP socket.
//
// ****************************************************************************

class MessageHandler
{
	// Change to "true" to enable message logging
	const MSG_LOGGING_ENABLED = false;

	const ADDRESS = "localhost";
	const PORT = 23456;
	const DEFAULT_MSG_TIMEOUT = 20;

	// Note: The message fragment (packet) size MUST be set to a value greater than
	// or equal to the corresponding setting in C_Application/SC_OAM_FZM/OM_WebUI/WebUIAgent/Include/CGIConnector.h
	const MSG_FRAGMENT_SIZE = 4096;
	const MSG_END_MARKER = "</CGIMessage>";
	const PACKET_TIMEOUT = 2;

	private $socket;

	public function __construct()
	{
		$this->socket = $this->CreateUdpSocket();
	}

	public function __destruct()
	{
		socket_close( $this->socket );
	}

	// ********************************************************************
	// Method: SendAndReceive
	//
	// Sends passed-in message (string) to the WebUIAgent and waits for a
	// response. This version of the method handles response messages sent
	// as fragmented packages to allow for messages exceeding the UDP max
	// packet size.
	//
	// Returns string containing response if received, or null on failure.
	// ********************************************************************
	public static function SendAndReceive( $msg, $timeout = self::DEFAULT_MSG_TIMEOUT )
	{
		try
		{
			$agentMessenger = new self();
			$agentMessenger->SendMsg( $msg );

			if( $agentMessenger->ReceiveData( $timeout ) )
			{
				return $agentMessenger->ReadMsg();
			}

			throw new Exception( "Timeout waiting for WebUIAgent response." );
		}

		catch( Exception $e )
		{
			syslog( LOG_ERR, "Message send or receive failure: " . $e->getMessage() );
			return null;
		}
	}


	private function CreateUdpSocket()
	{
		$socket = socket_create( AF_INET, SOCK_DGRAM, SOL_UDP );
		if( $socket === false )
		{
			throw new Exception( "Failed to create UDP socket: " . socket_strerror(socket_last_error()) );
		}

		return $socket;
	}


	private function SendMsg( $msg )
	{
		self::LogSentMessage( $msg );

		$len = strlen( $msg );
		$retval = socket_sendto( $this->socket,
					$msg,
					$len,
					0,
					self::ADDRESS,
					self::PORT );

		if( $retval === false )
		{
			throw new Exception( "Failed to send message." );
		}
	}


	private function ReceiveData( $timeout )
	{
		// Wait for messages by select'ing on socket. Configure arrays for select.
		// Only using read array.
		$readArray   = array( $this->socket );
		$writeArray  = null;
		$exceptArray = null;

		$numSockets = socket_select( $readArray, $writeArray, $exceptArray, $timeout );

		if( $numSockets === false )
		{
			throw new Exception( "Socket select failed." );
		}

		if( $numSockets === 0 )
		{
			// Timeout without receiving any data on the socket.
			syslog( LOG_INFO, "Socket select timeout" );
			return false;
		}

		// Num sockets > 0 indicates data available to be read.
		return true;
	}


	private function ReadMsg()
	{
		$response = "";

		do
		{
			$packet = socket_read( $this->socket, self::MSG_FRAGMENT_SIZE );

			if( $packet === false )
			{
				throw new Exception( "Socket receive failure." );
			}

			self::LogPacket( $packet );

			$response .= $packet;
			$endOfMsg = $this->CheckForEndOfMsg( $response );
		}
		while( !$endOfMsg && $this->ReceiveData( self::PACKET_TIMEOUT ) );

		self::LogReceivedMessage( $response );
		return $response;
	}


	private function CheckForEndOfMsg( $msg )
	{
		// Search for end marker. Apply offset to avoid searching the whole (potentially
		// large) message buffer.
		$offset = strlen( $msg ) - strlen( self::MSG_END_MARKER );

		// "Add" some arbitrary margin to allow for newline character, null, etc, at end
		// of message (reduce offset). Note: offset argument tells strpos() where to start
		// searching the string. Must be non-negative.
		$offset -= 3;
		if( $offset < 0 )
		{
			$offset = 0;
		}

		if( strpos( $msg, self::MSG_END_MARKER, $offset ) !== false )
		{
			return true;
		}

		return false;
	}

        private static function RemoveSecurityInfo( $msg,$tag,$replacement)
        {
            $tagBegin = strpos($msg,"<".$tag.">");
            $tagEnd = strpos($msg,"</".$tag.">");
            if($tagBegin === false)
            {
                return $msg;
            }
            else
            {
                  $tagBegin = $tagBegin + strlen("<".$tag.">");   
                  $tagLength = $tagEnd - $tagBegin; 
                  if($tagLength > 0)
                  {
                      $msg = substr_replace($msg,$replacement,$tagBegin,$tagLength);
                  }
                  return $msg;
            }
        }

	private static function LogSentMessage( $msg )
	{
		if( self::MSG_LOGGING_ENABLED )
		{
			// Remove newline characters to allow syslog printing.
			$msgString = str_replace( "\n", null, $msg );
                        $msgString = self::RemoveSecurityInfo( $msgString,"_userName","######");
                        $msgString = self::RemoveSecurityInfo( $msgString,"_password","@*@*@*@");
			$len = strlen( $msg );
                        syslog( LOG_DEBUG, "Sending  message of size " . $len . " bytes: " . $msgString );
                }
	}


	private static function LogReceivedMessage( $msg )
	{
		if( self::MSG_LOGGING_ENABLED )
		{
			// Remove newline characters to allow syslog printing.
			$msgString = str_replace( "\n", null, $msg );
                        $msgString = self::RemoveSecurityInfo( $msgString,"_userName","######");
                        $msgString = self::RemoveSecurityInfo( $msgString,"_password","@*@*@*@");
                        $len = strlen( $msg );
                        syslog( LOG_DEBUG, "Received message of size " . $len . " bytes: " . $msgString );
       		}
	}


	private static function LogPacket( $packet )
	{
		if( self::MSG_LOGGING_ENABLED )
		{
			$len = strlen( $packet );
			syslog( LOG_DEBUG, "Received packet of size: " . $len . " bytes" );
		}
	}

}

?>
