#!/bin/ash

set -u -e

cleanup()
{
  echo 'Cleaning up ...'
  rm -f /ramdisk/mkversioninfo
  rm -f /ramdisk/versioninfo.bin
  rm -f /ramdisk/update.sh
  rm -f $ARCHIVES
}

# Cleanup on exit
ARCHIVES=""
trap "cleanup" 0

echo `basename $0`-script
echo "PATH = $PATH"
PATH=/ramdisk:/rrh/bin:$PATH
export PATH

# Parameter parsing
while getopts n:v: f
do
	case $f in
	n)   NSN_FILE_NAME=$OPTARG;;
	v)   NSN_FILE_VERSION=$OPTARG;;
	\?)  echo "wrong usage"; exit 1;;
	esac
done
shift `expr $OPTIND - 1`

ARCHIVES="$@"

if [ $# -eq 0 ]; then
  echo "usage: $0 <path to tar archiv>"
  exit 1
fi

for x in "$@" ; do
  if [ ! -f $x ]; then
    echo "File \"$x\" doesn't exist!"
    exit 2
  fi 
done

# Check hardware revision
UPDATE_HARDWARE="Kathrein FLEXI RRH WCDMA 60W"
UPDATE_REVISION=1

HARDWARE=$(grep "Hardware" /proc/cpuinfo | sed -e "s@.*: @@")
REVISION=$(grep "Revision" /proc/cpuinfo | sed -e "s@.*: @@")

if [ "${HARDWARE}" != "${UPDATE_HARDWARE}" ] ; then
	echo "Wrong hardware"
	exit 3
fi

if [ ${REVISION} -ne ${UPDATE_REVISION} ] ; then
    echo "Wrong hardware revision"
    exit 3
fi

BOOT_IMAGE_FILE="/ramdisk/boot_image"

if [ -f $BOOT_IMAGE_FILE ]; then
	BOOT_IMAGE=`cat $BOOT_IMAGE_FILE`
elif [ ${BOOT_IMAGE:-2} -ne 2 ]; then
	echo "File /ramdisk/boot_image is missing but variable BOOT_IMAGE is present."
else
	echo "Neither /ramdisk/boot_image is present nor the environment variable BOOT_IMAGE is set."
	exit 1
fi

if [ $BOOT_IMAGE -eq 0 ]; then
	INDEX=1
else
	INDEX=0
fi

if [ $INDEX -eq 0 ]; then
	# Settings for flashing to location 0
	echo "Writing update to position 0"
else
	# Settings for flashing to location 1
	echo "Writing update to position 1"
fi

# Update versioninfo
cat $ARCHIVES | tar -x -f - -O mkversioninfo > /ramdisk/mkversioninfo
chmod a+x /ramdisk/mkversioninfo
cat $ARCHIVES | tar -x -f - -O versioninfo.bin > /ramdisk/versioninfo.bin
/ramdisk/mkversioninfo -u -o /ramdisk/versioninfo.bin -v "${NSN_FILE_VERSION:-unset}" -n "${NSN_FILE_NAME:-unset}"

echo "Flashing Bootloader"
cat $ARCHIVES | tar -x -f - -O bootloader.bin | flash -v --erase --copy "bootloader" || exit 4

echo "Verifying Bootloader"
cat $ARCHIVES | tar -x -f - -O bootloader.bin | flash -v --verify "bootloader" || exit 4

echo "Flashing Kernel"
cat $ARCHIVES | tar -x -f - -O uImage | cat /ramdisk/versioninfo.bin - | flash -v --output-offset 0x000000 --size 0x100000 --erase --copy "partition$INDEX" || exit 4

echo "Verifying Kernel"
cat $ARCHIVES | tar -x -f - -O uImage | cat /ramdisk/versioninfo.bin - | flash -v --output-offset 0x000000 --size 0x100000 --verify "partition$INDEX" || exit 4

echo "Flashing Rootfs"
cat $ARCHIVES | tar -x -f - -O rootfs.squashfs | flash -v --output-offset 0x100000 --size 0x6C0000 --erase --copy "partition$INDEX" || exit 4

echo "Verifying Rootfs"
cat $ARCHIVES | tar -x -f - -O rootfs.squashfs | flash -v --output-offset 0x100000 --size 0x6C0000 --verify "partition$INDEX" || exit 4

echo "Finished"
