#include "mainwindow.h"
#include "ui_mainwindow.h"
#include <QByteArray>
#include <QSize>
#include <QDataStream>
#include <QMenu>
#include <QIODevice>
#include <QApplication>
#include <QPixmap>
#include <QMessageBox>
#include <QColor>
#include <QBrush>
#include <QRect>
#include <QFont>
#include <QPushButton>
#include <QTextStream>
#include <QVariant>
#include <QDir>
#include <QFile>
#include <math.h>

using namespace std;

MainWindow::MainWindow(QWidget *parent) : QMainWindow(parent), ui(new Ui::MainWindow) {
    ui->setupUi(this);
#ifndef QT_DEBUG
    ui->menuDebug->setVisible(false);
#endif
    QApplication::setApplicationVersion("1.11");
    QApplication::setApplicationName("Tait SFE Feature Key Loader");
    this->setWindowTitle(QApplication::applicationName() + " ver " + QApplication::applicationVersion());
    myPorts = QSerialPortInfo::availablePorts();
    TAIT_RADIO_CONNECTED = false;
    canExit=true;
    connect(this,SIGNAL(TAIT_CONNECTED()),this,SLOT(tait_is_now_connected()));
    connect(this,SIGNAL(TAIT_DISCONNECTED()),this,SLOT(tait_is_now_disconnected()));
    connect(this,SIGNAL(TAIT_RADIO_FINISHED_INSPECTION()),this,SLOT(tait_inspected()));
    connect(this,SIGNAL(TAIT_RADIO_NOREPLY()),this,SLOT(tait_no_reply()));
    LoadConfiguration(); // Setup the variables.
    myStatus = new QLabel();
    myStatus->setStyleSheet("font-family:Courier;");
    ui->statusBar->setLayoutDirection(Qt::LeftToRight);
    ui->statusBar->addWidget(myStatus);
    myStatus->setText("");
    myStatus->setStyleSheet("font-family:Courier;");
    myModel = new QStandardItemModel(taitFeatures.size(),3,this);
    ui->winMain->horizontalHeader()->setStretchLastSection(true);
    ui->winMain->setShowGrid(false);
    ui->winMain->horizontalHeader()->setSectionResizeMode(QHeaderView::ResizeToContents);
    ui->winMain->verticalHeader()->hide();
    ui->winMain->setSelectionMode(QAbstractItemView::NoSelection);
    ui->winMain->setSelectionBehavior(QAbstractItemView::SelectRows);
    ui->winMain->setStyleSheet("QHeaderView::section { margin:0px; border:0px; }");
    ui->winMain->horizontalHeader()->setSectionResizeMode(QHeaderView::Fixed);
    ui->winMain->verticalHeader()->setDefaultSectionSize(1);
    ui->winMain->verticalHeader()->setMinimumSectionSize(1);
    ui->winMain->verticalHeader()->setSectionResizeMode(QHeaderView::ResizeToContents);
    ui->winMain->setUpdatesEnabled(false);
    myModel->setHorizontalHeaderLabels(QString("|Feature|Key").split("|"));
    QStandardItem *myItem;
    for (int a=0;a<myModel->rowCount();a++) {
        for (int b=0;b<myModel->columnCount();b++) {
            myItem = new QStandardItem;
            if (b < 2) myItem->setTextAlignment(Qt::AlignTop);
            if (b == 2) myItem->setTextAlignment(Qt::AlignTop|Qt::AlignHCenter);
            myItem->setFlags(myItem->flags() ^ Qt::ItemIsEditable);
            myModel->setItem(a,b,myItem);
        }
        ui->winMain->hideRow(a);
    }
    ui->winMain->setModel(myModel);
    ui->cmbPorts->clear();
    ui->butInspect->setEnabled(false);
    ui->butInspect->setCursor(Qt::PointingHandCursor);
    ui->butLoadKeys->setEnabled(false);
    ui->butLoadKeys->setCursor(Qt::PointingHandCursor);
    ui->butDisableAllKeys->setEnabled(false);
    ui->butDisableAllKeys->setCursor(Qt::PointingHandCursor);
    ui->txtModel->setText("");
    ui->txtChassis->setText("");
#ifdef WIN32
    foreach (QSerialPortInfo myPort,myPorts) { ui->cmbPorts->addItem(myPort.portName()); }
#else
    foreach(QSerialPortInfo myPort,myPorts) { ui->cmbPorts->addItem(myPort.systemLocation()); }
#endif
    if (ui->cmbPorts->count() > 0) ui->butInspect->setEnabled(true);
    ui->winMain->setUpdatesEnabled(true);
    ui->winMain->setStyleSheet("background-color:white;color:black;font-family:Courier;");
    currentAttackNum = 0;
    resetDisplay();
    ui->txtModel->setTextInteractionFlags(Qt::TextSelectableByMouse);
    ui->txtChassis->setTextInteractionFlags(Qt::TextSelectableByMouse);
    ui->txtBand->setTextInteractionFlags(Qt::TextSelectableByMouse);
    ui->txtFlash->setTextInteractionFlags(Qt::TextSelectableByMouse);
    ui->txtROM->setTextInteractionFlags(Qt::TextSelectableByMouse);
    ui->txtTEA1->setTextInteractionFlags(Qt::TextSelectableByMouse);
    ui->txtTEA2->setTextInteractionFlags(Qt::TextSelectableByMouse);
    ui->chkLoadable->setEnabled(false);
    pTick = QImage(":/tick.png");
    smTick = pTick.scaled(10,10,Qt::KeepAspectRatio);
    pCross = QImage(":/cross.png");
    smCross = pCross.scaled(10,10,Qt::KeepAspectRatio);
    ui->butDisableAllKeys->setEnabled(false);
    //ui->butDisableAllKeys->setVisible(false);
    sleep(2000);
    //radioInfo["BOOTVER"] = "1";
    //QStringList tmp2 = taitGenerateSFE("00",3);
    //qDebug() << QString(tmp2.join(" "));
}

MainWindow::~MainWindow() {
    disconnect(this,SIGNAL(TAIT_CONNECTED()),this,SLOT(tait_is_now_connected()));
    disconnect(this,SIGNAL(TAIT_DISCONNECTED()),this,SLOT(tait_is_now_disconnected()));
    disconnect(this,SIGNAL(TAIT_RADIO_FINISHED_INSPECTION()),this,SLOT(tait_inspected()));
    disconnect(this,SIGNAL(TAIT_RADIO_NOREPLY()),this,SLOT(tait_no_reply()));
    delete ui;
}

void MainWindow::on_butInspect_clicked() {
    QString myReturn, tmpStr, varCmd;
    ui->cmbPorts->setEnabled(false);
    ui->butInspect->setEnabled(false);
    ui->butLoadKeys->setEnabled(false);
    ui->butDisableAllKeys->setEnabled(false);
    ui->cmbPorts->setFocus();
    canExit=false;
    myStatus->setText("Attempting to connect to the Tait Radio on " + ui->cmbPorts->currentText());
    myReturn = taitWrite("^","v");      // Attempt to contact the radio
    if (myReturn=="ERROR") return;
    myReturn = taitWrite("%","-");      // Lets start with TEST MODE
     if (myReturn=="ERROR") return;
    if (mySerial.isOpen()) emit TAIT_CONNECTED();
    taitInterrogate();
    myStatus->setText("Switching Tait Radio on " + ui->cmbPorts->currentText() + " to programming mode to continue interrogation");
    myReturn = taitWrite("^","v");
    myReturn = taitWrite("#","v");       // Move to PROGRAMMING MODE
    QStringList myCmds = QString("ld p01 r00 d01 r00 d00 p01 r22 d00 r00").split(" ");
    QStringList varList;
    foreach(varCmd,myCmds) {
        myReturn = taitWrite(varCmd,">");
        varList = myReturn.split("\r");
        foreach(tmpStr,varList) {
            tmpStr.chop(2);
            if (varCmd=="r22") { radioInfo["R22"] = tmpStr.mid(4).trimmed(); }
            if ((varCmd=="r00") && (tmpStr.left(4)=="0008")) { ui->txtModel->setText(HextoAscii(tmpStr.mid(6)).trimmed().toUpper()); radioInfo["MODEL"] = ui->txtModel->text(); }
        }
    }
    ui->txtModel->setText(ui->txtModel->text() + " (ver " + radioInfo.value("BOOTVER") + ")");
    // Now we have checked for the extra information lets move on to retrieving the SFE keys currently in the radio.
    QFont myFont = QFont();
    QStandardItem *myItem;
    int a = 0;
    QStringList keyStatus;
    myStatus->setText("Now querying SFE keys for Tait Radio connected to " + ui->cmbPorts->currentText());
    foreach(QString varCmd,taitQueryOrder) {
        tmpStr = taitChecksum("02" + varCmd);
        tmpStr.prepend("f");
        myReturn = taitWrite(tmpStr,">");
        if (myReturn.length()==32) keyStatus = taitSFETest(myReturn);
        if (myReturn=="ERROR") continue;
        if (myReturn=="01FF") {
            myReturn = "FEATURE NOT SUPPORTED";
            continue;
        } else if (myReturn=="{C02}") {
            myReturn = "INVALID CHECKSUM"; // Really hope not.
        } else if ((myReturn!="01FF") && (myReturn!="{C02}")) {
            if (ui->txtFlash->text().trimmed()=="") { // We don't already know the ESN, lets grab it now.
                bool ok;
                uint tnum = QString("0x" + myReturn.mid(20,7)).toUInt(&ok,16);
                tnum >>=1 ;
                ui->txtFlash->setText(QString::number(tnum,10).rightJustified(8,'0'));
                radioInfo["ESN"] = ui->txtFlash->text().trimmed();
            }
            myReturn = taitKeyFromHex(myReturn);
        }
        (keyStatus.at(0)=="ENABLED") ? myFont.setBold(true) : myFont.setBold(false);
        myItem = myModel->item(a,0);
        if (keyStatus.at(0)=="ENABLED") myItem->setData(QVariant(QPixmap::fromImage(smTick)),Qt::DecorationRole);
        if (keyStatus.at(0)=="DISABLED") myItem->setData(QVariant(QPixmap::fromImage(smCross)),Qt::DecorationRole);
        myFont.setFamily("Courier");
        myFont.setStyleHint(QFont::TypeWriter);
        myFont.setPointSize(10);
        myItem = myModel->item(a,1);
        myItem->setFont(myFont);
        myItem->setText(taitFeatures.value(varCmd));
        myItem = myModel->item(a,2);
        myItem->setFont(myFont);
        myItem->setText(myReturn.trimmed());
        ui->winMain->showRow(a);
        a++;
    }
    myStatus->setText("All SFE Keys retrieved. Resetting the radio on " + ui->cmbPorts->currentText());
    myReturn = taitWrite("^","v"); // Reset the radio.
    sleep(4000);
    if (!radioInfo.contains("TEA")) radioInfo.insert("TEA","UNKNOWN");
    if (!radioInfo.contains("R22")) radioInfo.insert("R22","UNKNOWN");
    radioDB[ui->txtChassis->text()] = radioInfo;
    emit TAIT_RADIO_FINISHED_INSPECTION();
    emit TAIT_DISCONNECTED();
    myStatus->setText("Inspection of Tait Radio on " + ui->cmbPorts->currentText() + " complete.");
    ui->cmbPorts->setFocus();
}

void MainWindow::closeEvent(QCloseEvent *event) {
    if (mySerial.isOpen() || (!canExit)) {
        event->ignore();
        QMessageBox::warning(this,"Error","Please wait for the radio to reset before closing the application");
        return;
    }
    if (QMessageBox::question(this,"Confirm","Are you sure?",QMessageBox::Yes,QMessageBox::No) == QMessageBox::No) { event->ignore(); return; }
    if (mySerial.isOpen()) {
        taitWrite("^",">");
        mySerial.close();
    }
}

void MainWindow::resizeEvent(QResizeEvent *) {
    QSize mySize;
    mySize = ui->winMain->size();
    ui->winMain->setColumnWidth(0,20);
    ui->winMain->setColumnWidth(1, mySize.width() * 0.5);
    QApplication::processEvents(QEventLoop::AllEvents);
}

void MainWindow::on_cmbPorts_currentIndexChanged(const QString &arg1) {
    if (arg1==0) { }
    if (mySerial.isOpen()) mySerial.close();
    resetDisplay();
    ui->butInspect->setEnabled(true);
    QStandardItem *myItem;
    for (int a=0;a<myModel->rowCount();a++) {
        myItem = myModel->item(a,1);
        myItem->setText("");
        ui->winMain->hideRow(a);
    }

}

void MainWindow::sleep(int myInt) {
    QDateTime startTime = QDateTime::currentDateTime();
    while (startTime.msecsTo(QDateTime::currentDateTime()) < myInt) {
        QApplication::processEvents(QEventLoop::AllEvents);
    }
}

QString MainWindow::HextoAscii(QString String) {
    QByteArray ByteArray1=String.toUtf8();
    const char* chArr1=ByteArray1.constData();
    QByteArray ByteArray2=QByteArray::fromHex(chArr1);
    const char* chArr2=ByteArray2.constData();
    return QString::fromUtf8(chArr2);
}
QString MainWindow::AsciitoHex(QString String) {
    QByteArray ByteArray1=String.toUtf8();
    QByteArray ByteArray2=ByteArray1.toHex();
    const char* chArr1=ByteArray2.constData();
    return QString::fromUtf8(chArr1);
}

void MainWindow::LoadConfiguration() {
    ui->menuDebug->setChecked(false);
    // Some tables to make conversion between bytes and 5 bit Tait Ascii easy. Adapted from CRCinAU's perl script.
    taitAlphabet.insert("00000","T"); // 00
    taitAlphabet.insert("00001","3"); // 01
    taitAlphabet.insert("00010","F"); // 02
    taitAlphabet.insert("00011","J"); // 03
    taitAlphabet.insert("00100","D"); // 04
    taitAlphabet.insert("00101","6"); // 05
    taitAlphabet.insert("00110","W"); // 06
    taitAlphabet.insert("00111","8"); // 07
    taitAlphabet.insert("01000","A"); // 08
    taitAlphabet.insert("01001","C"); // 09
    taitAlphabet.insert("01010","7"); // 10
    taitAlphabet.insert("01011","H"); // 11
    taitAlphabet.insert("01100","N"); // 12
    taitAlphabet.insert("01101","E"); // 13
    taitAlphabet.insert("01110","G"); // 14
    taitAlphabet.insert("01111","4"); // 15
    taitAlphabet.insert("10000","2"); // 16
    taitAlphabet.insert("10001","V"); // 17
    taitAlphabet.insert("10010","S"); // 18
    taitAlphabet.insert("10011","Z"); // 19
    taitAlphabet.insert("10100","5"); // 20
    taitAlphabet.insert("10101","P"); // 21
    taitAlphabet.insert("10110","Y"); // 22
    taitAlphabet.insert("10111","U"); // 23
    taitAlphabet.insert("11000","9"); // 24
    taitAlphabet.insert("11001","L"); // 25
    taitAlphabet.insert("11010","R"); // 26
    taitAlphabet.insert("11011","B"); // 27
    taitAlphabet.insert("11100","K"); // 28
    taitAlphabet.insert("11101","M"); // 29
    taitAlphabet.insert("11110","X"); // 30
    taitAlphabet.insert("11111","Q"); // 31
    taitAlphabetB[0] = "T"; //00
    taitAlphabetB[1] = "3"; // 01
    taitAlphabetB[2] = "F"; // 02
    taitAlphabetB[3] = "J"; // 03
    taitAlphabetB[4] = "D"; // 04
    taitAlphabetB[5] = "6"; // 05
    taitAlphabetB[6] = "W"; // 06
    taitAlphabetB[7] = "8"; // 07
    taitAlphabetB[8] = "A"; // 08
    taitAlphabetB[9] = "C"; // 09
    taitAlphabetB[10] = "7"; // 10
    taitAlphabetB[11] = "H"; // 11
    taitAlphabetB[12] = "N"; // 12
    taitAlphabetB[13] = "E"; // 13
    taitAlphabetB[14] = "G"; // 14
    taitAlphabetB[15] = "4"; // 15
    taitAlphabetB[16] = "2"; // 16
    taitAlphabetB[17] = "V"; // 17
    taitAlphabetB[18] = "S"; // 18
    taitAlphabetB[19] = "Z"; // 19
    taitAlphabetB[20] = "5"; // 20
    taitAlphabetB[21] = "P"; // 21
    taitAlphabetB[22] = "Y"; // 22
    taitAlphabetB[23] = "U"; // 23
    taitAlphabetB[24] = "9"; // 24
    taitAlphabetB[25] = "L"; // 25
    taitAlphabetB[26] = "R"; // 26
    taitAlphabetB[27] = "B"; // 27
    taitAlphabetB[28] = "K"; // 28
    taitAlphabetB[29] = "M"; // 29
    taitAlphabetB[30] = "X"; // 30
    taitAlphabetB[31] = "Q"; // 31
    hexTable.insert("0","0000");
    hexTable.insert("1","0001");
    hexTable.insert("2","0010");
    hexTable.insert("3","0011");
    hexTable.insert("4","0100");
    hexTable.insert("5","0101");
    hexTable.insert("6","0110");
    hexTable.insert("7","0111");
    hexTable.insert("8","1000");
    hexTable.insert("9","1001");
    hexTable.insert("A","1010");
    hexTable.insert("B","1011");
    hexTable.insert("C","1100");
    hexTable.insert("D","1101");
    hexTable.insert("E","1110");
    hexTable.insert("F","1111");
    bitTable.insert("0000","0");
    bitTable.insert("0001","1");
    bitTable.insert("0010","2");
    bitTable.insert("0011","3");
    bitTable.insert("0100","4");
    bitTable.insert("0101","5");
    bitTable.insert("0110","6");
    bitTable.insert("0111","7");
    bitTable.insert("1000","8");
    bitTable.insert("1001","9");
    bitTable.insert("1010","A");
    bitTable.insert("1011","B");
    bitTable.insert("1100","C");
    bitTable.insert("1101","D");
    bitTable.insert("1110","E");
    bitTable.insert("1111","F");
    // List of known Tait Features. If anyone knows of more (probably DMR ones) please publish them.
    taitFeatures.insert("00", "TMAS010 - Tait High Speed Data (8xxx)");
    taitFeatures.insert("01", "TMAS011 - MPT1327 Trunking (8xxx)");
    taitFeatures.insert("02", "TMAS012 - MDC1200 Encode (8xxx)");
    taitFeatures.insert("04", "TBAS050 - P25 Common Air Interface (TB9100)");
    taitFeatures.insert("05", "TxAS015 - GPS Display");
    taitFeatures.insert("06", "TMAS016 - Multi-Body Support (8xxx)");
    taitFeatures.insert("07", "TMAS017 - Multi-Head Support (8xxx)");
    taitFeatures.insert("08", "TMAS018 - TDMA Support (8xxx)");
    taitFeatures.insert("09", "TMAS019 - Conventional Call Queuing (8xxx) / TBAS055 - Transmit Enable (TB9100)");
    taitFeatures.insert("0A", "TMAS020 - Lone Worker Support");
    taitFeatures.insert("14", "TxAS050 - P25 Common Air Interface");
    taitFeatures.insert("15", "TxAS051 - P25 Adminitrator Services");
    taitFeatures.insert("16", "TMAS052 - P25 Graphical C/Head Operation");
    taitFeatures.insert("17", "TxAS053 - Single DES Encryption & Key Loading");
    taitFeatures.insert("18", "TxAS054 - P25 Base OTAR Re-Keying");
    taitFeatures.insert("19", "TxAS055 - P25 Trunking Services");
    taitFeatures.insert("1A", "TxAS056 - P25 User IP Data");
    taitFeatures.insert("1B", "TxAS057 - P25 Base Encryption (DES) & Key Loading");
    taitFeatures.insert("1C", "TxAS058 - P25 Encryption (AES)");
    taitFeatures.insert("1D", "TxAS059 - MDC1200");
    taitFeatures.insert("1E", "TxAS060 - Tait Radio API");
    taitFeatures.insert("1F", "TxAS061 - P25 Protocol API");
    taitFeatures.insert("20", "TMAS062 - P25 Digital Crossband");
    taitFeatures.insert("21", "TxAS063 - P25 DLI / Trunked OTAR");
    taitFeatures.insert("22", "TxAS064 - P25 Trunked PSTN");
    taitFeatures.insert("23", "TxAS065 - 2-Tone Decode");
    taitFeatures.insert("24", "TxAS066 - Reserved (P25 Diagnostic Menu)");
    taitFeatures.insert("25", "TxAS067 - GPS Transmission");
    taitFeatures.insert("26", "TMAS068 - TM9000 Multi-Body Support");
    taitFeatures.insert("27", "TMAS069 - TM9000 Multi-Head Support");
    taitFeatures.insert("28", "APCO TCI");
    taitFeatures.insert("29", "TMAS071 - GPS Logging");
    taitFeatures.insert("2A", "TMAS072 - Alphanumeric ID");
    taitFeatures.insert("2B", "TMAS073 - Emergency Acknowledgement");
    taitFeatures.insert("2C", "TMAS074 - Terminal Repeater Detection Collision Avoidance");
    taitFeatures.insert("2D", "TPAS075 - SFE - OTAP (Over-The-Air Programming)");
    taitFeatures.insert("30", "Tait Radio API");
    taitFeatures.insert("32", "TPAS080 - SFE - DMR Trunking Digital");
    taitFeatures.insert("33", "TPAS081 - SFE - GPS Hardware Location Services");
    taitFeatures.insert("34", "TPAS082 - SFE - Bluetooth");
    taitFeatures.insert("35", "TMAS083 - 20/25kHz Unrestricted Wideband");
    taitFeatures.insert("36", "TPAS084 - SFE - WiFi");
    taitFeatures.insert("38", "Enhanced Channel Capacity");
    taitFeatures.insert("39", "TPAS087 - SFE - Voice Annunciations");
    taitFeatures.insert("3A", "P25 Digital Crossband Remote Control");
    taitFeatures.insert("3B", "TMAS089 - Enhanced Location Reporting");
    taitFeatures.insert("3F", "TMAS093 - Keyloading");
    taitFeatures.insert("40", "TPAS094 - SFE - Unrestricted Dist DES Encryption (9335,9337)");
    taitFeatures.insert("41", "TPAS095 - SFE - DES Encryption (9355,9357)");
    taitFeatures.insert("43", "TPAS097 - SFE - DMR Conventional");
    taitFeatures.insert("44", "Trunked GPS Transmission");
    taitFeatures.insert("45", "BIOLINK");
    taitFeatures.insert("48", "TPAS102 - SFE - ARC4 Encryption");
    taitFeatures.insert("49", "TPAS103 - SFE - Radio Logging");
    taitFeatures.insert("4A", "TPAS104 - SFE - Extended Hunt List Capacity");
    taitFeatures.insert("4B", "TPAS105 - SFE - Geofencing Services");
    taitFeatures.insert("4C", "TMAS106 - SFE - File Transfer Services");
    taitFeatures.insert("4D", "TPAS107 - SFE - Embedded GPS Decoding");
    taitFeatures.insert("4E", "TMAS108 - SFE - Unify API");
    taitFeatures.insert("64", "TMAS130 - SFE - Tait Internal 4");
    taitFeatures.insert("65", "TMAS131 - SFE - Tait Internal D");
    //taitQueryOrder = QString("00 01 02 05 06 07 08 09 0A 14 15 16 17 18 19 1A 1B 1C 1E 1F 20 21 22 23 24 25 26 27 29 2A 2B 2C 35").split(" "); // Needed as Qt Randomly changes QHash iteration order.
    taitQueryOrder.clear();         // Lets autofill in numeric order. QHash randomly changes the order of items and that irritates me.
    QString tmpStr;
    for (int a=0;a<255;a++) {
        tmpStr = QString::number(a,16).rightJustified(2,'0').toUpper();
        if (taitFeatures.contains(tmpStr)) taitQueryOrder.append(tmpStr);
    }
    // List of known band splits for standard issue radios
    taitBands.insert("A4","VHF 66 - 88 Mhz");
    taitBands.insert("B1","VHF 136 - 174 Mhz");
    taitBands.insert("C0","VHF 174 - 225 Mhz");
    taitBands.insert("D1","VHF 216 - 266 Mhz");
    taitBands.insert("G2","UHF 350 - 400 Mhz");
    taitBands.insert("H5","UHF 400 - 470 Mhz");
    taitBands.insert("H6","UHF 450 - 530 Mhz");
    taitBands.insert("H7","UHF 450 - 520 Mhz");
    taitBands.insert("K5","UHF 762 - 870 Mhz");
    // Supported models for SFE loading
    taitSupportedModels = QString("TMAB22|TMAB23|TMAB24").split("|");
}

void MainWindow::resetDisplay() {
    radioInfo.clear();
#ifdef WIN32
    ui->butInspect->setStyleSheet("background-color:darkgreen;color:white;border-style:ridge;border-width:1px;padding:2px;font-family:Courier;");
    ui->butLoadKeys->setStyleSheet("background-color:darkred;color:white;border-style:ridge;border-width:1px;padding:2px;font-family:Courier;");
    ui->butDisableAllKeys->setStyleSheet("background-color:darkblue;color:white;border-style:ridge;border-width:1px;padding:2px;font-family:Courier;");
#else
    ui->butInspect->setStyleSheet("background-color:darkgreen;color:white;font-family:Courier;");
    ui->butLoadKeys->setStyleSheet("background-color:darkred;color:white;font-family:Courier;");
    ui->butDisableAllKeys->setStyleSheet("background-color:darkblue;color:white;font-family:Courier;");
#endif
    ui->txtModel->setStyleSheet("background-color:white;font-family:Courier;");
    ui->txtModel->setText("");
    ui->txtChassis->setStyleSheet("background-color:white;font-family:Courier;");
    ui->txtChassis->setText("");
    ui->txtROM->setStyleSheet("background-color:white;font-family:Courier;");
    ui->txtROM->setText("");
    ui->txtBand->setStyleSheet("background-color:white;font-family:Courier;");
    ui->txtBand->setText("");
    ui->txtFlash->setStyleSheet("background-color:white;font-family:Courier;");
    ui->txtFlash->setText("");
    ui->txtTEA1->setStyleSheet("background-color:white;font-family:Courier;");
    ui->txtTEA1->setText("");
    ui->txtTEA2->setStyleSheet("background-color:white;font-family:Courier;");
    ui->txtTEA2->setText("");
    ui->cmbPorts->setStyleSheet("background-color:white;color:black;font-family:Courier;");
    ui->chkLoadable->setChecked(false);
    QStandardItem *myItem;
    for (int a=0;a<myModel->rowCount();a++) {
        myItem = myModel->item(a,1);
        myItem->setText("");
        myItem->setBackground(QBrush(QColor(Qt::white)));
        myItem->setForeground(QBrush(QColor(Qt::black)));
        ui->winMain->hideRow(a);
    }
    ui->winMain->clearSelection();
    ui->cmbPorts->setFocus();
}

void MainWindow::on_actionApplication_triggered(){
    QMessageBox about;
    QPixmap pix(":/8200.jpg");
    about.setText(QApplication::applicationName() + " " + QApplication::applicationVersion());
    about.setInformativeText("Developed February 2017 thanks to work by CRCinAU and mnix\nModel Support :\n- TM82xx\n- TM9xxx (Read Only)\n");
    about.setStandardButtons(QMessageBox::Ok);
    about.setDefaultButton(QMessageBox::Ok);
    about.setIconPixmap(pix.scaled(400,400,Qt::KeepAspectRatio));
    about.show();
    about.exec();
}

void MainWindow::testfunction(QString newFunction, int newSequence) {
    if ((newFunction=="") || (newSequence==0)) return;
}

void MainWindow::on_butLoadKeys_clicked() {
    doKeys("ENABLEALL");
}

void MainWindow::on_butDisableAllKeys_clicked() {
    doKeys("DISABLEALL");
}

void MainWindow::toggleOneSFE() {
    doKeys("TOGGLE");
}

void MainWindow::on_winMain_customContextMenuRequested(const QPoint &pos) {
    if (!canExit || mySerial.isOpen()) { QMessageBox::warning(this,"Wait","Please wait for the current serial port operations to finish",QMessageBox::Ok); return; }
    varRowNum = ui->winMain->rowAt(pos.y());
    if (varRowNum < 0) { ui->winMain->clearSelection(); ui->cmbPorts->setFocus(); return; }
    QMenu myMenu(this);
    myMenu.addSection(myModel->item(varRowNum,1)->text());
    myMenu.addSeparator();
    myMenu.addAction("Toggle SFE Feature Option",this,SLOT(toggleOneSFE()));
    myMenu.popup(ui->winMain->mapToGlobal(pos));
    myMenu.show();
    myMenu.exec();
}

void MainWindow::doKeys(QString myOption) {
    ui->cmbPorts->setEnabled(false);
    ui->butInspect->setEnabled(false);
    ui->butLoadKeys->setEnabled(false);
    ui->butDisableAllKeys->setEnabled(false);
    ui->cmbPorts->setFocus();
    canExit=false;
    int a = 0;
    int keysAttempted = 0;
    int keysSuccessful = 0;
    int keysFailed = 0;
    QString myReturn = "";
    QString tmpStr = "";
    QStringList newKey;
    bool continueProcessing = true;
    bool ok = true;
    QStandardItem *myItem;
    myItem = myModel->item(0,1);
    QString myFeatureNum = "00";
    QStringList keyStatus;
    bool EnableFeature = true;
    while (continueProcessing) {
        QFont myFont = myModel->item(0,0)->font();
        myFont.setFamily("Courier");
        myFont.setBold(true);
        if (!taitSupportedModels.contains(radioInfo.value("BODY"))) { QMessageBox::warning(this,"Not Supported","Sorry - this model of radio is not currently supported",QMessageBox::Ok); break; }
        if (radioInfo.value("TEA").length()!=64) { QMessageBox::warning(this,"No TEA Key","No TEA Key was found. Ensure you have run this application after loading mnix firmware. See the doucmentation for help.",QMessageBox::Ok); break; }
        myStatus->setText("Attempting to connect to the Tait Radio on " + ui->cmbPorts->currentText());
        myReturn = taitWrite("^","v");
        if (myReturn == "ERROR") break;
        myStatus->setText("Switching Tait Radio on " + ui->cmbPorts->currentText() + " to programming mode");
        myReturn = taitWrite("#",">");          // Move to PROGRAMMING MODE
        if (myReturn == "ERROR") break;
        sleep(5000);                            // Or else radio falls out of sequence
        if (myOption=="TOGGLE") a = varRowNum;
        myItem = myModel->item(a,2);            // Lets do some sanity checking and make sure that our SFE generation is matching a loaded key. Needed due to bootversion.
        keyStatus = taitSFETest(taitHexFromKey(myItem->text()));                                // This gives is ?ENABLED|CurrentSeq|NextSequence
        // Lets ensure we can accurately determine keys for this radio.
        tmpStr = taitGenerateSFE(keyStatus.at(3),QString(keyStatus.at(1)).toInt(&ok)).at(0);       // Generate a key using the current feature number and current sequence number.
        if (tmpStr != myItem->text()) {                                                         // Failed! Try changing the bootversion.
            radioInfo["BOOTVER"] = (radioInfo.value("BOOTVER")=="1") ? "2" : "1";
            tmpStr = taitGenerateSFE(myFeatureNum,QString(keyStatus.at(1)).toInt(&ok)).at(0);   // Retry with different bootversion.
            if (tmpStr != myItem->text()) {
#ifdef QT_DEBUG
                qDebug() << "Boot version failure despite attempting crash fix.";
#endif
                keysFailed++;
                break;
            }                           // Crash and burn.
        }
        // OK - we know that we can successfully replicate a key already loaded into the radio. Lets process all the keys.
        myStatus->setText("Now processing SFE key(s) for Tait Radio connected to " + ui->cmbPorts->currentText());
        for (a=0;a<taitFeatures.size();a++) {
            EnableFeature = true;
            if (myOption=="TOGGLE") a = varRowNum;
            if (ui->winMain->isRowHidden(a)) break;                                             // The row is hidden - so the previous line was the last valid key.
            myItem = myModel->item(a,2);                                                        // Get the current key being looked at from the display
            keyStatus = taitSFETest(taitHexFromKey(myItem->text()));                            // This gives is ?ENABLED|CurrentSeq|NextSequence
            myFeatureNum = keyStatus.at(3);                                                     // The feature number for this key.
            // Do we actually need to do anyting at all.
            if ((myOption=="ENABLEALL") && (keyStatus.at(0)=="ENABLED")) continue;              // Key already enabled (which we want).. no action required. Next
            if ((myOption=="DISABLEALL") && (keyStatus.at(1)=="DISABLED")) continue;            // Key already disabled (which we want).. no action required. Next.
            if (myOption=="DISABLEALL") EnableFeature = false;
            if ((myOption=="TOGGLE") && (keyStatus.at(0)=="ENABLED")) EnableFeature = false;
            if (EnableFeature) {
                newKey = taitGenerateSFE(myFeatureNum,QString(keyStatus.at(2)).toInt(&ok,10));
                tmpStr = "f" + newKey.at(1);
            } else {
                tmpStr = "f" + taitChecksum("01" + myFeatureNum);
            }
            myFont = myModel->item(a,2)->font();
            // So we have a new SFE. Lets try to send it to the radio.
            myReturn = taitWrite(tmpStr,">");
#ifdef QT_DEBUG
            qDebug() << QString(keyStatus.join("|")) << "Trying" << QString(tmpStr) << "GOT" << myReturn;
#endif
            if (!EnableFeature) {
                if (myReturn.length()==32) { // Yay - we managed to disable the key successfully.
                    keysSuccessful++;
                    tmpStr = taitKeyFromHex(myReturn);
                    myFont.setBold(false);
                    myModel->item(a,0)->setData(QVariant(QPixmap::fromImage(smCross)),Qt::DecorationRole);
                    myModel->item(a,2)->setText(tmpStr);
                    myModel->item(a,1)->setFont(myFont);
                    myModel->item(a,2)->setFont(myFont);
                } else {
                    keysFailed++;
                }
            } else if (myReturn=="0000") {
                keysSuccessful++;
                myFont.setBold(true);
                myModel->item(a,2)->setText(newKey.at(0));
                myModel->item(a,0)->setData(QVariant(QPixmap::fromImage(smTick)),Qt::DecorationRole);
                myModel->item(a,1)->setFont(myFont);
                myModel->item(a,2)->setFont(myFont);
            } else {
                keysFailed++;
            }
            if (myOption=="TOGGLE") break;
        }
        if (keysFailed > 0) QMessageBox::information(this,"Complete","SFE Key change sequence complete.\nKeys Attempted: " + QString::number(keysAttempted) + "\nKeys Successful: " + QString::number(keysSuccessful) + "\nKeys Failed:" + QString::number(keysFailed),QMessageBox::Ok);
        if (myOption=="TOGGLE") myStatus->setText("SFE Key toggle complete. Resetting radio.");
        if (myOption=="ENABLEALL") myStatus->setText("SFE Key enabling complete. Resetting radio.");
        if (myOption=="DISABLEALL") myStatus->setText("SFE Key disabling complete. Resetting radio.");
        break;  // End of function. Exit.
    }
    // End of function - clean up.
    if (mySerial.isOpen()) {
        myReturn = taitWrite("^","v"); // Reset the radio.
        sleep(5000);
        mySerial.close();
    }
    ui->butLoadKeys->setEnabled(true);
    ui->butInspect->setEnabled(true);
    ui->butDisableAllKeys->setEnabled(true);
    ui->cmbPorts->setEnabled(true);
    ui->cmbPorts->setFocus();
    canExit=true;
    myStatus->setText("Tait radio on " + ui->cmbPorts->currentText() + " disconnected.");
}

// ===================================================================================================================================
// Cryptography Functions
// ===================================================================================================================================
QString MainWindow::TEAEncrypt(QString vData, QString vKey) {
    if (vData.length() < 16) return "ERROR";                        // This code reworked from wikipedia provided information
    if (vKey.length() < 32) return "ERROR";
    bool ok;
    uint32_t v[2] = { 0,0 };
    uint32_t k[4] = { 0,0,0,0 };
    v[0] = vData.mid(0,8).toULong(&ok,16);                          // Since we are converting QStrings of hex we do not need to do the
    v[1] = vData.mid(8,8).toULong(&ok,16);                          // bit shifting you see on the wiki page to get the two datablocks
    k[0] = vKey.mid(0,8).toULong(&ok,16);
    k[1] = vKey.mid(8,8).toULong(&ok,16);
    k[2] = vKey.mid(16,8).toULong(&ok,16);
    k[3] = vKey.mid(24,8).toULong(&ok,16);
    uint32_t v0=v[0], v1=v[1], sum=0;                               // Set up
    uint32_t delta=0x9e3779b9;                                      // Key schedule constant
    uint32_t k0 = k[0];
    uint32_t k1 = k[1];
    uint32_t k2 = k[2];
    uint32_t k3 = k[3];
    for (int i=0; i < 32; i++) {                                    // Cycle start
        sum += delta;
        v0 += ((v1 << 4) + k0) ^ (v1 + sum) ^ ((v1 >> 5) + k1);
        v1 += ((v0 << 4) + k2) ^ (v0 + sum) ^ ((v0 >> 5) + k3);
    }                                                               // End cycle
    v[0]=v0; v[1]=v1;                                               // Absolutely vital to pad the front with 0's or else things break badly.
    return QString(QString::number(v0,16).rightJustified(8,'0') + QString::number(v1,16).rightJustified(8,'0')).toUpper();
}


QString MainWindow::TEADecrypt(QString vData, QString vKey) {
    if (vData.length() < 16) return "ERROR";                        // The comments here are the same as for TEAEncrypt. This function is
    if (vKey.length() < 32) return "ERROR";                         // also from wikipedia and simply reverses the encryption routine.
    bool ok;
    uint32_t v[2];
    uint32_t k[4];
    v[0] = vData.mid(0,8).toULong(&ok,16);
    v[1] = vData.mid(8,8).toULong(&ok,16);
    k[0] = vKey.mid(0,8).toULong(&ok,16);
    k[1] = vKey.mid(8,8).toULong(&ok,16);
    k[2] = vKey.mid(16,8).toULong(&ok,16);
    k[3] = vKey.mid(24,8).toULong(&ok,16);
    uint32_t v0=v[0], v1=v[1], sum=0xC6EF3720;
    uint32_t delta=0x9e3779b9;
    uint32_t k0 = k[0];
    uint32_t k1 = k[1];
    uint32_t k2 = k[2];
    uint32_t k3 = k[3];
    for (int i=0; i < 32; i++) {
        v1 -= ((v0 << 4) + k2) ^ (v0 + sum) ^ ((v0 >> 5) + k3);
        v0 -= ((v1 << 4) + k0) ^ (v1 + sum) ^ ((v1 >> 5) + k1);
        sum -= delta;
    }
    v[0]=v0; v[1]=v1;                                               // Don't forget the padding!
    return QString(QString::number(v0,16).rightJustified(8,'0') + QString::number(v1,16).rightJustified(8,'0')).toUpper();
}
// ===================================================================================================================================
// Tait Functions
// ===================================================================================================================================
QStringList MainWindow::taitGenerateSFE(QString newFeature, int newSequence) { // Sequence defaults to 1 if not provided.
    QString DefaultTEA = radioInfo.value("TEA");    // "0010000106B32F2BFFFF55B08F4C003204248298A8B4C32183E1C36D11F27032"
    DefaultTEA = "0010000106B32F2BFFFF55B08F4C003204248298A8B4C32183E1C36D11F27032";
    QString tmpStr;                                 // Magic2 "52F3F3DCEA43E28C"
    QString magic1 = "3467E59B1F95FEB9";
    QString varKey = QString(DefaultTEA).mid(12,4) + QString(DefaultTEA).mid(8,4) + QString(DefaultTEA).mid(20,4) + QString(DefaultTEA).mid(16,4) + magic1;
    QString radioESN = "00000000";
    QString varData = "";
    QString decryptedTEAkey = "";
    QString defaultTEAkey = "";
    QString SFE_TEA_Key = "";
    int a=0;
    bool ok;
    // ESN byte shuffled 00328F4C = 8F4C0032
    tmpStr = QString(DefaultTEA).mid(28,4) + QString(DefaultTEA).mid(24,4); // This is the system ESN.
    radioESN = QString::number(tmpStr.toULong(&ok,16),10);
    // We will get the decryptedTEAkey. Key = (Some bytes Seed1 swapped) + magic. Data = Bytes from Seed2 swapped around.
    tmpStr = QString(DefaultTEA).mid(48,16);
    varData = tmpStr.mid(4,4) + tmpStr.left(4) + tmpStr.right(4) + tmpStr.mid(8,4); // called hswap_block1
    varKey = QString(DefaultTEA).mid(12,4) + QString(DefaultTEA).mid(8,4) + QString(DefaultTEA).mid(20,4) + QString(DefaultTEA).mid(16,4) + magic1;
    // Generate the deCryptedTEAKey and from there shifts bytes around to get the defaultTEAKey
    decryptedTEAkey = TEADecrypt(varData,varKey);
    defaultTEAkey = decryptedTEAkey.mid(4,4) + decryptedTEAkey.left(4) + decryptedTEAkey.right(4) + decryptedTEAkey.mid(8,4); // called hswap_block1
    tmpStr = defaultTEAkey.mid(4,4) + defaultTEAkey.left(4) + defaultTEAkey.right(4) + defaultTEAkey.mid(8,4);
    tmpStr += "59" + defaultTEAkey.mid(10,4) + "A22C" + defaultTEAkey.mid(2,2) + defaultTEAkey.mid(8,2) + "F7";
    SFE_TEA_Key = tmpStr;
    tmpStr = TEADecrypt("0000000000000000",SFE_TEA_Key);
    uint16_t tmp[16];
    for (a=0;a<tmpStr.length();a++) {
        tmp[a] = tmpStr.left(2).toUInt(&ok,16);
        tmpStr = tmpStr.mid(2);
    }
    if (newSequence > 255) newSequence = 0;                     // Loop to avoid > 0xFF
    tmp[2] = (newSequence >= 0) ? newSequence : tmp[2] + 1;     // Set or Increment the SFE sequence...
    tmp[3] = newFeature.toUInt(&ok,16);                         // update the feature code
    // set the ESN/RSN from the system-data we have
    tmp[7] = (radioESN.toUInt(&ok,10) >> 16) & 0xFF;
    tmp[0] = (radioESN.toUInt(&ok,10) >>  8) & 0xFF;
    tmp[1] = radioESN.toUInt(&ok,10) & 0xFF;
    // update the other bytes to match...
    // if (BOOTFW >= 2) { // Add this in once this is known - QMAB2B_STD_1.04.01.0001, so take the number after STD_, in this case 1.
    if (radioInfo.value("BOOTVER") == "1") {
        tmp[5] = ((tmp[1]>>4) & 0x0F) | ((tmp[0]<<4) & 0xF0);
    } else {
        tmp[5] = ((tmp[1]>>4) & 0x0F) | ((tmp[1]<<4) & 0xF0);
    }
    tmp[6] = ((tmp[3]>>4) & 0x0F) | ((tmp[3]<<4) & 0xF0);
    tmp[4] = ((tmp[7]>>4) & 0x0F) | ((tmp[2]<<4) & 0xF0);
    tmp[12] = tmp[3];
    tmp[13] = tmp[2];
    tmp[8] = (radioESN.toUInt(&ok,10) >> 24) & 0xFF;
    tmp[9] = (radioESN.toUInt(&ok,10) >> 16) & 0xFF;
    tmp[10] = (radioESN.toUInt(&ok,10) >> 8) & 0xFF;
    tmp[11] = radioESN.toUInt(&ok,10) & 0xFF;
    tmpStr.clear();
    for (a=0;a<14;a++) {
        tmpStr += QString::number((tmp[a] & 0xFF),16).rightJustified(2,'0');
    }
    QString ts = tmpStr.toUpper();
    tmpStr = TEAEncrypt(tmpStr,SFE_TEA_Key);    // This step prevents reverse decoding the string. Damn it.
    tmpStr += ts.right(12);                     // Re add the non encrypted data to end.
    a = 0;
    while (tmpStr.length()) {                   // Encode the existing data
        tmp[a] = tmpStr.left(2).toInt(&ok,16);
        tmpStr = tmpStr.mid(2);
        a++;
    }
    uint16_t sfe[16];
    for (a=8;a<14;a++) { sfe[a] = tmp[a]; }     // Copy in the feature code and ESN
    int o1, o3, rsn;
    o1 = 0xE000 & ((((tmp[0x0A] << 8) | tmp[0x0B]) << 13) | (tmp[0x0D]<<4));
    rsn = (tmp[0x08] << 24) | (tmp[0x09] << 16) | (tmp[0x0A]<<8) | tmp[0x0B];
    o3 = ((rsn >> 19) & 0xFF) | (tmp[0x0C] << 8);
    o1 |= 0x1000 | (tmp[0x0D] << 4);
    sfe[0x08] = (o3 >> 8) & 0xFF;
    sfe[0x09] = (o3 & 0xFF);
    sfe[0x0A] = (rsn >> 11) & 0xFF;
    sfe[0x0B] = (rsn >> 3) & 0xFF;
    sfe[0x0C] = (o1 >> 8) & 0xFF;
    sfe[0x0D] = o1 & 0xFF;
    for (a=0;a<4;a++) { sfe[a] = tmp[a+4]; sfe[a+4] = tmp[a]; } // Some reversing.
    tmpStr = "00";
    for(a=0;a<14;a++) {
        tmpStr += QString::number(sfe[a] & 0xFF,16).rightJustified(2,'0');
    }
    tmpStr = tmpStr.rightJustified(28,'0');
    tmpStr = taitChecksum(tmpStr).toUpper();
    varData = taitKeyFromHex(tmpStr);
    return QString(varData + "|" + tmpStr).split("|");
}

QString MainWindow::taitEncodeKey(QString myStr) {
    QString myReturn = "";
    QString tmpStr = "";
    int z = 0;
    for (int a=0;a<myStr.length();a++) tmpStr += hexTable.value(myStr.at(a));
    while (tmpStr.length()) {
        myReturn += taitAlphabet.value(tmpStr.left(5));
        tmpStr = tmpStr.mid(5);
        if (z==3) myReturn += ".";
        if (z==7) myReturn += ".";
        if (z==11) myReturn += ".";
        if (z==15) myReturn += ".";
        if (z==19) myReturn += ".";
        z++;
    }
    return myReturn;
}

QString MainWindow::taitKeyFromHex(QString myStr) {
    if (myStr.startsWith("f")) myStr = myStr.mid(1);
    myStr = myStr.mid(2);
    myStr.chop(3);
    QString tmpStr = "";
    QString myReturn = "";
    for (int a=0;a<myStr.length();a++) {
        tmpStr += hexTable.value(myStr.at(a));
    }
    tmpStr += "0000";
    int z = 0;
    while (z < 30) {
        myReturn += taitAlphabet.value(tmpStr.left(5));
        tmpStr = tmpStr.mid(5);
        if (z==3) myReturn += ".";
        if (z==7) myReturn += ".";
        if (z==11) myReturn += ".";
        if (z==15) myReturn += ".";
        if (z==19) myReturn += ".";
        z++;
        if (z==22) break;
    }
    return myReturn.trimmed();
}

QString MainWindow::taitWrite(QString myStr,QString myReply) {
    // =====================================================================================================================
    // Generically write a HEX ASCII string to the serial port. Send back any reply to the calling function.
    // The default expected reply is a '>' text character. In Test mode that will be a '-' character.
    // =====================================================================================================================
    QByteArray varSend;
    QDateTime myTime = QDateTime::currentDateTime();
    if (!mySerial.isOpen()) {
        mySerial.setPortName(ui->cmbPorts->currentText());
        mySerial.setBaudRate(QSerialPort::Baud19200);
        mySerial.setDataBits(QSerialPort::Data8);
        mySerial.setStopBits(QSerialPort::OneStop);
        mySerial.setFlowControl(QSerialPort::SoftwareControl);  // Fix for FTDI usb converters.
        mySerial.setParity(QSerialPort::NoParity);
        if (!mySerial.open(QIODevice::ReadWrite)) {
            emit TAIT_RADIO_NOREPLY();
            return "ERROR";
        }
    }
    QString tmpString = "";
    int a = 0;
    while (a < 5) {
        if (myStr!="^") myStr.append("\r");                         // A special character. This is grabbed by the radio without a \r
        varSend.clear();
        varSend.append(myStr);                                      // Prepare the byte array that will be sent
        mySerial.flush();                                           // Clean out any junk in the buffer.
        mySerial.write(varSend);                                    // Send off the data to the radio
        myTime = QDateTime::currentDateTime();                      // Reset the timer.
        sleep(200);
        while (!tmpString.endsWith(myReply)) {
            if (mySerial.bytesAvailable()) {
                tmpString += QString(mySerial.readAll()).trimmed(); myTime = QDateTime::currentDateTime();
            } else {
                sleep(50);
            }
            if (myTime.msecsTo(QDateTime::currentDateTime()) > 10000) break; // Emergency trap.
        }
#ifdef QT_DEBUG
        if (ui->menuDebug->isChecked()) qDebug() << mySerial.portName() << "SENT:" << myStr.trimmed() << "RECV:" << tmpString;
#endif
        if (tmpString.trimmed()!="") a = 10;                        // Break out - the radio said something.
        if (myStr!="^") a = 10;                                     // Break out - the radio said something.
        a++;                                                        // No reply - increment the counter and try again (10 second delay)
    }
    if (tmpString.trimmed()=="") { emit TAIT_RADIO_NOREPLY(); return "ERROR"; }
    if (tmpString.endsWith(myReply)) { tmpString.chop(1); tmpString = tmpString.trimmed(); }
    return tmpString.trimmed();
}

QString MainWindow::taitChecksum(QString myStr) {
    // =====================================================================================================================
    // Code adapted from 8 bit 2's complement converter at http://easyonlineconverter.com/converters/checksum_converter.html
    // =====================================================================================================================
    myStr = myStr.toUpper();
    QString strHex = "0123456789ABCDEF";        // This is quite a bit different to the perl code which does not work when
    uint8_t z;                                     // directly translated across. I suspect this can be cleaned - but is working so not going to rush to change it.
    int mytotal = 0;                            // Must be initialized to 0 or things break.
    int fctr = 16;
    for (int a=0;a<myStr.length();a++) {
        z = strHex.indexOf(myStr.at(a));        // Get a char from QString and store the hex value of the ASCII character.
        mytotal += z * fctr;                    // Modulo 2 - add the new byte to the existing total.
        (fctr == 16) ? fctr = 1 : fctr = 16;
    }
    mytotal &= 0xFF;                            // Only take the least significant 7 bits.
    mytotal = ~mytotal;                         // Invert the bit values
    mytotal += 0x01;                            // Add one to the total. Inversion + 1 = Two Complement.
    mytotal &= 0xFF;                            // Only use the last byte for the checksum.
    return myStr + QString::number(mytotal,16).toUpper().rightJustified(2,'0');
}

void MainWindow::tait_is_now_connected() {
    TAIT_RADIO_CONNECTED = true;
    myStatus->setText("Tait Radio connected on " + mySerial.portName());
}

void MainWindow::tait_is_now_disconnected() {
    myStatus->setText("Tait Radio disconnected from " + mySerial.portName());
    TAIT_RADIO_CONNECTED = false;
    if (mySerial.isOpen()) mySerial.close();
    canExit=true;
}

void MainWindow::tait_inspected() {
    ui->cmbPorts->setEnabled(true);
    ui->butInspect->setEnabled(true);
    ui->butLoadKeys->setEnabled(true);
    ui->butDisableAllKeys->setEnabled(true);
    canExit=true;
}

void MainWindow::tait_no_reply() {
    myStatus->setText("Tait Radio @ " + mySerial.portName() + " did not reply and the serial port was shutdown");
    if (mySerial.isOpen()) mySerial.close();
    TAIT_RADIO_CONNECTED = false;
    ui->cmbPorts->setEnabled(true);
    ui->butInspect->setEnabled(true);
    ui->butLoadKeys->setEnabled(true);
    ui->butDisableAllKeys->setEnabled(true);
    canExit=true;
}

void MainWindow::taitInterrogate() {
    radioInfo.clear();
    resetDisplay();
    myStatus->setText("Interrogating Tait Radio on " + ui->cmbPorts->currentText() + " in test mode for basic information");
    sleep(5000);
    QString myReturn,tmpStrB;
    QStringList varList;
    QStringList myCmds = QString("94 96 97 98 133 93 134 93 203 204").split(" ");
    QDir myDir(QApplication::applicationDirPath() + "/radios/");
    QFile myFile(QApplication::applicationDirPath() + "/radios/0000000.txt");
    foreach(QString varCmd,myCmds) {
        myReturn = taitWrite(varCmd,"-");
        if (myReturn.startsWith("{")) continue;                                                                     // Feature not supported - skip processing.
        if ((varCmd=="133") && myReturn.startsWith("TMAB2")) { myCmds.removeAll("93"); }                            // No return on TM8200's so no point.
        if (varCmd=="93") { radioInfo["ESN"] = myReturn.rightJustified(8,'0'); ui->txtFlash->setText(radioInfo.value("ESN")); }
        if (varCmd=="94") {                                                                                         // Radio Serial Number
            ui->txtChassis->setText(myReturn); radioInfo["CHASSIS"] = myReturn;
            if (!myDir.exists()) myDir.mkdir(QApplication::applicationDirPath() + "/radiodb/");
            myFile.setFileName(QApplication::applicationDirPath() + "/radiodb/" + myReturn + ".txt");           // Lets see if we have seen this radio before
            if (myFile.exists()) {                                                                              // Handy if you got the TEA but forgot to load
                if (myFile.open(QIODevice::ReadOnly | QIODevice::Text)) {                                       // The keys last time, or want to see the raw data
                    while (!myFile.atEnd()) {
                        tmpStrB = QString(myFile.readLine()).trimmed();
                        if (tmpStrB.startsWith("#")) continue;
                        varList = tmpStrB.split("=");
                        if (varList.size()==2) { radioInfo[varList.at(0)] = varList.at(1); }
                    }
                    myFile.close();
                }
            }
        }
        if (varCmd=="96") { radioInfo["BODYFW"] = myReturn.toUpper().trimmed(); }                                   // Body Firmware
        if (varCmd=="97") { radioInfo["BOOTFW"] = myReturn.toUpper().trimmed(); }                                   // Head Firmware
        if (varCmd=="98") { radioInfo["FPGAFW"] = myReturn.toUpper().trimmed(); }                                   // FPGA Firmware
        if (varCmd=="133") {                                                                                        // Model Information
            if (myReturn.at(4)=="2") { ui->txtModel->setText("TM8200 Series"); }
            if (myReturn.at(4)=="3") { ui->txtModel->setText("TM9000 Series"); }
            if (myReturn.at(5)=="1") { ui->txtModel->setText(ui->txtModel->text() + " Conventional Mobile"); }
            if (myReturn.at(5)=="2") { ui->txtModel->setText(ui->txtModel->text() + " Trunking Mobile"); }
            if (myReturn.at(5)=="3") { ui->txtModel->setText(ui->txtModel->text() + " USA Signalling Mobile"); }
            if (myReturn.at(5)=="4") { ui->txtModel->setText(ui->txtModel->text() + " Conventional/Trunking Mobile"); }
            ui->txtBand->setText("UNKNOWN BAND");                                                                   // Frequency Band
            if (taitBands.contains(myReturn.mid(7,2))) ui->txtBand->setText(taitBands.value(myReturn.mid(7,2)));
            radioInfo["BAND"] = ui->txtBand->text();
            radioInfo["MODEL"] = ui->txtModel->text();
            radioInfo["BODY"] = myReturn.left(6);
        }
        if (varCmd=="134") { ui->txtROM->setText(myReturn); radioInfo["ROM"] = myReturn; }
        if (varCmd=="93") { ui->txtFlash->setText(myReturn.rightJustified(8,'0')); radioInfo["ESN"] = ui->txtFlash->text(); }
        if ((varCmd=="203" ) || (varCmd=="204")) {
            varList = myReturn.split("\r");
            foreach(QString tmpStr,varList) {
                if (tmpStr.startsWith("00000120:")) {
                    tmpStrB = tmpStr.mid(9).replace(" ","").trimmed();
                    if (tmpStrB.length()==32) radioInfo["TEA1"] = tmpStrB;
                }
                if (tmpStr.startsWith("00000130:")) {
                    tmpStrB = tmpStr.mid(9).replace(" ","").trimmed();
                    if (tmpStrB.length()==32) radioInfo["TEA2"] = tmpStrB;
                }
            }
        }
    }
    if (radioInfo.contains("TEA1")) {
        ui->txtTEA1->setText(radioInfo.value("TEA1"));
        tmpStrB = radioInfo.value("TEA1").right(8);
        tmpStrB = tmpStrB.right(4) + tmpStrB.left(4);
        bool ok;
        quint32 varESN = tmpStrB.toInt(&ok,16);
        ui->txtFlash->setText(QString::number(varESN,10).rightJustified(8,'0'));
    }
    if (radioInfo.contains("TEA2")) ui->txtTEA2->setText(radioInfo.value("TEA2"));
    if ((ui->txtTEA1->text().length()==32) && (ui->txtTEA2->text().length()==32)) {
        ui->chkLoadable->setChecked(true);
        radioInfo["TEA"] = ui->txtTEA1->text() + ui->txtTEA2->text();
        myStatus->setText("This radio has or has had mnix firmware. SFE keys can be generated.");
    }
    if (radioInfo.contains("BOOTFW")) {
        varList = radioInfo.value("BOOTFW").split(".");
        tmpStrB = varList.at(0);
        radioInfo["BOOTVER"] = (tmpStrB.endsWith("1")) ? "1" : "2";
    }
    if (myFile.exists()) myFile.remove();
    if (myFile.open(QIODevice::ReadWrite | QIODevice::Text)) {
        QTextStream out(&myFile);
        QHashIterator<QString,QString> i(radioInfo);
        while (i.hasNext()) {
            i.next();
            tmpStrB = i.key() + "=" + i.value();
            out << tmpStrB << endl;
        }
        myFile.close();
    }
}

QString MainWindow::taitHexFromKey(QString myKey) {
    // Lets translate a key into its HEX string equivalent. This changes 5 bit ASCII into 4 bit HEX.
    QString tmpStr = myKey.replace(".","").trimmed();
    QString byteStr = "";
    QHashIterator<QString,QString> i(taitAlphabet);
    for(int a=0;a<tmpStr.length();a++) {
        i.toFront();
        while (i.hasNext()) {
            i.next();
            if (i.value()==tmpStr.at(a)) { byteStr.append(i.key()); break; }
        }
    }
    // Okay we now have a binary string. Lets convert that to hex.
    tmpStr.clear();
    for (int a=0;a<byteStr.length();a+=4) {
        tmpStr.append(bitTable.value(byteStr.mid(a,4)));
    }
    tmpStr.append("0");
    tmpStr = tmpStr.rightJustified(30,'0');
    tmpStr = taitChecksum(tmpStr);
    return tmpStr.trimmed();
}

QStringList MainWindow::taitSFETest(QString myKey) { // Hex key - 32 bytes long.
    if (myKey.length()!=32) return QString("ERROR|0|0|00").split("|");
    QStringList myReturn = QString("DISABLED|0|0|00").split("|");
    bool ok = true;
    QString taitKey = taitKeyFromHex(myKey).right(2);
    uint16_t tmpNum = 0;
    for (int a=0;a<32;a++) {
        if (taitAlphabetB[a]==taitKey.at(0)) tmpNum = a;
    }
    for (int a=0;a<32;a++) {
        if (taitAlphabetB[a]==taitKey.at(1)) tmpNum = tmpNum & a;
    }
    int varSeq = myKey.mid(27,2).toInt(&ok,16);
    ok = ((varSeq % 2) == 0); // true = even 0, 2, 4, 6, 8 etc is a disabled feature key.
    myReturn[0] = (ok) ? "DISABLED" : "ENABLED";    // Current status of the key
    myReturn[1] = QString::number(varSeq);          // Current sequence.
    varSeq++;
    if (varSeq > 255) varSeq = 0;
    myReturn[2] = QString::number(varSeq);          // Next in the sequence.
    myReturn[3] = myKey.mid(18,2).rightJustified(2,'0');
    return myReturn;
}
