/*
 * Copyright (C) 2007 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <getopt.h>
#include <limits.h>
#include <linux/input.h>
#include <linux/reboot.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <time.h>
#include <unistd.h>
#include <dirent.h>
#include <syscall.h>

#include "bootloader.h"
#include "common.h"
#include "cutils/properties.h"
#include "install.h"
#include "minui/minui.h"
#include "minzip/DirUtil.h"
#include "roots.h"
#include "recovery_ui.h"

#include "deltaupdate_config.h"

/* 2015/9/22 added by tommy for backup */
#define QUECTEL_BACKUP
#ifdef QUECTEL_BACKUP
#define QUECTEL_RECOVERY_DEBUG	"/cache/quectel_recovery_debug"
#define QUECTEL_SYSTEM_BACKUP  "/res/system-bak"
#define QUECTEL_BACKUP_LOG	"/cache/quectel_backup_log"
#endif

static const deltaupdate_config_st DELTA_UPDATE_STATUS_DB[] = {
        {NO_DELTA_UPDATE, "IP_NO_UPDATE"},
        {START_DELTA_UPDATE, "IP_START_UPDATE"},
        {DELTA_UPDATE_IN_PROGRESS, "IP_PREVIOUS_UPDATE_IN_PROGRESS"},
        {DELTA_UPDATE_SUCCESSFUL, "IP_PREVIOUS_UPDATE_SUCCESSFUL"},
        {DELTA_UPDATE_FAILED, "IP_PREVIOUS_UPDATE_FAILED"}
};

static char diff_pkg_path_name[PATH_MAX];

static const struct option OPTIONS[] = {
  { "send_intent", required_argument, NULL, 's' },
  { "update_package", required_argument, NULL, 'u' },
  { "wipe_data", no_argument, NULL, 'w' },
  { "wipe_cache", no_argument, NULL, 'c' },
  { "show_text", no_argument, NULL, 't' },
  { NULL, 0, NULL, 0 },
};

static const char *COMMAND_FILE = "/cache/recovery/command";
static const char *INTENT_FILE = "/cache/recovery/intent";
static const char *LOG_FILE = "/cache/recovery/log";
static const char *LAST_LOG_FILE = "/cache/recovery/last_log";
static const char *LAST_INSTALL_FILE = "/cache/recovery/last_install";
static const char *CACHE_ROOT = "/cache";
static const char *SDCARD_ROOT = "/sdcard";
static const char *TEMPORARY_LOG_FILE = "/tmp/recovery.log";
static const char *TEMPORARY_INSTALL_FILE = "/tmp/last_install";
static const char *SIDELOAD_TEMP_DIR = "/tmp/sideload";

/*
 * The recovery tool communicates with the main system through /cache files.
 *   /cache/recovery/command - INPUT - command line for tool, one arg per line
 *   /cache/recovery/log - OUTPUT - combined log file from recovery run(s)
 *   /cache/recovery/intent - OUTPUT - intent that was passed in
 *
 * The arguments which may be supplied in the recovery.command file:
 *   --send_intent=anystring - write the text out to recovery.intent
 *   --update_package=path - verify install an OTA package file
 *   --wipe_data - erase user data (and cache), then reboot
 *   --wipe_cache - wipe cache (but not user data), then reboot
 *   --set_encrypted_filesystem=on|off - enables / diasables encrypted fs
 *
 * After completing, we remove /cache/recovery/command and reboot.
 * Arguments may also be supplied in the bootloader control block (BCB).
 * These important scenarios must be safely restartable at any point:
 *
 * FACTORY RESET
 * 1. user selects "factory reset"
 * 2. main system writes "--wipe_data" to /cache/recovery/command
 * 3. main system reboots into recovery
 * 4. get_args() writes BCB with "boot-recovery" and "--wipe_data"
 *    -- after this, rebooting will restart the erase --
 * 5. erase_volume() reformats /data
 * 6. erase_volume() reformats /cache
 * 7. finish_recovery() erases BCB
 *    -- after this, rebooting will restart the main system --
 * 8. main() calls reboot() to boot main system
 *
 * OTA INSTALL
 * 1. main system downloads OTA package to /cache/some-filename.zip
 * 2. main system writes "--update_package=/cache/some-filename.zip"
 * 3. main system reboots into recovery
 * 4. get_args() writes BCB with "boot-recovery" and "--update_package=..."
 *    -- after this, rebooting will attempt to reinstall the update --
 * 5. install_package() attempts to install the update
 *    NOTE: the package install must itself be restartable from any point
 * 6. finish_recovery() erases BCB
 *    -- after this, rebooting will (try to) restart the main system --
 * 7. ** if install failed **
 *    7a. prompt_and_wait() shows an error icon and waits for the user
 *    7b; the user reboots (pulling the battery, etc) into the main system
 * 8. main() calls maybe_install_firmware_update()
 *    ** if the update contained radio/hboot firmware **:
 *    8a. m_i_f_u() writes BCB with "boot-recovery" and "--wipe_cache"
 *        -- after this, rebooting will reformat cache & restart main system --
 *    8b. m_i_f_u() writes firmware image into raw cache partition
 *    8c. m_i_f_u() writes BCB with "update-radio/hboot" and "--wipe_cache"
 *        -- after this, rebooting will attempt to reinstall firmware --
 *    8d. bootloader tries to flash firmware
 *    8e. bootloader writes BCB with "boot-recovery" (keeping "--wipe_cache")
 *        -- after this, rebooting will reformat cache & restart main system --
 *    8f. erase_volume() reformats /cache
 *    8g. finish_recovery() erases BCB
 *        -- after this, rebooting will (try to) restart the main system --
 * 9. main() calls reboot() to boot main system
 */

static const int MAX_ARG_LENGTH = 4096;
static const int MAX_ARGS = 100;

// open a given path, mounting partitions as necessary
FILE*
fopen_path(const char *path, const char *mode) {
    if (ensure_path_mounted(path) != 0) {
        LOGE("Can't mount %s\n", path);
        return NULL;
    }

    // When writing, try to create the containing directory, if necessary.
    // Use generous permissions, the system (init.rc) will reset them.
    if (strchr("wa", mode[0])) dirCreateHierarchy(path, 0777, NULL, 1);

    FILE *fp = fopen(path, mode);
    return fp;
}

// close a file, log an error if the error indicator is set
static void
check_and_fclose(FILE *fp, const char *name) {
    fflush(fp);
    if (ferror(fp)) LOGE("Error in %s\n(%s)\n", name, strerror(errno));
    fclose(fp);
}

// command line args come from, in decreasing precedence:
//   - the actual command line
//   - the bootloader control block (one per line, after "recovery")
//   - the contents of COMMAND_FILE (one per line)
static void
get_args(int *argc, char ***argv) {
    struct bootloader_message boot;
    memset(&boot, 0, sizeof(boot));
    get_bootloader_message(&boot);  // this may fail, leaving a zeroed structure

    if (boot.command[0] != 0 && boot.command[0] != 255) {
        LOGI("Boot command: %.*s\n", sizeof(boot.command), boot.command);
    }

    if (boot.status[0] != 0 && boot.status[0] != 255) {
        LOGI("Boot status: %.*s\n", sizeof(boot.status), boot.status);
    }

    // --- if arguments weren't supplied, look in the bootloader control block
    if (*argc <= 1) {
        boot.recovery[sizeof(boot.recovery) - 1] = '\0';  // Ensure termination
        const char *arg = strtok(boot.recovery, "\n");
        if (arg != NULL && !strcmp(arg, "recovery")) {
            *argv = (char **) malloc(sizeof(char *) * MAX_ARGS);
            (*argv)[0] = strdup(arg);
            for (*argc = 1; *argc < MAX_ARGS; ++*argc) {
                if ((arg = strtok(NULL, "\n")) == NULL) break;
                (*argv)[*argc] = strdup(arg);
            }
            LOGI("Got arguments from boot message\n");
        } else if (boot.recovery[0] != 0 && boot.recovery[0] != 255) {
            LOGE("Bad boot message\n\"%.20s\"\n", boot.recovery);
        }
    }

    // --- if that doesn't work, try the command file
    if (*argc <= 1) {
        FILE *fp = fopen_path(COMMAND_FILE, "r");
        if (fp != NULL) {
            char *argv0 = (*argv)[0];
            *argv = (char **) malloc(sizeof(char *) * MAX_ARGS);
            (*argv)[0] = argv0;  // use the same program name

            char buf[MAX_ARG_LENGTH];
            for (*argc = 1; *argc < MAX_ARGS; ++*argc) {
                if (!fgets(buf, sizeof(buf), fp)) break;
                (*argv)[*argc] = strdup(strtok(buf, "\r\n"));  // Strip newline.
            }

            check_and_fclose(fp, COMMAND_FILE);
            LOGI("Got arguments from %s\n", COMMAND_FILE);
        }
    }

    // --> write the arguments we have back into the bootloader control block
    // always boot into recovery after this (until finish_recovery() is called)
    strlcpy(boot.command, "boot-recovery", sizeof(boot.command));
    strlcpy(boot.recovery, "recovery\n", sizeof(boot.recovery));
    int i;
    for (i = 1; i < *argc; ++i) {
        strlcat(boot.recovery, (*argv)[i], sizeof(boot.recovery));
        strlcat(boot.recovery, "\n", sizeof(boot.recovery));
    }
    set_bootloader_message(&boot);
}

static void
set_sdcard_update_bootloader_message() {
    struct bootloader_message boot;
    memset(&boot, 0, sizeof(boot));
    strlcpy(boot.command, "boot-recovery", sizeof(boot.command));
    strlcpy(boot.recovery, "recovery\n", sizeof(boot.recovery));
    set_bootloader_message(&boot);
}

// How much of the temp log we have copied to the copy in cache.
static long tmplog_offset = 0;

static void
copy_log_file(const char* source, const char* destination, int append) {
    FILE *log = fopen_path(destination, append ? "a" : "w");
    if (log == NULL) {
        LOGE("Can't open %s\n", destination);
    } else {
        FILE *tmplog = fopen(source, "r");
        if (tmplog != NULL) {
            if (append) {
                fseek(tmplog, tmplog_offset, SEEK_SET);  // Since last write
            }
            char buf[4096];
            while (fgets(buf, sizeof(buf), tmplog)) fputs(buf, log);
            if (append) {
                tmplog_offset = ftell(tmplog);
            }
            check_and_fclose(tmplog, source);
        }
        check_and_fclose(log, destination);
    }
}


// clear the recovery command and prepare to boot a (hopefully working) system,
// copy our log file to cache as well (for the system to read), and
// record any intent we were asked to communicate back to the system.
// this function is idempotent: call it as many times as you like.
static void
finish_recovery(const char *send_intent) {
    // By this point, we're ready to return to the main system...
    if (send_intent != NULL) {
        FILE *fp = fopen_path(INTENT_FILE, "w");
        if (fp == NULL) {
            LOGE("Can't open %s\n", INTENT_FILE);
        } else {
            fputs(send_intent, fp);
            check_and_fclose(fp, INTENT_FILE);
        }
    }

    // Copy logs to cache so the system can find out what happened.
    copy_log_file(TEMPORARY_LOG_FILE, LOG_FILE, true);
    copy_log_file(TEMPORARY_LOG_FILE, LAST_LOG_FILE, false);
    unlink(TEMPORARY_LOG_FILE);
    copy_log_file(TEMPORARY_INSTALL_FILE, LAST_INSTALL_FILE, false);
    unlink(TEMPORARY_INSTALL_FILE);
    chmod(LOG_FILE, 0600);
    chown(LOG_FILE, 1000, 1000);   // system user
    chmod(LAST_LOG_FILE, 0640);
    chmod(LAST_INSTALL_FILE, 0644);

    // Reset to mormal system boot so recovery won't cycle indefinitely.
    struct bootloader_message boot;
    memset(&boot, 0, sizeof(boot));
    set_bootloader_message(&boot);

    // Remove the command file, so recovery won't repeat indefinitely.
    if (ensure_path_mounted(COMMAND_FILE) != 0 ||
        (unlink(COMMAND_FILE) && errno != ENOENT)) {
        LOGW("Can't unlink %s\n", COMMAND_FILE);
    }

    ensure_path_unmounted(CACHE_ROOT);
    sync();  // For good measure.
}

static int
erase_volume(const char *volume) {
    ui_set_background(BACKGROUND_ICON_INSTALLING);
    ui_show_indeterminate_progress();
    ui_print("Formatting %s...\n", volume);

    ensure_path_unmounted(volume);

    if (strcmp(volume, "/cache") == 0) {
        // Any part of the log we'd copied to cache is now gone.
        // Reset the pointer so we copy from the beginning of the temp
        // log.
        tmplog_offset = 0;
    }

    return format_volume(volume);
}

static char*
copy_sideloaded_package(const char* original_path) {
  if (ensure_path_mounted(original_path) != 0) {
    LOGE("Can't mount %s\n", original_path);
    return NULL;
  }

  if (ensure_path_mounted(SIDELOAD_TEMP_DIR) != 0) {
    LOGE("Can't mount %s\n", SIDELOAD_TEMP_DIR);
    return NULL;
  }

  if (mkdir(SIDELOAD_TEMP_DIR, 0700) != 0) {
    if (errno != EEXIST) {
      LOGE("Can't mkdir %s (%s)\n", SIDELOAD_TEMP_DIR, strerror(errno));
      return NULL;
    }
  }

  // verify that SIDELOAD_TEMP_DIR is exactly what we expect: a
  // directory, owned by root, readable and writable only by root.
  struct stat st;
  if (stat(SIDELOAD_TEMP_DIR, &st) != 0) {
    LOGE("failed to stat %s (%s)\n", SIDELOAD_TEMP_DIR, strerror(errno));
    return NULL;
  }
  if (!S_ISDIR(st.st_mode)) {
    LOGE("%s isn't a directory\n", SIDELOAD_TEMP_DIR);
    return NULL;
  }
  if ((st.st_mode & 0777) != 0700) {
    LOGE("%s has perms %o\n", SIDELOAD_TEMP_DIR, st.st_mode);
    return NULL;
  }
  if (st.st_uid != 0) {
    LOGE("%s owned by %lu; not root\n", SIDELOAD_TEMP_DIR, st.st_uid);
    return NULL;
  }

  char copy_path[PATH_MAX];
  strcpy(copy_path, SIDELOAD_TEMP_DIR);
  strcat(copy_path, "/package.zip");

  char* buffer = malloc(BUFSIZ);
  if (buffer == NULL) {
    LOGE("Failed to allocate buffer\n");
    return NULL;
  }

  size_t read;
  FILE* fin = fopen(original_path, "rb");
  if (fin == NULL) {
    LOGE("Failed to open %s (%s)\n", original_path, strerror(errno));
    return NULL;
  }
  FILE* fout = fopen(copy_path, "wb");
  if (fout == NULL) {
    LOGE("Failed to open %s (%s)\n", copy_path, strerror(errno));
    return NULL;
  }

  while ((read = fread(buffer, 1, BUFSIZ, fin)) > 0) {
    if (fwrite(buffer, 1, read, fout) != read) {
      LOGE("Short write of %s (%s)\n", copy_path, strerror(errno));
      return NULL;
    }
  }

  free(buffer);

  if (fclose(fout) != 0) {
    LOGE("Failed to close %s (%s)\n", copy_path, strerror(errno));
    return NULL;
  }

  if (fclose(fin) != 0) {
    LOGE("Failed to close %s (%s)\n", original_path, strerror(errno));
    return NULL;
  }

  // "adb push" is happy to overwrite read-only files when it's
  // running as root, but we'll try anyway.
  if (chmod(copy_path, 0400) != 0) {
    LOGE("Failed to chmod %s (%s)\n", copy_path, strerror(errno));
    return NULL;
  }

  return strdup(copy_path);
}

static char**
prepend_title(const char** headers) {
    char* title[] = { "Android system recovery <"
                          EXPAND(RECOVERY_API_VERSION) "e>",
                      "",
                      NULL };

    // count the number of lines in our title, plus the
    // caller-provided headers.
    int count = 0;
    char** p;
    for (p = title; *p; ++p, ++count);
    for (p = headers; *p; ++p, ++count);

    char** new_headers = malloc((count+1) * sizeof(char*));
    char** h = new_headers;
    for (p = title; *p; ++p, ++h) *h = *p;
    for (p = headers; *p; ++p, ++h) *h = *p;
    *h = NULL;

    return new_headers;
}

static int
get_menu_selection(char** headers, char** items, int menu_only,
                   int initial_selection) {
    // throw away keys pressed previously, so user doesn't
    // accidentally trigger menu items.
    ui_clear_key_queue();

    ui_start_menu(headers, items, initial_selection);
    int selected = initial_selection;
    int chosen_item = -1;

    while (chosen_item < 0) {
        int key = ui_wait_key();
        int visible = ui_text_visible();

        if (key == -1) {   // ui_wait_key() timed out
            if (ui_text_ever_visible()) {
                continue;
            } else {
                LOGI("timed out waiting for key input; rebooting.\n");
                ui_end_menu();
                return ITEM_REBOOT;
            }
        }

        int action = device_handle_key(key, visible);

        if (action < 0) {
            switch (action) {
                case HIGHLIGHT_UP:
                    --selected;
                    selected = ui_menu_select(selected);
                    break;
                case HIGHLIGHT_DOWN:
                    ++selected;
                    selected = ui_menu_select(selected);
                    break;
                case SELECT_ITEM:
                    chosen_item = selected;
                    break;
                case NO_ACTION:
                    break;
            }
        } else if (!menu_only) {
            chosen_item = action;
        }
    }

    ui_end_menu();
    return chosen_item;
}

static int compare_string(const void* a, const void* b) {
    return strcmp(*(const char**)a, *(const char**)b);
}

static int
update_directory(const char* path, const char* unmount_when_done,
                 int* wipe_cache) {
    ensure_path_mounted(path);

    const char* MENU_HEADERS[] = { "Choose a package to install:",
                                   path,
                                   "",
                                   NULL };
    DIR* d;
    struct dirent* de;
    d = opendir(path);
    if (d == NULL) {
        LOGE("error opening %s: %s\n", path, strerror(errno));
        if (unmount_when_done != NULL) {
            ensure_path_unmounted(unmount_when_done);
        }
        return 0;
    }

    char** headers = prepend_title(MENU_HEADERS);

    int d_size = 0;
    int d_alloc = 10;
    char** dirs = malloc(d_alloc * sizeof(char*));
    int z_size = 1;
    int z_alloc = 10;
    char** zips = malloc(z_alloc * sizeof(char*));
    zips[0] = strdup("../");

    while ((de = readdir(d)) != NULL) {
        int name_len = strlen(de->d_name);

        if (de->d_type == DT_DIR) {
            // skip "." and ".." entries
            if (name_len == 1 && de->d_name[0] == '.') continue;
            if (name_len == 2 && de->d_name[0] == '.' &&
                de->d_name[1] == '.') continue;

            if (d_size >= d_alloc) {
                d_alloc *= 2;
                dirs = realloc(dirs, d_alloc * sizeof(char*));
            }
            dirs[d_size] = malloc(name_len + 2);
            strcpy(dirs[d_size], de->d_name);
            dirs[d_size][name_len] = '/';
            dirs[d_size][name_len+1] = '\0';
            ++d_size;
        } else if (de->d_type == DT_REG &&
                   name_len >= 4 &&
                   strncasecmp(de->d_name + (name_len-4), ".zip", 4) == 0) {
            if (z_size >= z_alloc) {
                z_alloc *= 2;
                zips = realloc(zips, z_alloc * sizeof(char*));
            }
            zips[z_size++] = strdup(de->d_name);
        }
    }
    closedir(d);

    qsort(dirs, d_size, sizeof(char*), compare_string);
    qsort(zips, z_size, sizeof(char*), compare_string);

    // append dirs to the zips list
    if (d_size + z_size + 1 > z_alloc) {
        z_alloc = d_size + z_size + 1;
        zips = realloc(zips, z_alloc * sizeof(char*));
    }
    memcpy(zips + z_size, dirs, d_size * sizeof(char*));
    free(dirs);
    z_size += d_size;
    zips[z_size] = NULL;

    int result;
    int chosen_item = 0;
    do {
        chosen_item = get_menu_selection(headers, zips, 1, chosen_item);

        char* item = zips[chosen_item];
        int item_len = strlen(item);
        if (chosen_item == 0) {          // item 0 is always "../"
            // go up but continue browsing (if the caller is update_directory)
            result = -1;
            break;
        } else if (item[item_len-1] == '/') {
            // recurse down into a subdirectory
            char new_path[PATH_MAX];
            strlcpy(new_path, path, PATH_MAX);
            strlcat(new_path, "/", PATH_MAX);
            strlcat(new_path, item, PATH_MAX);
            new_path[strlen(new_path)-1] = '\0';  // truncate the trailing '/'
            result = update_directory(new_path, unmount_when_done, wipe_cache);
            if (result >= 0) break;
        } else {
            // selected a zip file:  attempt to install it, and return
            // the status to the caller.
            char new_path[PATH_MAX];
            strlcpy(new_path, path, PATH_MAX);
            strlcat(new_path, "/", PATH_MAX);
            strlcat(new_path, item, PATH_MAX);

            ui_print("\n-- Install %s ...\n", path);
            set_sdcard_update_bootloader_message();
            char* copy = copy_sideloaded_package(new_path);
            if (unmount_when_done != NULL) {
                ensure_path_unmounted(unmount_when_done);
            }
            if (copy) {
                result = install_package(copy, wipe_cache, TEMPORARY_INSTALL_FILE);
                free(copy);
            } else {
                result = INSTALL_ERROR;
            }
            break;
        }
    } while (true);

    int i;
    for (i = 0; i < z_size; ++i) free(zips[i]);
    free(zips);
    free(headers);

    if (unmount_when_done != NULL) {
        ensure_path_unmounted(unmount_when_done);
    }
    return result;
}

static void
wipe_data(int confirm) {
    if (confirm) {
        static char** title_headers = NULL;

        if (title_headers == NULL) {
            char* headers[] = { "Confirm wipe of all user data?",
                                "  THIS CAN NOT BE UNDONE.",
                                "",
                                NULL };
            title_headers = prepend_title((const char**)headers);
        }

        char* items[] = { " No",
                          " No",
                          " No",
                          " No",
                          " No",
                          " No",
                          " No",
                          " Yes -- delete all user data",   // [7]
                          " No",
                          " No",
                          " No",
                          NULL };

        int chosen_item = get_menu_selection(title_headers, items, 1, 0);
        if (chosen_item != 7) {
            return;
        }
    }

    ui_print("\n-- Wiping data...\n");
    erase_volume("/data");
    erase_volume("/cache");
    ui_print("Data wipe complete.\n");
}

static void
print_property(const char *key, const char *name, void *cookie) {
    printf("%s=%s\n", key, name);
}

static const char* skip_whitespaces(const char* ptr)
{
    while(*ptr != '\0')
    {
        if(*ptr == 0x09 /* Tab */
            || *ptr == 0x0A /* LF */
            || *ptr == 0x0D /* CR */
            || *ptr == 0x20 /* SP */
            )
            ptr++;
        else
            return ptr;
    }
    return NULL;
}

static char* trim_whitespace(char *str)
{
    char *end;

    // Trim leading space
    while(isspace(*str)) str++;

    if(*str == 0)
        return str;

    // Trim trailing space
    end = str + strlen(str) - 1;
    while(end > str && isspace(*end)) end--;

    // Write new null terminator
    *(end+1) = 0;

    return str;
}

static int deltaupdate_pkg_location(char* diff_pkg_path_name)
{
    static char line[MAX_STRING_LEN];
    struct stat status;
    FILE* fp;
    char* tmp_str, saveptr;
    bool found = false;
    int i = 0;

    while (i++ < 3)
    {
        LOGI("fopen_path %d %s\n", i, FOTA_PROP_FILE);
        fp = fopen_path(FOTA_PROP_FILE, "r");
        if (fp)
            break;
        sleep(1);
    }
    if (fp == NULL)
    {
        LOGI("Failed to open %s, use default pkg location:%s\n",
             FOTA_PROP_FILE, DEFAULT_PKG_LOCATION);
        strlcpy(diff_pkg_path_name, DEFAULT_PKG_LOCATION, PATH_MAX);
    }
    else
    {
        while(fgets(line, MAX_STRING_LEN, fp)!=NULL)
        {
            tmp_str = strtok_r(line, "=", &saveptr);
            if(strcmp(tmp_str, PKG_LOCATION_STRING_NAME) == 0)
            {
               tmp_str = strtok_r(NULL, "=\n", &saveptr);
               strlcpy(diff_pkg_path_name, tmp_str, PATH_MAX);
               diff_pkg_path_name = trim_whitespace(diff_pkg_path_name);
               LOGI("Package location: %s\r\n", diff_pkg_path_name);
               found = true;
               break;
            }
        }
        if (!found)
        {
            LOGI("Package location is not defined in %s. Use default location: %s\n",
            FOTA_PROP_FILE, DEFAULT_PKG_LOCATION );
            strlcpy(diff_pkg_path_name, DEFAULT_PKG_LOCATION, PATH_MAX);
        }
        fclose(fp);
    }

    if (ensure_path_mounted(diff_pkg_path_name) != 0) {
        LOGI("Cannot mount %s\n", diff_pkg_path_name);
        reset_fota_cookie_mtd();
        return -1;
    }

    strlcat(diff_pkg_path_name, "/", PATH_MAX);
    strlcat(diff_pkg_path_name, DIFF_PACKAGE_NAME, PATH_MAX);

    if (access(diff_pkg_path_name, F_OK) != 0) {
        LOGI("Delta package does not exist %s\n", diff_pkg_path_name);
        reset_fota_cookie_mtd();
        return -1;
    }

    LOGI("Delta package path name: %s\n", diff_pkg_path_name);

    return 0;
}

static int get_deltaupdate_recoverycount(void)
{
    FILE* f;
    int len, num;
    char* buf;

    f = fopen_path(NUM_OF_RECOVERY, "r");
    if(f == NULL)
    {
       LOGI("Error opening recovery count file. Ignore.\n");
       return 0;
    }

    fseek(f, 0, SEEK_END);
    len = ftell(f);
    buf = malloc(len+1);
    if (buf == NULL) {
        LOGI("Failed to allocate buffer\n");
        return 0;
    }
    memset(buf,0x0,len+1);
    fseek(f, 0, SEEK_SET);
    fread(buf, sizeof(char), len, f);
    check_and_fclose(f,NUM_OF_RECOVERY);

    if ((buf = strstr((const char*)buf, "numRecovery")) == NULL)
    {
        LOGI("Recovery count string doesn't match.Ignore.\n");
    }
    else
    {
        buf += 11;
        if ((buf = strstr((const char*)buf, "=")) == NULL)
        {
            LOGI("Invalid recovery count value. Ignore.\n");
        }
        else
        {
            buf += 1;
            buf = (char*)skip_whitespaces((const char*)buf);
            num = atoi((const char*)buf);
            return num;
        }
    }
    return 0;
}

static int get_deltaupdate_status(void)
{
    FILE* f;
    int len;
    char* buf;
    int i, num_index;

    LOGI("Checking delta update status...\n");

    f = fopen_path(DELTA_UPDATE_STATUS_FILE, "r");
    if(f == NULL)
    {
       LOGI("fopen error(%s)\n",DELTA_UPDATE_STATUS_FILE);
       return -1;
    }

    fseek(f, 0, SEEK_END);
    len = ftell(f);
    buf = malloc(len+1);
    if (buf == NULL) {
        LOGI("Failed to allocate buffer\n");
        return -1;
    }
    memset(buf,0x0,len+1);
    fseek(f, 0, SEEK_SET);
    fread(buf, sizeof(char), len, f);
    check_and_fclose(f,DELTA_UPDATE_STATUS_FILE);

    num_index = sizeof(DELTA_UPDATE_STATUS_DB)/sizeof(deltaupdate_config_st);

    for (i = 0; i < num_index; i++)
    {
        if (strstr((const char*)buf, DELTA_UPDATE_STATUS_DB[i].str)!=NULL)
        {
            return DELTA_UPDATE_STATUS_DB[i].idx;
        }
    }

    LOGI("NO UPDATE SET\n");
    return NO_DELTA_UPDATE;
}

static int set_deltaupdate_status(int status, int error_code)
{
    FILE* f;
    char strbuf[64];

    LOGI("Setting delta update status...\n");

    f = fopen_path(DELTA_UPDATE_STATUS_FILE, "w");
    if(f == NULL)
    {
       LOGI("fopen error(%s)\n",DELTA_UPDATE_STATUS_FILE);
       return -1;
    }

    switch(status)
    {
        case START_DELTA_UPDATE:
        case DELTA_UPDATE_IN_PROGRESS:
        case DELTA_UPDATE_SUCCESSFUL:
        case DELTA_UPDATE_FAILED:
            if ((snprintf(strbuf, sizeof(strbuf), "%s %d", DELTA_UPDATE_STATUS_DB[status].str, error_code)) >= sizeof(strbuf)) {
               LOGI("Output Truncated while setting error code\n");
            }
            fwrite(strbuf, sizeof(char), strlen(strbuf), f);
            break;
        default:
            if ((snprintf(strbuf, sizeof(strbuf), "DELTA_NO_UPDATE %d", error_code)) >= sizeof(strbuf)) {
               LOGI("Output Truncated while setting error code\n");
            }
            fwrite(strbuf, sizeof(char), strlen(strbuf), f);
            break;
    }

    LOGI("Delta update status is set to (%s)\n",strbuf);
    check_and_fclose(f,DELTA_UPDATE_STATUS_FILE);
    return 0;
}

static void set_deltaupdate_recovery_bootmessage(void)
{
    struct bootloader_message boot;
    memset(&boot, 0, sizeof(boot));

    LOGI("Setting recovery boot...\n");

    if(MAX_NUM_UPDATE_RECOVERY > get_deltaupdate_recoverycount())
    {
       strlcpy(boot.command, "boot-recovery", sizeof(boot.command));
       strlcpy(boot.recovery, "recovery\n", sizeof(boot.recovery));
    }
    else
    {
       LOGI("Recovery mode reached maximum retry. Clear boot message.\n");
    }
    set_bootloader_message(&boot);

    LOGI("boot.command=%s\n",boot.command);
    LOGI("boot.recovery=%s\n",boot.recovery);
}

static void reset_deltaupdate_recovery_bootmessage(void)
{
    struct bootloader_message boot;

    memset(&boot, 0, sizeof(boot));

    LOGI("Resetting recovery boot...\n");

    set_bootloader_message(&boot);

    LOGI("boot.command=%s\n",boot.command);
    LOGI("boot.recovery=%s\n",boot.recovery);
}

static void increment_deltaupdate_recoverycount(void)
{
    FILE* f;
    int num;
    char numbuf[8];
    char strbuf[32];

    num = get_deltaupdate_recoverycount();
    num += 1;
    snprintf(numbuf, sizeof(numbuf), "%d", num);

    memset(strbuf,0x0,sizeof(strbuf));
    strlcpy(strbuf,"numRecovery=",sizeof(strbuf));
    strlcat(strbuf,numbuf,sizeof(strbuf));

    f = fopen_path(NUM_OF_RECOVERY, "w");
    if(f == NULL)
    {
       LOGI("Error Creating file %s\n",NUM_OF_RECOVERY);
       return;
    }
    fwrite(strbuf, sizeof(char), strlen(strbuf), f);
    check_and_fclose(f, NUM_OF_RECOVERY);
}

static int remove_tempfiles(char* diff_pkg_path_name)
{
   if (unlink(diff_pkg_path_name) && errno != ENOENT)
        LOGI("Cannot unlink %s\n", diff_pkg_path_name);
   if (unlink(NUM_OF_RECOVERY) && errno != ENOENT) 
        LOGI("Cannot unlink %s\n", NUM_OF_RECOVERY);
   return 0;
}

static int read_buildprop(char **ver)
{
    FILE* b_fp;
    char line[MAX_STRING_LEN];
    char *tmpStr, *saveptr;

    LOGI("read_buildprop.\n");

    b_fp = fopen_path(BUILD_PROP_FILE, "r");
    if(!b_fp)
        return -1;

    while(fgets(line, sizeof(line), b_fp))
    {
        tmpStr = strtok_r(line, "=", &saveptr);
        if(strcmp(tmpStr, BUILD_PROP_NAME) == 0)
        {
           tmpStr = strtok_r(NULL, "=", &saveptr);
           strlcpy(*ver, tmpStr, MAX_STRING_LEN);
           fclose(b_fp);
           return 0;
        }
    }
    fclose(b_fp);
    return -1;

}

static char *delta_update_replace_str(char *str, char *org, char *rep)
{
    static char buffer[MAX_STRING_LEN];
    char *p;

    if(!(p = strstr(str, org)))
       return str;

    if ((strlcpy(buffer, str, MAX_STRING_LEN)) >= MAX_STRING_LEN) {
        LOGI("Version Update string truncated\n");
        return NULL;
    }
    buffer[p-str] = '\0';

    strlcat(buffer, rep, MAX_STRING_LEN);
    strlcat(buffer, p + strlen(org), MAX_STRING_LEN);

    return buffer;
}

static int update_fotapropver(char *ver)
{
    int size;
    FILE *b_fp;
    char *buff;
    char *newbuff;
    char *orgstr=NULL;
    char newstr[MAX_STRING_LEN];
    char line[MAX_STRING_LEN];

    LOGI("update_ver:%s\r\n",ver);

    b_fp = fopen_path(FOTA_PROP_FILE, "r");
    if(!b_fp)
       return -1;

    //Read Old Version
    while(fgets(line, sizeof(line), b_fp))
    {
        orgstr = strstr(line, VERSION_STRING_NAME);
        if(orgstr)
        {
           break;
        }
    }

    if(orgstr == NULL)
    {
       LOGI("No firmware property.\r\n");
       return -1;
    }

    //Build New Version
    snprintf(newstr, MAX_STRING_LEN, "%s=%s",VERSION_STRING_NAME, ver);

    //Read Org File
    fseek(b_fp, 0, SEEK_END);
    size = ftell(b_fp);
    buff = (char*)malloc(size+1);
    if (buff == NULL) {
        LOGI("Failed to allocate buffer\n");
        return -1;
    }
    memset(buff, 0x0, size);

    //Update Version
    fseek(b_fp, 0, SEEK_SET);
    fread(buff, sizeof(char), size, b_fp);
    fclose(b_fp);
    buff[size] = '\0';
    newbuff = delta_update_replace_str(buff, orgstr, newstr);

    if (newbuff) {
        b_fp = fopen_path(FOTA_PROP_FILE, "w+");
        fwrite(newbuff, sizeof(char), strlen(newbuff), b_fp);
        fclose(b_fp);
        return 0;
    }

    return -1;
}

static int update_fotaprop(void)
{
    int ret;
    char *ver;

    ui_print("update_fotaprop.\n");

    ver = (char *)malloc(MAX_STRING_LEN);
    if (ver == NULL) {
        LOGI("Failed to allocate buffer\n");
        return 0;
    }
    memset(ver, 0x0, MAX_STRING_LEN);
    if(ensure_path_mounted(FOTA_PROP_FILE) != 0)
        LOGI("failed to locate fota prop file \n");
    if(ensure_path_mounted(BUILD_PROP_PATH) != 0)
        LOGI("failed to locate build.prop \n");
    ret = read_buildprop(&ver);
    if(ret != 0)
    {
       LOGI("Failed reading build version.\n");
       return -1;
    }
    LOGI("Found build version:%s\n",ver);

    ret = update_fotapropver(ver);
    if(ret != 0)
    {
       LOGI("Failed update version.\n");
       return -1;
    }

    return 0;
}

int start_deltaupdate(char* diff_pkg_path_name)
{
    int status;
    int wipe_cache = 0;

    LOGI("Start delta update...\n");

    set_deltaupdate_recovery_bootmessage();

    status = install_package(diff_pkg_path_name, &wipe_cache, TEMPORARY_INSTALL_FILE);
    if (status != INSTALL_SUCCESS)
    {
        ui_set_background(BACKGROUND_ICON_ERROR);
        ui_print("Delta update failed.\n");
        finish_recovery("--send_intent=DELTA_UPDATE_FAILED");
        set_deltaupdate_status(DELTA_UPDATE_FAILED, DELTA_UPDATE_FAILED_410);
        reset_fota_cookie_mtd();
        return -1;
    }

    // modem update starts only if android update is successful
    status = start_delta_image_update(diff_pkg_path_name);
    reset_fota_cookie_mtd();

    // modem update is complete. Handle update result.
    if (status != DELTA_UPDATE_SUCCESS_200)
    {
        ui_set_background(BACKGROUND_ICON_ERROR);
        ui_print("Delta update failed(%d)\n",status);
        finish_recovery("--send_intent=DELTA_UPDATE_FAILED");
        set_deltaupdate_status(DELTA_UPDATE_FAILED, DELTA_UPDATE_FAILED_410);
        return -1;
    }

    // file system update starts only if android update is successful
    status = start_delta_fs_update(diff_pkg_path_name);
    reset_fota_cookie_mtd();
    // file system update is complete. Handle update result.
    if (status != DELTA_UPDATE_SUCCESS_200)
    {
        //ui_set_background(BACKGROUND_ICON_ERROR);
        //ui_print("Delta update failed(%d)\n",status);
        finish_recovery("--send_intent=DELTA_UPDATE_FAILED");
        set_deltaupdate_status(DELTA_UPDATE_FAILED, DELTA_UPDATE_FAILED_500);
        return -1;
    }

    finish_recovery("--send_intent=DELTA_UPDATE_SUCCESSFUL");
    set_deltaupdate_status(DELTA_UPDATE_SUCCESSFUL, DELTA_UPDATE_SUCCESS_200);

    ui_print("\n Delta Update Completed \n");

    // Remove all temp files
    remove_tempfiles(diff_pkg_path_name);
    update_fotaprop();
    return 0;
}

/* FOTA(Delta Update) INSTALL
 * 1. main system downloads delta update package to location specified in
 *    FOTA_PROP_FILE if it exists.
 *    -- Otherwise, downloads into default package location -
 *    cache/fota/DIFF_PACKAGE_NAME
 * 2. main system reboots into recovery
 * 3. get_args() writes BCB with "boot-recovery"
 *    -- after this, fota cookie is set to enable modem image update --
 *    -- rebooting into recovery to start android update --
 * 4. main system reboots into recovery
 * 5. get_args() writes BCB with "boot-recovery"
 * 6. install_package() attempts to install android delta update
 *    NOTE: the package install must itself be restartable from any point
 * 7. If update succeeds, calls start_delta_modemupdate() to begin
 *    modem update.
 *    NOTE: the package install must itself be restartable from any point
 * 8. If update succeeds, reset fota cookie.
 * 9. finish_recovery() erases BCB
 *    -- after this, rebooting will (try to) restart the main system --
 * 10. ** if install failed **
 *    10a. Show error icon, reset fota cookie.
 *    10b. finish_recovery() erases BCB
 *    -- after this, rebooting will (try to) restart the main system --
 * 11. handle_deltaupdate_status() calls reboot() to boot main system
 */
static int handle_deltaupdate_status(void)
{
    int update_status;
    struct stat status;

    if (deltaupdate_pkg_location(diff_pkg_path_name) == -1 )
    {
        return -1;
    }

    //Increment count that indicates number of times device enters into recovery
    //during delta update. This prevents the device recycling endlessly in recovery mode.
    increment_deltaupdate_recoverycount();

    update_status = get_deltaupdate_status();
    LOGI("update_status = %d\n", update_status);

    switch(update_status)
    {
    case START_DELTA_UPDATE:
          set_deltaupdate_status(DELTA_UPDATE_IN_PROGRESS, 0);

    case DELTA_UPDATE_IN_PROGRESS:
          if (MAX_NUM_UPDATE_RECOVERY < get_deltaupdate_recoverycount()){
             reset_deltaupdate_recovery_bootmessage();
             reset_fota_cookie_mtd();
          }
          start_deltaupdate(diff_pkg_path_name);
          break;

    default:
          LOGI("No update set\n");
          if (MAX_NUM_UPDATE_RECOVERY < get_deltaupdate_recoverycount()){
             reset_deltaupdate_recovery_bootmessage();
             reset_fota_cookie_mtd();
          }
          return EXIT_SUCCESS;
    }
    sync();
    LOGI("Rebooting after recovery\n");
    syscall(SYS_reboot, LINUX_REBOOT_MAGIC1, LINUX_REBOOT_MAGIC2, LINUX_REBOOT_CMD_RESTART, NULL);
    return EXIT_SUCCESS;
}

#ifdef EMBED_UPDATE_BINARY

#define MSM_FOTA_FILE	"/cache/update.zip"
extern void urc_to_ttygs_modem(char *buf);
extern int quec_get_update_result(unsigned int *buf);
static int handle_redbend_update()
{
    int update_status;
    struct stat status;
    unsigned int quec_state[5];

#if 1
	system("mount");
	system("ls -l /usr/bin/usb");
	system("ls -l /dev/ttyGS*");
	//urc_to_ttygs_modem("+QIND: \"FOTA\",\"START\"\r\n");//2015/11/26 deled by tommy
#endif

    if (access(MSM_FOTA_FILE, F_OK) != 0)
	{
		printf("Delta package does not exist %s\n", MSM_FOTA_FILE);
		reset_fota_cookie_mtd();
		return -1;
	}

    increment_deltaupdate_recoverycount();
	if (MAX_NUM_UPDATE_RECOVERY < get_deltaupdate_recoverycount())
	{
		system("rm -rf /cache/fota /cache/recovery "MSM_FOTA_FILE);
		reset_deltaupdate_recovery_bootmessage();
		reset_fota_cookie_mtd();
		return EXIT_SUCCESS;
	}

    urc_to_ttygs_modem("+QIND: \"FOTA\",\"START\"\r\n");/* 2015/11/26 added by tommy */
    update_status = get_deltaupdate_status();
    switch(update_status)
    {
    case START_DELTA_UPDATE:
			printf("%s: START_DELTA_UPDATE\n", __func__);
			set_deltaupdate_status(DELTA_UPDATE_IN_PROGRESS, 0);
			start_deltaupdate(MSM_FOTA_FILE);
			//break;/* 2015/11/26 added by tommy */
    case DELTA_UPDATE_IN_PROGRESS:
			printf("%s: DELTA_UPDATE_IN_PROGRESS\n", __func__);
			set_deltaupdate_status(DELTA_UPDATE_IN_PROGRESS, 0);
			start_deltaupdate(MSM_FOTA_FILE);
          break;

    default:
          printf("No update set\n");
          return EXIT_SUCCESS;
    }
    sync();
	if(!quec_get_update_result(quec_state) && 100 == quec_state[0] && quec_state[1]==quec_state[2])
		urc_to_ttygs_modem("+QIND: \"FOTA\",\"END\",0\r\n");
	else{
		urc_to_ttygs_modem("+QIND: \"FOTA\",\"END\",501\r\n");
		printf("precent:%d total:%d cur:%d\n",quec_state[0],quec_state[1],quec_state[2]);
	}
    printf("Rebooting after recovery\n");
    syscall(SYS_reboot, LINUX_REBOOT_MAGIC1, LINUX_REBOOT_MAGIC2, LINUX_REBOOT_CMD_RESTART, NULL);
    return EXIT_SUCCESS;

}
#endif

int
main(int argc, char **argv) {
    time_t start = time(NULL);
    //freopen(TEMPORARY_LOG_FILE, "a", stdout);
    //freopen(TEMPORARY_LOG_FILE, "a", stderr);
    // Disable stream buffering (note that /tmp is a ramdisk)
    //setvbuf(stdout, NULL, _IONBF, 0);
    //setvbuf(stderr, NULL, _IONBF, 0);

    system("mount -t debugfs none /sys/kernel/debug");

#ifdef QUECTEL_BACKUP
	system("echo 1 > /sys/module/qstart/parameters/start_mode");
#endif

	sleep(5);
#ifdef QUECTEL_BACKUP
	if(access(QUECTEL_RECOVERY_DEBUG, F_OK) == 0)  //debug mode
	{
		system("rm -f "QUECTEL_RECOVERY_DEBUG);
		return 0;
	}
	if(access(MSM_FOTA_FILE, F_OK) != 0)  //backup mode
	{
		//system("mount");
		//system("ls -l /usr/bin/usb");
		//system("ls -l /dev/ttyGS*");
		urc_to_ttygs_modem("+QIND: \"RECOVERY\",\"START\"\r\n");
    	printf("start to recovery\n");
		system("mkdir /cache/system-bak; \
		mkdir /tmp/systemfs; \
		mount -t yaffs /dev/mtdblock14 /tmp/systemfs; \
		tar -xzf "QUECTEL_SYSTEM_BACKUP" -C /cache/system-bak; \
		cp -rf /cache/system-bak/* /tmp/systemfs/; \
		umount /tmp/systemfs; \
		rm -rf /cache/system-bak; \
		echo \"quectel_backup system success.\" >>"QUECTEL_BACKUP_LOG";sync");
    	printf("recovery end\n");
		reset_fota_cookie_mtd();
		sync();
		urc_to_ttygs_modem("+QIND: \"RECOVERY\",\"END\"\r\n");
		syscall(SYS_reboot, LINUX_REBOOT_MAGIC1, LINUX_REBOOT_MAGIC2, LINUX_REBOOT_CMD_RESTART, NULL);
		return 0;
	}
#endif
    printf("Starting recovery on %s", ctime(&start));

    ui_init();
    ui_set_background(BACKGROUND_ICON_INSTALLING);
    load_volume_table();

    int previous_runs = 0;
    const char *send_intent = NULL;
    const char *update_package = NULL;
    int wipe_data = 0, wipe_cache = 0;

#ifndef EMBED_UPDATE_BINARY
    //check delta update first
    handle_deltaupdate_status();
#else
	// goto redbend delta update
	handle_redbend_update();
#endif

#ifdef ENABLE_RECOVERY_COMMAND_LINE_ARG
    int arg;
    get_args(&argc, &argv);
    while ((arg = getopt_long(argc, argv, "", OPTIONS, NULL)) != -1) {
        switch (arg) {
        case 'p': previous_runs = atoi(optarg); break;
        case 's': send_intent = optarg; break;
        case 'u': update_package = optarg; break;
        case 'w': wipe_data = wipe_cache = 1; break;
        case 'c': wipe_cache = 1; break;
        case 't': ui_show_text(1); break;
        case '?':
            LOGE("Invalid command argument\n");
            continue;
        }
    }

    printf("Command:");
    for (arg = 0; arg < argc; arg++) {
        printf(" \"%s\"", argv[arg]);
    }
    printf("\n");

    if (update_package) {
        // For backwards compatibility on the cache partition only, if
        // we're given an old 'root' path "CACHE:foo", change it to
        // "/cache/foo".
        if (strncmp(update_package, "CACHE:", 6) == 0) {
            int len = strlen(update_package) + 10;
            char* modified_path = malloc(len);
            strlcpy(modified_path, "/cache/", len);
            strlcat(modified_path, update_package+6, len);
            printf("(replacing path \"%s\" with \"%s\")\n",
                   update_package, modified_path);
            update_package = modified_path;
        }
    }
    printf("\n");

    property_list(print_property, NULL);
    printf("\n");

    int status = INSTALL_SUCCESS;

    if (update_package != NULL) {
        status = install_package(update_package, &wipe_cache, TEMPORARY_INSTALL_FILE);
        if (status == INSTALL_SUCCESS && wipe_cache) {
            if (erase_volume("/cache")) {
                LOGE("Cache wipe (requested by package) failed.");
            }
        }
        if (status != INSTALL_SUCCESS) ui_print("Installation aborted.\n");
    } else if (wipe_data) {
        if (erase_volume("/data")) status = INSTALL_ERROR;
        if (wipe_cache && erase_volume("/cache")) status = INSTALL_ERROR;
        if (status != INSTALL_SUCCESS) ui_print("Data wipe failed.\n");
    } else if (wipe_cache) {
        if (wipe_cache && erase_volume("/cache")) status = INSTALL_ERROR;
        if (status != INSTALL_SUCCESS) ui_print("Cache wipe failed.\n");
    } else {
        status = INSTALL_ERROR;  // No command specified
    }

    if (status != INSTALL_SUCCESS) ui_set_background(BACKGROUND_ICON_ERROR);
    if (status != INSTALL_SUCCESS || ui_text_visible()) {
        finish_recovery(NULL);
    }

    // Otherwise, get ready to boot the main system...
    finish_recovery(send_intent);
    ui_print("Rebooting...\n");
    __reboot(LINUX_REBOOT_MAGIC1, LINUX_REBOOT_MAGIC2, LINUX_REBOOT_CMD_RESTART, NULL);
#endif
    LOGI("Rebooting at the end of recovery module.\n");
    finish_recovery(NULL);
    syscall(SYS_reboot, LINUX_REBOOT_MAGIC1, LINUX_REBOOT_MAGIC2, LINUX_REBOOT_CMD_RESTART, NULL);
    return EXIT_SUCCESS;
}
