	/**************************************************************
	  **					C O N F I D E N T I A L					**
	 **					Copyright 2002-2011					**
	  **					Red Bend Software					**
	 **************************************************************/
	
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/vfs.h>
#include <stdarg.h>
#include <fcntl.h>
#include "redbend/redbend.h" 

#include "redbend/include/RB_FileSystemUpdate.h"
#include "redbend/include/RB_ImageUpdate.h"
#include "redbend/include/RB_vRM_Errors.h"
#include "redbend/include/RB_vRM_Update.h"

#undef MAX_PATH
#define MAX_PATH 1024

static unsigned int quec_update_percent = 0;

void build_compile(void)
{
	printf("hello world\n");
}

/* Prints a string like the C printf() function */
RB_UINT32 RB_Trace(
	void*			pUser,
	const char*		aFormat,
	...)
{
/* 2015/11/10 added by tommy for debug */
#if 1
	va_list list;

	va_start(list, aFormat);
	vprintf(aFormat, list);
	va_end(list);
#endif
	return S_RB_SUCCESS;
}

long RB_ResetTimerA(void)
{
  return S_RB_SUCCESS;
}

void RB_Progress(void *pbUserData, RB_UINT32 uPercent)
{
#if 1
  static char buf[128];
  int value[2];

  if (get_fota_update_file_count(value)) return;

  quec_update_percent = uPercent;
/* 2015/6/5 added by tommy for urc format  */
#if 0
  sprintf(buf, "+QIND: \"FOTA\",\"UPDATING\", %d/%d--(%d%)\r\n", value[0], value[1], uPercent);
#else
  sprintf(buf, "+QIND: \"FOTA\",\"UPDATING\",%d,%d,%d\r\n",uPercent,value[0],value[1]);
#endif
  urc_to_ttygs_modem(buf);
  RB_Trace(NULL, "%s", buf);
  return;
#else
	static char buf[100];
	int tmp;

  	//sprintf(buf, "+QIND: \"FOTA\",\"UPDATING\",%d\r\n", uPercent);
  	//RB_Trace(NULL, "%s", buf);

	FILE *fp = fopen("/cache/reboot.txt", "rb+");
	char pbBuffer[16];
	if (fp == NULL)
	{
		RB_Trace(NULL, "RB_Progress FOPEN FAIL\n");
		return;
	}

	if (fread(pbBuffer, sizeof(char), 2, fp) != 2)
	{
		RB_Trace(NULL, "RB_Progress: error fread\r\n");
		fclose(fp);
		return;
	}
	pbBuffer[2] = '\0';
	tmp = atoi(pbBuffer);
	RB_Trace(NULL, "RB_Progress: uPercent: %d - %s(%d)\r\n", uPercent, pbBuffer, tmp);

	if (uPercent >= tmp)
	{
		fseek(fp, 3, SEEK_SET);
		if (fread(pbBuffer, sizeof(char), 1, fp) != 1)
		{
			RB_Trace(NULL, "RB_Progress: error fread\r\n");
			fclose(fp);
			return;
		}
		RB_Trace(NULL, "RB_Progress: read ch : %c\n", pbBuffer[0]);
		if (pbBuffer[0] == '0')
		{
			system("echo 10,1 > /cache/reboot.txt");
			system("reboot");
		}
	}
	fclose(fp);
  	return;
#endif
}

int quec_get_update_result(unsigned int *buf)
{
	buf[0] = quec_update_percent;
	return get_fota_update_file_count(&buf[1]);
}

void quec_clr_percent(void)
{
	quec_update_percent = 0;
	return;
}
/*!
 ************************************************************
 *                     RB_FSGetDelta
 ************************************************************
 *
 * @brief
 *	Reads data from the delta.
 *
 *	A glue function that needs to be implemented by the customer.
 *
 *	It should follow the following restrictions:
 *
 *	1. Returning the proper error level \see RbErrors.h
 *
 *	@param pbBuffer					The gives buffer that the data from the open file should be.
 *									copy into.
 *
 *	@param dwStartAddressOffset		The offset in the read file that should be
 *									the starting point for the copy.
 *
 *	@param dwSize					The size in bytes that should be copied from the open file,
 *									starting from the given position offset.
 *
 *	@return							One of the return codes as defined in RbErrors.h
 *
 ************************************************************
 */
long RB_GetDelta(void *pbUserData, unsigned char *pbBuffer, RB_UINT32 dwStartAddressOffset, RB_UINT32 dwSize)
{
	redbend_fota *rbf = pbUserData;

	if (dwSize == 0) return S_RB_SUCCESS;
	if (rbf->fota_flag == FOTA_IMAGE_PART)
	{
		FILE *fp =NULL;

		RB_Trace(NULL, "RB_GetDelta: addr=0x%08x, size=0x%x\r\n", dwStartAddressOffset, dwSize);
		fp = fopen(rbf->update_name, "r");
		if (fp == NULL)
		{
			RB_Trace(NULL, "RB_GetDelta: fopen %s err\n", rbf->update_name);
			goto fopen_err;
		}

		if (fseek(fp, dwStartAddressOffset, SEEK_SET) != 0)
		{
			RB_Trace(NULL, "RB_GetDelta: fseek %s err\n", rbf->update_name);
			goto fseek_err;
		}

		if (fread(pbBuffer, sizeof(char), dwSize, fp) == 0)
		{
			RB_Trace(NULL, "RB_GetDelta: fread %s err\n", rbf->update_name);
			goto fseek_err;
		}
		
		fclose(fp);
		return S_RB_SUCCESS;

		fseek_err:
			fclose(fp);
		fopen_err:
			return E_RB_FAILURE;
	}
	else if (rbf->fota_flag == FOTA_FS_PART)
	{
		int readCount = 0, ret = 0;
		FILE* fp;
		redbend_fota *rfd = pbUserData;

		RB_Trace(NULL, "%s: offset 0x%lx(%ld), size 0x%lx(%ld)\n", __func__,
			dwStartAddressOffset, dwStartAddressOffset, dwSize, dwSize);

		fp = fopen(rfd->update_name, "r");
		if (!fp)
		{
			RB_Trace(NULL, "%s: open delta file %s failed.\n", __func__, rfd->update_name);
			return E_RB_OPENFILE_ONLYR;
		}
		ret = fseek(fp, dwStartAddressOffset, 0);
		if (ret == -1)
		{
			fclose(fp);
			return E_RB_READ_ERROR;
		}
		readCount = fread(pbBuffer, 1, dwSize, fp);
		if (readCount != dwSize)
		{
			RB_Trace(NULL, "%s: error in read size\n", __func__);
			fclose(fp);
			ret = E_RB_READ_ERROR;
		}
		fclose(fp);
		return S_RB_SUCCESS;
	}
	else
	{
		RB_Trace(NULL, "RB_GetDelta: don't konw update part type\n");
		return E_RB_FAILURE;
	}
}

/* IMAGE UPDATE */

long RB_ReadImage(void *pbUserData, unsigned char *pbBuffer, RB_UINT32 dwStartAddress, RB_UINT32 dwSize)
{
	redbend_fota *rbf = pbUserData;
	size_t count;

	RB_Trace(NULL, "RB_ReadImage: addr=0x%08x, size=0x%08x\r\n", dwStartAddress, dwSize);
	count = mtd_read_data_offset(rbf, pbBuffer, dwSize, dwStartAddress);
	if (count != dwSize)
	{
		RB_Trace(NULL, "RB_ReadImage: read err addr=0x%08x, size=0x%08x\r\n", dwStartAddress, dwSize);
		return E_RB_FAILURE;
	}
	return S_RB_SUCCESS;
}

long RB_GetBlockSize(void *pbUserData)
{
	redbend_fota *rbf = pbUserData;
	RB_Trace(NULL, "RB_GetBlockSize: size=0x%08x\n", rbf->flash_block_size);
	return rbf->flash_block_size;
}

long RB_ReadImageNewKey(void *pbUserData, unsigned char *pbBuffer, RB_UINT32 dwStartAddress, RB_UINT32 dwSize)
{
	RB_Trace(NULL, "RB_ReadImageNewKey..addr=0x%08x, size=0x%x\r\n", dwStartAddress, dwSize);
	return RB_ReadImage(pbUserData, pbBuffer, dwStartAddress, dwSize);
}

long RB_WriteBlock(void *pbUserData, RB_UINT32 dwBlockAddress, unsigned char *pbBuffer)
{
	redbend_fota *rbf = pbUserData;
	size_t count;

	RB_Trace(NULL, "RB_WriteBlock: addr=0x%08x\r\n", dwBlockAddress);
	count = mtd_write_data_offset(rbf, pbBuffer, rbf->flash_block_size, dwBlockAddress);
	if (count != rbf->flash_block_size)
	{
		return E_RB_FAILURE;
	}
	return S_RB_SUCCESS;
}

long RB_GetRBDeltaOffset(void *pbUserData, RB_UINT32 signed_delta_offset, RB_UINT32* delta_offset)
{
	*delta_offset = signed_delta_offset;
	return S_RB_SUCCESS;
}

long RB_ReadBackupBlock(void *pbUserData, unsigned char *pbBuffer, RB_UINT32 dwBlockAddress, RB_UINT32 dwSize)
{
	RB_Trace(NULL, "RB_ReadBackupBlock: addr=0x%08x, size=0x%x\r\n", dwBlockAddress, dwSize);

	redbend_fota *rbf = pbUserData;
	FILE *fp = fopen(rbf->backup_name, "rb");
	if (fp == NULL)
	{
		RB_Trace(NULL, "RB_ReadBackupBlock: error open (addr=0x%08x, size=0x%x)\r\n", dwBlockAddress, dwSize);
		return E_RB_FAILURE;
	}

	if (fseek(fp, dwBlockAddress, SEEK_SET) != 0)
	{
		RB_Trace(NULL, "RB_ReadBackupBlock: error fseek (addr=0x%08x, size=0x%x)\r\n", dwBlockAddress, dwSize);
		fclose(fp);
		return E_RB_FAILURE;
	}

	if (fread(pbBuffer, sizeof(char), dwSize, fp) != dwSize)
	{
		RB_Trace(NULL, "RB_ReadBackupBlock: error fread (addr=0x%08x, size=0x%x)\r\n", dwBlockAddress, dwSize);
		fclose(fp);
		return E_RB_FAILURE;
	}

	fclose(fp);
	return S_RB_SUCCESS;
}

long RB_WriteBackupBlock(void *pbUserData, RB_UINT32 dwBlockStartAddress, unsigned char *pbBuffer)
{
	redbend_fota *rbf = pbUserData;
	RB_Trace(NULL, "RB_WriteBackupBlock: addr=0x%08x\r\n", dwBlockStartAddress);
	return RB_WriteBackupPartOfBlock(pbUserData, dwBlockStartAddress, rbf->flash_block_size, pbBuffer);
}

long RB_EraseBackupBlock(void *pbUserData, RB_UINT32 dwStartAddress)
{
	RB_Trace(NULL, "RB_EraseBackupBlock: addr=0x%08x\r\n", dwStartAddress);

	redbend_fota *rbf = pbUserData;
	RB_UINT32 dwSize = (RB_UINT32)rbf->flash_block_size;
	char *pbBuffer = malloc(dwSize);
	if (pbBuffer == NULL)
	{
		RB_Trace(NULL, "RB_EraseBackupBlock: malloc fail");
		return E_RB_FAILURE;
	}

	FILE *fp = fopen(rbf->backup_name, "rb+");
	if (fp == NULL)
	{
		RB_Trace(NULL, "RB_EraseBackupBlock: error fopen (addr=0x%08x, size=0x%x)\r\n", dwStartAddress, dwSize);
		free(pbBuffer);
		return E_RB_FAILURE;
	}

	if (fseek(fp, dwStartAddress, SEEK_SET) != 0)
	{
		RB_Trace(NULL, "RB_EraseBackupBlock: error fseek (addr=0x%08x, size=0x%x)\r\n", dwStartAddress, dwSize);
		fclose(fp);
		free(pbBuffer);
		return E_RB_FAILURE;
	}

	memset(pbBuffer, 0xff, dwSize);
	if (fwrite(pbBuffer, sizeof(char), dwSize, fp) != dwSize)
	{
		RB_Trace(NULL, "RB_EraseBackupBlock: error fwrite (addr=0x%08x, size=0x%x)\r\n", dwStartAddress, dwSize);
		fclose(fp);
		free(pbBuffer);
		return E_RB_FAILURE;
	}

	free(pbBuffer);
	fsync(fp);
	fclose(fp);
	return S_RB_SUCCESS;
}

long RB_WriteBackupPartOfBlock(void *pbUserData, RB_UINT32 dwStartAddress, RB_UINT32 dwSize, unsigned char* pbBuffer)
{
	RB_Trace(NULL, "RB_WriteBackupPartOfBlock: addr=0x%08x size=0x%08x\r\n", dwStartAddress, dwSize);
	redbend_fota *rbf = pbUserData;
	FILE *fp = fopen(rbf->backup_name, "rb+");
	if (fp == NULL)
	{
		RB_Trace(NULL, "RB_WriteBackupPartOfBlock: error fopen (addr=0x%08x, size=0x%x)\r\n", dwStartAddress, dwSize);
		return E_RB_FAILURE;
	}

	if (fseek(fp, dwStartAddress, SEEK_SET) != 0)
	{
		RB_Trace(NULL, "RB_WriteBackupPartOfBlock: error fseek (addr=0x%08x, size=0x%x)\r\n", dwStartAddress, dwSize);
		fclose(fp);
		return E_RB_FAILURE;
	}

	if (fwrite(pbBuffer, sizeof(char), dwSize, fp) != dwSize)
	{
		RB_Trace(NULL, "RB_WriteBackupPartOfBlock: error fwrite (addr=0x%08x, size=0x%x)\r\n", dwStartAddress, dwSize);
		fclose(fp);
		return E_RB_FAILURE;
	}

	fsync(fp);
	fclose(fp);
	return S_RB_SUCCESS;
}

/* FS UPDATE */
long RecursiveFolderCreater(
	const char*	folderpath,
	const		mode_t mode)
{
	int ret = 0;
	char temppath[MAX_PATH] =  {'\0'};
	int pathOffset = strlen(folderpath);// For counting back until the '/' delimiter

	RB_Trace(NULL, " %s path: %s\n", __func__, folderpath);
	
	if(pathOffset == 0)
		return -1;//if from some reason we got to the end return error!!!.
		
	while(folderpath[pathOffset] != '/')// get to the next '/' place
		pathOffset--;
			
	strncpy(temppath, folderpath, pathOffset);// copy one depth below till and without the char '/'
	RB_Trace(NULL, " temppath: %s\n", temppath);
	
	ret = mkdir(temppath, mode);
	RB_Trace(NULL, " mkdir result: %d errno: %d\n", ret, errno);
	if (ret == 0 || ((ret == -1) && (errno == EEXIST)))
	{
		return 0;//meaning the depth creation is success.
	}
	else if((ret == -1) && (errno == ENOENT))
	{
		ret = RecursiveFolderCreater(temppath, mode);
		if (ret == 0)
		{
			ret = mkdir(temppath, mode);
		}
		return ret;
	}
	else
	{
		return -1;
	}
}

/*!
 ************************************************************
 *                     RB_CopyFile
 ************************************************************
 *
 * @brief
 *	This function copies a file to another file with a different name or path.
 *
 *
 *	@param pbUserData	Any user data structure, that may be useful for the user
 *					implementation If not needed set to NULL.
 *					The calling function supply you with the user data,
 *					previously supplied in the RB_FileSystemUpdate.
 *
 *	@param strFromPath	The path where the file exist.
 *
 *	@param strToPath	New destination of the file.
 *
 *	@return			One of the return codes as defined in RbErrors.h
 *
 ************************************************************
 */

long RB_CopyFile(
	void*		pbUserData,
	const char*	strFromPath,
	const char*	strToPath)
{
	FILE* fp1 = NULL;
	FILE* fp2 = NULL;
	unsigned int readCount = 0, writeCount = 0;
	const unsigned long BUFFER_SIZE = 10240;
	char buf[BUFFER_SIZE];
	int ret = 0;
	char *path1;
	char *path2;

	path1 = strFromPath;
	path2 = strToPath;

	printf ("%s: %s -> %s ", __func__, path1, path2);
	if (!strFromPath || !strToPath)
	{
		RB_Trace(NULL, "NULL file name find. Abort.\n");
		return -1;			//should never happen
	}
	
	fp1 = fopen(path1, "r");
	if (!fp1)
	{
		RB_Trace(NULL, " Open %s ENOENT %d\n", path1, errno);
		RB_Trace(NULL, "Open %s failed. Abort.\n", path1);
		return E_RB_OPENFILE_ONLYR;
	}

	fp2 = fopen(path2, "w");
	if (!fp2)
	{
		char shortfolderpath[MAX_PATH] = {'\0'};
		char* folder = strrchr(path2,'/'); 
		char* folderPath = (char *) malloc(folder - path2 + 1); 
		
		if (folderPath == NULL)
		{
			RB_Trace(NULL, "malloc failure (folderPath).\n");
			return -4;
		}

		memset(folderPath,'\0',folder - path2 + 1); 
		strncpy(folderPath, path2, folder - path2);
		strcpy(shortfolderpath, folderPath);

		free(folderPath);
		if ( RB_CreateFolder(NULL,shortfolderpath) != S_RB_SUCCESS )
		{
			fclose(fp1);
			RB_Trace(NULL, "Open %s failed. Abort.\n", path2);
			return E_RB_OPENFILE_WRITE;
		}
		else
		{
			fp2 = fopen(path2, "w");
			if(!fp2)
			{
				fclose(fp1);
				RB_Trace(NULL, "Open %s failed. Abort.\n", path2);
				return E_RB_OPENFILE_WRITE;
			}
		}
	}

	while( (readCount = fread(buf, 1, BUFFER_SIZE, fp1))> 0)
	{
		writeCount = fwrite(buf, 1, readCount, fp2);
		if (writeCount != readCount)
		{
			RB_Trace(NULL, " read %d, but write %d, abort.\n", readCount, writeCount);
			ret = E_RB_WRITE_ERROR;
 			break;
		}
	}

	fclose(fp1);
	fclose(fp2);

	return ret;
}

/*!
 ************************************************************
 *                     RB_DeleteFile
 ************************************************************
 *
 * @brief
 *	This function deletes a specified file and removes it from the File System.
 *
 *
 *	@param pbUserData	Any user data structure, that may be useful for the user
 *						implementation If not needed set to NULL.
 *						The calling function supply you with the user data,
 *						previously supplied in the RB_FileSystemUpdate.
 *
 *	@param strPath		The path of the file.
 *
 *	@return				One of the return codes as defined in RbErrors.h
 *
 ************************************************************
 */

long RB_DeleteFile(
	void*					pbUserData,
	const char*				strPath)
{
	char *path;
	int ret = 0;
 
	path = strPath;
	printf ("%s: %s\n", __func__, path);
	ret = unlink(path);
	printf (" unlink value: %d, errno: %d\n", ret, errno);
	if (ret == 0)
		return S_RB_SUCCESS;
 
	if (ret < 0 && errno == ENOENT)	//if file does not exist then we can say that we deleted it successfully
		return S_RB_SUCCESS;
  	return E_RB_DELETEFILE;
}

/*!
 ************************************************************
 *                     RB_DeleteFolder
 ************************************************************
 *
 * @brief
 *	This function deletes a specified folder and removes it from the File System.
 *
 *
 *	@param pbUserData	Any user data structure, that may be useful for the user
 *						implementation If not needed set to NULL.
 *						The calling function supply you with the user data,
 *						previously supplied in the RB_FileSystemUpdate.
 *
 *	@param strPath		The path of the folder.
 *
 *	@return				One of the return codes as defined in RbErrors.h
 *
 ************************************************************
 */

long RB_DeleteFolder(
	void*					pbUserData,
	const char*				strPath)
{
	int ret = 0;
	char *path;

	path = strPath;
	printf ("%s: %s\n", __func__, path);

	ret = rmdir(path);
 	printf (" rmdir value: %d, errno: %d\n", ret, errno);
	
	if ((ret == 0) || ((ret < 0) && ((errno == ENOENT) || (errno == ENOTEMPTY ))))
		return S_RB_SUCCESS;
	
 	return E_RB_FAILURE;
}

/*!
 ************************************************************
 *                     RB_CreateFolder
 ************************************************************
 *
 * @brief
 *	This function deletes a specified folder and removes it from the File System.
 *
 *
 *	@param pbUserData	Any user data structure, that may be useful for the user
 *						implementation If not needed set to NULL.
 *						The calling function supply you with the user data,
 *						previously supplied in the RB_FileSystemUpdate.
 *
 *	@param strPath		The path of the folder.
 *
 *	@return				One of the return codes as defined in RbErrors.h
 *
 ************************************************************
 */

long RB_CreateFolder(
	void*					pbUserData,
	const char*				strPath)
{
	mode_t mode = 0;
	int ret = 0;
	char *path;

	path = strPath;
	mode = 
		S_IRUSR /*Read by owner*/ | 
		S_IWUSR /*Write by owner*/ | 
		S_IXUSR /*Execute by owner*/ | 
		S_IRGRP /*Read by group*/ | 
		S_IWGRP /*Write by group*/ | 
		S_IXGRP /*Execute by group*/ | 
		S_IROTH /*Read by others*/ | 
		S_IWOTH /*Write by others*/ | 
		S_IXOTH /*Execute by others*/;
	
	printf ("%s: %s, mode:0x%x\n", __func__, path, mode);
	
	ret = mkdir(path, mode);
	
	if (ret == 0 || ((ret == -1) && (errno == EEXIST)))
	{
		return S_RB_SUCCESS;
	}
	else if((ret == -1) && (errno == ENOENT))//maybe multi directory problem
	{
		ret = RecursiveFolderCreater(path, mode);
		if(ret == 0) 
		{
			ret = mkdir(path, mode);//After creating all the depth Directories we try to create the Original one again.
			if(ret == 0)
				return S_RB_SUCCESS;
			else
				return E_RB_FAILURE;
		}
		else
		{
			return E_RB_FAILURE;
		}
	}
	else
	{
		return E_RB_FAILURE;
	}
}

/*!
 ************************************************************
 *                     RB_OpenFile
 ************************************************************
 *
 * @brief
 *	Opens a file in the file system.
 *
 *	A glue function that needs to be implemented by the customer.
 *
 *	It should follow the following restrictions:
 *
 * 1. Returning the proper error level \see RbErrors.h
 *
 *
 *	@param pbUserData	Any user data structure, that may be useful for the user
 *						implementation If not needed set to NULL.
 *						The calling function supply you with the user data,
 *						previously supplied in the RB_FileSystemUpdate.
 *
 *	@param strPath		An absolute path to the file location in the FS.
 *
 *	@param wFlag		Controls the access mode read, write or both.
 *						opens a file to write deletes the file content.
 *
 *	@param pwHandle		A handle to the file.
 *
 *	@return				One of the return codes as defined in RbErrors.h
 *
 ************************************************************
 */
mode_t get_mode(E_RW_TYPE wFlag)
{
	switch (wFlag)
	{
	case ONLY_R:
		RB_Trace(NULL, " RDONLY \n");
		return O_RDONLY;
	case ONLY_W:
		RB_Trace(NULL, " WRONLY \n");
		return O_WRONLY | O_CREAT;
	case BOTH_RW:
		RB_Trace(NULL, " RDWR \n");
		return O_RDWR | O_CREAT;
	default:
		RB_Trace(NULL, " Unknown \n");
		return 0;
	}
}

long RB_OpenFile(
	void*					pbUserData,
	const char*				strPath,
	E_RW_TYPE				wFlag,
	long*					pwHandle)
{
	mode_t mode;
	char *path;

	path = strPath;
	RB_Trace(NULL, "open file %s\n", path);
	
	RB_Trace(NULL, "%s: Path:%s | Mode:", __func__, path);
	mode = get_mode(wFlag);

	*pwHandle = open(path, mode);
	if (*pwHandle == -1)
	{
		*pwHandle = 0;
		RB_Trace(NULL, " First open() with error %d\n", errno);
		if (wFlag == ONLY_R)
			return E_RB_OPENFILE_ONLYR;

		//if  we need to open the file for write or read/write then we need to create the folder (in case it does not exist)
		if ((wFlag != ONLY_R) && (errno == ENOENT))
		{
			char dir[MAX_PATH] = {'\0'};
			char dirShort[MAX_PATH] = {'\0'};
			int i = 0;
			//copy the full file path to directory path variable
			while (path[i] != '\0')
			{
				dir[i] = path[i];
				i++;
			}
			RB_Trace(NULL, " copy dir[]=%s\n", dir);
			//search for the last '/' char
			while (dir[i--] != '/')
				;
			dir[i+1] = '\0';
			RB_Trace(NULL, " remove dir[]=%s\n", dir);
			
			//convert_char_to_unicode((unsigned char*)dir, dirShort);
			strcpy(dirShort, dir);

			if (RB_CreateFolder(pbUserData, dirShort))
			{
				RB_Trace(NULL, " Fail create folder, Leave RB_OpenFile\n");
				return E_RB_OPENFILE_WRITE;
			}
	
			*pwHandle = open(path, mode);
			if (*pwHandle == -1 || *pwHandle == 0)
			{
				*pwHandle = 0;
				RB_Trace(NULL, " After successful creating folder, fail open() with error %d\n", errno);
				return E_RB_OPENFILE_WRITE;
			}
		}
 	}
	RB_Trace(NULL, " Successful open() *pwHandle:%ld\n", *pwHandle);

	return S_RB_SUCCESS;
}

/*!
 ************************************************************
 *                     RB_ResizeFile
 ************************************************************
 *
 * @brief
 *	set the size of a file in the file system.
 *
 *	@param pbUserData	Any user data structure, that may be useful for the user
 *						implementation If not needed set to NULL.
 *						The calling function supply you with the user data,
 *						previously supplied in the RB_FileSystemUpdate.
 *
 *	@param wHandle		A handle to the file.
 *
 *	@param dwSize		The new size of the file.
 *
 *	@return				One of the return codes as defined in RbErrors.h
 *
 ************************************************************
 */

long RB_ResizeFile(
	void*			pbUserData,
	long			wHandle,
	RB_UINT32		dwSize)
{
	int ret = -1;

	RB_Trace(NULL, "%s: handle %ld, dwSize %d\n", __func__, wHandle, dwSize);

	if (wHandle)
		ret = ftruncate(wHandle, dwSize);

	if (ret)
		ret = E_RB_RESIZEFILE;
	
	RB_Trace(NULL, "%s: ret %d handle %ld %d\n", __func__, ret, wHandle, errno);
	
	return ret;
}

/*!
 ************************************************************
 *                     RB_CloseFile
 ************************************************************
 *
 * @brief
 *	Close a file in the file system.
 *
 *	A glue function that needs to be implemented by the customer.
 *
 *	It should follow the following restrictions:
 *
 * 1. Returning the proper error level \see RbErrors.h
 *
 *
 *	@param pbUserData	Any user data structure, that may be useful for the user
 *						implementation, if not needed set to NULL.
 *						The calling function supply you with the user data,
 *						previously supplied in the RB_FileSystemUpdate.
 *
 *	@param wHandle		A handle to the file.
 *
 *	@return				One of the return codes as defined in RbErrors.h
 *
 ************************************************************
 */

long RB_CloseFile(
	void*	pbUserData,
	long 	wHandle)
{
	int ret = E_RB_CLOSEFILE_ERROR;
	RB_Trace(NULL, "%s: wHandle = %ld\n", __func__, wHandle);
	
	if (wHandle>=0)
		ret = close(wHandle);
	
	if (ret == 0)
		return S_RB_SUCCESS;
	
	return E_RB_CLOSEFILE_ERROR;
}

/*!
 ************************************************************
 *                     RB_WriteFile
 ************************************************************
 *
 * @brief
 *	Writes block of data to an open file in a reliable manner.
 *
 *	A glue function that needs to be implemented by the customer.
 *
 *	It should follow the following restrictions:
 *
 *	1. Returning the proper error level \see RbErrors.h
 *	2. The writing procedure should be a transaction.
 *		In case of returning successfully after writing a block means that
 *		the block has been written to its target location, or at least resides
 *		in a NV memory, and an automatic procedure will restore it to its target
 *		location. e.g. a power fail right after returning from the function invocation.
 *
 *	@param pbUserData		Any user data structure, that may be useful for the user
 *							implementation, if not needed set to NULL.
 *							The calling function supply you with the user data,
 *							previously supplied in the RB_FileSystemUpdate.
 *
 *	@param wHandle			Handle to the file.
 *
 *	@param dwPosition		Position were to write
 *
 *	@param pbBuffer			The block of data that should be written.
 *
 *	@param dwSize			The size in bytes of the block to be written.
 *
 *	@return					One of the return codes as defined in RbErrors.h
 *
 ************************************************************
 */

long RB_WriteFile(
	void*			pbUserData,
	long			wHandle,
	RB_UINT32		dwPosition,
	unsigned char*	pbBuffer,
	RB_UINT32		dwSize)
{
	int ret = 0, size = 0;

	RB_Trace(NULL, "%s: Handle:%ld , Pos:%ld , Size: %ld", __func__, wHandle,dwPosition,dwSize);
	size = lseek(wHandle, 0, SEEK_END);
	/* from the guide: if dwPosition is beyond size of file the gap between end-of-file and the position should be filled with 0xff */
	if (size < dwPosition)
	{
		int heap_size = dwPosition - size;
		unsigned char* p_heap = malloc(heap_size);
		
		if (p_heap == NULL)
		{
			RB_Trace(NULL, "malloc failure (p_heap).\n");
			return E_RB_WRITE_ERROR;
		}
		memset(p_heap, 0xFF, heap_size);
		ret = write(wHandle, p_heap, heap_size);
		free(p_heap);
		if (ret < 0)
		{
			RB_Trace(NULL, "write failed with return value: %d\n",ret);
			return E_RB_WRITE_ERROR;
		}
	}
	ret = lseek(wHandle, dwPosition, SEEK_SET);
	if (ret < 0)
	{
		RB_Trace(NULL, "lseek failed with return value: %d\n",ret);
		return E_RB_WRITE_ERROR;
	}

	ret = write(wHandle, pbBuffer, dwSize);
	if (ret < 0)
	{
		RB_Trace(NULL, "Failed with return value: %d\n",ret);
		return E_RB_WRITE_ERROR;
	}
	RB_Trace(NULL, "Bytes Write: %d\n",ret);

	return S_RB_SUCCESS;
}

/*!
 ************************************************************
 *                     RB_ReadFile
 ************************************************************
 *
 * @brief
 *	Reads data from an open file.
 *
 *	A glue function that needs to be implemented by the customer.
 *
 *	It should follow the following restrictions:
 *
 *	1. Returning the proper error level \see RbErrors.h
 *
 *	@param pbUserData	Any user data structure, that may be useful for the user
 *						implementation, if not needed set to NULL.
 *						The calling function supply you with the user data,
 *						previously supplied in the RB_FileSystemUpdate.
 *
 *	@param wHandle		Handle to the file.
 *
 *	@param dwPosition	The offset in the read file that should be
 *						the starting point for the copy.
 *
 *	@param pbBuffer		The gives buffer that the data from the open file should be.
 *						copy into.
 *
 *	@param dwSize		The size in bytes that should be copied from the open file,
 *						starting from the given position offset.
 *
 *	@return				One of the return codes as defined in RbErrors.h
 *
 ************************************************************
 */

long RB_ReadFile(
	void*			pbUserData,
	long			wHandle,
	RB_UINT32		dwPosition,
	unsigned char*	pbBuffer,
	RB_UINT32		dwSize)
{
	int ret = 0;

	RB_Trace(NULL, " %s: Handle:%ld , Pos:%ld , Size: %ld", __func__, wHandle,dwPosition,dwSize);
	
	ret = lseek (wHandle, dwPosition, SEEK_SET);
	if (ret < 0)
	{
		RB_Trace(NULL, " lseek failed with return value: %d\n",ret);
		return E_RB_READ_ERROR;
	}
	ret = read(wHandle, pbBuffer, dwSize);
	if (ret < 0)
	{
		RB_Trace(NULL, " read failed with return value: %d\n",ret);
		return E_RB_READ_ERROR;
	}
	
	RB_Trace(NULL, " Bytes Read: %d\n",ret);
	if (ret != dwSize && ((ret + dwPosition) != RB_GetFileSize(pbUserData, wHandle)))
		return E_RB_READ_ERROR;
	
	return S_RB_SUCCESS;
}

long RB_GetFileSize(
	void*	pbUserData,
	long	wHandle)
{
	int ret = 0;

	RB_Trace(NULL, "%s: %ld ", __func__, wHandle);

	ret = lseek(wHandle, 0, SEEK_END);

	if (ret == -1)
	{
		RB_Trace(NULL, " lseek errno: %d\n", errno);
		return E_RB_READFILE_SIZE;
	}
	RB_Trace(NULL, "Returning Size = 0x%lx\n",ret);
	
	return ret;
}

long RB_Unlink(
	void*			pbUserData,
	const char*		pLinkName)
{
	int ret = 0;
	char *path;

	RB_Trace(NULL, "%s \n", __func__);

	path = pLinkName;
	
	ret = unlink(path);
	if(ret < 0)
	{
		RB_Trace(NULL, "unlink failed with return value: %d\n",ret);
		return E_RB_FAILURE;
	}
	RB_Trace(NULL, "unlink with return value: %d\n",ret);

	return S_RB_SUCCESS;
}

long RB_VerifyLinkReference(
	void*		pbUserData,
	const char*	pLinkName,
	const char*	pReferenceFileName)
{
	int ret = 0;
	char *path;
	char *refPath;
	char linkedpath[MAX_PATH]={'\0'};

	RB_Trace(NULL, "%s \n", __func__);

	path = pLinkName;
	refPath = pReferenceFileName;
	
	ret = readlink(path, linkedpath, MAX_PATH);
	if (ret < 0)
	{
		RB_Trace(NULL, "readlink failed with return value: %d\n",ret);
		return E_RB_FAILURE;
	}
	
	if ((memcmp(&linkedpath, &refPath, ret))!=0)
	{
		RB_Trace(NULL, "not same linked path - linkedpath[%s] pReferenceFileName[%s]\n", linkedpath, refPath);
		return E_RB_FAILURE;
	}
	RB_Trace(NULL, " same linked path\n");

	return S_RB_SUCCESS;
}

long RB_Link(
	void*			pbUserData,
	const char*		pLinkName,
	const char*		pReferenceFileName)
{
	int ret = 0;
	char *sympath;
	char *refpath;
	
	RB_Trace(NULL, "%s \n", __func__);
	
	sympath = pLinkName;
	refpath = pReferenceFileName;

	if(!RB_VerifyLinkReference(pbUserData, pLinkName, pReferenceFileName))
		return S_RB_SUCCESS;

	ret = symlink(refpath, sympath);
	if (ret != 0)
	{
		RB_Trace(NULL, " symlink failed with return value: %d, errno: %d\n", ret, errno);
		if (errno == EEXIST && RB_VerifyLinkReference(pbUserData, pLinkName, pReferenceFileName))
		{
			return S_RB_SUCCESS;
		}
		return E_RB_FAILED_CREATING_SYMBOLIC_LINK;
	}
	RB_Trace(NULL, " symlink with return value: %d\n",ret);

	return S_RB_SUCCESS;
}

long RB_SetFileAttributes(
	void*					pbUserData,
	const char*				ui16pFilePath,
	const RB_UINT32			ui32AttribSize,
	const unsigned char*	ui8pAttribs)
{
	const int ATTRSIZE = 25;
	char tmpAttribs[ATTRSIZE];
	char *tp;
	char *endstr;
	char *rb_sig;
	
	uid_t setUserID		= 0;
	gid_t setGroupID	= 0;
	mode_t setFileMode	= 0;
	char *setFilePath;
	struct stat sbuf;
	int ret = 0;
	// debug start
	int count = 0;
	// debug end
	
	RB_Trace(NULL, "%s \n", __func__);
	
	if(NULL == ui16pFilePath)
	{
		RB_Trace(NULL, "ui16pFilePath NULL [error]\n");
		return E_RB_BAD_PARAMS;
	}
	else if(NULL == ui8pAttribs)
	{
		RB_Trace(NULL, "ui8pAttribs NULL [error]\n");
		return E_RB_BAD_PARAMS;
	}

	setFilePath = ui16pFilePath;
		
	ret = lstat(setFilePath, &sbuf);
	if(ret < 0)
	{	
			RB_Trace(NULL, "stat failed with return value: %d\n",ret);
			return E_RB_FAILURE;
	}
	else
	{
		if(S_ISLNK(sbuf.st_mode))
		{
			RB_Trace(NULL, " stat->st_mode = symbolic link file\n");
			return S_RB_SUCCESS;
		}
		if(S_ISREG(sbuf.st_mode))
		{
			RB_Trace(NULL, " stat->st_mode = regular file\n");
		}
		if (S_ISDIR(sbuf.st_mode))
		{
			RB_Trace(NULL, " stat->st_mode = directory\n");
		}
	}
	
	if(0 == ui32AttribSize)
	{
		RB_Trace(NULL, "ui32AttribSize 0\n");
		return S_RB_SUCCESS;
	}
	
	RB_Trace(NULL, "ui16pFilePath = %s\n", setFilePath);
	RB_Trace(NULL, "ui32AttribSize = %u\n", ui32AttribSize);
	RB_Trace(NULL, "ui8pAttribs = %s\n", ui8pAttribs);
	// debug start
	for(count=0;count<ATTRSIZE;count++)
	{
		RB_Trace(NULL, "ui8pAttribs[%d] = %c\n", count, ui8pAttribs[count]);
	}
	// debug end

	memset((void*)tmpAttribs, 0x0, ATTRSIZE);
	memcpy(tmpAttribs, ui8pAttribs, (size_t)ui32AttribSize);
	
	//Check that the structure is Valid
	if(NULL == strstr(tmpAttribs,"_redbend_"))
		return E_RB_FAILURE;


	tp = strtok((unsigned char *)tmpAttribs, ":");

	//Remove the _redbend_ SAFIX
	rb_sig = strrchr(tp,'_');
	rb_sig++;

	
	// Get FileMode
	setFileMode = strtol(rb_sig, &endstr, 8);
	tp = strtok(NULL, ":");

	// Get UserID
	if (tp != NULL)
	{
		setUserID = (uid_t)strtol(tp, &endstr, 16);
		tp = strtok(NULL, ":");
	}

	// Get GroupID
	if (tp != NULL)
	{
		setGroupID = (gid_t)strtol(tp, &endstr, 16);
	}

	// Set FileMode
	RB_Trace(NULL, "setFilePath = %s\n",setFilePath);
	RB_Trace(NULL, "setFileMode = %d\n", setFileMode);
	RB_Trace(NULL, "setFileMode = %o\n", setFileMode);
	RB_Trace(NULL, "setUserID = %d\n", setUserID);
	RB_Trace(NULL, "setGroupID = %d\n", setGroupID);
	
	// Set UserID,GroupID
	if( chown(setFilePath, setUserID, setGroupID) )
	{
		RB_Trace(NULL, "%s chown error\n", __func__);
		// debug start
		RB_Trace(NULL, "%s setUserID = %d\n", __func__, setUserID);
		RB_Trace(NULL, "%s setGroupID = %d\n", __func__, setGroupID);
		RB_Trace(NULL, "%s chown errno = %d\n", __func__, errno);
		// debug end
		return E_RB_FAILURE;
	}

	if( chmod(setFilePath, setFileMode) )
	{
		RB_Trace(NULL, "%s chmod error\n", __func__);
		return E_RB_FAILURE;
	}

	RB_Trace(NULL, "%s SUCCESS\n", __func__);
	return S_RB_SUCCESS;
}

long RB_CompareFileAttributes(
	void*			pbUserData,
	unsigned short *	pFilePath,
	unsigned char*	pAdditionalAttribs,
	unsigned long	iAddiInfoSize)
{
	return S_RB_SUCCESS;
}

/*!
 ************************************************************
 *                     RB_FSTrace
 ************************************************************
 *
 * @brief
 *	Generic log
 *
 *	A glue function that needs to be implemented by the customer.
 *	Gives the Update procedure the ability to send out status indications and debug
 *	information.
 *
 *	@param pbUserData	Any user data structure, that may be useful for the user
 *						implementation, if not needed set to NULL.
 *						The calling function supply you with the user data,
 *						previously supplied in the RB_FileSystemUpdate.
 *
 *	@param aFormat		a NULL terminated string that support a subset of the known
 *						standard c library printf.
 *						Supports:
 *							%x - hex number
 *							%0x - hex number with leading zeros
 *							%fx - hex number with leading spaces
 *							%u - unsigned decimal
 *							%s - null terminated string
 *
 *	@param ...			List of the format corresponding variable
 *
 *	@return				One of the return codes as defined in RbErrors.h
 *
 ************************************************************
 */
long RB_FSTrace(
	void*					pUser,
	const char*				aFormat,
	...)
{
#if 0
	va_list list;

	va_start(list, aFormat);
	vprintf(aFormat, list);
	va_end(list);
	printf("\n");
#endif
	return S_RB_SUCCESS;
}

long RB_MoveFile(
	void* pbUserData,
	const char* strFromPath,
	const char* strToPath)
{
	int ret = 0;
	char *path1;
	char *path2;

	path1 = strFromPath;
	path2 = strToPath;

	printf ("%s: %s -> %s ", __func__, path1, path2);

	if (!strFromPath || !strToPath)
	{
		RB_Trace(NULL, "NULL file name find. Abort.\n");
		return -1;			//should never happen
	}

	ret = rename (path1,path2);	
	if (ret < 0)
	{
		RB_Trace(NULL, " Move File Failed with return value: %d\n",ret);
		return E_RB_WRITE_ERROR;
	}
	return S_RB_SUCCESS;
}

long RB_SyncFile(
	void*	pbUserData,
	long	wHandle)
{
	long ret = -1;
	ret = fsync(wHandle);
	if (ret < 0)
	{
		RB_Trace(NULL, "fsync Failed with return value: %d\n",ret);
		return E_RB_WRITE_ERROR;
	}
	RB_Trace(NULL, "fsync after write: %d\n",ret);

	return S_RB_SUCCESS;
}

long RB_GetAvailableFreeSpace(
	void       *pbUserData,
	const char *partition_name,
	RB_UINT32  *available_flash_size)
{
	struct statfs vfs;

	RB_Trace(NULL, "%s : %s\n", __func__, partition_name);
	if ( statfs(partition_name, &vfs) < 0 )
	{
		RB_Trace(NULL, "%s : failed to get statvfs\n", __func__);
		*available_flash_size = 0;
		return -1;
	}

	*available_flash_size = vfs.f_bsize * vfs.f_bfree;
	RB_Trace(NULL, "%s : success %d\n", __func__, *available_flash_size);
	return S_RB_SUCCESS;
}
