/* Copyright (c) 2010-2011, The Linux Foundation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */
/*
 * Quectel power_manager  driver
 *
 */

#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/err.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <asm/atomic.h>
#include <linux/init.h>
#include <linux/gpio.h>
#include <linux/spinlock.h>
#include <linux/quectel_power_manager.h>

#include <linux/init.h>
#include <linux/workqueue.h>
#include <linux/module.h>

#define DTR_IRQ_WORK

#define DTR_ESCAPE_MODE

#ifdef DTR_ESCAPE_MODE//francis
#include <mach/msm_smd.h>
#endif
/* Private data */

static DEFINE_MUTEX(wakeup_lock);
static DEFINE_MUTEX(dtr_lock);
/* Internal functions */

	
#ifdef DTR_ESCAPE_MODE

#define	DTR_SMD_CHANNEL_NAME "DATA3"

static DEFINE_MUTEX(dtr_ch_lock);

struct dtr_device_t{
  
	struct smd_channel *ch;
    char name[10];
	int dtr_value;//0 dtr low 1:dtr high
	int wakeup_value;
};

struct dtr_device_t *dtr_devp = NULL;

struct dtr_device_t dtr_device = {
    .name = "dtr", 
    .dtr_value = -1,	
	.wakeup_value = -1,
};

#endif//DTR_ESCAPE_MODE

#ifdef DTR_IRQ_WORK
static struct workqueue_struct *queue = NULL;
static struct work_struct work;
static struct quectel_power_manager *dtr_data = NULL;

void send_data_to_smd(int dtr_value,int w_value);

static void work_handler(struct work_struct *data)
{

	struct quectel_power_manager_pdata *pdata = dtr_data->pdata;
	
    int	dtr_value = !!gpio_get_value(pdata->gpio_dtr);
    int	wakeup_value = !!gpio_get_value(pdata->gpio_wakeup);

    printk("dtr work handler dtrvalue = %d,wakeupin_value = %d\n",dtr_value,wakeup_value);

	send_data_to_smd(dtr_value,wakeup_value);

	if(dtr_data != NULL)
	{
	  printk("dtr_data != NULL irq = %d\n",dtr_data->irq_dtr);
	}
}

#endif

static irqreturn_t quectel_wakeup_irq(int irq, void *d)
{
#ifndef DTR_IRQ_WORK
	struct quectel_power_manager *data = (struct quectel_power_manager *)d;
	if (data->wakeup_flag)
#endif
	{
		printk("%s: irq\n", __func__);
	}

#ifdef DTR_IRQ_WORK
	if(queue)
	{
	    queue_work(queue,&work);
	}
#endif
	return IRQ_HANDLED;
}

static irqreturn_t quectel_dtr_irq(int irq, void *d)
{
#ifndef DTR_IRQ_WORK
	struct quectel_power_manager *data = (struct quectel_power_manager *)d;
	if (data->dtr_flag)
#endif
	{
		printk("%s: irq\n", __func__);
	}
#ifdef DTR_IRQ_WORK
	if(queue)
	{
	    queue_work(queue,&work);
	}
#endif

	return IRQ_HANDLED;
}

static int quectel_wakeup_init(struct quectel_power_manager *data)
{
	struct quectel_power_manager_pdata *pdata = data->pdata;
	int err;

	err = gpio_request(pdata->gpio_wakeup, "wakeup_in");
	if (err < 0)
	{
		printk("%s: request gpio: %d error: %d\n", __func__, pdata->gpio_wakeup, err);
		goto err_gpio_request;
	}

	err = gpio_direction_input(pdata->gpio_wakeup);
	if (err < 0)
	{
		printk("%s: set gpio direction input (%d) fail\n", __func__, pdata->gpio_wakeup);
		goto err_gpio_to_irq;
	}

	err = gpio_to_irq(pdata->gpio_wakeup);
	if (err < 0)
	{
		printk("%s: gpio to irq: %d error: %d\n", __func__, pdata->gpio_wakeup, err);
		goto err_gpio_to_irq;
	}
	data->irq_wakeup = err;

	err = request_any_context_irq(data->irq_wakeup, quectel_wakeup_irq, \
			IRQF_TRIGGER_FALLING, "quectel_wakeup_event", data);
	if (err < 0)
	{
		printk("%s: Can't get %d IRQ for wakeup_in: %d\n", __func__, data->irq_wakeup, err);
		goto err_gpio_to_irq;
	}

#ifdef DTR_IRQ_WORK
	enable_irq_wake(data->irq_wakeup);
	printk("init gpio to irq = %d\n",data->irq_dtr);
#else
	disable_irq(data->irq_wakeup);
#endif
	

	return 0;

err_gpio_to_irq:
	gpio_free(pdata->gpio_wakeup);
err_gpio_request:
	return err;
}

#ifdef DTR_ESCAPE_MODE

void send_data_to_smd(int dtr_value,int wakeup_value)
{
  static unsigned cbits_to_modem = 2;
  int i = 0;
  printk("send_data_to_smd\n");
  if(dtr_devp == NULL)
  {
    printk("tr_devp == NULL\n");
	return;
  }
  else
  {
	printk("get dtr_devp continue\n");
  }
  if(dtr_devp->ch == 0)
  {
    printk("tr_devp->chn == NULL\n");
	return;
  }
  else
  {
	printk("get dtr_devp->ch continue\n");
  }
  i = smd_tiocmget(dtr_devp->ch);
  if(i & 0x002)//TIOCM_DTR
  { 
     cbits_to_modem = 0;
  }
  else
  {
     cbits_to_modem = 2;
  }
  if((dtr_devp->dtr_value != dtr_value) || (dtr_devp->wakeup_value != wakeup_value))
  {
    printk("cbits_to_modem = %d,~cbits_to_modem = %d func = %s\n",cbits_to_modem,~cbits_to_modem,__func__);
    printk("pre dtr_devp->dtr_value = %d,dtr_devp->wakeup_value = %d\n",dtr_devp->dtr_value,dtr_devp->wakeup_value);
    
	dtr_devp->dtr_value = dtr_value;
	dtr_devp->wakeup_value = wakeup_value;
    smd_tiocmset(dtr_devp->ch,cbits_to_modem,~cbits_to_modem);
  }
}

static void dtr_notify(void *priv, unsigned event)
{
  switch(event)
  {
    case SMD_EVENT_DATA:
		printk("DTR SMD EVENT DATA\n");
		break;

    case SMD_EVENT_OPEN:
		printk("DTR SMD EVENT OPEN\n");
		break;
  
    case SMD_EVENT_CLOSE:
		printk("DTR SMD EVENT CLOSE\n");
		break;
  }

}

static int dtr_smd_open( void )
{
  int r = 1;//
  mutex_lock(&dtr_ch_lock);
  dtr_devp = &dtr_device;

  if(dtr_devp->ch == 0)
  {
    r = smd_open(DTR_SMD_CHANNEL_NAME,&dtr_devp->ch,dtr_devp,dtr_notify);
    //open success return 0
	printk("dtr_smd_open smd_open_ret = %d(0 SUCCESS)\n",r);
  }
  mutex_unlock(&dtr_ch_lock);
  return r;
}

static int dtr_smd_release( void )
{
  int r = 0;
  mutex_lock(&dtr_ch_lock);
  if(dtr_devp->ch != 0)
  {
    r = smd_close(dtr_devp->ch);
	dtr_devp->ch = 0;
  }
  mutex_unlock(&dtr_ch_lock);
  return r;
}
#endif//DTR_ESCAPE_MODE

static int quectel_dtr_init(struct quectel_power_manager *data)
{
	struct quectel_power_manager_pdata *pdata = data->pdata;
	int err;

	err = gpio_request(pdata->gpio_dtr, "dtr_in");
	if (err < 0)
	{
		printk("%s: request gpio: %d error: %d\n", __func__, pdata->gpio_dtr, err);
		goto err_gpio_request;
	}

	err = gpio_direction_input(pdata->gpio_dtr);
	if (err < 0)
	{
		printk("%s: set gpio direction input (%d) fail\n", __func__, pdata->gpio_dtr);
		goto err_gpio_to_irq;
	}

	err = gpio_to_irq(pdata->gpio_dtr);
	if (err < 0)
	{
		printk("%s: gpio to irq: %d error: %d\n", __func__, pdata->gpio_dtr, err);
		goto err_gpio_to_irq;
	}
	data->irq_dtr = err;
	err = request_any_context_irq(data->irq_dtr, quectel_dtr_irq, \
			IRQF_TRIGGER_FALLING|IRQF_TRIGGER_RISING, "quectel_dtr_event", data);
	if (err < 0)
	{
		printk("%s: Can't get %d IRQ for dtr_in: %d\n", __func__, data->irq_dtr, err);
		goto err_gpio_to_irq;
	}
#ifdef DTR_IRQ_WORK
	enable_irq_wake(data->irq_dtr);
	printk("init gpio to irq = %d\n",data->irq_dtr);
#else
	disable_irq(data->irq_dtr);
#endif
	return 0;

err_gpio_to_irq:
	gpio_free(pdata->gpio_dtr);
err_gpio_request:
	return err;
}

static int quectel_sleep_state_init(struct quectel_power_manager *data)
{
	struct quectel_power_manager_pdata *pdata = data->pdata;
	int err;

	if (!pdata->gpio_sleep_state)
		return 0;

	err = gpio_request(pdata->gpio_sleep_state, "sleep_state_out");
	if (err < 0)
	{
		printk("%s: request gpio: %d error: %d\n", __func__, pdata->gpio_sleep_state, err);
		goto err_gpio_request;
	}

	if (gpio_direction_output(pdata->gpio_sleep_state, 0) < 0)
	{
		printk("%s: set gpio direction output err\n", __func__);
		goto err_gpio_direction_output;
	}

	return 0;

err_gpio_direction_output:
	gpio_free(pdata->gpio_sleep_state);
err_gpio_request:
	return err;
}

static ssize_t quectel_wakeup_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct quectel_power_manager *data = dev_get_drvdata(dev);
	struct quectel_power_manager_pdata *pdata = data->pdata;
	int val;
	ssize_t status;

	mutex_lock(&wakeup_lock);

	val = !!gpio_get_value(pdata->gpio_wakeup);
	status = sprintf(buf, "%d", val);

	mutex_unlock(&wakeup_lock);

	return status;
}

static ssize_t quectel_dtr_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct quectel_power_manager *data = dev_get_drvdata(dev);
	struct quectel_power_manager_pdata *pdata = data->pdata;
	int val;
	ssize_t status;

#ifdef DTR_ESCAPE_MODE//francis
	//open smd data3
	int ret = dtr_smd_open();
#endif//DTTR_ESCAPE_MODE
	mutex_lock(&dtr_lock);

	val = !!gpio_get_value(pdata->gpio_dtr);
	status = sprintf(buf, "%d", val);

#ifdef DTR_ESCAPE_MODE
   if(ret > 0)//==0 open smd, <0 open failed >0 had opensmd
   {
     //send_data_to_smd(val);
   }
#endif
	mutex_unlock(&dtr_lock);

	return status;
}

static DEVICE_ATTR(wakeup_in, 0664, quectel_wakeup_show, NULL);
static DEVICE_ATTR(dtr_in, 0664, quectel_dtr_show, NULL);

static int __devinit quectel_power_manager_probe(struct platform_device *pdev)
{

	struct quectel_power_manager *data;

	struct quectel_power_manager_pdata *pdata = pdev->dev.platform_data;

	data = kzalloc(sizeof *data, GFP_KERNEL);
	if (data == NULL) {
		printk("%s:kzalloc() failed.\n", __func__);
		return -ENOMEM;
	}

#ifdef DTR_IRQ_WORK
	queue = create_singlethread_workqueue("quec_dtr");
	if(!queue)
	{
	  printk("creat dtr workqueue failed\n");
	}
	INIT_WORK(&work,work_handler);
	dtr_data = data;
	dtr_data->pdata = pdata;
#endif

	data->pdata = pdata;
	platform_set_drvdata(pdev, data);
	data->wakeup_flag = 0;
	data->dtr_flag = 0;

	if (quectel_wakeup_init(data) < 0)
	{
		printk("%s: init wakeup gpio fail\n", __func__);
		goto err_wakeup_init;
	}

	if (quectel_sleep_state_init(data) < 0)
	{
		printk("%s: init sleep state gpio fail\n", __func__);
	}

	if (quectel_dtr_init(data) < 0)
	{
		printk("%s: init dtr gpio fail\n", __func__);
	}

	if (device_create_file(&pdev->dev, &dev_attr_wakeup_in) < 0)
	{
		printk("%s: dev file creation for wakeup_in failed\n", __func__);
	}

	if (device_create_file(&pdev->dev, &dev_attr_dtr_in) < 0)
	{
		printk("%s: dev file creation for dtr_in failed\n", __func__);
	}

	printk("%s: OK ###############################\n", __func__);
	return 0;

err_wakeup_init:
	platform_set_drvdata(pdev, NULL);
	kfree(data);

	return -1;
}

static int __devexit quectel_power_manager_remove(struct platform_device *pdev)
{
	struct quectel_power_manager *data = platform_get_drvdata(pdev);
	struct quectel_power_manager_pdata *pdata = data->pdata;

	free_irq(data->irq_wakeup, data);
	gpio_free(pdata->gpio_wakeup);
	disable_irq_wake(data->irq_wakeup);
	platform_set_drvdata(pdev, NULL);
	device_remove_file(&pdev->dev, &dev_attr_wakeup_in);
	device_remove_file(&pdev->dev, &dev_attr_dtr_in);
	kfree(data);
#ifdef DTR_ESCAPE_MODE//francis
	dtr_smd_release();
#endif
#ifdef DTR_IRQ_WORK
	if(queue)
	{
	  destroy_workqueue(queue);
	}
#endif
	return 0;
}

static int __devexit quectel_power_manager_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct quectel_power_manager *data = platform_get_drvdata(pdev);
	struct quectel_power_manager_pdata *pdata = data->pdata;

	if (pdata->gpio_sleep_state)
		gpio_set_value(pdata->gpio_sleep_state, 1);

#ifndef DTR_IRQ_WORK
	enable_irq(data->irq_wakeup);
	if (enable_irq_wake(data->irq_wakeup) < 0)
	{
		printk("%s: enable irq wake fail\n", __func__);
		return -1;
	}
	else
	{
		data->wakeup_flag = 1;
	}
	enable_irq(data->irq_dtr);
	if (enable_irq_wake(data->irq_dtr) < 0)
	{
		printk("%s: enable irq dtr fail\n", __func__);
		return -1;
	}
	else
	{
		data->dtr_flag = 1;
	}
#endif
	return 0;
}

static int __devexit quectel_power_manager_resume(struct platform_device *pdev)
{
	struct quectel_power_manager *data = platform_get_drvdata(pdev);
	struct quectel_power_manager_pdata *pdata = data->pdata;

	if (pdata->gpio_sleep_state)
		gpio_set_value(pdata->gpio_sleep_state, 0);
#ifndef DTR_IRQ_WORK
	disable_irq(data->irq_wakeup);
	if (disable_irq_wake(data->irq_wakeup) < 0)
	{
		printk("%s: disable irq wake fail\n", __func__);
		return -1;
	}

	disable_irq(data->irq_dtr);
	if (disable_irq_wake(data->irq_dtr) < 0)
	{
		printk("%s: disable irq dtr fail\n", __func__);
		return -1;
	}
#endif
	return 0;
}

static struct platform_driver quectel_power_manager_driver = {
	.probe		= quectel_power_manager_probe,
	.remove		= __devexit_p(quectel_power_manager_remove),
	.suspend	= quectel_power_manager_suspend,
	.resume		= quectel_power_manager_resume,
	.driver		= {
		.name = MSM9615_QUECTEL_POWER_MANAGER_NAME,
		.owner = THIS_MODULE,
	},
};

static int __init quectel_power_manager_init(void)
{
	return platform_driver_register(&quectel_power_manager_driver);
}

static void __exit quectel_power_manager_exit(void)
{
	platform_driver_unregister(&quectel_power_manager_driver);
}

module_init(quectel_power_manager_init);
module_exit(quectel_power_manager_exit);

MODULE_LICENSE("GPL v2");
MODULE_DESCRIPTION("QUECTEL SLEEP driver");
MODULE_VERSION("1.0");
MODULE_ALIAS("sam.wu@quectel.com");
