/*
 * spidev.c -- simple synchronous userspace interface to SPI devices
 *
 * Copyright (C) 2006 SWAPP
 *	Andrea Paterniani <a.paterniani@swapp-eng.it>
 * Copyright (C) 2007 David Brownell (simplification, cleanup)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/ioctl.h>
#include <linux/fs.h>
#include <linux/device.h>
#include <linux/err.h>
#include <linux/list.h>
#include <linux/errno.h>
#include <linux/mutex.h>
#include <linux/completion.h>
#include <linux/slab.h>
#include <linux/compat.h>
#include <linux/spi/spi.h>
#include <linux/spi/spidev.h>
#include <linux/kfifo.h>
#include <linux/poll.h>
#include <asm/uaccess.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <linux/workqueue.h>
#include <linux/io.h>
#include <linux/debugfs.h>
#include <asm/atomic.h>
#include <linux/mutex.h>
#include <linux/gpio.h>

#ifndef INIT_COMPLETION
#define INIT_COMPLETION(x) reinit_completion(&(x))
#endif 
#ifndef MIN
#define	MIN(a, b)		(((a) < (b))?(a):(b))
#endif 
#define SPI_MAX_CHN 8

struct quectel_msg {
    u8 flag;  //0x5a
    u8 ch; // 0~15
    u16 len; //0 ~ bufsiz
    u8 data[0];
};

struct quectel_channel_info {
    unsigned int ch_num;
    unsigned int users;
    struct mutex tx_mutex;
    struct mutex rx_mutex;
    u8 *tx_buf;
    ssize_t tx_len;
    int tx_status;
    struct completion tx_done;
    struct list_head tx_queue;
    struct kfifo rx_fifo;
    wait_queue_head_t rx_wq;
    struct device *dev;
};

struct quectel_spidev {
    struct spi_device *spidev;
    spinlock_t dev_lock;
    struct mutex dev_mutex;
    struct list_head msg_queue;
    struct workqueue_struct *msg_workqueue;
    struct work_struct msg_work;
    struct mutex msg_work_mutex;
    struct completion tx_done;
    wait_queue_head_t slave_ready_wq;
    struct quectel_msg *tx_msg;
    struct quectel_msg *rx_msg;
    int gpio_master_ready;
    int gpio_slave_ready;
    int irq_slave_ready;
    int major;
   struct quectel_channel_info ch_info[SPI_MAX_CHN];
};

static struct quectel_spidev *ql_spidev;
static unsigned int frame_size = 512;
static unsigned int speed_hz = 8000000;
static unsigned int spi_mode = SPI_MODE_0;
static unsigned int debug = 0xFF;
module_param(frame_size, uint, 0644);
module_param(speed_hz, uint, 0644);
module_param(spi_mode, uint, 0644);
module_param(debug, uint, 0644);

static int spidev_get_slave_ready(void) {
    return gpio_get_value(ql_spidev->gpio_slave_ready);
}

static void spidev_set_master_ready(int ready) {
    gpio_set_value(ql_spidev->gpio_master_ready, !!ready);
}

static irqreturn_t spidev_slave_ready_irq_handler(int irq, void *para)  {
    struct quectel_spidev *ql_spidev = (struct quectel_spidev *)para;
    wake_up_interruptible(&ql_spidev->slave_ready_wq);
    queue_work(ql_spidev->msg_workqueue, &ql_spidev->msg_work);
    return IRQ_HANDLED;
}

static void spidev_complete(void *arg) {
    complete(arg);
}

/* workqueue - pull messages from queue & process */
static void spidev_workq(struct work_struct *work) {
    struct quectel_spidev *ql_spidev =  container_of(work, struct quectel_spidev, msg_work);
    int status;
    unsigned long flags;

    mutex_lock(&ql_spidev->msg_work_mutex);
    if (debug & 0x4) printk("+++ %s\n", __func__);

#if 1
    spin_lock_irqsave(&ql_spidev->dev_lock, flags);   
    if (spidev_get_slave_ready() && list_empty(&ql_spidev->msg_queue)) {
        ql_spidev->ch_info[0].tx_len = frame_size - sizeof(struct quectel_msg);
        list_add_tail(&ql_spidev->ch_info[0].tx_queue, &ql_spidev->msg_queue);
    }
    spin_unlock_irqrestore(&ql_spidev->dev_lock, flags);
#endif

    while (1) {
        struct quectel_channel_info *tx_ch_info = NULL;
        spin_lock_irqsave(&ql_spidev->dev_lock, flags);   
        if (!list_empty(&ql_spidev->msg_queue)) {

                tx_ch_info = list_entry(ql_spidev->msg_queue.next, struct quectel_channel_info, tx_queue);
                list_del_init(&tx_ch_info->tx_queue);
        }
        spin_unlock_irqrestore(&ql_spidev->dev_lock, flags);
        if (!tx_ch_info)
            break;

        tx_ch_info->tx_status = 0;
        status = -EINVAL;
        while(tx_ch_info->tx_status < tx_ch_info->tx_len) {
            unsigned long long master_ready_usec;
            spidev_set_master_ready(1);
            master_ready_usec = cpu_clock(0) / 1024;
            status = wait_event_interruptible(ql_spidev->slave_ready_wq, spidev_get_slave_ready());
            if (status == 0) {
                struct spi_transfer t = {
                    .tx_buf = ql_spidev->tx_msg,
                    .rx_buf = ql_spidev->rx_msg,
                    .len = frame_size,
                    //.delay_usecs = 0,
                    //.speed_hz = tx_spidev->spidev->max_speed_hz,
                    //.bits_per_word = tx_spidev->spidev->bits_per_word,                    
                };
                struct spi_message m;

                spi_message_init(&m);
                spi_message_add_tail(&t, &m);
                m.complete = spidev_complete;        
                m.context = &ql_spidev->tx_done;                   

                INIT_COMPLETION(tx_ch_info->tx_done);
                ql_spidev->tx_msg->flag = 0x5a;
                ql_spidev->tx_msg->ch = tx_ch_info->ch_num;
                ql_spidev->tx_msg->len = MIN((tx_ch_info->tx_len - tx_ch_info->tx_status), (frame_size - sizeof(struct quectel_msg)));
                if (tx_ch_info->tx_buf)
                    memcpy(ql_spidev->tx_msg->data, tx_ch_info->tx_buf + tx_ch_info->tx_status, ql_spidev->tx_msg->len);
                else
                    ql_spidev->tx_msg->len = 0;

                //printk("> flag: %02x ch: %x len:%d\n", ql_spidev->tx_msg->flag, ql_spidev->tx_msg->ch, ql_spidev->tx_msg->len);
                 //make gpio_master_ready keep high at least 3 usec, make sure slave can receive falling interrupt!
                while ((master_ready_usec + 3) > (cpu_clock(0) / 1024))
                    udelay(1);            
                spidev_set_master_ready(0);
                
                status = spi_async(ql_spidev->spidev, &m);
                if (status == 0) {
                    wait_for_completion(&ql_spidev->tx_done);
                    status = m.status;
                    if (status == 0) {
                        if (/*m.actual_length && */(ql_spidev->rx_msg->flag == 0x5a) && (ql_spidev->rx_msg->ch < SPI_MAX_CHN)) {
                            struct quectel_channel_info *rx_ch_info = NULL;
                            //ql_spidev->rx_msg->ch = jiffies % SPI_NUM_CHIPSELECTS;
                            rx_ch_info = &ql_spidev->ch_info[ql_spidev->rx_msg->ch];
                            if (rx_ch_info && rx_ch_info->users) {
                                if (ql_spidev->rx_msg->len <= kfifo_avail(&rx_ch_info->rx_fifo)) {
                                    kfifo_in(&rx_ch_info->rx_fifo, ql_spidev->rx_msg->data, ql_spidev->rx_msg->len);
                                    wake_up_interruptible(&rx_ch_info->rx_wq);
                                } else {
                                    if (debug & 0x4) dev_info(rx_ch_info->dev,  "over run!\n");
                                }
                            } else {
                              if (debug & 0x4) dev_info(rx_ch_info->dev,  "no users\n");
                            }
                        } else {
                           if (debug & 0x4) printk("< flag: %02x ch: %x len:%d\n", ql_spidev->rx_msg->flag, ql_spidev->rx_msg->ch, ql_spidev->rx_msg->len);
                        }  
                        tx_ch_info->tx_status += ql_spidev->tx_msg->len ? ql_spidev->tx_msg->len : tx_ch_info->tx_len;
                    }
                }                       
            }

            if (status)
                break;
        }

        if (status && !tx_ch_info->tx_status )
            tx_ch_info->tx_status = status;
        complete(&tx_ch_info->tx_done);
    }
    
    if (debug & 0x4) printk("--- %s\n", __func__);
    mutex_unlock(&ql_spidev->msg_work_mutex);
}

static int spidev_sync_write(struct quectel_channel_info *ch_info, size_t len) {
    unsigned long flags;
    INIT_COMPLETION(ch_info->tx_done);

    spin_lock_irqsave(&ql_spidev->dev_lock, flags);
    list_add_tail(&ch_info->tx_queue, &ql_spidev->msg_queue);
    spin_unlock_irqrestore(&ql_spidev->dev_lock, flags);
    queue_work(ql_spidev->msg_workqueue, &ql_spidev->msg_work);
    
    wait_for_completion(&ch_info->tx_done);
    return ch_info->tx_status;
}

/*-------------------------------------------------------------------------*/
static unsigned int spidev_poll(struct file *filp, struct poll_table_struct *wait) {
    struct quectel_channel_info *ch_info = (struct quectel_channel_info *)  filp->private_data;
    unsigned int mask = 0;
    mutex_lock(&ch_info->rx_mutex);
    poll_wait(filp, &ch_info->rx_wq, wait);
    if(!kfifo_is_empty(&ch_info->rx_fifo))
        mask |=  POLLIN | POLLRDNORM;
    if (debug & 0x2) dev_info(ch_info->dev, "%s mask %x\n", __func__, mask);
    mutex_unlock(&ch_info->rx_mutex);

    return mask;
}

/* Read-only message with current device setup */
static ssize_t spidev_read(struct file *filp, char __user *buf, size_t count, loff_t *f_pos) {
    struct quectel_channel_info *ch_info = (struct quectel_channel_info *)  filp->private_data;
    int status = 0, copied;

    mutex_lock(&ch_info->rx_mutex);
#if 1 //block util have data to read
    while (kfifo_is_empty(&ch_info->rx_fifo))
        wait_event_interruptible(ch_info->rx_wq, !kfifo_is_empty(&ch_info->rx_fifo));
#endif
    status = kfifo_to_user(&ch_info->rx_fifo, buf, count, &copied);
    if (debug & 0x2) dev_info(ch_info->dev, "%s status = %d\n", __func__, copied);
    mutex_unlock(&ch_info->rx_mutex);

    return status ? status : (copied ? copied : -EAGAIN);
}

/* Write-only message with current device setup */
static ssize_t spidev_write(struct file *filp, const char __user *buf, size_t count, loff_t *f_pos) {
    struct quectel_channel_info *ch_info = (struct quectel_channel_info *)  filp->private_data;
    ssize_t	 status = 0;
    
    if (count > PAGE_SIZE)
        count = PAGE_SIZE;

    mutex_lock(&ch_info->tx_mutex);
    if (debug & 0x2) dev_info(ch_info->dev, "%s\n", __func__);
    if (copy_from_user(ch_info->tx_buf, buf, count))
        status = -EFAULT;
#if 0 //for at debug
    else if (ch_info->ch_num == 1) {
        dev_info(ch_info->dev, "AT > %s", ch_info->tx_buf);
        status = count;
   } else if (ch_info->ch_num == 4) {
        struct quectel_channel_info *rx_ch_info = &ql_spidev->ch_info[1];
        dev_info(ch_info->dev, "AT < %s", ch_info->tx_buf);
        kfifo_in(&rx_ch_info->rx_fifo, ch_info->tx_buf, count);
        wake_up_interruptible(&rx_ch_info->rx_wq);
        status = count;
    }
#endif
    else  {
        ch_info->tx_len = count;
        status = spidev_sync_write(ch_info, count);
    }
    if (debug & 0x2) dev_info(ch_info->dev, "%s status = %ld\n", __func__, (long)status);
    mutex_unlock(&ch_info->tx_mutex);

    return status;
}

static int spidev_open(struct inode *inode, struct file *filp) {
    struct quectel_channel_info *ch_info = &ql_spidev->ch_info[MINOR(inode->i_rdev)];
    int status = 0;

    mutex_lock(&ql_spidev->dev_mutex);
    if (debug) dev_info(ch_info->dev, "+++ %s\n", __func__);

    if (!ch_info->tx_buf) {
        ch_info->tx_buf = kmalloc(PAGE_SIZE, GFP_KERNEL);
        if (!ch_info->tx_buf) {
            dev_err(ch_info->dev, "tx_buf/ENOMEM\n");
            status = -ENOMEM;
        }

        if (kfifo_alloc(&ch_info->rx_fifo, PAGE_SIZE, GFP_KERNEL)) {
            dev_err(ch_info->dev, "rx_fifo/ENOMEM\n");
            kfree(ch_info->tx_buf);
            ch_info->tx_buf = NULL;
            status = -ENOMEM;
        }       
    }

    if (status == 0) {
        ch_info->users++;
        filp->private_data = ch_info;
        nonseekable_open(inode, filp);
    }

    mutex_unlock(&ql_spidev->dev_mutex);
    return status;
}

static int spidev_release(struct inode *inode, struct file *filp) {
    struct quectel_channel_info *ch_info = (struct quectel_channel_info *)  filp->private_data;
    int status = 0;

    mutex_lock(&ql_spidev->dev_mutex);
    mutex_lock(&ch_info->tx_mutex);
    mutex_lock(&ch_info->rx_mutex);
    if (debug) dev_info(ch_info->dev, "+++ %s\n", __func__);
    filp->private_data = NULL;
    ch_info->users--;
    if (!ch_info->users) {
       // kfree(spidev->tx_buf);
        //if (spidev->rx_buf != spidev->tx_buf)
       //     kfree(spidev->rx_buf);
       // kfifo_free(&spidev->rx_fifo);
    }
    mutex_unlock(&ch_info->rx_mutex);
    mutex_unlock(&ch_info->tx_mutex);
    mutex_unlock(&ql_spidev->dev_mutex);

    return status;
}

static const struct file_operations spidev_fops = {
    .owner =	THIS_MODULE,
    .write =	spidev_write,
    .read =    	spidev_read,
    .poll = spidev_poll,
    .open =    	spidev_open,
    .release =	spidev_release,
    .llseek =	no_llseek,
};


static struct class *spidev_class;
static int spidev_probe(struct spi_device *spi) {
    int status;
    int i;

    mutex_lock(&ql_spidev->dev_mutex);
    if (debug) dev_info(&spi->dev, "+++ %s\n", __func__);
    ql_spidev->spidev = spi;
    spi_set_drvdata(spi, ql_spidev);
    
    spi->mode = spi_mode;
    spi->max_speed_hz = speed_hz;
    status = spi_setup(spi);
    dev_info(&spi->dev, "setup mode %d, %s%s%s%s%u bits/w, %u Hz max --> %d\n",
    		(int) (spi->mode & (SPI_CPOL | SPI_CPHA)),
    		(spi->mode & SPI_CS_HIGH) ? "cs_high, " : "",
    		(spi->mode & SPI_LSB_FIRST) ? "lsb, " : "",
    		(spi->mode & SPI_3WIRE) ? "3wire, " : "",
    		(spi->mode & SPI_LOOP) ? "loopback, " : "",
    		spi->bits_per_word, spi->max_speed_hz,
    		status);

    for (i = 0; i < SPI_MAX_CHN; i++) {
        struct quectel_channel_info *ch_info = &ql_spidev->ch_info[i];
        struct device *dev = NULL;

        if (i) {
            dev = device_create(spidev_class, &spi->dev, MKDEV(ql_spidev->major, i), ql_spidev, "spichn%d", i);
            status = IS_ERR(dev) ? PTR_ERR(dev) : 0;
            if (status)
                break;
        }
        ch_info->dev = dev;
        ch_info->ch_num = i;
        mutex_init(&ch_info->tx_mutex);
        mutex_init(&ch_info->rx_mutex);
        init_completion(&ch_info->tx_done);
        INIT_LIST_HEAD(&ch_info->tx_queue);
        init_waitqueue_head(&ch_info->rx_wq);
    }
    
    if (debug) dev_info(&spi->dev, "%s status = %d\n", __func__, status);
    mutex_unlock(&ql_spidev->dev_mutex);
    enable_irq(ql_spidev->irq_slave_ready);
  
    return status;
}

static int spidev_remove(struct spi_device *spi) {
    struct quectel_spidev *ql_spidev = spi_get_drvdata(spi);
    unsigned long flags;
    int i;
    if (debug) dev_info(&spi->dev, "%s\n", __func__);
    /* make sure ops on existing fds can abort cleanly */
    spin_lock_irqsave(&ql_spidev->dev_lock, flags);
    ql_spidev->spidev = NULL;
    spin_unlock_irqrestore(&ql_spidev->dev_lock, flags);

    for (i = 1; i < SPI_MAX_CHN; i++) {
        device_destroy(spidev_class, MKDEV(ql_spidev->major, i));
    }

    return 0;
}


static struct spi_driver spidev_spi_driver = {
    .driver = {
        .name =        "spichn",
        .owner =	THIS_MODULE,
    },
    .probe =	spidev_probe,
    .remove = spidev_remove,
};

/*-------------------------------------------------------------------------*/

static int __init spidev_init(void) {
    int status;
    struct spi_master *master;
    struct spi_board_info chip = {
        .modalias = "spichn",
        .mode = spi_mode,
        .bus_num = 0,
        .chip_select = 0,
        .max_speed_hz = speed_hz,
    };

    ql_spidev = kzalloc(sizeof(struct quectel_spidev), GFP_KERNEL);
    if (ql_spidev == NULL)
        return -ENOMEM;

    spin_lock_init(&ql_spidev->dev_lock);
    mutex_init(&ql_spidev->dev_mutex);
    INIT_LIST_HEAD(&ql_spidev->msg_queue);
    INIT_WORK(&ql_spidev->msg_work, spidev_workq);
    mutex_init(&ql_spidev->msg_work_mutex);
    init_completion(&ql_spidev->tx_done);
    init_waitqueue_head(&ql_spidev->slave_ready_wq);
    ql_spidev->msg_workqueue = create_singlethread_workqueue("spidev");
    ql_spidev->tx_msg = (struct quectel_msg *) __get_free_page(GFP_KERNEL | GFP_DMA);
    ql_spidev->rx_msg = (struct quectel_msg *) __get_free_page(GFP_KERNEL | GFP_DMA);
    if (!ql_spidev->tx_msg || !ql_spidev->tx_msg) {
        kfree(ql_spidev);
        return -ENOMEM;
    }
    ql_spidev->gpio_master_ready = 18;
    ql_spidev->gpio_slave_ready = 16;

    status = gpio_request(ql_spidev->gpio_master_ready, "QUECTEL_SPIDEV_MRDY");
    if (status) {
        if (debug) printk("Fail to request gpio_master_ready PIN %d.\n", ql_spidev->gpio_master_ready);
    } else {
        gpio_direction_output(ql_spidev->gpio_master_ready, 0);
    }
    
    status = gpio_request(ql_spidev->gpio_slave_ready, "QUECTEL_SPIDEV_SRDY");
    if (status) {
        if (debug) printk("Fail to request gpio_slave_ready PIN %d.\n", ql_spidev->gpio_slave_ready);
    } else {
         gpio_direction_input(ql_spidev->gpio_slave_ready);
         status = request_irq(gpio_to_irq(ql_spidev->gpio_slave_ready), spidev_slave_ready_irq_handler,
                                            IRQF_TRIGGER_RISING | IRQF_DISABLED, "ql-spi", ql_spidev);
        if (status) {
            if (debug) printk("Fail to request gpio_slave_ready IRQ\n");
        } else {
            ql_spidev->irq_slave_ready = gpio_to_irq(ql_spidev->gpio_slave_ready);
            disable_irq(ql_spidev->irq_slave_ready);
        }
    }

    status = register_chrdev(0, "spichn", &spidev_fops);
    if (status < 0)
        return status;
    ql_spidev->major = status;

    spidev_class = class_create(THIS_MODULE, "spichn");
    if (IS_ERR(spidev_class)) {
        status = PTR_ERR(spidev_class);
        goto error_class;
    }

    status = spi_register_driver(&spidev_spi_driver);
    if (status < 0)
        goto error_register;

    master = spi_busnum_to_master(0);
    if (!master) {
        status = -ENODEV;
        goto error_busnum;
    }
    spi_new_device(master, &chip);
    
    return 0;
    
error_busnum:
    spi_unregister_driver(&spidev_spi_driver);
error_register:
    class_destroy(spidev_class);
error_class:
    unregister_chrdev(ql_spidev->major, spidev_spi_driver.driver.name);
    free_page((unsigned long) ql_spidev->rx_msg);
    free_page((unsigned long) ql_spidev->tx_msg);
    kfree(ql_spidev);
    return status;
}
module_init(spidev_init);

static void __exit spidev_exit(void) {
    spi_unregister_driver(&spidev_spi_driver);
    class_destroy(spidev_class);
    unregister_chrdev(ql_spidev->major, spidev_spi_driver.driver.name);
    gpio_free(ql_spidev->gpio_master_ready);
    gpio_free(ql_spidev->gpio_slave_ready);
    if (ql_spidev->irq_slave_ready)
        free_irq(ql_spidev->irq_slave_ready, ql_spidev);
    free_page((unsigned long)ql_spidev->tx_msg);
    free_page((unsigned long)ql_spidev->rx_msg);
    kfree(ql_spidev);    
}
module_exit(spidev_exit);

MODULE_LICENSE("GPL"); 
