
/* Copyright (c) 2009, The Linux Foundation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/time.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/spinlock.h>
#include <linux/hrtimer.h>
#include <linux/clk.h>
#include <mach/hardware.h>
#include <linux/io.h>
#include <linux/debugfs.h>

#include <asm/system.h>
#include <asm/mach-types.h>
#include <linux/semaphore.h>
#include <linux/uaccess.h>

#include "mdp.h"
#include "msm_fb.h"
#include "mdp4.h"

void mdp4_sw_reset(ulong bits)
{
	bits &= 0x1f;	/* 5 bits */
	outpdw(MDP_BASE + 0x001c, bits);	/* MDP_SW_RESET */

	while (inpdw(MDP_BASE + 0x001c) & bits) /* self clear when complete */
		;
	MSM_FB_INFO("mdp4_sw_reset: 0x%x\n", (int)bits);
}

void mdp4_overlay_cfg(int overlayer, int blt_mode, int refresh, int direct_out)
{
	ulong bits = 0;

	if (blt_mode)
		bits |= (1 << 3);
	refresh &= 0x03;	/* 2 bites */
	bits |= (refresh << 1);
	direct_out &= 0x01;
	bits |= direct_out;

	if (overlayer == MDP4_MIXER0)
		outpdw(MDP_BASE + 0x10004, bits); /* MDP_OVERLAY0_CFG */
	else
		outpdw(MDP_BASE + 0x18004, bits); /* MDP_OVERLAY1_CFG */

	MSM_FB_INFO("mdp4_overlay_cfg: 0x%x\n", (int)inpdw(MDP_BASE + 0x10004));
}

void mdp4_display_intf_sel(int output, ulong intf)
{
	ulong bits, mask;

	bits = inpdw(MDP_BASE + 0x0038);	/* MDP_DISP_INTF_SEL */

	mask = 0x03;	/* 2 bits */
	intf &= 0x03;	/* 2 bits */

	switch (output) {
	case EXTERNAL_INTF_SEL:
		intf <<= 4;
		mask <<= 4;
		break;
	case SECONDARY_INTF_SEL:
		intf &= 0x02;	/* only MDDI and EBI2 support */
		intf <<= 2;
		mask <<= 2;
		break;
	default:
		break;
	}


	bits &= ~mask;
	bits |= intf;

	outpdw(MDP_BASE + 0x0038, bits);	/* MDP_DISP_INTF_SEL */

  MSM_FB_INFO("mdp4_display_intf_sel: 0x%x\n", (int)inpdw(MDP_BASE + 0x0038));
}

unsigned long mdp4_display_status(void)
{
	return inpdw(MDP_BASE + 0x0018) & 0x3ff;	/* MDP_DISPLAY_STATUS */
}

void mdp4_ebi2_lcd_setup(int lcd, ulong base, int ystride)
{
	/* always use memory map */
	ystride &= 0x01fff;	/* 13 bits */
	if (lcd == EBI2_LCD0) {
		outpdw(MDP_BASE + 0x0060, base);/* MDP_EBI2_LCD0 */
		outpdw(MDP_BASE + 0x0068, ystride);/* MDP_EBI2_LCD0_YSTRIDE */
	} else {
		outpdw(MDP_BASE + 0x0064, base);/* MDP_EBI2_LCD1 */
		outpdw(MDP_BASE + 0x006c, ystride);/* MDP_EBI2_LCD1_YSTRIDE */
	}
}

void mdp4_mddi_setup(int mddi, unsigned long id)
{
	ulong 	bits;

	if (mddi == MDDI_EXTERNAL_SET)
		bits = 0x02;
	else if (mddi == MDDI_SECONDARY_SET)
		bits = 0x01;
	else
		bits = 0;	/* PRIMARY_SET */

	id <<= 16;

	bits |= id;

	outpdw(MDP_BASE + 0x0090, bits); /* MDP_MDDI_PARAM_WR_SEL */
}

int mdp_ppp_blit(struct fb_info *info, struct mdp_blit_req *req,
	struct file **pp_src_file, struct file **pp_dst_file)
{

	/* not implemented yet */
	return -1;
}

void mdp4_hw_init(void)
{
	ulong bits;

	/* MDP cmd block enable */
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);

#ifdef MDP4_ERROR
	/*
	 * Issue software reset on DMA_P will casue DMA_P dma engine stall
	 * on LCDC mode. However DMA_P does not stall at MDDI mode.
	 * This need further investigation.
	 */
	mdp4_sw_reset(0x17);
#endif

	mdp4_clear_lcdc();

	mdp4_mixer_blend_init(0);
	mdp4_mixer_blend_init(1);
	mdp4_vg_qseed_init(0);
	mdp4_vg_qseed_init(1);
	mdp4_vg_csc_mv_setup(0);
	mdp4_vg_csc_mv_setup(1);
	mdp4_vg_csc_pre_bv_setup(0);
	mdp4_vg_csc_pre_bv_setup(1);
	mdp4_vg_csc_post_bv_setup(0);
	mdp4_vg_csc_post_bv_setup(1);
	mdp4_vg_csc_pre_lv_setup(0);
	mdp4_vg_csc_pre_lv_setup(1);
	mdp4_vg_csc_post_lv_setup(0);
	mdp4_vg_csc_post_lv_setup(1);

	mdp4_mixer_gc_lut_setup(0);
	mdp4_mixer_gc_lut_setup(1);

	mdp4_vg_igc_lut_setup(0);
	mdp4_vg_igc_lut_setup(1);

	 mdp4_rgb_igc_lut_setup(0);
	 mdp4_rgb_igc_lut_setup(1);

	outp32(MDP_EBI2_PORTMAP_MODE, 0x3);

	/* system interrupts */

	bits =  mdp_intr_mask;
	outpdw(MDP_BASE + 0x0050, bits);/* enable specififed interrupts */

	/* histogram */
	MDP_OUTP(MDP_BASE + 0x95010, 1);	/* auto clear HIST */

	/* enable histogram interrupts */
	outpdw(MDP_BASE + 0x9501c, INTR_HIST_DONE);

	/* For the max read pending cmd config below, if the MDP clock     */
	/* is less than the AXI clock, then we must use 3 pending          */
	/* pending requests.  Otherwise, we should use 8 pending requests. */
	/* In the future we should do this detection automatically.	   */

	/* max read pending cmd config */
	outpdw(MDP_BASE + 0x004c, 0x02222);	/* 3 pending requests */

	/* dma_p fetch config */
	outpdw(MDP_BASE + 0x91004, 0x27);	/* burst size of 8 */

#ifndef CONFIG_FB_MSM_OVERLAY
	/* both REFRESH_MODE and DIRECT_OUT are ignored at BLT mode */
	mdp4_overlay_cfg(MDP4_MIXER0, OVERLAY_MODE_BLT, 0, 0);
	mdp4_overlay_cfg(MDP4_MIXER1, OVERLAY_MODE_BLT, 0, 0);
#endif

	/* MDP cmd block disable */
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
}


void mdp4_clear_lcdc(void)
{
	uint32 bits;

	bits = inpdw(MDP_BASE + 0xc0000);
	if (bits & 0x01) /* enabled already */
		return;

	outpdw(MDP_BASE + 0xc0004, 0);	/* vsync ctrl out */
	outpdw(MDP_BASE + 0xc0008, 0);	/* vsync period */
	outpdw(MDP_BASE + 0xc000c, 0);	/* vsync pusle width */
	outpdw(MDP_BASE + 0xc0010, 0);	/* lcdc display HCTL */
	outpdw(MDP_BASE + 0xc0014, 0);	/* lcdc display v start */
	outpdw(MDP_BASE + 0xc0018, 0);	/* lcdc display v end */
	outpdw(MDP_BASE + 0xc001c, 0);	/* lcdc active hctl */
	outpdw(MDP_BASE + 0xc0020, 0);	/* lcdc active v start */
	outpdw(MDP_BASE + 0xc0024, 0);	/* lcdc active v end */
	outpdw(MDP_BASE + 0xc0028, 0);	/* lcdc board color */
	outpdw(MDP_BASE + 0xc002c, 0);	/* lcdc underflow ctrl */
	outpdw(MDP_BASE + 0xc0030, 0);	/* lcdc hsync skew */
	outpdw(MDP_BASE + 0xc0034, 0);	/* lcdc test ctl */
	outpdw(MDP_BASE + 0xc0038, 0);	/* lcdc ctl polarity */
}

static struct mdp_dma_data overlay1_data;
static int intr_dma_p;
static int intr_dma_s;
static int intr_dma_e;
static int intr_overlay0;
static int intr_overlay1;

irqreturn_t mdp4_isr(int irq, void *ptr)
{
	uint32 isr, mask, lcdc;
	struct mdp_dma_data *dma;

	mdp_is_in_isr = TRUE;

	while (1) {
		isr = inpdw(MDP_INTR_STATUS);
		if (isr == 0)
			break;

		mask = inpdw(MDP_INTR_ENABLE);
		outpdw(MDP_INTR_CLEAR, isr);

		isr &= mask;

		if (unlikely(isr == 0))
			break;

		if (isr & INTR_DMA_P_DONE) {
			intr_dma_p++;
			lcdc = inpdw(MDP_BASE + 0xc0000);
			dma = &dma2_data;
			if (lcdc & 0x01) {	/* LCDC enable */
				/* disable LCDC interrupt */
				mdp_intr_mask &= ~INTR_DMA_P_DONE;
				outp32(MDP_INTR_ENABLE, mdp_intr_mask);
				dma->waiting = FALSE;
			} else {
				dma->busy = FALSE;
				mdp_pipe_ctrl(MDP_DMA2_BLOCK,
					MDP_BLOCK_POWER_OFF, TRUE);
			}
			complete(&dma->comp);
		}
		if (isr & INTR_DMA_S_DONE) {
			intr_dma_s++;
			dma = &dma_s_data;
			dma->busy = FALSE;
			mdp_pipe_ctrl(MDP_DMA_S_BLOCK,
					MDP_BLOCK_POWER_OFF, TRUE);
			complete(&dma->comp);
		}
		if (isr & INTR_DMA_E_DONE) {
			intr_dma_e++;
			dma = &dma_e_data;
			mdp_intr_mask &= ~INTR_DMA_E_DONE;
			outp32(MDP_INTR_ENABLE, mdp_intr_mask);
			dma->busy = FALSE;

			if (dma->waiting) {
				dma->waiting = FALSE;
				complete(&dma->comp);
			}
		}
		if (isr & INTR_OVERLAY0_DONE) {
			intr_overlay0++;
			lcdc = inpdw(MDP_BASE + 0xc0000);
			dma = &dma2_data;
			if (lcdc & 0x01) {	/* LCDC enable */
				/* disable LCDC interrupt */
				mdp_intr_mask &= ~INTR_OVERLAY0_DONE;
				outp32(MDP_INTR_ENABLE, mdp_intr_mask);
				dma->waiting = FALSE;
				mdp4_overlay0_done_lcdc();
			} else {	/* MDDI */
				dma->busy = FALSE;
#ifdef MDP4_NONBLOCKING
				mdp_disable_irq_nolock(MDP_OVERLAY0_TERM);
#endif
				mdp_pipe_ctrl(MDP_OVERLAY0_BLOCK,
					MDP_BLOCK_POWER_OFF, TRUE);
				mdp4_overlay0_done_mddi();
			}
		}
		if (isr & INTR_OVERLAY1_DONE) {
			intr_overlay1++;
			dma = &overlay1_data;
			dma->busy = FALSE;
			mdp_pipe_ctrl(MDP_OVERLAY1_BLOCK,
					MDP_BLOCK_POWER_OFF, TRUE);
			complete(&dma->comp);
		}
		if (isr & INTR_DMA_P_HISTOGRAM) {
			isr = inpdw(MDP_DMA_P_HIST_INTR_STATUS);
			mask = inpdw(MDP_DMA_P_HIST_INTR_ENABLE);
			outpdw(MDP_DMA_P_HIST_INTR_CLEAR, isr);
			isr &= mask;
			if (isr & INTR_HIST_DONE) {
				if (mdp_hist.r)
					memcpy(mdp_hist.r, MDP_BASE + 0x95100,
							mdp_hist.bin_cnt*4);
				if (mdp_hist.g)
					memcpy(mdp_hist.g, MDP_BASE + 0x95200,
							mdp_hist.bin_cnt*4);
				if (mdp_hist.b)
					memcpy(mdp_hist.b, MDP_BASE + 0x95300,
						mdp_hist.bin_cnt*4);
				complete(&mdp_hist_comp);
			}
		}
	}

	mdp_is_in_isr = FALSE;

	return IRQ_HANDLED;
}


/*
 * QSEED tables
 */

static uint32 vg_qseed_table0[] = {
	0x5556aaff, 0x00000000, 0x00000000, 0x00000000
};

static uint32 vg_qseed_table1[] = {
	0x76543210, 0xfedcba98
};

static uint32 vg_qseed_table2[] = {
	0x02000000, 0x00000000, 0x02060ff2, 0x00000008,
	0x02090fe4, 0x00000013, 0x020a0fd9, 0x0ffc0021,
	0x02080fce, 0x0ffa0030, 0x02030fc5, 0x0ff60042,
	0x01fd0fbe, 0x0ff10054, 0x01f50fb6, 0x0fed0068,
	0x01e90fb1, 0x0fe60080, 0x01dc0fae, 0x0fe10095,
	0x01ca0fae, 0x0fda00ae, 0x01b70fad, 0x0fd600c6,
	0x01a40fad, 0x0fcf00e0, 0x018f0faf, 0x0fc800fa,
	0x01780fb1, 0x0fc30114, 0x015f0fb5, 0x0fbf012d,
	0x01490fb7, 0x0fb70149, 0x012d0fbf, 0x0fb5015f,
	0x01140fc3, 0x0fb10178, 0x00fa0fc8, 0x0faf018f,
	0x00e00fcf, 0x0fad01a4, 0x00c60fd6, 0x0fad01b7,
	0x00ae0fda, 0x0fae01ca, 0x00950fe1, 0x0fae01dc,
	0x00800fe6, 0x0fb101e9, 0x00680fed, 0x0fb601f5,
	0x00540ff1, 0x0fbe01fd, 0x00420ff6, 0x0fc50203,
	0x00300ffa, 0x0fce0208, 0x00210ffc, 0x0fd9020a,
	0x00130000, 0x0fe40209, 0x00080000, 0x0ff20206,
	0x02000000, 0x00000000, 0x02040ff2, 0x0000000a,
	0x02040fe4, 0x00000018, 0x02010fda, 0x0ffc0029,
	0x01fc0fcf, 0x0ffa003b, 0x01f30fc7, 0x0ff60050,
	0x01e90fc0, 0x0ff20065, 0x01dc0fba, 0x0fee007c,
	0x01cc0fb6, 0x0fe80096, 0x01ba0fb4, 0x0fe400ae,
	0x01a70fb4, 0x0fdd00c8, 0x018f0fb5, 0x0fda00e2,
	0x017a0fb5, 0x0fd400fd, 0x01630fb8, 0x0fce0117,
	0x014c0fba, 0x0fca0130, 0x01320fbf, 0x0fc70148,
	0x011b0fc1, 0x0fc10163, 0x01010fc8, 0x0fc00177,
	0x00e90fcd, 0x0fbd018d, 0x00d10fd1, 0x0fbc01a2,
	0x00ba0fd7, 0x0fbb01b4, 0x00a30fdd, 0x0fbc01c4,
	0x008e0fe1, 0x0fbd01d4, 0x00790fe7, 0x0fbe01e2,
	0x00670feb, 0x0fc001ee, 0x00540ff1, 0x0fc501f6,
	0x00430ff4, 0x0fcb01fe, 0x00340ff8, 0x0fd10203,
	0x00260ffb, 0x0fd80207, 0x001a0ffd, 0x0fe10208,
	0x000f0000, 0x0fea0207, 0x00060000, 0x0ff50205,
	0x02000000, 0x00000000, 0x02020ff2, 0x0000000c,
	0x02000fe4, 0x0000001c, 0x01fa0fda, 0x0ffc0030,
	0x01f10fd0, 0x0ffa0045, 0x01e50fc8, 0x0ff6005d,
	0x01d60fc3, 0x0ff30074, 0x01c60fbd, 0x0fef008e,
	0x01b30fba, 0x0fe900aa, 0x019e0fb9, 0x0fe500c4,
	0x01870fba, 0x0fe000df, 0x016f0fbb, 0x0fdd00f9,
	0x01580fbc, 0x0fd80114, 0x01400fbf, 0x0fd3012e,
	0x01280fc2, 0x0fd00146, 0x010f0fc6, 0x0fce015d,
	0x00f90fc9, 0x0fc90175, 0x00e00fcf, 0x0fc90188,
	0x00ca0fd4, 0x0fc6019c, 0x00b40fd8, 0x0fc601ae,
	0x009f0fdd, 0x0fc501bf, 0x008b0fe3, 0x0fc601cc,
	0x00780fe6, 0x0fc701db, 0x00660feb, 0x0fc801e7,
	0x00560fef, 0x0fcb01f0, 0x00460ff3, 0x0fcf01f8,
	0x00380ff6, 0x0fd401fe, 0x002c0ff9, 0x0fd90202,
	0x00200ffc, 0x0fdf0205, 0x00160ffe, 0x0fe60206,
	0x000c0000, 0x0fed0207, 0x00050000, 0x0ff70204,
	0x02000000, 0x00000000, 0x01fe0ff3, 0x0000000f,
	0x01f60fe5, 0x00000025, 0x01ea0fdb, 0x0ffd003e,
	0x01db0fd2, 0x0ffb0058, 0x01c80fcc, 0x0ff70075,
	0x01b50fc7, 0x0ff40090, 0x01a00fc3, 0x0ff000ad,
	0x01880fc1, 0x0feb00cc, 0x01700fc1, 0x0fe800e7,
	0x01550fc3, 0x0fe40104, 0x013b0fc5, 0x0fe2011e,
	0x01240fc6, 0x0fde0138, 0x010c0fca, 0x0fda0150,
	0x00f40fcd, 0x0fd90166, 0x00dd0fd1, 0x0fd7017b,
	0x00c80fd4, 0x0fd40190, 0x00b20fd9, 0x0fd401a1,
	0x009f0fdd, 0x0fd301b1, 0x008c0fe1, 0x0fd301c0,
	0x007b0fe5, 0x0fd301cd, 0x006a0fea, 0x0fd401d8,
	0x005c0fec, 0x0fd501e3, 0x004d0ff0, 0x0fd601ed,
	0x00410ff3, 0x0fd801f4, 0x00340ff7, 0x0fdb01fa,
	0x002a0ff9, 0x0fdf01fe, 0x00200ffb, 0x0fe30202,
	0x00180ffd, 0x0fe70204, 0x00100ffe, 0x0fed0205,
	0x00090000, 0x0ff20205, 0x00040000, 0x0ff90203,
	0x02000000, 0x00000000, 0x02050ff5, 0x00000006,
	0x02070fea, 0x0000000f, 0x02080fe1, 0x0ffd001a,
	0x02070fd8, 0x0ffb0026, 0x02030fd1, 0x0ff80034,
	0x01fe0fcb, 0x0ff40043, 0x01f60fc5, 0x0ff10054,
	0x01ee0fc0, 0x0feb0067, 0x01e20fbe, 0x0fe70079,
	0x01d40fbd, 0x0fe1008e, 0x01c40fbc, 0x0fdd00a3,
	0x01b40fbb, 0x0fd700ba, 0x01a20fbc, 0x0fd100d1,
	0x018d0fbd, 0x0fcd00e9, 0x01770fc0, 0x0fc80101,
	0x01630fc1, 0x0fc1011b, 0x01480fc7, 0x0fbf0132,
	0x01300fca, 0x0fba014c, 0x01170fce, 0x0fb80163,
	0x00fd0fd4, 0x0fb5017a, 0x00e20fda, 0x0fb5018f,
	0x00c80fdd, 0x0fb401a7, 0x00ae0fe4, 0x0fb401ba,
	0x00960fe8, 0x0fb601cc, 0x007c0fee, 0x0fba01dc,
	0x00650ff2, 0x0fc001e9, 0x00500ff6, 0x0fc701f3,
	0x003b0ffa, 0x0fcf01fc, 0x00290ffc, 0x0fda0201,
	0x00180000, 0x0fe40204, 0x000a0000, 0x0ff20204,
	0x02000000, 0x00000000, 0x02030ff5, 0x00000008,
	0x02030fea, 0x00000013, 0x02020fe1, 0x0ffd0020,
	0x01fc0fd9, 0x0ffc002f, 0x01f60fd2, 0x0ff80040,
	0x01ed0fcd, 0x0ff50051, 0x01e30fc7, 0x0ff10065,
	0x01d70fc3, 0x0fec007a, 0x01c60fc2, 0x0fe9008f,
	0x01b60fc1, 0x0fe300a6, 0x01a20fc1, 0x0fe000bd,
	0x018f0fc1, 0x0fdb00d5, 0x017b0fc2, 0x0fd500ee,
	0x01640fc4, 0x0fd20106, 0x014d0fc8, 0x0fce011d,
	0x01370fc9, 0x0fc90137, 0x011d0fce, 0x0fc8014d,
	0x01060fd2, 0x0fc40164, 0x00ee0fd5, 0x0fc2017b,
	0x00d50fdb, 0x0fc1018f, 0x00bd0fe0, 0x0fc101a2,
	0x00a60fe3, 0x0fc101b6, 0x008f0fe9, 0x0fc201c6,
	0x007a0fec, 0x0fc301d7, 0x00650ff1, 0x0fc701e3,
	0x00510ff5, 0x0fcd01ed, 0x00400ff8, 0x0fd201f6,
	0x002f0ffc, 0x0fd901fc, 0x00200ffd, 0x0fe10202,
	0x00130000, 0x0fea0203, 0x00080000, 0x0ff50203,
	0x02000000, 0x00000000, 0x02020ff5, 0x00000009,
	0x01ff0fea, 0x00000017, 0x01fb0fe2, 0x0ffd0026,
	0x01f30fda, 0x0ffc0037, 0x01ea0fd3, 0x0ff8004b,
	0x01df0fce, 0x0ff5005e, 0x01d10fc9, 0x0ff20074,
	0x01c10fc6, 0x0fed008c, 0x01ae0fc5, 0x0fea00a3,
	0x019b0fc5, 0x0fe500bb, 0x01850fc6, 0x0fe200d3,
	0x01700fc6, 0x0fde00ec, 0x015a0fc8, 0x0fd90105,
	0x01430fca, 0x0fd6011d, 0x012b0fcd, 0x0fd30135,
	0x01150fcf, 0x0fcf014d, 0x00fc0fd4, 0x0fce0162,
	0x00e50fd8, 0x0fcc0177, 0x00cf0fdb, 0x0fca018c,
	0x00b80fe0, 0x0fc9019f, 0x00a20fe5, 0x0fca01af,
	0x008e0fe8, 0x0fcb01bf, 0x00790fec, 0x0fcb01d0,
	0x00670fef, 0x0fcd01dd, 0x00550ff4, 0x0fd001e7,
	0x00440ff7, 0x0fd501f0, 0x00350ffa, 0x0fda01f7,
	0x00270ffc, 0x0fdf01fe, 0x001b0ffe, 0x0fe70200,
	0x00100000, 0x0fee0202, 0x00060000, 0x0ff70203,
	0x02000000, 0x00000000, 0x01ff0ff5, 0x0000000c,
	0x01f80fea, 0x0000001e, 0x01ef0fe2, 0x0ffd0032,
	0x01e20fdb, 0x0ffc0047, 0x01d30fd5, 0x0ff9005f,
	0x01c20fd1, 0x0ff60077, 0x01b00fcd, 0x0ff30090,
	0x019b0fcb, 0x0fef00ab, 0x01850fcb, 0x0fec00c4,
	0x016e0fcc, 0x0fe800de, 0x01550fcd, 0x0fe600f8,
	0x013f0fce, 0x0fe20111, 0x01280fd0, 0x0fdf0129,
	0x01110fd2, 0x0fdd0140, 0x00f90fd6, 0x0fdb0156,
	0x00e40fd8, 0x0fd8016c, 0x00cd0fdd, 0x0fd8017e,
	0x00b80fe0, 0x0fd60192, 0x00a40fe3, 0x0fd601a3,
	0x00910fe7, 0x0fd501b3, 0x007f0feb, 0x0fd601c0,
	0x006e0fed, 0x0fd701ce, 0x005d0ff1, 0x0fd701db,
	0x004f0ff3, 0x0fd901e5, 0x00400ff7, 0x0fdc01ed,
	0x00330ff9, 0x0fe001f4, 0x00280ffb, 0x0fe301fa,
	0x001d0ffd, 0x0fe801fe, 0x00140ffe, 0x0fed0201,
	0x000c0000, 0x0ff20202, 0x00050000, 0x0ff90202,
	0x02000000, 0x00000000, 0x02040ff7, 0x00000005,
	0x02070fed, 0x0000000c, 0x02060fe6, 0x0ffe0016,
	0x02050fdf, 0x0ffc0020, 0x02020fd9, 0x0ff9002c,
	0x01fe0fd4, 0x0ff60038, 0x01f80fcf, 0x0ff30046,
	0x01f00fcb, 0x0fef0056, 0x01e70fc8, 0x0feb0066,
	0x01db0fc7, 0x0fe60078, 0x01cc0fc6, 0x0fe3008b,
	0x01bf0fc5, 0x0fdd009f, 0x01ae0fc6, 0x0fd800b4,
	0x019c0fc6, 0x0fd400ca, 0x01880fc9, 0x0fcf00e0,
	0x01750fc9, 0x0fc900f9, 0x015d0fce, 0x0fc6010f,
	0x01460fd0, 0x0fc20128, 0x012e0fd3, 0x0fbf0140,
	0x01140fd8, 0x0fbc0158, 0x00f90fdd, 0x0fbb016f,
	0x00df0fe0, 0x0fba0187, 0x00c40fe5, 0x0fb9019e,
	0x00aa0fe9, 0x0fba01b3, 0x008e0fef, 0x0fbd01c6,
	0x00740ff3, 0x0fc301d6, 0x005d0ff6, 0x0fc801e5,
	0x00450ffa, 0x0fd001f1, 0x00300ffc, 0x0fda01fa,
	0x001c0000, 0x0fe40200, 0x000c0000, 0x0ff20202,
	0x02000000, 0x00000000, 0x02030ff7, 0x00000006,
	0x02020fee, 0x00000010, 0x02000fe7, 0x0ffe001b,
	0x01fe0fdf, 0x0ffc0027, 0x01f70fda, 0x0ffa0035,
	0x01f00fd5, 0x0ff70044, 0x01e70fd0, 0x0ff40055,
	0x01dd0fcd, 0x0fef0067, 0x01d00fcb, 0x0fec0079,
	0x01bf0fcb, 0x0fe8008e, 0x01af0fca, 0x0fe500a2,
	0x019f0fc9, 0x0fe000b8, 0x018c0fca, 0x0fdb00cf,
	0x01770fcc, 0x0fd800e5, 0x01620fce, 0x0fd400fc,
	0x014d0fcf, 0x0fcf0115, 0x01350fd3, 0x0fcd012b,
	0x011d0fd6, 0x0fca0143, 0x01050fd9, 0x0fc8015a,
	0x00ec0fde, 0x0fc60170, 0x00d30fe2, 0x0fc60185,
	0x00bb0fe5, 0x0fc5019b, 0x00a30fea, 0x0fc501ae,
	0x008c0fed, 0x0fc601c1, 0x00740ff2, 0x0fc901d1,
	0x005e0ff5, 0x0fce01df, 0x004b0ff8, 0x0fd301ea,
	0x00370ffc, 0x0fda01f3, 0x00260ffd, 0x0fe201fb,
	0x00170000, 0x0fea01ff, 0x00090000, 0x0ff50202,
	0x02000000, 0x00000000, 0x02010ff7, 0x00000008,
	0x01ff0fee, 0x00000013, 0x01fb0fe7, 0x0ffe0020,
	0x01f60fe0, 0x0ffc002e, 0x01ed0fda, 0x0ffa003f,
	0x01e40fd6, 0x0ff7004f, 0x01d80fd2, 0x0ff40062,
	0x01ca0fcf, 0x0ff00077, 0x01bb0fcd, 0x0fed008b,
	0x01a90fcd, 0x0fe900a1, 0x01960fcd, 0x0fe600b7,
	0x01830fcd, 0x0fe200ce, 0x016d0fcf, 0x0fde00e6,
	0x01580fd0, 0x0fdb00fd, 0x01410fd3, 0x0fd80114,
	0x012c0fd4, 0x0fd4012c, 0x01140fd8, 0x0fd30141,
	0x00fd0fdb, 0x0fd00158, 0x00e60fde, 0x0fcf016d,
	0x00ce0fe2, 0x0fcd0183, 0x00b70fe6, 0x0fcd0196,
	0x00a10fe9, 0x0fcd01a9, 0x008b0fed, 0x0fcd01bb,
	0x00770ff0, 0x0fcf01ca, 0x00620ff4, 0x0fd201d8,
	0x004f0ff7, 0x0fd601e4, 0x003f0ffa, 0x0fda01ed,
	0x002e0ffc, 0x0fe001f6, 0x00200ffe, 0x0fe701fb,
	0x00130000, 0x0fee01ff, 0x00080000, 0x0ff70201,
	0x02000000, 0x00000000, 0x01ff0ff7, 0x0000000a,
	0x01f90fee, 0x00000019, 0x01f10fe7, 0x0ffe002a,
	0x01e60fe1, 0x0ffd003c, 0x01d90fdc, 0x0ffa0051,
	0x01cc0fd8, 0x0ff70065, 0x01bb0fd5, 0x0ff5007b,
	0x01a80fd3, 0x0ff10094, 0x01950fd2, 0x0fef00aa,
	0x01800fd2, 0x0feb00c3, 0x016a0fd3, 0x0fe900da,
	0x01540fd3, 0x0fe600f3, 0x013f0fd5, 0x0fe2010a,
	0x01280fd7, 0x0fe00121, 0x01100fda, 0x0fde0138,
	0x00fb0fdb, 0x0fdb014f, 0x00e40fdf, 0x0fdb0162,
	0x00ce0fe2, 0x0fd90177, 0x00b90fe4, 0x0fd8018b,
	0x00a50fe8, 0x0fd8019b, 0x00910fec, 0x0fd801ab,
	0x007e0fee, 0x0fd801bc, 0x006c0ff2, 0x0fd901c9,
	0x005c0ff4, 0x0fda01d6, 0x004b0ff7, 0x0fdd01e1,
	0x003c0ff9, 0x0fe001eb, 0x002f0ffb, 0x0fe401f2,
	0x00230ffd, 0x0fe801f8, 0x00180ffe, 0x0fed01fd,
	0x000e0000, 0x0ff20200, 0x00060000, 0x0ff90201,
	0x02000000, 0x00000000, 0x02030ff9, 0x00000004,
	0x02050ff2, 0x00000009, 0x02050fed, 0x0ffe0010,
	0x02040fe7, 0x0ffd0018, 0x02020fe3, 0x0ffb0020,
	0x01fe0fdf, 0x0ff9002a, 0x01fa0fdb, 0x0ff70034,
	0x01f40fd8, 0x0ff30041, 0x01ed0fd6, 0x0ff0004d,
	0x01e30fd5, 0x0fec005c, 0x01d80fd4, 0x0fea006a,
	0x01cd0fd3, 0x0fe5007b, 0x01c00fd3, 0x0fe1008c,
	0x01b10fd3, 0x0fdd009f, 0x01a10fd4, 0x0fd900b2,
	0x01900fd4, 0x0fd400c8, 0x017b0fd7, 0x0fd100dd,
	0x01660fd9, 0x0fcd00f4, 0x01500fda, 0x0fca010c,
	0x01380fde, 0x0fc60124, 0x011e0fe2, 0x0fc5013b,
	0x01040fe4, 0x0fc30155, 0x00e70fe8, 0x0fc10170,
	0x00cc0feb, 0x0fc10188, 0x00ad0ff0, 0x0fc301a0,
	0x00900ff4, 0x0fc701b5, 0x00750ff7, 0x0fcc01c8,
	0x00580ffb, 0x0fd201db, 0x003e0ffd, 0x0fdb01ea,
	0x00250000, 0x0fe501f6, 0x000f0000, 0x0ff301fe,
	0x02000000, 0x00000000, 0x02020ff9, 0x00000005,
	0x02020ff2, 0x0000000c, 0x02010fed, 0x0ffe0014,
	0x01fe0fe8, 0x0ffd001d, 0x01fa0fe3, 0x0ffb0028,
	0x01f40fe0, 0x0ff90033, 0x01ed0fdc, 0x0ff70040,
	0x01e50fd9, 0x0ff3004f, 0x01db0fd7, 0x0ff1005d,
	0x01ce0fd7, 0x0fed006e, 0x01c00fd6, 0x0feb007f,
	0x01b30fd5, 0x0fe70091, 0x01a30fd6, 0x0fe300a4,
	0x01920fd6, 0x0fe000b8, 0x017e0fd8, 0x0fdd00cd,
	0x016c0fd8, 0x0fd800e4, 0x01560fdb, 0x0fd600f9,
	0x01400fdd, 0x0fd20111, 0x01290fdf, 0x0fd00128,
	0x01110fe2, 0x0fce013f, 0x00f80fe6, 0x0fcd0155,
	0x00de0fe8, 0x0fcc016e, 0x00c40fec, 0x0fcb0185,
	0x00ab0fef, 0x0fcb019b, 0x00900ff3, 0x0fcd01b0,
	0x00770ff6, 0x0fd101c2, 0x005f0ff9, 0x0fd501d3,
	0x00470ffc, 0x0fdb01e2, 0x00320ffd, 0x0fe201ef,
	0x001e0000, 0x0fea01f8, 0x000c0000, 0x0ff501ff,
	0x02000000, 0x00000000, 0x02010ff9, 0x00000006,
	0x02000ff2, 0x0000000e, 0x01fd0fed, 0x0ffe0018,
	0x01f80fe8, 0x0ffd0023, 0x01f20fe4, 0x0ffb002f,
	0x01eb0fe0, 0x0ff9003c, 0x01e10fdd, 0x0ff7004b,
	0x01d60fda, 0x0ff4005c, 0x01c90fd9, 0x0ff2006c,
	0x01bc0fd8, 0x0fee007e, 0x01ab0fd8, 0x0fec0091,
	0x019b0fd8, 0x0fe800a5, 0x018b0fd8, 0x0fe400b9,
	0x01770fd9, 0x0fe200ce, 0x01620fdb, 0x0fdf00e4,
	0x014f0fdb, 0x0fdb00fb, 0x01380fde, 0x0fda0110,
	0x01210fe0, 0x0fd70128, 0x010a0fe2, 0x0fd5013f,
	0x00f30fe6, 0x0fd30154, 0x00da0fe9, 0x0fd3016a,
	0x00c30feb, 0x0fd20180, 0x00aa0fef, 0x0fd20195,
	0x00940ff1, 0x0fd301a8, 0x007b0ff5, 0x0fd501bb,
	0x00650ff7, 0x0fd801cc, 0x00510ffa, 0x0fdc01d9,
	0x003c0ffd, 0x0fe101e6, 0x002a0ffe, 0x0fe701f1,
	0x00190000, 0x0fee01f9, 0x000a0000, 0x0ff701ff,
	0x02000000, 0x00000000, 0x01ff0ff9, 0x00000008,
	0x01fb0ff2, 0x00000013, 0x01f50fed, 0x0ffe0020,
	0x01ed0fe8, 0x0ffd002e, 0x01e30fe4, 0x0ffb003e,
	0x01d80fe1, 0x0ff9004e, 0x01cb0fde, 0x0ff70060,
	0x01bc0fdc, 0x0ff40074, 0x01ac0fdb, 0x0ff20087,
	0x019a0fdb, 0x0fef009c, 0x01870fdb, 0x0fed00b1,
	0x01740fdb, 0x0fea00c7, 0x01600fdc, 0x0fe700dd,
	0x014b0fdd, 0x0fe500f3, 0x01350fdf, 0x0fe30109,
	0x01200fe0, 0x0fe00120, 0x01090fe3, 0x0fdf0135,
	0x00f30fe5, 0x0fdd014b, 0x00dd0fe7, 0x0fdc0160,
	0x00c70fea, 0x0fdb0174, 0x00b10fed, 0x0fdb0187,
	0x009c0fef, 0x0fdb019a, 0x00870ff2, 0x0fdb01ac,
	0x00740ff4, 0x0fdc01bc, 0x00600ff7, 0x0fde01cb,
	0x004e0ff9, 0x0fe101d8, 0x003e0ffb, 0x0fe401e3,
	0x002e0ffd, 0x0fe801ed, 0x00200ffe, 0x0fed01f5,
	0x00130000, 0x0ff201fb, 0x00080000, 0x0ff901ff,
	0x02000000, 0x00000000, 0x02060ff2, 0x00000008,
	0x02090fe4, 0x00000013, 0x020a0fd9, 0x0ffc0021,
	0x02080fce, 0x0ffa0030, 0x02030fc5, 0x0ff60042,
	0x01fd0fbe, 0x0ff10054, 0x01f50fb6, 0x0fed0068,
	0x01e90fb1, 0x0fe60080, 0x01dc0fae, 0x0fe10095,
	0x01ca0fae, 0x0fda00ae, 0x01b70fad, 0x0fd600c6,
	0x01a40fad, 0x0fcf00e0, 0x018f0faf, 0x0fc800fa,
	0x01780fb1, 0x0fc30114, 0x015f0fb5, 0x0fbf012d,
	0x01490fb7, 0x0fb70149, 0x012d0fbf, 0x0fb5015f,
	0x01140fc3, 0x0fb10178, 0x00fa0fc8, 0x0faf018f,
	0x00e00fcf, 0x0fad01a4, 0x00c60fd6, 0x0fad01b7,
	0x00ae0fda, 0x0fae01ca, 0x00950fe1, 0x0fae01dc,
	0x00800fe6, 0x0fb101e9, 0x00680fed, 0x0fb601f5,
	0x00540ff1, 0x0fbe01fd, 0x00420ff6, 0x0fc50203,
	0x00300ffa, 0x0fce0208, 0x00210ffc, 0x0fd9020a,
	0x00130000, 0x0fe40209, 0x00080000, 0x0ff20206,
	0x02000000, 0x00000000, 0x02040ff2, 0x0000000a,
	0x02040fe4, 0x00000018, 0x02010fda, 0x0ffc0029,
	0x01fc0fcf, 0x0ffa003b, 0x01f30fc7, 0x0ff60050,
	0x01e90fc0, 0x0ff20065, 0x01dc0fba, 0x0fee007c,
	0x01cc0fb6, 0x0fe80096, 0x01ba0fb4, 0x0fe400ae,
	0x01a70fb4, 0x0fdd00c8, 0x018f0fb5, 0x0fda00e2,
	0x017a0fb5, 0x0fd400fd, 0x01630fb8, 0x0fce0117,
	0x014c0fba, 0x0fca0130, 0x01320fbf, 0x0fc70148,
	0x011b0fc1, 0x0fc10163, 0x01010fc8, 0x0fc00177,
	0x00e90fcd, 0x0fbd018d, 0x00d10fd1, 0x0fbc01a2,
	0x00ba0fd7, 0x0fbb01b4, 0x00a30fdd, 0x0fbc01c4,
	0x008e0fe1, 0x0fbd01d4, 0x00790fe7, 0x0fbe01e2,
	0x00670feb, 0x0fc001ee, 0x00540ff1, 0x0fc501f6,
	0x00430ff4, 0x0fcb01fe, 0x00340ff8, 0x0fd10203,
	0x00260ffb, 0x0fd80207, 0x001a0ffd, 0x0fe10208,
	0x000f0000, 0x0fea0207, 0x00060000, 0x0ff50205,
	0x02000000, 0x00000000, 0x02020ff2, 0x0000000c,
	0x02000fe4, 0x0000001c, 0x01fa0fda, 0x0ffc0030,
	0x01f10fd0, 0x0ffa0045, 0x01e50fc8, 0x0ff6005d,
	0x01d60fc3, 0x0ff30074, 0x01c60fbd, 0x0fef008e,
	0x01b30fba, 0x0fe900aa, 0x019e0fb9, 0x0fe500c4,
	0x01870fba, 0x0fe000df, 0x016f0fbb, 0x0fdd00f9,
	0x01580fbc, 0x0fd80114, 0x01400fbf, 0x0fd3012e,
	0x01280fc2, 0x0fd00146, 0x010f0fc6, 0x0fce015d,
	0x00f90fc9, 0x0fc90175, 0x00e00fcf, 0x0fc90188,
	0x00ca0fd4, 0x0fc6019c, 0x00b40fd8, 0x0fc601ae,
	0x009f0fdd, 0x0fc501bf, 0x008b0fe3, 0x0fc601cc,
	0x00780fe6, 0x0fc701db, 0x00660feb, 0x0fc801e7,
	0x00560fef, 0x0fcb01f0, 0x00460ff3, 0x0fcf01f8,
	0x00380ff6, 0x0fd401fe, 0x002c0ff9, 0x0fd90202,
	0x00200ffc, 0x0fdf0205, 0x00160ffe, 0x0fe60206,
	0x000c0000, 0x0fed0207, 0x00050000, 0x0ff70204,
	0x02000000, 0x00000000, 0x01fe0ff3, 0x0000000f,
	0x01f60fe5, 0x00000025, 0x01ea0fdb, 0x0ffd003e,
	0x01db0fd2, 0x0ffb0058, 0x01c80fcc, 0x0ff70075,
	0x01b50fc7, 0x0ff40090, 0x01a00fc3, 0x0ff000ad,
	0x01880fc1, 0x0feb00cc, 0x01700fc1, 0x0fe800e7,
	0x01550fc3, 0x0fe40104, 0x013b0fc5, 0x0fe2011e,
	0x01240fc6, 0x0fde0138, 0x010c0fca, 0x0fda0150,
	0x00f40fcd, 0x0fd90166, 0x00dd0fd1, 0x0fd7017b,
	0x00c80fd4, 0x0fd40190, 0x00b20fd9, 0x0fd401a1,
	0x009f0fdd, 0x0fd301b1, 0x008c0fe1, 0x0fd301c0,
	0x007b0fe5, 0x0fd301cd, 0x006a0fea, 0x0fd401d8,
	0x005c0fec, 0x0fd501e3, 0x004d0ff0, 0x0fd601ed,
	0x00410ff3, 0x0fd801f4, 0x00340ff7, 0x0fdb01fa,
	0x002a0ff9, 0x0fdf01fe, 0x00200ffb, 0x0fe30202,
	0x00180ffd, 0x0fe70204, 0x00100ffe, 0x0fed0205,
	0x00090000, 0x0ff20205, 0x00040000, 0x0ff90203,
	0x02000000, 0x00000000, 0x02050ff5, 0x00000006,
	0x02070fea, 0x0000000f, 0x02080fe1, 0x0ffd001a,
	0x02070fd8, 0x0ffb0026, 0x02030fd1, 0x0ff80034,
	0x01fe0fcb, 0x0ff40043, 0x01f60fc5, 0x0ff10054,
	0x01ee0fc0, 0x0feb0067, 0x01e20fbe, 0x0fe70079,
	0x01d40fbd, 0x0fe1008e, 0x01c40fbc, 0x0fdd00a3,
	0x01b40fbb, 0x0fd700ba, 0x01a20fbc, 0x0fd100d1,
	0x018d0fbd, 0x0fcd00e9, 0x01770fc0, 0x0fc80101,
	0x01630fc1, 0x0fc1011b, 0x01480fc7, 0x0fbf0132,
	0x01300fca, 0x0fba014c, 0x01170fce, 0x0fb80163,
	0x00fd0fd4, 0x0fb5017a, 0x00e20fda, 0x0fb5018f,
	0x00c80fdd, 0x0fb401a7, 0x00ae0fe4, 0x0fb401ba,
	0x00960fe8, 0x0fb601cc, 0x007c0fee, 0x0fba01dc,
	0x00650ff2, 0x0fc001e9, 0x00500ff6, 0x0fc701f3,
	0x003b0ffa, 0x0fcf01fc, 0x00290ffc, 0x0fda0201,
	0x00180000, 0x0fe40204, 0x000a0000, 0x0ff20204,
	0x02000000, 0x00000000, 0x02030ff5, 0x00000008,
	0x02030fea, 0x00000013, 0x02020fe1, 0x0ffd0020,
	0x01fc0fd9, 0x0ffc002f, 0x01f60fd2, 0x0ff80040,
	0x01ed0fcd, 0x0ff50051, 0x01e30fc7, 0x0ff10065,
	0x01d70fc3, 0x0fec007a, 0x01c60fc2, 0x0fe9008f,
	0x01b60fc1, 0x0fe300a6, 0x01a20fc1, 0x0fe000bd,
	0x018f0fc1, 0x0fdb00d5, 0x017b0fc2, 0x0fd500ee,
	0x01640fc4, 0x0fd20106, 0x014d0fc8, 0x0fce011d,
	0x01370fc9, 0x0fc90137, 0x011d0fce, 0x0fc8014d,
	0x01060fd2, 0x0fc40164, 0x00ee0fd5, 0x0fc2017b,
	0x00d50fdb, 0x0fc1018f, 0x00bd0fe0, 0x0fc101a2,
	0x00a60fe3, 0x0fc101b6, 0x008f0fe9, 0x0fc201c6,
	0x007a0fec, 0x0fc301d7, 0x00650ff1, 0x0fc701e3,
	0x00510ff5, 0x0fcd01ed, 0x00400ff8, 0x0fd201f6,
	0x002f0ffc, 0x0fd901fc, 0x00200ffd, 0x0fe10202,
	0x00130000, 0x0fea0203, 0x00080000, 0x0ff50203,
	0x02000000, 0x00000000, 0x02020ff5, 0x00000009,
	0x01ff0fea, 0x00000017, 0x01fb0fe2, 0x0ffd0026,
	0x01f30fda, 0x0ffc0037, 0x01ea0fd3, 0x0ff8004b,
	0x01df0fce, 0x0ff5005e, 0x01d10fc9, 0x0ff20074,
	0x01c10fc6, 0x0fed008c, 0x01ae0fc5, 0x0fea00a3,
	0x019b0fc5, 0x0fe500bb, 0x01850fc6, 0x0fe200d3,
	0x01700fc6, 0x0fde00ec, 0x015a0fc8, 0x0fd90105,
	0x01430fca, 0x0fd6011d, 0x012b0fcd, 0x0fd30135,
	0x01150fcf, 0x0fcf014d, 0x00fc0fd4, 0x0fce0162,
	0x00e50fd8, 0x0fcc0177, 0x00cf0fdb, 0x0fca018c,
	0x00b80fe0, 0x0fc9019f, 0x00a20fe5, 0x0fca01af,
	0x008e0fe8, 0x0fcb01bf, 0x00790fec, 0x0fcb01d0,
	0x00670fef, 0x0fcd01dd, 0x00550ff4, 0x0fd001e7,
	0x00440ff7, 0x0fd501f0, 0x00350ffa, 0x0fda01f7,
	0x00270ffc, 0x0fdf01fe, 0x001b0ffe, 0x0fe70200,
	0x00100000, 0x0fee0202, 0x00060000, 0x0ff70203,
	0x02000000, 0x00000000, 0x01ff0ff5, 0x0000000c,
	0x01f80fea, 0x0000001e, 0x01ef0fe2, 0x0ffd0032,
	0x01e20fdb, 0x0ffc0047, 0x01d30fd5, 0x0ff9005f,
	0x01c20fd1, 0x0ff60077, 0x01b00fcd, 0x0ff30090,
	0x019b0fcb, 0x0fef00ab, 0x01850fcb, 0x0fec00c4,
	0x016e0fcc, 0x0fe800de, 0x01550fcd, 0x0fe600f8,
	0x013f0fce, 0x0fe20111, 0x01280fd0, 0x0fdf0129,
	0x01110fd2, 0x0fdd0140, 0x00f90fd6, 0x0fdb0156,
	0x00e40fd8, 0x0fd8016c, 0x00cd0fdd, 0x0fd8017e,
	0x00b80fe0, 0x0fd60192, 0x00a40fe3, 0x0fd601a3,
	0x00910fe7, 0x0fd501b3, 0x007f0feb, 0x0fd601c0,
	0x006e0fed, 0x0fd701ce, 0x005d0ff1, 0x0fd701db,
	0x004f0ff3, 0x0fd901e5, 0x00400ff7, 0x0fdc01ed,
	0x00330ff9, 0x0fe001f4, 0x00280ffb, 0x0fe301fa,
	0x001d0ffd, 0x0fe801fe, 0x00140ffe, 0x0fed0201,
	0x000c0000, 0x0ff20202, 0x00050000, 0x0ff90202,
	0x02000000, 0x00000000, 0x02040ff7, 0x00000005,
	0x02070fed, 0x0000000c, 0x02060fe6, 0x0ffe0016,
	0x02050fdf, 0x0ffc0020, 0x02020fd9, 0x0ff9002c,
	0x01fe0fd4, 0x0ff60038, 0x01f80fcf, 0x0ff30046,
	0x01f00fcb, 0x0fef0056, 0x01e70fc8, 0x0feb0066,
	0x01db0fc7, 0x0fe60078, 0x01cc0fc6, 0x0fe3008b,
	0x01bf0fc5, 0x0fdd009f, 0x01ae0fc6, 0x0fd800b4,
	0x019c0fc6, 0x0fd400ca, 0x01880fc9, 0x0fcf00e0,
	0x01750fc9, 0x0fc900f9, 0x015d0fce, 0x0fc6010f,
	0x01460fd0, 0x0fc20128, 0x012e0fd3, 0x0fbf0140,
	0x01140fd8, 0x0fbc0158, 0x00f90fdd, 0x0fbb016f,
	0x00df0fe0, 0x0fba0187, 0x00c40fe5, 0x0fb9019e,
	0x00aa0fe9, 0x0fba01b3, 0x008e0fef, 0x0fbd01c6,
	0x00740ff3, 0x0fc301d6, 0x005d0ff6, 0x0fc801e5,
	0x00450ffa, 0x0fd001f1, 0x00300ffc, 0x0fda01fa,
	0x001c0000, 0x0fe40200, 0x000c0000, 0x0ff20202,
	0x02000000, 0x00000000, 0x02030ff7, 0x00000006,
	0x02020fee, 0x00000010, 0x02000fe7, 0x0ffe001b,
	0x01fe0fdf, 0x0ffc0027, 0x01f70fda, 0x0ffa0035,
	0x01f00fd5, 0x0ff70044, 0x01e70fd0, 0x0ff40055,
	0x01dd0fcd, 0x0fef0067, 0x01d00fcb, 0x0fec0079,
	0x01bf0fcb, 0x0fe8008e, 0x01af0fca, 0x0fe500a2,
	0x019f0fc9, 0x0fe000b8, 0x018c0fca, 0x0fdb00cf,
	0x01770fcc, 0x0fd800e5, 0x01620fce, 0x0fd400fc,
	0x014d0fcf, 0x0fcf0115, 0x01350fd3, 0x0fcd012b,
	0x011d0fd6, 0x0fca0143, 0x01050fd9, 0x0fc8015a,
	0x00ec0fde, 0x0fc60170, 0x00d30fe2, 0x0fc60185,
	0x00bb0fe5, 0x0fc5019b, 0x00a30fea, 0x0fc501ae,
	0x008c0fed, 0x0fc601c1, 0x00740ff2, 0x0fc901d1,
	0x005e0ff5, 0x0fce01df, 0x004b0ff8, 0x0fd301ea,
	0x00370ffc, 0x0fda01f3, 0x00260ffd, 0x0fe201fb,
	0x00170000, 0x0fea01ff, 0x00090000, 0x0ff50202,
	0x02000000, 0x00000000, 0x02010ff7, 0x00000008,
	0x01ff0fee, 0x00000013, 0x01fb0fe7, 0x0ffe0020,
	0x01f60fe0, 0x0ffc002e, 0x01ed0fda, 0x0ffa003f,
	0x01e40fd6, 0x0ff7004f, 0x01d80fd2, 0x0ff40062,
	0x01ca0fcf, 0x0ff00077, 0x01bb0fcd, 0x0fed008b,
	0x01a90fcd, 0x0fe900a1, 0x01960fcd, 0x0fe600b7,
	0x01830fcd, 0x0fe200ce, 0x016d0fcf, 0x0fde00e6,
	0x01580fd0, 0x0fdb00fd, 0x01410fd3, 0x0fd80114,
	0x012c0fd4, 0x0fd4012c, 0x01140fd8, 0x0fd30141,
	0x00fd0fdb, 0x0fd00158, 0x00e60fde, 0x0fcf016d,
	0x00ce0fe2, 0x0fcd0183, 0x00b70fe6, 0x0fcd0196,
	0x00a10fe9, 0x0fcd01a9, 0x008b0fed, 0x0fcd01bb,
	0x00770ff0, 0x0fcf01ca, 0x00620ff4, 0x0fd201d8,
	0x004f0ff7, 0x0fd601e4, 0x003f0ffa, 0x0fda01ed,
	0x002e0ffc, 0x0fe001f6, 0x00200ffe, 0x0fe701fb,
	0x00130000, 0x0fee01ff, 0x00080000, 0x0ff70201,
	0x02000000, 0x00000000, 0x01ff0ff7, 0x0000000a,
	0x01f90fee, 0x00000019, 0x01f10fe7, 0x0ffe002a,
	0x01e60fe1, 0x0ffd003c, 0x01d90fdc, 0x0ffa0051,
	0x01cc0fd8, 0x0ff70065, 0x01bb0fd5, 0x0ff5007b,
	0x01a80fd3, 0x0ff10094, 0x01950fd2, 0x0fef00aa,
	0x01800fd2, 0x0feb00c3, 0x016a0fd3, 0x0fe900da,
	0x01540fd3, 0x0fe600f3, 0x013f0fd5, 0x0fe2010a,
	0x01280fd7, 0x0fe00121, 0x01100fda, 0x0fde0138,
	0x00fb0fdb, 0x0fdb014f, 0x00e40fdf, 0x0fdb0162,
	0x00ce0fe2, 0x0fd90177, 0x00b90fe4, 0x0fd8018b,
	0x00a50fe8, 0x0fd8019b, 0x00910fec, 0x0fd801ab,
	0x007e0fee, 0x0fd801bc, 0x006c0ff2, 0x0fd901c9,
	0x005c0ff4, 0x0fda01d6, 0x004b0ff7, 0x0fdd01e1,
	0x003c0ff9, 0x0fe001eb, 0x002f0ffb, 0x0fe401f2,
	0x00230ffd, 0x0fe801f8, 0x00180ffe, 0x0fed01fd,
	0x000e0000, 0x0ff20200, 0x00060000, 0x0ff90201,
	0x02000000, 0x00000000, 0x02030ff9, 0x00000004,
	0x02050ff2, 0x00000009, 0x02050fed, 0x0ffe0010,
	0x02040fe7, 0x0ffd0018, 0x02020fe3, 0x0ffb0020,
	0x01fe0fdf, 0x0ff9002a, 0x01fa0fdb, 0x0ff70034,
	0x01f40fd8, 0x0ff30041, 0x01ed0fd6, 0x0ff0004d,
	0x01e30fd5, 0x0fec005c, 0x01d80fd4, 0x0fea006a,
	0x01cd0fd3, 0x0fe5007b, 0x01c00fd3, 0x0fe1008c,
	0x01b10fd3, 0x0fdd009f, 0x01a10fd4, 0x0fd900b2,
	0x01900fd4, 0x0fd400c8, 0x017b0fd7, 0x0fd100dd,
	0x01660fd9, 0x0fcd00f4, 0x01500fda, 0x0fca010c,
	0x01380fde, 0x0fc60124, 0x011e0fe2, 0x0fc5013b,
	0x01040fe4, 0x0fc30155, 0x00e70fe8, 0x0fc10170,
	0x00cc0feb, 0x0fc10188, 0x00ad0ff0, 0x0fc301a0,
	0x00900ff4, 0x0fc701b5, 0x00750ff7, 0x0fcc01c8,
	0x00580ffb, 0x0fd201db, 0x003e0ffd, 0x0fdb01ea,
	0x00250000, 0x0fe501f6, 0x000f0000, 0x0ff301fe,
	0x02000000, 0x00000000, 0x02020ff9, 0x00000005,
	0x02020ff2, 0x0000000c, 0x02010fed, 0x0ffe0014,
	0x01fe0fe8, 0x0ffd001d, 0x01fa0fe3, 0x0ffb0028,
	0x01f40fe0, 0x0ff90033, 0x01ed0fdc, 0x0ff70040,
	0x01e50fd9, 0x0ff3004f, 0x01db0fd7, 0x0ff1005d,
	0x01ce0fd7, 0x0fed006e, 0x01c00fd6, 0x0feb007f,
	0x01b30fd5, 0x0fe70091, 0x01a30fd6, 0x0fe300a4,
	0x01920fd6, 0x0fe000b8, 0x017e0fd8, 0x0fdd00cd,
	0x016c0fd8, 0x0fd800e4, 0x01560fdb, 0x0fd600f9,
	0x01400fdd, 0x0fd20111, 0x01290fdf, 0x0fd00128,
	0x01110fe2, 0x0fce013f, 0x00f80fe6, 0x0fcd0155,
	0x00de0fe8, 0x0fcc016e, 0x00c40fec, 0x0fcb0185,
	0x00ab0fef, 0x0fcb019b, 0x00900ff3, 0x0fcd01b0,
	0x00770ff6, 0x0fd101c2, 0x005f0ff9, 0x0fd501d3,
	0x00470ffc, 0x0fdb01e2, 0x00320ffd, 0x0fe201ef,
	0x001e0000, 0x0fea01f8, 0x000c0000, 0x0ff501ff,
	0x02000000, 0x00000000, 0x02010ff9, 0x00000006,
	0x02000ff2, 0x0000000e, 0x01fd0fed, 0x0ffe0018,
	0x01f80fe8, 0x0ffd0023, 0x01f20fe4, 0x0ffb002f,
	0x01eb0fe0, 0x0ff9003c, 0x01e10fdd, 0x0ff7004b,
	0x01d60fda, 0x0ff4005c, 0x01c90fd9, 0x0ff2006c,
	0x01bc0fd8, 0x0fee007e, 0x01ab0fd8, 0x0fec0091,
	0x019b0fd8, 0x0fe800a5, 0x018b0fd8, 0x0fe400b9,
	0x01770fd9, 0x0fe200ce, 0x01620fdb, 0x0fdf00e4,
	0x014f0fdb, 0x0fdb00fb, 0x01380fde, 0x0fda0110,
	0x01210fe0, 0x0fd70128, 0x010a0fe2, 0x0fd5013f,
	0x00f30fe6, 0x0fd30154, 0x00da0fe9, 0x0fd3016a,
	0x00c30feb, 0x0fd20180, 0x00aa0fef, 0x0fd20195,
	0x00940ff1, 0x0fd301a8, 0x007b0ff5, 0x0fd501bb,
	0x00650ff7, 0x0fd801cc, 0x00510ffa, 0x0fdc01d9,
	0x003c0ffd, 0x0fe101e6, 0x002a0ffe, 0x0fe701f1,
	0x00190000, 0x0fee01f9, 0x000a0000, 0x0ff701ff,
	0x02000000, 0x00000000, 0x01ff0ff9, 0x00000008,
	0x01fb0ff2, 0x00000013, 0x01f50fed, 0x0ffe0020,
	0x01ed0fe8, 0x0ffd002e, 0x01e30fe4, 0x0ffb003e,
	0x01d80fe1, 0x0ff9004e, 0x01cb0fde, 0x0ff70060,
	0x01bc0fdc, 0x0ff40074, 0x01ac0fdb, 0x0ff20087,
	0x019a0fdb, 0x0fef009c, 0x01870fdb, 0x0fed00b1,
	0x01740fdb, 0x0fea00c7, 0x01600fdc, 0x0fe700dd,
	0x014b0fdd, 0x0fe500f3, 0x01350fdf, 0x0fe30109,
	0x01200fe0, 0x0fe00120, 0x01090fe3, 0x0fdf0135,
	0x00f30fe5, 0x0fdd014b, 0x00dd0fe7, 0x0fdc0160,
	0x00c70fea, 0x0fdb0174, 0x00b10fed, 0x0fdb0187,
	0x009c0fef, 0x0fdb019a, 0x00870ff2, 0x0fdb01ac,
	0x00740ff4, 0x0fdc01bc, 0x00600ff7, 0x0fde01cb,
	0x004e0ff9, 0x0fe101d8, 0x003e0ffb, 0x0fe401e3,
	0x002e0ffd, 0x0fe801ed, 0x00200ffe, 0x0fed01f5,
	0x00130000, 0x0ff201fb, 0x00080000, 0x0ff901ff
};


#define MDP4_QSEED_TABLE0_OFF 0x8100
#define MDP4_QSEED_TABLE1_OFF 0x8200
#define MDP4_QSEED_TABLE2_OFF 0x9000

void mdp4_vg_qseed_init(int vp_num)
{
	uint32 *off;
	int i, voff;

	voff = MDP4_VIDEO_OFF * vp_num;
	off = (uint32 *)(MDP_BASE + MDP4_VIDEO_BASE + voff +
						MDP4_QSEED_TABLE0_OFF);
	for (i = 0; i < (sizeof(vg_qseed_table0) / sizeof(uint32)); i++) {
		outpdw(off, vg_qseed_table0[i]);
		off++;
	}

	off = (uint32 *)(MDP_BASE + MDP4_VIDEO_BASE + voff +
						MDP4_QSEED_TABLE1_OFF);
	for (i = 0; i < (sizeof(vg_qseed_table1) / sizeof(uint32)); i++) {
		outpdw(off, vg_qseed_table1[i]);
		off++;
	}

	off = (uint32 *)(MDP_BASE + MDP4_VIDEO_BASE + voff +
						MDP4_QSEED_TABLE2_OFF);
	for (i = 0; i < (sizeof(vg_qseed_table2) / sizeof(uint32)); i++) {
		outpdw(off, vg_qseed_table2[i]);
		off++;
	}

}

void mdp4_mixer_blend_init(mixer_num)
{
	unsigned char *overlay_base;
	int off;

	if (mixer_num) 	/* mixer number, /dev/fb0, /dev/fb1 */
		overlay_base = MDP_BASE + MDP4_OVERLAYPROC1_BASE;/* 0x18000 */
	else
		overlay_base = MDP_BASE + MDP4_OVERLAYPROC0_BASE;/* 0x10000 */

	/* stage 0 to stage 2 */
	off = 0;
	outpdw(overlay_base + off + 0x104, 0x010);
	outpdw(overlay_base + off + 0x108, 0xff);/* FG */
	outpdw(overlay_base + off + 0x10c, 0x00);/* BG */

	off += 0x20;
	outpdw(overlay_base + off + 0x104, 0x010);
	outpdw(overlay_base + off + 0x108, 0xff);/* FG */
	outpdw(overlay_base + off + 0x10c, 0x00);/* BG */

	off += 0x20;
	outpdw(overlay_base + off + 0x104, 0x010);
	outpdw(overlay_base + off + 0x108, 0xff);/* FG */
	outpdw(overlay_base + off + 0x10c, 0x00);/* BG */
}


static uint32 csc_matrix_tab[9] = {
	0x0254, 0x0000, 0x0331,
	0x0254, 0xff37, 0xfe60,
	0x0254, 0x0409, 0x0000
};

static uint32 csc_pre_bv_tab[3] = {0xfff0, 0xff80, 0xff80 };
static uint32 csc_post_bv_tab[3] = {0, 0, 0 };

static  uint32 csc_pre_lv_tab[6] =  {0, 0xff, 0, 0xff, 0, 0xff };
static  uint32 csc_post_lv_tab[6] = {0, 0xff, 0, 0xff, 0, 0xff };

#define MDP4_CSC_MV_OFF 	0x4400
#define MDP4_CSC_PRE_BV_OFF 	0x4500
#define MDP4_CSC_POST_BV_OFF 	0x4580
#define MDP4_CSC_PRE_LV_OFF 	0x4600
#define MDP4_CSC_POST_LV_OFF 	0x4680

void mdp4_vg_csc_mv_setup(int vp_num)
{
	uint32 *off;
	int i, voff;

	voff = MDP4_VIDEO_OFF * vp_num;
	off = (uint32 *)(MDP_BASE + MDP4_VIDEO_BASE + voff +
					MDP4_CSC_MV_OFF);
	for (i = 0; i < 9; i++) {
		outpdw(off, csc_matrix_tab[i]);
		off++;
	}
}

void mdp4_vg_csc_pre_bv_setup(int vp_num)
{
	uint32 *off;
	int i, voff;

	voff = MDP4_VIDEO_OFF * vp_num;
	off = (uint32 *)(MDP_BASE + MDP4_VIDEO_BASE + voff +
					MDP4_CSC_PRE_BV_OFF);
	for (i = 0; i < 3; i++) {
		outpdw(off, csc_pre_bv_tab[i]);
		off++;
	}
}

void mdp4_vg_csc_post_bv_setup(int vp_num)
{
	uint32 *off;
	int i, voff;

	voff = MDP4_VIDEO_OFF * vp_num;
	off = (uint32 *)(MDP_BASE + MDP4_VIDEO_BASE + voff +
					MDP4_CSC_POST_BV_OFF);
	for (i = 0; i < 3; i++) {
		outpdw(off, csc_post_bv_tab[i]);
		off++;
	}
}

void mdp4_vg_csc_pre_lv_setup(int vp_num)
{
	uint32 *off;
	int i, voff;

	voff = MDP4_VIDEO_OFF * vp_num;
	off = (uint32 *)(MDP_BASE + MDP4_VIDEO_BASE + voff +
					MDP4_CSC_PRE_LV_OFF);

	for (i = 0; i < 6; i++) {
		outpdw(off, csc_pre_lv_tab[i]);
		off++;
	}
}

void mdp4_vg_csc_post_lv_setup(int vp_num)
{
	uint32 *off;
	int i, voff;

	voff = MDP4_VIDEO_OFF * vp_num;
	off = (uint32 *)(MDP_BASE + MDP4_VIDEO_BASE + voff +
					MDP4_CSC_POST_LV_OFF);

	for (i = 0; i < 6; i++) {
		outpdw(off, csc_post_lv_tab[i]);
		off++;
	}
}

char gc_lut[] = {
	0x0, 0x1, 0x2, 0x2, 0x3, 0x4, 0x5, 0x6,
	0x6, 0x7, 0x8, 0x9, 0xA, 0xA, 0xB, 0xC,
	0xD, 0xD, 0xE, 0xF, 0xF, 0x10, 0x10, 0x11,
	0x12, 0x12, 0x13, 0x13, 0x14, 0x14, 0x15, 0x15,
	0x16, 0x16, 0x17, 0x17, 0x17, 0x18, 0x18, 0x19,
	0x19, 0x19, 0x1A, 0x1A, 0x1B, 0x1B, 0x1B, 0x1C,
	0x1C, 0x1D, 0x1D, 0x1D, 0x1E, 0x1E, 0x1E, 0x1F,
	0x1F, 0x1F, 0x20, 0x20, 0x20, 0x21, 0x21, 0x21,
	0x22, 0x22, 0x22, 0x22, 0x23, 0x23, 0x23, 0x24,
	0x24, 0x24, 0x25, 0x25, 0x25, 0x25, 0x26, 0x26,
	0x26, 0x26, 0x27, 0x27, 0x27, 0x28, 0x28, 0x28,
	0x28, 0x29, 0x29, 0x29, 0x29, 0x2A, 0x2A, 0x2A,
	0x2A, 0x2B, 0x2B, 0x2B, 0x2B, 0x2B, 0x2C, 0x2C,
	0x2C, 0x2C, 0x2D, 0x2D, 0x2D, 0x2D, 0x2E, 0x2E,
	0x2E, 0x2E, 0x2E, 0x2F, 0x2F, 0x2F, 0x2F, 0x30,
	0x30, 0x30, 0x30, 0x30, 0x31, 0x31, 0x31, 0x31,
	0x31, 0x32, 0x32, 0x32, 0x32, 0x32, 0x33, 0x33,
	0x33, 0x33, 0x33, 0x34, 0x34, 0x34, 0x34, 0x34,
	0x35, 0x35, 0x35, 0x35, 0x35, 0x36, 0x36, 0x36,
	0x36, 0x36, 0x37, 0x37, 0x37, 0x37, 0x37, 0x37,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x39, 0x39, 0x39,
	0x39, 0x39, 0x39, 0x3A, 0x3A, 0x3A, 0x3A, 0x3A,
	0x3A, 0x3B, 0x3B, 0x3B, 0x3B, 0x3B, 0x3B, 0x3C,
	0x3C, 0x3C, 0x3C, 0x3C, 0x3C, 0x3D, 0x3D, 0x3D,
	0x3D, 0x3D, 0x3D, 0x3E, 0x3E, 0x3E, 0x3E, 0x3E,
	0x3E, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x40,
	0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x41, 0x41,
	0x41, 0x41, 0x41, 0x41, 0x42, 0x42, 0x42, 0x42,
	0x42, 0x42, 0x42, 0x43, 0x43, 0x43, 0x43, 0x43,
	0x43, 0x43, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
	0x44, 0x45, 0x45, 0x45, 0x45, 0x45, 0x45, 0x45,
	0x46, 0x46, 0x46, 0x46, 0x46, 0x46, 0x46, 0x47,
	0x47, 0x47, 0x47, 0x47, 0x47, 0x47, 0x48, 0x48,
	0x48, 0x48, 0x48, 0x48, 0x48, 0x48, 0x49, 0x49,
	0x49, 0x49, 0x49, 0x49, 0x49, 0x4A, 0x4A, 0x4A,
	0x4A, 0x4A, 0x4A, 0x4A, 0x4A, 0x4B, 0x4B, 0x4B,
	0x4B, 0x4B, 0x4B, 0x4B, 0x4B, 0x4C, 0x4C, 0x4C,
	0x4C, 0x4C, 0x4C, 0x4C, 0x4D, 0x4D, 0x4D, 0x4D,
	0x4D, 0x4D, 0x4D, 0x4D, 0x4E, 0x4E, 0x4E, 0x4E,
	0x4E, 0x4E, 0x4E, 0x4E, 0x4E, 0x4F, 0x4F, 0x4F,
	0x4F, 0x4F, 0x4F, 0x4F, 0x4F, 0x50, 0x50, 0x50,
	0x50, 0x50, 0x50, 0x50, 0x50, 0x51, 0x51, 0x51,
	0x51, 0x51, 0x51, 0x51, 0x51, 0x51, 0x52, 0x52,
	0x52, 0x52, 0x52, 0x52, 0x52, 0x52, 0x53, 0x53,
	0x53, 0x53, 0x53, 0x53, 0x53, 0x53, 0x53, 0x54,
	0x54, 0x54, 0x54, 0x54, 0x54, 0x54, 0x54, 0x54,
	0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
	0x55, 0x56, 0x56, 0x56, 0x56, 0x56, 0x56, 0x56,
	0x56, 0x56, 0x57, 0x57, 0x57, 0x57, 0x57, 0x57,
	0x57, 0x57, 0x57, 0x58, 0x58, 0x58, 0x58, 0x58,
	0x58, 0x58, 0x58, 0x58, 0x58, 0x59, 0x59, 0x59,
	0x59, 0x59, 0x59, 0x59, 0x59, 0x59, 0x5A, 0x5A,
	0x5A, 0x5A, 0x5A, 0x5A, 0x5A, 0x5A, 0x5A, 0x5A,
	0x5B, 0x5B, 0x5B, 0x5B, 0x5B, 0x5B, 0x5B, 0x5B,
	0x5B, 0x5B, 0x5C, 0x5C, 0x5C, 0x5C, 0x5C, 0x5C,
	0x5C, 0x5C, 0x5C, 0x5C, 0x5D, 0x5D, 0x5D, 0x5D,
	0x5D, 0x5D, 0x5D, 0x5D, 0x5D, 0x5D, 0x5E, 0x5E,
	0x5E, 0x5E, 0x5E, 0x5E, 0x5E, 0x5E, 0x5E, 0x5E,
	0x5F, 0x5F, 0x5F, 0x5F, 0x5F, 0x5F, 0x5F, 0x5F,
	0x5F, 0x5F, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60,
	0x60, 0x60, 0x60, 0x60, 0x60, 0x61, 0x61, 0x61,
	0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x62,
	0x62, 0x62, 0x62, 0x62, 0x62, 0x62, 0x62, 0x62,
	0x62, 0x62, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63,
	0x63, 0x63, 0x63, 0x63, 0x63, 0x64, 0x64, 0x64,
	0x64, 0x64, 0x64, 0x64, 0x64, 0x64, 0x64, 0x64,
	0x65, 0x65, 0x65, 0x65, 0x65, 0x65, 0x65, 0x65,
	0x65, 0x65, 0x65, 0x66, 0x66, 0x66, 0x66, 0x66,
	0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x67, 0x67,
	0x67, 0x67, 0x67, 0x67, 0x67, 0x67, 0x67, 0x67,
	0x67, 0x67, 0x68, 0x68, 0x68, 0x68, 0x68, 0x68,
	0x68, 0x68, 0x68, 0x68, 0x68, 0x69, 0x69, 0x69,
	0x69, 0x69, 0x69, 0x69, 0x69, 0x69, 0x69, 0x69,
	0x69, 0x6A, 0x6A, 0x6A, 0x6A, 0x6A, 0x6A, 0x6A,
	0x6A, 0x6A, 0x6A, 0x6A, 0x6A, 0x6B, 0x6B, 0x6B,
	0x6B, 0x6B, 0x6B, 0x6B, 0x6B, 0x6B, 0x6B, 0x6B,
	0x6B, 0x6C, 0x6C, 0x6C, 0x6C, 0x6C, 0x6C, 0x6C,
	0x6C, 0x6C, 0x6C, 0x6C, 0x6C, 0x6D, 0x6D, 0x6D,
	0x6D, 0x6D, 0x6D, 0x6D, 0x6D, 0x6D, 0x6D, 0x6D,
	0x6D, 0x6E, 0x6E, 0x6E, 0x6E, 0x6E, 0x6E, 0x6E,
	0x6E, 0x6E, 0x6E, 0x6E, 0x6E, 0x6F, 0x6F, 0x6F,
	0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F,
	0x6F, 0x6F, 0x70, 0x70, 0x70, 0x70, 0x70, 0x70,
	0x70, 0x70, 0x70, 0x70, 0x70, 0x70, 0x71, 0x71,
	0x71, 0x71, 0x71, 0x71, 0x71, 0x71, 0x71, 0x71,
	0x71, 0x71, 0x71, 0x72, 0x72, 0x72, 0x72, 0x72,
	0x72, 0x72, 0x72, 0x72, 0x72, 0x72, 0x72, 0x72,
	0x73, 0x73, 0x73, 0x73, 0x73, 0x73, 0x73, 0x73,
	0x73, 0x73, 0x73, 0x73, 0x73, 0x74, 0x74, 0x74,
	0x74, 0x74, 0x74, 0x74, 0x74, 0x74, 0x74, 0x74,
	0x74, 0x74, 0x75, 0x75, 0x75, 0x75, 0x75, 0x75,
	0x75, 0x75, 0x75, 0x75, 0x75, 0x75, 0x75, 0x75,
	0x76, 0x76, 0x76, 0x76, 0x76, 0x76, 0x76, 0x76,
	0x76, 0x76, 0x76, 0x76, 0x76, 0x77, 0x77, 0x77,
	0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77,
	0x77, 0x77, 0x77, 0x78, 0x78, 0x78, 0x78, 0x78,
	0x78, 0x78, 0x78, 0x78, 0x78, 0x78, 0x78, 0x78,
	0x78, 0x79, 0x79, 0x79, 0x79, 0x79, 0x79, 0x79,
	0x79, 0x79, 0x79, 0x79, 0x79, 0x79, 0x7A, 0x7A,
	0x7A, 0x7A, 0x7A, 0x7A, 0x7A, 0x7A, 0x7A, 0x7A,
	0x7A, 0x7A, 0x7A, 0x7A, 0x7A, 0x7B, 0x7B, 0x7B,
	0x7B, 0x7B, 0x7B, 0x7B, 0x7B, 0x7B, 0x7B, 0x7B,
	0x7B, 0x7B, 0x7B, 0x7C, 0x7C, 0x7C, 0x7C, 0x7C,
	0x7C, 0x7C, 0x7C, 0x7C, 0x7C, 0x7C, 0x7C, 0x7C,
	0x7C, 0x7D, 0x7D, 0x7D, 0x7D, 0x7D, 0x7D, 0x7D,
	0x7D, 0x7D, 0x7D, 0x7D, 0x7D, 0x7D, 0x7D, 0x7D,
	0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E,
	0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7F, 0x7F,
	0x7F, 0x7F, 0x7F, 0x7F, 0x7F, 0x7F, 0x7F, 0x7F,
	0x7F, 0x7F, 0x7F, 0x7F, 0x7F, 0x80, 0x80, 0x80,
	0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
	0x80, 0x80, 0x80, 0x80, 0x81, 0x81, 0x81, 0x81,
	0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81,
	0x81, 0x81, 0x81, 0x82, 0x82, 0x82, 0x82, 0x82,
	0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82,
	0x82, 0x82, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83,
	0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83,
	0x83, 0x83, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84,
	0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84,
	0x84, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85,
	0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85,
	0x85, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86,
	0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86,
	0x86, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87,
	0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87,
	0x87, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
	0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
	0x88, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89,
	0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89,
	0x89, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A,
	0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A,
	0x8A, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B,
	0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B,
	0x8B, 0x8B, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C,
	0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C,
	0x8C, 0x8C, 0x8C, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D,
	0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D,
	0x8D, 0x8D, 0x8D, 0x8D, 0x8E, 0x8E, 0x8E, 0x8E,
	0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E,
	0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8F, 0x8F, 0x8F,
	0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F,
	0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x90, 0x90,
	0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
	0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x91,
	0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91,
	0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91,
	0x91, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92,
	0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92,
	0x92, 0x92, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93,
	0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93,
	0x93, 0x93, 0x93, 0x93, 0x94, 0x94, 0x94, 0x94,
	0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94,
	0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x95, 0x95,
	0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95,
	0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95,
	0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96,
	0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96,
	0x96, 0x96, 0x96, 0x97, 0x97, 0x97, 0x97, 0x97,
	0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97,
	0x97, 0x97, 0x97, 0x97, 0x97, 0x98, 0x98, 0x98,
	0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98,
	0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98,
	0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99,
	0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99,
	0x99, 0x99, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A,
	0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A,
	0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9B, 0x9B, 0x9B,
	0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B,
	0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B,
	0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C,
	0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C,
	0x9C, 0x9C, 0x9C, 0x9C, 0x9D, 0x9D, 0x9D, 0x9D,
	0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D,
	0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9E,
	0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E,
	0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E,
	0x9E, 0x9E, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F,
	0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F,
	0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0xA0, 0xA0,
	0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0,
	0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0,
	0xA0, 0xA0, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1,
	0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1,
	0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA2, 0xA2,
	0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2,
	0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2,
	0xA2, 0xA2, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
	0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
	0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA4, 0xA4,
	0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4,
	0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4,
	0xA4, 0xA4, 0xA4, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5,
	0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5,
	0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5,
	0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6,
	0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6,
	0xA6, 0xA6, 0xA6, 0xA6, 0xA7, 0xA7, 0xA7, 0xA7,
	0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7,
	0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7,
	0xA7, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8,
	0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8,
	0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA9,
	0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9,
	0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9,
	0xA9, 0xA9, 0xA9, 0xA9, 0xAA, 0xAA, 0xAA, 0xAA,
	0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
	0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
	0xAA, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB,
	0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB,
	0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAC,
	0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC,
	0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC,
	0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAD, 0xAD, 0xAD,
	0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD,
	0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD,
	0xAD, 0xAD, 0xAD, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE,
	0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE,
	0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE,
	0xAE, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF,
	0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF,
	0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xB0,
	0xB0, 0xB0, 0xB0, 0xB0, 0xB0, 0xB0, 0xB0, 0xB0,
	0xB0, 0xB0, 0xB0, 0xB0, 0xB0, 0xB0, 0xB0, 0xB0,
	0xB0, 0xB0, 0xB0, 0xB0, 0xB0, 0xB0, 0xB1, 0xB1,
	0xB1, 0xB1, 0xB1, 0xB1, 0xB1, 0xB1, 0xB1, 0xB1,
	0xB1, 0xB1, 0xB1, 0xB1, 0xB1, 0xB1, 0xB1, 0xB1,
	0xB1, 0xB1, 0xB1, 0xB1, 0xB2, 0xB2, 0xB2, 0xB2,
	0xB2, 0xB2, 0xB2, 0xB2, 0xB2, 0xB2, 0xB2, 0xB2,
	0xB2, 0xB2, 0xB2, 0xB2, 0xB2, 0xB2, 0xB2, 0xB2,
	0xB2, 0xB2, 0xB2, 0xB3, 0xB3, 0xB3, 0xB3, 0xB3,
	0xB3, 0xB3, 0xB3, 0xB3, 0xB3, 0xB3, 0xB3, 0xB3,
	0xB3, 0xB3, 0xB3, 0xB3, 0xB3, 0xB3, 0xB3, 0xB3,
	0xB3, 0xB3, 0xB4, 0xB4, 0xB4, 0xB4, 0xB4, 0xB4,
	0xB4, 0xB4, 0xB4, 0xB4, 0xB4, 0xB4, 0xB4, 0xB4,
	0xB4, 0xB4, 0xB4, 0xB4, 0xB4, 0xB4, 0xB4, 0xB4,
	0xB4, 0xB5, 0xB5, 0xB5, 0xB5, 0xB5, 0xB5, 0xB5,
	0xB5, 0xB5, 0xB5, 0xB5, 0xB5, 0xB5, 0xB5, 0xB5,
	0xB5, 0xB5, 0xB5, 0xB5, 0xB5, 0xB5, 0xB5, 0xB5,
	0xB6, 0xB6, 0xB6, 0xB6, 0xB6, 0xB6, 0xB6, 0xB6,
	0xB6, 0xB6, 0xB6, 0xB6, 0xB6, 0xB6, 0xB6, 0xB6,
	0xB6, 0xB6, 0xB6, 0xB6, 0xB6, 0xB6, 0xB6, 0xB6,
	0xB7, 0xB7, 0xB7, 0xB7, 0xB7, 0xB7, 0xB7, 0xB7,
	0xB7, 0xB7, 0xB7, 0xB7, 0xB7, 0xB7, 0xB7, 0xB7,
	0xB7, 0xB7, 0xB7, 0xB7, 0xB7, 0xB7, 0xB7, 0xB8,
	0xB8, 0xB8, 0xB8, 0xB8, 0xB8, 0xB8, 0xB8, 0xB8,
	0xB8, 0xB8, 0xB8, 0xB8, 0xB8, 0xB8, 0xB8, 0xB8,
	0xB8, 0xB8, 0xB8, 0xB8, 0xB8, 0xB8, 0xB8, 0xB9,
	0xB9, 0xB9, 0xB9, 0xB9, 0xB9, 0xB9, 0xB9, 0xB9,
	0xB9, 0xB9, 0xB9, 0xB9, 0xB9, 0xB9, 0xB9, 0xB9,
	0xB9, 0xB9, 0xB9, 0xB9, 0xB9, 0xB9, 0xB9, 0xBA,
	0xBA, 0xBA, 0xBA, 0xBA, 0xBA, 0xBA, 0xBA, 0xBA,
	0xBA, 0xBA, 0xBA, 0xBA, 0xBA, 0xBA, 0xBA, 0xBA,
	0xBA, 0xBA, 0xBA, 0xBA, 0xBA, 0xBA, 0xBA, 0xBB,
	0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB,
	0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB,
	0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB,
	0xBC, 0xBC, 0xBC, 0xBC, 0xBC, 0xBC, 0xBC, 0xBC,
	0xBC, 0xBC, 0xBC, 0xBC, 0xBC, 0xBC, 0xBC, 0xBC,
	0xBC, 0xBC, 0xBC, 0xBC, 0xBC, 0xBC, 0xBC, 0xBC,
	0xBD, 0xBD, 0xBD, 0xBD, 0xBD, 0xBD, 0xBD, 0xBD,
	0xBD, 0xBD, 0xBD, 0xBD, 0xBD, 0xBD, 0xBD, 0xBD,
	0xBD, 0xBD, 0xBD, 0xBD, 0xBD, 0xBD, 0xBD, 0xBD,
	0xBD, 0xBE, 0xBE, 0xBE, 0xBE, 0xBE, 0xBE, 0xBE,
	0xBE, 0xBE, 0xBE, 0xBE, 0xBE, 0xBE, 0xBE, 0xBE,
	0xBE, 0xBE, 0xBE, 0xBE, 0xBE, 0xBE, 0xBE, 0xBE,
	0xBE, 0xBE, 0xBF, 0xBF, 0xBF, 0xBF, 0xBF, 0xBF,
	0xBF, 0xBF, 0xBF, 0xBF, 0xBF, 0xBF, 0xBF, 0xBF,
	0xBF, 0xBF, 0xBF, 0xBF, 0xBF, 0xBF, 0xBF, 0xBF,
	0xBF, 0xBF, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0,
	0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0,
	0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0,
	0xC0, 0xC0, 0xC0, 0xC0, 0xC1, 0xC1, 0xC1, 0xC1,
	0xC1, 0xC1, 0xC1, 0xC1, 0xC1, 0xC1, 0xC1, 0xC1,
	0xC1, 0xC1, 0xC1, 0xC1, 0xC1, 0xC1, 0xC1, 0xC1,
	0xC1, 0xC1, 0xC1, 0xC1, 0xC1, 0xC2, 0xC2, 0xC2,
	0xC2, 0xC2, 0xC2, 0xC2, 0xC2, 0xC2, 0xC2, 0xC2,
	0xC2, 0xC2, 0xC2, 0xC2, 0xC2, 0xC2, 0xC2, 0xC2,
	0xC2, 0xC2, 0xC2, 0xC2, 0xC2, 0xC2, 0xC3, 0xC3,
	0xC3, 0xC3, 0xC3, 0xC3, 0xC3, 0xC3, 0xC3, 0xC3,
	0xC3, 0xC3, 0xC3, 0xC3, 0xC3, 0xC3, 0xC3, 0xC3,
	0xC3, 0xC3, 0xC3, 0xC3, 0xC3, 0xC3, 0xC3, 0xC3,
	0xC4, 0xC4, 0xC4, 0xC4, 0xC4, 0xC4, 0xC4, 0xC4,
	0xC4, 0xC4, 0xC4, 0xC4, 0xC4, 0xC4, 0xC4, 0xC4,
	0xC4, 0xC4, 0xC4, 0xC4, 0xC4, 0xC4, 0xC4, 0xC4,
	0xC4, 0xC4, 0xC5, 0xC5, 0xC5, 0xC5, 0xC5, 0xC5,
	0xC5, 0xC5, 0xC5, 0xC5, 0xC5, 0xC5, 0xC5, 0xC5,
	0xC5, 0xC5, 0xC5, 0xC5, 0xC5, 0xC5, 0xC5, 0xC5,
	0xC5, 0xC5, 0xC5, 0xC5, 0xC6, 0xC6, 0xC6, 0xC6,
	0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6,
	0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6,
	0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC7, 0xC7,
	0xC7, 0xC7, 0xC7, 0xC7, 0xC7, 0xC7, 0xC7, 0xC7,
	0xC7, 0xC7, 0xC7, 0xC7, 0xC7, 0xC7, 0xC7, 0xC7,
	0xC7, 0xC7, 0xC7, 0xC7, 0xC7, 0xC7, 0xC7, 0xC7,
	0xC8, 0xC8, 0xC8, 0xC8, 0xC8, 0xC8, 0xC8, 0xC8,
	0xC8, 0xC8, 0xC8, 0xC8, 0xC8, 0xC8, 0xC8, 0xC8,
	0xC8, 0xC8, 0xC8, 0xC8, 0xC8, 0xC8, 0xC8, 0xC8,
	0xC8, 0xC8, 0xC8, 0xC9, 0xC9, 0xC9, 0xC9, 0xC9,
	0xC9, 0xC9, 0xC9, 0xC9, 0xC9, 0xC9, 0xC9, 0xC9,
	0xC9, 0xC9, 0xC9, 0xC9, 0xC9, 0xC9, 0xC9, 0xC9,
	0xC9, 0xC9, 0xC9, 0xC9, 0xC9, 0xC9, 0xCA, 0xCA,
	0xCA, 0xCA, 0xCA, 0xCA, 0xCA, 0xCA, 0xCA, 0xCA,
	0xCA, 0xCA, 0xCA, 0xCA, 0xCA, 0xCA, 0xCA, 0xCA,
	0xCA, 0xCA, 0xCA, 0xCA, 0xCA, 0xCA, 0xCA, 0xCA,
	0xCA, 0xCB, 0xCB, 0xCB, 0xCB, 0xCB, 0xCB, 0xCB,
	0xCB, 0xCB, 0xCB, 0xCB, 0xCB, 0xCB, 0xCB, 0xCB,
	0xCB, 0xCB, 0xCB, 0xCB, 0xCB, 0xCB, 0xCB, 0xCB,
	0xCB, 0xCB, 0xCB, 0xCB, 0xCC, 0xCC, 0xCC, 0xCC,
	0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC,
	0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC,
	0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCD,
	0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
	0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
	0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
	0xCD, 0xCD, 0xCE, 0xCE, 0xCE, 0xCE, 0xCE, 0xCE,
	0xCE, 0xCE, 0xCE, 0xCE, 0xCE, 0xCE, 0xCE, 0xCE,
	0xCE, 0xCE, 0xCE, 0xCE, 0xCE, 0xCE, 0xCE, 0xCE,
	0xCE, 0xCE, 0xCE, 0xCE, 0xCE, 0xCE, 0xCF, 0xCF,
	0xCF, 0xCF, 0xCF, 0xCF, 0xCF, 0xCF, 0xCF, 0xCF,
	0xCF, 0xCF, 0xCF, 0xCF, 0xCF, 0xCF, 0xCF, 0xCF,
	0xCF, 0xCF, 0xCF, 0xCF, 0xCF, 0xCF, 0xCF, 0xCF,
	0xCF, 0xCF, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
	0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
	0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
	0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD1, 0xD1, 0xD1,
	0xD1, 0xD1, 0xD1, 0xD1, 0xD1, 0xD1, 0xD1, 0xD1,
	0xD1, 0xD1, 0xD1, 0xD1, 0xD1, 0xD1, 0xD1, 0xD1,
	0xD1, 0xD1, 0xD1, 0xD1, 0xD1, 0xD1, 0xD1, 0xD1,
	0xD1, 0xD1, 0xD2, 0xD2, 0xD2, 0xD2, 0xD2, 0xD2,
	0xD2, 0xD2, 0xD2, 0xD2, 0xD2, 0xD2, 0xD2, 0xD2,
	0xD2, 0xD2, 0xD2, 0xD2, 0xD2, 0xD2, 0xD2, 0xD2,
	0xD2, 0xD2, 0xD2, 0xD2, 0xD2, 0xD2, 0xD3, 0xD3,
	0xD3, 0xD3, 0xD3, 0xD3, 0xD3, 0xD3, 0xD3, 0xD3,
	0xD3, 0xD3, 0xD3, 0xD3, 0xD3, 0xD3, 0xD3, 0xD3,
	0xD3, 0xD3, 0xD3, 0xD3, 0xD3, 0xD3, 0xD3, 0xD3,
	0xD3, 0xD3, 0xD4, 0xD4, 0xD4, 0xD4, 0xD4, 0xD4,
	0xD4, 0xD4, 0xD4, 0xD4, 0xD4, 0xD4, 0xD4, 0xD4,
	0xD4, 0xD4, 0xD4, 0xD4, 0xD4, 0xD4, 0xD4, 0xD4,
	0xD4, 0xD4, 0xD4, 0xD4, 0xD4, 0xD4, 0xD4, 0xD5,
	0xD5, 0xD5, 0xD5, 0xD5, 0xD5, 0xD5, 0xD5, 0xD5,
	0xD5, 0xD5, 0xD5, 0xD5, 0xD5, 0xD5, 0xD5, 0xD5,
	0xD5, 0xD5, 0xD5, 0xD5, 0xD5, 0xD5, 0xD5, 0xD5,
	0xD5, 0xD5, 0xD5, 0xD5, 0xD6, 0xD6, 0xD6, 0xD6,
	0xD6, 0xD6, 0xD6, 0xD6, 0xD6, 0xD6, 0xD6, 0xD6,
	0xD6, 0xD6, 0xD6, 0xD6, 0xD6, 0xD6, 0xD6, 0xD6,
	0xD6, 0xD6, 0xD6, 0xD6, 0xD6, 0xD6, 0xD6, 0xD6,
	0xD6, 0xD7, 0xD7, 0xD7, 0xD7, 0xD7, 0xD7, 0xD7,
	0xD7, 0xD7, 0xD7, 0xD7, 0xD7, 0xD7, 0xD7, 0xD7,
	0xD7, 0xD7, 0xD7, 0xD7, 0xD7, 0xD7, 0xD7, 0xD7,
	0xD7, 0xD7, 0xD7, 0xD7, 0xD7, 0xD7, 0xD8, 0xD8,
	0xD8, 0xD8, 0xD8, 0xD8, 0xD8, 0xD8, 0xD8, 0xD8,
	0xD8, 0xD8, 0xD8, 0xD8, 0xD8, 0xD8, 0xD8, 0xD8,
	0xD8, 0xD8, 0xD8, 0xD8, 0xD8, 0xD8, 0xD8, 0xD8,
	0xD8, 0xD8, 0xD8, 0xD9, 0xD9, 0xD9, 0xD9, 0xD9,
	0xD9, 0xD9, 0xD9, 0xD9, 0xD9, 0xD9, 0xD9, 0xD9,
	0xD9, 0xD9, 0xD9, 0xD9, 0xD9, 0xD9, 0xD9, 0xD9,
	0xD9, 0xD9, 0xD9, 0xD9, 0xD9, 0xD9, 0xD9, 0xD9,
	0xD9, 0xDA, 0xDA, 0xDA, 0xDA, 0xDA, 0xDA, 0xDA,
	0xDA, 0xDA, 0xDA, 0xDA, 0xDA, 0xDA, 0xDA, 0xDA,
	0xDA, 0xDA, 0xDA, 0xDA, 0xDA, 0xDA, 0xDA, 0xDA,
	0xDA, 0xDA, 0xDA, 0xDA, 0xDA, 0xDA, 0xDB, 0xDB,
	0xDB, 0xDB, 0xDB, 0xDB, 0xDB, 0xDB, 0xDB, 0xDB,
	0xDB, 0xDB, 0xDB, 0xDB, 0xDB, 0xDB, 0xDB, 0xDB,
	0xDB, 0xDB, 0xDB, 0xDB, 0xDB, 0xDB, 0xDB, 0xDB,
	0xDB, 0xDB, 0xDB, 0xDB, 0xDC, 0xDC, 0xDC, 0xDC,
	0xDC, 0xDC, 0xDC, 0xDC, 0xDC, 0xDC, 0xDC, 0xDC,
	0xDC, 0xDC, 0xDC, 0xDC, 0xDC, 0xDC, 0xDC, 0xDC,
	0xDC, 0xDC, 0xDC, 0xDC, 0xDC, 0xDC, 0xDC, 0xDC,
	0xDC, 0xDC, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
	0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
	0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
	0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
	0xDD, 0xDE, 0xDE, 0xDE, 0xDE, 0xDE, 0xDE, 0xDE,
	0xDE, 0xDE, 0xDE, 0xDE, 0xDE, 0xDE, 0xDE, 0xDE,
	0xDE, 0xDE, 0xDE, 0xDE, 0xDE, 0xDE, 0xDE, 0xDE,
	0xDE, 0xDE, 0xDE, 0xDE, 0xDE, 0xDE, 0xDE, 0xDF,
	0xDF, 0xDF, 0xDF, 0xDF, 0xDF, 0xDF, 0xDF, 0xDF,
	0xDF, 0xDF, 0xDF, 0xDF, 0xDF, 0xDF, 0xDF, 0xDF,
	0xDF, 0xDF, 0xDF, 0xDF, 0xDF, 0xDF, 0xDF, 0xDF,
	0xDF, 0xDF, 0xDF, 0xDF, 0xDF, 0xDF, 0xE0, 0xE0,
	0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0,
	0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0,
	0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0,
	0xE0, 0xE0, 0xE0, 0xE0, 0xE1, 0xE1, 0xE1, 0xE1,
	0xE1, 0xE1, 0xE1, 0xE1, 0xE1, 0xE1, 0xE1, 0xE1,
	0xE1, 0xE1, 0xE1, 0xE1, 0xE1, 0xE1, 0xE1, 0xE1,
	0xE1, 0xE1, 0xE1, 0xE1, 0xE1, 0xE1, 0xE1, 0xE1,
	0xE1, 0xE1, 0xE1, 0xE2, 0xE2, 0xE2, 0xE2, 0xE2,
	0xE2, 0xE2, 0xE2, 0xE2, 0xE2, 0xE2, 0xE2, 0xE2,
	0xE2, 0xE2, 0xE2, 0xE2, 0xE2, 0xE2, 0xE2, 0xE2,
	0xE2, 0xE2, 0xE2, 0xE2, 0xE2, 0xE2, 0xE2, 0xE2,
	0xE2, 0xE2, 0xE3, 0xE3, 0xE3, 0xE3, 0xE3, 0xE3,
	0xE3, 0xE3, 0xE3, 0xE3, 0xE3, 0xE3, 0xE3, 0xE3,
	0xE3, 0xE3, 0xE3, 0xE3, 0xE3, 0xE3, 0xE3, 0xE3,
	0xE3, 0xE3, 0xE3, 0xE3, 0xE3, 0xE3, 0xE3, 0xE3,
	0xE3, 0xE3, 0xE4, 0xE4, 0xE4, 0xE4, 0xE4, 0xE4,
	0xE4, 0xE4, 0xE4, 0xE4, 0xE4, 0xE4, 0xE4, 0xE4,
	0xE4, 0xE4, 0xE4, 0xE4, 0xE4, 0xE4, 0xE4, 0xE4,
	0xE4, 0xE4, 0xE4, 0xE4, 0xE4, 0xE4, 0xE4, 0xE4,
	0xE4, 0xE5, 0xE5, 0xE5, 0xE5, 0xE5, 0xE5, 0xE5,
	0xE5, 0xE5, 0xE5, 0xE5, 0xE5, 0xE5, 0xE5, 0xE5,
	0xE5, 0xE5, 0xE5, 0xE5, 0xE5, 0xE5, 0xE5, 0xE5,
	0xE5, 0xE5, 0xE5, 0xE5, 0xE5, 0xE5, 0xE5, 0xE5,
	0xE5, 0xE6, 0xE6, 0xE6, 0xE6, 0xE6, 0xE6, 0xE6,
	0xE6, 0xE6, 0xE6, 0xE6, 0xE6, 0xE6, 0xE6, 0xE6,
	0xE6, 0xE6, 0xE6, 0xE6, 0xE6, 0xE6, 0xE6, 0xE6,
	0xE6, 0xE6, 0xE6, 0xE6, 0xE6, 0xE6, 0xE6, 0xE6,
	0xE6, 0xE7, 0xE7, 0xE7, 0xE7, 0xE7, 0xE7, 0xE7,
	0xE7, 0xE7, 0xE7, 0xE7, 0xE7, 0xE7, 0xE7, 0xE7,
	0xE7, 0xE7, 0xE7, 0xE7, 0xE7, 0xE7, 0xE7, 0xE7,
	0xE7, 0xE7, 0xE7, 0xE7, 0xE7, 0xE7, 0xE7, 0xE7,
	0xE7, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8,
	0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8,
	0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8,
	0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8,
	0xE8, 0xE9, 0xE9, 0xE9, 0xE9, 0xE9, 0xE9, 0xE9,
	0xE9, 0xE9, 0xE9, 0xE9, 0xE9, 0xE9, 0xE9, 0xE9,
	0xE9, 0xE9, 0xE9, 0xE9, 0xE9, 0xE9, 0xE9, 0xE9,
	0xE9, 0xE9, 0xE9, 0xE9, 0xE9, 0xE9, 0xE9, 0xE9,
	0xE9, 0xE9, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA,
	0xEA, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA,
	0xEA, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA,
	0xEA, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA,
	0xEA, 0xEA, 0xEB, 0xEB, 0xEB, 0xEB, 0xEB, 0xEB,
	0xEB, 0xEB, 0xEB, 0xEB, 0xEB, 0xEB, 0xEB, 0xEB,
	0xEB, 0xEB, 0xEB, 0xEB, 0xEB, 0xEB, 0xEB, 0xEB,
	0xEB, 0xEB, 0xEB, 0xEB, 0xEB, 0xEB, 0xEB, 0xEB,
	0xEB, 0xEB, 0xEB, 0xEC, 0xEC, 0xEC, 0xEC, 0xEC,
	0xEC, 0xEC, 0xEC, 0xEC, 0xEC, 0xEC, 0xEC, 0xEC,
	0xEC, 0xEC, 0xEC, 0xEC, 0xEC, 0xEC, 0xEC, 0xEC,
	0xEC, 0xEC, 0xEC, 0xEC, 0xEC, 0xEC, 0xEC, 0xEC,
	0xEC, 0xEC, 0xEC, 0xEC, 0xED, 0xED, 0xED, 0xED,
	0xED, 0xED, 0xED, 0xED, 0xED, 0xED, 0xED, 0xED,
	0xED, 0xED, 0xED, 0xED, 0xED, 0xED, 0xED, 0xED,
	0xED, 0xED, 0xED, 0xED, 0xED, 0xED, 0xED, 0xED,
	0xED, 0xED, 0xED, 0xED, 0xED, 0xEE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEF, 0xEF,
	0xEF, 0xEF, 0xEF, 0xEF, 0xEF, 0xEF, 0xEF, 0xEF,
	0xEF, 0xEF, 0xEF, 0xEF, 0xEF, 0xEF, 0xEF, 0xEF,
	0xEF, 0xEF, 0xEF, 0xEF, 0xEF, 0xEF, 0xEF, 0xEF,
	0xEF, 0xEF, 0xEF, 0xEF, 0xEF, 0xEF, 0xEF, 0xEF,
	0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
	0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
	0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
	0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
	0xF0, 0xF0, 0xF1, 0xF1, 0xF1, 0xF1, 0xF1, 0xF1,
	0xF1, 0xF1, 0xF1, 0xF1, 0xF1, 0xF1, 0xF1, 0xF1,
	0xF1, 0xF1, 0xF1, 0xF1, 0xF1, 0xF1, 0xF1, 0xF1,
	0xF1, 0xF1, 0xF1, 0xF1, 0xF1, 0xF1, 0xF1, 0xF1,
	0xF1, 0xF1, 0xF1, 0xF1, 0xF2, 0xF2, 0xF2, 0xF2,
	0xF2, 0xF2, 0xF2, 0xF2, 0xF2, 0xF2, 0xF2, 0xF2,
	0xF2, 0xF2, 0xF2, 0xF2, 0xF2, 0xF2, 0xF2, 0xF2,
	0xF2, 0xF2, 0xF2, 0xF2, 0xF2, 0xF2, 0xF2, 0xF2,
	0xF2, 0xF2, 0xF2, 0xF2, 0xF2, 0xF2, 0xF3, 0xF3,
	0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3,
	0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3,
	0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3,
	0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3,
	0xF4, 0xF4, 0xF4, 0xF4, 0xF4, 0xF4, 0xF4, 0xF4,
	0xF4, 0xF4, 0xF4, 0xF4, 0xF4, 0xF4, 0xF4, 0xF4,
	0xF4, 0xF4, 0xF4, 0xF4, 0xF4, 0xF4, 0xF4, 0xF4,
	0xF4, 0xF4, 0xF4, 0xF4, 0xF4, 0xF4, 0xF4, 0xF4,
	0xF4, 0xF4, 0xF5, 0xF5, 0xF5, 0xF5, 0xF5, 0xF5,
	0xF5, 0xF5, 0xF5, 0xF5, 0xF5, 0xF5, 0xF5, 0xF5,
	0xF5, 0xF5, 0xF5, 0xF5, 0xF5, 0xF5, 0xF5, 0xF5,
	0xF5, 0xF5, 0xF5, 0xF5, 0xF5, 0xF5, 0xF5, 0xF5,
	0xF5, 0xF5, 0xF5, 0xF5, 0xF5, 0xF6, 0xF6, 0xF6,
	0xF6, 0xF6, 0xF6, 0xF6, 0xF6, 0xF6, 0xF6, 0xF6,
	0xF6, 0xF6, 0xF6, 0xF6, 0xF6, 0xF6, 0xF6, 0xF6,
	0xF6, 0xF6, 0xF6, 0xF6, 0xF6, 0xF6, 0xF6, 0xF6,
	0xF6, 0xF6, 0xF6, 0xF6, 0xF6, 0xF6, 0xF6, 0xF6,
	0xF7, 0xF7, 0xF7, 0xF7, 0xF7, 0xF7, 0xF7, 0xF7,
	0xF7, 0xF7, 0xF7, 0xF7, 0xF7, 0xF7, 0xF7, 0xF7,
	0xF7, 0xF7, 0xF7, 0xF7, 0xF7, 0xF7, 0xF7, 0xF7,
	0xF7, 0xF7, 0xF7, 0xF7, 0xF7, 0xF7, 0xF7, 0xF7,
	0xF7, 0xF7, 0xF7, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
	0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
	0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
	0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
	0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF9, 0xF9,
	0xF9, 0xF9, 0xF9, 0xF9, 0xF9, 0xF9, 0xF9, 0xF9,
	0xF9, 0xF9, 0xF9, 0xF9, 0xF9, 0xF9, 0xF9, 0xF9,
	0xF9, 0xF9, 0xF9, 0xF9, 0xF9, 0xF9, 0xF9, 0xF9,
	0xF9, 0xF9, 0xF9, 0xF9, 0xF9, 0xF9, 0xF9, 0xF9,
	0xF9, 0xFA, 0xFA, 0xFA, 0xFA, 0xFA, 0xFA, 0xFA,
	0xFA, 0xFA, 0xFA, 0xFA, 0xFA, 0xFA, 0xFA, 0xFA,
	0xFA, 0xFA, 0xFA, 0xFA, 0xFA, 0xFA, 0xFA, 0xFA,
	0xFA, 0xFA, 0xFA, 0xFA, 0xFA, 0xFA, 0xFA, 0xFA,
	0xFA, 0xFA, 0xFA, 0xFA, 0xFA, 0xFB, 0xFB, 0xFB,
	0xFB, 0xFB, 0xFB, 0xFB, 0xFB, 0xFB, 0xFB, 0xFB,
	0xFB, 0xFB, 0xFB, 0xFB, 0xFB, 0xFB, 0xFB, 0xFB,
	0xFB, 0xFB, 0xFB, 0xFB, 0xFB, 0xFB, 0xFB, 0xFB,
	0xFB, 0xFB, 0xFB, 0xFB, 0xFB, 0xFB, 0xFB, 0xFB,
	0xFB, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC,
	0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC,
	0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC,
	0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC,
	0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFD, 0xFD, 0xFD,
	0xFD, 0xFD, 0xFD, 0xFD, 0xFD, 0xFD, 0xFD, 0xFD,
	0xFD, 0xFD, 0xFD, 0xFD, 0xFD, 0xFD, 0xFD, 0xFD,
	0xFD, 0xFD, 0xFD, 0xFD, 0xFD, 0xFD, 0xFD, 0xFD,
	0xFD, 0xFD, 0xFD, 0xFD, 0xFD, 0xFD, 0xFD, 0xFD,
	0xFD, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
	0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
	0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
	0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
	0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
};

void mdp4_mixer_gc_lut_setup(int mixer_num)
{
	unsigned char *base;
	uint32 data;
	char val;
	int i, off;

	if (mixer_num) 	/* mixer number, /dev/fb0, /dev/fb1 */
		base = MDP_BASE + MDP4_OVERLAYPROC1_BASE;/* 0x18000 */
	else
		base = MDP_BASE + MDP4_OVERLAYPROC0_BASE;/* 0x10000 */

	base += 0x4000;	/* GC_LUT offset */

	off = 0;
	for (i = 0; i < 4096; i++) {
		val = gc_lut[i];
		data = (val << 16 | val << 8 | val); /* R, B, and G are same */
		outpdw(base + off, data);
		off += 4;
	}
}

uint32 igc_video_lut[] = {	 /* non linear */
	0x0, 0x1, 0x2, 0x4, 0x5, 0x6, 0x7, 0x9,
	0xA, 0xB, 0xC, 0xE, 0xF, 0x10, 0x12, 0x14,
	0x15, 0x17, 0x19, 0x1B, 0x1D, 0x1F, 0x21, 0x23,
	0x25, 0x28, 0x2A, 0x2D, 0x30, 0x32, 0x35, 0x38,
	0x3B, 0x3E, 0x42, 0x45, 0x48, 0x4C, 0x4F, 0x53,
	0x57, 0x5B, 0x5F, 0x63, 0x67, 0x6B, 0x70, 0x74,
	0x79, 0x7E, 0x83, 0x88, 0x8D, 0x92, 0x97, 0x9C,
	0xA2, 0xA8, 0xAD, 0xB3, 0xB9, 0xBF, 0xC5, 0xCC,
	0xD2, 0xD8, 0xDF, 0xE6, 0xED, 0xF4, 0xFB, 0x102,
	0x109, 0x111, 0x118, 0x120, 0x128, 0x130, 0x138, 0x140,
	0x149, 0x151, 0x15A, 0x162, 0x16B, 0x174, 0x17D, 0x186,
	0x190, 0x199, 0x1A3, 0x1AC, 0x1B6, 0x1C0, 0x1CA, 0x1D5,
	0x1DF, 0x1EA, 0x1F4, 0x1FF, 0x20A, 0x215, 0x220, 0x22B,
	0x237, 0x242, 0x24E, 0x25A, 0x266, 0x272, 0x27F, 0x28B,
	0x298, 0x2A4, 0x2B1, 0x2BE, 0x2CB, 0x2D8, 0x2E6, 0x2F3,
	0x301, 0x30F, 0x31D, 0x32B, 0x339, 0x348, 0x356, 0x365,
	0x374, 0x383, 0x392, 0x3A1, 0x3B1, 0x3C0, 0x3D0, 0x3E0,
	0x3F0, 0x400, 0x411, 0x421, 0x432, 0x443, 0x454, 0x465,
	0x476, 0x487, 0x499, 0x4AB, 0x4BD, 0x4CF, 0x4E1, 0x4F3,
	0x506, 0x518, 0x52B, 0x53E, 0x551, 0x565, 0x578, 0x58C,
	0x5A0, 0x5B3, 0x5C8, 0x5DC, 0x5F0, 0x605, 0x61A, 0x62E,
	0x643, 0x659, 0x66E, 0x684, 0x699, 0x6AF, 0x6C5, 0x6DB,
	0x6F2, 0x708, 0x71F, 0x736, 0x74D, 0x764, 0x77C, 0x793,
	0x7AB, 0x7C3, 0x7DB, 0x7F3, 0x80B, 0x824, 0x83D, 0x855,
	0x86F, 0x888, 0x8A1, 0x8BB, 0x8D4, 0x8EE, 0x908, 0x923,
	0x93D, 0x958, 0x973, 0x98E, 0x9A9, 0x9C4, 0x9DF, 0x9FB,
	0xA17, 0xA33, 0xA4F, 0xA6C, 0xA88, 0xAA5, 0xAC2, 0xADF,
	0xAFC, 0xB19, 0xB37, 0xB55, 0xB73, 0xB91, 0xBAF, 0xBCE,
	0xBEC, 0xC0B, 0xC2A, 0xC4A, 0xC69, 0xC89, 0xCA8, 0xCC8,
	0xCE8, 0xD09, 0xD29, 0xD4A, 0xD6B, 0xD8C, 0xDAD, 0xDCF,
	0xDF0, 0xE12, 0xE34, 0xE56, 0xE79, 0xE9B, 0xEBE, 0xEE1,
	0xF04, 0xF27, 0xF4B, 0xF6E, 0xF92, 0xFB6, 0xFDB, 0xFFF,
};

void mdp4_vg_igc_lut_setup(int vp_num)
{
	unsigned char *base;
	int i, voff, off;
	uint32 data, val;

	voff = MDP4_VIDEO_OFF * vp_num;
	base = MDP_BASE + MDP4_VIDEO_BASE + voff + 0x5000;

	off = 0;
	for (i = 0; i < 256; i++) {
		val = igc_video_lut[i];
		data = (val << 16 | val);	/* color 0 and 1 */
		outpdw(base + off, data);
		outpdw(base + off + 0x800, val);	/* color 2 */
		off += 4;
	}
}

uint32 igc_rgb_lut[] = {   /* linear */
	0x0, 0x10, 0x20, 0x30, 0x40, 0x50, 0x60, 0x70,
	0x80, 0x91, 0xA1, 0xB1, 0xC1, 0xD1, 0xE1, 0xF1,
	0x101, 0x111, 0x121, 0x131, 0x141, 0x151, 0x161, 0x171,
	0x181, 0x191, 0x1A2, 0x1B2, 0x1C2, 0x1D2, 0x1E2, 0x1F2,
	0x202, 0x212, 0x222, 0x232, 0x242, 0x252, 0x262, 0x272,
	0x282, 0x292, 0x2A2, 0x2B3, 0x2C3, 0x2D3, 0x2E3, 0x2F3,
	0x303, 0x313, 0x323, 0x333, 0x343, 0x353, 0x363, 0x373,
	0x383, 0x393, 0x3A3, 0x3B3, 0x3C4, 0x3D4, 0x3E4, 0x3F4,
	0x404, 0x414, 0x424, 0x434, 0x444, 0x454, 0x464, 0x474,
	0x484, 0x494, 0x4A4, 0x4B4, 0x4C4, 0x4D5, 0x4E5, 0x4F5,
	0x505, 0x515, 0x525, 0x535, 0x545, 0x555, 0x565, 0x575,
	0x585, 0x595, 0x5A5, 0x5B5, 0x5C5, 0x5D5, 0x5E6, 0x5F6,
	0x606, 0x616, 0x626, 0x636, 0x646, 0x656, 0x666, 0x676,
	0x686, 0x696, 0x6A6, 0x6B6, 0x6C6, 0x6D6, 0x6E6, 0x6F7,
	0x707, 0x717, 0x727, 0x737, 0x747, 0x757, 0x767, 0x777,
	0x787, 0x797, 0x7A7, 0x7B7, 0x7C7, 0x7D7, 0x7E7, 0x7F7,
	0x808, 0x818, 0x828, 0x838, 0x848, 0x858, 0x868, 0x878,
	0x888, 0x898, 0x8A8, 0x8B8, 0x8C8, 0x8D8, 0x8E8, 0x8F8,
	0x908, 0x919, 0x929, 0x939, 0x949, 0x959, 0x969, 0x979,
	0x989, 0x999, 0x9A9, 0x9B9, 0x9C9, 0x9D9, 0x9E9, 0x9F9,
	0xA09, 0xA19, 0xA2A, 0xA3A, 0xA4A, 0xA5A, 0xA6A, 0xA7A,
	0xA8A, 0xA9A, 0xAAA, 0xABA, 0xACA, 0xADA, 0xAEA, 0xAFA,
	0xB0A, 0xB1A, 0xB2A, 0xB3B, 0xB4B, 0xB5B, 0xB6B, 0xB7B,
	0xB8B, 0xB9B, 0xBAB, 0xBBB, 0xBCB, 0xBDB, 0xBEB, 0xBFB,
	0xC0B, 0xC1B, 0xC2B, 0xC3B, 0xC4C, 0xC5C, 0xC6C, 0xC7C,
	0xC8C, 0xC9C, 0xCAC, 0xCBC, 0xCCC, 0xCDC, 0xCEC, 0xCFC,
	0xD0C, 0xD1C, 0xD2C, 0xD3C, 0xD4C, 0xD5D, 0xD6D, 0xD7D,
	0xD8D, 0xD9D, 0xDAD, 0xDBD, 0xDCD, 0xDDD, 0xDED, 0xDFD,
	0xE0D, 0xE1D, 0xE2D, 0xE3D, 0xE4D, 0xE5D, 0xE6E, 0xE7E,
	0xE8E, 0xE9E, 0xEAE, 0xEBE, 0xECE, 0xEDE, 0xEEE, 0xEFE,
	0xF0E, 0xF1E, 0xF2E, 0xF3E, 0xF4E, 0xF5E, 0xF6E, 0xF7F,
	0xF8F, 0xF9F, 0xFAF, 0xFBF, 0xFCF, 0xFDF, 0xFEF, 0xFFF,
};

void mdp4_rgb_igc_lut_setup(int num)
{
	unsigned char *base;
	int i, voff, off;
	uint32 data, val;

	voff = MDP4_RGB_OFF * num;
	base = MDP_BASE + MDP4_RGB_BASE + voff + 0x5000;

	off = 0;
	for (i = 0; i < 256; i++) {
		val = igc_rgb_lut[i];
		data = (val << 16 | val);	/* color 0 and 1 */
		outpdw(base + off, data);
		outpdw(base + off + 0x800, val);	/* color 2 */
		off += 4;
	}
}
