/*
 *  QXDM diag though UART.
 *
 *  filename : uart_diag.c
 *  design by: scott.hu
 *  date     : 2014-12-12 
 */
#include <linux/atomic.h>
#include <linux/hrtimer.h>
#include <linux/module.h>
#include <linux/io.h>
#include <linux/ioport.h>
#include <linux/irq.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/console.h>
#include <linux/tty.h>
#include <linux/tty_flip.h>
#include <linux/serial_core.h>
#include <linux/serial.h>
#include <linux/nmi.h>
#include <linux/clk.h>
#include <linux/platform_device.h>
#include <linux/pm_runtime.h>
#include <linux/gpio.h>
#include <linux/debugfs.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <mach/board.h>
#include <asm/mach-types.h>

#include "msm_serial_hs_hwreg.h"

#include <mach/usbdiag.h>

//2014-12-24 add by tommy
#ifdef CONFIG_QUECTEL_UART_DIAG
#include <mach/msm_iomap.h>
#endif
//end tommy

#define UART_DIAG_BUF_SIZE  UART_XMIT_SIZE

#define UART_DIAG_FRMAE_END 0x7e

typedef void(*uart_diag_notifier)(void *, unsigned, struct diag_request *);

struct uart_diag_port {
    struct uart_port uart;
    char             name[16];
    struct clk      *clk;
    struct clk      *pclk;
    unsigned int     imr;
    unsigned int    *gsbi_mapbase;
    unsigned int    *mapped_gsbi;
    int              is_uartdm;
    unsigned int     old_snap_state;
    int              tx_timeout;
	struct circ_buf	 xmit;
	struct circ_buf	 rcvd;
    uart_diag_notifier notifier;
	spinlock_t		 lock,read_lock,write_lock;

	struct workqueue_struct *uart_diag_wq;
	struct work_struct read_work;
    struct diag_request *d_req;
};

#define UART_TO_DIAG(uart_port)	((struct uart_diag_port *) uart_port)

static const unsigned int regmap[UARTDM_LAST] = {
    [UARTDM_MR1] = UARTDM_MR1_ADDR,
    [UARTDM_MR2] = UARTDM_MR2_ADDR,
    [UARTDM_IMR] = UARTDM_IMR_ADDR,
    [UARTDM_SR] = UARTDM_SR_ADDR,
    [UARTDM_CR] = UARTDM_CR_ADDR,
    [UARTDM_CSR] = UARTDM_CSR_ADDR,
    [UARTDM_IPR] = UARTDM_IPR_ADDR,
    [UARTDM_ISR] = UARTDM_ISR_ADDR,
    [UARTDM_RX_TOTAL_SNAP] = UARTDM_RX_TOTAL_SNAP_ADDR,
    [UARTDM_TFWR] = UARTDM_TFWR_ADDR,
    [UARTDM_RFWR] = UARTDM_RFWR_ADDR,
    [UARTDM_RF] = UARTDM_RF_ADDR,
    [UARTDM_TF] = UARTDM_TF_ADDR,
    [UARTDM_MISR] = UARTDM_MISR_ADDR,
    [UARTDM_DMRX] = UARTDM_DMRX_ADDR,
    [UARTDM_NCF_TX] = UARTDM_NCF_TX_ADDR,
    [UARTDM_DMEN] = UARTDM_DMEN_ADDR,
    [UARTDM_TXFS] = UARTDM_TXFS_ADDR,
    [UARTDM_RXFS] = UARTDM_RXFS_ADDR,
};

static struct uart_diag_port diag_uart_port = {
    .uart = {
        .fifosize = 64,
    },
};

static inline void uart_write(struct uart_port *port, unsigned int val, unsigned int off);
static inline unsigned int uart_read(struct uart_port *port, unsigned int off);

static void dump_uart_regs(struct uart_port *port)
{
    struct uart_diag_port *uart_diag_port = UART_TO_DIAG(port);
    unsigned int sr, isr, mr1, mr2, ncf, txfs, rxfs;

    sr = uart_read(port, regmap[UARTDM_SR]);
    isr = uart_read(port, regmap[UARTDM_ISR]);
    mr1 = uart_read(port, regmap[UARTDM_MR1]);
    mr2 = uart_read(port, regmap[UARTDM_MR2]);
    ncf = uart_read(port, regmap[UARTDM_NCF_TX]);
    txfs = uart_read(port, regmap[UARTDM_TXFS]);
    rxfs = uart_read(port, regmap[UARTDM_RXFS]);

    pr_info("%s(): Timeout: %d uS\n", __func__, uart_diag_port->tx_timeout);
    pr_info("%s(): SR:  %08x\n", __func__, sr);
    pr_info("%s(): ISR: %08x\n", __func__, isr);
    pr_info("%s(): MR1: %08x\n", __func__, mr1);
    pr_info("%s(): MR2: %08x\n", __func__, mr2);
    pr_info("%s(): NCF: %08x\n", __func__, ncf);
    pr_info("%s(): TXFS: %08x\n", __func__, txfs);
    pr_info("%s(): RXFS: %08x\n", __func__, rxfs);
}

/*
 *  Wait for transmitter & holding register to empty
 *  Derived from wait_for_xmitr in 8250 serial driver by Russell King  */
static inline void wait_for_xmitr(struct uart_port *port)
{
    struct uart_diag_port *uart_diag_port = UART_TO_DIAG(port);
    int count = 0;

    if (!(uart_read(port, regmap[UARTDM_SR]) &
            UARTDM_SR_TXEMT_BMSK))
    {
        while (!(uart_read(port, regmap[UARTDM_ISR]) &
                UARTDM_ISR_TX_READY_BMSK))
        {
            udelay(1);
            touch_nmi_watchdog();
            cpu_relax();
            if (++count == uart_diag_port->tx_timeout)
            {
                dump_uart_regs(port);
                panic("MSM HSL wait_for_xmitr is stuck!");
            }
        }
        uart_write(port, CLEAR_TX_READY, regmap[UARTDM_CR]);
    }
}

static inline void uart_write(struct uart_port *port,
                 unsigned int val, unsigned int off)
{
    iowrite32(val, port->membase + off);
}

static inline unsigned int uart_read(struct uart_port *port,
             unsigned int off)
{
    return ioread32(port->membase + off);
}

static unsigned int port_is_gsbi(struct uart_port *port)
{
    return UART_TO_DIAG(port)->is_uartdm;
}

static int clk_en(struct uart_port *port, int enable)
{
    struct uart_diag_port *uart_diag_port = UART_TO_DIAG(port);
    int ret = 0;

    if (enable)
    {
        ret = clk_prepare_enable(uart_diag_port->clk);
        if (ret)
        {
            goto err;
        }
        if (uart_diag_port->pclk)
        {
            ret = clk_prepare_enable(uart_diag_port->pclk);
            if (ret)
            {
                clk_disable_unprepare(uart_diag_port->clk);
                goto err;
            }
        }
    }
    else
    {
        clk_disable_unprepare(uart_diag_port->clk);
        if (uart_diag_port->pclk)
        {
            clk_disable_unprepare(uart_diag_port->pclk);
        }
    }
err:
    return ret;
}

static void uart_diag_init_clock(struct uart_port *port)
{
    clk_en(port, 1);
}

static void uart_diag_deinit_clock(struct uart_port *port)
{
   clk_en(port, 0);
}

static void uart_diag_stop_tx(struct uart_port *port)
{
    struct uart_diag_port *uart_diag_port = UART_TO_DIAG(port);
    
    uart_diag_port->imr &= ~UARTDM_ISR_TXLEV_BMSK;
    uart_write(port, uart_diag_port->imr, regmap[UARTDM_IMR]);
}

static void uart_diag_start_tx(struct uart_port *port)
{
    struct uart_diag_port *uart_diag_port = UART_TO_DIAG(port);
    
    uart_diag_port->imr |= UARTDM_ISR_TXLEV_BMSK;
    uart_write(port, uart_diag_port->imr, regmap[UARTDM_IMR]);
}

static void handle_rx(struct uart_port *port, unsigned int misr)
{
    unsigned long flags;
    unsigned int sr;
    int count = 0,cnt,w_cnt;
    char *w_buf;
	struct circ_buf *circ = &diag_uart_port.rcvd;
    struct uart_diag_port *diag_port = UART_TO_DIAG(port);

    /*
     * Handle overrun. My understanding of the hardware is that overrun
     * is not tied to the RX buffer, so we handle the case out of band.
     */
    if ((uart_read(port, regmap[UARTDM_SR]) &
                UARTDM_SR_OVERRUN_BMSK))
    {
        port->icount.overrun++;
        uart_write(port, RESET_ERROR_STATUS,
            regmap[UARTDM_CR]);
    }

    if (misr & UARTDM_ISR_RXSTALE_BMSK)
    {
        count = uart_read(port,
            regmap[UARTDM_RX_TOTAL_SNAP]) -
            diag_port->old_snap_state;
        diag_port->old_snap_state = 0;
    }
    else
    {
        count = 4 * (uart_read(port, regmap[UARTDM_RFWR]));
        diag_port->old_snap_state += count;
    }
    
    /* and now the main RX loop */
    while (count > 0)
    {
        unsigned int c;
        char flag = TTY_NORMAL;

        sr = uart_read(port, regmap[UARTDM_SR]);
        if ((sr & UARTDM_SR_RXRDY_BMSK) == 0)
        {
            diag_port->old_snap_state -= count;
            break;
        }
        c = uart_read(port, regmap[UARTDM_RF]);
        if (sr & UARTDM_SR_RX_BREAK_BMSK)
        {
            port->icount.brk++;
            if (uart_handle_break(port))
                continue;
        }
        else if (sr & UARTDM_SR_PAR_FRAME_BMSK)
        {
            port->icount.frame++;
        }
        else
        {
            port->icount.rx++;
        }

        /* Mask conditions we're ignorning. */
        sr &= port->read_status_mask;
        if (sr & UARTDM_SR_RX_BREAK_BMSK)
            flag = TTY_BREAK;
        else if (sr & UARTDM_SR_PAR_FRAME_BMSK)
            flag = TTY_FRAME;

        spin_lock_irqsave(&diag_port->read_lock,flags);
        w_cnt = (count > 4) ? 4 : count;
        w_buf = (char *)&c;

        while(1)
        {
            cnt = CIRC_SPACE_TO_END(circ->head, circ->tail, UART_DIAG_BUF_SIZE);
            
            if (w_cnt < cnt)
                cnt = w_cnt;
            if (cnt <= 0)
                break;
            memcpy(circ->buf + circ->head, w_buf, cnt);
            circ->head = (circ->head + cnt) & (UART_DIAG_BUF_SIZE - 1);
            w_buf += cnt;
            w_cnt -= cnt;
        }
        spin_unlock_irqrestore(&diag_port->read_lock,flags);
        count -= 4;
    }
    
    queue_work(diag_port->uart_diag_wq, &(diag_port->read_work));
}

static void handle_tx(struct uart_port *port)
{
    struct uart_diag_port *diag_port = UART_TO_DIAG(port);
    struct circ_buf *xmit = &diag_port->xmit;
    int tx_count;
    int x;
    unsigned int tf_pointer = 0;
    unsigned long flags;

    tx_count = uart_circ_chars_pending(xmit);
    
    if (tx_count > (UART_DIAG_BUF_SIZE - xmit->tail))
        tx_count = UART_DIAG_BUF_SIZE - xmit->tail;
    
    if (tx_count >= port->fifosize)
        tx_count = port->fifosize;

    if (tx_count)
    {
        wait_for_xmitr(port);
        uart_write(port, tx_count, regmap[UARTDM_NCF_TX]);
        uart_read(port, regmap[UARTDM_NCF_TX]);
    }
    
    if (!tx_count)
    {
        uart_diag_stop_tx(port);
        return;
    }

    spin_lock_irqsave(&diag_port->write_lock,flags);

    while (tf_pointer < tx_count)
    {
        if (unlikely(!(uart_read(port, regmap[UARTDM_SR]) &
                   UARTDM_SR_TXRDY_BMSK)))
        {
            continue;
        }
        
        switch (tx_count - tf_pointer)
        {
            case 1:
            {
                x = xmit->buf[xmit->tail];
                port->icount.tx++;
                break;
            }
            case 2:
            {
                x = xmit->buf[xmit->tail]
                    | xmit->buf[xmit->tail+1] << 8;
                port->icount.tx += 2;
                break;
            }
            case 3:
            {
                x = xmit->buf[xmit->tail]
                    | xmit->buf[xmit->tail+1] << 8
                    | xmit->buf[xmit->tail + 2] << 16;
                port->icount.tx += 3;
                break;
            }
            default:
            {
                x = *((int *)&(xmit->buf[xmit->tail]));
                port->icount.tx += 4;
                break;
            }
        }
        uart_write(port, x, regmap[UARTDM_TF]);
        xmit->tail = ((tx_count - tf_pointer < 4) ?
                  (tx_count - tf_pointer + xmit->tail) :
                  (xmit->tail + 4)) & (UART_DIAG_BUF_SIZE - 1);
        tf_pointer += 4;
    }

    if (uart_circ_empty(xmit))
        uart_diag_stop_tx(port);

    spin_unlock_irqrestore(&diag_port->write_lock,flags);
}

static irqreturn_t uart_diag_irq(int irq, void *dev_id)
{
    struct uart_port *port = dev_id;
    struct uart_diag_port *diag_port = UART_TO_DIAG(port);
    unsigned int misr;
    unsigned long flags;

    spin_lock_irqsave(&port->lock, flags);
    misr = uart_read(port, regmap[UARTDM_MISR]);
    /* disable interrupt */
    uart_write(port, 0, regmap[UARTDM_IMR]);

    if (misr & (UARTDM_ISR_RXSTALE_BMSK | UARTDM_ISR_RXLEV_BMSK))
    {
        handle_rx(port, misr);
        if (misr & (UARTDM_ISR_RXSTALE_BMSK))
        {
            uart_write(port, RESET_STALE_INT, regmap[UARTDM_CR]);
        }
        uart_write(port, 6500, regmap[UARTDM_DMRX]);
        uart_write(port, STALE_EVENT_ENABLE, regmap[UARTDM_CR]);
    }
    
    if (misr & UARTDM_ISR_TXLEV_BMSK)
    {
        handle_tx(port);
    }

    /* restore interrupt */
    uart_write(port, diag_port->imr, regmap[UARTDM_IMR]);
    spin_unlock_irqrestore(&port->lock, flags);

    return IRQ_HANDLED;
}

static void uart_diag_reset(struct uart_port *port)
{
    /* reset everything */
    uart_write(port, RESET_RX, regmap[UARTDM_CR]);
    uart_write(port, RESET_TX, regmap[UARTDM_CR]);
    uart_write(port, RESET_ERROR_STATUS, regmap[UARTDM_CR]);
    uart_write(port, RESET_BREAK_INT, regmap[UARTDM_CR]);
    uart_write(port, RESET_CTS, regmap[UARTDM_CR]);
    uart_write(port, RFR_LOW, regmap[UARTDM_CR]);
}

static void uart_diag_set_baud_rate(struct uart_port *port, unsigned int baud)
{
    unsigned int baud_code, rxstale, watermark;
    unsigned int data;
    struct uart_diag_port *uart_diag_port = UART_TO_DIAG(port);

    switch (baud) {
    case 300:
        baud_code = UARTDM_CSR_75;
        rxstale = 1;
        break;
    case 600:
        baud_code = UARTDM_CSR_150;
        rxstale = 1;
        break;
    case 1200:
        baud_code = UARTDM_CSR_300;
        rxstale = 1;
        break;
    case 2400:
        baud_code = UARTDM_CSR_600;
        rxstale = 1;
        break;
    case 4800:
        baud_code = UARTDM_CSR_1200;
        rxstale = 1;
        break;
    case 9600:
        baud_code = UARTDM_CSR_2400;
        rxstale = 2;
        break;
    case 14400:
        baud_code = UARTDM_CSR_3600;
        rxstale = 3;
        break;
    case 19200:
        baud_code = UARTDM_CSR_4800;
        rxstale = 4;
        break;
    case 28800:
        baud_code = UARTDM_CSR_7200;
        rxstale = 6;
        break;
    case 38400:
        baud_code = UARTDM_CSR_9600;
        rxstale = 8;
        break;
    case 57600:
        baud_code = UARTDM_CSR_14400;
        rxstale = 16;
        break;
    case 115200:
        baud_code = UARTDM_CSR_28800;
        rxstale = 31;
        break;
    case 230400:
        baud_code = UARTDM_CSR_57600;
        rxstale = 31;
        break;
    case 460800:
        baud_code = UARTDM_CSR_115200;
        rxstale = 31;
        break;
    default: /* 115200 baud rate */
        baud_code = UARTDM_CSR_28800;
        rxstale = 31;
        break;
    }

    /* Set timeout to be ~600x the character transmit time */
    uart_diag_port->tx_timeout = (1000000000 / baud) * 6;

    uart_write(port, baud_code, regmap[UARTDM_CSR]);

    /* RX stale watermark */
    watermark = UARTDM_IPR_STALE_LSB_BMSK & rxstale;
    watermark |= UARTDM_IPR_STALE_TIMEOUT_MSB_BMSK & (rxstale << 2);
    uart_write(port, watermark, regmap[UARTDM_IPR]);

    /* Set RX watermark
     * Configure Rx Watermark as 3/4 size of Rx FIFO.
     * RFWR register takes value in Words for UARTDM Core
     * whereas it is consider to be in Bytes for UART Core.
     * Hence configuring Rx Watermark as 12 Words.
     */
    watermark = (port->fifosize * 3) / (4*4);
    uart_write(port, watermark, regmap[UARTDM_RFWR]);

    /* set TX watermark */
    uart_write(port, 0, regmap[UARTDM_TFWR]);

    uart_write(port, CR_PROTECTION_EN, regmap[UARTDM_CR]);
    uart_diag_reset(port);

    data = UARTDM_CR_TX_EN_BMSK;
    data |= UARTDM_CR_RX_EN_BMSK;
    /* enable TX & RX */
    uart_write(port, data, regmap[UARTDM_CR]);

    uart_write(port, RESET_STALE_INT, regmap[UARTDM_CR]);
    /* turn on RX and CTS interrupts */
    uart_diag_port->imr = UARTDM_ISR_RXSTALE_BMSK
        | UARTDM_ISR_DELTA_CTS_BMSK | UARTDM_ISR_RXLEV_BMSK;
    uart_write(port, uart_diag_port->imr, regmap[UARTDM_IMR]);
    uart_write(port, 6500, regmap[UARTDM_DMRX]);
    uart_write(port, STALE_EVENT_ENABLE, regmap[UARTDM_CR]);
}

static void uart_diag_set_termios(struct uart_port *port, int baudrate)
{
    unsigned long flags;
    unsigned int  mr, uartclk;
    struct uart_diag_port *diag_port = UART_TO_DIAG(port);

    spin_lock_irqsave(&port->lock, flags);

    if(baudrate > 460800)
    {
        uartclk = baudrate * 16;
    }
    else
    {
        uartclk = 7372800;
    }
    clk_set_rate(diag_port->clk, uartclk);

    uart_diag_set_baud_rate(port, baudrate);

    /* set parity to no parity. */
    mr = uart_read(port, regmap[UARTDM_MR2]);
    mr &= ~UARTDM_MR2_PARITY_MODE_BMSK;
    mr |= NO_PARITY;

    /* set 8 bits per char.*/
    mr &= ~UARTDM_MR2_BITS_PER_CHAR_BMSK;
    mr |= EIGHT_BPC;

    /* set 1 stop bit. */
    mr &= ~(STOP_BIT_ONE | STOP_BIT_TWO);
    mr |= STOP_BIT_ONE;

    /* set parity, bits per char, and stop bit */
    uart_write(port, mr, regmap[UARTDM_MR2]);

    /* no hardware flow control */
    mr = uart_read(port, regmap[UARTDM_MR1]);
    mr &= ~(UARTDM_MR1_CTS_CTL_BMSK | UARTDM_MR1_RX_RDY_CTL_BMSK);
    uart_write(port, mr, regmap[UARTDM_MR1]);

    /* Configure status bits to ignore based on termio flags. */
    port->read_status_mask = 0;
    port->read_status_mask |= UARTDM_SR_PAR_FRAME_BMSK;
    port->read_status_mask |= UARTDM_SR_RX_BREAK_BMSK;

    spin_unlock_irqrestore(&port->lock, flags);
}

static void uart_diag_release_port(struct uart_port *port)
{
    struct uart_diag_port *uart_diag_port = UART_TO_DIAG(port);
    struct platform_device *pdev = to_platform_device(port->dev);
    struct resource *uart_resource;
    resource_size_t size;

    uart_resource = platform_get_resource_byname(pdev, IORESOURCE_MEM,
                             "uartdm_resource");
    if (!uart_resource)
    {
        uart_resource = platform_get_resource(pdev, IORESOURCE_MEM, 0);
    }
    if (unlikely(!uart_resource))
    {
        return;
    }
    size = uart_resource->end - uart_resource->start + 1;

    release_mem_region(port->mapbase, size);
    iounmap(port->membase);
    port->membase = NULL;

    if (port_is_gsbi(port))
    {
        iowrite32(GSBI_PROTOCOL_IDLE, uart_diag_port->mapped_gsbi + GSBI_CONTROL_ADDR);
        iounmap(uart_diag_port->mapped_gsbi);
        uart_diag_port->mapped_gsbi = NULL;
    }

    uart_diag_port->imr = 0;
    /* disable interrupts */
    uart_write(port, 0, regmap[UARTDM_IMR]);

    free_irq(port->irq, port);

    uart_diag_deinit_clock(port);
    pm_runtime_put_sync(port->dev);
}

static void uart_diag_config_port(struct uart_port *port)
{
    struct uart_diag_port *uart_diag_port = UART_TO_DIAG(port);
    struct platform_device *pdev = to_platform_device(port->dev);
    struct resource *uart_resource;
    struct resource *gsbi_resource;
	unsigned int data, rfr_level;
    resource_size_t size;
	unsigned long flags;
    int ret;

    uart_resource = platform_get_resource_byname(pdev, IORESOURCE_MEM,
                             "uartdm_resource");
    if (!uart_resource)
    {
        uart_resource = platform_get_resource(pdev, IORESOURCE_MEM, 0);
    }
    if (unlikely(!uart_resource))
    {
        pr_err("%s: can't get uartdm resource\n", __func__);
        return;
    }
    size = uart_resource->end - uart_resource->start + 1;

    if (unlikely(!request_mem_region(port->mapbase, size, "uart_diag")))
    {
        pr_err("%s: can't get mem region for uartdm\n", __func__);
        return;
    }

    port->membase = ioremap(port->mapbase, size);
    if (!port->membase)
    {
        release_mem_region(port->mapbase, size);
        return;
    }

    if (port_is_gsbi(port))
    {
        gsbi_resource = platform_get_resource_byname(pdev,
                                 IORESOURCE_MEM,
                                 "gsbi_resource");
        if (!gsbi_resource)
            gsbi_resource = platform_get_resource(pdev,
                        IORESOURCE_MEM, 1);
        if (unlikely(!gsbi_resource))
        {
            pr_err("%s: can't get gsbi resource\n", __func__);
            return;
        }

        size = gsbi_resource->end - gsbi_resource->start + 1;
        uart_diag_port->mapped_gsbi = ioremap(gsbi_resource->start, size);
        if (!uart_diag_port->mapped_gsbi)
        {
            return;
        }

        if (uart_diag_port->pclk)
        {
            clk_prepare_enable(uart_diag_port->pclk);
        }
        
        if ((ioread32(uart_diag_port->mapped_gsbi + GSBI_CONTROL_ADDR) &
            GSBI_PROTOCOL_I2C_UART) != GSBI_PROTOCOL_I2C_UART)
        {
            iowrite32(GSBI_PROTOCOL_I2C_UART,
                uart_diag_port->mapped_gsbi + GSBI_CONTROL_ADDR);
        }
        
        if (uart_diag_port->pclk)
        {
            clk_disable_unprepare(uart_diag_port->pclk);
        }
    }

    
    clk_set_rate(uart_diag_port->clk, 7372800);

    // GPIO was inited in msm9615_init_gpiomux.
	pm_runtime_get_noresume(port->dev);
    uart_diag_init_clock(port);
    pm_runtime_get_sync(port->dev);

    /* Set RFR Level as 3/4 of UARTDM FIFO Size */
    if (likely(port->fifosize > 48))
        rfr_level = port->fifosize - 16;
    else
        rfr_level = port->fifosize;

    /*
     * Use rfr_level value in Words to program
     * MR1 register for UARTDM Core.
     */
    rfr_level = (rfr_level / 4);

    spin_lock_irqsave(&port->lock, flags);

    /* set automatic RFR level */
    data = uart_read(port, regmap[UARTDM_MR1]);
    data &= ~UARTDM_MR1_AUTO_RFR_LEVEL1_BMSK;
    data &= ~UARTDM_MR1_AUTO_RFR_LEVEL0_BMSK;
    data |= UARTDM_MR1_AUTO_RFR_LEVEL1_BMSK & (rfr_level << 2);
    data |= UARTDM_MR1_AUTO_RFR_LEVEL0_BMSK & rfr_level;
    uart_write(port, data, regmap[UARTDM_MR1]);
    spin_unlock_irqrestore(&port->lock, flags);

    ret = request_irq(port->irq, uart_diag_irq, IRQF_TRIGGER_HIGH,
                        uart_diag_port->name, port);
    if (unlikely(ret))
    {
        printk(KERN_ERR "%s: failed to request_irq\n", __func__);
    }
}

static int uart_diag_read_a_frame
(
    char *dst_buf, 
    int *dst_len, 
    struct circ_buf *c_buf, 
    int src_len
)
{
    int dst_cnt = 0, ret = 0;
    char c;

    while(src_len--)
    {
        c = c_buf->buf[c_buf->tail++];
        if(c_buf->tail >= UART_DIAG_BUF_SIZE)
        {
            c_buf->tail = 0;
        }
        dst_buf[dst_cnt++] = c;
        
        if(c == UART_DIAG_FRMAE_END && dst_cnt != 1)
        {
            ret = 1;
            break;
        }
    }

    *dst_len = dst_cnt;

    return ret;
}

static void uart_diag_read_work_fn(struct work_struct *work)
{
    int buf_cnt, copied;
    unsigned long flags;
	struct circ_buf *circ = &diag_uart_port.rcvd;

    if(diag_uart_port.d_req == NULL)
    {
        pr_err("%s: null req.\n", __func__);
        return;
    }    
    
    spin_lock_irqsave(&diag_uart_port.read_lock, flags);

    buf_cnt = CIRC_CNT(circ->head, circ->tail, UART_DIAG_BUF_SIZE);
    
    if(buf_cnt >= 4)
    {
        if(uart_diag_read_a_frame(
           (diag_uart_port.d_req->buf + diag_uart_port.d_req->actual),
           &copied, circ, buf_cnt))
        {
            diag_uart_port.d_req->actual += copied;
            diag_uart_port.notifier(NULL, USB_DIAG_READ_DONE, diag_uart_port.d_req);
            diag_uart_port.d_req = NULL;
        }
        else
        {
            diag_uart_port.d_req->actual += copied;
        }
    }
    spin_unlock_irqrestore(&diag_uart_port.read_lock, flags);
}

int uart_diag_open(void *notifier)
{
    pr_err("%s: entry.\n", __func__);
    
    //create tx queue.
    diag_uart_port.xmit.buf = (char *)kzalloc(UART_DIAG_BUF_SIZE, GFP_KERNEL);
    diag_uart_port.xmit.head = diag_uart_port.xmit.tail = 0;

    //create rx queue.
    diag_uart_port.rcvd.buf = (char *)kzalloc(UART_DIAG_BUF_SIZE, GFP_KERNEL);
    diag_uart_port.rcvd.head = diag_uart_port.rcvd.tail = 0;

    diag_uart_port.notifier = (uart_diag_notifier)notifier;

	diag_uart_port.uart_diag_wq = create_singlethread_workqueue("uart_diag_wq");
    INIT_WORK(&diag_uart_port.read_work, uart_diag_read_work_fn);
    
    uart_diag_config_port(&diag_uart_port.uart);
    uart_diag_set_termios(&diag_uart_port.uart, 115200);

    diag_uart_port.notifier(NULL, USB_DIAG_CONNECT, NULL);
    
    return 0;
}
EXPORT_SYMBOL(uart_diag_open);
    
void uart_diag_close(void)
{
    uart_diag_release_port(&diag_uart_port.uart);

    kfree(diag_uart_port.xmit.buf);
    kfree(diag_uart_port.rcvd.buf);
    destroy_workqueue(diag_uart_port.uart_diag_wq);

    diag_uart_port.notifier(NULL, USB_DIAG_DISCONNECT, NULL);

    diag_uart_port.notifier = NULL;
}
EXPORT_SYMBOL(uart_diag_close);

int uart_diag_read(struct diag_request *d_req)
{
    unsigned long flags;

    spin_lock_irqsave(&diag_uart_port.read_lock,flags);

    d_req->actual = 0;
    diag_uart_port.d_req = d_req;
    
    spin_unlock_irqrestore(&diag_uart_port.read_lock,flags);

    return 0;
}
EXPORT_SYMBOL(uart_diag_read);

int uart_diag_write(struct diag_request *d_req)
{
    unsigned long flags;
    
	int c, len = d_req->length;
    char *in_buf = d_req->buf;
	struct circ_buf *circ = &diag_uart_port.xmit;

    if(!len)
    {
        return -1;
    }

    spin_lock_irqsave(&diag_uart_port.write_lock,flags);
	while(1)
    {
		c = CIRC_SPACE_TO_END(circ->head, circ->tail, UART_DIAG_BUF_SIZE);
		if (len < c)
			c = len;
		if (c <= 0)
			break;
		memcpy(circ->buf + circ->head, in_buf, c);
		circ->head = (circ->head + c) & (UART_DIAG_BUF_SIZE - 1);
		in_buf += c;
        len -= c;
	}
    spin_unlock_irqrestore(&diag_uart_port.write_lock,flags);

    uart_diag_start_tx(&diag_uart_port.uart);

    diag_uart_port.notifier(NULL, USB_DIAG_WRITE_DONE, d_req);

    return 0;
}
EXPORT_SYMBOL(uart_diag_write);

static int __devinit uart_diag_probe(struct platform_device *pdev)
{
    struct uart_diag_port *uart_diag_port;
    struct resource *uart_resource;
    struct resource *gsbi_resource;
    struct uart_port *port;

    printk("%s: detected port %d\n",__func__, pdev->id);

    port = &diag_uart_port.uart;
    port->dev = &pdev->dev;
    uart_diag_port = UART_TO_DIAG(port);

    gsbi_resource = platform_get_resource_byname(pdev,
                                                IORESOURCE_MEM,
                                                "gsbi_resource");
    if (!gsbi_resource)
    {
        gsbi_resource = platform_get_resource(pdev, IORESOURCE_MEM, 1);
    }

    uart_diag_port->clk = clk_get(&pdev->dev, "core_clk");
    if (gsbi_resource)
    {
        uart_diag_port->is_uartdm = 1;
        uart_diag_port->pclk = clk_get(&pdev->dev, "iface_clk");
    }
    else
    {
        uart_diag_port->is_uartdm = 0;
        uart_diag_port->pclk = NULL;
    }

    if (unlikely(IS_ERR(uart_diag_port->clk)))
    {
        printk(KERN_ERR "%s: Error getting clk\n", __func__);
        return PTR_ERR(uart_diag_port->clk);
    }
    if (unlikely(IS_ERR(uart_diag_port->pclk)))
    {
        printk(KERN_ERR "%s: Error getting pclk\n", __func__);
        return PTR_ERR(uart_diag_port->pclk);
    }

    uart_resource = platform_get_resource_byname(pdev,
                                                IORESOURCE_MEM,
                                                "uartdm_resource");
    if (!uart_resource)
    {
        uart_resource = platform_get_resource(pdev, IORESOURCE_MEM, 0);
    }
    if (unlikely(!uart_resource))
    {
        printk(KERN_ERR "getting uartdm_resource failed\n");
        return -ENXIO;
    }
    port->mapbase = uart_resource->start;

    port->irq = platform_get_irq(pdev, 0);
    if (unlikely((int)port->irq < 0))
    {
        printk(KERN_ERR "%s: getting irq failed\n", __func__);
        return -ENXIO;
    }

    device_set_wakeup_capable(&pdev->dev, 1);
    platform_set_drvdata(pdev, port);

    /* Temporarily increase the refcount on the GSBI clock to avoid a race
    * condition with the earlyprintk handover mechanism.
    */
    if (uart_diag_port->pclk)
        clk_prepare_enable(uart_diag_port->pclk);
    if (uart_diag_port->pclk)
        clk_disable_unprepare(uart_diag_port->pclk);
    
    return 0;
}

static int __devexit uart_diag_remove(struct platform_device *pdev)
{
    struct uart_diag_port *uart_diag_port = platform_get_drvdata(pdev);
    struct uart_port *port;

    port = &diag_uart_port.uart;

    pm_runtime_put_sync(&pdev->dev);
    pm_runtime_disable(&pdev->dev);

    device_set_wakeup_capable(&pdev->dev, 0);
    platform_set_drvdata(pdev, NULL);

    clk_put(uart_diag_port->pclk);
    clk_put(uart_diag_port->clk);

    return 0;
}

#ifdef CONFIG_PM
static int uart_diag_suspend(struct device *dev)
{
    struct uart_port *port;
    port = &diag_uart_port.uart;

    if (port)
    {
        uart_diag_deinit_clock(port);

        if (device_may_wakeup(dev))
        {
            enable_irq_wake(port->irq);
        }
    }

    return 0;
}

static int uart_diag_resume(struct device *dev)
{
    struct uart_port *port;
    port = &diag_uart_port.uart;

    if (port)
    {
        if (device_may_wakeup(dev))
        {
            disable_irq_wake(port->irq);
        }

        uart_diag_init_clock(port);
    }

    return 0;
}
#else
#define uart_diag_suspend NULL
#define uart_diag_resume NULL
#endif

static int uart_diag_runtime_suspend(struct device *dev)
{
    struct uart_port *port;
    port = &diag_uart_port.uart;

    dev_dbg(dev, "pm_runtime: suspending\n");
    uart_diag_deinit_clock(port);
    return 0;
}

static int uart_diag_runtime_resume(struct device *dev)
{
    struct uart_port *port;
    port = &diag_uart_port.uart;

    dev_dbg(dev, "pm_runtime: resuming\n");
    uart_diag_init_clock(port);
    return 0;
}

static struct dev_pm_ops uart_diag_dev_pm_ops = {
    .suspend = uart_diag_suspend,
    .resume = uart_diag_resume,
    .runtime_suspend = uart_diag_runtime_suspend,
    .runtime_resume = uart_diag_runtime_resume,
};

static struct platform_driver uart_diag_platform_driver = {
    .probe = uart_diag_probe,
    .remove = __devexit_p(uart_diag_remove),
    .driver = {
        .name = "uart_diag",
        .owner = THIS_MODULE,
        .pm = &uart_diag_dev_pm_ops,
    },
};

extern unsigned quectel_diag_val;
static int __init uart_diag_init(void)
{
    int ret=0;
    void *quectel_diag_flag;
    #define QUECTEL_UART_DIAG_FLAG_ADDR 0x65c + 0x4
    quectel_diag_flag = MSM_IMEM_BASE + QUECTEL_UART_DIAG_FLAG_ADDR;
    quectel_diag_val = __raw_readl(quectel_diag_flag);
 if(0x96159615 == quectel_diag_val)   
 {
    printk(KERN_INFO "%s: entry.\n", __func__);

    ret = platform_driver_register(&uart_diag_platform_driver);
    if (unlikely(ret))
    {
        printk(KERN_ERR "driver register error.\n");
    }
 }   
    return ret;
}

static void __exit uart_diag_exit(void)
{
//2014-12-24 add by tommy
if(0x96159615 == quectel_diag_val)
    platform_driver_unregister(&uart_diag_platform_driver);
}

module_init(uart_diag_init);
module_exit(uart_diag_exit);

MODULE_AUTHOR("scott Hu <scott.hu@quectel.com>");
MODULE_DESCRIPTION("Driver for UART diag");
MODULE_LICENSE("GPL v2");

