/*
    UART used to AT command.

    filename: uart_smd.c
    design by: scott.hu
    date: 2014-12-25
 */

#include <linux/module.h>

#include <linux/serial.h>
#include <linux/serial_core.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/io.h>
#include <linux/ioport.h>
#include <linux/kernel.h>
#include <linux/timer.h>
#include <linux/delay.h>
#include <linux/clk.h>
#include <linux/platform_device.h>
#include <linux/pm_runtime.h>
#include <linux/dma-mapping.h>
#include <linux/dmapool.h>
#include <linux/wait.h>
#include <linux/sysfs.h>
#include <linux/stat.h>
#include <linux/device.h>
#include <linux/wakelock.h>
#include <linux/debugfs.h>
#include <linux/gpio.h>
#include <asm/atomic.h>
#include <asm/irq.h>
#include <asm/system.h>

#include <mach/hardware.h>
#include <mach/dma.h>
#include <mach/msm_serial_hs.h>

#include <mach/msm_smd.h>

#include "msm_serial_hs_hwreg.h"

enum flush_reason {
	FLUSH_NONE,
	FLUSH_DATA_READY,
	FLUSH_DATA_INVALID,  /* values after this indicate invalid data */
	FLUSH_IGNORE = FLUSH_DATA_INVALID,
	FLUSH_STOP,
	FLUSH_SHUTDOWN,
};

enum msm_hs_clk_states_e {
	MSM_HS_CLK_PORT_OFF,     /* port not in use */
	MSM_HS_CLK_OFF,          /* clock disabled */
	MSM_HS_CLK_REQUEST_OFF,  /* disable after TX and RX flushed */
	MSM_HS_CLK_ON,           /* clock enabled */
};

/* Track the forced RXSTALE flush during clock off sequence.
 * These states are only valid during MSM_HS_CLK_REQUEST_OFF */
enum msm_hs_clk_req_off_state_e {
	CLK_REQ_OFF_START,
	CLK_REQ_OFF_RXSTALE_ISSUED,
	CLK_REQ_OFF_FLUSH_ISSUED,
	CLK_REQ_OFF_RXSTALE_FLUSHED,
};

struct smd_port_tx {
	unsigned int tx_ready_int_en;  /* ok to dma more tx */
	unsigned int dma_in_flight;    /* tx dma in progress */
	enum flush_reason flush;
	wait_queue_head_t wait;
	struct msm_dmov_cmd xfer;
	dmov_box *command_ptr;
	u32 *command_ptr_ptr;
	dma_addr_t mapped_cmd_ptr;
	dma_addr_t mapped_cmd_ptr_ptr;
	int tx_count;
	dma_addr_t dma_base;
	struct tasklet_struct tlet;
};

struct smd_port_rx {
	enum flush_reason flush;
	struct msm_dmov_cmd xfer;
	dma_addr_t cmdptr_dmaaddr;
	dmov_box *command_ptr;
	u32 *command_ptr_ptr;
	dma_addr_t mapped_cmd_ptr;
	wait_queue_head_t wait;
	dma_addr_t rbuffer;
	unsigned char *buffer;
	unsigned int buffer_pending;
	struct dma_pool *pool;
	struct wake_lock wake_lock;
	struct delayed_work write_smd_work;
	struct tasklet_struct tlet;
	bool dma_in_flight;
	bool cmd_exec;
};

enum buffer_states {
	NONE_PENDING = 0x0,
	FIFO_OVERRUN = 0x1,
	PARITY_ERROR = 0x2,
	CHARS_NORMAL = 0x4,
};

/* optional low power wakeup, typically on a GPIO RX irq */
struct uart_smd_wakeup {
	int irq;  /* < 0 indicates low power wakeup disabled */
	unsigned char ignore;  /* bool */

	/* bool: inject char into rx tty on wakeup */
	unsigned char inject_rx;
	char rx_to_inject;
};

/*
 * UART can be used in 2-wire or 4-wire mode.
 * Use uart_func_mode to set 2-wire or 4-wire mode.
 */
enum uart_func_mode {
	UART_TWO_WIRE, /* can't support HW Flow control. */
	UART_FOUR_WIRE,/* can support HW Flow control. */
};

struct uart_smd_uconfig {
	int baudrate;
	int parity;
	int bit_per_char;
	int stop_bits;
	bool hw_fc_enable;
};

struct uart_smd_port {
	struct uart_port uport;
	unsigned int imr_reg;  /* shadow value of UARTDM_IMR */
	struct clk *clk;
	struct clk *pclk;
	struct smd_port_tx tx;
	struct smd_port_rx rx;
	/* gsbi uarts have to do additional writes to gsbi memory */
	/* block and top control status block. The following pointers */
	/* keep a handle to these blocks. */
	unsigned char __iomem	*mapped_gsbi;
	int dma_tx_channel;
	int dma_rx_channel;
	int dma_tx_crci;
	int dma_rx_crci;
	struct hrtimer clk_off_timer;  /* to poll TXEMT before clock off */
	ktime_t clk_off_delay;
	enum msm_hs_clk_states_e clk_state;
	enum msm_hs_clk_req_off_state_e clk_req_off_state;

	struct uart_smd_wakeup wakeup;
	struct wake_lock dma_wake_lock;  /* held while any DMA active */

	struct dentry *loopback_dir;
	struct work_struct clock_off_w; /* work for actual clock off */
	struct workqueue_struct *uart_smd_wq; /* workqueue */
	struct mutex clk_mutex; /* mutex to guard against clock off/clock on */
	bool tty_flush_receive;
	bool rx_discard_flush_issued;
	enum uart_func_mode func_mode;
	bool is_shutdown;
	bool termios_in_progress;
	int rx_buf_size;
	smd_channel_t *uart_smd_ch;
	struct delayed_work smd_delayed_work;
	struct uart_smd_uconfig uconfig;
	char   *tx_buf;
};

#define UART_SMD_BURST_SIZE   16   /* DM burst size (in bytes) */
#define UART_SMD_TX_BUF_SIZE  UART_XMIT_SIZE
#define UART_SMD_RX_BUF_SIZE  512

#define UART_SMD_RETRY_TIMEOUT 			   5
#define UART_SMD_RX_FLUSH_COMPLETE_TIMEOUT 300 /* In jiffies */
#define UART_SMD_NR 1

#define UART_SMD_CH_NAME	"DATA3"

static struct uart_smd_port smd_port_instance[UART_SMD_NR];

static void uart_smd_start_rx_locked(struct uart_port *uport);
static void uart_smd_set_uconfig(struct uart_port *uport, struct uart_smd_uconfig *uconfig);

#define UPORT_TO_SMD(uart_port) \
	container_of((uart_port), struct uart_smd_port, uport)

static inline int is_valid_baud(int bps)
{
	int ret;
	
	switch (bps)
	{
		case 300:
		case 600:
		case 1200:
		case 2400:
		case 4800:
		case 9600:
		case 14400:
		case 19200:
		case 28800:
		case 38400:
		case 57600:
		case 76800:
		case 115200:
		case 230400:
		case 460800:
		case 921600:
		case 3200000:
		case 3686400:
		case 4000000:
			ret = 1;
			break;
		default:
			ret = 0;
			break;
	}

	return ret;
}

static ssize_t baudrate_show
(
	struct device *dev,
	struct device_attribute *attr,
	char *buf
)
{
	struct platform_device *pdev = container_of(dev, struct platform_device, dev);
	struct uart_smd_port *smd_port = &smd_port_instance[pdev->id];
	
	return snprintf(buf, PAGE_SIZE, "%d\n", smd_port->uconfig.baudrate);
}

static ssize_t baudrate_store
(
	struct device *dev,
	struct device_attribute *attr,
	const char *buff, size_t size
)
{
	int baudrate;
	struct platform_device *pdev = container_of(dev, struct platform_device, dev);
	struct uart_smd_port *smd_port = &smd_port_instance[pdev->id];

	sscanf(buff, "%d", &baudrate);
	if(is_valid_baud(baudrate) && smd_port->uconfig.baudrate != baudrate)
	{
		smd_port->uconfig.baudrate = baudrate;
		uart_smd_set_uconfig(&smd_port->uport, &smd_port->uconfig);
	}
	else
	{
		pr_warn("%s: baud = %d, invalid baudrate or already.\n", __func__, baudrate);
	}
		
	return size;
}

static ssize_t hw_flowctrl_show
(
	struct device *dev,
	struct device_attribute *attr,
	char *buf
)
{
	struct platform_device *pdev = container_of(dev, struct platform_device, dev);
	struct uart_smd_port *smd_port = &smd_port_instance[pdev->id];
	
	return snprintf(buf, PAGE_SIZE, "%d\n", smd_port->uconfig.hw_fc_enable);
}

static ssize_t hw_flowctrl_store
(
	struct device *dev,
	struct device_attribute *attr,
	const char *buff, size_t size
)
{
	int hw_fc_enable;
	struct platform_device *pdev = container_of(dev, struct platform_device, dev);
	struct uart_smd_port *smd_port = &smd_port_instance[pdev->id];

	sscanf(buff, "%d", &hw_fc_enable);
	if((hw_fc_enable == 0 || hw_fc_enable == 1) 
		&& smd_port->uconfig.hw_fc_enable != hw_fc_enable)
	{
		smd_port->uconfig.hw_fc_enable = hw_fc_enable;
		uart_smd_set_uconfig(&smd_port->uport, &smd_port->uconfig);
	}
	else
	{
		pr_warn("%s: hw_fc = %d, invalid hw_fc or already.\n", __func__, hw_fc_enable);
	}
		
	return size;
}

static DEVICE_ATTR(baudrate, S_IRUGO | S_IWUSR, baudrate_show, baudrate_store);
static DEVICE_ATTR(hw_flowctrl, S_IRUGO | S_IWUSR, hw_flowctrl_show, hw_flowctrl_store);

static struct device_attribute *uart_smd_attributes[] = {
	&dev_attr_baudrate,
	&dev_attr_hw_flowctrl,
	NULL
};

static inline unsigned int uart_read_reg(struct uart_port *uport,
				       unsigned int offset)
{
	return readl_relaxed(uport->membase + offset);
}

static inline void uart_write_reg(struct uart_port *uport, unsigned int offset,
				 unsigned int value)
{
	writel_relaxed(value, uport->membase + offset);
}

static inline int write_data_to_modem(struct uart_smd_port *smd_port,
											const unsigned char *buf, size_t size)
{
	int avail, w_len;
	
	avail = smd_write_avail(smd_port->uart_smd_ch);

	w_len = avail > size ? size : avail;

	if(w_len)
	{
		if(smd_write(smd_port->uart_smd_ch, buf, w_len) < 0)
		{
			w_len = 0;
		}
	}
	
	return w_len;
}

static int uart_smd_clk_en(struct uart_smd_port *smd_port, int enable)
{
    int ret = 0;

    if (enable)
    {
        ret = clk_prepare_enable(smd_port->clk);
        if (ret)
        {
            goto err;
        }
        if (smd_port->pclk)
        {
            ret = clk_prepare_enable(smd_port->pclk);
            if (ret)
            {
                clk_disable_unprepare(smd_port->clk);
                goto err;
            }
        }
    }
    else
    {
        clk_disable_unprepare(smd_port->clk);
        if (smd_port->pclk)
        {
            clk_disable_unprepare(smd_port->pclk);
        }
    }
err:
    return ret;
}

/*
 * This API read and provides UART Core registers information.
 */
static void print_uart_registers(struct uart_smd_port *smd_port)
{
	struct uart_port *uport = &(smd_port->uport);

	pr_info("============= UART Registers ================\n");

	pr_info("UART_DM_MR1:%x\n", uart_read_reg(uport, UARTDM_MR1_ADDR));
	pr_info("UART_DM_MR2:%x\n", uart_read_reg(uport, UARTDM_MR2_ADDR));
	pr_info("UART_DM_IPR:%x\n", uart_read_reg(uport, UARTDM_IPR_ADDR));
	pr_info("UART_DM_TFWR:%x\n", uart_read_reg(uport, UARTDM_TFWR_ADDR));
	pr_info("UART_DM_RFWR:%x\n", uart_read_reg(uport, UARTDM_RFWR_ADDR));
	pr_info("UART_DM_DMEN:%x\n", uart_read_reg(uport, UARTDM_DMEN_ADDR));
	pr_info("UART_DM_SR:%x\n", uart_read_reg(uport, UARTDM_SR_ADDR));
	pr_info("UART_DM_ISR:%x\n", uart_read_reg(uport, UARTDM_ISR_ADDR));
	pr_info("UART_DM_TXFS:%x\n", uart_read_reg(uport, UARTDM_TXFS_ADDR));
	pr_info("UART_DM_RXFS:%x\n", uart_read_reg(uport, UARTDM_RXFS_ADDR));
	pr_info("UART_DM_IMR:%x\n", smd_port->imr_reg);

	pr_info("=============================================\n");
}

static int uart_smd_init_clk(struct uart_port *uport)
{
	int ret;
	struct uart_smd_port *smd_port = UPORT_TO_SMD(uport);

	/* Set up the MREG/NREG/DREG/MNDREG */
	ret = clk_set_rate(smd_port->clk, uport->uartclk);
	if (ret)
	{
		printk(KERN_WARNING "Error setting clock rate on UART\n");
		return ret;
	}

	ret = clk_prepare_enable(smd_port->clk);
	if (ret)
	{
		printk(KERN_ERR "Error could not turn on UART clk\n");
		return ret;
	}
	if (smd_port->pclk)
	{
		ret = clk_prepare_enable(smd_port->pclk);
		if (ret)
		{
			clk_disable_unprepare(smd_port->clk);
			dev_err(uport->dev,
				"Error could not turn on UART pclk\n");
			return ret;
		}
	}

	smd_port->clk_state = MSM_HS_CLK_ON;
	return 0;
}

/*
 * programs the UARTDM_CSR register with correct bit rates
 *
 * Interrupts should be disabled before we are called, as
 * we modify Set Baud rate
 * Set receive stale interrupt level, dependant on Bit Rate
 * Goal is to have around 8 ms before indicate stale.
 * roundup (((Bit Rate * .008) / 10) + 1
 */
static unsigned long uart_smd_set_bps_locked(struct uart_port *uport,
			       unsigned int bps,
				unsigned long flags)
{
	unsigned long rxstale;
	unsigned long data;
	unsigned int curr_uartclk;
	struct uart_smd_port *smd_port = UPORT_TO_SMD(uport);

	switch (bps)
	{
	case 300:
		uart_write_reg(uport, UARTDM_CSR_ADDR, 0x00);
		rxstale = 1;
		break;
	case 600:
		uart_write_reg(uport, UARTDM_CSR_ADDR, 0x11);
		rxstale = 1;
		break;
	case 1200:
		uart_write_reg(uport, UARTDM_CSR_ADDR, 0x22);
		rxstale = 1;
		break;
	case 2400:
		uart_write_reg(uport, UARTDM_CSR_ADDR, 0x33);
		rxstale = 1;
		break;
	case 4800:
		uart_write_reg(uport, UARTDM_CSR_ADDR, 0x44);
		rxstale = 1;
		break;
	case 9600:
		uart_write_reg(uport, UARTDM_CSR_ADDR, 0x55);
		rxstale = 2;
		break;
	case 14400:
		uart_write_reg(uport, UARTDM_CSR_ADDR, 0x66);
		rxstale = 3;
		break;
	case 19200:
		uart_write_reg(uport, UARTDM_CSR_ADDR, 0x77);
		rxstale = 4;
		break;
	case 28800:
		uart_write_reg(uport, UARTDM_CSR_ADDR, 0x88);
		rxstale = 6;
		break;
	case 38400:
		uart_write_reg(uport, UARTDM_CSR_ADDR, 0x99);
		rxstale = 8;
		break;
	case 57600:
		uart_write_reg(uport, UARTDM_CSR_ADDR, 0xaa);
		rxstale = 16;
		break;
	case 76800:
		uart_write_reg(uport, UARTDM_CSR_ADDR, 0xbb);
		rxstale = 16;
		break;
	case 115200:
		uart_write_reg(uport, UARTDM_CSR_ADDR, 0xcc);
		rxstale = 31;
		break;
	case 230400:
		uart_write_reg(uport, UARTDM_CSR_ADDR, 0xee);
		rxstale = 31;
		break;
	case 460800:
		uart_write_reg(uport, UARTDM_CSR_ADDR, 0xff);
		rxstale = 31;
		break;
	case 4000000:
	case 3686400:
	case 3200000:
	case 3500000:
	case 3000000:
	case 2500000:
	case 1500000:
	case 1152000:
	case 1000000:
	case 921600:
		uart_write_reg(uport, UARTDM_CSR_ADDR, 0xff);
		rxstale = 31;
		break;
	default:
		uart_write_reg(uport, UARTDM_CSR_ADDR, 0xff);
		/* default to 9600 */
		bps = 9600;
		rxstale = 2;
		break;
	}
	/*
	 * uart baud rate depends on CSR and MND Values
	 * we are updating CSR before and then calling
	 * clk_set_rate which updates MND Values. Hence
	 * dsb requires here.
	 */
	mb();
	curr_uartclk = uport->uartclk;
	if (bps > 460800)
	{
		uport->uartclk = bps * 16;
	}
	else
	{
		uport->uartclk = 7372800;
	}

	spin_unlock_irqrestore(&uport->lock, flags);

	if (curr_uartclk != uport->uartclk)
	{
		if (clk_set_rate(smd_port->clk, uport->uartclk))
		{
			pr_err("%s(): Error setting clock rate on UART\n",
								__func__);
			WARN_ON(1);
			spin_lock_irqsave(&uport->lock, flags);
			return flags;
		}
	}

	spin_lock_irqsave(&uport->lock, flags);
	data = rxstale & UARTDM_IPR_STALE_LSB_BMSK;
	data |= UARTDM_IPR_STALE_TIMEOUT_MSB_BMSK & (rxstale << 2);

	uart_write_reg(uport, UARTDM_IPR_ADDR, data);
	return flags;
}

/*
 * termios :  new ktermios
 * oldtermios:  old ktermios previous setting
 *
 * Configure the serial port
 */
static void uart_smd_set_uconfig(struct uart_port *uport,
				   struct uart_smd_uconfig *uconfig)
{
	int ret;
	unsigned int bits;
	unsigned long data;
	unsigned long flags;
	bool error_case = false;
	struct uart_smd_port *smd_port = UPORT_TO_SMD(uport);

	mutex_lock(&smd_port->clk_mutex);
	spin_lock_irqsave(&uport->lock, flags);

	smd_port->termios_in_progress = true;

	/* Disable RxStale Event Mechanism */
	uart_write_reg(uport, UARTDM_CR_ADDR, STALE_EVENT_DISABLE);

	/* Disable all UART interrupts */
	uart_write_reg(uport, UARTDM_IMR_ADDR, 0);

	/* Enable RFR so remote UART doesn't send any data. */
	uart_write_reg(uport, UARTDM_CR_ADDR, RFR_HIGH);

	/*
	 * It is quite possible that previous graceful flush is not
	 * completed and set_termios() request has been received.
	 * Hence wait here to make sure that it is completed and
	 * queued one more UART RX CMD with ADM.
	 */
	if (smd_port->rx.dma_in_flight &&
			smd_port->rx.flush == FLUSH_DATA_READY)
	{
		spin_unlock_irqrestore(&uport->lock, flags);
		ret = wait_event_timeout(smd_port->rx.wait,
			smd_port->rx.flush == FLUSH_NONE,
			UART_SMD_RX_FLUSH_COMPLETE_TIMEOUT);
		if (!ret)
		{
			pr_err("%s(): timeout for Rx cmd completion\n",
							__func__);
			spin_lock_irqsave(&uport->lock, flags);
			print_uart_registers(smd_port);
			spin_unlock_irqrestore(&uport->lock, flags);
			BUG_ON(1);
		}

		spin_lock_irqsave(&uport->lock, flags);
	}

	/*
	 * Wait for queued Rx CMD to ADM driver to be programmed
	 * with ADM hardware before going and changing UART baud rate.
	 * Below udelay(500) is required as exec_cmd callback is called
	 * before actually programming ADM hardware with cmd.
	 */
	if (smd_port->rx.dma_in_flight)
	{
		spin_unlock_irqrestore(&uport->lock, flags);
		ret = wait_event_timeout(smd_port->rx.wait,
			smd_port->rx.cmd_exec == true,
			UART_SMD_RX_FLUSH_COMPLETE_TIMEOUT);
		if (!ret)
		{
			pr_err("%s(): timeout for rx cmd to be program\n", __func__);
		}
		else
		{
			udelay(500);
		}

		spin_lock_irqsave(&uport->lock, flags);
	}

	uport->uartclk = clk_get_rate(smd_port->clk);
	if (!uport->uartclk)
	{
		BUG_ON(1);
	}
	flags = uart_smd_set_bps_locked(uport, uconfig->baudrate, flags);

	data = uart_read_reg(uport, UARTDM_MR2_ADDR);
	/* set parity */
	data &= ~UARTDM_MR2_PARITY_MODE_BMSK;
	data |= uconfig->parity;

	/* Set bits per char */
	data &= ~UARTDM_MR2_BITS_PER_CHAR_BMSK;
	switch (uconfig->bit_per_char)
	{
		case 5:
			data |= FIVE_BPC;
			break;
		case 6:
			data |= SIX_BPC;
			break;
		case 7:
			data |= SEVEN_BPC;
			break;
		default:
			data |= EIGHT_BPC;
			break;
	}
	
	/* set stop bits */
	if (uconfig->stop_bits == 2)
	{
		data |= STOP_BIT_TWO;
	}
	else
	{
		/* otherwise 1 stop bit */
		data |= STOP_BIT_ONE;
	}
	data |= UARTDM_MR2_ERROR_MODE_BMSK;
	/* write parity/bits per char/stop bit configuration */
	uart_write_reg(uport, UARTDM_MR2_ADDR, data);

	//uport->ignore_status_mask = termios->c_iflag & INPCK;
	//uport->ignore_status_mask |= termios->c_iflag & IGNPAR;
	//uport->ignore_status_mask |= termios->c_iflag & IGNBRK;

	uport->read_status_mask = CREAD;

	/* Wait for baud rate and UART protocol parameters to set. */
	udelay(200);

	/* Set Transmit software time out */
	//uart_update_timeout(uport, c_cflag, bps);
	bits = uconfig->bit_per_char + 2;
	if(uconfig->stop_bits == 2)
	{
		bits++;
	}
	if(uconfig->parity != 0)
	{
		bits++;
	}

	/*
	 * The total number of bits to be transmitted in the fifo.
	 */
	bits = bits * uport->fifosize;

	/*
	 * Figure the timeout to send the above number of bits.
	 * Add .02 seconds of slop
	 */
	uport->timeout = (HZ * bits) / uconfig->baudrate + HZ/50;

	/*
	 * Configure HW flow control
	 * UART Core would see status of CTS line when it is sending data
	 * to remote uart to confirm that it can receive or not.
	 * UART Core would trigger RFR if it is not having any space with
	 * RX FIFO.
	 */
	data = uart_read_reg(uport, UARTDM_MR1_ADDR);
	data &= ~(UARTDM_MR1_CTS_CTL_BMSK | UARTDM_MR1_RX_RDY_CTL_BMSK);
	if (uconfig->hw_fc_enable)
	{
		data |= UARTDM_MR1_CTS_CTL_BMSK;
		data |= UARTDM_MR1_RX_RDY_CTL_BMSK;
	}

	uart_write_reg(uport, UARTDM_MR1_ADDR, data);

	/* Enable previously enabled all UART interrupts. */
	uart_write_reg(uport, UARTDM_IMR_ADDR, smd_port->imr_reg);

	/*
	 * Invoke Force RxStale Interrupt
	 * On receiving this interrupt, send discard flush request
	 * to ADM driver and ignore all received data.
	 */
	wake_lock(&smd_port->rx.wake_lock);
	uart_write_reg(uport, UARTDM_CR_ADDR, FORCE_STALE_EVENT);
	mb();

	smd_port->rx_discard_flush_issued = true;

	/*
	 * Wait for above discard flush request for UART RX CMD to be
	 * completed. completion would be signal from rx_tlet without
	 * queueing any next UART RX CMD.
	 */
	if (smd_port->rx.dma_in_flight)
	{
		spin_unlock_irqrestore(&uport->lock, flags);
		ret = wait_event_timeout(smd_port->rx.wait,
			smd_port->rx_discard_flush_issued == false,
			msecs_to_jiffies(300));
		if (!ret)
		{
			pr_err("%s(): timeout for rx discard flush\n",
							__func__);
			spin_lock_irqsave(&uport->lock, flags);
			print_uart_registers(smd_port);
			error_case = true;
		}
		if (!error_case)
		{
			spin_lock_irqsave(&uport->lock, flags);
		}
	}
	else
	{
		spin_unlock_irqrestore(&uport->lock, flags);
		pr_err("%s(): called with rx.dma_in_flight:%d\n",
				__func__, smd_port->rx.dma_in_flight);
		print_uart_registers(smd_port);
		spin_lock_irqsave(&uport->lock, flags);
	}

	/* Start Rx Transfer */
	uart_smd_start_rx_locked(&smd_port->uport);

	/* Disable RFR so remote UART can send data. */
	uart_write_reg(uport, UARTDM_CR_ADDR, RFR_LOW);
	mb();

	smd_port->termios_in_progress = false;

	spin_unlock_irqrestore(&uport->lock, flags);
	mutex_unlock(&smd_port->clk_mutex);
}

/*
 *  Standard API, Transmitter
 *  Any character in the transmit shift register is sent
 */
unsigned int uart_smd_tx_empty(struct uart_port *uport)
{
	unsigned int data;
	unsigned int ret = 0;

	data = uart_read_reg(uport, UARTDM_SR_ADDR);
	if (data & UARTDM_SR_TXEMT_BMSK)
		ret = TIOCSER_TEMT;

	return ret;
}

/*
 *  Standard API, Stop transmitter.
 *  Any character in the transmit shift register is sent as
 *  well as the current data mover transfer .
 */
static void uart_smd_stop_tx_locked(struct uart_port *uport)
{
	struct uart_smd_port *smd_port = UPORT_TO_SMD(uport);

	smd_port->tx.tx_ready_int_en = 0;
}

/*
 *  Standard API, Stop receiver as soon as possible.
 *
 *  Function immediately terminates the operation of the
 *  channel receiver and any incoming characters are lost. None
 *  of the receiver status bits are affected by this command and
 *  characters that are already in the receive FIFO there.
 */
static void uart_smd_stop_rx_locked(struct uart_port *uport)
{
	struct uart_smd_port *smd_port = UPORT_TO_SMD(uport);

	/* Disable RxStale Event Mechanism */
	uart_write_reg(uport, UARTDM_CR_ADDR, STALE_EVENT_DISABLE);

	/* Enable RFR so remote UART doesn't send any data. */
	uart_write_reg(uport, UARTDM_CR_ADDR, RFR_HIGH);

	/* Allow to receive all pending data from UART RX FIFO */
	udelay(100);

	if (smd_port->rx.flush == FLUSH_NONE)
	{
		wake_lock(&smd_port->rx.wake_lock);

		smd_port->rx_discard_flush_issued = true;

		/*
		 * Invoke Force RxStale Interrupt
		 * On receiving this interrupt, send discard flush request
		 * to ADM driver and ignore all received data.
		 */
		uart_write_reg(uport, UARTDM_CR_ADDR, FORCE_STALE_EVENT);
		mb();
	}

	if (smd_port->rx.flush != FLUSH_SHUTDOWN)
		smd_port->rx.flush = FLUSH_STOP;
}

/*  Transmit the next chunk of data */
static void uart_smd_submit_tx_locked(struct uart_port *uport)
{
	int tx_count;
	int aligned_tx_count;
	dma_addr_t src_addr;
	dma_addr_t aligned_src_addr;
	struct uart_smd_port *smd_port = UPORT_TO_SMD(uport);
	struct smd_port_tx *tx = &smd_port->tx;

	if (tx->dma_in_flight || smd_port->is_shutdown)
		return;

	if((tx_count = smd_read_avail(smd_port->uart_smd_ch)) == 0)
	{
		uart_smd_stop_tx_locked(uport);
		return;
	}

	tx->dma_in_flight = 1;

	if (UART_SMD_TX_BUF_SIZE < tx_count)
	{
		tx_count = UART_SMD_TX_BUF_SIZE;
	}

	smd_read(smd_port->uart_smd_ch, smd_port->tx_buf, tx_count);

	src_addr = tx->dma_base;
	/* Mask the src_addr to align on a cache
	 * and add those bytes to tx_count */
	aligned_src_addr = src_addr & ~(dma_get_cache_alignment() - 1);
	aligned_tx_count = tx_count + src_addr - aligned_src_addr;

	dma_sync_single_for_device(uport->dev, aligned_src_addr,
			aligned_tx_count, DMA_TO_DEVICE);

	tx->command_ptr->num_rows = (((tx_count + 15) >> 4) << 16) |
				     ((tx_count + 15) >> 4);
	tx->command_ptr->src_row_addr = src_addr;

	dma_sync_single_for_device(uport->dev, tx->mapped_cmd_ptr,
				   sizeof(dmov_box), DMA_TO_DEVICE);

	*tx->command_ptr_ptr = CMD_PTR_LP | DMOV_CMD_ADDR(tx->mapped_cmd_ptr);

	/* Save tx_count to use in Callback */
	tx->tx_count = tx_count;
	uart_write_reg(uport, UARTDM_NCF_TX_ADDR, tx_count);

	/* Disable the tx_ready interrupt */
	smd_port->imr_reg &= ~UARTDM_ISR_TX_READY_BMSK;
	uart_write_reg(uport, UARTDM_IMR_ADDR, smd_port->imr_reg);
	/* Calling next DMOV API. Hence mb() here. */
	mb();

	dma_sync_single_for_device(uport->dev, tx->mapped_cmd_ptr_ptr,
				   sizeof(u32), DMA_TO_DEVICE);
	smd_port->tx.flush = FLUSH_NONE;
	msm_dmov_enqueue_cmd(smd_port->dma_tx_channel, &tx->xfer);
}

/* Start to receive the next chunk of data */
static void uart_smd_start_rx_locked(struct uart_port *uport)
{
	struct uart_smd_port *smd_port = UPORT_TO_SMD(uport);
	unsigned int buffer_pending = smd_port->rx.buffer_pending;
	unsigned int data;

	if (smd_port->rx.dma_in_flight)
	{
		pr_err("%s(): RX CMD is already queued.\n", __func__);
		print_uart_registers(smd_port);
		BUG_ON(1);
	}

	smd_port->rx.buffer_pending = 0;
	if (buffer_pending)
	{
		printk(KERN_ERR "Error: rx started in buffer state = %x",
		       buffer_pending);
	}
	
	/*
	 * Zeroed out UART RX software buffer which would help to
	 * check how much data is copied if there is any RX stall.
	 */
	memset(smd_port->rx.buffer, 0x00, smd_port->rx_buf_size);
	uart_write_reg(uport, UARTDM_CR_ADDR, RESET_STALE_INT);
	uart_write_reg(uport, UARTDM_DMRX_ADDR, smd_port->rx_buf_size);
	uart_write_reg(uport, UARTDM_CR_ADDR, STALE_EVENT_ENABLE);
	smd_port->imr_reg |= UARTDM_ISR_RXLEV_BMSK;

	/*
	 * Enable UARTDM Rx Interface as previously it has been
	 * disable in set_termios before configuring baud rate.
	 */
	data = uart_read_reg(uport, UARTDM_DMEN_ADDR);
	data |= UARTDM_RX_DM_EN_BMSK;
	uart_write_reg(uport, UARTDM_DMEN_ADDR, data);
	uart_write_reg(uport, UARTDM_IMR_ADDR, smd_port->imr_reg);
	/* Calling next DMOV API. Hence mb() here. */
	mb();

	smd_port->rx.flush = FLUSH_NONE;
	smd_port->rx.dma_in_flight = true;
	smd_port->rx.cmd_exec = false;
	msm_dmov_enqueue_cmd_ext(smd_port->dma_rx_channel,
					&smd_port->rx.xfer);
	/*
	 * wake up set_termios() process if it is waiting for
	 * completion of previous UART RX CMD and queueing of
	 * new UART RX CMD before going ahead with baud rate
	 * change process.
	 */
	if (smd_port->termios_in_progress)
		wake_up(&smd_port->rx.wait);
}

static void write_smd_work(struct work_struct *work)
{
	unsigned long flags;
	int retval;
	struct uart_smd_port *smd_port =
		container_of(work, struct uart_smd_port,
			     rx.write_smd_work.work);

	spin_lock_irqsave(&smd_port->uport.lock, flags);
	if (smd_port->rx.buffer_pending == NONE_PENDING)
	{
		printk(KERN_ERR "Error: No buffer pending in %s", __func__);
		return;
	}
#if 0
	if (smd_port->rx.buffer_pending & FIFO_OVERRUN)
	{
		retval = tty_insert_flip_char(tty, 0, TTY_OVERRUN);
		if (retval)
			smd_port->rx.buffer_pending &= ~FIFO_OVERRUN;
	}
	if (smd_port->rx.buffer_pending & PARITY_ERROR)
	{
		retval = tty_insert_flip_char(tty, 0, TTY_PARITY);
		if (retval)
			smd_port->rx.buffer_pending &= ~PARITY_ERROR;
	}
#endif
	if (smd_port->rx.buffer_pending & CHARS_NORMAL)
	{
		int rx_count, rx_offset;
		rx_count = (smd_port->rx.buffer_pending & 0xFFFF0000) >> 16;
		rx_offset = (smd_port->rx.buffer_pending & 0xFFD0) >> 5;
		retval = write_data_to_modem(smd_port, smd_port->rx.buffer + rx_offset, rx_count);
		smd_port->rx.buffer_pending &= (FIFO_OVERRUN | PARITY_ERROR);
		if (retval != rx_count)
		{
			smd_port->rx.buffer_pending |= CHARS_NORMAL |
				retval << 8 | (rx_count - retval) << 16;
		}
	}
	if (smd_port->rx.buffer_pending)
	{
		schedule_delayed_work(&smd_port->rx.write_smd_work,
				      msecs_to_jiffies(UART_SMD_RETRY_TIMEOUT));
	}
	else if ((smd_port->clk_state == MSM_HS_CLK_ON) &&
		    (smd_port->rx.flush <= FLUSH_IGNORE))
	{
		printk(KERN_WARNING
			   "uart_smd: "
			   "Pending buffers cleared. "
			   "Restarting\n");
		uart_smd_start_rx_locked(&smd_port->uport);
	}
	spin_unlock_irqrestore(&smd_port->uport.lock, flags);
}

static void uart_smd_rx_tlet(unsigned long tlet_ptr)
{
	int retval;
	int rx_count;
	unsigned long status;
	unsigned long flags;
	unsigned int error_f = 0;
	struct uart_port *uport;
	struct uart_smd_port *smd_port;
	unsigned int flush;

	smd_port = container_of((struct tasklet_struct *)tlet_ptr,
				 struct uart_smd_port, rx.tlet);
	uport = &smd_port->uport;

	status = uart_read_reg(uport, UARTDM_SR_ADDR);

	spin_lock_irqsave(&uport->lock, flags);

	uart_write_reg(uport, UARTDM_CR_ADDR, STALE_EVENT_DISABLE);

	/* Rx DMA cmd is completed here. */
	smd_port->rx.dma_in_flight = false;

	/* overflow is not connect to data in a FIFO */
	if (unlikely((status & UARTDM_SR_OVERRUN_BMSK) &&
		     (uport->read_status_mask & CREAD)))
	{
		printk(KERN_WARNING "uart_smd: overrun error\n");
		uport->icount.buf_overrun++;
		error_f = 1;
	}

	if (!(uport->ignore_status_mask & INPCK))
		status = status & ~(UARTDM_SR_PAR_FRAME_BMSK);

	if (unlikely(status & UARTDM_SR_PAR_FRAME_BMSK))
	{
		/* Can not tell difference between parity & frame error */
		printk(KERN_WARNING "uart_smd: parity error\n");
		uport->icount.parity++;
		error_f = 1;
	}

	if (unlikely(status & UARTDM_SR_RX_BREAK_BMSK))
	{
		printk(KERN_WARNING "uart_smd: Rx break\n");
		uport->icount.brk++;
		error_f = 1;
		uart_write_reg(uport, UARTDM_CR_ADDR, RESET_BREAK_INT);
	}

	if (error_f)
	{
		uart_write_reg(uport, UARTDM_CR_ADDR, RESET_ERROR_STATUS);
	}

	if (smd_port->clk_req_off_state == CLK_REQ_OFF_FLUSH_ISSUED)
	{
		smd_port->clk_req_off_state = CLK_REQ_OFF_RXSTALE_FLUSHED;
	}
	flush = smd_port->rx.flush;

	/*
	 * Part hs_isr() sets the flush as FLUSH_IGNORE if it is explictily
	 * trigger from set_termios().
	 */
	if (flush == FLUSH_IGNORE)
	{
		if (!smd_port->rx_discard_flush_issued &&
					!smd_port->rx.buffer_pending)
		{
			uart_smd_start_rx_locked(uport);
		}
		else
		{
			smd_port->rx_discard_flush_issued = false;
			wake_up(&smd_port->rx.wait);
			goto out;
		}
	}

	/* Part of stop_rx sets the flush as FLUSH_STOP */
	if (flush == FLUSH_STOP)
	{
		if (smd_port->rx_discard_flush_issued)
			smd_port->rx_discard_flush_issued = false;
		smd_port->rx.flush = FLUSH_SHUTDOWN;
		wake_up(&smd_port->rx.wait);
	}

	if (flush >= FLUSH_DATA_INVALID)
		goto out;

	rx_count = uart_read_reg(uport, UARTDM_RX_TOTAL_SNAP_ADDR);

	/* order the read of rx.buffer */
	rmb();

	if (0 != (uport->read_status_mask & CREAD))
	{
		retval = write_data_to_modem(smd_port, smd_port->rx.buffer, rx_count);
		if (retval != rx_count)
		{
			smd_port->rx.buffer_pending |= CHARS_NORMAL |
				retval << 5 | (rx_count - retval) << 16;
		}
	}

	/* order the read of rx.buffer and the start of next rx xfer */
	wmb();

	if (!smd_port->rx.buffer_pending)
		uart_smd_start_rx_locked(uport);

out:
	if (smd_port->rx.buffer_pending)
	{
		printk(KERN_WARNING
		       "uart_smd: "
		       "tty buffer exhausted. "
		       "Stalling\n");
		schedule_delayed_work(&smd_port->rx.write_smd_work
				      , msecs_to_jiffies(UART_SMD_RETRY_TIMEOUT));
	}
	/* release wakelock in 500ms, not immediately, because higher layers
	 * don't always take wakelocks when they should */
	wake_lock_timeout(&smd_port->rx.wake_lock, HZ / 2);
	/* tty_flip_buffer_push() might call msm_hs_start(), so unlock */
	spin_unlock_irqrestore(&uport->lock, flags);
}

/* Enable the transmitter Interrupt */
static void uart_smd_start_tx_locked(struct uart_port *uport )
{
	struct uart_smd_port *smd_port = UPORT_TO_SMD(uport);

	if (smd_port->is_shutdown)
		return;

	if (smd_port->tx.tx_ready_int_en == 0)
	{
		smd_port->tx.tx_ready_int_en = 1;
		if (smd_port->tx.dma_in_flight == 0)
			uart_smd_submit_tx_locked(uport);
	}
}

/*
 *  This routine is called when we are done with a DMA transfer
 *
 *  This routine is registered with Data mover when we set
 *  up a Data Mover transfer. It is called from Data mover ISR
 *  when the DMA transfer is done.
 */
static void uart_smd_dmov_tx_callback(struct msm_dmov_cmd *cmd_ptr,
					unsigned int result,
					struct msm_dmov_errdata *err)
{
	struct uart_smd_port *smd_port;

	smd_port = container_of(cmd_ptr, struct uart_smd_port, tx.xfer);
	if (smd_port->tx.flush == FLUSH_STOP)
		/* DMA FLUSH unsuccesfful */
		WARN_ON(!(result & DMOV_RSLT_FLUSH));
	else
		/* DMA did not finish properly */
		WARN_ON(!(result & DMOV_RSLT_DONE));

	tasklet_schedule(&smd_port->tx.tlet);
}

static void uart_smd_tx_tlet(unsigned long tlet_ptr)
{
	unsigned long flags;
	struct uart_smd_port *smd_port = container_of((struct tasklet_struct *)
				tlet_ptr, struct uart_smd_port, tx.tlet);
	struct smd_port_tx *tx = &smd_port->tx;

	spin_lock_irqsave(&(smd_port->uport.lock), flags);

	tx->dma_in_flight = 0;
	if (smd_port->tx.flush == FLUSH_STOP)
	{
		smd_port->tx.flush = FLUSH_SHUTDOWN;
		wake_up(&smd_port->tx.wait);
		spin_unlock_irqrestore(&(smd_port->uport.lock), flags);
		return;
	}

	smd_port->imr_reg |= UARTDM_ISR_TX_READY_BMSK;
	uart_write_reg(&(smd_port->uport), UARTDM_IMR_ADDR, smd_port->imr_reg);
	/* Calling clk API. Hence mb() requires. */
	mb();

	spin_unlock_irqrestore(&(smd_port->uport.lock), flags);
}

/*
 * This routine is called when ADM driver is about to program queued UART
 * RX CMD with ADM hardware.
 */
static void uart_smd_dmov_rx_exec_callback(struct msm_dmov_cmd *cmd_ptr)
{
	struct uart_smd_port *smd_port;
	struct smd_port_rx *rx;

	smd_port = container_of(cmd_ptr, struct uart_smd_port, rx.xfer);
	rx = &smd_port->rx;

	rx->cmd_exec = true;
	/*
	 * wakeup set_termios() as  it waits for UART RX CMD to be programmed
	 * with ADM hardware.
	 */
	wake_up(&smd_port->rx.wait);
}

/*
 * This routine is called when we are done with a DMA transfer or the
 * a flush has been sent to the data mover driver.
 *
 * This routine is registered with Data mover when we set up a Data Mover
 *  transfer. It is called from Data mover ISR when the DMA transfer is done.
 */
static void uart_smd_dmov_rx_callback(struct msm_dmov_cmd *cmd_ptr,
					unsigned int result,
					struct msm_dmov_errdata *err)
{
	struct uart_smd_port *smd_port;
	
	if (result & DMOV_RSLT_ERROR)
		pr_err("%s(): DMOV_RSLT_ERROR\n", __func__);

	smd_port = container_of(cmd_ptr, struct uart_smd_port, rx.xfer);

	tasklet_schedule(&smd_port->rx.tlet);
}

/*  Handle CTS changes (Called from interrupt handler) */
static void uart_smd_handle_delta_cts_locked(struct uart_port *uport)
{
	/* clear interrupt */
	uart_write_reg(uport, UARTDM_CR_ADDR, RESET_CTS);
	/* Calling CLOCK API. Hence mb() requires here. */
	mb();
	uport->icount.cts++;

	/* clear the IOCTL TIOCMIWAIT if called */
	wake_up_interruptible(&uport->state->port.delta_msr_wait);
}

/* check if the TX path is flushed, and if so clock off
 * returns 0 did not clock off, need to retry (still sending final byte)
 *        -1 did not clock off, do not retry
 *         1 if we clocked off
 */
static int uart_smd_check_clock_off(struct uart_port *uport)
{
	unsigned long sr_status;
	unsigned long flags;
	int ret;
	struct uart_smd_port *smd_port = UPORT_TO_SMD(uport);

	mutex_lock(&smd_port->clk_mutex);
	spin_lock_irqsave(&uport->lock, flags);

	/* Cancel if tx tty buffer is not empty, dma is in flight,
	 * or tx fifo is not empty */
	if (smd_port->clk_state != MSM_HS_CLK_REQUEST_OFF ||
	    smd_read_avail(smd_port->uart_smd_ch) || smd_port->tx.dma_in_flight ||
	    smd_port->imr_reg & UARTDM_ISR_TXLEV_BMSK)
	{
		spin_unlock_irqrestore(&uport->lock, flags);
		mutex_unlock(&smd_port->clk_mutex);
		return -1;
	}

	/* Make sure the uart is finished with the last byte */
	sr_status = uart_read_reg(uport, UARTDM_SR_ADDR);
	if (!(sr_status & UARTDM_SR_TXEMT_BMSK))
	{
		spin_unlock_irqrestore(&uport->lock, flags);
		mutex_unlock(&smd_port->clk_mutex);
		return 0;  /* retry */
	}

	/* Make sure forced RXSTALE flush complete */
	switch (smd_port->clk_req_off_state)
	{
	case CLK_REQ_OFF_START:
		smd_port->clk_req_off_state = CLK_REQ_OFF_RXSTALE_ISSUED;
		wake_lock(&smd_port->rx.wake_lock);
		uart_write_reg(uport, UARTDM_CR_ADDR, FORCE_STALE_EVENT);
		/*
		 * Before returning make sure that device writel completed.
		 * Hence mb() requires here.
		 */
		mb();
		spin_unlock_irqrestore(&uport->lock, flags);
		mutex_unlock(&smd_port->clk_mutex);
		return 0;  /* RXSTALE flush not complete - retry */
	case CLK_REQ_OFF_RXSTALE_ISSUED:
	case CLK_REQ_OFF_FLUSH_ISSUED:
		spin_unlock_irqrestore(&uport->lock, flags);
		mutex_unlock(&smd_port->clk_mutex);
		return 0;  /* RXSTALE flush not complete - retry */
	case CLK_REQ_OFF_RXSTALE_FLUSHED:
		break;  /* continue */
	}

	if (smd_port->rx.flush != FLUSH_SHUTDOWN)
	{
		if (smd_port->rx.flush == FLUSH_NONE)
		{
			uart_smd_stop_rx_locked(uport);
			smd_port->rx_discard_flush_issued = true;
		}

		spin_unlock_irqrestore(&uport->lock, flags);
		if (smd_port->rx_discard_flush_issued)
		{
			pr_debug("%s(): wainting for flush completion.\n",
								__func__);
			ret = wait_event_timeout(smd_port->rx.wait,
				smd_port->rx_discard_flush_issued == false,
				UART_SMD_RX_FLUSH_COMPLETE_TIMEOUT);
			if (!ret)
				pr_err("%s(): Flush complete pending.\n",
								__func__);
		}

		mutex_unlock(&smd_port->clk_mutex);
		return 0;  /* come back later to really clock off */
	}

	spin_unlock_irqrestore(&uport->lock, flags);

	/* we really want to clock off */
	clk_disable_unprepare(smd_port->clk);
	if (smd_port->pclk)
		clk_disable_unprepare(smd_port->pclk);

	smd_port->clk_state = MSM_HS_CLK_OFF;

	spin_lock_irqsave(&uport->lock, flags);
	wake_unlock(&smd_port->dma_wake_lock);

	spin_unlock_irqrestore(&uport->lock, flags);
	mutex_unlock(&smd_port->clk_mutex);
	return 1;
}

static void uart_smd_clock_off_work(struct work_struct *w)
{
	struct uart_smd_port *smd_port = container_of(w, struct uart_smd_port,
							clock_off_w);
	struct uart_port *uport = &smd_port->uport;

	if (!uart_smd_check_clock_off(uport))
	{
		hrtimer_start(&smd_port->clk_off_timer,
				smd_port->clk_off_delay,
				HRTIMER_MODE_REL);
	}
}

static enum hrtimer_restart uart_smd_clk_off_retry(struct hrtimer *timer)
{
	struct uart_smd_port *smd_port = container_of(timer, struct uart_smd_port,
							clk_off_timer);

	queue_work(smd_port->uart_smd_wq, &smd_port->clock_off_w);
	return HRTIMER_NORESTART;
}

static irqreturn_t uart_smd_isr(int irq, void *dev)
{
	unsigned long flags;
	unsigned long isr_status;
	struct uart_smd_port *smd_port = (struct uart_smd_port *)dev;
	struct uart_port *uport = &smd_port->uport;
	struct smd_port_tx *tx = &smd_port->tx;
	struct smd_port_rx *rx = &smd_port->rx;

	spin_lock_irqsave(&uport->lock, flags);

	if (smd_port->is_shutdown)
	{
		pr_err("%s(): Received UART interrupt after shutdown.\n",
								__func__);
		spin_unlock_irqrestore(&uport->lock, flags);
		return IRQ_HANDLED;
	}

	isr_status = uart_read_reg(uport, UARTDM_MISR_ADDR);

	/* Uart RX starting */
	if (isr_status & UARTDM_ISR_RXLEV_BMSK)
	{
		wake_lock(&rx->wake_lock);  /* hold wakelock while rx dma */
		smd_port->imr_reg &= ~UARTDM_ISR_RXLEV_BMSK;
		uart_write_reg(uport, UARTDM_IMR_ADDR, smd_port->imr_reg);
		/* Complete device write for IMR. Hence mb() requires. */
		mb();
	}
	/* Stale rx interrupt */
	if (isr_status & UARTDM_ISR_RXSTALE_BMSK)
	{
		uart_write_reg(uport, UARTDM_CR_ADDR, STALE_EVENT_DISABLE);
		uart_write_reg(uport, UARTDM_CR_ADDR, RESET_STALE_INT);
		/*
		 * Complete device write before calling DMOV API. Hence
		 * mb() requires here.
		 */
		mb();

		if (smd_port->clk_req_off_state == CLK_REQ_OFF_RXSTALE_ISSUED)
			smd_port->clk_req_off_state =
				CLK_REQ_OFF_FLUSH_ISSUED;

		if (rx->flush == FLUSH_NONE)
		{
			if (!smd_port->termios_in_progress)
			{
				rx->flush = FLUSH_DATA_READY;
				/* Graceful Flush */
				msm_dmov_flush(smd_port->dma_rx_channel, 1);
			}
			else
			{
				rx->flush = FLUSH_IGNORE;
				/* Discard Flush */
				msm_dmov_flush(smd_port->dma_rx_channel, 0);
			}
		}

		/*
		 * Force RxStale is performed from uart_smd_stop_rx_locked() and
		 * change rx.flush to FLUSH_STOP.
		 */
		if (rx->flush == FLUSH_STOP)
		{
			if (smd_port->rx_discard_flush_issued)
				/* Discard Flush */
				msm_dmov_flush(smd_port->dma_rx_channel, 0);
		}
	}

	/* tx ready interrupt */
	if (isr_status & UARTDM_ISR_TX_READY_BMSK)
	{
		/* Clear  TX Ready */
		uart_write_reg(uport, UARTDM_CR_ADDR, CLEAR_TX_READY);

		if (smd_port->clk_state == MSM_HS_CLK_REQUEST_OFF)
		{
			smd_port->imr_reg |= UARTDM_ISR_TXLEV_BMSK;
			uart_write_reg(uport, UARTDM_IMR_ADDR,
				     smd_port->imr_reg);
		}
		/*
		 * Complete both writes before starting new TX.
		 * Hence mb() requires here.
		 */
		mb();
		/* Complete DMA TX transactions and submit new transactions */

		uport->icount.tx += tx->tx_count;
		if (tx->tx_ready_int_en)
			uart_smd_submit_tx_locked(uport);
	}
	
	if (isr_status & UARTDM_ISR_TXLEV_BMSK)
	{
		/* TX FIFO is empty */
		smd_port->imr_reg &= ~UARTDM_ISR_TXLEV_BMSK;
		uart_write_reg(uport, UARTDM_IMR_ADDR, smd_port->imr_reg);
		/*
		 * Complete device write before starting clock_off request.
		 * Hence mb() requires here.
		 */
		mb();
		queue_work(smd_port->uart_smd_wq, &smd_port->clock_off_w);
	}

	/* Change in CTS interrupt */
	if (isr_status & UARTDM_ISR_DELTA_CTS_BMSK)
		uart_smd_handle_delta_cts_locked(uport);

	spin_unlock_irqrestore(&uport->lock, flags);

	return IRQ_HANDLED;
}

/* request to turn off uart clock once pending TX is flushed */
void uart_smd_request_clock_off(struct uart_port *uport)
{
	unsigned long flags;
	struct uart_smd_port *smd_port = UPORT_TO_SMD(uport);

	spin_lock_irqsave(&uport->lock, flags);
	if (smd_port->clk_state == MSM_HS_CLK_ON) {
		smd_port->clk_state = MSM_HS_CLK_REQUEST_OFF;
		smd_port->clk_req_off_state = CLK_REQ_OFF_START;
		smd_port->imr_reg |= UARTDM_ISR_TXLEV_BMSK;
		uart_write_reg(uport, UARTDM_IMR_ADDR, smd_port->imr_reg);
		/*
		 * Complete device write before retuning back.
		 * Hence mb() requires here.
		 */
		mb();
	}
	spin_unlock_irqrestore(&uport->lock, flags);
}

void uart_smd_request_clock_on(struct uart_port *uport)
{
	struct uart_smd_port *smd_port = UPORT_TO_SMD(uport);
	unsigned long flags;
	unsigned int data;
	int ret = 0;

	mutex_lock(&smd_port->clk_mutex);
	spin_lock_irqsave(&uport->lock, flags);

	switch (smd_port->clk_state)
	{
	case MSM_HS_CLK_OFF:
		wake_lock(&smd_port->dma_wake_lock);
		disable_irq_nosync(smd_port->wakeup.irq);
		spin_unlock_irqrestore(&uport->lock, flags);
		ret = clk_prepare_enable(smd_port->clk);
		if (ret)
		{
			dev_err(uport->dev, "Clock ON Failure"
			"For UART CLK Stalling HSUART\n");
			wake_unlock(&smd_port->dma_wake_lock);
			break;
		}

		if (smd_port->pclk)
		{
			ret = clk_prepare_enable(smd_port->pclk);
			if (unlikely(ret)) {
				clk_disable_unprepare(smd_port->clk);
				dev_err(uport->dev, "Clock ON Failure"
				"For UART Pclk Stalling HSUART\n");
				wake_unlock(&smd_port->dma_wake_lock);
				break;
			}
		}
		spin_lock_irqsave(&uport->lock, flags);
		/* else fall-through */
	case MSM_HS_CLK_REQUEST_OFF:
		if (smd_port->rx.flush == FLUSH_STOP ||
		    smd_port->rx.flush == FLUSH_SHUTDOWN) {
			uart_write_reg(uport, UARTDM_CR_ADDR, RESET_RX);
			data = uart_read_reg(uport, UARTDM_DMEN_ADDR);
			data |= UARTDM_RX_DM_EN_BMSK;
			uart_write_reg(uport, UARTDM_DMEN_ADDR, data);
			/* Complete above device write. Hence mb() here. */
			mb();
		}
		hrtimer_try_to_cancel(&smd_port->clk_off_timer);
		if (smd_port->rx.flush == FLUSH_SHUTDOWN)
			uart_smd_start_rx_locked(uport);
		if (smd_port->rx.flush == FLUSH_STOP)
			smd_port->rx.flush = FLUSH_IGNORE;
		smd_port->clk_state = MSM_HS_CLK_ON;
		break;
	case MSM_HS_CLK_ON:
		break;
	case MSM_HS_CLK_PORT_OFF:
		break;
	}

	spin_unlock_irqrestore(&uport->lock, flags);
	mutex_unlock(&smd_port->clk_mutex);
}

static int uart_smd_config_port(struct uart_port *uport)
{
	unsigned long flags;
	resource_size_t size;
	struct resource *gsbi_resource;
	struct uart_smd_port *smd_port = UPORT_TO_SMD(uport);
	struct platform_device *pdev = to_platform_device(uport->dev);

	gsbi_resource = platform_get_resource_byname(pdev,
						     IORESOURCE_MEM,
						     "gsbi_resource");

	if(gsbi_resource)
	{
		size = resource_size(gsbi_resource);
		if (unlikely(!request_mem_region(gsbi_resource->start, size,
						 "uart_smd")))
		{
			return -EBUSY;
		}
		smd_port->mapped_gsbi = ioremap(gsbi_resource->start, size);
		if (!smd_port->mapped_gsbi)
		{
			release_mem_region(gsbi_resource->start, size);
			return -EBUSY;
		}
	}
	
	if (smd_port->pclk)
	{
		clk_prepare_enable(smd_port->pclk);
	}
	spin_lock_irqsave(&uport->lock, flags);
	iowrite32(GSBI_PROTOCOL_UART, smd_port->mapped_gsbi +
		  GSBI_CONTROL_ADDR);
	spin_unlock_irqrestore(&uport->lock, flags);
	if (smd_port->pclk)
	{
		clk_disable_unprepare(smd_port->pclk);
	}

	return 0;
}

/* Called when port is opened */
static int uart_smd_startup(struct uart_port *uport)
{
	int ret;
	int rfr_level;
	unsigned long flags;
	unsigned int data;
	struct uart_smd_port *smd_port = UPORT_TO_SMD(uport);
	struct smd_port_tx *tx = &smd_port->tx;
	struct smd_port_rx *rx = &smd_port->rx;

	if(uart_smd_config_port(uport))
	{
		return -EBUSY;
	}
	
	smd_port->is_shutdown = false;
	smd_port->termios_in_progress = false;

	rfr_level = uport->fifosize;
	if (rfr_level > 16)
		rfr_level -= 16;

	tx->dma_base = dma_map_single(uport->dev, smd_port->tx_buf, UART_XMIT_SIZE,
				      DMA_TO_DEVICE);

	wake_lock(&smd_port->dma_wake_lock);
	/* turn on uart clk */
	ret = uart_smd_init_clk(uport);
	if (unlikely(ret))
	{
		pr_err("Turning ON uartclk error\n");
		wake_unlock(&smd_port->dma_wake_lock);
		return ret;
	}

	/* Stop remote UART to send data by setting RFR GPIO to LOW. */
	uart_write_reg(uport, UARTDM_CR_ADDR, RFR_HIGH);

	/* Set auto RFR Level */
	data = uart_read_reg(uport, UARTDM_MR1_ADDR);
	data &= ~UARTDM_MR1_AUTO_RFR_LEVEL1_BMSK;
	data &= ~UARTDM_MR1_AUTO_RFR_LEVEL0_BMSK;
	data |= (UARTDM_MR1_AUTO_RFR_LEVEL1_BMSK & (rfr_level << 2));
	data |= (UARTDM_MR1_AUTO_RFR_LEVEL0_BMSK & rfr_level);
	uart_write_reg(uport, UARTDM_MR1_ADDR, data);

	/* Make sure RXSTALE count is non-zero */
	data = uart_read_reg(uport, UARTDM_IPR_ADDR);
	if (!data)
	{
		data |= 0x1f & UARTDM_IPR_STALE_LSB_BMSK;
		uart_write_reg(uport, UARTDM_IPR_ADDR, data);
	}

	/* Enable Data Mover Mode */
	data = UARTDM_TX_DM_EN_BMSK | UARTDM_RX_DM_EN_BMSK;
	uart_write_reg(uport, UARTDM_DMEN_ADDR, data);


	/* Reset both RX and TX HW state machine */
	uart_write_reg(uport, UARTDM_CR_ADDR, (RESET_RX | RESET_TX));
	/*
	 * Rx and Tx reset operation takes few clock cycles, hence as
	 * safe side adding 10us delay.
	 */
	udelay(10);

	uart_write_reg(uport, UARTDM_CR_ADDR, RESET_ERROR_STATUS);
	uart_write_reg(uport, UARTDM_CR_ADDR, RESET_BREAK_INT);
	uart_write_reg(uport, UARTDM_CR_ADDR, RESET_STALE_INT);
	uart_write_reg(uport, UARTDM_CR_ADDR, RESET_CTS);
	/* Turn on Uart Receiver */
	uart_write_reg(uport, UARTDM_CR_ADDR, UARTDM_CR_RX_EN_BMSK);

	/* Turn on Uart Transmitter */
	uart_write_reg(uport, UARTDM_CR_ADDR, UARTDM_CR_TX_EN_BMSK);

	/* Initialize the tx */
	tx->tx_ready_int_en = 0;
	tx->dma_in_flight = 0;
	rx->dma_in_flight = false;
	rx->cmd_exec = false;

	tx->xfer.complete_func = uart_smd_dmov_tx_callback;

	tx->command_ptr->cmd = CMD_LC |
	    CMD_DST_CRCI(smd_port->dma_tx_crci) | CMD_MODE_BOX;

	tx->command_ptr->src_dst_len = (UART_SMD_BURST_SIZE << 16)
					   | (UART_SMD_BURST_SIZE);

	tx->command_ptr->row_offset = (UART_SMD_BURST_SIZE << 16);

	tx->command_ptr->dst_row_addr =
	    smd_port->uport.mapbase + UARTDM_TF_ADDR;

	smd_port->imr_reg |= UARTDM_ISR_RXSTALE_BMSK;
	/* Enable reading the current CTS, no harm even if CTS is ignored */
	smd_port->imr_reg |= UARTDM_ISR_CURRENT_CTS_BMSK;

	uart_write_reg(uport, UARTDM_TFWR_ADDR, 0);  /* TXLEV on empty TX fifo */
	/*
	 * Complete all device write related configuration before
	 * queuing RX request. Hence mb() requires here.
	 */
	mb();


	ret = request_irq(uport->irq, uart_smd_isr, IRQF_TRIGGER_HIGH,
			  "msm_uart_smd", smd_port);
	if (unlikely(ret)) 
	{
		pr_err("%s():Error getting uart irq\n", __func__);
		goto uart_irq_err;
	}

	spin_lock_irqsave(&uport->lock, flags);

	uart_smd_start_rx_locked(uport);

	/* Allow remote UART to send data by setting RFR GPIO to HIGH. */
	uart_write_reg(uport, UARTDM_CR_ADDR, RFR_LOW);

	spin_unlock_irqrestore(&uport->lock, flags);

	pm_runtime_enable(uport->dev);

	return 0;

uart_irq_err:
	wake_unlock(&smd_port->dma_wake_lock);

	return ret;
}

static void uart_smd_notify(void *priv, unsigned event)
{
	struct uart_smd_port *smd_port = priv;

	switch (event)
	{
	case SMD_EVENT_DATA:
		if(smd_read_avail(smd_port->uart_smd_ch))
		{
			uart_smd_start_tx_locked(&smd_port->uport);
		}
		break;
		
	case SMD_EVENT_OPEN:
		pr_warn("%s: Event Open\n", __func__);
		break;
		
	case SMD_EVENT_CLOSE:
		pr_warn("%s: Event Close\n", __func__);
		break;
		
	case SMD_EVENT_STATUS:
		break;
	}
}

static void uart_smd_init_worker(struct work_struct *work)
{
	int ret;
	
	struct uart_smd_port *smd_port 
		= container_of(container_of(work, struct delayed_work, work), 
						struct uart_smd_port,
					  	smd_delayed_work);

	pr_warn("%s: smd_port=%d.\n", __func__, (int)smd_port);
	
	ret = smd_open(UART_SMD_CH_NAME,
				  &smd_port->uart_smd_ch, 
				   smd_port, 
				   uart_smd_notify);
	
	if (ret)
	{
		if (ret == -EAGAIN)
		{
			/* port not ready  - retry */
			pr_debug("%s: SMD port not ready - rescheduling:%s err:%d\n",
						__func__, UART_SMD_CH_NAME, ret);

			schedule_delayed_work(&smd_port->smd_delayed_work,
								   msecs_to_jiffies(UART_SMD_RETRY_TIMEOUT));
			return;
		}
		else
		{
			pr_err("%s: unable to open smd port:%s err:%d\n",
					__func__, UART_SMD_CH_NAME, ret);
		}
	}

	//init uart.
	uart_smd_startup(&smd_port->uport);
	uart_smd_set_uconfig(&smd_port->uport, &smd_port->uconfig);
}

/* Initialize tx and rx data structures */
static int uart_smd_init_port(struct uart_port *uport)
{
	int ret = 0;
	struct uart_smd_port *smd_port = UPORT_TO_SMD(uport);
	struct smd_port_tx *tx = &smd_port->tx;
	struct smd_port_rx *rx = &smd_port->rx;

	smd_port->tx_buf = kmalloc(UART_XMIT_SIZE, GFP_KERNEL | __GFP_DMA);
	if(!(smd_port->tx_buf))
	{
		pr_err("%s: can not get a buffer.\n", __func__);
	}
	
	/* Allocate the command pointer. Needs to be 64 bit aligned */
	tx->command_ptr = kmalloc(sizeof(dmov_box), GFP_KERNEL | __GFP_DMA);
	if (!tx->command_ptr)
		return -ENOMEM;

	tx->command_ptr_ptr = kmalloc(sizeof(u32), GFP_KERNEL | __GFP_DMA);
	if (!tx->command_ptr_ptr)
	{
		ret = -ENOMEM;
		goto free_tx_command_ptr;
	}

	tx->mapped_cmd_ptr = dma_map_single(uport->dev, tx->command_ptr,
					    sizeof(dmov_box), DMA_TO_DEVICE);
	tx->mapped_cmd_ptr_ptr = dma_map_single(uport->dev,
						tx->command_ptr_ptr,
						sizeof(u32), DMA_TO_DEVICE);
	tx->xfer.cmdptr = DMOV_CMD_ADDR(tx->mapped_cmd_ptr_ptr);

	init_waitqueue_head(&rx->wait);
	init_waitqueue_head(&tx->wait);
	wake_lock_init(&rx->wake_lock, WAKE_LOCK_SUSPEND, "uart_smd_rx");
	wake_lock_init(&smd_port->dma_wake_lock, WAKE_LOCK_SUSPEND, "uart_smd_dma");

	tasklet_init(&rx->tlet, uart_smd_rx_tlet,
			(unsigned long) &rx->tlet);
	tasklet_init(&tx->tlet, uart_smd_tx_tlet,
			(unsigned long) &tx->tlet);

	rx->pool = dma_pool_create("rx_buffer_pool", uport->dev,
				   smd_port->rx_buf_size, 16, 0);
	if (!rx->pool)
	{
		pr_err("%s(): cannot allocate rx_buffer_pool", __func__);
		ret = -ENOMEM;
		goto exit_tasket_init;
	}

	rx->buffer = dma_pool_alloc(rx->pool, GFP_KERNEL, &rx->rbuffer);
	if (!rx->buffer)
	{
		pr_err("%s(): cannot allocate rx->buffer", __func__);
		ret = -ENOMEM;
		goto free_pool;
	}

	/* Allocate the command pointer. Needs to be 64 bit aligned */
	rx->command_ptr = kmalloc(sizeof(dmov_box), GFP_KERNEL | __GFP_DMA);
	if (!rx->command_ptr)
	{
		pr_err("%s(): cannot allocate rx->command_ptr", __func__);
		ret = -ENOMEM;
		goto free_rx_buffer;
	}

	rx->command_ptr_ptr = kmalloc(sizeof(u32), GFP_KERNEL | __GFP_DMA);
	if (!rx->command_ptr_ptr)
	{
		pr_err("%s(): cannot allocate rx->command_ptr_ptr", __func__);
		ret = -ENOMEM;
		goto free_rx_command_ptr;
	}

	rx->command_ptr->num_rows = ((smd_port->rx_buf_size >> 4) << 16) |
					 (smd_port->rx_buf_size >> 4);

	rx->command_ptr->dst_row_addr = rx->rbuffer;

	/* Set up Uart Receive */
	uart_write_reg(uport, UARTDM_RFWR_ADDR, 0);

	rx->xfer.complete_func = uart_smd_dmov_rx_callback;
	rx->xfer.exec_func = uart_smd_dmov_rx_exec_callback;

	rx->command_ptr->cmd = CMD_LC |
	    CMD_SRC_CRCI(smd_port->dma_rx_crci) | CMD_MODE_BOX;

	rx->command_ptr->src_dst_len = (UART_SMD_BURST_SIZE << 16)
					   | (UART_SMD_BURST_SIZE);
	rx->command_ptr->row_offset =  UART_SMD_BURST_SIZE;
	rx->command_ptr->src_row_addr = uport->mapbase + UARTDM_RF_ADDR;

	rx->mapped_cmd_ptr = dma_map_single(uport->dev, rx->command_ptr,
					    sizeof(dmov_box), DMA_TO_DEVICE);

	*rx->command_ptr_ptr = CMD_PTR_LP | DMOV_CMD_ADDR(rx->mapped_cmd_ptr);

	rx->cmdptr_dmaaddr = dma_map_single(uport->dev, rx->command_ptr_ptr,
					    sizeof(u32), DMA_TO_DEVICE);
	rx->xfer.cmdptr = DMOV_CMD_ADDR(rx->cmdptr_dmaaddr);

	INIT_DELAYED_WORK(&rx->write_smd_work, write_smd_work);

	return ret;

free_rx_command_ptr:
	kfree(rx->command_ptr);

free_rx_buffer:
	dma_pool_free(smd_port->rx.pool, smd_port->rx.buffer,
			smd_port->rx.rbuffer);

free_pool:
	dma_pool_destroy(smd_port->rx.pool);

exit_tasket_init:
	wake_lock_destroy(&smd_port->rx.wake_lock);
	wake_lock_destroy(&smd_port->dma_wake_lock);
	tasklet_kill(&smd_port->tx.tlet);
	tasklet_kill(&smd_port->rx.tlet);
	dma_unmap_single(uport->dev, smd_port->tx.mapped_cmd_ptr_ptr,
			sizeof(u32), DMA_TO_DEVICE);
	dma_unmap_single(uport->dev, smd_port->tx.mapped_cmd_ptr,
			sizeof(dmov_box), DMA_TO_DEVICE);
	kfree(smd_port->tx.command_ptr_ptr);

free_tx_command_ptr:
	kfree(smd_port->tx.command_ptr);
	return ret;
}

static int __devinit uart_smd_probe(struct platform_device *pdev)
{
	int ret;
	struct uart_port *uport;
	struct uart_smd_port *smd_port;
	unsigned int data;
	struct resource *resource;
	struct device_attribute *attr;
	struct device_attribute **attrs;
	struct msm_serial_hs_platform_data *pdata = pdev->dev.platform_data;

	if (pdev->id < 0 || pdev->id >= UART_SMD_NR)
	{
		printk(KERN_ERR "Invalid plaform device ID = %d\n", pdev->id);
		return -EINVAL;
	}

	smd_port = &smd_port_instance[pdev->id];
	uport = &smd_port->uport;

	uport->dev = &pdev->dev;

	resource = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (unlikely(!resource))
		return -ENXIO;
	uport->mapbase = resource->start;  /* virtual address */

	uport->membase = ioremap(uport->mapbase, PAGE_SIZE);
	if (unlikely(!uport->membase))
		return -ENOMEM;

	uport->irq = platform_get_irq(pdev, 0);
	if (unlikely((int)uport->irq < 0))
		return -ENXIO;

	if (pdata == NULL)
		smd_port->wakeup.irq = -1;
	else
	{
		smd_port->wakeup.irq = pdata->wakeup_irq;
		smd_port->wakeup.ignore = 1;
		smd_port->wakeup.inject_rx = pdata->inject_rx_on_wakeup;
		smd_port->wakeup.rx_to_inject = pdata->rx_to_inject;

		if (unlikely(smd_port->wakeup.irq < 0))
			return -ENXIO;
	}

	if (pdata && pdata->uartdm_rx_buf_size)
		smd_port->rx_buf_size = pdata->uartdm_rx_buf_size;
	else
		smd_port->rx_buf_size = UART_SMD_RX_BUF_SIZE;

	resource = platform_get_resource_byname(pdev, IORESOURCE_DMA,
						"uartdm_channels");
	if (unlikely(!resource))
		return -ENXIO;
	smd_port->dma_tx_channel = resource->start;
	smd_port->dma_rx_channel = resource->end;

	resource = platform_get_resource_byname(pdev, IORESOURCE_DMA,
						"uartdm_crci");
	if (unlikely(!resource))
		return -ENXIO;
	smd_port->dma_tx_crci = resource->start;
	smd_port->dma_rx_crci = resource->end;

	uport->iotype = UPIO_MEM;
	uport->fifosize = 64;
	uport->flags = UPF_BOOT_AUTOCONF;
	uport->uartclk = 7372800;
	smd_port->imr_reg = 0x0;

	smd_port->clk = clk_get(&pdev->dev, "core_clk");
	if (IS_ERR(smd_port->clk))
		return PTR_ERR(smd_port->clk);

	smd_port->pclk = clk_get(&pdev->dev, "iface_clk");
	/*
	 * Some configurations do not require explicit pclk control so
	 * do not flag error on pclk get failure.
	 */
	if (IS_ERR(smd_port->pclk))
		smd_port->pclk = NULL;

	pr_warn("%s: clk=%d, pclk=%d.\n", __func__, (int)(smd_port->clk),
		(int)(smd_port->clk));

	ret = clk_set_rate(smd_port->clk, uport->uartclk);
	if (ret)
	{
		printk(KERN_WARNING "Error setting clock rate on UART\n");
		return ret;
	}

	smd_port->uart_smd_wq = alloc_workqueue("uart_smd_wq",
					WQ_UNBOUND | WQ_MEM_RECLAIM, 1);
	if (!smd_port->uart_smd_wq)
	{
		pr_err("%s(): Unable to create workqueue uart_smd_wq\n",
								__func__);
		return -ENOMEM;
	}

	INIT_WORK(&smd_port->clock_off_w, uart_smd_clock_off_work);
	mutex_init(&smd_port->clk_mutex);

	clk_prepare_enable(smd_port->clk);
	if (smd_port->pclk)
		clk_prepare_enable(smd_port->pclk);

	ret = uart_smd_init_port(uport);
	if (unlikely(ret))
	{
		clk_disable_unprepare(smd_port->clk);
		if (smd_port->pclk)
			clk_disable_unprepare(smd_port->pclk);
		return ret;
	}

	/* configure the CR Protection to Enable */
	uart_write_reg(uport, UARTDM_CR_ADDR, CR_PROTECTION_EN);

	/*
	 * Enable Command register protection before going ahead as this hw
	 * configuration makes sure that issued cmd to CR register gets complete
	 * before next issued cmd start. Hence mb() requires here.
	 */
	mb();

	/*
	 * Set RX_BREAK_ZERO_CHAR_OFF and RX_ERROR_CHAR_OFF
	 * so any rx_break and character having parity of framing
	 * error don't enter inside UART RX FIFO.
	 */
	data = uart_read_reg(uport, UARTDM_MR2_ADDR);
	data |= (UARTDM_MR2_RX_BREAK_ZERO_CHAR_OFF |
			UARTDM_MR2_RX_ERROR_CHAR_OFF);
	uart_write_reg(uport, UARTDM_MR2_ADDR, data);
	mb();

	clk_disable_unprepare(smd_port->clk);
	if (smd_port->pclk)
		clk_disable_unprepare(smd_port->pclk);

	smd_port->clk_state = MSM_HS_CLK_PORT_OFF;
	hrtimer_init(&smd_port->clk_off_timer, CLOCK_MONOTONIC,
		     HRTIMER_MODE_REL);
	smd_port->clk_off_timer.function = uart_smd_clk_off_retry;
	smd_port->clk_off_delay = ktime_set(0, 1000000);  /* 1ms */

	//crate sysfs files.
	attrs = uart_smd_attributes;
	while((attr = *attrs++))
	{
		ret = device_create_file(&pdev->dev, attr);
		if (ret)
		{
			pr_err("create auxpcm interface error: %d", ret);
			return ret;
		}
	}

	smd_port->uconfig.baudrate = 115200;
	smd_port->uconfig.bit_per_char = 8;
	smd_port->uconfig.parity = 0;
	smd_port->uconfig.stop_bits = 1;
	smd_port->uconfig.hw_fc_enable = 0;

	INIT_DELAYED_WORK(&smd_port->smd_delayed_work, uart_smd_init_worker);

	schedule_delayed_work(&smd_port->smd_delayed_work, msecs_to_jiffies(UART_SMD_RETRY_TIMEOUT));
	
	return 0;
}

/*
 *  Called by the upper layer when port is closed.
 *     - Disables the port
 *     - Unhook the ISR
 */
static void uart_smd_shutdown(struct uart_port *uport)
{
	int ret;
	unsigned int data;
	unsigned long flags;
	struct uart_smd_port *smd_port = UPORT_TO_SMD(uport);

	spin_lock_irqsave(&uport->lock, flags);

	/* deactivate if any clock off hrtimer is active. */
	hrtimer_try_to_cancel(&smd_port->clk_off_timer);

	/* disable UART TX interface to DM */
	data = uart_read_reg(uport, UARTDM_DMEN_ADDR);
	data &= ~UARTDM_TX_DM_EN_BMSK;
	uart_write_reg(uport, UARTDM_DMEN_ADDR, data);
	mb();

	if (smd_port->tx.dma_in_flight)
	{
		smd_port->tx.flush = FLUSH_STOP;
		/* discard flush */
		msm_dmov_flush(smd_port->dma_tx_channel, 0);
		spin_unlock_irqrestore(&uport->lock, flags);
		ret = wait_event_timeout(smd_port->tx.wait,
			smd_port->tx.flush == FLUSH_SHUTDOWN, 100);
		if (!ret)
		{
			pr_err("%s():HSUART TX Stalls.\n", __func__);
			print_uart_registers(smd_port);
		}
	}
	else
	{
		spin_unlock_irqrestore(&uport->lock, flags);
	}

	tasklet_kill(&smd_port->tx.tlet);

	if (smd_port->rx.dma_in_flight)
	{
		if (smd_port->rx.flush < FLUSH_STOP)
		{
			pr_err("%s(): rx.flush is not correct.\n",
							__func__);
			print_uart_registers(smd_port);
			BUG_ON(1);
		}

		ret = wait_event_timeout(smd_port->rx.wait,
			smd_port->rx.flush == FLUSH_SHUTDOWN,
			UART_SMD_RX_FLUSH_COMPLETE_TIMEOUT);
		if (!ret)
		{
			pr_err("%s(): Rx completion failed.\n", __func__);
			print_uart_registers(smd_port);
		}
	}

	tasklet_kill(&smd_port->rx.tlet);

	spin_lock_irqsave(&uport->lock, flags);
	/* Disable all UART interrupts */
	smd_port->imr_reg = 0;
	uart_write_reg(uport, UARTDM_IMR_ADDR, smd_port->imr_reg);

	/* Free the UART IRQ line */
	free_irq(uport->irq, smd_port);

	smd_port->is_shutdown = true;
	spin_unlock_irqrestore(&uport->lock, flags);

	/* disable UART RX interface to DM */
	data = uart_read_reg(uport, UARTDM_DMEN_ADDR);
	data &= ~UARTDM_RX_DM_EN_BMSK;
	uart_write_reg(uport, UARTDM_DMEN_ADDR, data);

	cancel_delayed_work_sync(&smd_port->rx.write_smd_work);
	flush_workqueue(smd_port->uart_smd_wq);
	pm_runtime_disable(uport->dev);

	/* Disable the transmitter */
	uart_write_reg(uport, UARTDM_CR_ADDR, UARTDM_CR_TX_DISABLE_BMSK);
	/* Disable the receiver */
	uart_write_reg(uport, UARTDM_CR_ADDR, UARTDM_CR_RX_DISABLE_BMSK);

	/*
	 * Complete all device write before actually disabling uartclk.
	 * Hence mb() requires here.
	 */
	mb();

	if (smd_port->clk_state != MSM_HS_CLK_OFF)
	{
		/* to balance clk_state */
		clk_disable_unprepare(smd_port->clk);
		if (smd_port->pclk)
			clk_disable_unprepare(smd_port->pclk);
		wake_unlock(&smd_port->dma_wake_lock);
	}

	smd_port->clk_state = MSM_HS_CLK_PORT_OFF;
	dma_unmap_single(uport->dev, smd_port->tx.dma_base,
			 UART_XMIT_SIZE, DMA_TO_DEVICE);
}

static int __devexit uart_smd_remove(struct platform_device *pdev)
{
	struct uart_smd_port *smd_port;
	struct device *dev;
	struct device_attribute *attr;
	struct device_attribute **attrs;

	if (pdev->id < 0 || pdev->id >= UART_SMD_NR)
	{
		printk(KERN_ERR "Invalid plaform device ID = %d\n", pdev->id);
		return -EINVAL;
	}

	smd_port = &smd_port_instance[pdev->id];
	dev = smd_port->uport.dev;

	uart_smd_shutdown(&smd_port->uport);

	//remove sysfs files.
	attrs = uart_smd_attributes;
	while((attr = *attrs++))
	{
		device_remove_file(&pdev->dev, attr);
	}

	dma_unmap_single(dev, smd_port->rx.mapped_cmd_ptr, sizeof(dmov_box),
			 DMA_TO_DEVICE);
	dma_pool_free(smd_port->rx.pool, smd_port->rx.buffer,
		      smd_port->rx.rbuffer);
	dma_pool_destroy(smd_port->rx.pool);

	dma_unmap_single(dev, smd_port->rx.cmdptr_dmaaddr, sizeof(u32),
			 DMA_TO_DEVICE);
	dma_unmap_single(dev, smd_port->tx.mapped_cmd_ptr_ptr, sizeof(u32),
			 DMA_TO_DEVICE);
	dma_unmap_single(dev, smd_port->tx.mapped_cmd_ptr, sizeof(dmov_box),
			 DMA_TO_DEVICE);

	wake_lock_destroy(&smd_port->rx.wake_lock);
	wake_lock_destroy(&smd_port->dma_wake_lock);
	destroy_workqueue(smd_port->uart_smd_wq);
	mutex_destroy(&smd_port->clk_mutex);

	clk_put(smd_port->clk);
	if (smd_port->pclk)
		clk_put(smd_port->pclk);

	/* Free the tx resources */
	kfree(smd_port->tx.command_ptr);
	kfree(smd_port->tx.command_ptr_ptr);

	/* Free the rx resources */
	kfree(smd_port->rx.command_ptr);
	kfree(smd_port->rx.command_ptr_ptr);

	iounmap(smd_port->uport.membase);

	kfree(smd_port->tx_buf);

	return 0;
}

static int uart_smd_runtime_idle(struct device *dev)
{
	/*
	 * returning success from idle results in runtime suspend to be
	 * called
	 */
	return 0;
}

static int uart_smd_runtime_resume(struct device *dev)
{
	struct platform_device *pdev = container_of(dev, struct
						    platform_device, dev);
	struct uart_smd_port *smd_port = &smd_port_instance[pdev->id];
	uart_smd_request_clock_on(&smd_port->uport);
	return 0;
}

static int uart_smd_runtime_suspend(struct device *dev)
{
	struct platform_device *pdev = container_of(dev, struct
						    platform_device, dev);
	struct uart_smd_port *smd_port = &smd_port_instance[pdev->id];
	uart_smd_request_clock_off(&smd_port->uport);
	return 0;
}

#ifdef CONFIG_PM
static int uart_smd_resume(struct device *dev)
{
	struct platform_device *pdev = container_of(dev, struct
							platform_device, dev);
	struct uart_smd_port *smd_port = &smd_port_instance[pdev->id];
	uart_smd_clk_en(smd_port, 1);
	return 0;
}

static int uart_smd_suspend(struct device *dev)
{
	struct platform_device *pdev = container_of(dev, struct
							platform_device, dev);
	struct uart_smd_port *smd_port = &smd_port_instance[pdev->id];
	uart_smd_clk_en(smd_port, 0);
	return 0;
}
#else
#define uart_smd_resume NULL
#define uart_smd_suspend NULL
#endif

static const struct dev_pm_ops uart_smd_dev_pm_ops = {
	.suspend         = uart_smd_suspend,
	.resume			 = uart_smd_resume,
	.runtime_suspend = uart_smd_runtime_suspend,
	.runtime_resume  = uart_smd_runtime_resume,
	.runtime_idle    = uart_smd_runtime_idle,
};

static struct platform_driver uart_smd_platform_driver = {
	.probe	= uart_smd_probe,
	.remove = __devexit_p(uart_smd_remove),
	.driver = {
		.name = "uart_smd",
		.pm   = &uart_smd_dev_pm_ops,
	},
};

static int __init uart_smd_init(void)
{
	int ret;
	int i;

	/* Init all UARTS as non-configured */
	for (i = 0; i < UART_SMD_NR; i++)
		smd_port_instance[i].uport.type = PORT_UNKNOWN;

	ret = platform_driver_register(&uart_smd_platform_driver);
	if (ret)
    {
		printk(KERN_ERR "%s failed to load\n", __FUNCTION__);
		return ret;
	}

	printk(KERN_INFO "uart_smd module loaded\n");
	return ret;
}

static void __exit uart_smd_exit(void)
{
	printk(KERN_INFO "uart_smd module removed\n");
	platform_driver_unregister(&uart_smd_platform_driver);
}

module_init(uart_smd_init);
module_exit(uart_smd_exit);
MODULE_VERSION("1.2");
MODULE_LICENSE("GPL v2");
