/*
 * u_smd.c - utilities for USB gadget serial over smd
 *
 * Copyright (c) 2011, The Linux Foundation. All rights reserved.
 *
 * This code also borrows from drivers/usb/gadget/u_serial.c, which is
 * Copyright (C) 2000 - 2003 Al Borchers (alborchers@steinerpoint.com)
 * Copyright (C) 2008 David Brownell
 * Copyright (C) 2008 by Nokia Corporation
 * Copyright (C) 1999 - 2002 Greg Kroah-Hartman (greg@kroah.com)
 * Copyright (C) 2000 Peter Berger (pberger@brimson.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */
#include <linux/kernel.h>
#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/termios.h>
#include <mach/msm_smd.h>
#include <linux/debugfs.h>

#include "u_serial.h"

#ifdef CONFIG_QUECTEL_FLOWCTRL
	volatile int q_flowctrl = 0;
#endif

#ifdef CONFIG_QUECTEL_ESCAPE_FEATURE
#include <linux/remote_spinlock.h>
#define QUECTEL_ESCAPE_DETECTED       43
#define QUECTEL_ESCAPE_UNDETECTED     0
typedef struct quectel_escape{
    unsigned int  smem_escape_spinlock;
    unsigned char escape;   
}QUECTEL_ESCAPE_T;

QUECTEL_ESCAPE_T *quectel_escape_addr = NULL;

#define SMEM_SPINLOCK_SMEM_ALLOC       "S:3"
static remote_spinlock_t remote_spinlock;
static int spinlocks_initialized = 0;

#endif

#ifdef CONFIG_QUECTEL_LINUX_FILE
#include <linux/fs.h>
#include <linux/syscalls.h>
#include <linux/statfs.h>
#include <linux/namei.h>
#include <linux/mount.h>

#define QUECTEL_DISABLE_PRINTK	0

	extern void pet_watchdog(void);
#if QUECTEL_DISABLE_PRINTK
	extern int q_linux_file_printk;
#endif
        struct file *fop=NULL;
        unsigned long long fsize=0,ftotal=0;
        bool fopen=false,fupl=false;
        struct timer_list file_timer;
	struct list_head usb_send_pool;

#define UFS_PATH_PRE		"/cache/ufs/"
#define UFS_TMP_FILE		UFS_PATH_PRE".ufs.tmp"

#define SD_PATH_PRE            "/media/card/"
#define SD_TMP_FILE            SD_PATH_PRE".sd.tmp"
#define NON_SD_DEV		"/dev/root"

#define UFS_PRE			"UFS"
#define RAM_PRE			"RAM"
#define SD_PRE                 "SD"

#define UFS_FP_INDEX_BASE       0
#define SD_FP_INDEX_BASE	100

typedef enum{
Q_TYPE_RAM,
Q_TYPE_UFS,
Q_TYPE_SD,

Q_TYPE_MAX
}QUECTEL_FILE_TYPE;

#define QACK_VALUE_LIMIT	1024
#define FILE_NAME_SIZE		128
#define BUFER_SIZE		128
#define FILE_NUM_MAX		15
#define FILE_BUFER_LEN		512
#define FILE_TIMEOUT_DEFAULT	5
#define FILE_UFS_SIZE		33554432 //(32MB)

#define MODEM_MODE_CMD			0
#define MODEM_MODE_AT_DATA		1
#define MODEM_MODE_MODEM_DATA           2

#define IF_DENTRY_VALID(fdty) if(fdty && virt_addr_valid(fdty->d_inode))

typedef struct filelist_head
{
	unsigned int num;
	unsigned long long used;
	unsigned long long total;
	struct list_head fh;
	unsigned int fp_index;
	struct file *fp[FILE_NUM_MAX];
}FILELIST_HEAD_T;

typedef struct file_info
{
	char name[FILE_NAME_SIZE];
	unsigned long long len;	
	struct list_head list;
}FILE_INFO_T;

typedef enum
{
	QUECTEL_APPEND=0,
	QUECTEL_OVERWRITE,
	QUECTEL_RDONLY,
	QUECTEL_MAX
}FILE_OPENMODE_E;


FILELIST_HEAD_T ufs_file,sd_file;
FILELIST_HEAD_T *qfile_head=NULL;
static int ftime=FILE_TIMEOUT_DEFAULT;
static int fack=0,facknum=1;
static unsigned int fchecksum = 0,fodd = 0;
static unsigned char foddflag = 0;
volatile int q_modemmode = MODEM_MODE_CMD; //default cmd mode
volatile int q_linuxmode = MODEM_MODE_CMD; //default cmd mode
volatile int q_current_mode = MODEM_MODE_AT_DATA;
static char ErrBuf[]="ERROR\r\n";

struct quectel_fdwl
{
	struct file *fp;
	struct gsmd_port *port;
	struct work_struct qfdwl_work;	
};
static void qfdwl_handle(struct work_struct *work);
struct quectel_fdwl qfdwl;

struct quectel_frd
{
        struct file *fp;
	unsigned long long size;
        struct gsmd_port *port;
        struct work_struct qfread_work;
};
static void qfread_handle(struct work_struct *work);
struct quectel_frd qfread;

#endif

#define SMD_RX_QUEUE_SIZE		8
#define SMD_RX_BUF_SIZE			2048

#define SMD_TX_QUEUE_SIZE		8
#define SMD_TX_BUF_SIZE			2048

static struct workqueue_struct *gsmd_wq;

#define SMD_N_PORTS	2
#define CH_OPENED	0
#define CH_READY	1
struct smd_port_info {
	struct smd_channel	*ch;
	char			*name;
	unsigned long		flags;
};

#ifdef CONFIG_QUECTEL_LINUX_FILE
int quectel_req_init(const char *src,char *dst,int len)
{
	int n = 0;
	n = strchr(src,'\n') - src;
        if(n > len - 2)
        {
		return 0;
        }
        memset(dst,0,len);
        strncpy(dst,src,n);
        dst[n-1]='\n';
        dst[n]='\0';

	n = 0;
	while(*dst != '\0')
        {
                if(*dst == ':' || *dst == '\"' || *dst == ',' || *dst == '=')
                *dst = ' ';
                dst++;
                n++;
        }

	return n;
}

#if QUECTEL_DISABLE_PRINTK
void quectel_printk_flag(bool flag)
{
	q_linux_file_printk=flag?0:1;
	return;
}
#endif

int quectel_strncmp(const char *cs, const char *ct, size_t count)
{
	unsigned char c1, c2;

	while (count) {
		c1 = *cs++;
		c2 = *ct++;
		if (c1 >= 'A' && c1 <= 'Z')
        	{
            		c1 += 'a' - 'A';
        	}
        	if (c2 >= 'A' && c2 <= 'Z')
        	{
            		c2 += 'a' - 'A';
        	}
		if (c1 != c2)
			return c1 < c2 ? -1 : 1;
		if (!c1)
			break;
		count--;
	}
	return 0;
}

struct file *quectel_fopen(const char *filename, int flags, int mode)
{ 
    struct file *filp = NULL;

    filp = filp_open(filename, flags, mode);
    return (IS_ERR(filp)) ? NULL : filp;
}

void quectel_fclose(struct file *filp)
{
    if (filp)
        fput(filp);
}

int quectel_fread(char *buf, int len, struct file *filp)
{
    int readlen;
 
    mm_segment_t oldfs;
 
    if (filp == NULL)
        return -ENOENT;
 
    if (filp->f_op->read == NULL)
        return -ENOSYS;
 
    if (((filp->f_flags & O_ACCMODE) & O_RDONLY) != 0)
        return -EACCES;


    oldfs = get_fs();
    set_fs(KERNEL_DS);
 
    readlen = filp->f_op->read(filp, buf, len, &filp->f_pos);
 
    set_fs(oldfs);
    return readlen;
}

int quectel_fwrite(char *buf, int len, struct file *filp)
{
    int writelen;
    mm_segment_t oldfs;
 
    if (filp == NULL)
        return -ENOENT;
 
    if (filp->f_op->write == NULL)
        return -ENOSYS;
 
    if (((filp->f_flags & O_ACCMODE) & (O_WRONLY | O_RDWR)) == 0)
        return -EACCES;
 
    oldfs = get_fs();
    set_fs(KERNEL_DS);
    writelen = filp->f_op->write(filp, buf, len, &filp->f_pos);
    set_fs(oldfs);
    return writelen;
}

int quectel_fseek(struct file *filp, unsigned long long offset, int whence)
{
 
    int pos = filp->f_pos;
 
    if (filp) {
 
	switch(whence){
	case SEEK_SET:
		if (offset > filp->f_dentry->d_inode->i_size)
			return -EINVAL; 
		pos = offset;
		break;
	case SEEK_CUR:
		if (offset > filp->f_dentry->d_inode->i_size - pos)
                        return -EINVAL;
		pos += offset;
		break;
	case SEEK_END:
		if (offset > filp->f_dentry->d_inode->i_size)
                        return -EINVAL;
		pos = filp->f_dentry->d_inode->i_size - offset;
		break;
	default:
		return -EINVAL;
	}
 
        if (pos < 0)
            pos = 0;
 
        return (filp->f_pos = pos);
    } else
        return -ENOENT;
}

void quectel_fdel(char *name)
{
	sys_unlinkat(AT_FDCWD,name,0);
	return;
}

bool quectel_file_init(QUECTEL_FILE_TYPE ftype)
{
        struct file *fp=NULL;
        struct list_head *tmp,*p;
        struct dentry *fdty;
	FILE_INFO_T *finfo;
	struct kstatfs sdfs;
	char tmpbuf[BUFER_SIZE];

	memset(tmpbuf,0,BUFER_SIZE);
	switch(ftype){
	case Q_TYPE_UFS:
		qfile_head = &ufs_file;
		qfile_head->total = FILE_UFS_SIZE;
		strcpy(tmpbuf,UFS_TMP_FILE);
		break;
	case Q_TYPE_SD:
		qfile_head = &sd_file;
		strcpy(tmpbuf,SD_TMP_FILE);
                break;
	default:
		break;
	}

	qfile_head->used = 0;
	qfile_head->num = 0;
	list_for_each_safe(p,tmp,&qfile_head->fh)
	{
		finfo=list_entry(p,FILE_INFO_T,list);
		list_del(p);
		kfree(finfo);
	}
        fp = quectel_fopen(tmpbuf,O_CREAT | O_RDWR,0);
        list_for_each(tmp,&fp->f_dentry->d_u.d_child) {
                fdty=list_entry(tmp,struct dentry,d_u.d_child);
                IF_DENTRY_VALID(fdty)
		{
			finfo = kmalloc(sizeof(FILE_INFO_T),GFP_ATOMIC);
			strcpy(finfo->name,fdty->d_iname);
			finfo->len = fdty->d_inode->i_size;
			list_add(&finfo->list,&qfile_head->fh);
                	qfile_head->used += fdty->d_inode->i_size;
			qfile_head->num++;
		}
        }

	if(ftype == Q_TYPE_SD){
		if(!strcmp(fp->f_vfsmnt->mnt_devname,NON_SD_DEV))
		{
			return false;
		}
		vfs_statfs(&fp->f_path,&sdfs);
		qfile_head->total = sdfs.f_bsize * sdfs.f_blocks;
		qfile_head->used = qfile_head->total - sdfs.f_bsize * sdfs.f_bfree;
	}
        quectel_fclose(fp);
        quectel_fdel(tmpbuf);

        return true;
}

bool quectel_file_exist(char *name)
{
        struct list_head *tmp;
#if 0
        struct dentry *fdty;
	struct file *fp=NULL;
        fp = quectel_fopen(UFS_TMP_FILE,O_CREAT | O_RDWR,0);
        list_for_each(tmp,&fp->f_dentry->d_u.d_child) {
                fdty=list_entry(tmp,struct dentry,d_u.d_child);
		IF_DENTRY_VALID(fdty){
                if(!strcmp(name,fdty->d_iname) ){ 
                        	quectel_fclose(fp);
                        	quectel_fdel(UFS_TMP_FILE);
                        	return true;
                }
		}
        }
        quectel_fclose(fp);
        quectel_fdel(UFS_TMP_FILE);
#else
	FILE_INFO_T *finfo;
	list_for_each(tmp,&qfile_head->fh) {
		finfo = list_entry(tmp,FILE_INFO_T,list);
		if(!strcmp(name,finfo->name)) return true;
	}
#endif
        return false;
}

bool quectel_update_size_byname(const char *name,unsigned long long len)
{
	struct list_head *tmp;
	FILE_INFO_T *finfo;
        list_for_each(tmp,&qfile_head->fh) {
                finfo = list_entry(tmp,FILE_INFO_T,list);
                if(!strcmp(name,finfo->name))
		{
			finfo->len = len;
			return true;
		}
        }	
	return false;
}

bool quectel_del_byname(const char *name)
{
        struct list_head *tmp,*p;
        FILE_INFO_T *finfo;

        list_for_each_safe(tmp,p,&qfile_head->fh) {
                finfo = list_entry(tmp,FILE_INFO_T,list);
                if(!strcmp(name,finfo->name))
                {
			qfile_head->used -= finfo->len;
			qfile_head->num--;
                        list_del(tmp);
			kfree(finfo);
                        return true;
                }
        }
        return false;
}

bool is_even(int num)
{
	return ((num%2 == 0)?true:false);
}

unsigned int file_CalcCheckSum(unsigned int *checksum, unsigned int *odd_tmp, unsigned char *odd_flag, unsigned char *buffer, unsigned int size, bool bend)
{
    unsigned int i = 0;
    unsigned int tmp = 0;

    //gibson:doesn't check the input param, because it is only allowed to call inside.
    if(0 != *odd_flag && 0 != size)
    { 
	 *odd_tmp |= buffer[0];
        *checksum ^= *odd_tmp;
        buffer++;
        size--;
        *odd_tmp = 0;
        *odd_flag  = 0;
    }
    else if(0 != *odd_flag && (0 == size) && (true == bend))
    {
        *checksum ^= *odd_tmp;
        *odd_tmp = 0;
        *odd_flag = 0;
    }

   
    for(i=0;i<size/2;i++)
    {
        tmp = buffer[2*i];
        tmp <<= 8;
        tmp |= buffer[2*i+1];
        *checksum ^= tmp;
        tmp = 0;
    }

    if(false == is_even(size))
    {
        *odd_tmp = buffer[size-1];
        *odd_tmp <<= 8;
        *odd_flag = 1;
        if(true == bend)
        {
            *checksum ^= *odd_tmp;
            *odd_tmp = 0;
            *odd_flag = 0;
        }
    }

    return *checksum;
}

#endif

// 2014-09-28 add by scott.hu
#define QUECTEL_2ND_AT_PORT

struct smd_port_info smd_pi[SMD_N_PORTS] = {
	{
		.name = "DS",
	},
	{

// 2014-09-28 add by scott.hu
#ifdef QUECTEL_2ND_AT_PORT
		.name = "DATA2",
#else
		.name = "UNUSED",
#endif
// end scott.hu
	},
};

struct gsmd_port {
	unsigned		port_num;
	spinlock_t		port_lock;

	unsigned		n_read;
	struct list_head	read_pool;
	struct list_head	read_queue;
	struct work_struct	push;

	struct list_head	write_pool;
	struct work_struct	pull;

	struct gserial		*port_usb;

	struct smd_port_info	*pi;
	struct delayed_work	connect_work;

	/* At present, smd does not notify
	 * control bit change info from modem
	 */
	struct work_struct	update_modem_ctrl_sig;

#define SMD_ACM_CTRL_DTR		0x01
#define SMD_ACM_CTRL_RTS		0x02
	unsigned		cbits_to_modem;

#define SMD_ACM_CTRL_DCD		0x01
#define SMD_ACM_CTRL_DSR		0x02
#define SMD_ACM_CTRL_BRK		0x04
#define SMD_ACM_CTRL_RI		0x08
	unsigned		cbits_to_laptop;

	/* pkt counters */
	unsigned long		nbytes_tomodem;
	unsigned long		nbytes_tolaptop;
};

static struct smd_portmaster {
	struct mutex lock;
	struct gsmd_port *port;
	struct platform_driver pdrv;
} smd_ports[SMD_N_PORTS];
static unsigned n_smd_ports;

#ifdef CONFIG_QUECTEL_LINUX_FILE
typedef struct file_cmd_list
{
        char cmd[BUFER_SIZE];
        bool (*func)(struct gsmd_port *port,struct usb_request *req);
}FILE_CMD_LIST_T;

bool quectel_qfupl_cmd(struct gsmd_port *port,struct usb_request *req);
bool quectel_qfdel_cmd(struct gsmd_port *port,struct usb_request *req);
bool quectel_qflds_cmd(struct gsmd_port *port,struct usb_request *req);
bool quectel_qflst_cmd(struct gsmd_port *port,struct usb_request *req);
bool quectel_qfdwl_cmd(struct gsmd_port *port,struct usb_request *req);
bool quectel_qfopen_cmd(struct gsmd_port *port,struct usb_request *req);
bool quectel_qfread_cmd(struct gsmd_port *port,struct usb_request *req);
bool quectel_qfwrite_cmd(struct gsmd_port *port,struct usb_request *req);
bool quectel_qfseek_cmd(struct gsmd_port *port,struct usb_request *req);
bool quectel_qfposition_cmd(struct gsmd_port *port,struct usb_request *req);
bool quectel_qfclose_cmd(struct gsmd_port *port,struct usb_request *req);

static FILE_CMD_LIST_T file_cmd[] = {
{"at+qfupl=",quectel_qfupl_cmd},
{"at+qfdel=",quectel_qfdel_cmd},
{"at+qflds",quectel_qflds_cmd},
{"at+qflst",quectel_qflst_cmd},
{"at+qfdwl=",quectel_qfdwl_cmd},
{"at+qfopen=",quectel_qfopen_cmd},
{"at+qfread=",quectel_qfread_cmd},
{"at+qfwrite=",quectel_qfwrite_cmd},
{"at+qfseek=",quectel_qfseek_cmd},
{"at+qfposition=",quectel_qfposition_cmd},
{"at+qfclose=",quectel_qfclose_cmd},

{"",NULL},
};
#endif

static void gsmd_free_req(struct usb_ep *ep, struct usb_request *req)
{
	kfree(req->buf);
	usb_ep_free_request(ep, req);
}

static void gsmd_free_requests(struct usb_ep *ep, struct list_head *head)
{
	struct usb_request	*req;

	while (!list_empty(head)) {
		req = list_entry(head->next, struct usb_request, list);
		list_del(&req->list);
		gsmd_free_req(ep, req);
	}
}

static struct usb_request *
gsmd_alloc_req(struct usb_ep *ep, unsigned len, gfp_t flags)
{
	struct usb_request *req;

	req = usb_ep_alloc_request(ep, flags);
	if (!req) {
		pr_err("%s: usb alloc request failed\n", __func__);
		return 0;
	}

	req->length = len;
	req->buf = kmalloc(len, flags);
	if (!req->buf) {
		pr_err("%s: request buf allocation failed\n", __func__);
		usb_ep_free_request(ep, req);
		return 0;
	}

	return req;
}

static int gsmd_alloc_requests(struct usb_ep *ep, struct list_head *head,
		int num, int size,
		void (*cb)(struct usb_ep *ep, struct usb_request *))
{
	int i;
	struct usb_request *req;

	pr_debug("%s: ep:%p head:%p num:%d size:%d cb:%p", __func__,
			ep, head, num, size, cb);

	for (i = 0; i < num; i++) {
		req = gsmd_alloc_req(ep, size, GFP_ATOMIC);
		if (!req) {
			pr_debug("%s: req allocated:%d\n", __func__, i);
			return list_empty(head) ? -ENOMEM : 0;
		}
		req->complete = cb;
		list_add(&req->list, head);
	}

	return 0;
}

static void gsmd_start_rx(struct gsmd_port *port)
{
	struct list_head	*pool;
	struct usb_ep		*out;
	unsigned long	flags;
	int ret;

	if (!port) {
		pr_err("%s: port is null\n", __func__);
		return;
	}

	spin_lock_irqsave(&port->port_lock, flags);

	if (!port->port_usb) {
		pr_debug("%s: USB disconnected\n", __func__);
		goto start_rx_end;
	}

	pool = &port->read_pool;
	out = port->port_usb->out;

	while (test_bit(CH_OPENED, &port->pi->flags) && !list_empty(pool)) {
		struct usb_request	*req;

		req = list_entry(pool->next, struct usb_request, list);
		list_del(&req->list);
		req->length = SMD_RX_BUF_SIZE;

		spin_unlock_irqrestore(&port->port_lock, flags);
		ret = usb_ep_queue(out, req, GFP_KERNEL);
		spin_lock_irqsave(&port->port_lock, flags);
		if (ret) {
			pr_err("%s: usb ep out queue failed"
					"port:%p, port#%d\n",
					 __func__, port, port->port_num);
			list_add_tail(&req->list, pool);
			break;
		}
	}
start_rx_end:
	spin_unlock_irqrestore(&port->port_lock, flags);
}

#ifdef CONFIG_QUECTEL_LINUX_FILE
static void usb_send_complete(struct usb_ep *ep, struct usb_request *req)
{
	struct gsmd_port *port = ep->driver_data;

	pr_debug("%s: ep:%p port:%p\n", __func__, ep, port);

	if (!port) {
		pr_err("%s: port is null\n", __func__);
		return;
	}

	spin_lock(&port->port_lock);
	if (req->status == -ESHUTDOWN) {
		spin_unlock(&port->port_lock);
		gsmd_free_req(ep, req);
		return;
	}

	if (req->status)
		pr_warning("%s: port:%p port#%d unexpected %s status %d\n",
				__func__, port, port->port_num,
				ep->name, req->status);

	list_add(&req->list, &usb_send_pool);
	spin_unlock(&port->port_lock);

	return;
}

void quectel_send(struct gsmd_port *port,char *data,unsigned int len)
{
	int ret=0;
	struct list_head *pool = &usb_send_pool;
	struct usb_ep *in;

	in = port->port_usb->in;
	if (!list_empty(pool)) {
	struct usb_request *req;
	req = list_entry(pool->next, struct usb_request, list);
	list_del(&req->list);
	len = len > FILE_BUFER_LEN?FILE_BUFER_LEN:len;
	memcpy(req->buf,data,len);
	req->length = len;
	ret = usb_ep_queue(in, req, GFP_KERNEL);
	if (ret) {
		pr_err("%s: usb ep out queue failed"
				"port:%p, port#%d err:%d\n",
				__func__, port, port->port_num, ret);
		/* could be usb disconnected */
		if (!port->port_usb)
			gsmd_free_req(in, req);
		else
			list_add(&req->list, pool);
	}
	}
	return ;
}

bool quectel_send_usb_urc(struct gsmd_port *port,char *data)
{	
	quectel_send(port,data,strlen(data));
	return true;
}

static void exit_data_mode(struct gsmd_port *port)
{
        quectel_update_size_byname(fop->f_dentry->d_iname,fop->f_dentry->d_inode->i_size);
	//file_CalcCheckSum(&fchecksum,&fodd,&foddflag,NULL,0,true);
        if(fupl)
        {
                fupl = false;
                quectel_fclose(fop);
                fop = NULL;
        }
        qfile_head->used+=ftotal;
        del_timer(&file_timer);
        //if(fack && facknum > 1) quectel_send_usb_urc(port,"\r\n");
        fopen = false;
        ftotal=0;
        fsize=0;
        fack=0;
        facknum=1;
	q_linuxmode = MODEM_MODE_CMD;
        return;
}

void file_timer_expire(unsigned long data)
{
        char buf[BUFER_SIZE];
	struct gsmd_port *port = (struct gsmd_port *)data;

	file_CalcCheckSum(&fchecksum,&fodd,&foddflag,NULL,0,true);
	if(fack && facknum > 1) quectel_send_usb_urc(port,"\r\n");	
	if(fupl) 
        	sprintf(buf,"+QFUPL: %lld,%x\r\n\r\n+CME ERROR: 421\r\n",ftotal,fchecksum);
        else    
        	sprintf(buf,"+QFWRITE: %lld,%lld\r\n\r\n+CME ERROR: 421\r\n",ftotal,fop->f_dentry->d_inode->i_size);
        exit_data_mode(port);
	//sprintf(buf,"+QFUPL: %d\r\n\r\n+CME ERROR: 421\r\n",ftotal);
        quectel_send_usb_urc(port,buf);
}


bool quectel_qcmd_show(QUECTEL_FILE_TYPE ftype,struct gsmd_port *port,struct usb_request *req)
{
	int n;
	char tmpbuf[BUFER_SIZE];
	char *packet = req->buf;

	n = strchr(packet,'\n') - packet;
        if(n > BUFER_SIZE - 2)
        {
		return false;
        }
        memset(tmpbuf,0,BUFER_SIZE);
        strncpy(tmpbuf,packet,n);
	tmpbuf[n-1]='\r';
        tmpbuf[n]='\n';
        tmpbuf[n+1]='\0';
        quectel_send_usb_urc(port,tmpbuf);

        return quectel_file_init(ftype);
}
bool quectel_qfupl_cmd(struct gsmd_port *port,struct usb_request *req)
{
	char *packet = req->buf;
	char fname[FILE_NAME_SIZE],tmpbuf[BUFER_SIZE],cbuf[BUFER_SIZE];
	FILE_INFO_T *finfo;
	QUECTEL_FILE_TYPE ftype;

        quectel_req_init(packet,cbuf,BUFER_SIZE);
        sscanf(cbuf,"%s%s%s%lld%d%d",fname,tmpbuf,fname,&fsize,&ftime,&fack);
	if(!strcmp(tmpbuf,RAM_PRE) || !strcmp(tmpbuf,"?")) return false;
	if(!strcmp(tmpbuf,SD_PRE)){
		ftype = Q_TYPE_SD;
		strcpy(tmpbuf,fname);
	}
	else
	{ 
		ftype = Q_TYPE_UFS;
		sscanf(cbuf,"%s%s%lld%d%d",fname,tmpbuf,&fsize,&ftime,&fack);
	}
	if(!quectel_qcmd_show(ftype,port,req))
		return quectel_send_usb_urc(port,ErrBuf);

	if(MODEM_MODE_CMD != q_linuxmode)
		return quectel_send_usb_urc(port,ErrBuf);

        if(quectel_file_exist(tmpbuf))
         {
                return quectel_send_usb_urc(port,"+CME ERROR: 407\r\n");
         }
	if(Q_TYPE_SD == ftype) strcpy(fname,SD_PATH_PRE);
	else
         	strcpy(fname,UFS_PATH_PRE);
         strcat(fname,tmpbuf);
         //printk("dbg:%s %d %d %d %d,%d\n",fname,fsize,ftime,fack,ufs_file.num,ufs_file.used);//tommy tem
         if(fsize > qfile_head->total - qfile_head->used || fsize == 0)
         {
                return quectel_send_usb_urc(port,"+CME ERROR: 423\r\n");
         }
	if(!ftime) ftime=FILE_TIMEOUT_DEFAULT;
	if(fsize)
         {
		if(port->port_num)
                       	q_linuxmode = MODEM_MODE_MODEM_DATA;
                else
                        q_linuxmode = MODEM_MODE_AT_DATA;

        	quectel_send_usb_urc(port,"CONNECT\r\n");
         	fop = quectel_fopen(fname,O_CREAT | O_RDWR,0);
                fopen = true;
		fupl = true;
		fchecksum = 0;
		fodd = 0;
		foddflag = 0;
		finfo = kmalloc(sizeof(FILE_INFO_T),GFP_ATOMIC);
		strcpy(finfo->name,tmpbuf);
		list_add(&finfo->list,&qfile_head->fh);
		qfile_head->num++;
                init_timer(&(file_timer));
                file_timer.function = file_timer_expire;
                file_timer.expires = jiffies + ftime*HZ;
                file_timer.data = (unsigned long)port;
                add_timer(&file_timer);
         }

	return false;
}

bool quectel_qfdel_cmd(struct gsmd_port *port,struct usb_request *req)
{
	char fname[FILE_NAME_SIZE],tmpbuf[BUFER_SIZE],cbuf[BUFER_SIZE];
	struct list_head *tmp,*p;
        //struct dentry *fdty;
	FILE_INFO_T *finfo;
	QUECTEL_FILE_TYPE ftype;
	//struct file *fp=NULL;
	
	quectel_req_init(req->buf,cbuf,BUFER_SIZE);
	fsize = 0;
        sscanf(cbuf,"%s%s%s%lld",fname,fname,tmpbuf,&fsize);
	if(!strcmp(fname,RAM_PRE) || !strcmp(fname,"?")) return false;
	if(!strcmp(fname,SD_PRE)) ftype = Q_TYPE_SD;
        else
	{
              ftype = Q_TYPE_UFS;
	sscanf(cbuf,"%s%s%lld",fname,tmpbuf,&fsize);
	}
	if(!quectel_qcmd_show(ftype,port,req))
                return quectel_send_usb_urc(port,ErrBuf);

	if(!strcmp(tmpbuf,"*"))
	{
		#if 0	
		fp = quectel_fopen(UFS_TMP_FILE,O_CREAT | O_RDWR,0);
		list_for_each(tmp,&fp->f_dentry->d_u.d_child) {
                fdty=list_entry(tmp,struct dentry,d_u.d_child);
		//printk("%s:%d\n",__func__,virt_addr_valid(fdty->d_inode));//tommy tem
                IF_DENTRY_VALID(fdty)
		{
			strcpy(fname,UFS_PATH_PRE);
                	strcat(fname,fdty->d_iname);
                	ufs_file.used -= fdty->d_inode->i_size;
			ufs_file.num--;
			quectel_fdel(fname);
			printk("del num:%d\n",++i);//tommy tem
		}
        	}
		quectel_fclose(fp);
                quectel_fdel(UFS_TMP_FILE);
		#else
		list_for_each_safe(p,tmp,&qfile_head->fh)
		{	
			finfo=list_entry(p,FILE_INFO_T,list);
			if(Q_TYPE_SD == ftype) strcpy(fname,SD_PATH_PRE);
			else
				strcpy(fname,UFS_PATH_PRE);
                        strcat(fname,finfo->name);
                        qfile_head->used -= finfo->len;
                        qfile_head->num--;
                        quectel_fdel(fname);
                        //printk("del i:%d total:%d num:%d\n",++i,ufs_file.used,ufs_file.num);
			list_del(p);
			kfree(finfo);
		}
		#endif
		
                quectel_send_usb_urc(port,"OK\r\n");
                return true;
	}
	if(fsize)
	{
		quectel_send_usb_urc(port,"+CME ERROR: 400\r\n");
		return true;
	}
        if(quectel_file_exist(tmpbuf))
        {
		if(Q_TYPE_SD == ftype) strcpy(fname,SD_PATH_PRE);
		else
			strcpy(fname,UFS_PATH_PRE);
        	strcat(fname,tmpbuf);
		quectel_del_byname(tmpbuf);
		//fop = quectel_fopen(fname, O_RDWR,0);
		//ufs_file.used -= fop->f_dentry->d_inode->i_size;
		//ufs_file.num--;
		//quectel_fclose(fop);
		quectel_fdel(fname);
		quectel_send_usb_urc(port,"OK\r\n");
        }
	else
	{
	 	quectel_send_usb_urc(port,"+CME ERROR: 405\r\n");
	}
	return true;
}

bool quectel_qflds_cmd(struct gsmd_port *port,struct usb_request *req)
{
	char fname[FILE_NAME_SIZE]={0},tmpbuf[BUFER_SIZE]={0},cbuf[BUFER_SIZE];
	QUECTEL_FILE_TYPE ftype;

	quectel_req_init(req->buf,tmpbuf,BUFER_SIZE);

        sscanf(tmpbuf,"%s%s",cbuf,fname);
	if(fname[0] == 0){ /* default UFS */
		quectel_qcmd_show(Q_TYPE_UFS,port,req);
		sprintf(cbuf,"+QFLDS: %lld,%d\r\n\r\nOK\r\n",ufs_file.used,ufs_file.num);
                quectel_send_usb_urc(port,cbuf);
		return true;
	}
	if(!strncmp(fname,UFS_PRE,strlen(UFS_PRE)))
        {
		ftype = Q_TYPE_UFS;
        }
	else if(!strncmp(fname,SD_PRE,strlen(SD_PRE)))
	{
		ftype = Q_TYPE_SD;
	}
	else
	{	
		return false;
	}

	if(!quectel_qcmd_show(ftype,port,req))
                return quectel_send_usb_urc(port,ErrBuf);
        sprintf(cbuf,"+QFLDS: %lld,%lld\r\n\r\nOK\r\n",qfile_head->total - qfile_head->used,qfile_head->total);
        quectel_send_usb_urc(port,cbuf);
        return true;
}

bool quectel_qflst_cmd(struct gsmd_port *port,struct usb_request *req)
{
        char fname[FILE_NAME_SIZE],tmpbuf[BUFER_SIZE],cbuf[FILE_BUFER_LEN];
	//struct file *fp=NULL;
	bool showlist=false;
	int buflen=0;
        struct list_head *tmp;
        //struct dentry *fdty;
	FILE_INFO_T *finfo;
	QUECTEL_FILE_TYPE ftype;

	quectel_req_init(req->buf,cbuf,BUFER_SIZE);
        fsize = 0;
        sscanf(cbuf,"%s%s%s",tmpbuf,fname,tmpbuf);
	if(!strcmp(fname,RAM_PRE) || !strcmp(fname,"?")) return false;
	if(!strcmp(fname,SD_PRE)){
 		ftype = Q_TYPE_SD;
		strcpy(fname,tmpbuf);
	}
        else 
              ftype = Q_TYPE_UFS;

	if(!quectel_qcmd_show(ftype,port,req))
                return quectel_send_usb_urc(port,ErrBuf);
        if(!quectel_strncmp(fname,"*",1) || fname[0] == 0 )
        {
		showlist = true;
        }
	#if 0
        fp = quectel_fopen(UFS_TMP_FILE,O_CREAT | O_RDWR,0);
        list_for_each(tmp,&fp->f_dentry->d_u.d_child) {
        fdty=list_entry(tmp,struct dentry,d_u.d_child);
	printk("%s:%p %p %d=%d %p %s %d\n",__func__,fdty,fdty->d_inode,fdty->d_flags,fp->f_dentry->d_flags,fp->f_dentry->d_u.d_child.next,fdty->d_iname,virt_addr_valid(fdty->d_inode));//tommy tem
        IF_DENTRY_VALID(fdty){
	#else
	
	memset(cbuf,0,FILE_BUFER_LEN);
	list_for_each(tmp,&qfile_head->fh) {
	finfo = list_entry(tmp,FILE_INFO_T,list);
	#endif
		if(Q_TYPE_SD == ftype) sprintf(tmpbuf,"+QFLST:\"SD:%s\",%lld\r\n",finfo->name,finfo->len);
		else
		sprintf(tmpbuf,"+QFLST:\"%s\",%lld\r\n",finfo->name,finfo->len);

		if(showlist){
			mdelay(1);
			buflen += strlen(tmpbuf);
			if(buflen > FILE_BUFER_LEN - 2)
			{
			quectel_send_usb_urc(port,cbuf);
			memset(cbuf,0,FILE_BUFER_LEN);
			buflen = 0;
			buflen += strlen(tmpbuf);
			}
			strcat(cbuf,tmpbuf);
		}
		else{
			if(!strcmp(finfo->name,fname)){
                        	quectel_send_usb_urc(port,tmpbuf);
				break;
			}
		}
		
        }
	if(showlist) quectel_send_usb_urc(port,cbuf);
        quectel_send_usb_urc(port,"\r\nOK\r\n");
	
        //quectel_fclose(fp);
        //quectel_fdel(UFS_TMP_FILE);

        return true;
}

bool quectel_qfdwl_cmd(struct gsmd_port *port,struct usb_request *req)
{
        char *packet = req->buf;
	struct file *fp=NULL;
        char fname[FILE_NAME_SIZE],tmpbuf[FILE_BUFER_LEN],cbuf[BUFER_SIZE];
	QUECTEL_FILE_TYPE ftype;

	quectel_req_init(packet,cbuf,BUFER_SIZE);
	fsize = 0;
        sscanf(cbuf,"%s%s%s%lld",fname,tmpbuf,fname,&fsize);
	if(!strcmp(tmpbuf,RAM_PRE) || !strcmp(tmpbuf,"?")) return false;
	if(!strcmp(tmpbuf,SD_PRE)){ 
		ftype = Q_TYPE_SD;
		strcpy(tmpbuf,fname);
	}
        else 
	{
              ftype = Q_TYPE_UFS;
	      sscanf(cbuf,"%s%s%lld",fname,tmpbuf,&fsize);
	}
	if(!quectel_qcmd_show(ftype,port,req))
                return quectel_send_usb_urc(port,ErrBuf);

	if(MODEM_MODE_CMD != q_linuxmode)
                return quectel_send_usb_urc(port,ErrBuf);

        if(fsize)
                return quectel_send_usb_urc(port,"+CME ERROR: 400\r\n");

        if(quectel_file_exist(tmpbuf))
         {
		quectel_send_usb_urc(port,"CONNECT\r\n");
		fchecksum = 0;
                fodd = 0;
                foddflag = 0;
		if(Q_TYPE_SD == ftype) strcpy(fname,SD_PATH_PRE);
		else
			strcpy(fname,UFS_PATH_PRE);
                strcat(fname,tmpbuf);
                fp = quectel_fopen(fname,O_RDWR,0);
		qfdwl.fp = fp;
		qfdwl.port = port;
		schedule_work(&qfdwl.qfdwl_work);
         }
	else
	{
		quectel_send_usb_urc(port,"+CME ERROR: 405\r\n");
	}
        return true;
}

bool quectel_qfopen_cmd(struct gsmd_port *port,struct usb_request *req)
{
        char *packet = req->buf;
	unsigned int fmode=0,i=0;
	bool fexist=false;
	FILE_INFO_T *finfo;
        char fname[FILE_NAME_SIZE],tmpbuf[BUFER_SIZE],cbuf[BUFER_SIZE];
	QUECTEL_FILE_TYPE ftype;

	quectel_req_init(packet,cbuf,BUFER_SIZE);
        sscanf(cbuf,"%s%s%s%d",fname,fname,tmpbuf,&fmode);
	if(!strcmp(fname,RAM_PRE) || !strcmp(fname,"?")) return false;
	if(!strcmp(fname,SD_PRE)) ftype = Q_TYPE_SD;
        else{ 
                ftype = Q_TYPE_UFS;
		sscanf(cbuf,"%s%s%d",fname,tmpbuf,&fmode);
	}
	if(!quectel_qcmd_show(ftype,port,req))
                return quectel_send_usb_urc(port,ErrBuf);
	if(fmode >= QUECTEL_MAX){
	quectel_send_usb_urc(port,"+CME ERROR: 400\r\n");
	return true;
	}

	if(Q_TYPE_SD == ftype) strcpy(fname,SD_PATH_PRE);
	else
		strcpy(fname,UFS_PATH_PRE);
        strcat(fname,tmpbuf);

        if(quectel_file_exist(tmpbuf))
         {
		fexist = true;
         }

	for(i=0;i<FILE_NUM_MAX;i++)
	{
		if(!qfile_head->fp[i]) break;
	}

	if(i>=FILE_NUM_MAX){
	quectel_send_usb_urc(port,"+CME ERROR: 410\r\n");
	return true;
	}

	switch(fmode)
	{
		case QUECTEL_APPEND:
			qfile_head->fp[i] = quectel_fopen(fname,O_CREAT | O_RDWR | O_APPEND,0);
			if(!fexist){
			finfo = kmalloc(sizeof(FILE_INFO_T),GFP_ATOMIC);
			strcpy(finfo->name,tmpbuf);
			list_add(&finfo->list,&qfile_head->fh);
			 qfile_head->num++;
			}
			break;
		case QUECTEL_OVERWRITE:
			qfile_head->fp[i] = quectel_fopen(fname,O_CREAT | O_RDWR,0);
			if(!fexist){
			finfo = kmalloc(sizeof(FILE_INFO_T),GFP_ATOMIC);
			strcpy(finfo->name,tmpbuf);
                        list_add(&finfo->list,&qfile_head->fh);
			qfile_head->num++;
			}
			break;
		case QUECTEL_RDONLY:
			if(fexist)
			 qfile_head->fp[i] = quectel_fopen(fname,O_RDONLY,0);
			else{
			quectel_send_usb_urc(port,"+CME ERROR: 410\r\n");
			return true;
			}
		default:
			return true;
	}
	if(Q_TYPE_SD == ftype) qfile_head->fp_index = SD_FP_INDEX_BASE + i;
	else
		qfile_head->fp_index = UFS_FP_INDEX_BASE + i;
	sprintf(tmpbuf,"+QFOPEN: %d\r\n\r\nOK\r\n",qfile_head->fp_index);		
	quectel_send_usb_urc(port,tmpbuf);
        return true;
}

bool quectel_qfclose_cmd(struct gsmd_port *port,struct usb_request *req)
{
	int i = -1;
	char cbuf[BUFER_SIZE],tmpbuf[BUFER_SIZE];
	QUECTEL_FILE_TYPE ftype;

	quectel_req_init(req->buf,tmpbuf,BUFER_SIZE);
	sscanf(tmpbuf,"%s%d",cbuf,&i);

	if(i >= UFS_FP_INDEX_BASE && i < UFS_FP_INDEX_BASE + FILE_NUM_MAX)
	{
		ftype = Q_TYPE_UFS;
		i -= UFS_FP_INDEX_BASE;
	}
	else if(i >= SD_FP_INDEX_BASE && i < SD_FP_INDEX_BASE + FILE_NUM_MAX)
	{
		ftype = Q_TYPE_SD;
                i -= SD_FP_INDEX_BASE;
	}
	else
		return false;/* AP不做处理，由modem去处理该命令 */

	if(!quectel_qcmd_show(ftype,port,req))
                return quectel_send_usb_urc(port,ErrBuf);
	if(qfile_head->fp[i])
	{
		quectel_fclose(qfile_head->fp[i]);
		qfile_head->fp[i] = NULL;
		quectel_send_usb_urc(port,"\r\nOK\r\n");
	}
	else
	{
		quectel_send_usb_urc(port,"+CME ERROR: 405\r\n");
	}
	return true;	
}

bool quectel_qfread_cmd(struct gsmd_port *port,struct usb_request *req)
{
        unsigned long long len=0;
	int i = -1;
        char cbuf[BUFER_SIZE],tmpbuf[FILE_BUFER_LEN];
	QUECTEL_FILE_TYPE ftype;

	quectel_req_init(req->buf,tmpbuf,BUFER_SIZE);
        sscanf(tmpbuf,"%s%d%lld",cbuf,&i,&len);

        if(i >= UFS_FP_INDEX_BASE && i < UFS_FP_INDEX_BASE + FILE_NUM_MAX)
        {
                ftype = Q_TYPE_UFS;
                i -= UFS_FP_INDEX_BASE;
        }
        else if(i >= SD_FP_INDEX_BASE && i < SD_FP_INDEX_BASE + FILE_NUM_MAX)
        {
                ftype = Q_TYPE_SD;
                i -= SD_FP_INDEX_BASE;
        }
        else
        	return false;/* AP不做处理，由modem去处理该命令 */

	if(!quectel_qcmd_show(ftype,port,req))
                return quectel_send_usb_urc(port,ErrBuf);
	if(MODEM_MODE_CMD != q_linuxmode)
                return quectel_send_usb_urc(port,ErrBuf);
        
	if(!qfile_head->fp[i])
                return quectel_send_usb_urc(port,"+CME ERROR: 405\r\n");

	if(len ==0 || len > qfile_head->fp[i]->f_dentry->d_inode->i_size - qfile_head->fp[i]->f_pos)
	{
		len = qfile_head->fp[i]->f_dentry->d_inode->i_size - qfile_head->fp[i]->f_pos;
	}

	sprintf(tmpbuf,"CONNECT %lld\r\n",len);
	quectel_send_usb_urc(port,tmpbuf);
	qfread.fp = qfile_head->fp[i];
	qfread.size = len;
	qfread.port = port;
	schedule_work(&qfread.qfread_work);
        return true;
}

bool quectel_qfwrite_cmd(struct gsmd_port *port,struct usb_request *req)
{
        int i=-1;
        char cbuf[BUFER_SIZE],tmpbuf[BUFER_SIZE];
	QUECTEL_FILE_TYPE ftype;

	quectel_req_init(req->buf,tmpbuf,BUFER_SIZE);
        sscanf(tmpbuf,"%s%d%lld%d",cbuf,&i,&fsize,&ftime);

        if(i >= UFS_FP_INDEX_BASE && i < UFS_FP_INDEX_BASE + FILE_NUM_MAX)
        {
                ftype = Q_TYPE_UFS;
                i -= UFS_FP_INDEX_BASE;
        }
        else if(i >= SD_FP_INDEX_BASE && i < SD_FP_INDEX_BASE + FILE_NUM_MAX)
        {
                ftype = Q_TYPE_SD;
                i -= SD_FP_INDEX_BASE;
        }
        else
        	return false;/* AP不做处理，由modem去处理该命令 */

	if(!quectel_qcmd_show(ftype,port,req))
                return quectel_send_usb_urc(port,ErrBuf);
	if(MODEM_MODE_CMD != q_linuxmode)
                return quectel_send_usb_urc(port,ErrBuf);

	fop = qfile_head->fp[i];
        if(!fop)
                return quectel_send_usb_urc(port,"+CME ERROR: 405\r\n");

         if(fsize > qfile_head->total - qfile_head->used)
         {
                return quectel_send_usb_urc(port,"+CME ERROR: 423\r\n");
         }
        if(!ftime) ftime=FILE_TIMEOUT_DEFAULT;
        if(fsize)
         {
		if(port->port_num)
                       	q_linuxmode = MODEM_MODE_MODEM_DATA;
                else
                        q_linuxmode = MODEM_MODE_AT_DATA;

                quectel_send_usb_urc(port,"CONNECT\r\n");
                fopen = true;
		fchecksum = 0;
		fodd = 0;
		foddflag = 0;
                init_timer(&(file_timer));
                file_timer.function = file_timer_expire;
                file_timer.expires = jiffies + ftime*HZ;
                file_timer.data = (unsigned long)port;
                add_timer(&file_timer);
         }

        return false;
}

bool quectel_qfseek_cmd(struct gsmd_port *port,struct usb_request *req)
{
        int i = -1,position = 0;
	unsigned long long offset = 0;
        char cbuf[BUFER_SIZE],tmpbuf[BUFER_SIZE];
	QUECTEL_FILE_TYPE ftype;

	quectel_req_init(req->buf,tmpbuf,BUFER_SIZE);
        sscanf(tmpbuf,"%s%d%lld%d",cbuf,&i,&offset,&position);

        if(i >= UFS_FP_INDEX_BASE && i < UFS_FP_INDEX_BASE + FILE_NUM_MAX)
        {
                ftype = Q_TYPE_UFS;
                i -= UFS_FP_INDEX_BASE;
        }
        else if(i >= SD_FP_INDEX_BASE && i < SD_FP_INDEX_BASE + FILE_NUM_MAX)
        {
                ftype = Q_TYPE_SD;
                i -= SD_FP_INDEX_BASE;
        }
        else
        	return false;/* AP不做处理，由modem去处理该命令 */

	if(!quectel_qcmd_show(ftype,port,req))
                return quectel_send_usb_urc(port,ErrBuf);
        if(qfile_head->fp[i])
        {
                if(quectel_fseek(qfile_head->fp[i],offset,position) < 0){
			quectel_send_usb_urc(port,ErrBuf);
			return true;
		}
                quectel_send_usb_urc(port,"\r\nOK\r\n");
        }
        else
        {
                quectel_send_usb_urc(port,"+CME ERROR: 405\r\n");
        }
        return true;
}

bool quectel_qfposition_cmd(struct gsmd_port *port,struct usb_request *req)
{
        int i = -1;
        char cbuf[BUFER_SIZE],tmpbuf[BUFER_SIZE];
	 QUECTEL_FILE_TYPE ftype;

	quectel_req_init(req->buf,tmpbuf,BUFER_SIZE);
        sscanf(tmpbuf,"%s%d",cbuf,&i);

        if(i >= UFS_FP_INDEX_BASE && i < UFS_FP_INDEX_BASE + FILE_NUM_MAX)
        {
                ftype = Q_TYPE_UFS;
                i -= UFS_FP_INDEX_BASE;
        }
        else if(i >= SD_FP_INDEX_BASE && i < SD_FP_INDEX_BASE + FILE_NUM_MAX)
        {
                ftype = Q_TYPE_SD;
                i -= SD_FP_INDEX_BASE;
        }
        else
        	return false;/* AP不做处理，由modem去处理该命令 */

	if(!quectel_qcmd_show(ftype,port,req))
                return quectel_send_usb_urc(port,ErrBuf);
        if(qfile_head->fp[i])
        {
                sprintf(tmpbuf,"+QFPOSITION: %lld\r\n\r\nOK\r\n",qfile_head->fp[i]->f_pos);
                quectel_send_usb_urc(port,tmpbuf);
        }
        else
        {
                quectel_send_usb_urc(port,"+CME ERROR: 405\r\n");
        }
        return true;
}

bool quectel_parse_cmd(struct gsmd_port *port,struct usb_request *req)
{
	FILE_CMD_LIST_T *fcmd;

	fcmd = file_cmd;	
	while(fcmd->func)
	{
		if(!quectel_strncmp(req->buf,fcmd->cmd,strlen(fcmd->cmd)))
		{
			return fcmd->func(port,req);
		}
		fcmd++;
	}
	return false;
}

static void qfdwl_handle(struct work_struct *work)
{
        char tmpbuf[FILE_BUFER_LEN];
        struct quectel_fdwl *qfdwl_t = container_of(work, struct quectel_fdwl,qfdwl_work); 
	unsigned long long total=0,readlen=0;

	if(qfdwl_t->port->port_num)
                q_linuxmode = MODEM_MODE_MODEM_DATA;
        else
        q_linuxmode = MODEM_MODE_AT_DATA;
        
        do{
                if(MODEM_MODE_CMD == q_linuxmode) break;
                readlen = quectel_fread(tmpbuf,FILE_BUFER_LEN,qfdwl_t->fp);
                file_CalcCheckSum(&fchecksum,&fodd,&foddflag,tmpbuf,readlen,false);
                quectel_send(qfdwl_t->port,tmpbuf,readlen);
                total += readlen;
                //printk("dbg-total:%d\n",total);
		pet_watchdog();
                mdelay(5);
        }while(total < qfdwl_t->fp->f_dentry->d_inode->i_size);
        q_linuxmode = MODEM_MODE_CMD;
        quectel_fclose(qfdwl_t->fp);
        file_CalcCheckSum(&fchecksum,&fodd,&foddflag,NULL,0,true);
        sprintf(tmpbuf,"\r\n+QFDWL: %lld,%x\r\n\r\nOK\r\n",total,fchecksum);
        quectel_send_usb_urc(qfdwl_t->port,tmpbuf);
        return;
}

static void qfread_handle(struct work_struct *work)
{
	char tmpbuf[FILE_BUFER_LEN];
        struct quectel_frd *qfread_t = container_of(work, struct quectel_frd,qfread_work);
	unsigned int len = qfread_t->size;
	unsigned int total=0,readlen=0;

	if(qfread_t->port->port_num)
                q_linuxmode = MODEM_MODE_MODEM_DATA;
        else
        	q_linuxmode = MODEM_MODE_AT_DATA;

	total=len;        
        do{
		if(MODEM_MODE_CMD == q_linuxmode) break;

                len = total > FILE_BUFER_LEN ? FILE_BUFER_LEN:total;
                readlen = quectel_fread(tmpbuf,len,qfread_t->fp);
                quectel_send(qfread_t->port,tmpbuf,readlen);
                total -= readlen;
		pet_watchdog();
                mdelay(5);
        }while(len);
	q_linuxmode = MODEM_MODE_CMD;
        quectel_send_usb_urc(qfread_t->port,"\r\nOK\r\n");
	return;
}
#endif

static void gsmd_rx_push(struct work_struct *w)
{
	struct gsmd_port *port = container_of(w, struct gsmd_port, push);
	struct smd_port_info *pi = port->pi;
	struct list_head *q;
#ifdef CONFIG_QUECTEL_LINUX_FILE
	int fcount=0;
	char tmpbuf[BUFER_SIZE];
#endif
#ifdef CONFIG_QUECTEL_FLOWCTRL
        static int flow_cnt = 0;
#endif
#ifdef CONFIG_QUECTEL_ESCAPE_FEATURE
	unsigned long flags = 0;
#endif

	pr_debug("%s: port:%p port#%d", __func__, port, port->port_num);

	spin_lock_irq(&port->port_lock);

	q = &port->read_queue;
	while (pi->ch && !list_empty(q)) {
		struct usb_request *req;
		int avail;

		req = list_first_entry(q, struct usb_request, list);

		switch (req->status) {
		case -ESHUTDOWN:
			pr_debug("%s: req status shutdown portno#%d port:%p\n",
					__func__, port->port_num, port);
			goto rx_push_end;
		default:
			pr_warning("%s: port:%p port#%d"
					" Unexpected Rx Status:%d\n", __func__,
					port, port->port_num, req->status);
		case 0:
			/* normal completion */
			break;
		}

		avail = smd_write_avail(pi->ch);
		if (!avail)
			goto rx_push_end;

		if (req->actual) {
			char		*packet = req->buf;
			unsigned	size = req->actual;
			unsigned	n;
			int		count;

			n = port->n_read;
			if (n) {
				packet += n;
				size -= n;
			}

			//count = smd_write(pi->ch, packet, size);
			#ifdef CONFIG_QUECTEL_ESCAPE_FEATURE
			if(port->port_num)
				q_current_mode = MODEM_MODE_MODEM_DATA;
			else
				q_current_mode = MODEM_MODE_AT_DATA;

			if(((MODEM_MODE_CMD != q_modemmode && q_modemmode == q_current_mode) \
			    || (MODEM_MODE_CMD != q_linuxmode && q_linuxmode == q_current_mode)) && \
                            (size == 3) && packet[0] =='+' && packet[1] =='+' && packet[2] =='+')
			{
				printk("+++ detected\n");
				if(MODEM_MODE_CMD != q_linuxmode && q_linuxmode == q_current_mode){
					 q_linuxmode = MODEM_MODE_CMD;
					q_current_mode = MODEM_MODE_CMD;
				}
				else{
				quectel_escape_addr = smem_find(SMEM_ID_VENDOR0,sizeof(QUECTEL_ESCAPE_T));
				if (spinlocks_initialized)
					remote_spin_lock_irqsave(&remote_spinlock, flags);
				if(quectel_escape_addr){
					quectel_escape_addr->escape = QUECTEL_ESCAPE_DETECTED;
				}
				if (spinlocks_initialized)
					remote_spin_unlock_irqrestore(&remote_spinlock, flags);

				q_modemmode = MODEM_MODE_CMD;
				port->nbytes_tomodem += size;
                                port->n_read = 0;
                                list_move(&req->list, &port->read_pool);
                                goto rx_push_end;
				}
			}
			
			#endif
			#ifdef CONFIG_QUECTEL_LINUX_FILE
			if(fopen && q_linuxmode == q_current_mode)
                        {
				if(q_linuxmode == MODEM_MODE_CMD)
				{
					file_CalcCheckSum(&fchecksum,&fodd,&foddflag,NULL,0,true);
					if(fack && facknum > 1) quectel_send_usb_urc(port,"\r\n");
					if(fupl) 
						sprintf(tmpbuf,"+QFUPL: %lld,%x\r\n\r\nOK\r\n",ftotal,fchecksum);
					else	
						sprintf(tmpbuf,"+QFWRITE: %lld,%lld\r\n\r\nOK\r\n",ftotal,fop->f_dentry->d_inode->i_size);
					exit_data_mode(port);
                                        quectel_send_usb_urc(port,tmpbuf);
					port->nbytes_tomodem += size;
                                        port->n_read = 0;
                                        list_move(&req->list, &port->read_pool);
                                        goto rx_push_end;
				}
				spin_unlock_irq(&port->port_lock);
                          	fcount=quectel_fwrite(packet,size,fop);
				spin_lock_irq(&port->port_lock);
				file_CalcCheckSum(&fchecksum,&fodd,&foddflag,packet,fcount,false);
				ftotal += fcount;
				if(fack &&  facknum==  (ftotal / QACK_VALUE_LIMIT)){
					facknum++;
					quectel_send_usb_urc(port,"A");	
				}
				//file_timer.data = (unsigned long)port;
				mod_timer(&file_timer,jiffies + ftime*HZ);
                        }
			else
			{
				fcount = size;
				spin_unlock_irq(&port->port_lock);
				if(!q_modemmode && quectel_parse_cmd(port,req)){ 
					spin_lock_irq(&port->port_lock);
					port->nbytes_tomodem += fcount;
        				port->n_read = 0;
        				list_move(&req->list, &port->read_pool);
					goto rx_push_end;
				}
				spin_lock_irq(&port->port_lock);
			}

			if(fopen && q_linuxmode == q_current_mode && ftotal && ftotal >= fsize)
			{
				file_CalcCheckSum(&fchecksum,&fodd,&foddflag,NULL,0,true);
				if(fack && facknum > 1) quectel_send_usb_urc(port,"\r\n");
				if(fupl)
                                	sprintf(tmpbuf,"+QFUPL: %lld,%x\r\n\r\nOK\r\n",ftotal,fchecksum);
                                else
                                	sprintf(tmpbuf,"+QFWRITE: %lld,%lld\r\n\r\nOK\r\n",ftotal,fop->f_dentry->d_inode->i_size);
				if(ftotal == fsize){
					//file_CalcCheckSum(&fchecksum,&fodd,&foddflag,NULL,0,true);
					//sprintf(tmpbuf,"+QFUPL: %d,%x\r\n\r\nOK\r\n",ftotal,fchecksum);
					quectel_send_usb_urc(port,tmpbuf);
				}		
				else
					quectel_send_usb_urc(port,"+CME ERROR: 409\r\n");
				exit_data_mode(port);
				port->nbytes_tomodem += fcount;
        			port->n_read = 0;
        			list_move(&req->list, &port->read_pool);
				goto rx_push_end;
			}
			
			if(!fopen || q_linuxmode != q_current_mode)
			{
#ifdef CONFIG_QUECTEL_FLOWCTRL
                        if(q_flowctrl){
                                flow_cnt++;
                                mdelay(50);
                        }
                        else
                                flow_cnt = 0;

                        if(flow_cnt > 100){
                                flow_cnt = 0;
                                q_flowctrl = 0;
                        }
#endif
				count = smd_write(pi->ch, packet, size);
			}
			else
			{
				count = fcount;
			}
			#else
			count = smd_write(pi->ch, packet, size);
			#endif
			if (count < 0) {
				pr_err("%s: smd write failed err:%d\n",
						__func__, count);
				goto rx_push_end;
			}

			if (count != size) {
				port->n_read += count;
				goto rx_push_end;
			}

			port->nbytes_tomodem += count;
		}

		port->n_read = 0;
		list_move(&req->list, &port->read_pool);
	}

rx_push_end:
	spin_unlock_irq(&port->port_lock);

	gsmd_start_rx(port);
}

static void gsmd_read_pending(struct gsmd_port *port)
{
	int avail;

	if (!port || !port->pi->ch)
		return;

	/* passing null buffer discards the data */
	while ((avail = smd_read_avail(port->pi->ch)))
		smd_read(port->pi->ch, 0, avail);

	return;
}

static void gsmd_tx_pull(struct work_struct *w)
{
	struct gsmd_port *port = container_of(w, struct gsmd_port, pull);
	struct list_head *pool = &port->write_pool;
	struct smd_port_info *pi = port->pi;
	struct usb_ep *in;

	pr_debug("%s: port:%p port#%d pool:%p\n", __func__,
			port, port->port_num, pool);

	spin_lock_irq(&port->port_lock);

	if (!port->port_usb) {
		pr_debug("%s: usb is disconnected\n", __func__);
		spin_unlock_irq(&port->port_lock);
		gsmd_read_pending(port);
		return;
	}

	in = port->port_usb->in;
	while (pi->ch && !list_empty(pool)) {
		struct usb_request *req;
		int avail;
		int ret;

		avail = smd_read_avail(pi->ch);
		if (!avail)
			break;

		avail = avail > SMD_TX_BUF_SIZE ? SMD_TX_BUF_SIZE : avail;

		req = list_entry(pool->next, struct usb_request, list);
		list_del(&req->list);
		req->length = smd_read(pi->ch, req->buf, avail);
		spin_unlock_irq(&port->port_lock);
		ret = usb_ep_queue(in, req, GFP_KERNEL);
		spin_lock_irq(&port->port_lock);
		if (ret) {
			pr_err("%s: usb ep out queue failed"
					"port:%p, port#%d err:%d\n",
					__func__, port, port->port_num, ret);
			/* could be usb disconnected */
			if (!port->port_usb)
				gsmd_free_req(in, req);
			else
				list_add(&req->list, pool);
			goto tx_pull_end;
		}

		port->nbytes_tolaptop += req->length;
	}

tx_pull_end:
	/* TBD: Check how code behaves on USB bus suspend */
	if (port->port_usb && smd_read_avail(port->pi->ch) && !list_empty(pool))
		queue_work(gsmd_wq, &port->pull);

	spin_unlock_irq(&port->port_lock);

	return;
}

static void gsmd_read_complete(struct usb_ep *ep, struct usb_request *req)
{
	struct gsmd_port *port = ep->driver_data;

	pr_debug("%s: ep:%p port:%p\n", __func__, ep, port);

	if (!port) {
		pr_err("%s: port is null\n", __func__);
		return;
	}

	spin_lock(&port->port_lock);
	if (!test_bit(CH_OPENED, &port->pi->flags) ||
			req->status == -ESHUTDOWN) {
		spin_unlock(&port->port_lock);
		gsmd_free_req(ep, req);
		return;
	}

	list_add_tail(&req->list, &port->read_queue);
	queue_work(gsmd_wq, &port->push);
	spin_unlock(&port->port_lock);

	return;
}

static void gsmd_write_complete(struct usb_ep *ep, struct usb_request *req)
{
	struct gsmd_port *port = ep->driver_data;

	pr_debug("%s: ep:%p port:%p\n", __func__, ep, port);

	if (!port) {
		pr_err("%s: port is null\n", __func__);
		return;
	}

	spin_lock(&port->port_lock);
	if (!test_bit(CH_OPENED, &port->pi->flags) ||
			req->status == -ESHUTDOWN) {
		spin_unlock(&port->port_lock);
		gsmd_free_req(ep, req);
		return;
	}

	if (req->status)
		pr_warning("%s: port:%p port#%d unexpected %s status %d\n",
				__func__, port, port->port_num,
				ep->name, req->status);

	list_add(&req->list, &port->write_pool);
	queue_work(gsmd_wq, &port->pull);
	spin_unlock(&port->port_lock);

	return;
}

static void gsmd_start_io(struct gsmd_port *port)
{
	int		ret = -ENODEV;

	pr_debug("%s: port: %p\n", __func__, port);

	spin_lock(&port->port_lock);

	if (!port->port_usb)
		goto start_io_out;

	smd_tiocmset_from_cb(port->pi->ch,
			port->cbits_to_modem,
			~port->cbits_to_modem);

	ret = gsmd_alloc_requests(port->port_usb->out,
				&port->read_pool,
				SMD_RX_QUEUE_SIZE, SMD_RX_BUF_SIZE,
				gsmd_read_complete);
	if (ret) {
		pr_err("%s: unable to allocate out requests\n",
				__func__);
		goto start_io_out;
	}

	ret = gsmd_alloc_requests(port->port_usb->in,
				&port->write_pool,
				SMD_TX_QUEUE_SIZE, SMD_TX_BUF_SIZE,
				gsmd_write_complete);
	if (ret) {
		gsmd_free_requests(port->port_usb->out, &port->read_pool);
		pr_err("%s: unable to allocate IN requests\n",
				__func__);
		goto start_io_out;
	}

#ifdef CONFIG_QUECTEL_LINUX_FILE
	ret = gsmd_alloc_requests(port->port_usb->in,
                                &usb_send_pool,
                                SMD_TX_QUEUE_SIZE, FILE_BUFER_LEN,
                                usb_send_complete);
        if (ret) {
                gsmd_free_requests(port->port_usb->in, &port->write_pool);
		gsmd_free_requests(port->port_usb->out, &port->read_pool);
                pr_err("%s: unable to allocate IN requests\n",
                                __func__);
                goto start_io_out;
        }
#endif

start_io_out:
	spin_unlock(&port->port_lock);

	if (ret)
		return;

	gsmd_start_rx(port);
}

static unsigned int convert_uart_sigs_to_acm(unsigned uart_sig)
{
	unsigned int acm_sig = 0;

	/* should this needs to be in calling functions ??? */
	uart_sig &= (TIOCM_RI | TIOCM_CD | TIOCM_DSR);

	if (uart_sig & TIOCM_RI)
		acm_sig |= SMD_ACM_CTRL_RI;
	if (uart_sig & TIOCM_CD)
		acm_sig |= SMD_ACM_CTRL_DCD;
	if (uart_sig & TIOCM_DSR)
		acm_sig |= SMD_ACM_CTRL_DSR;

	return acm_sig;
}

static unsigned int convert_acm_sigs_to_uart(unsigned acm_sig)
{
	unsigned int uart_sig = 0;

	/* should this needs to be in calling functions ??? */
	acm_sig &= (SMD_ACM_CTRL_DTR | SMD_ACM_CTRL_RTS);

	if (acm_sig & SMD_ACM_CTRL_DTR)
		uart_sig |= TIOCM_DTR;
	if (acm_sig & SMD_ACM_CTRL_RTS)
		uart_sig |= TIOCM_RTS;

	return uart_sig;
}


static void gsmd_stop_io(struct gsmd_port *port)
{
	struct usb_ep	*in;
	struct usb_ep	*out;
	unsigned long	flags;

	spin_lock_irqsave(&port->port_lock, flags);
	if (!port->port_usb) {
		spin_unlock_irqrestore(&port->port_lock, flags);
		return;
	}
	in = port->port_usb->in;
	out = port->port_usb->out;
	spin_unlock_irqrestore(&port->port_lock, flags);

	usb_ep_fifo_flush(in);
	usb_ep_fifo_flush(out);

	spin_lock(&port->port_lock);
	if (port->port_usb) {
		gsmd_free_requests(out, &port->read_pool);
		gsmd_free_requests(out, &port->read_queue);
		gsmd_free_requests(in, &port->write_pool);
#ifdef CONFIG_QUECTEL_LINUX_FILE
		gsmd_free_requests(in, &usb_send_pool);
#endif
		port->n_read = 0;
		port->cbits_to_laptop = 0;
	}

	if (port->port_usb->send_modem_ctrl_bits)
		port->port_usb->send_modem_ctrl_bits(
					port->port_usb,
					port->cbits_to_laptop);
	spin_unlock(&port->port_lock);

}

static void gsmd_notify(void *priv, unsigned event)
{
	struct gsmd_port *port = priv;
	struct smd_port_info *pi = port->pi;
	int i;

	switch (event) {
	case SMD_EVENT_DATA:
		pr_debug("%s: Event data\n", __func__);
		if (smd_read_avail(pi->ch))
			queue_work(gsmd_wq, &port->pull);
		if (smd_write_avail(pi->ch))
		{
			queue_work(gsmd_wq, &port->push);
		}
		break;
	case SMD_EVENT_OPEN:
		pr_debug("%s: Event Open\n", __func__);
		set_bit(CH_OPENED, &pi->flags);
		gsmd_start_io(port);
		break;
	case SMD_EVENT_CLOSE:
		pr_debug("%s: Event Close\n", __func__);
		clear_bit(CH_OPENED, &pi->flags);
		gsmd_stop_io(port);
		break;
	case SMD_EVENT_STATUS:
		i = smd_tiocmget(port->pi->ch);
		port->cbits_to_laptop = convert_uart_sigs_to_acm(i);
		if (port->port_usb && port->port_usb->send_modem_ctrl_bits)
			port->port_usb->send_modem_ctrl_bits(port->port_usb,
						port->cbits_to_laptop);
		break;
	}
}

static void gsmd_connect_work(struct work_struct *w)
{
	struct gsmd_port *port;
	struct smd_port_info *pi;
	int ret;

	port = container_of(w, struct gsmd_port, connect_work.work);
	pi = port->pi;

	pr_debug("%s: port:%p port#%d\n", __func__, port, port->port_num);

	if (!test_bit(CH_READY, &pi->flags))
		return;

	ret = smd_named_open_on_edge(pi->name, SMD_APPS_MODEM,
				&pi->ch, port, gsmd_notify);
	if (ret) {
		if (ret == -EAGAIN) {
			/* port not ready  - retry */
			pr_debug("%s: SMD port not ready - rescheduling:%s err:%d\n",
					__func__, pi->name, ret);
			queue_delayed_work(gsmd_wq, &port->connect_work,
				msecs_to_jiffies(250));
		} else {
			pr_err("%s: unable to open smd port:%s err:%d\n",
					__func__, pi->name, ret);
		}
	}
}

static void gsmd_notify_modem(void *gptr, u8 portno, int ctrl_bits)
{
	struct gsmd_port *port;
	int temp;
	struct gserial *gser = gptr;

	if (portno >= n_smd_ports) {
		pr_err("%s: invalid portno#%d\n", __func__, portno);
		return;
	}

	if (!gser) {
		pr_err("%s: gser is null\n", __func__);
		return;
	}

	port = smd_ports[portno].port;

	temp = convert_acm_sigs_to_uart(ctrl_bits);

	if (temp == port->cbits_to_modem)
		return;

	port->cbits_to_modem = temp;

	/* usb could send control signal before smd is ready */
	if (!test_bit(CH_OPENED, &port->pi->flags))
		return;

	/* if DTR is high, update latest modem info to laptop */
	if (port->cbits_to_modem & TIOCM_DTR) {
		unsigned i;

		i = smd_tiocmget(port->pi->ch);
		port->cbits_to_laptop = convert_uart_sigs_to_acm(i);

		if (gser->send_modem_ctrl_bits)
			gser->send_modem_ctrl_bits(
					port->port_usb,
					port->cbits_to_laptop);
	}

	smd_tiocmset(port->pi->ch,
			port->cbits_to_modem,
			~port->cbits_to_modem);
}

int gsmd_connect(struct gserial *gser, u8 portno)
{
	unsigned long flags;
	int ret;
	struct gsmd_port *port;

	pr_debug("%s: gserial:%p portno:%u\n", __func__, gser, portno);

	if (portno >= n_smd_ports) {
		pr_err("%s: Invalid port no#%d", __func__, portno);
		return -EINVAL;
	}

	if (!gser) {
		pr_err("%s: gser is null\n", __func__);
		return -EINVAL;
	}

	port = smd_ports[portno].port;

	spin_lock_irqsave(&port->port_lock, flags);
	port->port_usb = gser;
	gser->notify_modem = gsmd_notify_modem;
	port->nbytes_tomodem = 0;
	port->nbytes_tolaptop = 0;
	spin_unlock_irqrestore(&port->port_lock, flags);

	ret = usb_ep_enable(gser->in);
	if (ret) {
		pr_err("%s: usb_ep_enable failed eptype:IN ep:%p",
				__func__, gser->in);
		port->port_usb = 0;
		return ret;
	}
	gser->in->driver_data = port;

	ret = usb_ep_enable(gser->out);
	if (ret) {
		pr_err("%s: usb_ep_enable failed eptype:OUT ep:%p",
				__func__, gser->out);
		port->port_usb = 0;
		gser->in->driver_data = 0;
		return ret;
	}
	gser->out->driver_data = port;

	queue_delayed_work(gsmd_wq, &port->connect_work, msecs_to_jiffies(0));

	return 0;
}

void gsmd_disconnect(struct gserial *gser, u8 portno)
{
	unsigned long flags;
	struct gsmd_port *port;

	pr_debug("%s: gserial:%p portno:%u\n", __func__, gser, portno);

	if (portno >= n_smd_ports) {
		pr_err("%s: invalid portno#%d\n", __func__, portno);
		return;
	}

	if (!gser) {
		pr_err("%s: gser is null\n", __func__);
		return;
	}

	port = smd_ports[portno].port;

	spin_lock_irqsave(&port->port_lock, flags);
	port->port_usb = 0;
	spin_unlock_irqrestore(&port->port_lock, flags);

	/* disable endpoints, aborting down any active I/O */
	usb_ep_disable(gser->out);
	usb_ep_disable(gser->in);

	spin_lock_irqsave(&port->port_lock, flags);
	gsmd_free_requests(gser->out, &port->read_pool);
	gsmd_free_requests(gser->out, &port->read_queue);
	gsmd_free_requests(gser->in, &port->write_pool);
#ifdef CONFIG_QUECTEL_LINUX_FILE
        gsmd_free_requests(gser->in, &usb_send_pool);
#endif
	port->n_read = 0;
	spin_unlock_irqrestore(&port->port_lock, flags);

	if (test_and_clear_bit(CH_OPENED, &port->pi->flags)) {
		/* lower the dtr */
		port->cbits_to_modem = 0;
		smd_tiocmset(port->pi->ch,
				port->cbits_to_modem,
				~port->cbits_to_modem);
	}

	if (port->pi->ch) {
		smd_close(port->pi->ch);
		port->pi->ch = NULL;
	}
}

#define SMD_CH_MAX_LEN	20
static int gsmd_ch_probe(struct platform_device *pdev)
{
	struct gsmd_port *port;
	struct smd_port_info *pi;
	int i;
	unsigned long flags;

	pr_debug("%s: name:%s\n", __func__, pdev->name);

	for (i = 0; i < n_smd_ports; i++) {
		port = smd_ports[i].port;
		pi = port->pi;
		if (!strncmp(pi->name, pdev->name, SMD_CH_MAX_LEN)) {
			set_bit(CH_READY, &pi->flags);
			spin_lock_irqsave(&port->port_lock, flags);
			if (port->port_usb)
				queue_delayed_work(gsmd_wq, &port->connect_work,
					msecs_to_jiffies(0));
			spin_unlock_irqrestore(&port->port_lock, flags);
			break;
		}
	}
	return 0;
}

static int gsmd_ch_remove(struct platform_device *pdev)
{
	struct gsmd_port *port;
	struct smd_port_info *pi;
	int i;

	pr_debug("%s: name:%s\n", __func__, pdev->name);

	for (i = 0; i < n_smd_ports; i++) {
		port = smd_ports[i].port;
		pi = port->pi;

		if (!strncmp(pi->name, pdev->name, SMD_CH_MAX_LEN)) {
			clear_bit(CH_READY, &pi->flags);
			clear_bit(CH_OPENED, &pi->flags);
			if (pi->ch) {
				smd_close(pi->ch);
				pi->ch = NULL;
			}
			break;
		}
	}
	return 0;
}

static void gsmd_port_free(int portno)
{
	struct gsmd_port *port = smd_ports[portno].port;

	if (!port)
		kfree(port);
}

static int gsmd_port_alloc(int portno, struct usb_cdc_line_coding *coding)
{
	struct gsmd_port *port;
	struct platform_driver *pdrv;

	port = kzalloc(sizeof(struct gsmd_port), GFP_KERNEL);
	if (!port)
		return -ENOMEM;

	port->port_num = portno;
	port->pi = &smd_pi[portno];

	spin_lock_init(&port->port_lock);

	INIT_LIST_HEAD(&port->read_pool);
	INIT_LIST_HEAD(&port->read_queue);
	INIT_WORK(&port->push, gsmd_rx_push);

	INIT_LIST_HEAD(&port->write_pool);
	INIT_WORK(&port->pull, gsmd_tx_pull);

	INIT_DELAYED_WORK(&port->connect_work, gsmd_connect_work);

#ifdef CONFIG_QUECTEL_LINUX_FILE
	INIT_WORK(&qfdwl.qfdwl_work, qfdwl_handle);
	INIT_WORK(&qfread.qfread_work, qfread_handle);
	
	INIT_LIST_HEAD(&usb_send_pool);
	INIT_LIST_HEAD(&ufs_file.fh);
	INIT_LIST_HEAD(&sd_file.fh);
#endif
	smd_ports[portno].port = port;
	pdrv = &smd_ports[portno].pdrv;
	pdrv->probe = gsmd_ch_probe;
	pdrv->remove = gsmd_ch_remove;
	pdrv->driver.name = port->pi->name;
	pdrv->driver.owner = THIS_MODULE;
	platform_driver_register(pdrv);

	pr_debug("%s: port:%p portno:%d\n", __func__, port, portno);

	return 0;
}

#if defined(CONFIG_DEBUG_FS)
static ssize_t debug_smd_read_stats(struct file *file, char __user *ubuf,
		size_t count, loff_t *ppos)
{
	struct gsmd_port *port;
	struct smd_port_info *pi;
	char *buf;
	unsigned long flags;
	int temp = 0;
	int i;
	int ret;

	buf = kzalloc(sizeof(char) * 512, GFP_KERNEL);
	if (!buf)
		return -ENOMEM;

	for (i = 0; i < n_smd_ports; i++) {
		port = smd_ports[i].port;
		pi = port->pi;
		spin_lock_irqsave(&port->port_lock, flags);
		temp += scnprintf(buf + temp, 512 - temp,
				"###PORT:%d###\n"
				"nbytes_tolaptop: %lu\n"
				"nbytes_tomodem:  %lu\n"
				"cbits_to_modem:  %u\n"
				"cbits_to_laptop: %u\n"
				"n_read: %u\n"
				"smd_read_avail: %d\n"
				"smd_write_avail: %d\n"
				"CH_OPENED: %d\n"
				"CH_READY: %d\n",
				i, port->nbytes_tolaptop, port->nbytes_tomodem,
				port->cbits_to_modem, port->cbits_to_laptop,
				port->n_read,
				pi->ch ? smd_read_avail(pi->ch) : 0,
				pi->ch ? smd_write_avail(pi->ch) : 0,
				test_bit(CH_OPENED, &pi->flags),
				test_bit(CH_READY, &pi->flags));
		spin_unlock_irqrestore(&port->port_lock, flags);
	}

	ret = simple_read_from_buffer(ubuf, count, ppos, buf, temp);

	kfree(buf);

	return ret;

}

static ssize_t debug_smd_reset_stats(struct file *file, const char __user *buf,
				 size_t count, loff_t *ppos)
{
	struct gsmd_port *port;
	unsigned long flags;
	int i;

	for (i = 0; i < n_smd_ports; i++) {
		port = smd_ports[i].port;

		spin_lock_irqsave(&port->port_lock, flags);
		port->nbytes_tolaptop = 0;
		port->nbytes_tomodem = 0;
		spin_unlock_irqrestore(&port->port_lock, flags);
	}

	return count;
}

static int debug_smd_open(struct inode *inode, struct file *file)
{
	return 0;
}

static const struct file_operations debug_gsmd_ops = {
	.open = debug_smd_open,
	.read = debug_smd_read_stats,
	.write = debug_smd_reset_stats,
};

static void gsmd_debugfs_init(void)
{
	struct dentry *dent;

	dent = debugfs_create_dir("usb_gsmd", 0);
	if (IS_ERR(dent))
		return;

	debugfs_create_file("status", 0444, dent, 0, &debug_gsmd_ops);
}
#else
static void gsmd_debugfs_init(void) {}
#endif

int gsmd_setup(struct usb_gadget *g, unsigned count)
{
	struct usb_cdc_line_coding	coding;
	int ret;
	int i;

	pr_debug("%s: g:%p count: %d\n", __func__, g, count);

	if (!count || count > SMD_N_PORTS) {
		pr_err("%s: Invalid num of ports count:%d gadget:%p\n",
				__func__, count, g);
		return -EINVAL;
	}

	coding.dwDTERate = cpu_to_le32(9600);
	coding.bCharFormat = 8;
	coding.bParityType = USB_CDC_NO_PARITY;
	coding.bDataBits = USB_CDC_1_STOP_BITS;

	gsmd_wq = create_singlethread_workqueue("k_gsmd");
	if (!gsmd_wq) {
		pr_err("%s: Unable to create workqueue gsmd_wq\n",
				__func__);
		return -ENOMEM;
	}

	for (i = 0; i < count; i++) {
		mutex_init(&smd_ports[i].lock);
		n_smd_ports++;
		ret = gsmd_port_alloc(i, &coding);
		if (ret) {
			n_smd_ports--;
			pr_err("%s: Unable to alloc port:%d\n", __func__, i);
			goto free_smd_ports;
		}
	}
#ifdef CONFIG_QUECTEL_ESCAPE_FEATURE
	i = remote_spin_lock_init(&remote_spinlock, SMEM_SPINLOCK_SMEM_ALLOC);
	if (i) {
		pr_err("%s: remote spinlock init failed %d\n", __func__, i);
	}
	else
		spinlocks_initialized = 1;
#endif

	gsmd_debugfs_init();

	return 0;
free_smd_ports:
	for (i = 0; i < n_smd_ports; i++)
		gsmd_port_free(i);

	destroy_workqueue(gsmd_wq);

	return ret;
}

void gsmd_cleanup(struct usb_gadget *g, unsigned count)
{
	/* TBD */
}
