/* Copyright (c) 2009-2012, The Linux Foundation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/time.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/spinlock.h>
#include <linux/hrtimer.h>
#include <linux/clk.h>
#include <linux/io.h>
#include <linux/debugfs.h>
#include <linux/semaphore.h>
#include <linux/uaccess.h>
#include <linux/msm_mdp.h>
#include <asm/system.h>
#include <asm/mach-types.h>
#include <mach/hardware.h>
#include <mach/iommu_domains.h>
#include "mdp.h"
#include "msm_fb.h"
#include "mdp4.h"

struct mdp4_statistic mdp4_stat;

unsigned is_mdp4_hw_reset(void)
{
	unsigned hw_reset = 0;

	/* Only revisions > v2.1 may be reset or powered off/on at runtime */
	if (mdp_hw_revision > MDP4_REVISION_V2_1) {
		mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);
		hw_reset = !inpdw(MDP_BASE + 0x003c);
		mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
	}

	return hw_reset;
}

void mdp4_sw_reset(ulong bits)
{
	/* MDP cmd block enable */
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);

	bits &= 0x1f;	/* 5 bits */
	outpdw(MDP_BASE + 0x001c, bits);	/* MDP_SW_RESET */

	while (inpdw(MDP_BASE + 0x001c) & bits) /* self clear when complete */
		;
	/* MDP cmd block disable */
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);

	MSM_FB_DEBUG("mdp4_sw_reset: 0x%x\n", (int)bits);
}

void mdp4_overlay_cfg(int overlayer, int blt_mode, int refresh, int direct_out)
{
	ulong bits = 0;

	if (blt_mode)
		bits |= (1 << 3);
	refresh &= 0x03;	/* 2 bites */
	bits |= (refresh << 1);
	direct_out &= 0x01;
	bits |= direct_out;
	/* MDP cmd block enable */
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);


	if (overlayer == MDP4_MIXER0)
		outpdw(MDP_BASE + 0x10004, bits); /* MDP_OVERLAY0_CFG */
	else if (overlayer == MDP4_MIXER1)
		outpdw(MDP_BASE + 0x18004, bits); /* MDP_OVERLAY1_CFG */

	MSM_FB_DEBUG("mdp4_overlay_cfg: 0x%x\n",
		(int)inpdw(MDP_BASE + 0x10004));
	/* MDP cmd block disable */
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
}

void mdp4_display_intf_sel(int output, ulong intf)
{
	ulong bits, mask, data;
	/* MDP cmd block enable */
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);

	bits = inpdw(MDP_BASE + 0x0038);	/* MDP_DISP_INTF_SEL */

	if (intf == DSI_VIDEO_INTF) {
		data = 0x40;	/* bit 6 */
		intf = MDDI_LCDC_INTF;
		if (output == SECONDARY_INTF_SEL) {
			MSM_FB_INFO("%s: Illegal INTF selected, output=%d \
				intf=%d\n", __func__, output, (int)intf);
		}
	} else if (intf == DSI_CMD_INTF) {
		data = 0x80;	/* bit 7 */
		intf = MDDI_INTF;
		if (output == EXTERNAL_INTF_SEL) {
			MSM_FB_INFO("%s: Illegal INTF selected, output=%d \
				intf=%d\n", __func__, output, (int)intf);
		}
	} else
		data = 0;

	mask = 0x03;	/* 2 bits */
	intf &= 0x03;	/* 2 bits */

	switch (output) {
	case EXTERNAL_INTF_SEL:
		intf <<= 4;
		mask <<= 4;
		break;
	case SECONDARY_INTF_SEL:
		intf &= 0x02;	/* only MDDI and EBI2 support */
		intf <<= 2;
		mask <<= 2;
		break;
	default:
		break;
	}

	intf |= data;
	mask |= data;

	bits &= ~mask;
	bits |= intf;

	outpdw(MDP_BASE + 0x0038, bits);	/* MDP_DISP_INTF_SEL */
	/* MDP cmd block disable */
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);

  MSM_FB_DEBUG("mdp4_display_intf_sel: 0x%x\n", (int)inpdw(MDP_BASE + 0x0038));
}

unsigned long mdp4_display_status(void)
{
	ulong status;
	/* MDP cmd block enable */
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);

	status = inpdw(MDP_BASE + 0x0018) & 0x3ff;	/* MDP_DISPLAY_STATUS */

	/* MDP cmd block disable */
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
	return status;
}

void mdp4_ebi2_lcd_setup(int lcd, ulong base, int ystride)
{
	/* always use memory map */
	ystride &= 0x01fff;	/* 13 bits */
	/* MDP cmd block enable */
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);

	if (lcd == EBI2_LCD0) {
		outpdw(MDP_BASE + 0x0060, base);/* MDP_EBI2_LCD0 */
		outpdw(MDP_BASE + 0x0068, ystride);/* MDP_EBI2_LCD0_YSTRIDE */
	} else {
		outpdw(MDP_BASE + 0x0064, base);/* MDP_EBI2_LCD1 */
		outpdw(MDP_BASE + 0x006c, ystride);/* MDP_EBI2_LCD1_YSTRIDE */
	}
	/* MDP cmd block disable */
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
}

void mdp4_mddi_setup(int mddi, unsigned long id)
{
	ulong 	bits;

	if (mddi == MDDI_EXTERNAL_SET)
		bits = 0x02;
	else if (mddi == MDDI_SECONDARY_SET)
		bits = 0x01;
	else
		bits = 0;	/* PRIMARY_SET */

	id <<= 16;

	bits |= id;
	/* MDP cmd block enable */
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);

	outpdw(MDP_BASE + 0x0090, bits); /* MDP_MDDI_PARAM_WR_SEL */
	/* MDP cmd block disable */
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
}

int mdp_ppp_blit(struct fb_info *info, struct mdp_blit_req *req)
{

	/* not implemented yet */
	return -1;
}

void mdp4_fetch_cfg(uint32 core_clk)
{
	uint32 dmap_data, vg_data;
	char *base;
	int i;
	/* MDP cmd block enable */
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);

	if (mdp_rev >= MDP_REV_41 || core_clk >= 90000000) { /* 90 Mhz */
		dmap_data = 0x47; /* 16 bytes-burst x 8 req */
		vg_data = 0x47; /* 16 bytes-burs x 8 req */
	} else {
		dmap_data = 0x27; /* 8 bytes-burst x 8 req */
		vg_data = 0x43; /* 16 bytes-burst x 4 req */
	}

	MSM_FB_DEBUG("mdp4_fetch_cfg: dmap=%x vg=%x\n",
			dmap_data, vg_data);

	/* dma_p fetch config */
	outpdw(MDP_BASE + 0x91004, dmap_data);
	/* dma_e fetch config */
	outpdw(MDP_BASE + 0xB1004, dmap_data);

	/*
	 * set up two vg pipes and two rgb pipes
	 */
	base = MDP_BASE + MDP4_VIDEO_BASE;
	for (i = 0; i < 4; i++) {
		outpdw(base + 0x1004, vg_data);
		base += MDP4_VIDEO_OFF;
	}
	/* MDP cmd block disable */
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
}

void mdp4_hw_init(void)
{
	ulong bits;
	uint32 clk_rate;

	/* MDP cmd block enable */
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);

	mdp4_update_perf_level(OVERLAY_PERF_LEVEL4);

#ifdef MDP4_ERROR
	/*
	 * Issue software reset on DMA_P will casue DMA_P dma engine stall
	 * on LCDC mode. However DMA_P does not stall at MDDI mode.
	 * This need further investigation.
	 */
	mdp4_sw_reset(0x17);
#endif

	if (mdp_rev > MDP_REV_41) {
		/* mdp chip select controller */
		outpdw(MDP_BASE + 0x00c0, CS_CONTROLLER_0);
		outpdw(MDP_BASE + 0x00c4, CS_CONTROLLER_1);
	}

	mdp4_clear_lcdc();

	mdp4_mixer_blend_init(0);
	mdp4_mixer_blend_init(1);
	mdp4_vg_qseed_init(0);
	mdp4_vg_qseed_init(1);

	mdp4_vg_csc_setup(0);
	mdp4_vg_csc_setup(1);
	mdp4_mixer_csc_setup(1);
	mdp4_mixer_csc_setup(2);
	mdp4_dmap_csc_setup();

	if (mdp_rev <= MDP_REV_41) {
		mdp4_mixer_gc_lut_setup(0);
		mdp4_mixer_gc_lut_setup(1);
	}

	mdp4_vg_igc_lut_setup(0);
	mdp4_vg_igc_lut_setup(1);

	mdp4_rgb_igc_lut_setup(0);
	mdp4_rgb_igc_lut_setup(1);

	outp32(MDP_EBI2_PORTMAP_MODE, 0x3);

	/* system interrupts */

	bits =  mdp_intr_mask;
	outpdw(MDP_BASE + 0x0050, bits);/* enable specififed interrupts */

	/* For the max read pending cmd config below, if the MDP clock     */
	/* is less than the AXI clock, then we must use 3 pending          */
	/* pending requests.  Otherwise, we should use 8 pending requests. */
	/* In the future we should do this detection automatically.	   */

	/* max read pending cmd config */
	outpdw(MDP_BASE + 0x004c, 0x02222);	/* 3 pending requests */

#ifndef CONFIG_FB_MSM_OVERLAY
	/* both REFRESH_MODE and DIRECT_OUT are ignored at BLT mode */
	mdp4_overlay_cfg(MDP4_MIXER0, OVERLAY_MODE_BLT, 0, 0);
	mdp4_overlay_cfg(MDP4_MIXER1, OVERLAY_MODE_BLT, 0, 0);
#endif

	clk_rate = mdp_get_core_clk();
	mdp4_fetch_cfg(clk_rate);

	/* Mark hardware as initialized. Only revisions > v2.1 have a register
	 * for tracking core reset status. */
	if (mdp_hw_revision > MDP4_REVISION_V2_1)
		outpdw(MDP_BASE + 0x003c, 1);

	/* MDP cmd block disable */
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
}


void mdp4_clear_lcdc(void)
{
	uint32 bits;

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);

	bits = inpdw(MDP_BASE + 0xc0000);
	if (bits & 0x01) { /* enabled already */
		mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
		return;
	}

	outpdw(MDP_BASE + 0xc0004, 0);	/* vsync ctrl out */
	outpdw(MDP_BASE + 0xc0008, 0);	/* vsync period */
	outpdw(MDP_BASE + 0xc000c, 0);	/* vsync pusle width */
	outpdw(MDP_BASE + 0xc0010, 0);	/* lcdc display HCTL */
	outpdw(MDP_BASE + 0xc0014, 0);	/* lcdc display v start */
	outpdw(MDP_BASE + 0xc0018, 0);	/* lcdc display v end */
	outpdw(MDP_BASE + 0xc001c, 0);	/* lcdc active hctl */
	outpdw(MDP_BASE + 0xc0020, 0);	/* lcdc active v start */
	outpdw(MDP_BASE + 0xc0024, 0);	/* lcdc active v end */
	outpdw(MDP_BASE + 0xc0028, 0);	/* lcdc board color */
	outpdw(MDP_BASE + 0xc002c, 0);	/* lcdc underflow ctrl */
	outpdw(MDP_BASE + 0xc0030, 0);	/* lcdc hsync skew */
	outpdw(MDP_BASE + 0xc0034, 0);	/* lcdc test ctl */
	outpdw(MDP_BASE + 0xc0038, 0);	/* lcdc ctl polarity */

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
}

irqreturn_t mdp4_isr(int irq, void *ptr)
{
	uint32 isr, mask, panel;
	struct mdp_dma_data *dma;
	struct mdp_hist_mgmt *mgmt = NULL;
	char *base_addr;
	int i, ret;

	mdp_is_in_isr = TRUE;

	/* complete all the reads before reading the interrupt
	* status register - eliminate effects of speculative
	* reads by the cpu
	*/
	rmb();
	isr = inpdw(MDP_INTR_STATUS);
	if (isr == 0)
		goto out;

	mdp4_stat.intr_tot++;
	mask = inpdw(MDP_INTR_ENABLE);
	outpdw(MDP_INTR_CLEAR, isr);

	if (isr & INTR_PRIMARY_INTF_UDERRUN) {
		mdp4_stat.intr_underrun_p++;
		/* When underun occurs mdp clear the histogram registers
		that are set before in hw_init so restore them back so
		that histogram works.*/
		for (i = 0; i < MDP_HIST_MGMT_MAX; i++) {
			mgmt = mdp_hist_mgmt_array[i];
			if (!mgmt)
				continue;
			base_addr = MDP_BASE + mgmt->base;
			MDP_OUTP(base_addr + 0x010, 1);
			outpdw(base_addr + 0x01c, INTR_HIST_DONE |
						INTR_HIST_RESET_SEQ_DONE);
			mgmt->mdp_is_hist_valid = FALSE;
			__mdp_histogram_reset(mgmt);
		}
	}

	if (isr & INTR_EXTERNAL_INTF_UDERRUN)
		mdp4_stat.intr_underrun_e++;

	isr &= mask;

	if (isr == 0)
		goto out;

	panel = mdp4_overlay_panel_list();
	if (isr & INTR_PRIMARY_VSYNC) {
		mdp4_stat.intr_vsync_p++;
		dma = &dma2_data;
		spin_lock(&mdp_spin_lock);
		mdp_intr_mask &= ~INTR_PRIMARY_VSYNC;
		outp32(MDP_INTR_ENABLE, mdp_intr_mask);
		dma->waiting = FALSE;
		if (panel & MDP4_PANEL_LCDC)
			mdp4_primary_vsync_lcdc();
#ifdef CONFIG_FB_MSM_MIPI_DSI
		else if (panel & MDP4_PANEL_DSI_VIDEO)
			mdp4_primary_vsync_dsi_video();
#endif
		spin_unlock(&mdp_spin_lock);
	}
#ifdef CONFIG_FB_MSM_DTV
	if (isr & INTR_EXTERNAL_VSYNC) {
		mdp4_stat.intr_vsync_e++;
		dma = &dma_e_data;
		spin_lock(&mdp_spin_lock);
		mdp_intr_mask &= ~INTR_EXTERNAL_VSYNC;
		outp32(MDP_INTR_ENABLE, mdp_intr_mask);
		dma->waiting = FALSE;
		if (panel & MDP4_PANEL_DTV)
			mdp4_external_vsync_dtv();
		spin_unlock(&mdp_spin_lock);
	}
#endif

#ifdef CONFIG_FB_MSM_OVERLAY
	if (isr & INTR_OVERLAY0_DONE) {
		mdp4_stat.intr_overlay0++;
		dma = &dma2_data;
		if (panel & (MDP4_PANEL_LCDC | MDP4_PANEL_DSI_VIDEO)) {
			/* disable LCDC interrupt */
			spin_lock(&mdp_spin_lock);
			mdp_intr_mask &= ~INTR_OVERLAY0_DONE;
			outp32(MDP_INTR_ENABLE, mdp_intr_mask);
			dma->waiting = FALSE;
			spin_unlock(&mdp_spin_lock);
			if (panel & MDP4_PANEL_LCDC)
				mdp4_overlay0_done_lcdc(dma);
#ifdef CONFIG_FB_MSM_MIPI_DSI
			else if (panel & MDP4_PANEL_DSI_VIDEO)
				mdp4_overlay0_done_dsi_video(dma);
#endif
		} else {        /* MDDI, DSI_CMD  */
#ifdef CONFIG_FB_MSM_MIPI_DSI
			if (panel & MDP4_PANEL_DSI_CMD)
				mdp4_overlay0_done_dsi_cmd(dma);
#else
			if (panel & MDP4_PANEL_MDDI)
				mdp4_overlay0_done_mddi(dma);
#endif
		}
		mdp_hw_cursor_done();
	}
	if (isr & INTR_OVERLAY1_DONE) {
		mdp4_stat.intr_overlay1++;
		/* disable DTV interrupt */
		dma = &dma_e_data;
		spin_lock(&mdp_spin_lock);
		mdp_intr_mask &= ~INTR_OVERLAY1_DONE;
		outp32(MDP_INTR_ENABLE, mdp_intr_mask);
		dma->waiting = FALSE;
		spin_unlock(&mdp_spin_lock);
#if defined(CONFIG_FB_MSM_DTV)
		if (panel & MDP4_PANEL_DTV)
			mdp4_overlay1_done_dtv();
#endif
#if defined(CONFIG_FB_MSM_TVOUT)
		if (panel & MDP4_PANEL_ATV)
			mdp4_overlay1_done_atv();
#endif
	}
#if defined(CONFIG_FB_MSM_WRITEBACK_MSM_PANEL)
	if (isr & INTR_OVERLAY2_DONE) {
		mdp4_stat.intr_overlay2++;

		mdp_pipe_ctrl(MDP_OVERLAY2_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);

		/* disable DTV interrupt */
		dma = &dma_wb_data;
		spin_lock(&mdp_spin_lock);
		mdp_intr_mask &= ~INTR_OVERLAY2_DONE;
		outp32(MDP_INTR_ENABLE, mdp_intr_mask);
		dma->waiting = FALSE;
		spin_unlock(&mdp_spin_lock);
		if (panel & MDP4_PANEL_WRITEBACK)
			mdp4_overlay1_done_writeback(dma);
	}
#endif
#endif	/* OVERLAY */

	if (isr & INTR_DMA_P_DONE) {
		mdp4_stat.intr_dma_p++;
		dma = &dma2_data;
		if (panel & MDP4_PANEL_LCDC) {
			/* disable LCDC interrupt */
			spin_lock(&mdp_spin_lock);
			mdp_intr_mask &= ~INTR_DMA_P_DONE;
			outp32(MDP_INTR_ENABLE, mdp_intr_mask);
			dma->waiting = FALSE;
			mdp4_dma_p_done_lcdc();
			spin_unlock(&mdp_spin_lock);
		}
#ifdef CONFIG_FB_MSM_OVERLAY
#ifdef CONFIG_FB_MSM_MIPI_DSI
		else if (panel & MDP4_PANEL_DSI_VIDEO) {
			/* disable LCDC interrupt */
			spin_lock(&mdp_spin_lock);
			mdp_intr_mask &= ~INTR_DMA_P_DONE;
			outp32(MDP_INTR_ENABLE, mdp_intr_mask);
			dma->waiting = FALSE;
			mdp4_dma_p_done_dsi_video(dma);
			spin_unlock(&mdp_spin_lock);
		} else if (panel & MDP4_PANEL_DSI_CMD) {
			mdp4_dma_p_done_dsi(dma);
		}
#else
		else { /* MDDI */
			mdp4_dma_p_done_mddi(dma);
			mdp_pipe_ctrl(MDP_DMA2_BLOCK,
				MDP_BLOCK_POWER_OFF, TRUE);
			complete(&dma->comp);
		}
#endif
#else
		else {
			spin_lock(&mdp_spin_lock);
			dma->busy = FALSE;
			spin_unlock(&mdp_spin_lock);
			complete(&dma->comp);
		}
#endif
	}
	if (isr & INTR_DMA_S_DONE) {
		mdp4_stat.intr_dma_s++;
#if defined(CONFIG_FB_MSM_OVERLAY) && defined(CONFIG_FB_MSM_MDDI)
		dma = &dma2_data;
#else
		dma = &dma_s_data;
#endif

		dma->busy = FALSE;
		mdp_pipe_ctrl(MDP_DMA_S_BLOCK,
				MDP_BLOCK_POWER_OFF, TRUE);
		complete(&dma->comp);
	}
	if (isr & INTR_DMA_E_DONE) {
		mdp4_stat.intr_dma_e++;
		dma = &dma_e_data;
		spin_lock(&mdp_spin_lock);
		mdp_intr_mask &= ~INTR_DMA_E_DONE;
		outp32(MDP_INTR_ENABLE, mdp_intr_mask);
		dma->busy = FALSE;
		mdp4_dma_e_done_dtv();
		if (dma->waiting) {
			dma->waiting = FALSE;
			complete(&dma->comp);
		}
		spin_unlock(&mdp_spin_lock);
	}
	if (isr & INTR_DMA_P_HISTOGRAM) {
		mdp4_stat.intr_histogram++;
		ret = mdp_histogram_block2mgmt(MDP_BLOCK_DMA_P, &mgmt);
		if (!ret)
			mdp_histogram_handle_isr(mgmt);
	}
	if (isr & INTR_DMA_S_HISTOGRAM) {
		mdp4_stat.intr_histogram++;
		ret = mdp_histogram_block2mgmt(MDP_BLOCK_DMA_S, &mgmt);
		if (!ret)
			mdp_histogram_handle_isr(mgmt);
	}
	if (isr & INTR_VG1_HISTOGRAM) {
		mdp4_stat.intr_histogram++;
		ret = mdp_histogram_block2mgmt(MDP_BLOCK_VG_1, &mgmt);
		if (!ret)
			mdp_histogram_handle_isr(mgmt);
	}
	if (isr & INTR_VG2_HISTOGRAM) {
		mdp4_stat.intr_histogram++;
		ret = mdp_histogram_block2mgmt(MDP_BLOCK_VG_2, &mgmt);
		if (!ret)
			mdp_histogram_handle_isr(mgmt);
	}

out:
	mdp_is_in_isr = FALSE;

	return IRQ_HANDLED;
}


/*
 * QSEED tables
 */

static uint32 vg_qseed_table0[] = {
	0x5556aaff, 0x00000000, 0x00000000, 0x00000000
};

static uint32 vg_qseed_table1[] = {
	0x00000000, 0x20000000,
};

static uint32 vg_qseed_table2[] = {
	0x02000000, 0x00000000, 0x01ff0ff9, 0x00000008,
	0x01fb0ff2, 0x00000013, 0x01f50fed, 0x0ffe0020,
	0x01ed0fe8, 0x0ffd002e, 0x01e30fe4, 0x0ffb003e,
	0x01d80fe1, 0x0ff9004e, 0x01cb0fde, 0x0ff70060,
	0x01bc0fdc, 0x0ff40074, 0x01ac0fdb, 0x0ff20087,
	0x019a0fdb, 0x0fef009c, 0x01870fdb, 0x0fed00b1,
	0x01740fdb, 0x0fea00c7, 0x01600fdc, 0x0fe700dd,
	0x014b0fdd, 0x0fe500f3, 0x01350fdf, 0x0fe30109,
	0x01200fe0, 0x0fe00120, 0x01090fe3, 0x0fdf0135,
	0x00f30fe5, 0x0fdd014b, 0x00dd0fe7, 0x0fdc0160,
	0x00c70fea, 0x0fdb0174, 0x00b10fed, 0x0fdb0187,
	0x009c0fef, 0x0fdb019a, 0x00870ff2, 0x0fdb01ac,
	0x00740ff4, 0x0fdc01bc, 0x00600ff7, 0x0fde01cb,
	0x004e0ff9, 0x0fe101d8, 0x003e0ffb, 0x0fe401e3,
	0x002e0ffd, 0x0fe801ed, 0x00200ffe, 0x0fed01f5,
	0x00130000, 0x0ff201fb, 0x00080000, 0x0ff901ff,

	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,

	0x02000000, 0x00000000, 0x01fc0ff9, 0x0ffe000d,
	0x01f60ff3, 0x0ffb001c, 0x01ef0fed, 0x0ff9002b,
	0x01e60fe8, 0x0ff6003c, 0x01dc0fe4, 0x0ff3004d,
	0x01d00fe0, 0x0ff1005f, 0x01c30fde, 0x0fee0071,
	0x01b50fdb, 0x0feb0085, 0x01a70fd9, 0x0fe80098,
	0x01960fd8, 0x0fe600ac, 0x01850fd7, 0x0fe300c1,
	0x01730fd7, 0x0fe100d5, 0x01610fd7, 0x0fdf00e9,
	0x014e0fd8, 0x0fdd00fd, 0x013b0fd8, 0x0fdb0112,
	0x01250fda, 0x0fda0127, 0x01120fdb, 0x0fd8013b,
	0x00fd0fdd, 0x0fd8014e, 0x00e90fdf, 0x0fd70161,
	0x00d50fe1, 0x0fd70173, 0x00c10fe3, 0x0fd70185,
	0x00ac0fe6, 0x0fd80196, 0x00980fe8, 0x0fd901a7,
	0x00850feb, 0x0fdb01b5, 0x00710fee, 0x0fde01c3,
	0x005f0ff1, 0x0fe001d0, 0x004d0ff3, 0x0fe401dc,
	0x003c0ff6, 0x0fe801e6, 0x002b0ff9, 0x0fed01ef,
	0x001c0ffb, 0x0ff301f6, 0x000d0ffe, 0x0ff901fc,

	0x020f0034, 0x0f7a0043, 0x01e80023, 0x0fa8004d,
	0x01d30016, 0x0fbe0059, 0x01c6000a, 0x0fc90067,
	0x01bd0000, 0x0fce0075, 0x01b50ff7, 0x0fcf0085,
	0x01ae0fee, 0x0fcf0095, 0x01a70fe6, 0x0fcd00a6,
	0x019d0fe0, 0x0fcb00b8, 0x01940fd9, 0x0fc900ca,
	0x01890fd4, 0x0fc700dc, 0x017d0fcf, 0x0fc600ee,
	0x01700fcc, 0x0fc40100, 0x01620fc9, 0x0fc40111,
	0x01540fc6, 0x0fc30123, 0x01430fc5, 0x0fc40134,
	0x01340fc4, 0x0fc50143, 0x01230fc3, 0x0fc60154,
	0x01110fc4, 0x0fc90162, 0x01000fc4, 0x0fcc0170,
	0x00ee0fc6, 0x0fcf017d, 0x00dc0fc7, 0x0fd40189,
	0x00ca0fc9, 0x0fd90194, 0x00b80fcb, 0x0fe0019d,
	0x00a60fcd, 0x0fe601a7, 0x00950fcf, 0x0fee01ae,
	0x00850fcf, 0x0ff701b5, 0x00750fce, 0x000001bd,
	0x00670fc9, 0x000a01c6, 0x00590fbe, 0x001601d3,
	0x004d0fa8, 0x002301e8, 0x00430f7a, 0x0034020f,

	0x015c005e, 0x0fde0068, 0x015c0054, 0x0fdd0073,
	0x015b004b, 0x0fdc007e, 0x015a0042, 0x0fdb0089,
	0x01590039, 0x0fda0094, 0x01560030, 0x0fda00a0,
	0x01530028, 0x0fda00ab, 0x014f0020, 0x0fda00b7,
	0x014a0019, 0x0fdb00c2, 0x01450011, 0x0fdc00ce,
	0x013e000b, 0x0fde00d9, 0x01390004, 0x0fdf00e4,
	0x01310ffe, 0x0fe200ef, 0x01290ff9, 0x0fe400fa,
	0x01200ff4, 0x0fe80104, 0x01180fef, 0x0feb010e,
	0x010e0feb, 0x0fef0118, 0x01040fe8, 0x0ff40120,
	0x00fa0fe4, 0x0ff90129, 0x00ef0fe2, 0x0ffe0131,
	0x00e40fdf, 0x00040139, 0x00d90fde, 0x000b013e,
	0x00ce0fdc, 0x00110145, 0x00c20fdb, 0x0019014a,
	0x00b70fda, 0x0020014f, 0x00ab0fda, 0x00280153,
	0x00a00fda, 0x00300156, 0x00940fda, 0x00390159,
	0x00890fdb, 0x0042015a, 0x007e0fdc, 0x004b015b,
	0x00730fdd, 0x0054015c, 0x00680fde, 0x005e015c,

	0x01300068, 0x0ff80070, 0x01300060, 0x0ff80078,
	0x012f0059, 0x0ff80080, 0x012d0052, 0x0ff80089,
	0x012b004b, 0x0ff90091, 0x01290044, 0x0ff9009a,
	0x0126003d, 0x0ffa00a3, 0x01220037, 0x0ffb00ac,
	0x011f0031, 0x0ffc00b4, 0x011a002b, 0x0ffe00bd,
	0x01150026, 0x000000c5, 0x010f0021, 0x000200ce,
	0x010a001c, 0x000400d6, 0x01030018, 0x000600df,
	0x00fd0014, 0x000900e6, 0x00f60010, 0x000c00ee,
	0x00ee000c, 0x001000f6, 0x00e60009, 0x001400fd,
	0x00df0006, 0x00180103, 0x00d60004, 0x001c010a,
	0x00ce0002, 0x0021010f, 0x00c50000, 0x00260115,
	0x00bd0ffe, 0x002b011a, 0x00b40ffc, 0x0031011f,
	0x00ac0ffb, 0x00370122, 0x00a30ffa, 0x003d0126,
	0x009a0ff9, 0x00440129, 0x00910ff9, 0x004b012b,
	0x00890ff8, 0x0052012d, 0x00800ff8, 0x0059012f,
	0x00780ff8, 0x00600130, 0x00700ff8, 0x00680130,

	0x01050079, 0x0003007f, 0x01040073, 0x00030086,
	0x0103006d, 0x0004008c, 0x01030066, 0x00050092,
	0x01010060, 0x00060099, 0x0100005a, 0x0007009f,
	0x00fe0054, 0x000900a5, 0x00fa004f, 0x000b00ac,
	0x00f80049, 0x000d00b2, 0x00f50044, 0x000f00b8,
	0x00f2003f, 0x001200bd, 0x00ef0039, 0x001500c3,
	0x00ea0035, 0x001800c9, 0x00e60030, 0x001c00ce,
	0x00e3002b, 0x001f00d3, 0x00dd0027, 0x002300d9,
	0x00d90023, 0x002700dd, 0x00d3001f, 0x002b00e3,
	0x00ce001c, 0x003000e6, 0x00c90018, 0x003500ea,
	0x00c30015, 0x003900ef, 0x00bd0012, 0x003f00f2,
	0x00b8000f, 0x004400f5, 0x00b2000d, 0x004900f8,
	0x00ac000b, 0x004f00fa, 0x00a50009, 0x005400fe,
	0x009f0007, 0x005a0100, 0x00990006, 0x00600101,
	0x00920005, 0x00660103, 0x008c0004, 0x006d0103,
	0x00860003, 0x00730104, 0x007f0003, 0x00790105,

	0x00cf0088, 0x001d008c, 0x00ce0084, 0x0020008e,
	0x00cd0080, 0x00210092, 0x00cd007b, 0x00240094,
	0x00ca0077, 0x00270098, 0x00c90073, 0x0029009b,
	0x00c8006f, 0x002c009d, 0x00c6006b, 0x002f00a0,
	0x00c50067, 0x003200a2, 0x00c30062, 0x003600a5,
	0x00c0005f, 0x003900a8, 0x00c0005b, 0x003b00aa,
	0x00be0057, 0x003e00ad, 0x00ba0054, 0x004200b0,
	0x00b90050, 0x004500b2, 0x00b7004c, 0x004900b4,
	0x00b40049, 0x004c00b7, 0x00b20045, 0x005000b9,
	0x00b00042, 0x005400ba, 0x00ad003e, 0x005700be,
	0x00aa003b, 0x005b00c0, 0x00a80039, 0x005f00c0,
	0x00a50036, 0x006200c3, 0x00a20032, 0x006700c5,
	0x00a0002f, 0x006b00c6, 0x009d002c, 0x006f00c8,
	0x009b0029, 0x007300c9, 0x00980027, 0x007700ca,
	0x00940024, 0x007b00cd, 0x00920021, 0x008000cd,
	0x008e0020, 0x008400ce, 0x008c001d, 0x008800cf,

	0x008e0083, 0x006b0084, 0x008d0083, 0x006c0084,
	0x008d0082, 0x006d0084, 0x008d0081, 0x006d0085,
	0x008d0080, 0x006e0085, 0x008c007f, 0x006f0086,
	0x008b007f, 0x00700086, 0x008b007e, 0x00710086,
	0x008b007d, 0x00720086, 0x008a007d, 0x00730086,
	0x008a007c, 0x00730087, 0x008a007b, 0x00740087,
	0x0089007b, 0x00750087, 0x008a0079, 0x00750088,
	0x008a0078, 0x00760088, 0x008a0077, 0x00770088,
	0x00880077, 0x0077008a, 0x00880076, 0x0078008a,
	0x00880075, 0x0079008a, 0x00870075, 0x007b0089,
	0x00870074, 0x007b008a, 0x00870073, 0x007c008a,
	0x00860073, 0x007d008a, 0x00860072, 0x007d008b,
	0x00860071, 0x007e008b, 0x00860070, 0x007f008b,
	0x0086006f, 0x007f008c, 0x0085006e, 0x0080008d,
	0x0085006d, 0x0081008d, 0x0084006d, 0x0082008d,
	0x0084006c, 0x0083008d, 0x0084006b, 0x0083008e,

	0x023c0fe2, 0x00000fe2, 0x023a0fdb, 0x00000feb,
	0x02360fd3, 0x0fff0ff8, 0x022e0fcf, 0x0ffc0007,
	0x02250fca, 0x0ffa0017, 0x021a0fc6, 0x0ff70029,
	0x020c0fc4, 0x0ff4003c, 0x01fd0fc1, 0x0ff10051,
	0x01eb0fc0, 0x0fed0068, 0x01d80fc0, 0x0fe9007f,
	0x01c30fc1, 0x0fe50097, 0x01ac0fc2, 0x0fe200b0,
	0x01960fc3, 0x0fdd00ca, 0x017e0fc5, 0x0fd900e4,
	0x01650fc8, 0x0fd500fe, 0x014b0fcb, 0x0fd20118,
	0x01330fcd, 0x0fcd0133, 0x01180fd2, 0x0fcb014b,
	0x00fe0fd5, 0x0fc80165, 0x00e40fd9, 0x0fc5017e,
	0x00ca0fdd, 0x0fc30196, 0x00b00fe2, 0x0fc201ac,
	0x00970fe5, 0x0fc101c3, 0x007f0fe9, 0x0fc001d8,
	0x00680fed, 0x0fc001eb, 0x00510ff1, 0x0fc101fd,
	0x003c0ff4, 0x0fc4020c, 0x00290ff7, 0x0fc6021a,
	0x00170ffa, 0x0fca0225, 0x00070ffc, 0x0fcf022e,
	0x0ff80fff, 0x0fd30236, 0x0feb0000, 0x0fdb023a,

	0x02780fc4, 0x00000fc4, 0x02770fbc, 0x0fff0fce,
	0x02710fb5, 0x0ffe0fdc, 0x02690fb0, 0x0ffa0fed,
	0x025f0fab, 0x0ff70fff, 0x02500fa8, 0x0ff30015,
	0x02410fa6, 0x0fef002a, 0x022f0fa4, 0x0feb0042,
	0x021a0fa4, 0x0fe5005d, 0x02040fa5, 0x0fe10076,
	0x01eb0fa7, 0x0fdb0093, 0x01d20fa9, 0x0fd600af,
	0x01b80fab, 0x0fd000cd, 0x019d0faf, 0x0fca00ea,
	0x01810fb2, 0x0fc50108, 0x01620fb7, 0x0fc10126,
	0x01440fbb, 0x0fbb0146, 0x01260fc1, 0x0fb70162,
	0x01080fc5, 0x0fb20181, 0x00ea0fca, 0x0faf019d,
	0x00cd0fd0, 0x0fab01b8, 0x00af0fd6, 0x0fa901d2,
	0x00930fdb, 0x0fa701eb, 0x00760fe1, 0x0fa50204,
	0x005d0fe5, 0x0fa4021a, 0x00420feb, 0x0fa4022f,
	0x002a0fef, 0x0fa60241, 0x00150ff3, 0x0fa80250,
	0x0fff0ff7, 0x0fab025f, 0x0fed0ffa, 0x0fb00269,
	0x0fdc0ffe, 0x0fb50271, 0x0fce0fff, 0x0fbc0277,

	0x02a00fb0, 0x00000fb0, 0x029e0fa8, 0x0fff0fbb,
	0x02980fa1, 0x0ffd0fca, 0x028f0f9c, 0x0ff90fdc,
	0x02840f97, 0x0ff50ff0, 0x02740f94, 0x0ff10007,
	0x02640f92, 0x0fec001e, 0x02500f91, 0x0fe70038,
	0x023a0f91, 0x0fe00055, 0x02220f92, 0x0fdb0071,
	0x02080f95, 0x0fd4008f, 0x01ec0f98, 0x0fce00ae,
	0x01cf0f9b, 0x0fc700cf, 0x01b10f9f, 0x0fc100ef,
	0x01920fa4, 0x0fbb010f, 0x01710faa, 0x0fb50130,
	0x01520fae, 0x0fae0152, 0x01300fb5, 0x0faa0171,
	0x010f0fbb, 0x0fa40192, 0x00ef0fc1, 0x0f9f01b1,
	0x00cf0fc7, 0x0f9b01cf, 0x00ae0fce, 0x0f9801ec,
	0x008f0fd4, 0x0f950208, 0x00710fdb, 0x0f920222,
	0x00550fe0, 0x0f91023a, 0x00380fe7, 0x0f910250,
	0x001e0fec, 0x0f920264, 0x00070ff1, 0x0f940274,
	0x0ff00ff5, 0x0f970284, 0x0fdc0ff9, 0x0f9c028f,
	0x0fca0ffd, 0x0fa10298, 0x0fbb0fff, 0x0fa8029e,

	0x02c80f9c, 0x00000f9c, 0x02c70f94, 0x0ffe0fa7,
	0x02c10f8c, 0x0ffc0fb7, 0x02b70f87, 0x0ff70fcb,
	0x02aa0f83, 0x0ff30fe0, 0x02990f80, 0x0fee0ff9,
	0x02870f7f, 0x0fe80012, 0x02720f7e, 0x0fe2002e,
	0x025a0f7e, 0x0fdb004d, 0x02400f80, 0x0fd5006b,
	0x02230f84, 0x0fcd008c, 0x02050f87, 0x0fc700ad,
	0x01e60f8b, 0x0fbf00d0, 0x01c60f90, 0x0fb700f3,
	0x01a30f96, 0x0fb00117, 0x01800f9c, 0x0faa013a,
	0x015d0fa2, 0x0fa2015f, 0x013a0faa, 0x0f9c0180,
	0x01170fb0, 0x0f9601a3, 0x00f30fb7, 0x0f9001c6,
	0x00d00fbf, 0x0f8b01e6, 0x00ad0fc7, 0x0f870205,
	0x008c0fcd, 0x0f840223, 0x006b0fd5, 0x0f800240,
	0x004d0fdb, 0x0f7e025a, 0x002e0fe2, 0x0f7e0272,
	0x00120fe8, 0x0f7f0287, 0x0ff90fee, 0x0f800299,
	0x0fe00ff3, 0x0f8302aa, 0x0fcb0ff7, 0x0f8702b7,
	0x0fb70ffc, 0x0f8c02c1, 0x0fa70ffe, 0x0f9402c7,

	0x02f00f88, 0x00000f88, 0x02ee0f80, 0x0ffe0f94,
	0x02e70f78, 0x0ffc0fa5, 0x02dd0f73, 0x0ff60fba,
	0x02ce0f6f, 0x0ff20fd1, 0x02be0f6c, 0x0feb0feb,
	0x02aa0f6b, 0x0fe50006, 0x02940f6a, 0x0fde0024,
	0x02790f6c, 0x0fd60045, 0x025e0f6e, 0x0fcf0065,
	0x023f0f72, 0x0fc60089, 0x021d0f77, 0x0fbf00ad,
	0x01fd0f7b, 0x0fb600d2, 0x01da0f81, 0x0fad00f8,
	0x01b50f87, 0x0fa6011e, 0x018f0f8f, 0x0f9e0144,
	0x016b0f95, 0x0f95016b, 0x01440f9e, 0x0f8f018f,
	0x011e0fa6, 0x0f8701b5, 0x00f80fad, 0x0f8101da,
	0x00d20fb6, 0x0f7b01fd, 0x00ad0fbf, 0x0f77021d,
	0x00890fc6, 0x0f72023f, 0x00650fcf, 0x0f6e025e,
	0x00450fd6, 0x0f6c0279, 0x00240fde, 0x0f6a0294,
	0x00060fe5, 0x0f6b02aa, 0x0feb0feb, 0x0f6c02be,
	0x0fd10ff2, 0x0f6f02ce, 0x0fba0ff6, 0x0f7302dd,
	0x0fa50ffc, 0x0f7802e7, 0x0f940ffe, 0x0f8002ee,

	0x03180f74, 0x00000f74, 0x03160f6b, 0x0ffe0f81,
	0x030e0f64, 0x0ffb0f93, 0x03030f5f, 0x0ff50fa9,
	0x02f40f5b, 0x0ff00fc1, 0x02e20f58, 0x0fe90fdd,
	0x02cd0f57, 0x0fe20ffa, 0x02b60f57, 0x0fda0019,
	0x02990f59, 0x0fd1003d, 0x027b0f5c, 0x0fc90060,
	0x02590f61, 0x0fc00086, 0x02370f66, 0x0fb700ac,
	0x02130f6b, 0x0fae00d4, 0x01ee0f72, 0x0fa400fc,
	0x01c70f79, 0x0f9b0125, 0x019f0f81, 0x0f93014d,
	0x01760f89, 0x0f890178, 0x014d0f93, 0x0f81019f,
	0x01250f9b, 0x0f7901c7, 0x00fc0fa4, 0x0f7201ee,
	0x00d40fae, 0x0f6b0213, 0x00ac0fb7, 0x0f660237,
	0x00860fc0, 0x0f610259, 0x00600fc9, 0x0f5c027b,
	0x003d0fd1, 0x0f590299, 0x00190fda, 0x0f5702b6,
	0x0ffa0fe2, 0x0f5702cd, 0x0fdd0fe9, 0x0f5802e2,
	0x0fc10ff0, 0x0f5b02f4, 0x0fa90ff5, 0x0f5f0303,
	0x0f930ffb, 0x0f64030e, 0x0f810ffe, 0x0f6b0316,

	0x03400f60, 0x00000f60, 0x033e0f57, 0x0ffe0f6d,
	0x03370f4f, 0x0ffa0f80, 0x032a0f4b, 0x0ff30f98,
	0x031a0f46, 0x0fee0fb2, 0x03070f44, 0x0fe60fcf,
	0x02f10f44, 0x0fde0fed, 0x02d70f44, 0x0fd6000f,
	0x02b80f46, 0x0fcc0036, 0x02990f4a, 0x0fc3005a,
	0x02750f4f, 0x0fb90083, 0x02500f55, 0x0fb000ab,
	0x022a0f5b, 0x0fa500d6, 0x02020f63, 0x0f9a0101,
	0x01d80f6b, 0x0f91012c, 0x01ae0f74, 0x0f870157,
	0x01840f7c, 0x0f7c0184, 0x01570f87, 0x0f7401ae,
	0x012c0f91, 0x0f6b01d8, 0x01010f9a, 0x0f630202,
	0x00d60fa5, 0x0f5b022a, 0x00ab0fb0, 0x0f550250,
	0x00830fb9, 0x0f4f0275, 0x005a0fc3, 0x0f4a0299,
	0x00360fcc, 0x0f4602b8, 0x000f0fd6, 0x0f4402d7,
	0x0fed0fde, 0x0f4402f1, 0x0fcf0fe6, 0x0f440307,
	0x0fb20fee, 0x0f46031a, 0x0f980ff3, 0x0f4b032a,
	0x0f800ffa, 0x0f4f0337, 0x0f6d0ffe, 0x0f57033e,

	0x02000000, 0x00000000, 0x01ff0ff9, 0x00000008,
	0x01fb0ff2, 0x00000013, 0x01f50fed, 0x0ffe0020,
	0x01ed0fe8, 0x0ffd002e, 0x01e30fe4, 0x0ffb003e,
	0x01d80fe1, 0x0ff9004e, 0x01cb0fde, 0x0ff70060,
	0x01bc0fdc, 0x0ff40074, 0x01ac0fdb, 0x0ff20087,
	0x019a0fdb, 0x0fef009c, 0x01870fdb, 0x0fed00b1,
	0x01740fdb, 0x0fea00c7, 0x01600fdc, 0x0fe700dd,
	0x014b0fdd, 0x0fe500f3, 0x01350fdf, 0x0fe30109,
	0x01200fe0, 0x0fe00120, 0x01090fe3, 0x0fdf0135,
	0x00f30fe5, 0x0fdd014b, 0x00dd0fe7, 0x0fdc0160,
	0x00c70fea, 0x0fdb0174, 0x00b10fed, 0x0fdb0187,
	0x009c0fef, 0x0fdb019a, 0x00870ff2, 0x0fdb01ac,
	0x00740ff4, 0x0fdc01bc, 0x00600ff7, 0x0fde01cb,
	0x004e0ff9, 0x0fe101d8, 0x003e0ffb, 0x0fe401e3,
	0x002e0ffd, 0x0fe801ed, 0x00200ffe, 0x0fed01f5,
	0x00130000, 0x0ff201fb, 0x00080000, 0x0ff901ff,

	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,
	0x02000000, 0x00000000, 0x02000000, 0x00000000,

	0x02000000, 0x00000000, 0x01fc0ff9, 0x0ffe000d,
	0x01f60ff3, 0x0ffb001c, 0x01ef0fed, 0x0ff9002b,
	0x01e60fe8, 0x0ff6003c, 0x01dc0fe4, 0x0ff3004d,
	0x01d00fe0, 0x0ff1005f, 0x01c30fde, 0x0fee0071,
	0x01b50fdb, 0x0feb0085, 0x01a70fd9, 0x0fe80098,
	0x01960fd8, 0x0fe600ac, 0x01850fd7, 0x0fe300c1,
	0x01730fd7, 0x0fe100d5, 0x01610fd7, 0x0fdf00e9,
	0x014e0fd8, 0x0fdd00fd, 0x013b0fd8, 0x0fdb0112,
	0x01250fda, 0x0fda0127, 0x01120fdb, 0x0fd8013b,
	0x00fd0fdd, 0x0fd8014e, 0x00e90fdf, 0x0fd70161,
	0x00d50fe1, 0x0fd70173, 0x00c10fe3, 0x0fd70185,
	0x00ac0fe6, 0x0fd80196, 0x00980fe8, 0x0fd901a7,
	0x00850feb, 0x0fdb01b5, 0x00710fee, 0x0fde01c3,
	0x005f0ff1, 0x0fe001d0, 0x004d0ff3, 0x0fe401dc,
	0x003c0ff6, 0x0fe801e6, 0x002b0ff9, 0x0fed01ef,
	0x001c0ffb, 0x0ff301f6, 0x000d0ffe, 0x0ff901fc,

	0x020f0034, 0x0f7a0043, 0x01e80023, 0x0fa8004d,
	0x01d30016, 0x0fbe0059, 0x01c6000a, 0x0fc90067,
	0x01bd0000, 0x0fce0075, 0x01b50ff7, 0x0fcf0085,
	0x01ae0fee, 0x0fcf0095, 0x01a70fe6, 0x0fcd00a6,
	0x019d0fe0, 0x0fcb00b8, 0x01940fd9, 0x0fc900ca,
	0x01890fd4, 0x0fc700dc, 0x017d0fcf, 0x0fc600ee,
	0x01700fcc, 0x0fc40100, 0x01620fc9, 0x0fc40111,
	0x01540fc6, 0x0fc30123, 0x01430fc5, 0x0fc40134,
	0x01340fc4, 0x0fc50143, 0x01230fc3, 0x0fc60154,
	0x01110fc4, 0x0fc90162, 0x01000fc4, 0x0fcc0170,
	0x00ee0fc6, 0x0fcf017d, 0x00dc0fc7, 0x0fd40189,
	0x00ca0fc9, 0x0fd90194, 0x00b80fcb, 0x0fe0019d,
	0x00a60fcd, 0x0fe601a7, 0x00950fcf, 0x0fee01ae,
	0x00850fcf, 0x0ff701b5, 0x00750fce, 0x000001bd,
	0x00670fc9, 0x000a01c6, 0x00590fbe, 0x001601d3,
	0x004d0fa8, 0x002301e8, 0x00430f7a, 0x0034020f,

	0x015c005e, 0x0fde0068, 0x015c0054, 0x0fdd0073,
	0x015b004b, 0x0fdc007e, 0x015a0042, 0x0fdb0089,
	0x01590039, 0x0fda0094, 0x01560030, 0x0fda00a0,
	0x01530028, 0x0fda00ab, 0x014f0020, 0x0fda00b7,
	0x014a0019, 0x0fdb00c2, 0x01450011, 0x0fdc00ce,
	0x013e000b, 0x0fde00d9, 0x01390004, 0x0fdf00e4,
	0x01310ffe, 0x0fe200ef, 0x01290ff9, 0x0fe400fa,
	0x01200ff4, 0x0fe80104, 0x01180fef, 0x0feb010e,
	0x010e0feb, 0x0fef0118, 0x01040fe8, 0x0ff40120,
	0x00fa0fe4, 0x0ff90129, 0x00ef0fe2, 0x0ffe0131,
	0x00e40fdf, 0x00040139, 0x00d90fde, 0x000b013e,
	0x00ce0fdc, 0x00110145, 0x00c20fdb, 0x0019014a,
	0x00b70fda, 0x0020014f, 0x00ab0fda, 0x00280153,
	0x00a00fda, 0x00300156, 0x00940fda, 0x00390159,
	0x00890fdb, 0x0042015a, 0x007e0fdc, 0x004b015b,
	0x00730fdd, 0x0054015c, 0x00680fde, 0x005e015c,

	0x01300068, 0x0ff80070, 0x01300060, 0x0ff80078,
	0x012f0059, 0x0ff80080, 0x012d0052, 0x0ff80089,
	0x012b004b, 0x0ff90091, 0x01290044, 0x0ff9009a,
	0x0126003d, 0x0ffa00a3, 0x01220037, 0x0ffb00ac,
	0x011f0031, 0x0ffc00b4, 0x011a002b, 0x0ffe00bd,
	0x01150026, 0x000000c5, 0x010f0021, 0x000200ce,
	0x010a001c, 0x000400d6, 0x01030018, 0x000600df,
	0x00fd0014, 0x000900e6, 0x00f60010, 0x000c00ee,
	0x00ee000c, 0x001000f6, 0x00e60009, 0x001400fd,
	0x00df0006, 0x00180103, 0x00d60004, 0x001c010a,
	0x00ce0002, 0x0021010f, 0x00c50000, 0x00260115,
	0x00bd0ffe, 0x002b011a, 0x00b40ffc, 0x0031011f,
	0x00ac0ffb, 0x00370122, 0x00a30ffa, 0x003d0126,
	0x009a0ff9, 0x00440129, 0x00910ff9, 0x004b012b,
	0x00890ff8, 0x0052012d, 0x00800ff8, 0x0059012f,
	0x00780ff8, 0x00600130, 0x00700ff8, 0x00680130,

	0x01050079, 0x0003007f, 0x01040073, 0x00030086,
	0x0103006d, 0x0004008c, 0x01030066, 0x00050092,
	0x01010060, 0x00060099, 0x0100005a, 0x0007009f,
	0x00fe0054, 0x000900a5, 0x00fa004f, 0x000b00ac,
	0x00f80049, 0x000d00b2, 0x00f50044, 0x000f00b8,
	0x00f2003f, 0x001200bd, 0x00ef0039, 0x001500c3,
	0x00ea0035, 0x001800c9, 0x00e60030, 0x001c00ce,
	0x00e3002b, 0x001f00d3, 0x00dd0027, 0x002300d9,
	0x00d90023, 0x002700dd, 0x00d3001f, 0x002b00e3,
	0x00ce001c, 0x003000e6, 0x00c90018, 0x003500ea,
	0x00c30015, 0x003900ef, 0x00bd0012, 0x003f00f2,
	0x00b8000f, 0x004400f5, 0x00b2000d, 0x004900f8,
	0x00ac000b, 0x004f00fa, 0x00a50009, 0x005400fe,
	0x009f0007, 0x005a0100, 0x00990006, 0x00600101,
	0x00920005, 0x00660103, 0x008c0004, 0x006d0103,
	0x00860003, 0x00730104, 0x007f0003, 0x00790105,

	0x00cf0088, 0x001d008c, 0x00ce0084, 0x0020008e,
	0x00cd0080, 0x00210092, 0x00cd007b, 0x00240094,
	0x00ca0077, 0x00270098, 0x00c90073, 0x0029009b,
	0x00c8006f, 0x002c009d, 0x00c6006b, 0x002f00a0,
	0x00c50067, 0x003200a2, 0x00c30062, 0x003600a5,
	0x00c0005f, 0x003900a8, 0x00c0005b, 0x003b00aa,
	0x00be0057, 0x003e00ad, 0x00ba0054, 0x004200b0,
	0x00b90050, 0x004500b2, 0x00b7004c, 0x004900b4,
	0x00b40049, 0x004c00b7, 0x00b20045, 0x005000b9,
	0x00b00042, 0x005400ba, 0x00ad003e, 0x005700be,
	0x00aa003b, 0x005b00c0, 0x00a80039, 0x005f00c0,
	0x00a50036, 0x006200c3, 0x00a20032, 0x006700c5,
	0x00a0002f, 0x006b00c6, 0x009d002c, 0x006f00c8,
	0x009b0029, 0x007300c9, 0x00980027, 0x007700ca,
	0x00940024, 0x007b00cd, 0x00920021, 0x008000cd,
	0x008e0020, 0x008400ce, 0x008c001d, 0x008800cf,

	0x008e0083, 0x006b0084, 0x008d0083, 0x006c0084,
	0x008d0082, 0x006d0084, 0x008d0081, 0x006d0085,
	0x008d0080, 0x006e0085, 0x008c007f, 0x006f0086,
	0x008b007f, 0x00700086, 0x008b007e, 0x00710086,
	0x008b007d, 0x00720086, 0x008a007d, 0x00730086,
	0x008a007c, 0x00730087, 0x008a007b, 0x00740087,
	0x0089007b, 0x00750087, 0x008a0079, 0x00750088,
	0x008a0078, 0x00760088, 0x008a0077, 0x00770088,
	0x00880077, 0x0077008a, 0x00880076, 0x0078008a,
	0x00880075, 0x0079008a, 0x00870075, 0x007b0089,
	0x00870074, 0x007b008a, 0x00870073, 0x007c008a,
	0x00860073, 0x007d008a, 0x00860072, 0x007d008b,
	0x00860071, 0x007e008b, 0x00860070, 0x007f008b,
	0x0086006f, 0x007f008c, 0x0085006e, 0x0080008d,
	0x0085006d, 0x0081008d, 0x0084006d, 0x0082008d,
	0x0084006c, 0x0083008d, 0x0084006b, 0x0083008e,

	0x023c0fe2, 0x00000fe2, 0x023a0fdb, 0x00000feb,
	0x02360fd3, 0x0fff0ff8, 0x022e0fcf, 0x0ffc0007,
	0x02250fca, 0x0ffa0017, 0x021a0fc6, 0x0ff70029,
	0x020c0fc4, 0x0ff4003c, 0x01fd0fc1, 0x0ff10051,
	0x01eb0fc0, 0x0fed0068, 0x01d80fc0, 0x0fe9007f,
	0x01c30fc1, 0x0fe50097, 0x01ac0fc2, 0x0fe200b0,
	0x01960fc3, 0x0fdd00ca, 0x017e0fc5, 0x0fd900e4,
	0x01650fc8, 0x0fd500fe, 0x014b0fcb, 0x0fd20118,
	0x01330fcd, 0x0fcd0133, 0x01180fd2, 0x0fcb014b,
	0x00fe0fd5, 0x0fc80165, 0x00e40fd9, 0x0fc5017e,
	0x00ca0fdd, 0x0fc30196, 0x00b00fe2, 0x0fc201ac,
	0x00970fe5, 0x0fc101c3, 0x007f0fe9, 0x0fc001d8,
	0x00680fed, 0x0fc001eb, 0x00510ff1, 0x0fc101fd,
	0x003c0ff4, 0x0fc4020c, 0x00290ff7, 0x0fc6021a,
	0x00170ffa, 0x0fca0225, 0x00070ffc, 0x0fcf022e,
	0x0ff80fff, 0x0fd30236, 0x0feb0000, 0x0fdb023a,

	0x02780fc4, 0x00000fc4, 0x02770fbc, 0x0fff0fce,
	0x02710fb5, 0x0ffe0fdc, 0x02690fb0, 0x0ffa0fed,
	0x025f0fab, 0x0ff70fff, 0x02500fa8, 0x0ff30015,
	0x02410fa6, 0x0fef002a, 0x022f0fa4, 0x0feb0042,
	0x021a0fa4, 0x0fe5005d, 0x02040fa5, 0x0fe10076,
	0x01eb0fa7, 0x0fdb0093, 0x01d20fa9, 0x0fd600af,
	0x01b80fab, 0x0fd000cd, 0x019d0faf, 0x0fca00ea,
	0x01810fb2, 0x0fc50108, 0x01620fb7, 0x0fc10126,
	0x01440fbb, 0x0fbb0146, 0x01260fc1, 0x0fb70162,
	0x01080fc5, 0x0fb20181, 0x00ea0fca, 0x0faf019d,
	0x00cd0fd0, 0x0fab01b8, 0x00af0fd6, 0x0fa901d2,
	0x00930fdb, 0x0fa701eb, 0x00760fe1, 0x0fa50204,
	0x005d0fe5, 0x0fa4021a, 0x00420feb, 0x0fa4022f,
	0x002a0fef, 0x0fa60241, 0x00150ff3, 0x0fa80250,
	0x0fff0ff7, 0x0fab025f, 0x0fed0ffa, 0x0fb00269,
	0x0fdc0ffe, 0x0fb50271, 0x0fce0fff, 0x0fbc0277,

	0x02a00fb0, 0x00000fb0, 0x029e0fa8, 0x0fff0fbb,
	0x02980fa1, 0x0ffd0fca, 0x028f0f9c, 0x0ff90fdc,
	0x02840f97, 0x0ff50ff0, 0x02740f94, 0x0ff10007,
	0x02640f92, 0x0fec001e, 0x02500f91, 0x0fe70038,
	0x023a0f91, 0x0fe00055, 0x02220f92, 0x0fdb0071,
	0x02080f95, 0x0fd4008f, 0x01ec0f98, 0x0fce00ae,
	0x01cf0f9b, 0x0fc700cf, 0x01b10f9f, 0x0fc100ef,
	0x01920fa4, 0x0fbb010f, 0x01710faa, 0x0fb50130,
	0x01520fae, 0x0fae0152, 0x01300fb5, 0x0faa0171,
	0x010f0fbb, 0x0fa40192, 0x00ef0fc1, 0x0f9f01b1,
	0x00cf0fc7, 0x0f9b01cf, 0x00ae0fce, 0x0f9801ec,
	0x008f0fd4, 0x0f950208, 0x00710fdb, 0x0f920222,
	0x00550fe0, 0x0f91023a, 0x00380fe7, 0x0f910250,
	0x001e0fec, 0x0f920264, 0x00070ff1, 0x0f940274,
	0x0ff00ff5, 0x0f970284, 0x0fdc0ff9, 0x0f9c028f,
	0x0fca0ffd, 0x0fa10298, 0x0fbb0fff, 0x0fa8029e,

	0x02c80f9c, 0x00000f9c, 0x02c70f94, 0x0ffe0fa7,
	0x02c10f8c, 0x0ffc0fb7, 0x02b70f87, 0x0ff70fcb,
	0x02aa0f83, 0x0ff30fe0, 0x02990f80, 0x0fee0ff9,
	0x02870f7f, 0x0fe80012, 0x02720f7e, 0x0fe2002e,
	0x025a0f7e, 0x0fdb004d, 0x02400f80, 0x0fd5006b,
	0x02230f84, 0x0fcd008c, 0x02050f87, 0x0fc700ad,
	0x01e60f8b, 0x0fbf00d0, 0x01c60f90, 0x0fb700f3,
	0x01a30f96, 0x0fb00117, 0x01800f9c, 0x0faa013a,
	0x015d0fa2, 0x0fa2015f, 0x013a0faa, 0x0f9c0180,
	0x01170fb0, 0x0f9601a3, 0x00f30fb7, 0x0f9001c6,
	0x00d00fbf, 0x0f8b01e6, 0x00ad0fc7, 0x0f870205,
	0x008c0fcd, 0x0f840223, 0x006b0fd5, 0x0f800240,
	0x004d0fdb, 0x0f7e025a, 0x002e0fe2, 0x0f7e0272,
	0x00120fe8, 0x0f7f0287, 0x0ff90fee, 0x0f800299,
	0x0fe00ff3, 0x0f8302aa, 0x0fcb0ff7, 0x0f8702b7,
	0x0fb70ffc, 0x0f8c02c1, 0x0fa70ffe, 0x0f9402c7,

	0x02f00f88, 0x00000f88, 0x02ee0f80, 0x0ffe0f94,
	0x02e70f78, 0x0ffc0fa5, 0x02dd0f73, 0x0ff60fba,
	0x02ce0f6f, 0x0ff20fd1, 0x02be0f6c, 0x0feb0feb,
	0x02aa0f6b, 0x0fe50006, 0x02940f6a, 0x0fde0024,
	0x02790f6c, 0x0fd60045, 0x025e0f6e, 0x0fcf0065,
	0x023f0f72, 0x0fc60089, 0x021d0f77, 0x0fbf00ad,
	0x01fd0f7b, 0x0fb600d2, 0x01da0f81, 0x0fad00f8,
	0x01b50f87, 0x0fa6011e, 0x018f0f8f, 0x0f9e0144,
	0x016b0f95, 0x0f95016b, 0x01440f9e, 0x0f8f018f,
	0x011e0fa6, 0x0f8701b5, 0x00f80fad, 0x0f8101da,
	0x00d20fb6, 0x0f7b01fd, 0x00ad0fbf, 0x0f77021d,
	0x00890fc6, 0x0f72023f, 0x00650fcf, 0x0f6e025e,
	0x00450fd6, 0x0f6c0279, 0x00240fde, 0x0f6a0294,
	0x00060fe5, 0x0f6b02aa, 0x0feb0feb, 0x0f6c02be,
	0x0fd10ff2, 0x0f6f02ce, 0x0fba0ff6, 0x0f7302dd,
	0x0fa50ffc, 0x0f7802e7, 0x0f940ffe, 0x0f8002ee,

	0x03180f74, 0x00000f74, 0x03160f6b, 0x0ffe0f81,
	0x030e0f64, 0x0ffb0f93, 0x03030f5f, 0x0ff50fa9,
	0x02f40f5b, 0x0ff00fc1, 0x02e20f58, 0x0fe90fdd,
	0x02cd0f57, 0x0fe20ffa, 0x02b60f57, 0x0fda0019,
	0x02990f59, 0x0fd1003d, 0x027b0f5c, 0x0fc90060,
	0x02590f61, 0x0fc00086, 0x02370f66, 0x0fb700ac,
	0x02130f6b, 0x0fae00d4, 0x01ee0f72, 0x0fa400fc,
	0x01c70f79, 0x0f9b0125, 0x019f0f81, 0x0f93014d,
	0x01760f89, 0x0f890178, 0x014d0f93, 0x0f81019f,
	0x01250f9b, 0x0f7901c7, 0x00fc0fa4, 0x0f7201ee,
	0x00d40fae, 0x0f6b0213, 0x00ac0fb7, 0x0f660237,
	0x00860fc0, 0x0f610259, 0x00600fc9, 0x0f5c027b,
	0x003d0fd1, 0x0f590299, 0x00190fda, 0x0f5702b6,
	0x0ffa0fe2, 0x0f5702cd, 0x0fdd0fe9, 0x0f5802e2,
	0x0fc10ff0, 0x0f5b02f4, 0x0fa90ff5, 0x0f5f0303,
	0x0f930ffb, 0x0f64030e, 0x0f810ffe, 0x0f6b0316,

	0x03400f60, 0x00000f60, 0x033e0f57, 0x0ffe0f6d,
	0x03370f4f, 0x0ffa0f80, 0x032a0f4b, 0x0ff30f98,
	0x031a0f46, 0x0fee0fb2, 0x03070f44, 0x0fe60fcf,
	0x02f10f44, 0x0fde0fed, 0x02d70f44, 0x0fd6000f,
	0x02b80f46, 0x0fcc0036, 0x02990f4a, 0x0fc3005a,
	0x02750f4f, 0x0fb90083, 0x02500f55, 0x0fb000ab,
	0x022a0f5b, 0x0fa500d6, 0x02020f63, 0x0f9a0101,
	0x01d80f6b, 0x0f91012c, 0x01ae0f74, 0x0f870157,
	0x01840f7c, 0x0f7c0184, 0x01570f87, 0x0f7401ae,
	0x012c0f91, 0x0f6b01d8, 0x01010f9a, 0x0f630202,
	0x00d60fa5, 0x0f5b022a, 0x00ab0fb0, 0x0f550250,
	0x00830fb9, 0x0f4f0275, 0x005a0fc3, 0x0f4a0299,
	0x00360fcc, 0x0f4602b8, 0x000f0fd6, 0x0f4402d7,
	0x0fed0fde, 0x0f4402f1, 0x0fcf0fe6, 0x0f440307,
	0x0fb20fee, 0x0f46031a, 0x0f980ff3, 0x0f4b032a,
	0x0f800ffa, 0x0f4f0337, 0x0f6d0ffe, 0x0f57033e
};


#define MDP4_QSEED_TABLE0_OFF 0x8100
#define MDP4_QSEED_TABLE1_OFF 0x8200
#define MDP4_QSEED_TABLE2_OFF 0x9000

void mdp4_vg_qseed_init(int vp_num)
{
	uint32 *off;
	int i, voff;

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);

	voff = MDP4_VIDEO_OFF * vp_num;
	off = (uint32 *)(MDP_BASE + MDP4_VIDEO_BASE + voff +
						MDP4_QSEED_TABLE0_OFF);
	for (i = 0; i < (sizeof(vg_qseed_table0) / sizeof(uint32)); i++) {
		outpdw(off, vg_qseed_table0[i]);
		off++;
		/* This code is added to workaround the 1K Boundary AXI
		Interleave operations from Scorpion that can potentially
		corrupt the QSEED table. The idea is to complete the prevous
		to the buffer before making the next write when address is
		1KB aligned to ensure the write has been committed prior to
		next instruction write that can go out from  the secondary AXI
		port.This happens also because of the expected write sequence
		from QSEED table, where LSP has to be written first then the
		MSP to trigger both to write out to SRAM, if this has not been
		the expectation, then corruption wouldn't have happened.*/

		if (!((uint32)off & 0x3FF))
			wmb();
	}

	off = (uint32 *)(MDP_BASE + MDP4_VIDEO_BASE + voff +
						MDP4_QSEED_TABLE1_OFF);
	for (i = 0; i < (sizeof(vg_qseed_table1) / sizeof(uint32)); i++) {
		outpdw(off, vg_qseed_table1[i]);
		off++;
		if (!((uint32)off & 0x3FF))
			wmb();
	}

	off = (uint32 *)(MDP_BASE + MDP4_VIDEO_BASE + voff +
						MDP4_QSEED_TABLE2_OFF);
	for (i = 0; i < (sizeof(vg_qseed_table2) / sizeof(uint32)); i++) {
		outpdw(off, vg_qseed_table2[i]);
		off++;
		if (!((uint32)off & 0x3FF))
			wmb();
	}

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);

}

void mdp4_mixer_blend_init(mixer_num)
{
	unsigned char *overlay_base;
	int off;

	if (mixer_num) 	/* mixer number, /dev/fb0, /dev/fb1 */
		overlay_base = MDP_BASE + MDP4_OVERLAYPROC1_BASE;/* 0x18000 */
	else
		overlay_base = MDP_BASE + MDP4_OVERLAYPROC0_BASE;/* 0x10000 */

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);

	/* stage 0 to stage 2 */
	off = 0;
	outpdw(overlay_base + off + 0x104, 0x010);
	outpdw(overlay_base + off + 0x108, 0xff);/* FG */
	outpdw(overlay_base + off + 0x10c, 0x00);/* BG */

	off += 0x20;
	outpdw(overlay_base + off + 0x104, 0x010);
	outpdw(overlay_base + off + 0x108, 0xff);/* FG */
	outpdw(overlay_base + off + 0x10c, 0x00);/* BG */

	off += 0x20;
	outpdw(overlay_base + off + 0x104, 0x010);
	outpdw(overlay_base + off + 0x108, 0xff);/* FG */
	outpdw(overlay_base + off + 0x10c, 0x00);/* BG */

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
}

struct mdp_csc_cfg mdp_csc_convert[4] = {
	{ /*RGB2RGB*/
		0,
		{
			0x0200, 0x0000, 0x0000,
			0x0000, 0x0200, 0x0000,
			0x0000, 0x0000, 0x0200,
		},
		{ 0x0, 0x0, 0x0, },
		{ 0x0, 0x0, 0x0, },
		{ 0x0, 0xff, 0x0, 0xff, 0x0, 0xff, },
		{ 0x0, 0xff, 0x0, 0xff, 0x0, 0xff, },
	},
	{ /*YUV2RGB*/
		0,
		{
			0x0254, 0x0000, 0x0331,
			0x0254, 0xff37, 0xfe60,
			0x0254, 0x0409, 0x0000,
		},
		{ 0xfff0, 0xff80, 0xff80, },
		{ 0x0, 0x0, 0x0, },
		{ 0x0, 0xff, 0x0, 0xff, 0x0, 0xff, },
		{ 0x0, 0xff, 0x0, 0xff, 0x0, 0xff, },
	},
	{ /*RGB2YUV*/
		0,
		{
			0x0083, 0x0102, 0x0032,
			0x1fb5, 0x1f6c, 0x00e1,
			0x00e1, 0x1f45, 0x1fdc
		},
		{ 0x0, 0x0, 0x0, },
		{ 0x0010, 0x0080, 0x0080, },
		{ 0x0, 0xff, 0x0, 0xff, 0x0, 0xff, },
		{ 0x0010, 0x00eb, 0x0010, 0x00f0, 0x0010, 0x00f0, },
	},
	{ /*YUV2YUV ???*/
		0,
		{
			0x0200, 0x0000, 0x0000,
			0x0000, 0x0200, 0x0000,
			0x0000, 0x0000, 0x0200,
		},
		{ 0x0, 0x0, 0x0, },
		{ 0x0, 0x0, 0x0, },
		{ 0x0, 0xff, 0x0, 0xff, 0x0, 0xff, },
		{ 0x0, 0xff, 0x0, 0xff, 0x0, 0xff, },
	},
};

struct mdp_csc_cfg csc_matrix[3] = {
	{
		(MDP_CSC_FLAG_YUV_OUT),
		{
			0x0254, 0x0000, 0x0331,
			0x0254, 0xff37, 0xfe60,
			0x0254, 0x0409, 0x0000,
		},
		{
			0xfff0, 0xff80, 0xff80,
		},
		{
			0, 0, 0,
		},
		{
			0, 0xff, 0, 0xff, 0, 0xff,
		},
		{
			0, 0xff, 0, 0xff, 0, 0xff,
		},
	},
	{
		(MDP_CSC_FLAG_YUV_OUT),
		{
			0x0254, 0x0000, 0x0331,
			0x0254, 0xff37, 0xfe60,
			0x0254, 0x0409, 0x0000,
		},
		{
			0xfff0, 0xff80, 0xff80,
		},
		{
			0, 0, 0,
		},
		{
			0, 0xff, 0, 0xff, 0, 0xff,
		},
		{
			0, 0xff, 0, 0xff, 0, 0xff,
		},
	},
	{
		(0),
		{
			0x0200, 0x0000, 0x0000,
			0x0000, 0x0200, 0x0000,
			0x0000, 0x0000, 0x0200,
		},
		{
			0x0, 0x0, 0x0,
		},
		{
			0, 0, 0,
		},
		{
			0, 0xff, 0, 0xff, 0, 0xff,
		},
		{
			0, 0xff, 0, 0xff, 0, 0xff,
		},
	},
};



#define MDP4_CSC_MV_OFF 	0x4400
#define MDP4_CSC_PRE_BV_OFF 	0x4500
#define MDP4_CSC_POST_BV_OFF 	0x4580
#define MDP4_CSC_PRE_LV_OFF 	0x4600
#define MDP4_CSC_POST_LV_OFF 	0x4680

void mdp4_vg_csc_mv_setup(int vp_num)
{
	uint32 *off;
	int i, voff;

	voff = MDP4_VIDEO_OFF * vp_num;
	off = (uint32 *)(MDP_BASE + MDP4_VIDEO_BASE + voff +
					MDP4_CSC_MV_OFF);

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);
	for (i = 0; i < 9; i++) {
		outpdw(off, csc_matrix[vp_num].csc_mv[i]);
		off++;
	}
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
}

void mdp4_vg_csc_pre_bv_setup(int vp_num)
{
	uint32 *off;
	int i, voff;

	voff = MDP4_VIDEO_OFF * vp_num;
	off = (uint32 *)(MDP_BASE + MDP4_VIDEO_BASE + voff +
					MDP4_CSC_PRE_BV_OFF);

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);
	for (i = 0; i < 3; i++) {
		outpdw(off, csc_matrix[vp_num].csc_pre_bv[i]);
		off++;
	}
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
}

void mdp4_vg_csc_post_bv_setup(int vp_num)
{
	uint32 *off;
	int i, voff;

	voff = MDP4_VIDEO_OFF * vp_num;
	off = (uint32 *)(MDP_BASE + MDP4_VIDEO_BASE + voff +
					MDP4_CSC_POST_BV_OFF);

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);
	for (i = 0; i < 3; i++) {
		outpdw(off, csc_matrix[vp_num].csc_post_bv[i]);
		off++;
	}
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
}

void mdp4_vg_csc_pre_lv_setup(int vp_num)
{
	uint32 *off;
	int i, voff;

	voff = MDP4_VIDEO_OFF * vp_num;
	off = (uint32 *)(MDP_BASE + MDP4_VIDEO_BASE + voff +
					MDP4_CSC_PRE_LV_OFF);

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);
	for (i = 0; i < 6; i++) {
		outpdw(off, csc_matrix[vp_num].csc_pre_lv[i]);
		off++;
	}
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
}

void mdp4_vg_csc_post_lv_setup(int vp_num)
{
	uint32 *off;
	int i, voff;

	voff = MDP4_VIDEO_OFF * vp_num;
	off = (uint32 *)(MDP_BASE + MDP4_VIDEO_BASE + voff +
					MDP4_CSC_POST_LV_OFF);

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);
	for (i = 0; i < 6; i++) {
		outpdw(off, csc_matrix[vp_num].csc_post_lv[i]);
		off++;
	}
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
}

void mdp4_vg_csc_convert_setup(int vp_num)
{
	struct mdp_csc_cfg_data cfg;

	switch (vp_num) {
	case 0:
		cfg.block = MDP_BLOCK_VG_1;
		break;
	case 1:
		cfg.block = MDP_BLOCK_VG_2;
		break;
	default:
		pr_err("%s - invalid vp_num = %d", __func__, vp_num);
		return;
	}
	cfg.csc_data = csc_matrix[vp_num];
	mdp4_csc_enable(&cfg);
}

void mdp4_vg_csc_setup(int vp_num)
{
		/* yuv2rgb */
		mdp4_vg_csc_mv_setup(vp_num);
		mdp4_vg_csc_pre_bv_setup(vp_num);
		mdp4_vg_csc_post_bv_setup(vp_num);
		mdp4_vg_csc_pre_lv_setup(vp_num);
		mdp4_vg_csc_post_lv_setup(vp_num);
		mdp4_vg_csc_convert_setup(vp_num);
}
void mdp4_vg_csc_update(struct mdp_csc *p)
{
	struct mdp4_overlay_pipe *pipe;
	int vp_num;

	pipe = mdp4_overlay_ndx2pipe(p->id);
	if (pipe == NULL) {
		pr_err("%s: p->id = %d Error\n", __func__, p->id);
		return;
	}

	vp_num = pipe->pipe_num - OVERLAY_PIPE_VG1;

	if (vp_num == 0 || vp_num == 1) {
		memcpy(csc_matrix[vp_num].csc_mv, p->csc_mv, sizeof(p->csc_mv));
		memcpy(csc_matrix[vp_num].csc_pre_bv, p->csc_pre_bv,
			sizeof(p->csc_pre_bv));
		memcpy(csc_matrix[vp_num].csc_post_bv, p->csc_post_bv,
			sizeof(p->csc_post_bv));
		memcpy(csc_matrix[vp_num].csc_pre_lv, p->csc_pre_lv,
			sizeof(p->csc_pre_lv));
		memcpy(csc_matrix[vp_num].csc_post_lv, p->csc_post_lv,
			sizeof(p->csc_post_lv));
		mdp4_vg_csc_setup(vp_num);
	}
}
static uint32 csc_rgb2yuv_matrix_tab[9] = {
	0x0083, 0x0102, 0x0032,
	0x1fb5, 0x1f6c, 0x00e1,
	0x00e1, 0x1f45, 0x1fdc
};

static uint32 csc_rgb2yuv_pre_bv_tab[3] = {0, 0, 0};

static uint32 csc_rgb2yuv_post_bv_tab[3] = {0x0010, 0x0080, 0x0080};

static  uint32 csc_rgb2yuv_pre_lv_tab[6] = {
	0x00, 0xff, 0x00,
	0xff, 0x00, 0xff
};

static  uint32 csc_rgb2yuv_post_lv_tab[6] = {
	0x0010, 0x00eb, 0x0010,
	0x00f0, 0x0010, 0x00f0
};

void mdp4_mixer_csc_mv_setup(uint32 mixer)
{
	uint32 *off;
	int i;

	if (mixer == MDP4_MIXER1)
		off = (uint32 *)(MDP_BASE + MDP4_OVERLAYPROC1_BASE + 0x2400);
	else
		off = (uint32 *)(MDP_BASE + MDP4_OVERLAYPROC2_BASE + 0x2400);

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);
	for (i = 0; i < 9; i++) {
		outpdw(off, csc_rgb2yuv_matrix_tab[i]);
		off++;
	}
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
}

void mdp4_mixer_csc_pre_bv_setup(uint32 mixer)
{
	uint32 *off;
	int i;

	if (mixer == MDP4_MIXER1)
		off = (uint32 *)(MDP_BASE + MDP4_OVERLAYPROC1_BASE + 0x2500);
	else
		off = (uint32 *)(MDP_BASE + MDP4_OVERLAYPROC2_BASE + 0x2500);

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);
	for (i = 0; i < 3; i++) {
		outpdw(off, csc_rgb2yuv_pre_bv_tab[i]);
		off++;
	}
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
}

void mdp4_mixer_csc_post_bv_setup(uint32 mixer)
{
	uint32 *off;
	int i;

	if (mixer == MDP4_MIXER1)
		off = (uint32 *)(MDP_BASE + MDP4_OVERLAYPROC1_BASE + 0x2580);
	else
		off = (uint32 *)(MDP_BASE + MDP4_OVERLAYPROC2_BASE + 0x2580);

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);
	for (i = 0; i < 3; i++) {
		outpdw(off, csc_rgb2yuv_post_bv_tab[i]);
		off++;
	}
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
}

void mdp4_mixer_csc_pre_lv_setup(uint32 mixer)
{
	uint32 *off;
	int i;

	if (mixer == MDP4_MIXER1)
		off = (uint32 *)(MDP_BASE + MDP4_OVERLAYPROC1_BASE + 0x2600);
	else
		off = (uint32 *)(MDP_BASE + MDP4_OVERLAYPROC2_BASE + 0x2600);

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);
	for (i = 0; i < 6; i++) {
		outpdw(off, csc_rgb2yuv_pre_lv_tab[i]);
		off++;
	}
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
}

void mdp4_mixer_csc_post_lv_setup(uint32 mixer)
{
	uint32 *off;
	int i;

	if (mixer == MDP4_MIXER1)
		off = (uint32 *)(MDP_BASE + MDP4_OVERLAYPROC1_BASE + 0x2680);
	else
		off = (uint32 *)(MDP_BASE + MDP4_OVERLAYPROC2_BASE + 0x2680);

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);
	for (i = 0; i < 6; i++) {
		outpdw(off, csc_rgb2yuv_post_lv_tab[i]);
		off++;
	}
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
}

void mdp4_mixer_csc_setup(uint32 mixer)
{
	if (mixer >= MDP4_MIXER1) {
		/* rgb2yuv */
		mdp4_mixer_csc_mv_setup(mixer);
		mdp4_mixer_csc_pre_bv_setup(mixer);
		mdp4_mixer_csc_post_bv_setup(mixer);
		mdp4_mixer_csc_pre_lv_setup(mixer);
		mdp4_mixer_csc_post_lv_setup(mixer);
	}
}

#define DMA_P_BASE 0x90000
void mdp4_dmap_csc_mv_setup(void)
{
	uint32 *off;
	int i;

	off = (uint32 *)(MDP_BASE + DMA_P_BASE + 0x3400);

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);
	for (i = 0; i < 9; i++) {
		outpdw(off, csc_matrix[2].csc_mv[i]);
		off++;
	}
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
}

void mdp4_dmap_csc_pre_bv_setup(void)
{
	uint32 *off;
	int i;

	off = (uint32 *)(MDP_BASE + DMA_P_BASE + 0x3500);

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);
	for (i = 0; i < 3; i++) {
		outpdw(off, csc_matrix[2].csc_pre_bv[i]);
		off++;
	}
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
}

void mdp4_dmap_csc_post_bv_setup(void)
{
	uint32 *off;
	int i;

	off = (uint32 *)(MDP_BASE + DMA_P_BASE + 0x3580);

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);
	for (i = 0; i < 3; i++) {
		outpdw(off, csc_matrix[2].csc_post_bv[i]);
		off++;
	}
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
}

void mdp4_dmap_csc_pre_lv_setup(void)
{
	uint32 *off;
	int i;

	off = (uint32 *)(MDP_BASE + DMA_P_BASE + 0x3600);

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);
	for (i = 0; i < 6; i++) {
		outpdw(off, csc_matrix[2].csc_pre_lv[i]);
		off++;
	}
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
}

void mdp4_dmap_csc_post_lv_setup(void)
{
	uint32 *off;
	int i;

	off = (uint32 *)(MDP_BASE + DMA_P_BASE + 0x3680);

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);
	for (i = 0; i < 6; i++) {
		outpdw(off, csc_matrix[2].csc_post_lv[i]);
		off++;
	}
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
}

void mdp4_dmap_csc_setup(void)
{
	mdp4_dmap_csc_mv_setup();
	mdp4_dmap_csc_pre_bv_setup();
	mdp4_dmap_csc_post_bv_setup();
	mdp4_dmap_csc_pre_lv_setup();
	mdp4_dmap_csc_post_lv_setup();
}

char gc_lut[] = {
	0x0, 0x1, 0x2, 0x2, 0x3, 0x4, 0x5, 0x6,
	0x6, 0x7, 0x8, 0x9, 0xA, 0xA, 0xB, 0xC,
	0xD, 0xD, 0xE, 0xF, 0xF, 0x10, 0x10, 0x11,
	0x12, 0x12, 0x13, 0x13, 0x14, 0x14, 0x15, 0x15,
	0x16, 0x16, 0x17, 0x17, 0x17, 0x18, 0x18, 0x19,
	0x19, 0x19, 0x1A, 0x1A, 0x1B, 0x1B, 0x1B, 0x1C,
	0x1C, 0x1D, 0x1D, 0x1D, 0x1E, 0x1E, 0x1E, 0x1F,
	0x1F, 0x1F, 0x20, 0x20, 0x20, 0x21, 0x21, 0x21,
	0x22, 0x22, 0x22, 0x22, 0x23, 0x23, 0x23, 0x24,
	0x24, 0x24, 0x25, 0x25, 0x25, 0x25, 0x26, 0x26,
	0x26, 0x26, 0x27, 0x27, 0x27, 0x28, 0x28, 0x28,
	0x28, 0x29, 0x29, 0x29, 0x29, 0x2A, 0x2A, 0x2A,
	0x2A, 0x2B, 0x2B, 0x2B, 0x2B, 0x2B, 0x2C, 0x2C,
	0x2C, 0x2C, 0x2D, 0x2D, 0x2D, 0x2D, 0x2E, 0x2E,
	0x2E, 0x2E, 0x2E, 0x2F, 0x2F, 0x2F, 0x2F, 0x30,
	0x30, 0x30, 0x30, 0x30, 0x31, 0x31, 0x31, 0x31,
	0x31, 0x32, 0x32, 0x32, 0x32, 0x32, 0x33, 0x33,
	0x33, 0x33, 0x33, 0x34, 0x34, 0x34, 0x34, 0x34,
	0x35, 0x35, 0x35, 0x35, 0x35, 0x36, 0x36, 0x36,
	0x36, 0x36, 0x37, 0x37, 0x37, 0x37, 0x37, 0x37,
	0x38, 0x38, 0x38, 0x38, 0x38, 0x39, 0x39, 0x39,
	0x39, 0x39, 0x39, 0x3A, 0x3A, 0x3A, 0x3A, 0x3A,
	0x3A, 0x3B, 0x3B, 0x3B, 0x3B, 0x3B, 0x3B, 0x3C,
	0x3C, 0x3C, 0x3C, 0x3C, 0x3C, 0x3D, 0x3D, 0x3D,
	0x3D, 0x3D, 0x3D, 0x3E, 0x3E, 0x3E, 0x3E, 0x3E,
	0x3E, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x40,
	0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x41, 0x41,
	0x41, 0x41, 0x41, 0x41, 0x42, 0x42, 0x42, 0x42,
	0x42, 0x42, 0x42, 0x43, 0x43, 0x43, 0x43, 0x43,
	0x43, 0x43, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
	0x44, 0x45, 0x45, 0x45, 0x45, 0x45, 0x45, 0x45,
	0x46, 0x46, 0x46, 0x46, 0x46, 0x46, 0x46, 0x47,
	0x47, 0x47, 0x47, 0x47, 0x47, 0x47, 0x48, 0x48,
	0x48, 0x48, 0x48, 0x48, 0x48, 0x48, 0x49, 0x49,
	0x49, 0x49, 0x49, 0x49, 0x49, 0x4A, 0x4A, 0x4A,
	0x4A, 0x4A, 0x4A, 0x4A, 0x4A, 0x4B, 0x4B, 0x4B,
	0x4B, 0x4B, 0x4B, 0x4B, 0x4B, 0x4C, 0x4C, 0x4C,
	0x4C, 0x4C, 0x4C, 0x4C, 0x4D, 0x4D, 0x4D, 0x4D,
	0x4D, 0x4D, 0x4D, 0x4D, 0x4E, 0x4E, 0x4E, 0x4E,
	0x4E, 0x4E, 0x4E, 0x4E, 0x4E, 0x4F, 0x4F, 0x4F,
	0x4F, 0x4F, 0x4F, 0x4F, 0x4F, 0x50, 0x50, 0x50,
	0x50, 0x50, 0x50, 0x50, 0x50, 0x51, 0x51, 0x51,
	0x51, 0x51, 0x51, 0x51, 0x51, 0x51, 0x52, 0x52,
	0x52, 0x52, 0x52, 0x52, 0x52, 0x52, 0x53, 0x53,
	0x53, 0x53, 0x53, 0x53, 0x53, 0x53, 0x53, 0x54,
	0x54, 0x54, 0x54, 0x54, 0x54, 0x54, 0x54, 0x54,
	0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
	0x55, 0x56, 0x56, 0x56, 0x56, 0x56, 0x56, 0x56,
	0x56, 0x56, 0x57, 0x57, 0x57, 0x57, 0x57, 0x57,
	0x57, 0x57, 0x57, 0x58, 0x58, 0x58, 0x58, 0x58,
	0x58, 0x58, 0x58, 0x58, 0x58, 0x59, 0x59, 0x59,
	0x59, 0x59, 0x59, 0x59, 0x59, 0x59, 0x5A, 0x5A,
	0x5A, 0x5A, 0x5A, 0x5A, 0x5A, 0x5A, 0x5A, 0x5A,
	0x5B, 0x5B, 0x5B, 0x5B, 0x5B, 0x5B, 0x5B, 0x5B,
	0x5B, 0x5B, 0x5C, 0x5C, 0x5C, 0x5C, 0x5C, 0x5C,
	0x5C, 0x5C, 0x5C, 0x5C, 0x5D, 0x5D, 0x5D, 0x5D,
	0x5D, 0x5D, 0x5D, 0x5D, 0x5D, 0x5D, 0x5E, 0x5E,
	0x5E, 0x5E, 0x5E, 0x5E, 0x5E, 0x5E, 0x5E, 0x5E,
	0x5F, 0x5F, 0x5F, 0x5F, 0x5F, 0x5F, 0x5F, 0x5F,
	0x5F, 0x5F, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60,
	0x60, 0x60, 0x60, 0x60, 0x60, 0x61, 0x61, 0x61,
	0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x61, 0x62,
	0x62, 0x62, 0x62, 0x62, 0x62, 0x62, 0x62, 0x62,
	0x62, 0x62, 0x63, 0x63, 0x63, 0x63, 0x63, 0x63,
	0x63, 0x63, 0x63, 0x63, 0x63, 0x64, 0x64, 0x64,
	0x64, 0x64, 0x64, 0x64, 0x64, 0x64, 0x64, 0x64,
	0x65, 0x65, 0x65, 0x65, 0x65, 0x65, 0x65, 0x65,
	0x65, 0x65, 0x65, 0x66, 0x66, 0x66, 0x66, 0x66,
	0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x67, 0x67,
	0x67, 0x67, 0x67, 0x67, 0x67, 0x67, 0x67, 0x67,
	0x67, 0x67, 0x68, 0x68, 0x68, 0x68, 0x68, 0x68,
	0x68, 0x68, 0x68, 0x68, 0x68, 0x69, 0x69, 0x69,
	0x69, 0x69, 0x69, 0x69, 0x69, 0x69, 0x69, 0x69,
	0x69, 0x6A, 0x6A, 0x6A, 0x6A, 0x6A, 0x6A, 0x6A,
	0x6A, 0x6A, 0x6A, 0x6A, 0x6A, 0x6B, 0x6B, 0x6B,
	0x6B, 0x6B, 0x6B, 0x6B, 0x6B, 0x6B, 0x6B, 0x6B,
	0x6B, 0x6C, 0x6C, 0x6C, 0x6C, 0x6C, 0x6C, 0x6C,
	0x6C, 0x6C, 0x6C, 0x6C, 0x6C, 0x6D, 0x6D, 0x6D,
	0x6D, 0x6D, 0x6D, 0x6D, 0x6D, 0x6D, 0x6D, 0x6D,
	0x6D, 0x6E, 0x6E, 0x6E, 0x6E, 0x6E, 0x6E, 0x6E,
	0x6E, 0x6E, 0x6E, 0x6E, 0x6E, 0x6F, 0x6F, 0x6F,
	0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F,
	0x6F, 0x6F, 0x70, 0x70, 0x70, 0x70, 0x70, 0x70,
	0x70, 0x70, 0x70, 0x70, 0x70, 0x70, 0x71, 0x71,
	0x71, 0x71, 0x71, 0x71, 0x71, 0x71, 0x71, 0x71,
	0x71, 0x71, 0x71, 0x72, 0x72, 0x72, 0x72, 0x72,
	0x72, 0x72, 0x72, 0x72, 0x72, 0x72, 0x72, 0x72,
	0x73, 0x73, 0x73, 0x73, 0x73, 0x73, 0x73, 0x73,
	0x73, 0x73, 0x73, 0x73, 0x73, 0x74, 0x74, 0x74,
	0x74, 0x74, 0x74, 0x74, 0x74, 0x74, 0x74, 0x74,
	0x74, 0x74, 0x75, 0x75, 0x75, 0x75, 0x75, 0x75,
	0x75, 0x75, 0x75, 0x75, 0x75, 0x75, 0x75, 0x75,
	0x76, 0x76, 0x76, 0x76, 0x76, 0x76, 0x76, 0x76,
	0x76, 0x76, 0x76, 0x76, 0x76, 0x77, 0x77, 0x77,
	0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77,
	0x77, 0x77, 0x77, 0x78, 0x78, 0x78, 0x78, 0x78,
	0x78, 0x78, 0x78, 0x78, 0x78, 0x78, 0x78, 0x78,
	0x78, 0x79, 0x79, 0x79, 0x79, 0x79, 0x79, 0x79,
	0x79, 0x79, 0x79, 0x79, 0x79, 0x79, 0x7A, 0x7A,
	0x7A, 0x7A, 0x7A, 0x7A, 0x7A, 0x7A, 0x7A, 0x7A,
	0x7A, 0x7A, 0x7A, 0x7A, 0x7A, 0x7B, 0x7B, 0x7B,
	0x7B, 0x7B, 0x7B, 0x7B, 0x7B, 0x7B, 0x7B, 0x7B,
	0x7B, 0x7B, 0x7B, 0x7C, 0x7C, 0x7C, 0x7C, 0x7C,
	0x7C, 0x7C, 0x7C, 0x7C, 0x7C, 0x7C, 0x7C, 0x7C,
	0x7C, 0x7D, 0x7D, 0x7D, 0x7D, 0x7D, 0x7D, 0x7D,
	0x7D, 0x7D, 0x7D, 0x7D, 0x7D, 0x7D, 0x7D, 0x7D,
	0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E,
	0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7E, 0x7F, 0x7F,
	0x7F, 0x7F, 0x7F, 0x7F, 0x7F, 0x7F, 0x7F, 0x7F,
	0x7F, 0x7F, 0x7F, 0x7F, 0x7F, 0x80, 0x80, 0x80,
	0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80,
	0x80, 0x80, 0x80, 0x80, 0x81, 0x81, 0x81, 0x81,
	0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81, 0x81,
	0x81, 0x81, 0x81, 0x82, 0x82, 0x82, 0x82, 0x82,
	0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82,
	0x82, 0x82, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83,
	0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83, 0x83,
	0x83, 0x83, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84,
	0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84,
	0x84, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85,
	0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85, 0x85,
	0x85, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86,
	0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86, 0x86,
	0x86, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87,
	0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87, 0x87,
	0x87, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
	0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x88,
	0x88, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89,
	0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89, 0x89,
	0x89, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A,
	0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A, 0x8A,
	0x8A, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B,
	0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B, 0x8B,
	0x8B, 0x8B, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C,
	0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C, 0x8C,
	0x8C, 0x8C, 0x8C, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D,
	0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D, 0x8D,
	0x8D, 0x8D, 0x8D, 0x8D, 0x8E, 0x8E, 0x8E, 0x8E,
	0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8E,
	0x8E, 0x8E, 0x8E, 0x8E, 0x8E, 0x8F, 0x8F, 0x8F,
	0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F,
	0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x8F, 0x90, 0x90,
	0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
	0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x91,
	0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91,
	0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91, 0x91,
	0x91, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92,
	0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92,
	0x92, 0x92, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93,
	0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93, 0x93,
	0x93, 0x93, 0x93, 0x93, 0x94, 0x94, 0x94, 0x94,
	0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x94,
	0x94, 0x94, 0x94, 0x94, 0x94, 0x94, 0x95, 0x95,
	0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95,
	0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95, 0x95,
	0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96,
	0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96, 0x96,
	0x96, 0x96, 0x96, 0x97, 0x97, 0x97, 0x97, 0x97,
	0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97, 0x97,
	0x97, 0x97, 0x97, 0x97, 0x97, 0x98, 0x98, 0x98,
	0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98,
	0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98, 0x98,
	0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99,
	0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99,
	0x99, 0x99, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A,
	0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9A,
	0x9A, 0x9A, 0x9A, 0x9A, 0x9A, 0x9B, 0x9B, 0x9B,
	0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B,
	0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B, 0x9B,
	0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C,
	0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C, 0x9C,
	0x9C, 0x9C, 0x9C, 0x9C, 0x9D, 0x9D, 0x9D, 0x9D,
	0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D,
	0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9D, 0x9E,
	0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E,
	0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E, 0x9E,
	0x9E, 0x9E, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F,
	0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F,
	0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0x9F, 0xA0, 0xA0,
	0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0,
	0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0, 0xA0,
	0xA0, 0xA0, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1,
	0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1,
	0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA1, 0xA2, 0xA2,
	0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2,
	0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2, 0xA2,
	0xA2, 0xA2, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
	0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3,
	0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA3, 0xA4, 0xA4,
	0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4,
	0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4, 0xA4,
	0xA4, 0xA4, 0xA4, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5,
	0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5,
	0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5, 0xA5,
	0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6,
	0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6, 0xA6,
	0xA6, 0xA6, 0xA6, 0xA6, 0xA7, 0xA7, 0xA7, 0xA7,
	0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7,
	0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7, 0xA7,
	0xA7, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8,
	0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8,
	0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA8, 0xA9,
	0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9,
	0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9, 0xA9,
	0xA9, 0xA9, 0xA9, 0xA9, 0xAA, 0xAA, 0xAA, 0xAA,
	0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
	0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
	0xAA, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB,
	0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB,
	0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAB, 0xAC,
	0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC,
	0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAC,
	0xAC, 0xAC, 0xAC, 0xAC, 0xAC, 0xAD, 0xAD, 0xAD,
	0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD,
	0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD, 0xAD,
	0xAD, 0xAD, 0xAD, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE,
	0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE,
	0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE, 0xAE,
	0xAE, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF,
	0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF,
	0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xAF, 0xB0,
	0xB0, 0xB0, 0xB0, 0xB0, 0xB0, 0xB0, 0xB0, 0xB0,
	0xB0, 0xB0, 0xB0, 0xB0, 0xB0, 0xB0, 0xB0, 0xB0,
	0xB0, 0xB0, 0xB0, 0xB0, 0xB0, 0xB0, 0xB1, 0xB1,
	0xB1, 0xB1, 0xB1, 0xB1, 0xB1, 0xB1, 0xB1, 0xB1,
	0xB1, 0xB1, 0xB1, 0xB1, 0xB1, 0xB1, 0xB1, 0xB1,
	0xB1, 0xB1, 0xB1, 0xB1, 0xB2, 0xB2, 0xB2, 0xB2,
	0xB2, 0xB2, 0xB2, 0xB2, 0xB2, 0xB2, 0xB2, 0xB2,
	0xB2, 0xB2, 0xB2, 0xB2, 0xB2, 0xB2, 0xB2, 0xB2,
	0xB2, 0xB2, 0xB2, 0xB3, 0xB3, 0xB3, 0xB3, 0xB3,
	0xB3, 0xB3, 0xB3, 0xB3, 0xB3, 0xB3, 0xB3, 0xB3,
	0xB3, 0xB3, 0xB3, 0xB3, 0xB3, 0xB3, 0xB3, 0xB3,
	0xB3, 0xB3, 0xB4, 0xB4, 0xB4, 0xB4, 0xB4, 0xB4,
	0xB4, 0xB4, 0xB4, 0xB4, 0xB4, 0xB4, 0xB4, 0xB4,
	0xB4, 0xB4, 0xB4, 0xB4, 0xB4, 0xB4, 0xB4, 0xB4,
	0xB4, 0xB5, 0xB5, 0xB5, 0xB5, 0xB5, 0xB5, 0xB5,
	0xB5, 0xB5, 0xB5, 0xB5, 0xB5, 0xB5, 0xB5, 0xB5,
	0xB5, 0xB5, 0xB5, 0xB5, 0xB5, 0xB5, 0xB5, 0xB5,
	0xB6, 0xB6, 0xB6, 0xB6, 0xB6, 0xB6, 0xB6, 0xB6,
	0xB6, 0xB6, 0xB6, 0xB6, 0xB6, 0xB6, 0xB6, 0xB6,
	0xB6, 0xB6, 0xB6, 0xB6, 0xB6, 0xB6, 0xB6, 0xB6,
	0xB7, 0xB7, 0xB7, 0xB7, 0xB7, 0xB7, 0xB7, 0xB7,
	0xB7, 0xB7, 0xB7, 0xB7, 0xB7, 0xB7, 0xB7, 0xB7,
	0xB7, 0xB7, 0xB7, 0xB7, 0xB7, 0xB7, 0xB7, 0xB8,
	0xB8, 0xB8, 0xB8, 0xB8, 0xB8, 0xB8, 0xB8, 0xB8,
	0xB8, 0xB8, 0xB8, 0xB8, 0xB8, 0xB8, 0xB8, 0xB8,
	0xB8, 0xB8, 0xB8, 0xB8, 0xB8, 0xB8, 0xB8, 0xB9,
	0xB9, 0xB9, 0xB9, 0xB9, 0xB9, 0xB9, 0xB9, 0xB9,
	0xB9, 0xB9, 0xB9, 0xB9, 0xB9, 0xB9, 0xB9, 0xB9,
	0xB9, 0xB9, 0xB9, 0xB9, 0xB9, 0xB9, 0xB9, 0xBA,
	0xBA, 0xBA, 0xBA, 0xBA, 0xBA, 0xBA, 0xBA, 0xBA,
	0xBA, 0xBA, 0xBA, 0xBA, 0xBA, 0xBA, 0xBA, 0xBA,
	0xBA, 0xBA, 0xBA, 0xBA, 0xBA, 0xBA, 0xBA, 0xBB,
	0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB,
	0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB,
	0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB,
	0xBC, 0xBC, 0xBC, 0xBC, 0xBC, 0xBC, 0xBC, 0xBC,
	0xBC, 0xBC, 0xBC, 0xBC, 0xBC, 0xBC, 0xBC, 0xBC,
	0xBC, 0xBC, 0xBC, 0xBC, 0xBC, 0xBC, 0xBC, 0xBC,
	0xBD, 0xBD, 0xBD, 0xBD, 0xBD, 0xBD, 0xBD, 0xBD,
	0xBD, 0xBD, 0xBD, 0xBD, 0xBD, 0xBD, 0xBD, 0xBD,
	0xBD, 0xBD, 0xBD, 0xBD, 0xBD, 0xBD, 0xBD, 0xBD,
	0xBD, 0xBE, 0xBE, 0xBE, 0xBE, 0xBE, 0xBE, 0xBE,
	0xBE, 0xBE, 0xBE, 0xBE, 0xBE, 0xBE, 0xBE, 0xBE,
	0xBE, 0xBE, 0xBE, 0xBE, 0xBE, 0xBE, 0xBE, 0xBE,
	0xBE, 0xBE, 0xBF, 0xBF, 0xBF, 0xBF, 0xBF, 0xBF,
	0xBF, 0xBF, 0xBF, 0xBF, 0xBF, 0xBF, 0xBF, 0xBF,
	0xBF, 0xBF, 0xBF, 0xBF, 0xBF, 0xBF, 0xBF, 0xBF,
	0xBF, 0xBF, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0,
	0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0,
	0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0,
	0xC0, 0xC0, 0xC0, 0xC0, 0xC1, 0xC1, 0xC1, 0xC1,
	0xC1, 0xC1, 0xC1, 0xC1, 0xC1, 0xC1, 0xC1, 0xC1,
	0xC1, 0xC1, 0xC1, 0xC1, 0xC1, 0xC1, 0xC1, 0xC1,
	0xC1, 0xC1, 0xC1, 0xC1, 0xC1, 0xC2, 0xC2, 0xC2,
	0xC2, 0xC2, 0xC2, 0xC2, 0xC2, 0xC2, 0xC2, 0xC2,
	0xC2, 0xC2, 0xC2, 0xC2, 0xC2, 0xC2, 0xC2, 0xC2,
	0xC2, 0xC2, 0xC2, 0xC2, 0xC2, 0xC2, 0xC3, 0xC3,
	0xC3, 0xC3, 0xC3, 0xC3, 0xC3, 0xC3, 0xC3, 0xC3,
	0xC3, 0xC3, 0xC3, 0xC3, 0xC3, 0xC3, 0xC3, 0xC3,
	0xC3, 0xC3, 0xC3, 0xC3, 0xC3, 0xC3, 0xC3, 0xC3,
	0xC4, 0xC4, 0xC4, 0xC4, 0xC4, 0xC4, 0xC4, 0xC4,
	0xC4, 0xC4, 0xC4, 0xC4, 0xC4, 0xC4, 0xC4, 0xC4,
	0xC4, 0xC4, 0xC4, 0xC4, 0xC4, 0xC4, 0xC4, 0xC4,
	0xC4, 0xC4, 0xC5, 0xC5, 0xC5, 0xC5, 0xC5, 0xC5,
	0xC5, 0xC5, 0xC5, 0xC5, 0xC5, 0xC5, 0xC5, 0xC5,
	0xC5, 0xC5, 0xC5, 0xC5, 0xC5, 0xC5, 0xC5, 0xC5,
	0xC5, 0xC5, 0xC5, 0xC5, 0xC6, 0xC6, 0xC6, 0xC6,
	0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6,
	0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6,
	0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0xC7, 0xC7,
	0xC7, 0xC7, 0xC7, 0xC7, 0xC7, 0xC7, 0xC7, 0xC7,
	0xC7, 0xC7, 0xC7, 0xC7, 0xC7, 0xC7, 0xC7, 0xC7,
	0xC7, 0xC7, 0xC7, 0xC7, 0xC7, 0xC7, 0xC7, 0xC7,
	0xC8, 0xC8, 0xC8, 0xC8, 0xC8, 0xC8, 0xC8, 0xC8,
	0xC8, 0xC8, 0xC8, 0xC8, 0xC8, 0xC8, 0xC8, 0xC8,
	0xC8, 0xC8, 0xC8, 0xC8, 0xC8, 0xC8, 0xC8, 0xC8,
	0xC8, 0xC8, 0xC8, 0xC9, 0xC9, 0xC9, 0xC9, 0xC9,
	0xC9, 0xC9, 0xC9, 0xC9, 0xC9, 0xC9, 0xC9, 0xC9,
	0xC9, 0xC9, 0xC9, 0xC9, 0xC9, 0xC9, 0xC9, 0xC9,
	0xC9, 0xC9, 0xC9, 0xC9, 0xC9, 0xC9, 0xCA, 0xCA,
	0xCA, 0xCA, 0xCA, 0xCA, 0xCA, 0xCA, 0xCA, 0xCA,
	0xCA, 0xCA, 0xCA, 0xCA, 0xCA, 0xCA, 0xCA, 0xCA,
	0xCA, 0xCA, 0xCA, 0xCA, 0xCA, 0xCA, 0xCA, 0xCA,
	0xCA, 0xCB, 0xCB, 0xCB, 0xCB, 0xCB, 0xCB, 0xCB,
	0xCB, 0xCB, 0xCB, 0xCB, 0xCB, 0xCB, 0xCB, 0xCB,
	0xCB, 0xCB, 0xCB, 0xCB, 0xCB, 0xCB, 0xCB, 0xCB,
	0xCB, 0xCB, 0xCB, 0xCB, 0xCC, 0xCC, 0xCC, 0xCC,
	0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC,
	0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC,
	0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCD,
	0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
	0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
	0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD,
	0xCD, 0xCD, 0xCE, 0xCE, 0xCE, 0xCE, 0xCE, 0xCE,
	0xCE, 0xCE, 0xCE, 0xCE, 0xCE, 0xCE, 0xCE, 0xCE,
	0xCE, 0xCE, 0xCE, 0xCE, 0xCE, 0xCE, 0xCE, 0xCE,
	0xCE, 0xCE, 0xCE, 0xCE, 0xCE, 0xCE, 0xCF, 0xCF,
	0xCF, 0xCF, 0xCF, 0xCF, 0xCF, 0xCF, 0xCF, 0xCF,
	0xCF, 0xCF, 0xCF, 0xCF, 0xCF, 0xCF, 0xCF, 0xCF,
	0xCF, 0xCF, 0xCF, 0xCF, 0xCF, 0xCF, 0xCF, 0xCF,
	0xCF, 0xCF, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
	0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
	0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD0,
	0xD0, 0xD0, 0xD0, 0xD0, 0xD0, 0xD1, 0xD1, 0xD1,
	0xD1, 0xD1, 0xD1, 0xD1, 0xD1, 0xD1, 0xD1, 0xD1,
	0xD1, 0xD1, 0xD1, 0xD1, 0xD1, 0xD1, 0xD1, 0xD1,
	0xD1, 0xD1, 0xD1, 0xD1, 0xD1, 0xD1, 0xD1, 0xD1,
	0xD1, 0xD1, 0xD2, 0xD2, 0xD2, 0xD2, 0xD2, 0xD2,
	0xD2, 0xD2, 0xD2, 0xD2, 0xD2, 0xD2, 0xD2, 0xD2,
	0xD2, 0xD2, 0xD2, 0xD2, 0xD2, 0xD2, 0xD2, 0xD2,
	0xD2, 0xD2, 0xD2, 0xD2, 0xD2, 0xD2, 0xD3, 0xD3,
	0xD3, 0xD3, 0xD3, 0xD3, 0xD3, 0xD3, 0xD3, 0xD3,
	0xD3, 0xD3, 0xD3, 0xD3, 0xD3, 0xD3, 0xD3, 0xD3,
	0xD3, 0xD3, 0xD3, 0xD3, 0xD3, 0xD3, 0xD3, 0xD3,
	0xD3, 0xD3, 0xD4, 0xD4, 0xD4, 0xD4, 0xD4, 0xD4,
	0xD4, 0xD4, 0xD4, 0xD4, 0xD4, 0xD4, 0xD4, 0xD4,
	0xD4, 0xD4, 0xD4, 0xD4, 0xD4, 0xD4, 0xD4, 0xD4,
	0xD4, 0xD4, 0xD4, 0xD4, 0xD4, 0xD4, 0xD4, 0xD5,
	0xD5, 0xD5, 0xD5, 0xD5, 0xD5, 0xD5, 0xD5, 0xD5,
	0xD5, 0xD5, 0xD5, 0xD5, 0xD5, 0xD5, 0xD5, 0xD5,
	0xD5, 0xD5, 0xD5, 0xD5, 0xD5, 0xD5, 0xD5, 0xD5,
	0xD5, 0xD5, 0xD5, 0xD5, 0xD6, 0xD6, 0xD6, 0xD6,
	0xD6, 0xD6, 0xD6, 0xD6, 0xD6, 0xD6, 0xD6, 0xD6,
	0xD6, 0xD6, 0xD6, 0xD6, 0xD6, 0xD6, 0xD6, 0xD6,
	0xD6, 0xD6, 0xD6, 0xD6, 0xD6, 0xD6, 0xD6, 0xD6,
	0xD6, 0xD7, 0xD7, 0xD7, 0xD7, 0xD7, 0xD7, 0xD7,
	0xD7, 0xD7, 0xD7, 0xD7, 0xD7, 0xD7, 0xD7, 0xD7,
	0xD7, 0xD7, 0xD7, 0xD7, 0xD7, 0xD7, 0xD7, 0xD7,
	0xD7, 0xD7, 0xD7, 0xD7, 0xD7, 0xD7, 0xD8, 0xD8,
	0xD8, 0xD8, 0xD8, 0xD8, 0xD8, 0xD8, 0xD8, 0xD8,
	0xD8, 0xD8, 0xD8, 0xD8, 0xD8, 0xD8, 0xD8, 0xD8,
	0xD8, 0xD8, 0xD8, 0xD8, 0xD8, 0xD8, 0xD8, 0xD8,
	0xD8, 0xD8, 0xD8, 0xD9, 0xD9, 0xD9, 0xD9, 0xD9,
	0xD9, 0xD9, 0xD9, 0xD9, 0xD9, 0xD9, 0xD9, 0xD9,
	0xD9, 0xD9, 0xD9, 0xD9, 0xD9, 0xD9, 0xD9, 0xD9,
	0xD9, 0xD9, 0xD9, 0xD9, 0xD9, 0xD9, 0xD9, 0xD9,
	0xD9, 0xDA, 0xDA, 0xDA, 0xDA, 0xDA, 0xDA, 0xDA,
	0xDA, 0xDA, 0xDA, 0xDA, 0xDA, 0xDA, 0xDA, 0xDA,
	0xDA, 0xDA, 0xDA, 0xDA, 0xDA, 0xDA, 0xDA, 0xDA,
	0xDA, 0xDA, 0xDA, 0xDA, 0xDA, 0xDA, 0xDB, 0xDB,
	0xDB, 0xDB, 0xDB, 0xDB, 0xDB, 0xDB, 0xDB, 0xDB,
	0xDB, 0xDB, 0xDB, 0xDB, 0xDB, 0xDB, 0xDB, 0xDB,
	0xDB, 0xDB, 0xDB, 0xDB, 0xDB, 0xDB, 0xDB, 0xDB,
	0xDB, 0xDB, 0xDB, 0xDB, 0xDC, 0xDC, 0xDC, 0xDC,
	0xDC, 0xDC, 0xDC, 0xDC, 0xDC, 0xDC, 0xDC, 0xDC,
	0xDC, 0xDC, 0xDC, 0xDC, 0xDC, 0xDC, 0xDC, 0xDC,
	0xDC, 0xDC, 0xDC, 0xDC, 0xDC, 0xDC, 0xDC, 0xDC,
	0xDC, 0xDC, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
	0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
	0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
	0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
	0xDD, 0xDE, 0xDE, 0xDE, 0xDE, 0xDE, 0xDE, 0xDE,
	0xDE, 0xDE, 0xDE, 0xDE, 0xDE, 0xDE, 0xDE, 0xDE,
	0xDE, 0xDE, 0xDE, 0xDE, 0xDE, 0xDE, 0xDE, 0xDE,
	0xDE, 0xDE, 0xDE, 0xDE, 0xDE, 0xDE, 0xDE, 0xDF,
	0xDF, 0xDF, 0xDF, 0xDF, 0xDF, 0xDF, 0xDF, 0xDF,
	0xDF, 0xDF, 0xDF, 0xDF, 0xDF, 0xDF, 0xDF, 0xDF,
	0xDF, 0xDF, 0xDF, 0xDF, 0xDF, 0xDF, 0xDF, 0xDF,
	0xDF, 0xDF, 0xDF, 0xDF, 0xDF, 0xDF, 0xE0, 0xE0,
	0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0,
	0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0,
	0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0, 0xE0,
	0xE0, 0xE0, 0xE0, 0xE0, 0xE1, 0xE1, 0xE1, 0xE1,
	0xE1, 0xE1, 0xE1, 0xE1, 0xE1, 0xE1, 0xE1, 0xE1,
	0xE1, 0xE1, 0xE1, 0xE1, 0xE1, 0xE1, 0xE1, 0xE1,
	0xE1, 0xE1, 0xE1, 0xE1, 0xE1, 0xE1, 0xE1, 0xE1,
	0xE1, 0xE1, 0xE1, 0xE2, 0xE2, 0xE2, 0xE2, 0xE2,
	0xE2, 0xE2, 0xE2, 0xE2, 0xE2, 0xE2, 0xE2, 0xE2,
	0xE2, 0xE2, 0xE2, 0xE2, 0xE2, 0xE2, 0xE2, 0xE2,
	0xE2, 0xE2, 0xE2, 0xE2, 0xE2, 0xE2, 0xE2, 0xE2,
	0xE2, 0xE2, 0xE3, 0xE3, 0xE3, 0xE3, 0xE3, 0xE3,
	0xE3, 0xE3, 0xE3, 0xE3, 0xE3, 0xE3, 0xE3, 0xE3,
	0xE3, 0xE3, 0xE3, 0xE3, 0xE3, 0xE3, 0xE3, 0xE3,
	0xE3, 0xE3, 0xE3, 0xE3, 0xE3, 0xE3, 0xE3, 0xE3,
	0xE3, 0xE3, 0xE4, 0xE4, 0xE4, 0xE4, 0xE4, 0xE4,
	0xE4, 0xE4, 0xE4, 0xE4, 0xE4, 0xE4, 0xE4, 0xE4,
	0xE4, 0xE4, 0xE4, 0xE4, 0xE4, 0xE4, 0xE4, 0xE4,
	0xE4, 0xE4, 0xE4, 0xE4, 0xE4, 0xE4, 0xE4, 0xE4,
	0xE4, 0xE5, 0xE5, 0xE5, 0xE5, 0xE5, 0xE5, 0xE5,
	0xE5, 0xE5, 0xE5, 0xE5, 0xE5, 0xE5, 0xE5, 0xE5,
	0xE5, 0xE5, 0xE5, 0xE5, 0xE5, 0xE5, 0xE5, 0xE5,
	0xE5, 0xE5, 0xE5, 0xE5, 0xE5, 0xE5, 0xE5, 0xE5,
	0xE5, 0xE6, 0xE6, 0xE6, 0xE6, 0xE6, 0xE6, 0xE6,
	0xE6, 0xE6, 0xE6, 0xE6, 0xE6, 0xE6, 0xE6, 0xE6,
	0xE6, 0xE6, 0xE6, 0xE6, 0xE6, 0xE6, 0xE6, 0xE6,
	0xE6, 0xE6, 0xE6, 0xE6, 0xE6, 0xE6, 0xE6, 0xE6,
	0xE6, 0xE7, 0xE7, 0xE7, 0xE7, 0xE7, 0xE7, 0xE7,
	0xE7, 0xE7, 0xE7, 0xE7, 0xE7, 0xE7, 0xE7, 0xE7,
	0xE7, 0xE7, 0xE7, 0xE7, 0xE7, 0xE7, 0xE7, 0xE7,
	0xE7, 0xE7, 0xE7, 0xE7, 0xE7, 0xE7, 0xE7, 0xE7,
	0xE7, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8,
	0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8,
	0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8,
	0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8, 0xE8,
	0xE8, 0xE9, 0xE9, 0xE9, 0xE9, 0xE9, 0xE9, 0xE9,
	0xE9, 0xE9, 0xE9, 0xE9, 0xE9, 0xE9, 0xE9, 0xE9,
	0xE9, 0xE9, 0xE9, 0xE9, 0xE9, 0xE9, 0xE9, 0xE9,
	0xE9, 0xE9, 0xE9, 0xE9, 0xE9, 0xE9, 0xE9, 0xE9,
	0xE9, 0xE9, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA,
	0xEA, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA,
	0xEA, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA,
	0xEA, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA, 0xEA,
	0xEA, 0xEA, 0xEB, 0xEB, 0xEB, 0xEB, 0xEB, 0xEB,
	0xEB, 0xEB, 0xEB, 0xEB, 0xEB, 0xEB, 0xEB, 0xEB,
	0xEB, 0xEB, 0xEB, 0xEB, 0xEB, 0xEB, 0xEB, 0xEB,
	0xEB, 0xEB, 0xEB, 0xEB, 0xEB, 0xEB, 0xEB, 0xEB,
	0xEB, 0xEB, 0xEB, 0xEC, 0xEC, 0xEC, 0xEC, 0xEC,
	0xEC, 0xEC, 0xEC, 0xEC, 0xEC, 0xEC, 0xEC, 0xEC,
	0xEC, 0xEC, 0xEC, 0xEC, 0xEC, 0xEC, 0xEC, 0xEC,
	0xEC, 0xEC, 0xEC, 0xEC, 0xEC, 0xEC, 0xEC, 0xEC,
	0xEC, 0xEC, 0xEC, 0xEC, 0xED, 0xED, 0xED, 0xED,
	0xED, 0xED, 0xED, 0xED, 0xED, 0xED, 0xED, 0xED,
	0xED, 0xED, 0xED, 0xED, 0xED, 0xED, 0xED, 0xED,
	0xED, 0xED, 0xED, 0xED, 0xED, 0xED, 0xED, 0xED,
	0xED, 0xED, 0xED, 0xED, 0xED, 0xEE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE,
	0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEF, 0xEF,
	0xEF, 0xEF, 0xEF, 0xEF, 0xEF, 0xEF, 0xEF, 0xEF,
	0xEF, 0xEF, 0xEF, 0xEF, 0xEF, 0xEF, 0xEF, 0xEF,
	0xEF, 0xEF, 0xEF, 0xEF, 0xEF, 0xEF, 0xEF, 0xEF,
	0xEF, 0xEF, 0xEF, 0xEF, 0xEF, 0xEF, 0xEF, 0xEF,
	0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
	0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
	0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
	0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0, 0xF0,
	0xF0, 0xF0, 0xF1, 0xF1, 0xF1, 0xF1, 0xF1, 0xF1,
	0xF1, 0xF1, 0xF1, 0xF1, 0xF1, 0xF1, 0xF1, 0xF1,
	0xF1, 0xF1, 0xF1, 0xF1, 0xF1, 0xF1, 0xF1, 0xF1,
	0xF1, 0xF1, 0xF1, 0xF1, 0xF1, 0xF1, 0xF1, 0xF1,
	0xF1, 0xF1, 0xF1, 0xF1, 0xF2, 0xF2, 0xF2, 0xF2,
	0xF2, 0xF2, 0xF2, 0xF2, 0xF2, 0xF2, 0xF2, 0xF2,
	0xF2, 0xF2, 0xF2, 0xF2, 0xF2, 0xF2, 0xF2, 0xF2,
	0xF2, 0xF2, 0xF2, 0xF2, 0xF2, 0xF2, 0xF2, 0xF2,
	0xF2, 0xF2, 0xF2, 0xF2, 0xF2, 0xF2, 0xF3, 0xF3,
	0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3,
	0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3,
	0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3,
	0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3, 0xF3,
	0xF4, 0xF4, 0xF4, 0xF4, 0xF4, 0xF4, 0xF4, 0xF4,
	0xF4, 0xF4, 0xF4, 0xF4, 0xF4, 0xF4, 0xF4, 0xF4,
	0xF4, 0xF4, 0xF4, 0xF4, 0xF4, 0xF4, 0xF4, 0xF4,
	0xF4, 0xF4, 0xF4, 0xF4, 0xF4, 0xF4, 0xF4, 0xF4,
	0xF4, 0xF4, 0xF5, 0xF5, 0xF5, 0xF5, 0xF5, 0xF5,
	0xF5, 0xF5, 0xF5, 0xF5, 0xF5, 0xF5, 0xF5, 0xF5,
	0xF5, 0xF5, 0xF5, 0xF5, 0xF5, 0xF5, 0xF5, 0xF5,
	0xF5, 0xF5, 0xF5, 0xF5, 0xF5, 0xF5, 0xF5, 0xF5,
	0xF5, 0xF5, 0xF5, 0xF5, 0xF5, 0xF6, 0xF6, 0xF6,
	0xF6, 0xF6, 0xF6, 0xF6, 0xF6, 0xF6, 0xF6, 0xF6,
	0xF6, 0xF6, 0xF6, 0xF6, 0xF6, 0xF6, 0xF6, 0xF6,
	0xF6, 0xF6, 0xF6, 0xF6, 0xF6, 0xF6, 0xF6, 0xF6,
	0xF6, 0xF6, 0xF6, 0xF6, 0xF6, 0xF6, 0xF6, 0xF6,
	0xF7, 0xF7, 0xF7, 0xF7, 0xF7, 0xF7, 0xF7, 0xF7,
	0xF7, 0xF7, 0xF7, 0xF7, 0xF7, 0xF7, 0xF7, 0xF7,
	0xF7, 0xF7, 0xF7, 0xF7, 0xF7, 0xF7, 0xF7, 0xF7,
	0xF7, 0xF7, 0xF7, 0xF7, 0xF7, 0xF7, 0xF7, 0xF7,
	0xF7, 0xF7, 0xF7, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
	0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
	0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
	0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8,
	0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF8, 0xF9, 0xF9,
	0xF9, 0xF9, 0xF9, 0xF9, 0xF9, 0xF9, 0xF9, 0xF9,
	0xF9, 0xF9, 0xF9, 0xF9, 0xF9, 0xF9, 0xF9, 0xF9,
	0xF9, 0xF9, 0xF9, 0xF9, 0xF9, 0xF9, 0xF9, 0xF9,
	0xF9, 0xF9, 0xF9, 0xF9, 0xF9, 0xF9, 0xF9, 0xF9,
	0xF9, 0xFA, 0xFA, 0xFA, 0xFA, 0xFA, 0xFA, 0xFA,
	0xFA, 0xFA, 0xFA, 0xFA, 0xFA, 0xFA, 0xFA, 0xFA,
	0xFA, 0xFA, 0xFA, 0xFA, 0xFA, 0xFA, 0xFA, 0xFA,
	0xFA, 0xFA, 0xFA, 0xFA, 0xFA, 0xFA, 0xFA, 0xFA,
	0xFA, 0xFA, 0xFA, 0xFA, 0xFA, 0xFB, 0xFB, 0xFB,
	0xFB, 0xFB, 0xFB, 0xFB, 0xFB, 0xFB, 0xFB, 0xFB,
	0xFB, 0xFB, 0xFB, 0xFB, 0xFB, 0xFB, 0xFB, 0xFB,
	0xFB, 0xFB, 0xFB, 0xFB, 0xFB, 0xFB, 0xFB, 0xFB,
	0xFB, 0xFB, 0xFB, 0xFB, 0xFB, 0xFB, 0xFB, 0xFB,
	0xFB, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC,
	0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC,
	0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC,
	0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC,
	0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFD, 0xFD, 0xFD,
	0xFD, 0xFD, 0xFD, 0xFD, 0xFD, 0xFD, 0xFD, 0xFD,
	0xFD, 0xFD, 0xFD, 0xFD, 0xFD, 0xFD, 0xFD, 0xFD,
	0xFD, 0xFD, 0xFD, 0xFD, 0xFD, 0xFD, 0xFD, 0xFD,
	0xFD, 0xFD, 0xFD, 0xFD, 0xFD, 0xFD, 0xFD, 0xFD,
	0xFD, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
	0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
	0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
	0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFE,
	0xFE, 0xFE, 0xFE, 0xFE, 0xFE, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
};

void mdp4_mixer_gc_lut_setup(int mixer_num)
{
	unsigned char *base;
	uint32 data;
	char val;
	int i, off;

	if (mixer_num) 	/* mixer number, /dev/fb0, /dev/fb1 */
		base = MDP_BASE + MDP4_OVERLAYPROC1_BASE;/* 0x18000 */
	else
		base = MDP_BASE + MDP4_OVERLAYPROC0_BASE;/* 0x10000 */

	base += 0x4000;	/* GC_LUT offset */

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);
	off = 0;
	for (i = 0; i < 4096; i++) {
		val = gc_lut[i];
		data = (val << 16 | val << 8 | val); /* R, B, and G are same */
		outpdw(base + off, data);
		off += 4;
	}
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
}

uint32 igc_video_lut[] = {	 /* non linear */
	0x0, 0x1, 0x2, 0x4, 0x5, 0x6, 0x7, 0x9,
	0xA, 0xB, 0xC, 0xE, 0xF, 0x10, 0x12, 0x14,
	0x15, 0x17, 0x19, 0x1B, 0x1D, 0x1F, 0x21, 0x23,
	0x25, 0x28, 0x2A, 0x2D, 0x30, 0x32, 0x35, 0x38,
	0x3B, 0x3E, 0x42, 0x45, 0x48, 0x4C, 0x4F, 0x53,
	0x57, 0x5B, 0x5F, 0x63, 0x67, 0x6B, 0x70, 0x74,
	0x79, 0x7E, 0x83, 0x88, 0x8D, 0x92, 0x97, 0x9C,
	0xA2, 0xA8, 0xAD, 0xB3, 0xB9, 0xBF, 0xC5, 0xCC,
	0xD2, 0xD8, 0xDF, 0xE6, 0xED, 0xF4, 0xFB, 0x102,
	0x109, 0x111, 0x118, 0x120, 0x128, 0x130, 0x138, 0x140,
	0x149, 0x151, 0x15A, 0x162, 0x16B, 0x174, 0x17D, 0x186,
	0x190, 0x199, 0x1A3, 0x1AC, 0x1B6, 0x1C0, 0x1CA, 0x1D5,
	0x1DF, 0x1EA, 0x1F4, 0x1FF, 0x20A, 0x215, 0x220, 0x22B,
	0x237, 0x242, 0x24E, 0x25A, 0x266, 0x272, 0x27F, 0x28B,
	0x298, 0x2A4, 0x2B1, 0x2BE, 0x2CB, 0x2D8, 0x2E6, 0x2F3,
	0x301, 0x30F, 0x31D, 0x32B, 0x339, 0x348, 0x356, 0x365,
	0x374, 0x383, 0x392, 0x3A1, 0x3B1, 0x3C0, 0x3D0, 0x3E0,
	0x3F0, 0x400, 0x411, 0x421, 0x432, 0x443, 0x454, 0x465,
	0x476, 0x487, 0x499, 0x4AB, 0x4BD, 0x4CF, 0x4E1, 0x4F3,
	0x506, 0x518, 0x52B, 0x53E, 0x551, 0x565, 0x578, 0x58C,
	0x5A0, 0x5B3, 0x5C8, 0x5DC, 0x5F0, 0x605, 0x61A, 0x62E,
	0x643, 0x659, 0x66E, 0x684, 0x699, 0x6AF, 0x6C5, 0x6DB,
	0x6F2, 0x708, 0x71F, 0x736, 0x74D, 0x764, 0x77C, 0x793,
	0x7AB, 0x7C3, 0x7DB, 0x7F3, 0x80B, 0x824, 0x83D, 0x855,
	0x86F, 0x888, 0x8A1, 0x8BB, 0x8D4, 0x8EE, 0x908, 0x923,
	0x93D, 0x958, 0x973, 0x98E, 0x9A9, 0x9C4, 0x9DF, 0x9FB,
	0xA17, 0xA33, 0xA4F, 0xA6C, 0xA88, 0xAA5, 0xAC2, 0xADF,
	0xAFC, 0xB19, 0xB37, 0xB55, 0xB73, 0xB91, 0xBAF, 0xBCE,
	0xBEC, 0xC0B, 0xC2A, 0xC4A, 0xC69, 0xC89, 0xCA8, 0xCC8,
	0xCE8, 0xD09, 0xD29, 0xD4A, 0xD6B, 0xD8C, 0xDAD, 0xDCF,
	0xDF0, 0xE12, 0xE34, 0xE56, 0xE79, 0xE9B, 0xEBE, 0xEE1,
	0xF04, 0xF27, 0xF4B, 0xF6E, 0xF92, 0xFB6, 0xFDB, 0xFFF,
};

void mdp4_vg_igc_lut_setup(int vp_num)
{
	unsigned char *base;
	int i, voff, off;
	uint32 data, val;

	voff = MDP4_VIDEO_OFF * vp_num;
	base = MDP_BASE + MDP4_VIDEO_BASE + voff + 0x5000;

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);
	off = 0;
	for (i = 0; i < 256; i++) {
		val = igc_video_lut[i];
		data = (val << 16 | val);	/* color 0 and 1 */
		outpdw(base + off, data);
		outpdw(base + off + 0x800, val);	/* color 2 */
		off += 4;
	}
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
}

uint32 igc_rgb_lut[] = {   /* linear */
	0x0, 0x10, 0x20, 0x30, 0x40, 0x50, 0x60, 0x70,
	0x80, 0x91, 0xA1, 0xB1, 0xC1, 0xD1, 0xE1, 0xF1,
	0x101, 0x111, 0x121, 0x131, 0x141, 0x151, 0x161, 0x171,
	0x181, 0x191, 0x1A2, 0x1B2, 0x1C2, 0x1D2, 0x1E2, 0x1F2,
	0x202, 0x212, 0x222, 0x232, 0x242, 0x252, 0x262, 0x272,
	0x282, 0x292, 0x2A2, 0x2B3, 0x2C3, 0x2D3, 0x2E3, 0x2F3,
	0x303, 0x313, 0x323, 0x333, 0x343, 0x353, 0x363, 0x373,
	0x383, 0x393, 0x3A3, 0x3B3, 0x3C4, 0x3D4, 0x3E4, 0x3F4,
	0x404, 0x414, 0x424, 0x434, 0x444, 0x454, 0x464, 0x474,
	0x484, 0x494, 0x4A4, 0x4B4, 0x4C4, 0x4D5, 0x4E5, 0x4F5,
	0x505, 0x515, 0x525, 0x535, 0x545, 0x555, 0x565, 0x575,
	0x585, 0x595, 0x5A5, 0x5B5, 0x5C5, 0x5D5, 0x5E6, 0x5F6,
	0x606, 0x616, 0x626, 0x636, 0x646, 0x656, 0x666, 0x676,
	0x686, 0x696, 0x6A6, 0x6B6, 0x6C6, 0x6D6, 0x6E6, 0x6F7,
	0x707, 0x717, 0x727, 0x737, 0x747, 0x757, 0x767, 0x777,
	0x787, 0x797, 0x7A7, 0x7B7, 0x7C7, 0x7D7, 0x7E7, 0x7F7,
	0x808, 0x818, 0x828, 0x838, 0x848, 0x858, 0x868, 0x878,
	0x888, 0x898, 0x8A8, 0x8B8, 0x8C8, 0x8D8, 0x8E8, 0x8F8,
	0x908, 0x919, 0x929, 0x939, 0x949, 0x959, 0x969, 0x979,
	0x989, 0x999, 0x9A9, 0x9B9, 0x9C9, 0x9D9, 0x9E9, 0x9F9,
	0xA09, 0xA19, 0xA2A, 0xA3A, 0xA4A, 0xA5A, 0xA6A, 0xA7A,
	0xA8A, 0xA9A, 0xAAA, 0xABA, 0xACA, 0xADA, 0xAEA, 0xAFA,
	0xB0A, 0xB1A, 0xB2A, 0xB3B, 0xB4B, 0xB5B, 0xB6B, 0xB7B,
	0xB8B, 0xB9B, 0xBAB, 0xBBB, 0xBCB, 0xBDB, 0xBEB, 0xBFB,
	0xC0B, 0xC1B, 0xC2B, 0xC3B, 0xC4C, 0xC5C, 0xC6C, 0xC7C,
	0xC8C, 0xC9C, 0xCAC, 0xCBC, 0xCCC, 0xCDC, 0xCEC, 0xCFC,
	0xD0C, 0xD1C, 0xD2C, 0xD3C, 0xD4C, 0xD5D, 0xD6D, 0xD7D,
	0xD8D, 0xD9D, 0xDAD, 0xDBD, 0xDCD, 0xDDD, 0xDED, 0xDFD,
	0xE0D, 0xE1D, 0xE2D, 0xE3D, 0xE4D, 0xE5D, 0xE6E, 0xE7E,
	0xE8E, 0xE9E, 0xEAE, 0xEBE, 0xECE, 0xEDE, 0xEEE, 0xEFE,
	0xF0E, 0xF1E, 0xF2E, 0xF3E, 0xF4E, 0xF5E, 0xF6E, 0xF7F,
	0xF8F, 0xF9F, 0xFAF, 0xFBF, 0xFCF, 0xFDF, 0xFEF, 0xFFF,
};

void mdp4_rgb_igc_lut_setup(int num)
{
	unsigned char *base;
	int i, voff, off;
	uint32 data, val;

	voff = MDP4_RGB_OFF * num;
	base = MDP_BASE + MDP4_RGB_BASE + voff + 0x5000;

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);
	off = 0;
	for (i = 0; i < 256; i++) {
		val = igc_rgb_lut[i];
		data = (val << 16 | val);	/* color 0 and 1 */
		outpdw(base + off, data);
		outpdw(base + off + 0x800, val);	/* color 2 */
		off += 4;
	}
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
}

uint32 mdp4_rgb_igc_lut_cvt(uint32 ndx)
{
	return igc_rgb_lut[ndx & 0x0ff];
}

uint32_t mdp4_ss_table_value(int8_t value, int8_t index)
{
	uint32_t out = 0x0;
	int8_t level = -1;
	uint32_t mask = 0xffffffff;

	if (value < 0) {
		if (value == -128)
			value = 127;
		else
			value = -value;
		out = 0x11111111;
	} else {
		out = 0x88888888;
		mask = 0x0fffffff;
	}

	if (value == 0)
		level = 0;
	else {
		while (value > 0 && level < 7) {
			level++;
			value -= 16;
		}
	}

	if (level == 0) {
		if (index == 0)
			out = 0x0;
		else
			out = 0x20000000;
	} else {
		out += (0x11111111 * level);
		if (index == 1)
			out &= mask;
	}

	return out;
}

static uint32_t mdp4_csc_block2base(uint32_t block)
{
	uint32_t base = 0x0;
	switch (block) {
	case MDP_BLOCK_OVERLAY_1:
		base = 0x1A000;
		break;
	case MDP_BLOCK_OVERLAY_2:
		base = (mdp_rev >= MDP_REV_44) ? 0x8A000 : 0x0;
		break;
	case MDP_BLOCK_VG_1:
		base = 0x24000;
		break;
	case MDP_BLOCK_VG_2:
		base = 0x34000;
		break;
	case MDP_BLOCK_DMA_P:
		base = 0x93000;
		break;
	case MDP_BLOCK_DMA_S:
		base = (mdp_rev >= MDP_REV_42) ? 0xA3000 : 0x0;
	default:
		break;
	}
	return base;
}

int mdp4_csc_enable(struct mdp_csc_cfg_data *config)
{
	uint32_t output, base, temp, mask;

	switch (config->block) {
	case MDP_BLOCK_DMA_P:
		base = 0x90070;
		output = (config->csc_data.flags << 3) & (0x08);
		temp = (config->csc_data.flags << 10) & (0x1800);
		output |= temp;
		mask = 0x08 | 0x1800;
		break;
	case MDP_BLOCK_DMA_S:
		base = 0xA0028;
		output = (config->csc_data.flags << 3) & (0x08);
		temp = (config->csc_data.flags << 10) & (0x1800);
		output |= temp;
		mask = 0x08 | 0x1800;
		break;
	case MDP_BLOCK_VG_1:
		base = 0x20058;
		output = (config->csc_data.flags << 11) & (0x800);
		temp = (config->csc_data.flags << 8) & (0x600);
		output |= temp;
		mask = 0x800 | 0x600;
		break;
	case MDP_BLOCK_VG_2:
		base = 0x30058;
		output = (config->csc_data.flags << 11) & (0x800);
		temp = (config->csc_data.flags << 8) & (0x600);
		output |= temp;
		mask = 0x800 | 0x600;
		break;
	case MDP_BLOCK_OVERLAY_1:
		base = 0x18200;
		output = config->csc_data.flags;
		mask = 0x07;
		break;
	case MDP_BLOCK_OVERLAY_2:
		base = 0x88200;
		output = config->csc_data.flags;
		mask = 0x07;
		break;
	default:
		pr_err("%s - CSC block does not exist on MDP_BLOCK = %d\n",
						__func__, config->block);
		return -EINVAL;
	}

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);
	temp = inpdw(MDP_BASE + base) & ~mask;
	output |= temp;
	outpdw(MDP_BASE + base, output);
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
	return 0;
}

#define CSC_MV_OFF	0x400
#define CSC_BV_OFF	0x500
#define CSC_LV_OFF	0x600
#define CSC_POST_OFF	0x80

void mdp4_csc_write(struct mdp_csc_cfg *data, uint32_t base)
{
	int i;
	uint32_t *off;

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);
	off = (uint32_t *) ((uint32_t) base + CSC_MV_OFF);
	for (i = 0; i < 9; i++) {
		outpdw(off, data->csc_mv[i]);
		off++;
	}

	off = (uint32_t *) ((uint32_t) base + CSC_BV_OFF);
	for (i = 0; i < 3; i++) {
		outpdw(off, data->csc_pre_bv[i]);
		outpdw((uint32_t *)((uint32_t)off + CSC_POST_OFF),
					data->csc_post_bv[i]);
		off++;
	}

	off = (uint32_t *) ((uint32_t) base + CSC_LV_OFF);
	for (i = 0; i < 6; i++) {
		outpdw(off, data->csc_pre_lv[i]);
		outpdw((uint32_t *)((uint32_t)off + CSC_POST_OFF),
					data->csc_post_lv[i]);
		off++;
	}
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
}

int mdp4_csc_config(struct mdp_csc_cfg_data *config)
{
	int ret = 0;
	uint32_t base;

	base = mdp4_csc_block2base(config->block);
	if (!base) {
		pr_warn("%s: Block type %d isn't supported by CSC.\n",
				__func__, config->block);
		return -EINVAL;
	}

	mdp4_csc_write(&config->csc_data, (uint32_t) (MDP_BASE + base));

	ret = mdp4_csc_enable(config);

	return ret;
}

void mdp4_init_writeback_buf(struct msm_fb_data_type *mfd, u32 mix_num)
{
	struct mdp_buf_type *buf;

	if (mix_num == MDP4_MIXER0)
		buf = mfd->ov0_wb_buf;
	else
		buf = mfd->ov1_wb_buf;

	buf->ihdl = NULL;
	buf->write_addr = 0;
	buf->read_addr = 0;
}

u32 mdp4_allocate_writeback_buf(struct msm_fb_data_type *mfd, u32 mix_num)
{
	struct mdp_buf_type *buf;
	ion_phys_addr_t	addr, read_addr = 0;
	size_t buffer_size;
	unsigned long len;

	if (mix_num == MDP4_MIXER0)
		buf = mfd->ov0_wb_buf;
	else
		buf = mfd->ov1_wb_buf;

	if (buf->write_addr || !IS_ERR_OR_NULL(buf->ihdl))
		return 0;

	if (!buf->size) {
		pr_err("%s:%d In valid size\n", __func__, __LINE__);
		return -EINVAL;
	}

	buffer_size = roundup(mfd->panel_info.xres * \
		mfd->panel_info.yres * 3 * 2, SZ_4K);

	if (!IS_ERR_OR_NULL(mfd->iclient)) {
		pr_info("%s:%d ion based allocation mfd->mem_hid 0x%x\n",
			__func__, __LINE__, mfd->mem_hid);
		buf->ihdl = ion_alloc(mfd->iclient, buffer_size, SZ_4K,
			mfd->mem_hid);
		if (!IS_ERR_OR_NULL(buf->ihdl)) {
			if (mdp_iommu_split_domain) {
				if (ion_map_iommu(mfd->iclient, buf->ihdl,
					DISPLAY_READ_DOMAIN, GEN_POOL, SZ_4K,
					buffer_size * 2, &read_addr, &len,
					0, 0)) {
					pr_err("ion_map_iommu() read failed\n");
					return -ENOMEM;
				}
				if (mfd->mem_hid & ION_SECURE) {
					if (ion_phys(mfd->iclient, buf->ihdl,
						&addr, (size_t *)&len)) {
						pr_err("%s:%d: ion_phys map failed\n",
							 __func__, __LINE__);
						return -ENOMEM;
					}
				} else {
					if (ion_map_iommu(mfd->iclient,
						buf->ihdl, DISPLAY_WRITE_DOMAIN,
						GEN_POOL, SZ_4K,
						buffer_size * 2, &addr, &len,
						0, 0)) {
						pr_err("split ion_map_iommu() failed\n");
						return -ENOMEM;
					}
				}
			} else {
				if (ion_map_iommu(mfd->iclient, buf->ihdl,
					DISPLAY_READ_DOMAIN, GEN_POOL, SZ_4K,
					buffer_size * 2, &addr, &len, 0, 0)) {
					pr_err("ion_map_iommu() write failed\n");
					return -ENOMEM;
				}
			}
		} else {
			pr_err("%s:%d: ion_alloc failed\n", __func__,
				__LINE__);
			return -ENOMEM;
		}
	} else {
		addr = allocate_contiguous_memory_nomap(buffer_size,
			mfd->mem_hid, 4);
	}
	if (addr) {
		pr_info("allocating %d bytes at %x for mdp writeback\n",
			buffer_size, (u32) addr);
		buf->write_addr = addr;

		if (read_addr)
			buf->read_addr = read_addr;
		else
			buf->read_addr = buf->write_addr;

		return 0;
	} else {
		pr_err("%s cannot allocate memory for mdp writeback!\n",
			 __func__);
		return -ENOMEM;
	}
}

void mdp4_free_writeback_buf(struct msm_fb_data_type *mfd, u32 mix_num)
{
	struct mdp_buf_type *buf;

	if (mix_num == MDP4_MIXER0)
		buf = mfd->ov0_wb_buf;
	else
		buf = mfd->ov1_wb_buf;

	if (!IS_ERR_OR_NULL(mfd->iclient)) {
		if (!IS_ERR_OR_NULL(buf->ihdl)) {
			if (mdp_iommu_split_domain) {
				if (!(mfd->mem_hid & ION_SECURE))
					ion_unmap_iommu(mfd->iclient, buf->ihdl,
						DISPLAY_WRITE_DOMAIN, GEN_POOL);
				ion_unmap_iommu(mfd->iclient, buf->ihdl,
					DISPLAY_READ_DOMAIN, GEN_POOL);
			} else {
				ion_unmap_iommu(mfd->iclient, buf->ihdl,
					DISPLAY_READ_DOMAIN, GEN_POOL);
			}
			ion_free(mfd->iclient, buf->ihdl);
			pr_debug("%s:%d free writeback imem\n", __func__,
				__LINE__);
			buf->ihdl = NULL;
		}
	} else {
		if (buf->write_addr) {
			free_contiguous_memory_by_paddr(buf->write_addr);
			pr_debug("%s:%d free writeback pmem\n", __func__,
				__LINE__);
		}
	}
	buf->write_addr = 0;
	buf->read_addr = 0;
}

static int mdp4_update_pcc_regs(uint32_t offset,
				struct mdp_pcc_cfg_data *cfg_ptr)
{
	int ret = -1;

	if (offset && cfg_ptr) {

		outpdw(offset, cfg_ptr->r.c);
		outpdw(offset + 0x30, cfg_ptr->g.c);
		outpdw(offset + 0x60, cfg_ptr->b.c);
		offset += 4;

		outpdw(offset, cfg_ptr->r.r);
		outpdw(offset + 0x30, cfg_ptr->g.r);
		outpdw(offset + 0x60, cfg_ptr->b.r);
		offset += 4;

		outpdw(offset, cfg_ptr->r.g);
		outpdw(offset + 0x30, cfg_ptr->g.g);
		outpdw(offset + 0x60, cfg_ptr->b.g);
		offset += 4;

		outpdw(offset, cfg_ptr->r.b);
		outpdw(offset + 0x30, cfg_ptr->g.b);
		outpdw(offset + 0x60, cfg_ptr->b.b);
		offset += 4;

		outpdw(offset, cfg_ptr->r.rr);
		outpdw(offset + 0x30, cfg_ptr->g.rr);
		outpdw(offset + 0x60, cfg_ptr->b.rr);
		offset += 4;

		outpdw(offset, cfg_ptr->r.gg);
		outpdw(offset + 0x30, cfg_ptr->g.gg);
		outpdw(offset + 0x60, cfg_ptr->b.gg);
		offset += 4;

		outpdw(offset, cfg_ptr->r.bb);
		outpdw(offset + 0x30, cfg_ptr->g.bb);
		outpdw(offset + 0x60, cfg_ptr->b.bb);
		offset += 4;

		outpdw(offset, cfg_ptr->r.rg);
		outpdw(offset + 0x30, cfg_ptr->g.rg);
		outpdw(offset + 0x60, cfg_ptr->b.rg);
		offset += 4;

		outpdw(offset, cfg_ptr->r.gb);
		outpdw(offset + 0x30, cfg_ptr->g.gb);
		outpdw(offset + 0x60, cfg_ptr->b.gb);
		offset += 4;

		outpdw(offset, cfg_ptr->r.rb);
		outpdw(offset + 0x30, cfg_ptr->g.rb);
		outpdw(offset + 0x60, cfg_ptr->b.rb);
		offset += 4;

		outpdw(offset, cfg_ptr->r.rgb_0);
		outpdw(offset + 0x30, cfg_ptr->g.rgb_0);
		outpdw(offset + 0x60, cfg_ptr->b.rgb_0);
		offset += 4;

		outpdw(offset, cfg_ptr->r.rgb_1);
		outpdw(offset + 0x30, cfg_ptr->g.rgb_1);
		outpdw(offset + 0x60, cfg_ptr->b.rgb_1);

		ret = 0;
	}

	return ret;
}

static int mdp4_read_pcc_regs(uint32_t offset,
				struct mdp_pcc_cfg_data *cfg_ptr)
{
	int ret = -1;

	if (offset && cfg_ptr) {
		cfg_ptr->r.c = inpdw(offset);
		cfg_ptr->g.c = inpdw(offset + 0x30);
		cfg_ptr->b.c = inpdw(offset + 0x60);
		offset += 4;

		cfg_ptr->r.r = inpdw(offset);
		cfg_ptr->g.r = inpdw(offset + 0x30);
		cfg_ptr->b.r = inpdw(offset + 0x60);
		offset += 4;

		cfg_ptr->r.g = inpdw(offset);
		cfg_ptr->g.g = inpdw(offset + 0x30);
		cfg_ptr->b.g = inpdw(offset + 0x60);
		offset += 4;

		cfg_ptr->r.b = inpdw(offset);
		cfg_ptr->g.b = inpdw(offset + 0x30);
		cfg_ptr->b.b = inpdw(offset + 0x60);
		offset += 4;

		cfg_ptr->r.rr = inpdw(offset);
		cfg_ptr->g.rr = inpdw(offset + 0x30);
		cfg_ptr->b.rr = inpdw(offset + 0x60);
		offset += 4;

		cfg_ptr->r.gg = inpdw(offset);
		cfg_ptr->g.gg = inpdw(offset + 0x30);
		cfg_ptr->b.gg = inpdw(offset + 0x60);
		offset += 4;

		cfg_ptr->r.bb = inpdw(offset);
		cfg_ptr->g.bb = inpdw(offset + 0x30);
		cfg_ptr->b.bb = inpdw(offset + 0x60);
		offset += 4;

		cfg_ptr->r.rg = inpdw(offset);
		cfg_ptr->g.rg = inpdw(offset + 0x30);
		cfg_ptr->b.rg = inpdw(offset + 0x60);
		offset += 4;

		cfg_ptr->r.gb = inpdw(offset);
		cfg_ptr->g.gb = inpdw(offset + 0x30);
		cfg_ptr->b.gb = inpdw(offset + 0x60);
		offset += 4;

		cfg_ptr->r.rb = inpdw(offset);
		cfg_ptr->g.rb = inpdw(offset + 0x30);
		cfg_ptr->b.rb = inpdw(offset + 0x60);
		offset += 4;

		cfg_ptr->r.rgb_0 = inpdw(offset);
		cfg_ptr->g.rgb_0 = inpdw(offset + 0x30);
		cfg_ptr->b.rgb_0 = inpdw(offset + 0x60);
		offset += 4;

		cfg_ptr->r.rgb_1 = inpdw(offset);
		cfg_ptr->g.rgb_1 = inpdw(offset + 0x30);
		cfg_ptr->b.rgb_1 = inpdw(offset + 0x60);

		ret = 0;
	}

	return ret;
}


#define MDP_PCC_OFFSET 0xA000
#define MDP_DMA_GC_OFFSET 0x8800
#define MDP_LM_GC_OFFSET 0x4800

#define MDP_DMA_P_OP_MODE_OFFSET 0x70
#define MDP_DMA_S_OP_MODE_OFFSET 0x28
#define MDP_LM_OP_MODE_OFFSET 0x14

#define DMA_PCC_R2_OFFSET 0x100

#define MDP_GC_COLOR_OFFSET	0x100
#define MDP_GC_PARMS_OFFSET	0x80

#define MDP_AR_GC_MAX_STAGES	16

static uint32_t mdp_pp_block2pcc(uint32_t block)
{
	uint32_t valid = 0;

	switch (block) {
	case MDP_BLOCK_DMA_P:
	case MDP_BLOCK_DMA_S:
		valid = (mdp_rev >= MDP_REV_42) ? 1 : 0;
		break;

	default:
		break;
	}

	return valid;
}

int mdp4_pcc_cfg(struct mdp_pcc_cfg_data *cfg_ptr)
{
	int ret = -1;
	uint32_t pcc_offset = 0, mdp_cfg_offset = 0;
	uint32_t mdp_dma_op_mode = 0;
	uint32_t blockbase;

	if (!mdp_pp_block2pcc(cfg_ptr->block))
		return ret;

	blockbase = mdp_block2base(cfg_ptr->block);
	if (!blockbase)
		return ret;

	blockbase += (uint32_t) MDP_BASE;

	switch (cfg_ptr->block) {
	case MDP_BLOCK_DMA_P:
	case MDP_BLOCK_DMA_S:
		pcc_offset = blockbase + MDP_PCC_OFFSET;
		mdp_cfg_offset = blockbase;
		mdp_dma_op_mode = blockbase +
			(MDP_BLOCK_DMA_P == cfg_ptr->block ?
			 MDP_DMA_P_OP_MODE_OFFSET
			 : MDP_DMA_S_OP_MODE_OFFSET);
		break;

	default:
		break;
	}

	if (0x8 & cfg_ptr->ops)
		pcc_offset += DMA_PCC_R2_OFFSET;

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);

	switch ((0x6 & cfg_ptr->ops)>>1) {
	case 0x1:
		ret = mdp4_read_pcc_regs(pcc_offset, cfg_ptr);
		break;

	case 0x2:
		ret = mdp4_update_pcc_regs(pcc_offset, cfg_ptr);
		break;

	default:
		break;
	}

	if (0x8 & cfg_ptr->ops)
		outpdw(mdp_dma_op_mode,
			((inpdw(mdp_dma_op_mode) & ~(0x1<<10)) |
						((0x8 & cfg_ptr->ops)<<10)));

	outpdw(mdp_cfg_offset,
			((inpdw(mdp_cfg_offset) & ~(0x1<<29)) |
						((cfg_ptr->ops & 0x1)<<29)));

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);

	return ret;
}

static uint32_t mdp_pp_block2argc(uint32_t block)
{
	uint32_t valid = 0;

	switch (block) {
	case MDP_BLOCK_DMA_P:
	case MDP_BLOCK_DMA_S:
	case MDP_BLOCK_OVERLAY_0:
	case MDP_BLOCK_OVERLAY_1:
		valid = (mdp_rev >= MDP_REV_42) ? 1 : 0;
		break;

	case MDP_BLOCK_OVERLAY_2:
		valid = (mdp_rev >= MDP_REV_44) ? 1 : 0;
		break;

	default:
		break;
	}

	return valid;
}

static int update_ar_gc_lut(uint32_t *offset, struct mdp_pgc_lut_data *lut_data)
{
	int count = 0;

	uint32_t *c0_offset = offset;
	uint32_t *c0_params_offset = (uint32_t *)((uint32_t)c0_offset
							+ MDP_GC_PARMS_OFFSET);

	uint32_t *c1_offset = (uint32_t *)((uint32_t)offset
							+ MDP_GC_COLOR_OFFSET);

	uint32_t *c1_params_offset = (uint32_t *)((uint32_t)c1_offset
							+ MDP_GC_PARMS_OFFSET);

	uint32_t *c2_offset = (uint32_t *)((uint32_t)offset
						+ 2*MDP_GC_COLOR_OFFSET);

	uint32_t *c2_params_offset = (uint32_t *)((uint32_t)c2_offset
						+MDP_GC_PARMS_OFFSET);


	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);
	for (count = 0; count < MDP_AR_GC_MAX_STAGES; count++) {
		if (count < lut_data->num_r_stages) {
			outpdw(c0_offset+count,
				((0xfff & lut_data->r_data[count].x_start)
					| 0x10000));

			outpdw(c0_params_offset+count,
				((0x7fff & lut_data->r_data[count].slope)
					| ((0xffff
					& lut_data->r_data[count].offset)
						<< 16)));
		} else
			outpdw(c0_offset+count, 0);

		if (count < lut_data->num_b_stages) {
			outpdw(c1_offset+count,
				((0xfff & lut_data->b_data[count].x_start)
					| 0x10000));

			outpdw(c1_params_offset+count,
				((0x7fff & lut_data->b_data[count].slope)
					| ((0xffff
					& lut_data->b_data[count].offset)
						<< 16)));
		} else
			outpdw(c1_offset+count, 0);

		if (count < lut_data->num_g_stages) {
			outpdw(c2_offset+count,
				((0xfff & lut_data->g_data[count].x_start)
					| 0x10000));

			outpdw(c2_params_offset+count,
				((0x7fff & lut_data->g_data[count].slope)
				| ((0xffff
				& lut_data->g_data[count].offset)
					<< 16)));
		} else
			outpdw(c2_offset+count, 0);
	}

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);

	return 0;
}

static int mdp4_argc_process_write_req(uint32_t *offset,
		struct mdp_pgc_lut_data *pgc_ptr)
{
	int ret = -1;
	struct mdp_ar_gc_lut_data r[MDP_AR_GC_MAX_STAGES];
	struct mdp_ar_gc_lut_data g[MDP_AR_GC_MAX_STAGES];
	struct mdp_ar_gc_lut_data b[MDP_AR_GC_MAX_STAGES];

	ret = copy_from_user(&r[0], pgc_ptr->r_data,
		pgc_ptr->num_r_stages * sizeof(struct mdp_ar_gc_lut_data));

	if (!ret) {
		ret = copy_from_user(&g[0],
				pgc_ptr->g_data,
				pgc_ptr->num_g_stages
				* sizeof(struct mdp_ar_gc_lut_data));
		if (!ret)
			ret = copy_from_user(&b[0],
					pgc_ptr->b_data,
					pgc_ptr->num_b_stages
					* sizeof(struct mdp_ar_gc_lut_data));
	}

	if (ret)
		return ret;

	pgc_ptr->r_data = &r[0];
	pgc_ptr->g_data = &g[0];
	pgc_ptr->b_data = &b[0];

	ret = update_ar_gc_lut(offset, pgc_ptr);
	return ret;
}

int mdp4_argc_cfg(struct mdp_pgc_lut_data *pgc_ptr)
{
	int ret = -1;
	uint32_t *offset = 0, *pgc_enable_offset = 0, lshift_bits = 0;
	uint32_t blockbase;

	if (!mdp_pp_block2argc(pgc_ptr->block))
		return ret;

	blockbase = mdp_block2base(pgc_ptr->block);
	if (!blockbase)
		return ret;

	blockbase += (uint32_t) MDP_BASE;
	ret = 0;

	switch (pgc_ptr->block) {
	case MDP_BLOCK_DMA_P:
	case MDP_BLOCK_DMA_S:
		offset = (uint32_t *)(blockbase + MDP_DMA_GC_OFFSET);
		pgc_enable_offset = (uint32_t *) blockbase;
		lshift_bits = 28;
		break;

	case MDP_BLOCK_OVERLAY_0:
	case MDP_BLOCK_OVERLAY_1:
	case MDP_BLOCK_OVERLAY_2:
		offset = (uint32_t *)(blockbase + MDP_LM_GC_OFFSET);
		pgc_enable_offset = (uint32_t *)(blockbase
				+ MDP_LM_OP_MODE_OFFSET);
		lshift_bits = 2;
		break;

	default:
		ret = -1;
		break;
	}

	if (!ret) {

		switch ((0x6 & pgc_ptr->flags)>>1) {
		case 0x1:
			ret = -ENOTTY;
			break;

		case 0x2:
			ret = mdp4_argc_process_write_req(offset, pgc_ptr);
			break;

		default:
			break;
		}

		if (!ret) {
			mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);
			outpdw(pgc_enable_offset, (inpdw(pgc_enable_offset) &
							~(0x1<<lshift_bits)) |
				((0x1 & pgc_ptr->flags) << lshift_bits));
			mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF,
									FALSE);
		}
	}

	return ret;
}

static uint32_t mdp4_pp_block2igc(uint32_t block)
{
	uint32_t valid = 0;
	switch (block) {
	case MDP_BLOCK_VG_1:
		valid = 0x1;
		break;
	case MDP_BLOCK_VG_2:
		valid = 0x1;
		break;
	case MDP_BLOCK_RGB_1:
		valid = 0x1;
		break;
	case MDP_BLOCK_RGB_2:
		valid = 0x1;
		break;
	case MDP_BLOCK_DMA_P:
		valid = (mdp_rev >= MDP_REV_40) ? 1 : 0;
		break;
	case MDP_BLOCK_DMA_S:
		valid = (mdp_rev >= MDP_REV_40) ? 1 : 0;
		break;
	default:
		break;
	}
	return valid;
}

static int mdp4_igc_lut_write(struct mdp_igc_lut_data *cfg, uint32_t en_off,
		uint32_t lut_off)
{
	int i;
	uint32_t base, *off_low, *off_high;
	uint32_t low[cfg->len];
	uint32_t high[cfg->len];

	base = mdp_block2base(cfg->block);

	if (cfg->len != 256)
		return -EINVAL;

	off_low = (uint32_t *)(MDP_BASE + base + lut_off);
	off_high = (uint32_t *)(MDP_BASE + base + lut_off + 0x800);
	if (copy_from_user(&low, cfg->c0_c1_data, cfg->len * sizeof(uint32_t)))
		return -EFAULT;
	if (copy_from_user(&high, cfg->c2_data, cfg->len * sizeof(uint32_t)))
		return -EFAULT;

	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);
	for (i = 0; i < cfg->len; i++) {
		MDP_OUTP(off_low++, low[i]);
		/*low address write should occur before high address write*/
		wmb();
		MDP_OUTP(off_high++, high[i]);
	}
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
	return 0;
}

static int mdp4_igc_lut_ctrl(struct mdp_igc_lut_data *cfg)
{
	uint32_t mask, out;
	uint32_t base = mdp_block2base(cfg->block);
	int8_t shift = 0;

	switch (cfg->block) {
	case MDP_BLOCK_DMA_P:
	case MDP_BLOCK_DMA_S:
		base = base;
		shift = 30;
		break;
	case MDP_BLOCK_VG_1:
	case MDP_BLOCK_VG_2:
	case MDP_BLOCK_RGB_1:
	case MDP_BLOCK_RGB_2:
		base += 0x58;
		shift = 16;
		break;
	default:
		return -EINVAL;

	}
	out = 1<<shift;
	mask = ~out;
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);
	out = inpdw(MDP_BASE + base) & mask;
	MDP_OUTP(MDP_BASE + base, out | ((cfg->ops & 0x1)<<shift));
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);

	return 0;
}

static int mdp4_igc_lut_write_cfg(struct mdp_igc_lut_data *cfg)
{
	int ret = 0;

	switch (cfg->block) {
	case MDP_BLOCK_DMA_P:
	case MDP_BLOCK_DMA_S:
		ret = mdp4_igc_lut_write(cfg, 0x00, 0x9000);
		break;
	case MDP_BLOCK_VG_1:
	case MDP_BLOCK_VG_2:
	case MDP_BLOCK_RGB_1:
	case MDP_BLOCK_RGB_2:
		ret = mdp4_igc_lut_write(cfg, 0x58, 0x5000);
		break;
	default:
		ret = -EINVAL;
	}

	return ret;
}

int mdp4_igc_lut_config(struct mdp_igc_lut_data *cfg)
{
	int ret = 0;

	if (!mdp4_pp_block2igc(cfg->block)) {
		ret = -ENOTTY;
		goto error;
	}

	switch ((cfg->ops & 0x6) >> 1) {
	case 0x1:
		pr_info("%s: IGC LUT read not supported\n", __func__);
		break;
	case 0x2:
		ret = mdp4_igc_lut_write_cfg(cfg);
		if (ret)
			goto error;
		break;
	default:
		break;
	}

	ret = mdp4_igc_lut_ctrl(cfg);

error:
	return ret;
}

#define QSEED_TABLE_1_COUNT	2
#define QSEED_TABLE_2_COUNT	1024

static uint32_t mdp4_pp_block2qseed(uint32_t block)
{
	uint32_t valid = 0;
	switch (block) {
	case MDP_BLOCK_VG_1:
	case MDP_BLOCK_VG_2:
		valid = 0x1;
		break;
	default:
		break;
	}
	return valid;
}

int mdp4_qseed_access_cfg(struct mdp_qseed_cfg *config, uint32_t base)
{
	int i, ret = 0;
	uint32_t *values;

	if ((config->table_num != 1) && (config->table_num != 2)) {
		ret = -ENOTTY;
		goto error;
	}

	if (((config->table_num == 1) && (config->len != QSEED_TABLE_1_COUNT))
			|| ((config->table_num == 2) &&
				(config->len != QSEED_TABLE_2_COUNT))) {
		ret = -EINVAL;
		goto error;
	}

	values = kmalloc(config->len * sizeof(uint32_t), GFP_KERNEL);
	if (!values) {
		ret = -ENOMEM;
		goto error;
	}

	base += (config->table_num == 1) ? MDP4_QSEED_TABLE1_OFF :
							MDP4_QSEED_TABLE2_OFF;

	if (config->ops & MDP_PP_OPS_WRITE) {
		ret = copy_from_user(values, config->data,
						sizeof(uint32_t) * config->len);
		if (ret) {
			pr_warn("%s: Error copying from user, %d", __func__,
									ret);
			ret = -EINVAL;
			goto err_mem;
		}
		mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);
		for (i = 0; i < config->len; i++) {
			if (!(base & 0x3FF))
				wmb();
			MDP_OUTP(base , values[i]);
			base += sizeof(uint32_t);
		}
		mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
	} else if (config->ops & MDP_PP_OPS_READ) {
		mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);
		for (i = 0; i < config->len; i++) {
			values[i] = inpdw(base);
			if (!(base & 0x3FF))
				rmb();
			base += sizeof(uint32_t);
		}
		mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
		ret = copy_to_user(config->data, values,
						sizeof(uint32_t) * config->len);
		if (ret) {
			pr_warn("%s: Error copying to user, %d", __func__, ret);
			ret = -EINVAL;
			goto err_mem;
		}
	}

err_mem:
	kfree(values);
error:
	return ret;
}

int mdp4_qseed_cfg(struct mdp_qseed_cfg_data *config)
{
	int ret = 0;
	struct mdp_qseed_cfg *cfg = &config->qseed_data;
	uint32_t base;

	if (!mdp4_pp_block2qseed(config->block)) {
		ret = -ENOTTY;
		goto error;
	}

	if ((cfg->ops & MDP_PP_OPS_READ) && (cfg->ops & MDP_PP_OPS_WRITE)) {
		ret = -EPERM;
		pr_warn("%s: Cannot read and write on the same request\n",
								__func__);
		goto error;
	}
	base = (uint32_t) (MDP_BASE + mdp_block2base(config->block));
	ret = mdp4_qseed_access_cfg(cfg, base);

error:
	return ret;
}
