/*
 * nau8814.c  --  NAU8814 ALSA Soc Audio driver
 *
 * Copyright 2006 Wolfson Microelectronics PLC.
 *
 *          
 * Author: CFYang modify
 *         Liam Girdwood <lrg@slimlogic.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/pm.h>
#include <linux/i2c.h>
#include <linux/platform_device.h>
#include <linux/spi/spi.h>
#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/soc.h>
#include <sound/soc-dapm.h>
#include <sound/initval.h>

#include "nau8814.h"

#define NAU8814_VERSION 	"1.6.39"

/*
 * nau8814 register cache
 * We can't read the NAU8814 register space when we are
 * using 2 wire for device control, so we cache them instead.
 */
static const u16 nau8814_reg_init_tbl[][2] = 
{
    {0x01 , 0x011d},	/* CFG_POWER_MANAGEMENT1 */
    {0x02 , 0x0015},	/* CFG_POWER_MANAGEMENT2 */
    {0x03 , 0x0075},	/* CFG_POWER_MANAGEMENT3 */
    {0x04 , 0x0018},	/* CFG_AUDIO_INTERFACE */
    {0x06 , 0x0000},	/* CFG_CLK_GEN_CTRL */
    {0x07 , 0x000a},	/* CFG_ADDITIONAL_CTRL */
    {0x0A , 0x0008},	/* CFG_DAC_CTRL */
    {0x0E , 0x0108},	/* CFG_ADC_CTRL */
    {0x12 , 0x0000},	/* CFG_EQ1_SHELF_LOW */
    {0x13 , 0x0000},	/* CFG_EQ2_PEAK1 */
    {0x14 , 0x0000},	/* CFG_EQ3_PEAK2 */
    {0x15 , 0x0000},	/* CFG_EQ4_PEAK3 */
    {0x16 , 0x0000},	/* CFG_EQ5_HIGH_SHELF */
    {0x2C , 0x0003},	/* CFG_INPUT_CTRL */
    {0x2D , 0x0020},	/* CFG_LEFT_INP_PGA_GAIN_CTRL */
    {0x32 , 0x0001},	/* CFG_LEFT_MIXER_CTRL */
    {0x38 , 0x0000},	/* CFG_OUT3_MIXER_CTRL */
    {0x3A , 0x0000},
    {0x3B , 0x0000},
    {0x3C , 0x0000},
    {0x49 , 0x0000},
    {0x4B , 0x0000},
    {0x4C , 0x0000},
    {0x4D , 0x0000},
    {0x4E , 0x0000},
    {0x4F , 0x0108},
};

/*
 * read nau8814 register cache
 */
static inline unsigned int nau8814_read_reg_cache(struct snd_soc_codec  *codec,
  unsigned int reg)
{
	u16 *cache = codec->reg_cache;
	if (reg == NAU8814_RESET)
	{
		return 0;
	}
	if (reg >= NAU8814_CACHEREGNUM)
	{
		return -1;
	}
	return cache[reg];
}

/*
 * write nau8814 register cache
 */
static inline void nau8814_write_reg_cache(struct snd_soc_codec  *codec,
  u16 reg, unsigned int value)
{
	u16 *cache = codec->reg_cache;

	if (reg >= NAU8814_CACHEREGNUM)
	{
		return;
	}
	cache[reg] = value;
}

/*
 * write to the NAU8814 register space
 */
static int nau8814_write(struct snd_soc_codec  *codec, unsigned int reg, unsigned int value)
{
	u8 data[2];

	/* data is
	*   D15..D9 NAU8814 register offset
	*   D8...D0 register data
	*/
	data[0] = (reg << 1) | ((value >> 8) & 0x0001);
	data[1] = value & 0x00ff;

	nau8814_write_reg_cache (codec, reg, value);
	if (codec->hw_write(codec->control_data, data, 2) == 2)
	{
		return 0;
	}
	else
	{
		pr_err("%s=> reg:0x%02X,value:0x%04x error\n", __FUNCTION__, reg, value);
		return -1;
	}
}

static unsigned int nau8814_read(struct snd_soc_codec *codec, unsigned int reg)
{
	u8 data[2]={0};
	unsigned int value=0x0;
	
	if (reg == NAU8814_RESET)
		return 0;

	data[0] = (reg<<1);
	if(codec->hw_write(codec->control_data, data, 1) == 1)
	{
		i2c_master_recv(codec->control_data, data, 2);
		value = ((data[0] & 0x01) << 8) | data[1];
		return value;
	}
	else
	{
		pr_err("nau8814 read reg:0x%X failed.\n", reg);
		return -EIO;
	}
}

static inline void nau8814_reg_init(struct snd_soc_codec *codec)
{
	u32 i;

	for(i = 0; i < ARRAY_SIZE(nau8814_reg_init_tbl); i++)
	{
		nau8814_write(codec, nau8814_reg_init_tbl[i][0], nau8814_reg_init_tbl[i][1]);
	}
}

#if 0
static int nau8814_write_mask(struct snd_soc_codec *codec, unsigned int reg,unsigned int value,unsigned int mask)
{
  unsigned char RetVal=0;
  unsigned  int CodecData;
  if(!mask)
    return 0;
  if(mask!=0xffff) {
    CodecData=nau8814_read(codec,reg);
    CodecData&=~mask;
    CodecData|=(value&mask);
    RetVal=nau8814_write(codec,reg,CodecData);
  }
  else {
    RetVal=nau8814_write(codec,reg,value);
  }
  return RetVal;
}

void nau8814DumpRegister(struct snd_soc_codec *codec)
{
	int i;

	pr_warn("\n");
	pr_warn("\n>>>>>>>>>> Dump Reg Cache <<<<<<<<<<<<\n");
	for(i = 0; i < NAU8814_CACHEREGNUM; i++)
	{
		if((i%8 == 0) && i)
		{
			pr_warn("\n");
		}
		pr_warn("R0x%02X:0x%04x  ", i, ((u16 *)codec->reg_cache)[i]);
	}
	pr_warn("\n");
	pr_warn("\n");
}
#endif

/* codec private data */
struct nau8814_priv {
	struct snd_soc_codec codec;
	unsigned int f_pllout;
	unsigned int f_mclk;
	unsigned int f_256fs;
	unsigned int f_opclk;
	void *control_data;
	int mclk_idx;
	u16 reg_cache[NAU8814_CACHEREGNUM];
};

#define NAU8814_POWER1_BIASEN  0x08
#define NAU8814_POWER1_BUFIOEN 0x10

int update(struct snd_kcontrol *kcontrol, struct snd_ctl_elem_value *ucontrol)
{
	struct soc_mixer_control *mc =
		(struct soc_mixer_control *)kcontrol->private_value;
	struct snd_soc_codec *codec = snd_kcontrol_chip(kcontrol);
	unsigned int reg = mc->reg;
	unsigned int shift = mc->shift;
	unsigned int rshift = mc->rshift;
	int max = mc->max;
	unsigned int mask = (1 << fls(max)) - 1;
	unsigned int invert = mc->invert;
	unsigned int val, val2, val_mask;
//snd_soc_update_bits_locked declare
	int change;
	unsigned int old, new;
//snd_soc_update_bits_locked declare end //

	val = (ucontrol->value.integer.value[0] & mask);
	if (invert)
		val = max - val;
	val_mask = mask << shift;
	val = val << shift;
	if (shift != rshift) {
		val2 = (ucontrol->value.integer.value[1] & mask);
		if (invert)
			val2 = max - val2;
		val_mask |= mask << rshift;
		val |= val2 << rshift;
	}
//return snd_soc_update_bits_locked(codec, reg, val_mask, val);
	old = snd_soc_read(codec, reg);
	new = (old & ~mask) | val;
	change = old != new;
  new |= 0x100; //add update bit for  b,c,2d,2e,34,35,36,37
	if (change)
		snd_soc_write(codec, reg, new);

	return change;
}

static const char *nau8814_companding[] = { "Off", "NC", "u-law", "A-law" };
static const char *nau8814_deemp[] = { "None", "32kHz", "44.1kHz", "48kHz" };
static const char *nau8814_alc[] = { "ALC", "Limiter" };

static const struct soc_enum nau8814_enum[] = {
	SOC_ENUM_SINGLE(NAU8814_COMP, 1, 4, nau8814_companding), /* adc */
	SOC_ENUM_SINGLE(NAU8814_COMP, 3, 4, nau8814_companding), /* dac */
	SOC_ENUM_SINGLE(NAU8814_DAC,  4, 4, nau8814_deemp),
	SOC_ENUM_SINGLE(NAU8814_ALC3,  8, 2, nau8814_alc),
};

static const struct snd_kcontrol_new nau8814_snd_controls[] = {

SOC_SINGLE("Digital Loopback Switch", NAU8814_COMP, 0, 1, 0),

SOC_ENUM("DAC Companding", nau8814_enum[1]),
SOC_ENUM("ADC Companding", nau8814_enum[0]),

SOC_ENUM("Playback De-emphasis", nau8814_enum[2]),
SOC_SINGLE("DAC Inversion Switch", NAU8814_DAC, 0, 1, 0),

SOC_SINGLE_EXT("Master Playback Volume", NAU8814_DACVOL, 0, 127, 0, snd_soc_get_volsw, update),

SOC_SINGLE("High Pass Filter Switch", NAU8814_ADC, 8, 1, 0),
SOC_SINGLE("High Pass Cut Off", NAU8814_ADC, 4, 7, 0),
SOC_SINGLE("ADC Inversion Switch", NAU8814_COMP, 0, 1, 0),

SOC_SINGLE_EXT("Capture Volume", NAU8814_ADCVOL,  0, 127, 0, snd_soc_get_volsw, update),

//x11
//SOC_ENUM("Equaliser Function", nau8814_enum[3]),
//SOC_ENUM("EQ1 Cut Off", nau8814_enum[4]),
//SOC_SINGLE("EQ1 Volume", NAU8814_EQ1,  0, 31, 1),
//x13
//SOC_ENUM("Equaliser EQ2 Bandwith", nau8814_enum[5]),
//SOC_ENUM("EQ2 Cut Off", nau8814_enum[6]),
//SOC_SINGLE("EQ2 Volume", NAU8814_EQ2,  0, 31, 1),
//x14
//SOC_ENUM("Equaliser EQ3 Bandwith", nau8814_enum[7]),
//SOC_ENUM("EQ3 Cut Off", nau8814_enum[8]),
//SOC_SINGLE("EQ3 Volume", NAU8814_EQ3,  0, 31, 1),
//x15
//SOC_ENUM("Equaliser EQ4 Bandwith", nau8814_enum[9]),
//SOC_ENUM("EQ4 Cut Off", nau8814_enum[10]),
//SOC_SINGLE("EQ4 Volume", NAU8814_EQ4,  0, 31, 1),
//x16
//SOC_ENUM("Equaliser EQ5 Bandwith", nau8814_enum[11]),
//SOC_ENUM("EQ5 Cut Off", nau8814_enum[12]),
//SOC_SINGLE("EQ5 Volume", NAU8814_EQ5,  0, 31, 1),

SOC_SINGLE("DAC Playback Limiter Switch", NAU8814_DACLIM1,  8, 1, 0),
SOC_SINGLE("DAC Playback Limiter Decay", NAU8814_DACLIM1,  4, 15, 0),
SOC_SINGLE("DAC Playback Limiter Attack", NAU8814_DACLIM1,  0, 15, 0),

SOC_SINGLE("DAC Playback Limiter Threshold", NAU8814_DACLIM2,  4, 7, 0),
SOC_SINGLE("DAC Playback Limiter Boost", NAU8814_DACLIM2,  0, 15, 0),

SOC_SINGLE("ALC Enable Switch", NAU8814_ALC1,  8, 1, 0),
SOC_SINGLE("ALC Capture Max Gain", NAU8814_ALC1,  3, 7, 0),
SOC_SINGLE("ALC Capture Min Gain", NAU8814_ALC1,  0, 7, 0),

SOC_SINGLE("ALC Capture ZC Switch", NAU8814_ALC2,  8, 1, 0),
SOC_SINGLE("ALC Capture Hold", NAU8814_ALC2,  4, 7, 0),
SOC_SINGLE("ALC Capture Target", NAU8814_ALC2,  0, 15, 0),

SOC_ENUM("ALC Capture Mode", nau8814_enum[3]),
SOC_SINGLE("ALC Capture Decay", NAU8814_ALC3,  4, 15, 0),
SOC_SINGLE("ALC Capture Attack", NAU8814_ALC3,  0, 15, 0),

SOC_SINGLE("ALC Capture Noise Gate Switch", NAU8814_NGATE,  3, 1, 0),
SOC_SINGLE("ALC Capture Noise Gate Threshold", NAU8814_NGATE,  0, 7, 0),

SOC_SINGLE("Capture PGA ZC Switch", NAU8814_INPPGA,  7, 1, 0),
SOC_SINGLE("Capture PGA Volume", NAU8814_INPPGA,  0, 63, 0),

SOC_SINGLE("Speaker Playback ZC Switch", NAU8814_SPKVOL,  7, 1, 0),
SOC_SINGLE("Speaker Playback Switch", NAU8814_SPKVOL,  6, 1, 1),
SOC_SINGLE("Speaker Playback Volume", NAU8814_SPKVOL,  0, 63, 0),
SOC_SINGLE("Speaker Boost", NAU8814_OUTPUT, 2, 1, 0),

SOC_SINGLE("Capture Boost(+20dB)", NAU8814_ADCBOOST,  8, 1, 0),
SOC_SINGLE("Mono Playback Switch", NAU8814_MONOMIX, 6, 1, 1),
};

/* Speaker Output Mixer */
static const struct snd_kcontrol_new nau8814_speaker_mixer_controls[] = {
SOC_DAPM_SINGLE("Line Bypass Switch", NAU8814_SPKMIX, 1, 1, 0),
SOC_DAPM_SINGLE("Aux Playback Switch", NAU8814_SPKMIX, 5, 1, 0),
SOC_DAPM_SINGLE("PCM Playback Switch", NAU8814_SPKMIX, 0, 1, 0),
};

/* Mono Output Mixer */
static const struct snd_kcontrol_new nau8814_mono_mixer_controls[] = {
SOC_DAPM_SINGLE("Line Bypass Switch", NAU8814_MONOMIX, 1, 1, 0),
SOC_DAPM_SINGLE("Aux Playback Switch", NAU8814_MONOMIX, 2, 1, 0),
SOC_DAPM_SINGLE("PCM Playback Switch", NAU8814_MONOMIX, 0, 1, 0),
};

static const struct snd_kcontrol_new nau8814_boost_controls[] = {
SOC_DAPM_SINGLE("Mic PGA Switch", NAU8814_INPPGA,  6, 1, 1),
SOC_DAPM_SINGLE("Aux Volume", NAU8814_ADCBOOST, 0, 7, 0),
SOC_DAPM_SINGLE("Mic Volume", NAU8814_ADCBOOST, 4, 7, 0),
};

static const struct snd_kcontrol_new nau8814_micpga_controls[] = {
SOC_DAPM_SINGLE("MICP Switch", NAU8814_INPUT, 0, 1, 0),
SOC_DAPM_SINGLE("MICN Switch", NAU8814_INPUT, 1, 1, 0),
SOC_DAPM_SINGLE("AUX Switch", NAU8814_INPUT, 2, 1, 0),
};

static const struct snd_soc_dapm_widget nau8814_dapm_widgets[] = {
SND_SOC_DAPM_MIXER("Speaker Mixer", NAU8814_POWER3, 2, 0,
	&nau8814_speaker_mixer_controls[0],
	ARRAY_SIZE(nau8814_speaker_mixer_controls)),
SND_SOC_DAPM_MIXER("Mono Mixer", NAU8814_POWER3, 3, 0,
	&nau8814_mono_mixer_controls[0],
	ARRAY_SIZE(nau8814_mono_mixer_controls)),
SND_SOC_DAPM_DAC("DAC", "HiFi Playback", NAU8814_POWER3, 0, 0),
SND_SOC_DAPM_ADC("ADC", "HiFi Capture", NAU8814_POWER2, 0, 0),
SND_SOC_DAPM_PGA("Aux Input", NAU8814_POWER1, 6, 0, NULL, 0),
SND_SOC_DAPM_PGA("SpkN Out", NAU8814_POWER3, 5, 0, NULL, 0),
SND_SOC_DAPM_PGA("SpkP Out", NAU8814_POWER3, 6, 0, NULL, 0),
SND_SOC_DAPM_PGA("Mono Out", NAU8814_POWER3, 7, 0, NULL, 0),

SND_SOC_DAPM_MIXER("Mic PGA", NAU8814_POWER2, 2, 0,
		   &nau8814_micpga_controls[0],
		   ARRAY_SIZE(nau8814_micpga_controls)),
SND_SOC_DAPM_MIXER("Boost Mixer", NAU8814_POWER2, 4, 0,
	&nau8814_boost_controls[0],
	ARRAY_SIZE(nau8814_boost_controls)),

SND_SOC_DAPM_MICBIAS("Mic Bias", NAU8814_POWER1, 4, 0),

SND_SOC_DAPM_INPUT("MICN"),
SND_SOC_DAPM_INPUT("MICP"),
SND_SOC_DAPM_INPUT("AUX"),
SND_SOC_DAPM_OUTPUT("MONOOUT"),
SND_SOC_DAPM_OUTPUT("SPKOUTP"),
SND_SOC_DAPM_OUTPUT("SPKOUTN"),
};

static const struct snd_soc_dapm_route audio_map[] = {
	/* Mono output mixer */
	{"Mono Mixer", "PCM Playback Switch", "DAC"},
	{"Mono Mixer", "Aux Playback Switch", "Aux Input"},
	{"Mono Mixer", "Line Bypass Switch", "Boost Mixer"},

	/* Speaker output mixer */
	{"Speaker Mixer", "PCM Playback Switch", "DAC"},
	{"Speaker Mixer", "Aux Playback Switch", "Aux Input"},
	{"Speaker Mixer", "Line Bypass Switch", "Boost Mixer"},

	/* Outputs */
	{"Mono Out", NULL, "Mono Mixer"},
	{"MONOOUT", NULL, "Mono Out"},
	{"SpkN Out", NULL, "Speaker Mixer"},
	{"SpkP Out", NULL, "Speaker Mixer"},
	{"SPKOUTN", NULL, "SpkN Out"},
	{"SPKOUTP", NULL, "SpkP Out"},

	/* Microphone PGA */
	{"Mic PGA", "MICN Switch", "MICN"},
	{"Mic PGA", "MICP Switch", "MICP"},
	{"Mic PGA", "AUX Switch", "Aux Input" },

	/* Boost Mixer */
	{"Boost Mixer", "Mic PGA Switch", "Mic PGA"},
	{"Boost Mixer", "Mic Volume", "MICP"},
	{"Boost Mixer", "Aux Volume", "Aux Input"},

	{"ADC", NULL, "Boost Mixer"},
};

static int nau8814_add_widgets(struct snd_soc_codec *codec)
{
#if 0
	snd_soc_dapm_new_controls(&codec->dapm, nau8814_dapm_widgets,
				  ARRAY_SIZE(nau8814_dapm_widgets));

	snd_soc_dapm_add_routes(&codec->dapm, audio_map, ARRAY_SIZE(audio_map));

	snd_soc_dapm_new_widgets(&codec->dapm);
#endif
	return 0;
}

struct pll_ {
	unsigned int pre_div:4; /* prescale - 1 */
	unsigned int n:4;
	unsigned int k;
};

static struct pll_ pll_div;

/* The size in bits of the pll divide multiplied by 10
 * to allow rounding later */
#define FIXED_PLL_SIZE ((1 << 24) * 10)

static void pll_factors(unsigned int target, unsigned int source)
{
	unsigned long long Kpart;
	unsigned int K, Ndiv, Nmod;
	
	Ndiv = target / source;
	if (Ndiv < 6) {
		source >>= 1;
		pll_div.pre_div = 1;
		Ndiv = target / source;
	} else
		pll_div.pre_div = 0;

	if ((Ndiv < 6) || (Ndiv > 12))
		printk(KERN_WARNING
			"NAU8814 N value %u outwith recommended range!d\n",
			Ndiv);

	pll_div.n = Ndiv;
	Nmod = target % source;
	Kpart = FIXED_PLL_SIZE * (long long)Nmod;

	do_div(Kpart, source);

	K = Kpart & 0xFFFFFFFF;

	/* Check if we need to round */
	if ((K % 10) >= 5)
		K += 5;

	/* Move down to proper range now rounding is done */
	K /= 10;

	pll_div.k = K;
}

static int nau8814_set_dai_pll(struct snd_soc_dai *codec_dai,
		int pll_id, int source, unsigned int freq_in, unsigned int freq_out)
{
	struct snd_soc_codec *codec = codec_dai->codec;
	u16 reg;

	if (freq_in == 0 || freq_out == 0) {
		/* Clock CODEC directly from MCLK */
		reg = nau8814_read_reg_cache(codec, NAU8814_CLOCK);
		nau8814_write(codec, NAU8814_CLOCK, reg & 0x0ff);

		/* Turn off PLL */
		reg = nau8814_read_reg_cache(codec, NAU8814_POWER1);
		nau8814_write(codec, NAU8814_POWER1, reg & 0x1df);
		return 0;
	}

	pll_factors(freq_out*4, freq_in);

	nau8814_write(codec, NAU8814_PLLN, (pll_div.pre_div << 4) | pll_div.n);
	nau8814_write(codec, NAU8814_PLLK1, pll_div.k >> 18);
	nau8814_write(codec, NAU8814_PLLK2, (pll_div.k >> 9) & 0x1ff);
	nau8814_write(codec, NAU8814_PLLK3, pll_div.k & 0x1ff);
	reg = nau8814_read_reg_cache(codec, NAU8814_POWER1);
	nau8814_write(codec, NAU8814_POWER1, reg | 0x020);

	/* Run CODEC from PLL instead of MCLK */
	reg = nau8814_read_reg_cache(codec, NAU8814_CLOCK);
	nau8814_write(codec, NAU8814_CLOCK, reg | 0x100);

	return 0;
}

/*
 * Configure NAU8814 clock dividers.
 */
static int nau8814_set_dai_clkdiv(struct snd_soc_dai *codec_dai,
		int div_id, int div)
{
	struct snd_soc_codec *codec = codec_dai->codec;
	u16 reg;
	
	switch (div_id) {
	case NAU8814_OPCLKDIV:
		reg = nau8814_read_reg_cache(codec, NAU8814_GPIO) & 0x1cf;
		nau8814_write(codec, NAU8814_GPIO, reg | div);
		break;
	case NAU8814_MCLKDIV:
		reg = nau8814_read_reg_cache(codec, NAU8814_CLOCK) & 0x11f;
		nau8814_write(codec, NAU8814_CLOCK, reg | div);
		break;
	case NAU8814_ADCCLK:
		reg = nau8814_read_reg_cache(codec, NAU8814_ADC) & 0x1f7;
		nau8814_write(codec, NAU8814_ADC, reg | div);
		break;
	case NAU8814_DACCLK:
		reg = nau8814_read_reg_cache(codec, NAU8814_DAC) & 0x1f7;
		nau8814_write(codec, NAU8814_DAC, reg | div);
		break;
	case NAU8814_BCLKDIV:
		reg = nau8814_read_reg_cache(codec, NAU8814_CLOCK) & 0x1e3;
		nau8814_write(codec, NAU8814_CLOCK, reg | div);
		break;
	default:
		return -EINVAL;
	}

	return 0;
}

static int nau8814_set_dai_fmt(struct snd_soc_dai *codec_dai,
		unsigned int fmt)
{
	struct snd_soc_codec *codec = codec_dai->codec;
	u16 iface = 0;
	u16 clk = nau8814_read_reg_cache(codec, NAU8814_CLOCK) & 0x1fe;
	
	/* set master/slave audio interface */
	switch (fmt & SND_SOC_DAIFMT_MASTER_MASK) {
	case SND_SOC_DAIFMT_CBM_CFM:
		clk |= 0x0001;
		break;
	case SND_SOC_DAIFMT_CBS_CFS:
		break;
	default:
		return -EINVAL;
	}

	/* interface format */
	switch (fmt & SND_SOC_DAIFMT_FORMAT_MASK) {
	case SND_SOC_DAIFMT_I2S:
		iface |= 0x0010;
		break;
	case SND_SOC_DAIFMT_RIGHT_J:
		break;
	case SND_SOC_DAIFMT_LEFT_J:
		iface |= 0x0008;
		break;
	case SND_SOC_DAIFMT_DSP_A:
		iface |= 0x00018;
		break;
	default:
		return -EINVAL;
	}

	/* clock inversion */
	switch (fmt & SND_SOC_DAIFMT_INV_MASK) {
	case SND_SOC_DAIFMT_NB_NF:
		break;
	case SND_SOC_DAIFMT_IB_IF:
		iface |= 0x0180;
		break;
	case SND_SOC_DAIFMT_IB_NF:
		iface |= 0x0100;
		break;
	case SND_SOC_DAIFMT_NB_IF:
		iface |= 0x0080;
		break;
	default:
		return -EINVAL;
	}

	nau8814_write(codec, NAU8814_IFACE, iface);
	nau8814_write(codec, NAU8814_CLOCK, clk);
	return 0;
}

static int nau8814_hw_params(struct snd_pcm_substream *substream,
				struct snd_pcm_hw_params *params,
				struct snd_soc_dai *dai)
{
	struct snd_soc_codec *codec = dai->codec;

	u16 iface = nau8814_read_reg_cache(codec, NAU8814_IFACE) & 0x19f;
	u16 adn = nau8814_read_reg_cache(codec, NAU8814_ADD) & 0x1f1;

	/* bit size */
	switch (params_format(params))
	{
		case SNDRV_PCM_FORMAT_S16_LE:
			break;
		case SNDRV_PCM_FORMAT_S20_3LE:
			iface |= 0x0020;
			break;
		case SNDRV_PCM_FORMAT_S24_LE:
			iface |= 0x0040;
			break;
		case SNDRV_PCM_FORMAT_S32_LE:
			iface |= 0x0060;
			break;
	}

	/* filter coefficient */
	switch (params_rate(params))
	{
		case SNDRV_PCM_RATE_8000:
			adn |= 0x5 << 1;
			break;
		case SNDRV_PCM_RATE_11025:
			adn |= 0x4 << 1;
			break;
		case SNDRV_PCM_RATE_16000:
			adn |= 0x3 << 1;
			break;
		case SNDRV_PCM_RATE_22050:
			adn |= 0x2 << 1;
			break;
		case SNDRV_PCM_RATE_32000:
			adn |= 0x1 << 1;
			break;
		case SNDRV_PCM_RATE_44100:
		case SNDRV_PCM_RATE_48000:
			break;
	}

	nau8814_write(codec, NAU8814_IFACE, iface);
	nau8814_write(codec, NAU8814_ADD, adn);
	
	return 0;
}

static int nau8814_mute(struct snd_soc_dai *dai, int mute)
{
	struct snd_soc_codec *codec = dai->codec;
	u16 mute_reg = nau8814_read_reg_cache(codec, NAU8814_DAC) & 0xffbf;

	if (mute)
		nau8814_write(codec, NAU8814_DAC, mute_reg | 0x40);
	else
		nau8814_write(codec, NAU8814_DAC, mute_reg);
	return 0;
}

/* liam need to make this lower power with dapm */
static int nau8814_set_bias_level(struct snd_soc_codec *codec,
	enum snd_soc_bias_level level)
{
	u16 power1 = nau8814_read_reg_cache(codec, NAU8814_POWER1) & ~0x3;

	switch (level) {
	case SND_SOC_BIAS_ON:
	case SND_SOC_BIAS_PREPARE:
		power1 |= 0x1;  /* VMID 50k */
		nau8814_write(codec, NAU8814_POWER1, power1);
		break;

	case SND_SOC_BIAS_STANDBY:
		power1 |= NAU8814_POWER1_BIASEN | NAU8814_POWER1_BUFIOEN;

		if (codec->dapm.bias_level == SND_SOC_BIAS_OFF) {
			/* Initial cap charge at VMID 5k */
			nau8814_write(codec, NAU8814_POWER1, power1 | 0x3);
			mdelay(100);
		}

		power1 |= 0x2;  /* VMID 500k */
		nau8814_write(codec, NAU8814_POWER1, power1);
		break;

	case SND_SOC_BIAS_OFF:
		nau8814_write(codec, NAU8814_POWER1, 0);
		nau8814_write(codec, NAU8814_POWER2, 0);
		nau8814_write(codec, NAU8814_POWER3, 0);
		break;
	}

	codec->dapm.bias_level = level;
	return 0;
}

#define NAU8814_RATES (SNDRV_PCM_RATE_8000 | SNDRV_PCM_RATE_11025 |\
		SNDRV_PCM_RATE_16000 | SNDRV_PCM_RATE_22050 |\
		SNDRV_PCM_RATE_44100 | SNDRV_PCM_RATE_48000)

#define NAU8814_FORMATS (SNDRV_PCM_FMTBIT_S16_LE | SNDRV_PCM_FMTBIT_S20_3LE |\
	SNDRV_PCM_FMTBIT_S24_LE | SNDRV_PCM_FMTBIT_S32_LE)

static struct snd_soc_dai_ops nau8814_ops = {
	.hw_params	= nau8814_hw_params,
	.digital_mute	= nau8814_mute,
	.set_fmt	= nau8814_set_dai_fmt,
	.set_clkdiv	= nau8814_set_dai_clkdiv,
	.set_pll	= nau8814_set_dai_pll,
};

static struct snd_soc_dai_driver nau8814_dai[] = 
{
	{
		.name = "nau8814-aif1",
		.id = 1,
		.playback = {
			.stream_name = "NAU8814 Playback",
			.rates = NAU8814_RATES,
			.formats = NAU8814_FORMATS,
			.channels_min = 1,
			.channels_max = 1,
		},
		.capture = {
			.stream_name = "NAU8814 Capture",
			.rates = NAU8814_RATES,
			.formats = NAU8814_FORMATS,
			.channels_min = 1,
			.channels_max = 1,
		},
		.ops = &nau8814_ops,
	},
};

static int nau8814_suspend(struct snd_soc_codec *codec, pm_message_t state)
{
	nau8814_set_bias_level(codec, SND_SOC_BIAS_OFF);
	return 0;
}

static int nau8814_resume(struct snd_soc_codec *codec)
{
	int i;
	u8 data[2];
	u16 *cache = codec->reg_cache;

	/* Sync reg_cache with the hardware */
	for (i = 0; i < NAU8814_CACHEREGNUM; i++)
	{
		data[0] = (i << 1) | ((cache[i] >> 8) & 0x0001);
		data[1] = cache[i] & 0x00ff;
		codec->hw_write(codec->control_data, data, 2);
	}
	nau8814_set_bias_level(codec, SND_SOC_BIAS_STANDBY);
	return 0;
}

static int nau8814_probe(struct snd_soc_codec *codec)
{
	u32 tmp_reg;
	
	struct nau8814_priv *nau8814_private = snd_soc_codec_get_drvdata(codec);

	codec->hw_write = (hw_write_t) i2c_master_send;
	codec->control_data = nau8814_private->control_data;
	codec->reg_cache    = nau8814_private->reg_cache;

	nau8814_write(codec, NAU8814_RESET, 0x01);
	
	//add init code here.
	if((tmp_reg = snd_soc_read(codec,NAU8814_2WID)) != 0x1A)
	{
		pr_err("NAU8814 not found, nau8814 probe failed! tmp_reg = %d.\n", tmp_reg);
	}

	nau8814_reg_init(codec);
	
	nau8814_set_bias_level(codec, SND_SOC_BIAS_STANDBY);
	snd_soc_add_controls(codec, nau8814_snd_controls, ARRAY_SIZE(nau8814_snd_controls));
	nau8814_add_widgets(codec);
	
	return 0;
}

/* power down chip */
static int nau8814_remove(struct snd_soc_codec *codec)
{
	nau8814_set_bias_level(codec, SND_SOC_BIAS_OFF);
	return 0;
}

static struct snd_soc_codec_driver soc_codec_dev_nau8814 = {
	.read  = 	nau8814_read,
	.write = 	nau8814_write,
	.probe = 	nau8814_probe,
	.remove = 	nau8814_remove,
	.suspend = 	nau8814_suspend,
	.resume =	nau8814_resume,
	.set_bias_level = nau8814_set_bias_level,
};

/*
 * NAU8814 2 wire address is 0x1a
 */

static int nau8814_i2c_probe(struct i2c_client *i2c,
			    const struct i2c_device_id *id)
{
	struct nau8814_priv  *nau8814_private;
	int ret;

	nau8814_private = kzalloc(sizeof(struct nau8814_priv), GFP_KERNEL);
	if(nau8814_private == NULL)
	{
		pr_err("alloc nau8814_priv failed, no memory.\n");
		return -ENOMEM;
	}

	nau8814_private->control_data = i2c;
	memset(nau8814_private->reg_cache, 0, NAU8814_CACHEREGNUM);
	i2c_set_clientdata(i2c, nau8814_private);
	
	ret = snd_soc_register_codec(&i2c->dev, &soc_codec_dev_nau8814, nau8814_dai, ARRAY_SIZE(nau8814_dai));

	if(ret < 0)
	{
		pr_err("%s, snd_soc_register_codec failed! ret = %d.\n", __func__, ret);
		kfree(nau8814_private);
	}

	return ret;
}

static int __devexit nau8814_i2c_remove(struct i2c_client *client)
{
	snd_soc_unregister_codec(&client->dev);
	kfree(i2c_get_clientdata(client));
	return 0;
}

static const struct i2c_device_id nau8814_i2c_id[] = {
	{NAU8814_DEV_NAME, 0},{},
};
MODULE_DEVICE_TABLE(i2c, nau8814_i2c_id);

static struct i2c_driver nau8814_i2c_driver = {
	.driver = {
		.name = NAU8814_DEV_NAME,
		.owner = THIS_MODULE,
	},
	.probe    = nau8814_i2c_probe,
	.remove   = __devexit_p(nau8814_i2c_remove),
	.id_table = nau8814_i2c_id,
};

static int __init nau8814_modinit(void)
{
	int r = 0;
	
	r = i2c_add_driver(&nau8814_i2c_driver);
	if(r < 0)
	{
		pr_err("%s, nau8814_modinit failed! ret = %d.\n", __func__, r);
	}

	return r;
}
module_init(nau8814_modinit);

static void __exit nau8814_exit(void)
{
	i2c_del_driver(&nau8814_i2c_driver);
}
module_exit(nau8814_exit);

MODULE_DESCRIPTION("ASoC NAU8814 driver");
MODULE_AUTHOR("scott Hu");
MODULE_LICENSE("GPL");
