#!/bin/sh
#
# Script to suggest duplicate entries delivered by 109896-34 and
# 109896-35 for removal from input file (/etc/name_to_major).
# Based on switches used, it can create name_to_major file ready
# for copy (-c switch) and copy it over specified file (-f switch).
# Original file is backed up before overwriting.
# User must confirm overwriting of the file.

#
# Copyright 2007 Sun Microsystems, Inc.  All rights reserved.
# Use is subject to license terms.

usage() {
	echo "
This script suggests duplicate entries delivered by 109896-34 and
109896-35 for removal from input file (/etc/name_to_major).

Based on switches used, it can create name_to_major file ready
for copy (-c switch) and copy it over specified file (-f switch).
Original file is backed up before overwriting.
User must confirm overwriting of the file.

Usage: $0 [-c [-f]] <file> <directory>
       $0 -h

-c create file ready for manual copy to /etc/name_to_major
-f overwrite specified <file>
<file> should be in format typical for /etc/name_to_major
<directory> should point to directory with messages file(s)

Example: $0 -c -f /etc/name_to_major /var/adm/
"
	exit 1
}

if [ "$#" -eq 0 ]; then
	usage
fi

while getopts cfh name
do
	case $name in
		c)      cflag=1
			;;
		f)      fflag=1
			;;
		?)      usage
			;;
	esac
done
shift `expr $OPTIND - 1`

if [ \( ! -z "${fflag}" \) -a \( -z "${cflag}" \) ]; then
	usage
fi

FILE=$1
MESS_DIR=$2
TMPS="$$_`/usr/bin/date '+%Y_%m_%d_%H_%M_%S'`"

if [ \( ! -z ${FILE} \) -a \( ! -f "${FILE}" \) ]; then
	usage
fi

ls ${MESS_DIR}/messages* 2>&1 > /dev/null
if [ $? -ne 0 ]; then
	usage
fi

IFS_ORIGIN=${IFS}
TO_REMOVE=
LANG=C; export LANG
LC_CTYPE=C; export LC_CTYPE
LC_NUMERIC="C"; export LC_NUMERIC
LC_TIME="C"; export LC_TIME
LC_COLLATE="C"; export LC_COLLATE
LC_MONETARY="C"; export LC_MONETARY
LC_MESSAGES="C"; export LC_MESSAGES
LC_ALL=C; export LC_ALL
## This is /etc/name_to_major content as it is delivered
## in patch 109896-35
NAME_TO_MAJOR="cn 0
rootnex 1
pseudo 2
ip 3
logindmux 4
icmp 5
fas 6
hme 7
eri 8
pcf8584 9
max1617 10
clone 11
sad 12
mm 13
iwscn 14
wc 15
conskbd 16
consms 17
tda8444 18
dump 19
se 20
log 21
sy 22
ptm 23
pts 24
ptc 25
ptsl 26
bwtwo 27
audio 28
zs 29
cgthree 30
seeprom 31
sd 32
st 33
smbus 34
fd 36
su 37
openeepr 38
cgsix 39
le 40
udp 41
tcp 42
rts 43
arp 44
lo 46
tivc 47
tidg 48
tmux 49
glm 50
options 57
obio 58
sbus 59
fssnap 60
esp  61
dma  62
sbusmem 69
ramdisk 70
zsh 71
ksyms 72
bpp 73
iommu 76
lebuffer 77
ledma 78
cgeight 80
winlock 81
cpu 83
dbri 84
md 85
todds1287 87
devinfo 88
lockstat 89
kstat 90
vol 91
xbox 92
bootbus 93
sbi 94
cpc 95
sysmsg 97
sx 98
cgfourteen 99
sx_cmem 100
stc 101
isp 102
qec 103
qe 104
tl 105
upa64s 106
llc1 107
audiocs 108
pcisch 109
simba 110
pcipsy 111
socal 112
sf 113
tnf 114
pci_pci 115
soc 116
pln 117
ssd 118
central 119
fhc 120
environ 121
sysctrl 122
ac 123
sram 124
simmstat 125
ebus 126
pm  127
tod 128
pmc 129
mic 130
envctrl 131
cvc 132
cvcredir 133
power 134
uata 135
dad 136
envctrltwo 137
poll 138
keysock 139
ipsecah 140
ipsecesp 141
dr 142
ip6 143
icmp6 144
udp6 145
tcp6 146
lofi 147
usoc 148
fp 149
fcp 150
sgen 151
db21554 152
qlc 153
fcip 154
bbc_beep 155
schppm 156
xcalppm 157
mc-us3 158
pmubus 159
grfans 160
grppm 161
grbeep 162
audiots 163
isadma 164
sgsbbc 165
ssm 166
sgcn 167
sghsc 168
sgenv 169
sgfru 170
gpio_87317 171
pcf8574 172
pcf8591 173
sbbc 174
iosram 175
axq 176
schpc 177
dman 178
hpc3130 179
lm75 180
ltc1427 181
ssc050 182
ssc100 183
smbus_ara 186
sppp 187
sppptun 188
usb_ac 189
usb_as 190
rsm 191
lombus 192
lw8 193
mc-us3i 194
bge 195
rmc_comm 196
rmclomv 197
rmcadm 198
pmugpio 199
pca9556 200
bscbus 201
bscv 202
i2bsc 203
ppm 204
tomtppm 205
m1535ppm 206
ics951601 207
adm1031 208
usba10_usb_ac 209
usba10_usb_as 210
mpt 211
xcalwd 212
pic16f819 213"

echo "Searching messages files for driver binding conflicts"
IFS=
MESSAGES_LINES="`grep 'conflicts with a previous entry' ${MESS_DIR}/messages*`"
##echo ${MESSAGES_LINES}
IFS=${IFS_ORIGIN}

echo "Searching for duplicate entries in ${FILE}"
echo
IFS="
"
for line in ${NAME_TO_MAJOR}
do
##	echo "${line}"
	IFS=${IFS_ORIGIN}
	set -- $line
	name=$1 major=$2
##	echo "## $name ## $major" 
	IFS=
	LINES=`egrep "^$name[ 	][ 	]*" ${FILE}`
	COUNT=`echo "${LINES}" | awk 'BEGIN{i=0}{i++}END{print i}'`
	IFS=${IFS_ORIGIN}
	if [ ${COUNT} -gt 1 ]; then
		echo "Info: Found duplicate entries for ${name}:"
		echo "${LINES}"
##		MSG=`echo ${MESSAGES_LINES} | awk 'BEGIN{i=1} /'"\"${name} "'/{a[i++]=$0} END{print a[1]}'`
		IFS=
		MSG=`echo ${MESSAGES_LINES} | sed -n '/'"\"${name} "'/{p;q;}'`
		if [ "${MSG}" != "" ]; then
			echo "Info: ${MSG}"
		fi
		IFS=${IFS_ORIGIN}
		if [ ! -d "/dev/$name" ]; then
			device_line="`ls -lL /dev/$name 2>/dev/null`"
			if [ "${device_line}" != "" ]; then
				set -- `echo "${device_line}" | awk '{split($5,a,","); printf "%d %d\n", a[1], $6}'`
				maj=$1
				min=$2
				if [ $maj -eq 11 ]; then
					maj=$min
				fi
##				/usr/ucb/echo -n "## $name ## $major ($1 $2) $device_line"
##				echo
				if [ $major -ne $maj ]; then
					echo "Info: there is device \"$name\" with \"$major\" number in the /dev/"
					echo "Info: be sure to leave at least and only \"${name} ${major}\" line in ${FILE}"
				fi
			else
				echo "Info: there is no device with name \`$name' in the /dev/"
			fi
		fi
		IFS=
		TO_REMOVE=`echo "${name} ${major}"; echo ${TO_REMOVE}`
		IFS=${IFS_ORIGIN}
		echo "Info: line \"${name} ${major}\" suggested for removal"
		echo
	fi
done

if [ "" != "${TO_REMOVE}" ]; then
	if [ ! -z "${cflag}" ]; then
		NEW_FILE="/tmp/name_to_major_${TMPS}"
		TMP_NEW_FILE="/tmp/tmp_name_to_major_${TMPS}"
		tail -r ${FILE} > ${NEW_FILE}
		IFS="
"
		for line in ${TO_REMOVE}
		do
			nawk 'BEGIN {DEL=0} ; DEL == 0 && /^'${line}'$/ { DEL = 1 ; next ; } ; { print }' ${NEW_FILE} > ${TMP_NEW_FILE}
			mv ${TMP_NEW_FILE} ${NEW_FILE}
		done
		tail -r ${NEW_FILE} > ${TMP_NEW_FILE}
		mv ${TMP_NEW_FILE} ${NEW_FILE}
		echo "File ${NEW_FILE} without duplicate entries (see below) was created."
		echo
		echo "Duplicate entries:"
	else
		echo "Be careful when removing line(s), leave at least (and only)"
		echo "one line for every name (first column) entry."
		echo
		echo "Suggestion which line(s) to remove from ${FILE}:"
	fi
	echo
	IFS=
	echo ${TO_REMOVE} | tail -r
	if [ ! -z "${fflag}" ]; then
		echo
		echo "Do you want to back up ${FILE} and overwrite it with ${NEW_FILE} [*/yY]?"
		read a
		if [ \( "${a}" = "y" \) -o \( "${a}" = "Y" \) ]; then
			BACKUP_FILE="${FILE}_${TMPS}"
			cp "${FILE}" "${BACKUP_FILE}"
			if [ $? -eq 0 ]; then
				echo "${FILE} backed up as ${BACKUP_FILE}"
				cp "${NEW_FILE}" "${NEW_FILE}_$$_${TMPS}"
				if [ $? -ne 0 ]; then
					rm -f "${NEW_FILE}_$$_${TMPS}"
					echo "Error during copying of ${NEW_FILE} to ${FILE}"
					exit 2
				else
					mv -f "${NEW_FILE}_$$_${TMPS}" ${FILE}
					if [ $? -ne 0 ]; then
						echo "Error during copying of ${NEW_FILE} to ${FILE}"
						echo "Please compare content of ${FILE} and ${BACKUP_FILE}, restore backed up version if needed."
						exit 2
					else
						echo "${NEW_FILE} copied to ${FILE}"
						rm ${NEW_FILE}
						if [ $? -eq 0 ]; then
							echo "${NEW_FILE} removed."
						else
							echo "Error during ${NEW_FILE} removal."
						fi
					fi
				fi
			else
				echo "Error ocured during ${FILE} back up, ${FILE} will not be replaced."
			fi
		fi
	fi
else
	echo "No duplicate entries found in ${FILE}"
fi

exit 0
