/*
 *	@(#)cd.h 1.1 94/10/31 SMI
 *	Copyright (c) 1989 Sun Microsystems, Inc.
 */

#include <sys/buf.h>

#ifdef	 SVR4
#include <sys/dkio.h>
#include <sys/cdio.h>
#else
#include <sun/dkio.h>
#include <scsi/targets/srdef.h>
#endif	 SVR4


#define	CD_SECTOR_SIZE	2048
#define	MBLOCKS		(CD_SECTOR_SIZE/CDROM_MODE1_SIZE)

#define	MIN_TRACK	1		/* smallest valid track number */
#define	MAX_TRACK	0xAA		/* largest valid track number */
#define	MIN_INDEX	1		/* smallest valid index number */
#define	MAX_INDEX	1		/* largest valid index number */

#define	CDTRK_PREEMP	0x1		/* pre-emphasis bit field */
#define	CDTRK_COPY	0x2		/* copy permitted bit field */
#define	CDTRK_DATA	0x4		/* data track bit field */
#define	CDTRK_4CHANNEL	0x8		/* four channel bit field */

#define	CD_DATAMODE1	1		/* CD data mode 1 */
#define	CD_DATAMODE2	2		/* CD data mode 2 */


#define	MAX_DATACMP	512		/* max data size to be compared */

#define	CD_SONNY2	0		/* Sony Test Disc 2.0 */
#define	CD_HITACHI4	1		/* Hitachi Test Disc 4 */
#define	CD_ASSIST	2		/* CD Assist */
#define	CD_PDO		3		/* PDO disc */
#define	CD_OTHER	4		/* other (unknown) CD type */
#define	CD_DEFAULT	CD_OTHER	/* default is unknown type of CD */

#define	NBLOCKS_SEC	(75*MBLOCKS)	/* Number blocks per second */
#define	CD_PREGAP1	(1*NBLOCKS_SEC)	/* First part of pre-gap in blocks */
#define	CD_PREGAP2	(2*NBLOCKS_SEC)	/* Second part of pre-gap in blocks */
#define	CD_POSTGAP	(2*NBLOCKS_SEC)	/* post-gap in blocks */
#define	CD_PAUSEGAP	(2*NBLOCKS_SEC)	/* pause-gap in blocks */


/* Table of contents info structure */
struct toc_info
{
	unsigned char track;
	unsigned char ctrl;
	unsigned char datamode;
	int lba;
};


/* Data contents info structure */
struct datacon_info
{
	unsigned char track;
	int size;
	unsigned char expect[MAX_DATACMP];
};


/*
 * Sony test disc type 2.0 specific information.
 */
struct toc_info sony2_toc[] =		/* Sony disc 2.0 table of contents */
{
	/* track,	control,	data mode,	lba */
	1,		4,		1,		0*MBLOCKS,
	2,		4,		2,		54225*MBLOCKS,
	3,		4,		1,		63450*MBLOCKS,
	4,		0,		255,		72750*MBLOCKS,
	5,		0,		255,		75150*MBLOCKS,
	6,		0,		255,		86550*MBLOCKS,
	7,		0,		255,		97950*MBLOCKS,
	8,		0,		255,		136350*MBLOCKS,
	9,		0,		255,		138750*MBLOCKS,
	10,		0,		255,		143400*MBLOCKS,
	11,		1,		255,		148650*MBLOCKS,
	12,		4,		2,		151125*MBLOCKS,
	13,		4,		1,		155700*MBLOCKS,
	14,		1,		255,		165000*MBLOCKS,
	15,		0,		255,		183450*MBLOCKS,
	16,		4,		1,		202425*MBLOCKS,
	17,		4,		2,		207150*MBLOCKS,
	18,		0,		255,		211950*MBLOCKS,
	19,		0,		255,		213975*MBLOCKS,
	20,		4,		1,		240900*MBLOCKS,
	21,		0,		255,		245700*MBLOCKS,
	22,		4,		2,		263925*MBLOCKS,
	23,		4,		1,		266400*MBLOCKS,
	170,		4,		255,		270000*MBLOCKS,
};
struct datacon_info sony2_data[] =	/* Sony disc 2.0 tracks and data */
{
	/* track 1, size, data contents */
	1,		MAX_DATACMP,
	0x01, 0x00, 0x00, 0x20, 0x01, 0x00, 0x00, 0x20, 
	0x20, 0x00, 0x02, 0x00, 0x20, 0x30, 0x30, 0x6d, 
	0x30, 0x32, 0x73, 0x30, 0x30, 0x66, 0x20, 0x20, 
	0x00, 0x02, 0x00, 0x01, 0x80, 0x00, 0x40, 0x00, 
	0x20, 0x00, 0x10, 0x00, 0x08, 0x00, 0x04, 0x00, 
	0x02, 0x00, 0x01, 0x00, 0x00, 0x02, 0x00, 0x01, 
	0x80, 0x00, 0x40, 0x00, 0x20, 0x80, 0x10, 0xc0, 
	0x08, 0xe0, 0x04, 0x70, 0x02, 0x38, 0x01, 0x1c, 
	0x00, 0x0c, 0x00, 0x06, 0x00, 0x03, 0x80, 0x01, 
	0xc0, 0x80, 0x60, 0xc0, 0x30, 0xe0, 0x18, 0x70, 
	0x0c, 0x38, 0x06, 0x1c, 0x03, 0x0e, 0x01, 0x05, 
	0x80, 0x00, 0x40, 0x00, 0x20, 0x80, 0x10, 0xc0, 
	0x08, 0xe0, 0x04, 0x70, 0x02, 0x38, 0x01, 0x1c, 
	0x00, 0x0c, 0x00, 0x06, 0x00, 0x03, 0x80, 0x01, 
	0xc0, 0x00, 0x60, 0x80, 0x30, 0x40, 0x18, 0x20, 
	0x0c, 0x90, 0x06, 0x48, 0x03, 0x24, 0x01, 0x10, 
	0x00, 0x0a, 0x00, 0x05, 0x80, 0x82, 0x40, 0xc1, 
	0xa0, 0xe0, 0x50, 0x70, 0x28, 0x38, 0x14, 0x1c, 
	0x0a, 0x0e, 0x05, 0x07, 0x82, 0x01, 0xc1, 0x00, 
	0x60, 0x02, 0x30, 0x81, 0x98, 0x40, 0x4c, 0x20, 
	0x26, 0x90, 0x13, 0x48, 0x09, 0x26, 0x04, 0x11, 
	0x82, 0x08, 0x41, 0x04, 0x20, 0x00, 0x10, 0x00, 
	0x08, 0x80, 0x04, 0xc0, 0x02, 0x60, 0x01, 0xb0, 
	0x00, 0xda, 0x00, 0x6d, 0x80, 0x36, 0x40, 0x1b, 
	0xa0, 0x8d, 0xd0, 0x46, 0x68, 0x23, 0xb4, 0x91, 
	0xda, 0xc8, 0x6d, 0xe4, 0x36, 0xf0, 0x1b, 0x78, 
	0x0d, 0x3e, 0x06, 0x1d, 0x83, 0x8e, 0x41, 0xc5, 
	0xa0, 0xe0, 0x50, 0x70, 0x28, 0x38, 0x14, 0x1c, 
	0x0a, 0x0e, 0x05, 0x07, 0x82, 0x01, 0xc1, 0x00, 
	0x60, 0x82, 0x30, 0xc1, 0x98, 0x60, 0x4c, 0x30, 
	0x26, 0x18, 0x13, 0x0c, 0x09, 0x04, 0x04, 0x00, 
	0x02, 0x80, 0x01, 0x40, 0x00, 0x22, 0x00, 0x11,
	0x80, 0x88, 0x40, 0xc4, 0x20, 0x62, 0x10, 0xb1, 
	0x88, 0xd8, 0x44, 0x6c, 0x22, 0x36, 0x11, 0x1b, 
	0x88, 0x8f, 0xc4, 0x47, 0xe2, 0x23, 0xf1, 0x91, 
	0xf8, 0xca, 0x7c, 0xe5, 0xbe, 0xf2, 0x5f, 0x79, 
	0xaf, 0x3e, 0x57, 0x1d, 0xab, 0x0c, 0x55, 0x04, 
	0x2a, 0x00, 0x15, 0x00, 0x0a, 0x82, 0x05, 0xc1, 
	0x82, 0xe2, 0x41, 0x71, 0xa0, 0xba, 0x50, 0xdd, 
	0xa8, 0xee, 0x54, 0x77, 0xaa, 0xbb, 0xd5, 0x5d, 
	0xea, 0xac, 0x75, 0xd6, 0x3a, 0xe9, 0x9d, 0x74, 
	0x4e, 0x38, 0x27, 0x9c, 0x13, 0x4c, 0x09, 0xa4, 
	0x04, 0x50, 0x02, 0xa8, 0x01, 0xd4, 0x00, 0x68, 
	0x00, 0x34, 0x00, 0x1a, 0x00, 0x8d, 0x80, 0xc6, 
	0x40, 0x63, 0xa0, 0xb1, 0xd0, 0x58, 0x68, 0xac, 
	0x34, 0x56, 0x1a, 0x2b, 0x8d, 0x95, 0xc6, 0x48, 
	0x63, 0xa4, 0x31, 0xd0, 0x18, 0xea, 0x0c, 0xf5, 
	0x86, 0x7a, 0x43, 0x3d, 0xa1, 0x1c, 0x50, 0x0c, 
	0x28, 0x06, 0x14, 0x03, 0x8a, 0x01, 0xc5, 0x00, 
	0x62, 0x02, 0x31, 0x81, 0x98, 0xc2, 0x4c, 0x61, 
	0xa6, 0x30, 0x53, 0x18, 0x29, 0x8e, 0x14, 0x45, 
	0x8a, 0xa2, 0x45, 0x51, 0xa2, 0x2a, 0x51, 0x95, 
	0xa8, 0x48, 0x54, 0x24, 0x2a, 0x92, 0x15, 0xc9, 
	0x8a, 0x66, 0x45, 0xb3, 0xa2, 0xdb, 0xd1, 0x6d, 
	0xe8, 0xb4, 0x74, 0x5a, 0x3a, 0x2d, 0x9d, 0x96, 
	0x4e, 0xc9, 0xa7, 0xe4, 0x53, 0x70, 0x29, 0xba, 
	0x14, 0xdf, 0x8a, 0x6f, 0xc5, 0xb7, 0xe2, 0xd9, 
	0xf1, 0x6c, 0x78, 0xb4, 0x3c, 0x5a, 0x1e, 0xad, 
	0x8f, 0xd6, 0x47, 0x69, 0xa3, 0xb6, 0x51, 0xd9, 
	0xa8, 0x6e, 0x54, 0x37, 0xaa, 0x9b, 0xd5, 0x4d, 
	0xea, 0xa4, 0x75, 0xd2, 0x3a, 0xeb, 0x9d, 0x75, 
	0xce, 0xb8, 0x67, 0xdc, 0x33, 0x6c, 0x19, 0xb4, 
	0x0c, 0x58, 0x06, 0xac, 0x03, 0xd6, 0x01, 0x69, 
	0x80, 0xb6, 0x40, 0x5b, 0xa0, 0xad, 0xd0, 0xd6, 

	/* track 2, size, data contents */
	2,		MAX_DATACMP,
	0xd2, 0xd3, 0x00, 0x20, 0x26, 0x42, 0x05, 0x20, 
	0x20, 0x12, 0x05, 0x00, 0x20, 0x31, 0x32, 0x6d, 
	0x30, 0x35, 0x73, 0x30, 0x30, 0x66, 0x20, 0x20, 
	0xe9, 0x69, 0xf4, 0x36, 0x7a, 0x1b, 0xbd, 0x0d, 
	0xde, 0x04, 0x6f, 0x02, 0x37, 0x03, 0x9b, 0x03, 
	0xcd, 0x03, 0xe6, 0x03, 0xf3, 0x01, 0xf9, 0x02, 
	0x7c, 0x03, 0xbe, 0x01, 0xdf, 0x00, 0x6f, 0x82, 
	0x37, 0xc3, 0x9b, 0xe3, 0xcd, 0xf3, 0xe6, 0xfb, 
	0xf3, 0x7d, 0xf9, 0x3c, 0x7c, 0x9c, 0x3e, 0xce, 
	0x1f, 0x67, 0x8f, 0x31, 0xc7, 0x1a, 0x63, 0x0f, 
	0xb1, 0x85, 0xd8, 0xc0, 0x6c, 0x60, 0x36, 0xb0, 
	0x1b, 0xd8, 0x0d, 0xee, 0x06, 0x75, 0x83, 0x3a, 
	0x41, 0x1f, 0xa0, 0x0d, 0xd0, 0x86, 0x68, 0xc3, 
	0xb4, 0x61, 0xda, 0xb0, 0x6d, 0xd8, 0x36, 0xee, 
	0x1b, 0x77, 0x8d, 0xb9, 0xc6, 0x5e, 0x63, 0xaf, 
	0xb1, 0x55, 0xd8, 0x28, 0x6c, 0x14, 0x36, 0x0a, 
	0x1b, 0x05, 0x8d, 0x80, 0x46, 0x42, 0x23, 0x21, 
	0x91, 0x12, 0x48, 0x8b, 0xa4, 0xc5, 0xd2, 0x62, 
	0x69, 0x31, 0xb4, 0x1a, 0x5a, 0x0d, 0xad, 0x06, 
	0x56, 0x01, 0xab, 0x80, 0x55, 0x42, 0x2a, 0xa3, 
	0x95, 0x51, 0xca, 0x2a, 0x65, 0x15, 0xb2, 0x08, 
	0x59, 0x04, 0x2c, 0x80, 0x16, 0x40, 0x0b, 0xa0, 
	0x05, 0xd2, 0x02, 0xeb, 0x81, 0xf5, 0xc0, 0xf8, 
	0x60, 0x7c, 0x30, 0x3e, 0x18, 0x1f, 0x8c, 0x0f, 
	0xc6, 0x87, 0xe3, 0x43, 0xf1, 0xa3, 0xf8, 0x53, 
	0xfc, 0xa9, 0xfe, 0xd4, 0x7f, 0xea, 0x3f, 0x77, 
	0x9f, 0x39, 0xcf, 0x1e, 0x67, 0x8d, 0xb3, 0xc4, 
	0x59, 0x60, 0x2c, 0x32, 0x16, 0x19, 0x8b, 0x0c, 
	0x45, 0x84, 0x22, 0x40, 0x11, 0x20, 0x08, 0x92, 
	0x04, 0x49, 0x82, 0xa4, 0x41, 0x52, 0x20, 0xab, 
	0x90, 0xd5, 0xc8, 0xea, 0x64, 0xf5, 0xb2, 0xfa, 
	0x59, 0x7d, 0xac, 0xbc, 0x56, 0xde, 0x2b, 0x6f, 
	0x95, 0x35, 0xca, 0x18, 0x65, 0x8c, 0x32, 0xc4, 
	0x19, 0x62, 0x0c, 0xb3, 0x86, 0xd9, 0xc3, 0xec, 
	0x61, 0x74, 0x30, 0x38, 0x18, 0x1c, 0x0c, 0x8e, 
	0x06, 0xc7, 0x83, 0xe3, 0xc1, 0xf3, 0xe0, 0x7b, 
	0xf0, 0x3d, 0xf8, 0x1e, 0x7c, 0x8f, 0xbe, 0x47, 
	0xdf, 0xa3, 0xef, 0xd3, 0xf7, 0xeb, 0xfb, 0x77, 
	0xfd, 0x39, 0xfe, 0x9e, 0x7f, 0x4f, 0xbf, 0x25, 
	0xdf, 0x90, 0x6f, 0x4a, 0x37, 0x27, 0x9b, 0x91, 
	0xcd, 0x4a, 0x66, 0x27, 0xb3, 0x13, 0xd9, 0x8b, 
	0xec, 0x47, 0xf6, 0xa3, 0xfb, 0xd1, 0xfd, 0xea, 
	0x7e, 0x77, 0xbf, 0x3b, 0xdf, 0x1f, 0xef, 0x8d, 
	0xf7, 0xc4, 0x7b, 0xe0, 0x3d, 0xf2, 0x1e, 0xfb, 
	0x8f, 0x7d, 0xc7, 0x3c, 0x63, 0x1c, 0x31, 0x0c, 
	0x18, 0x84, 0x0c, 0xc2, 0x06, 0x61, 0x83, 0x30, 
	0x41, 0x1a, 0x20, 0x0f, 0x90, 0x87, 0xc8, 0xc3, 
	0xe4, 0xe1, 0xf2, 0xf0, 0x79, 0x78, 0x3c, 0x3e, 
	0x1e, 0x1f, 0x8f, 0x8f, 0xc7, 0xc5, 0xe3, 0x60, 
	0x71, 0xb2, 0x38, 0x5b, 0x9c, 0xad, 0xce, 0x56, 
	0x67, 0x2b, 0xb3, 0x97, 0xd9, 0x49, 0xec, 0x26, 
	0x76, 0x13, 0xbb, 0x89, 0xdd, 0xc6, 0x6e, 0xe1, 
	0xb7, 0xf0, 0x5b, 0x7a, 0x2d, 0xbf, 0x96, 0xdd, 
	0xcb, 0x6e, 0x65, 0xb5, 0xb2, 0x58, 0x59, 0xac, 
	0x2c, 0x54, 0x16, 0xaa, 0x0b, 0x55, 0x85, 0xa8, 
	0x42, 0xd6, 0x21, 0x6b, 0x90, 0x37, 0xc8, 0x1b, 
	0xe4, 0x0d, 0xf2, 0x86, 0x79, 0x43, 0xbc, 0x23, 
	0xde, 0x11, 0xef, 0x08, 0x77, 0x06, 0x3b, 0x81, 
	0x9d, 0xc2, 0x4e, 0xe3, 0xa7, 0x71, 0xd3, 0x3a, 
	0x69, 0x1f, 0xb4, 0x8d, 0xda, 0xc6, 0x6d, 0x63, 
	0xb6, 0x33, 0xdb, 0x99, 0xed, 0xce, 0x76, 0xe5, 
	0xbb, 0xf2, 0x5d, 0x7b, 0xae, 0xbf, 0xd7, 0xdf, 
	0xeb, 0x6d, 0xf5, 0x34, 0x7a, 0x18, 0x3d, 0x8c, 
	0x1e, 0x44, 0x0f, 0xa2, 0x07, 0xd3, 0x83, 0xeb,

	/* track 3, size, data contents */
	3,		MAX_DATACMP,
	0xdb, 0xf7, 0x00, 0x20, 0x51, 0x34, 0x06, 0x20, 
	0x20, 0x14, 0x08, 0x00, 0x20, 0x31, 0x34, 0x6d, 
	0x30, 0x38, 0x73, 0x30, 0x30, 0x66, 0x20, 0x20, 
	0xed, 0x79, 0xf6, 0x3e, 0x7b, 0x1f, 0xbd, 0x0d, 
	0xde, 0x04, 0x6f, 0x02, 0x37, 0x03, 0x9b, 0x03, 
	0xcd, 0x03, 0xe6, 0x03, 0xf3, 0x01, 0xf9, 0x02, 
	0x7c, 0x03, 0xbe, 0x01, 0xdf, 0x80, 0x6f, 0x42, 
	0x37, 0x23, 0x9b, 0x13, 0xcd, 0x0b, 0xe6, 0x07, 
	0xf3, 0x03, 0xf9, 0x03, 0xfc, 0x83, 0xfe, 0xc1, 
	0xff, 0x60, 0x7f, 0x32, 0x3f, 0x1b, 0x9f, 0x0f, 
	0xcf, 0x85, 0xe7, 0xc0, 0x73, 0x62, 0x39, 0xb3, 
	0x9c, 0xdb, 0xce, 0xed, 0xe7, 0x76, 0x73, 0x39, 
	0xb9, 0x1e, 0x5c, 0x0d, 0xae, 0x86, 0x57, 0xc3, 
	0xab, 0x63, 0xd5, 0xb3, 0xea, 0xdb, 0xf5, 0xed, 
	0xfa, 0xf4, 0x7d, 0xfa, 0x3e, 0xff, 0x9f, 0x7f, 
	0xcf, 0xbd, 0xe7, 0xdc, 0x73, 0x6c, 0x39, 0xb4, 
	0x1c, 0x58, 0x0e, 0xac, 0x07, 0xd6, 0x03, 0x69, 
	0x81, 0xb6, 0x40, 0x59, 0xa0, 0x2c, 0x50, 0x96, 
	0x28, 0x4b, 0x94, 0xa5, 0xca, 0x52, 0x65, 0x29, 
	0xb2, 0x96, 0x59, 0xcb, 0xac, 0xe7, 0xd6, 0x73, 
	0xeb, 0xb9, 0xf5, 0xde, 0x7a, 0x6d, 0xbd, 0xb6, 
	0x5e, 0x59, 0xaf, 0xac, 0x57, 0x54, 0x2b, 0x28, 
	0x15, 0x16, 0x0a, 0x09, 0x85, 0x84, 0x42, 0xc0, 
	0x21, 0x60, 0x10, 0xb2, 0x08, 0xd9, 0x84, 0xec, 
	0x42, 0x76, 0x21, 0x3b, 0x90, 0x1f, 0xc8, 0x8f, 
	0xe4, 0x47, 0xf2, 0x23, 0xf9, 0x91, 0xfc, 0xca, 
	0x7e, 0x65, 0xbf, 0x32, 0x5f, 0x9b, 0xaf, 0x4f, 
	0xd7, 0xa5, 0xeb, 0xd0, 0x75, 0x6a, 0x3a, 0x37, 
	0x9d, 0x9b, 0xce, 0xcf, 0xe7, 0xe7, 0xf3, 0x71, 
	0xf9, 0x3a, 0x7c, 0x9f, 0xbe, 0xcf, 0xdf, 0xe7, 
	0xef, 0xf1, 0xf7, 0x7a, 0x7b, 0xbf, 0xbd, 0xdd, 
	0xde, 0xec, 0x6f, 0xf6, 0x37, 0x79, 0x9b, 0x3e, 
	0x4d, 0x9d, 0xa6, 0x4c, 0x53, 0x26, 0x29, 0x91, 
	0x94, 0x4a, 0x4a, 0xa5, 0xa5, 0xd2, 0x52, 0x6b, 
	0xa9, 0xb5, 0xd4, 0xd8, 0x6a, 0xec, 0x35, 0x76, 
	0x1a, 0x39, 0x8d, 0x1c, 0x46, 0x8c, 0x23, 0xc6, 
	0x11, 0x61, 0x88, 0x32, 0x44, 0x19, 0xa2, 0x0c, 
	0x51, 0x86, 0x28, 0xc1, 0x94, 0xe0, 0x4a, 0x70, 
	0x25, 0xb8, 0x12, 0xde, 0x09, 0xef, 0x84, 0xf5, 
	0xc2, 0x7a, 0x61, 0xbd, 0xb0, 0x5c, 0x58, 0x2e, 
	0x2c, 0x17, 0x96, 0x8b, 0xcb, 0x45, 0xe5, 0xa0, 
	0x72, 0xd2, 0x39, 0xe9, 0x9c, 0x76, 0x4e, 0xbb, 
	0xa7, 0xdd, 0xd3, 0x6c, 0x69, 0xb4, 0x34, 0xd8, 
	0x1a, 0xec, 0x0d, 0xf6, 0x06, 0xf9, 0x83, 0x7c, 
	0x41, 0x3c, 0x20, 0x9c, 0x10, 0x4e, 0x08, 0x27, 
	0x84, 0x13, 0xc2, 0x89, 0xe1, 0xc4, 0x70, 0xe0, 
	0x38, 0x70, 0x1c, 0xb8, 0x0e, 0x5c, 0x07, 0xae, 
	0x03, 0xd5, 0x81, 0xe8, 0x40, 0x76, 0x20, 0x3b, 
	0x90, 0x1d, 0xc8, 0x8e, 0x64, 0x47, 0xb2, 0x23, 
	0xd9, 0x11, 0xec, 0x8a, 0x76, 0xc5, 0xbb, 0xe2, 
	0x5d, 0x73, 0xae, 0x3b, 0xd7, 0x9d, 0xeb, 0x4c, 
	0x75, 0xa4, 0x3a, 0x50, 0x1d, 0xa8, 0x0e, 0x56, 
	0x07, 0x2b, 0x83, 0x17, 0xc1, 0x09, 0xe0, 0x86, 
	0x70, 0xc3, 0xb8, 0xe1, 0xdc, 0xf0, 0x6e, 0x78, 
	0x37, 0x3c, 0x1b, 0x1c, 0x0d, 0x0c, 0x06, 0x84, 
	0x03, 0x42, 0x01, 0x23, 0x80, 0x93, 0xc0, 0x49, 
	0xe0, 0x24, 0x70, 0x92, 0x38, 0x49, 0x9c, 0xa4, 
	0x4e, 0x52, 0x27, 0x29, 0x93, 0x16, 0x49, 0x89, 
	0xa4, 0x46, 0x52, 0xa3, 0xa9, 0x51, 0xd4, 0x2a, 
	0x6a, 0x15, 0xb5, 0x0a, 0x5a, 0x87, 0xad, 0x43, 
	0xd6, 0x23, 0xeb, 0x11, 0xf5, 0x0a, 0x7a, 0x07, 
	0xbd, 0x83, 0xde, 0x43, 0xef, 0xa1, 0xf7, 0x52, 
	0x7b, 0xab, 0xbd, 0xd7, 0xde, 0xe9, 0xef, 0xf4, 
	0x77, 0xf8, 0x3b, 0xfe, 0x1d, 0x7d, 0x8e, 0xbc,

	/* track 12, size, data contents */
	12,		MAX_DATACMP,
	0x56, 0x4e, 0x02, 0x20, 0x26, 0x11, 0x15, 0x20, 
	0x20, 0x33, 0x37, 0x00, 0x20, 0x33, 0x33, 0x6d, 
	0x33, 0x37, 0x73, 0x30, 0x30, 0x66, 0x20, 0x20, 
	0x2b, 0x27, 0x95, 0x91, 0xca, 0x4a, 0x65, 0x25, 
	0xb2, 0x10, 0x59, 0x08, 0x2c, 0x06, 0x16, 0x03, 
	0x8b, 0x01, 0xc5, 0x02, 0x62, 0x03, 0xb1, 0x01, 
	0xd8, 0x02, 0x6c, 0x01, 0xb6, 0x00, 0x5b, 0x80, 
	0x2d, 0x42, 0x16, 0x23, 0x8b, 0x11, 0xc5, 0x8a, 
	0x62, 0x47, 0xb1, 0xa3, 0xd8, 0xd3, 0xec, 0xe9, 
	0xf6, 0x74, 0x7b, 0x3a, 0x3d, 0x1f, 0x9e, 0x8d, 
	0xcf, 0xc6, 0x67, 0x61, 0xb3, 0xb2, 0x59, 0x5b, 
	0xac, 0x2f, 0xd6, 0x17, 0xeb, 0x0b, 0xf5, 0x07, 
	0xfa, 0x01, 0xfd, 0x80, 0x7e, 0xc2, 0x3f, 0x61, 
	0x9f, 0xb2, 0x4f, 0x5b, 0xa7, 0x2f, 0xd3, 0x15, 
	0xe9, 0x08, 0x74, 0x86, 0x3a, 0xc3, 0x9d, 0x61, 
	0xce, 0xb2, 0x67, 0xd9, 0xb3, 0x6e, 0x59, 0x35, 
	0xac, 0x18, 0x56, 0x8c, 0x2b, 0x46, 0x15, 0x21, 
	0x8a, 0x92, 0x45, 0xc9, 0xa2, 0xe6, 0x51, 0xf3, 
	0xa8, 0x7b, 0xd4, 0xbd, 0xea, 0xde, 0x75, 0xef, 
	0xba, 0x75, 0xdd, 0x3a, 0x6e, 0x9f, 0xb7, 0x4f, 
	0xdb, 0xa5, 0xed, 0xd0, 0x76, 0x6a, 0x3b, 0x35, 
	0x9d, 0x18, 0x4e, 0x8e, 0x27, 0x47, 0x93, 0xa1, 
	0xc9, 0xd2, 0x64, 0x6b, 0xb2, 0xb5, 0xd9, 0xda, 
	0x6c, 0x6f, 0xb6, 0xb7, 0xdb, 0x5b, 0xed, 0x2f, 
	0xf6, 0x95, 0xfb, 0x4a, 0x7d, 0x27, 0xbe, 0x11, 
	0xdf, 0x08, 0x6f, 0x06, 0x37, 0x01, 0x9b, 0x82, 
	0x4d, 0x43, 0xa6, 0xa3, 0xd3, 0xd1, 0xe9, 0xea, 
	0x74, 0xf7, 0xba, 0xfb, 0xdd, 0x7d, 0xee, 0x3c, 
	0x77, 0x1e, 0x3b, 0x0d, 0x9d, 0x04, 0x4e, 0x00, 
	0x27, 0x00, 0x13, 0x82, 0x09, 0xc3, 0x84, 0xe3, 
	0xc2, 0x71, 0xe1, 0x38, 0x70, 0x1e, 0x38, 0x0f, 
	0x9c, 0x87, 0xce, 0xc3, 0xe7, 0xe1, 0xf3, 0x72, 
	0x79, 0x3b, 0xbc, 0x9f, 0xde, 0x4f, 0xef, 0x27, 
	0xf7, 0x11, 0xfb, 0x8a, 0x7d, 0x47, 0xbe, 0xa1, 
	0xdf, 0xd0, 0x6f, 0xea, 0x37, 0xf7, 0x9b, 0x79, 
	0xcd, 0x3e, 0x66, 0x9d, 0xb3, 0x4e, 0x59, 0xa5, 
	0xac, 0x50, 0x56, 0xa8, 0x2b, 0xd4, 0x15, 0x68, 
	0x0a, 0xb6, 0x05, 0xdb, 0x82, 0x6f, 0xc1, 0x37, 
	0xe0, 0x19, 0xf0, 0x8c, 0x78, 0xc6, 0x3c, 0xe3, 
	0x9e, 0x71, 0xcf, 0x38, 0x67, 0x1e, 0x33, 0x0d, 
	0x99, 0x84, 0x4c, 0xc0, 0x26, 0x60, 0x13, 0x30, 
	0x09, 0x1a, 0x04, 0x8f, 0x82, 0x47, 0xc1, 0xa3, 
	0xe0, 0xd3, 0xf0, 0x69, 0xf8, 0x34, 0x7c, 0x1a, 
	0x3e, 0x8d, 0x9f, 0x46, 0x4f, 0xa1, 0xa7, 0xd2, 
	0x53, 0x6b, 0xa9, 0xb7, 0xd4, 0xd9, 0xea, 0x6c, 
	0x75, 0x36, 0x3a, 0x99, 0x9d, 0xcc, 0x4e, 0xe4, 
	0x27, 0xf2, 0x13, 0x7b, 0x89, 0x3f, 0xc4, 0x1d, 
	0xe2, 0x8e, 0x71, 0x47, 0xb8, 0xa1, 0xdc, 0xd0, 
	0x6e, 0xe8, 0x37, 0x74, 0x1b, 0xb8, 0x0d, 0xde, 
	0x06, 0xed, 0x83, 0x76, 0x41, 0xb9, 0xa0, 0x5e, 
	0x50, 0x2f, 0xa8, 0x97, 0xd4, 0x4b, 0xea, 0xa5, 
	0xf5, 0xd2, 0x7a, 0xeb, 0xbd, 0xf5, 0xde, 0x78, 
	0x6f, 0xbc, 0x37, 0x5c, 0x1b, 0xac, 0x0d, 0x54, 
	0x06, 0x28, 0x03, 0x14, 0x01, 0x08, 0x00, 0x86, 
	0x00, 0x43, 0x80, 0x21, 0xc0, 0x10, 0x60, 0x88, 
	0x30, 0xc4, 0x18, 0x62, 0x0c, 0xb1, 0x86, 0x58, 
	0x43, 0x2c, 0x21, 0x94, 0x10, 0xc8, 0x08, 0x64, 
	0x04, 0x32, 0x02, 0x19, 0x81, 0x0c, 0x40, 0x84, 
	0x20, 0x42, 0x10, 0x21, 0x88, 0x10, 0x44, 0x08, 
	0x22, 0x04, 0x11, 0x02, 0x08, 0x03, 0x84, 0x81, 
	0xc2, 0x40, 0x61, 0x20, 0x30, 0x92, 0x18, 0x49, 
	0x8c, 0x24, 0x46, 0x92, 0x23, 0xc9, 0x91, 0xe6, 
	0x48, 0x71, 0xa4, 0x38, 0x52, 0x1c, 0x29, 0x0e, 
	0x14, 0x85, 0x8a, 0xc2, 0x45, 0xe1, 0xa2, 0x72,

	/* track 13, size, data contents */
	13,		MAX_DATACMP,
	0x35, 0x60, 0x02, 0x20, 0x01, 0x57, 0x15, 0x20, 
	0x20, 0x34, 0x38, 0x00, 0x20, 0x33, 0x34, 0x6d, 
	0x33, 0x38, 0x73, 0x30, 0x30, 0x66, 0x20, 0x20, 
	0x1a, 0x32, 0x0d, 0x99, 0x86, 0x4e, 0x43, 0x27, 
	0xa1, 0x11, 0xd0, 0x0a, 0x68, 0x05, 0xb4, 0x02, 
	0x5a, 0x01, 0xad, 0x00, 0x56, 0x02, 0x2b, 0x01, 
	0x95, 0x02, 0x4a, 0x03, 0xa5, 0x81, 0xd2, 0xc2, 
	0x69, 0x61, 0xb4, 0xb2, 0x5a, 0x59, 0xad, 0x2c, 
	0x56, 0x14, 0x2b, 0x8a, 0x15, 0x47, 0x8a, 0x21, 
	0xc5, 0x90, 0x62, 0xca, 0x31, 0x65, 0x98, 0x30, 
	0x4c, 0x18, 0x26, 0x0c, 0x13, 0x86, 0x09, 0x41, 
	0x84, 0xa2, 0x42, 0xd1, 0xa1, 0xe8, 0x50, 0xf6, 
	0x28, 0x7b, 0x94, 0x3d, 0xca, 0x1e, 0x65, 0x0f, 
	0xb2, 0x85, 0xd9, 0x42, 0x6c, 0xa3, 0xb6, 0xd1, 
	0xdb, 0x68, 0x6d, 0xb6, 0x36, 0x59, 0x9b, 0x2c, 
	0x4d, 0x94, 0x26, 0xc8, 0x13, 0xe4, 0x09, 0xf0, 
	0x04, 0x7a, 0x02, 0x3d, 0x81, 0x9e, 0x40, 0xcd, 
	0xa0, 0x66, 0x50, 0xb3, 0xa8, 0xd9, 0xd4, 0xec, 
	0x6a, 0x76, 0x35, 0x3b, 0x9a, 0x1f, 0xcd, 0x0f, 
	0xe6, 0x05, 0xf3, 0x02, 0x79, 0x03, 0xbc, 0x03, 
	0xde, 0x81, 0xef, 0xc0, 0x77, 0xe2, 0x3b, 0xf3, 
	0x9d, 0x7b, 0xce, 0x3f, 0xe7, 0x1f, 0xf3, 0x0d, 
	0xf9, 0x84, 0x7c, 0xc0, 0x3e, 0x60, 0x1f, 0xb0, 
	0x0f, 0x5a, 0x07, 0x2f, 0x83, 0x15, 0xc1, 0x88, 
	0x60, 0xc6, 0x30, 0x63, 0x98, 0xb1, 0xcc, 0xd8, 
	0x66, 0x6c, 0x33, 0x36, 0x19, 0x19, 0x8c, 0x8e, 
	0x46, 0x47, 0xa3, 0xa3, 0xd1, 0xd3, 0xe8, 0x6b, 
	0xf4, 0xb5, 0xfa, 0xda, 0x7d, 0xed, 0xbe, 0x74, 
	0x5f, 0xba, 0x2f, 0x5f, 0x97, 0x2d, 0xcb, 0x14, 
	0x65, 0x88, 0x32, 0x46, 0x19, 0x23, 0x8c, 0x93, 
	0xc6, 0xc9, 0xe3, 0x64, 0x71, 0x30, 0x38, 0x9a, 
	0x1c, 0x4d, 0x8e, 0xa6, 0x47, 0xd3, 0xa3, 0x6b,
	0xd1, 0x37, 0xe8, 0x99, 0xf4, 0xcc, 0x7a, 0xe6, 
	0x3d, 0x73, 0x9e, 0x3b, 0xcf, 0x9d, 0xe7, 0xcc, 
	0x73, 0xe4, 0x39, 0x70, 0x1c, 0x3a, 0x0e, 0x9d, 
	0x87, 0x4e, 0x43, 0xa5, 0xa1, 0x50, 0x50, 0xaa, 
	0x28, 0x55, 0x94, 0xaa, 0x4a, 0x55, 0xa5, 0xaa, 
	0x52, 0xd7, 0xa9, 0x6b, 0xd4, 0x37, 0xea, 0x9b, 
	0xf5, 0xcd, 0xfa, 0x64, 0x7d, 0xb2, 0x3e, 0xdb, 
	0x9f, 0x6d, 0xcf, 0xb4, 0x67, 0x58, 0x33, 0xae, 
	0x19, 0x55, 0x8c, 0x28, 0x46, 0x94, 0x23, 0xca, 
	0x11, 0x67, 0x88, 0xb1, 0xc4, 0xd8, 0x62, 0xec, 
	0x31, 0xf6, 0x18, 0x79, 0x8c, 0xbc, 0x46, 0x5e, 
	0x23, 0x2f, 0x91, 0x95, 0xc8, 0xc8, 0x64, 0xe4, 
	0x32, 0x72, 0x19, 0xb9, 0x8c, 0xde, 0x46, 0x6f, 
	0xa3, 0x37, 0xd1, 0x99, 0xe8, 0x4e, 0x74, 0xa7, 
	0xba, 0xd3, 0xdd, 0xe9, 0xee, 0x76, 0x77, 0xbb, 
	0xbb, 0x5f, 0xdd, 0x2d, 0xee, 0x94, 0x77, 0x4a, 
	0x3b, 0xa7, 0x9d, 0xd1, 0xce, 0xea, 0x67, 0xf5, 
	0xb3, 0x78, 0x59, 0x3e, 0x2c, 0x9d, 0x96, 0x4e, 
	0x4b, 0xa7, 0xa5, 0xd1, 0xd2, 0x6a, 0x69, 0x35, 
	0xb4, 0x98, 0x5a, 0x4c, 0x2d, 0x26, 0x16, 0x11, 
	0x8b, 0x88, 0x45, 0x46, 0x22, 0x21, 0x91, 0x10, 
	0x48, 0x8a, 0x24, 0x45, 0x92, 0xa2, 0x49, 0xd1, 
	0xa4, 0x6a, 0x52, 0x35, 0xa9, 0x1a, 0x54, 0x8f, 
	0xaa, 0xc7, 0xd5, 0xe3, 0xea, 0xf3, 0xf5, 0x79, 
	0xfa, 0x3e, 0x7d, 0x1f, 0xbe, 0x0d, 0xdf, 0x86, 
	0x6f, 0xc1, 0xb7, 0x62, 0x5b, 0xb3, 0xad, 0xdb, 
	0xd6, 0xef, 0xeb, 0xf7, 0xf5, 0xf9, 0xfa, 0xfe, 
	0x7d, 0xff, 0xbe, 0xfd, 0xdf, 0x7e, 0x6f, 0xbd, 
	0xb7, 0x5c, 0x5b, 0xac, 0x2d, 0x54, 0x16, 0x28, 
	0x0b, 0x94, 0x05, 0xc8, 0x02, 0xe6, 0x01, 0x73, 
	0x80, 0x3b, 0xc0, 0x1d, 0xe0, 0x0e, 0x70, 0x07, 
	0xb8, 0x83, 0xdc, 0x41, 0xee, 0x20, 0x77, 0x90,

	/* track 16, size, data contents */
	16,		MAX_DATACMP,
	0xba, 0x16, 0x03, 0x20, 0x26, 0x24, 0x20, 0x20, 
	0x20, 0x45, 0x01, 0x00, 0x20, 0x34, 0x35, 0x6d, 
	0x30, 0x31, 0x73, 0x30, 0x30, 0x66, 0x20, 0x20, 
	0x5d, 0x8b, 0xae, 0xc7, 0xd7, 0x63, 0xeb, 0x33, 
	0xf5, 0x1b, 0xfa, 0x0f, 0xfd, 0x07, 0xfe, 0x01, 
	0xff, 0x00, 0x7f, 0x02, 0x3f, 0x03, 0x9f, 0x03, 
	0xcf, 0x03, 0xe7, 0x03, 0xf3, 0x03, 0xf9, 0x83, 
	0xfc, 0xc3, 0xfe, 0x61, 0xff, 0x30, 0x7f, 0x9a, 
	0x3f, 0x4f, 0x9f, 0x25, 0xcf, 0x90, 0x67, 0x4a, 
	0x33, 0x27, 0x99, 0x91, 0xcc, 0xca, 0x66, 0xe5, 
	0xb3, 0xf2, 0x59, 0xfb, 0xac, 0xff, 0xd6, 0x7f, 
	0xeb, 0xbf, 0xf5, 0xdd, 0xfa, 0x6c, 0x7d, 0xb6, 
	0x3e, 0xd9, 0x9f, 0xec, 0x4f, 0xf4, 0x27, 0xf8, 
	0x13, 0xfe, 0x09, 0x7d, 0x84, 0xbc, 0x42, 0xde, 
	0x21, 0x6f, 0x90, 0x35, 0xc8, 0x9a, 0x64, 0xcd, 
	0xb2, 0x66, 0x59, 0x33, 0xac, 0x1b, 0xd6, 0x8d, 
	0xeb, 0x46, 0x75, 0x21, 0xba, 0x92, 0x5d, 0xc9, 
	0xae, 0xe6, 0x57, 0x73, 0xab, 0xbb, 0xd5, 0xdf, 
	0xea, 0xed, 0xf5, 0xf6, 0x7a, 0x79, 0xbd, 0x3c, 
	0x5e, 0x9c, 0x2f, 0xce, 0x17, 0xe5, 0x8b, 0xf0, 
	0x45, 0x7a, 0x22, 0x3f, 0x91, 0x1f, 0xc8, 0x8d, 
	0xe4, 0x46, 0x72, 0x23, 0xb9, 0x91, 0xdc, 0x4a, 
	0x6e, 0x25, 0xb7, 0x92, 0x5b, 0x4b, 0xad, 0xa7, 
	0xd6, 0x51, 0xeb, 0x28, 0x75, 0x96, 0x3a, 0xc9, 
	0x9d, 0x64, 0x4e, 0x30, 0x27, 0x98, 0x13, 0xce, 
	0x09, 0x65, 0x84, 0xb0, 0x42, 0xd8, 0x21, 0xec, 
	0x10, 0xf4, 0x08, 0x7a, 0x04, 0xbd, 0x82, 0x5e, 
	0x41, 0x2f, 0xa0, 0x15, 0xd0, 0x0a, 0x68, 0x05, 
	0xb4, 0x82, 0x5a, 0x41, 0xad, 0x20, 0x56, 0x92, 
	0x2b, 0x49, 0x95, 0xa6, 0x4a, 0xd1, 0xa5, 0xe8, 
	0x52, 0xf6, 0x29, 0x7b, 0x94, 0x3f, 0xca, 0x9f, 
	0xe5, 0xcf, 0xf2, 0xe5, 0xf9, 0x72, 0x7c, 0x3b,
	0xbe, 0x1d, 0xdf, 0x8e, 0x6f, 0xc5, 0xb7, 0x60, 
	0x5b, 0x32, 0x2d, 0x1b, 0x96, 0x0f, 0xcb, 0x87, 
	0xe5, 0x41, 0xf2, 0xa2, 0x79, 0xd1, 0xbc, 0xea, 
	0x5e, 0xf5, 0xaf, 0x7a, 0x57, 0xbf, 0xab, 0xdd, 
	0xd5, 0xec, 0x6a, 0x74, 0x35, 0xba, 0x1a, 0xdf, 
	0x8d, 0xef, 0xc6, 0x75, 0xe3, 0x3a, 0x71, 0x1f, 
	0xb8, 0x0d, 0xdc, 0x06, 0x6e, 0x83, 0xb7, 0xc1, 
	0xdb, 0xe2, 0x6d, 0x73, 0xb6, 0xbb, 0xdb, 0xdd, 
	0xed, 0x6c, 0x76, 0x34, 0x3b, 0x9a, 0x1d, 0x4f, 
	0x8e, 0xa5, 0xc7, 0x52, 0x63, 0x2b, 0xb1, 0x17, 
	0xd8, 0x89, 0xec, 0x44, 0x76, 0xa2, 0x3b, 0x51, 
	0x9d, 0xaa, 0x4e, 0x57, 0xa7, 0x2b, 0xd3, 0x17, 
	0xe9, 0x09, 0xf4, 0x06, 0x7a, 0x03, 0xbd, 0x01, 
	0xde, 0x02, 0x6f, 0x01, 0xb7, 0x82, 0x5b, 0x43, 
	0xad, 0xa3, 0xd6, 0x53, 0xeb, 0xa9, 0xf5, 0x56, 
	0x7a, 0x29, 0xbd, 0x14, 0x5e, 0x08, 0x2f, 0x84, 
	0x17, 0x40, 0x0b, 0xa2, 0x05, 0xd3, 0x82, 0xeb, 
	0xc1, 0x75, 0xe0, 0x38, 0x70, 0x9c, 0x38, 0xce, 
	0x1c, 0x67, 0x8e, 0x33, 0xc7, 0x19, 0xe3, 0x0e, 
	0x71, 0x05, 0xb8, 0x80, 0x5c, 0x40, 0x2e, 0x20, 
	0x17, 0x90, 0x0b, 0xca, 0x05, 0x67, 0x82, 0x31, 
	0xc1, 0x98, 0x60, 0xce, 0x30, 0x67, 0x98, 0x33, 
	0xcc, 0x19, 0xe6, 0x8c, 0x73, 0x46, 0x39, 0xa1, 
	0x9c, 0x52, 0x4e, 0xa9, 0xa7, 0x54, 0x53, 0xa8, 
	0x29, 0x56, 0x14, 0xa9, 0x8a, 0x54, 0x45, 0x2a, 
	0x22, 0x97, 0x91, 0xcb, 0xc8, 0x67, 0xe4, 0x33, 
	0xf2, 0x19, 0xf9, 0x8c, 0x7c, 0x44, 0x3e, 0x22, 
	0x1f, 0x91, 0x8f, 0xca, 0x47, 0x67, 0xa3, 0xb1, 
	0xd1, 0x5a, 0x68, 0xaf, 0xb4, 0xd7, 0xda, 0xeb, 
	0xed, 0x75, 0xf6, 0xb8, 0x7b, 0xdc, 0x3d, 0x6c, 
	0x1e, 0xb4, 0x0f, 0xda, 0x07, 0xef, 0x83, 0xf5, 
	0xc1, 0x78, 0x60, 0xbe, 0x30, 0xdf, 0x98, 0xef,

	/* track 17, size, data contents */
	17,		MAX_DATACMP,
	0x2f, 0x29, 0x03, 0x20, 0x51, 0x71, 0x20, 0x20, 
	0x20, 0x46, 0x04, 0x00, 0x20, 0x34, 0x36, 0x6d, 
	0x30, 0x34, 0x73, 0x30, 0x30, 0x66, 0x20, 0x20, 
	0x97, 0x96, 0x4b, 0xc9, 0xa5, 0x66, 0x52, 0x31, 
	0xa9, 0x18, 0x54, 0x0e, 0x2a, 0x07, 0x95, 0x03, 
	0xca, 0x03, 0xe5, 0x01, 0xf2, 0x02, 0x79, 0x01, 
	0xbc, 0x02, 0x5e, 0x01, 0xaf, 0x80, 0x57, 0x42, 
	0x2b, 0xa3, 0x95, 0xd3, 0xca, 0x6b, 0xe5, 0x35, 
	0xf2, 0x98, 0x79, 0xcc, 0x3c, 0xe4, 0x1e, 0xf2, 
	0x0f, 0x79, 0x87, 0xbe, 0x43, 0x5d, 0xa1, 0xac, 
	0x50, 0xd4, 0x28, 0xea, 0x14, 0x75, 0x8a, 0xba, 
	0x45, 0xdd, 0xa2, 0x6c, 0x51, 0x36, 0x28, 0x99, 
	0x94, 0x4c, 0x4a, 0xa6, 0x25, 0xd3, 0x92, 0xeb, 
	0xc9, 0x75, 0xe4, 0xb8, 0x72, 0xdc, 0x39, 0x6e, 
	0x1c, 0xb5, 0x8e, 0x5a, 0x47, 0x2d, 0xa3, 0x94, 
	0x51, 0xc8, 0x28, 0x66, 0x14, 0xb3, 0x8a, 0xd9, 
	0xc5, 0x6c, 0x62, 0xb4, 0x31, 0xda, 0x18, 0x6f, 
	0x8c, 0xb7, 0xc6, 0xdb, 0xe3, 0xed, 0xf1, 0xf4, 
	0x78, 0xf8, 0x3c, 0xfc, 0x1e, 0x7e, 0x0f, 0xbf, 
	0x87, 0x5d, 0xc3, 0xac, 0x61, 0x54, 0x30, 0xa8, 
	0x18, 0xd4, 0x0c, 0x6a, 0x06, 0xb5, 0x83, 0xda, 
	0x41, 0x6f, 0xa0, 0xb5, 0xd0, 0x5a, 0x68, 0x2d, 
	0xb4, 0x96, 0x5a, 0x4b, 0xad, 0xa5, 0xd6, 0x50, 
	0x6b, 0xa8, 0x35, 0xd6, 0x1a, 0x69, 0x8d, 0xb4, 
	0x46, 0x58, 0x23, 0x2c, 0x11, 0x94, 0x08, 0x48, 
	0x04, 0x24, 0x02, 0x92, 0x01, 0x49, 0x80, 0x26, 
	0x40, 0x13, 0xa0, 0x89, 0xd0, 0xc4, 0x68, 0x62, 
	0x34, 0x31, 0x9a, 0x18, 0x4d, 0x0c, 0x26, 0x84, 
	0x13, 0x42, 0x09, 0x23, 0x84, 0x13, 0xc2, 0x89, 
	0xe1, 0x44, 0x70, 0xa0, 0x38, 0xd0, 0x1c, 0xe8, 
	0x0e, 0x74, 0x07, 0x3a, 0x03, 0x1f, 0x81, 0x0d, 
	0xc0, 0x04, 0x60, 0x82, 0x30, 0xc1, 0x98, 0x60, 
	0x4c, 0x30, 0x26, 0x18, 0x13, 0x8c, 0x09, 0xc4, 
	0x04, 0xe0, 0x02, 0xf0, 0x01, 0x78, 0x00, 0x3e, 
	0x00, 0x9f, 0x80, 0x4f, 0xc0, 0xa7, 0xe0, 0x53, 
	0xf0, 0xa9, 0xf8, 0x54, 0x7c, 0x2a, 0x3e, 0x15, 
	0x9f, 0x0a, 0x4f, 0x87, 0xa7, 0x41, 0xd3, 0x22, 
	0x69, 0x93, 0xb4, 0xcb, 0xda, 0xe5, 0xed, 0xf2, 
	0x76, 0x7b, 0xbb, 0x3d, 0xdd, 0x1c, 0x6e, 0x0c, 
	0x37, 0x06, 0x1b, 0x01, 0x8d, 0x02, 0x46, 0x83, 
	0xa3, 0x41, 0xd1, 0xa2, 0x68, 0xd3, 0xb4, 0xe9, 
	0xda, 0x74, 0x6d, 0xba, 0x36, 0xdf, 0x9b, 0xef, 
	0xcd, 0x75, 0xe6, 0x38, 0x73, 0x1c, 0x39, 0x0c, 
	0x1c, 0x04, 0x0e, 0x82, 0x07, 0x41, 0x83, 0x22, 
	0x41, 0x13, 0xa0, 0x0b, 0xd0, 0x85, 0xe8, 0x42, 
	0x74, 0xa1, 0xba, 0xd0, 0x5d, 0x68, 0x2e, 0x36, 
	0x17, 0x1b, 0x8b, 0x0f, 0xc5, 0x05, 0xe2, 0x00, 
	0x71, 0x80, 0x38, 0xc2, 0x1c, 0x61, 0x8e, 0xb0, 
	0x47, 0x58, 0x23, 0xae, 0x11, 0x55, 0x88, 0x28, 
	0x44, 0x14, 0x22, 0x8a, 0x11, 0xc5, 0x88, 0x60, 
	0x44, 0x30, 0x22, 0x98, 0x11, 0x4c, 0x08, 0x24, 
	0x04, 0x92, 0x02, 0xc9, 0x81, 0x64, 0x40, 0xb0, 
	0x20, 0x58, 0x10, 0xac, 0x08, 0x56, 0x04, 0xab, 
	0x82, 0x55, 0xc1, 0xaa, 0x60, 0xd7, 0xb0, 0xeb, 
	0xd8, 0x75, 0xec, 0xba, 0x76, 0xdd, 0xbb, 0xee, 
	0x5d, 0x75, 0xae, 0xb8, 0x57, 0xdc, 0x2b, 0xec, 
	0x15, 0x74, 0x0a, 0x38, 0x05, 0x1c, 0x02, 0x0c, 
	0x01, 0x86, 0x00, 0xc1, 0x80, 0xe0, 0x40, 0x70, 
	0x20, 0x38, 0x10, 0x1c, 0x08, 0x8e, 0x04, 0x47, 
	0x82, 0x23, 0xc1, 0x11, 0xe0, 0x0a, 0x70, 0x85, 
	0xb8, 0x42, 0x5c, 0x21, 0xae, 0x90, 0x57, 0x48, 
	0x2b, 0xa6, 0x15, 0xd1, 0x8a, 0x6a, 0x45, 0x35, 
	0xa2, 0x18, 0x51, 0x0c, 0x28, 0x04, 0x14, 0x02, 
	0x0a, 0x81, 0x85, 0xc0, 0x42, 0xe2, 0x21, 0x71,

	/* track 20, size, data contents */
	20,		MAX_DATACMP,
	0x05, 0xad, 0x03, 0x20, 0x01, 0x09, 0x24, 0x20, 
	0x20, 0x53, 0x34, 0x00, 0x20, 0x35, 0x33, 0x6d, 
	0x33, 0x34, 0x73, 0x30, 0x30, 0x66, 0x20, 0x20, 
	0x82, 0xd4, 0x41, 0xea, 0x20, 0x77, 0x90, 0x3b, 
	0xc8, 0x1d, 0xe4, 0x0e, 0x72, 0x07, 0xb9, 0x03, 
	0xdc, 0x03, 0xee, 0x01, 0xf7, 0x00, 0x7b, 0x02, 
	0x3d, 0x03, 0x9e, 0x03, 0xcf, 0x81, 0xe7, 0xc2, 
	0x73, 0x63, 0xb9, 0xb3, 0xdc, 0xdb, 0xee, 0x6d, 
	0xf7, 0x36, 0x7b, 0x19, 0xbd, 0x8e, 0x5e, 0xc5, 
	0xaf, 0xe2, 0x57, 0xf3, 0xab, 0x7b, 0xd5, 0xbf, 
	0xea, 0x5d, 0xf5, 0x2e, 0x7a, 0x15, 0xbd, 0x8a, 
	0x5e, 0xc7, 0xaf, 0x63, 0xd7, 0xb3, 0xeb, 0xdb, 
	0xf5, 0x6f, 0xfa, 0xb5, 0xfd, 0x5a, 0x7e, 0x2f, 
	0xbf, 0x17, 0xdf, 0x89, 0xef, 0xc6, 0x77, 0x61, 
	0xbb, 0x32, 0x5d, 0x9b, 0xae, 0x4f, 0xd7, 0xa7, 
	0xeb, 0x51, 0xf5, 0x2a, 0x7a, 0x97, 0xbd, 0xcb, 
	0xde, 0x67, 0xef, 0x33, 0xf7, 0x9b, 0xfb, 0x4f, 
	0xfd, 0x25, 0xfe, 0x90, 0x7f, 0xc8, 0x3f, 0xe6, 
	0x1f, 0xf1, 0x8f, 0x7a, 0x47, 0x3f, 0xa3, 0x1d, 
	0xd1, 0x8c, 0x68, 0x44, 0x34, 0x22, 0x1a, 0x91, 
	0x8d, 0x48, 0x46, 0x26, 0x23, 0x93, 0x91, 0xcb, 
	0xc8, 0x67, 0xe4, 0x33, 0xf2, 0x19, 0xf9, 0x8c, 
	0x7c, 0xc4, 0x3e, 0xe2, 0x1f, 0xf1, 0x8f, 0xfa, 
	0x47, 0x7f, 0xa3, 0xbd, 0xd1, 0x5c, 0x68, 0xac, 
	0x34, 0xd6, 0x1a, 0x6b, 0x8d, 0x35, 0xc6, 0x18, 
	0x63, 0x8c, 0x31, 0x44, 0x18, 0xa0, 0x0c, 0xd0, 
	0x06, 0xe8, 0x03, 0xf4, 0x01, 0x78, 0x00, 0xbe, 
	0x00, 0xdf, 0x80, 0xef, 0xc0, 0x77, 0xe0, 0xbb, 
	0xf0, 0x5d, 0xf8, 0xae, 0x7c, 0xd7, 0xbe, 0x6b, 
	0xdf, 0x35, 0xef, 0x18, 0x77, 0x8e, 0x3b, 0x45, 
	0x9d, 0xa0, 0x4e, 0x52, 0x27, 0x29, 0x93, 0x96, 
	0x49, 0x49, 0xa4, 0x26, 0x52, 0x13, 0xa9, 0x09,
	0xd4, 0x06, 0x6a, 0x03, 0xb5, 0x01, 0xda, 0x82, 
	0x6d, 0x41, 0xb6, 0xa2, 0x5b, 0xd1, 0xad, 0xea, 
	0x56, 0x77, 0xab, 0x3b, 0xd5, 0x1f, 0xea, 0x8d, 
	0xf5, 0x46, 0x7a, 0xa1, 0xbd, 0xd0, 0x5e, 0xea, 
	0x2f, 0xf5, 0x97, 0xf8, 0x4b, 0xfe, 0x25, 0x7d, 
	0x92, 0xbc, 0x49, 0x5e, 0x24, 0x2d, 0x92, 0x16, 
	0x49, 0x0b, 0xa4, 0x07, 0xd2, 0x03, 0xe9, 0x01, 
	0xf4, 0x82, 0x7a, 0x41, 0xbd, 0xa0, 0x5e, 0x52, 
	0x2f, 0x29, 0x97, 0x96, 0x4b, 0xc9, 0xa5, 0x66, 
	0x52, 0x31, 0xa9, 0x98, 0x54, 0xce, 0x2a, 0xe7, 
	0x95, 0xf3, 0xca, 0xfb, 0xe5, 0xfd, 0xf2, 0xfc, 
	0x79, 0xfe, 0x3c, 0xfd, 0x9e, 0xfe, 0x4f, 0x7f, 
	0xa7, 0x3d, 0xd3, 0x9c, 0x69, 0xcc, 0x34, 0x64, 
	0x1a, 0x32, 0x0d, 0x99, 0x86, 0xce, 0x43, 0xe7, 
	0xa1, 0xf1, 0xd0, 0x7a, 0x68, 0xbd, 0xb4, 0x5e, 
	0x5a, 0xaf, 0xad, 0xd7, 0xd6, 0xe9, 0xeb, 0x74, 
	0x75, 0xb8, 0x3a, 0xde, 0x1d, 0x6f, 0x8e, 0xb5, 
	0xc7, 0xda, 0x63, 0xef, 0xb1, 0x75, 0xd8, 0x38, 
	0x6c, 0x1c, 0x36, 0x8e, 0x1b, 0x47, 0x8d, 0x21, 
	0xc6, 0x92, 0x63, 0xc9, 0xb1, 0x66, 0x58, 0x31, 
	0xac, 0x18, 0x56, 0x0c, 0x2b, 0x86, 0x15, 0x41, 
	0x8a, 0x22, 0x45, 0x91, 0xa2, 0x4a, 0x51, 0xa5, 
	0xa8, 0x50, 0x54, 0xa8, 0x2a, 0x54, 0x15, 0x2a, 
	0x0a, 0x17, 0x85, 0x0b, 0xc2, 0x07, 0xe1, 0x83, 
	0xf0, 0x43, 0xf8, 0xa1, 0xfc, 0x50, 0x7e, 0x28, 
	0x3f, 0x14, 0x1f, 0x88, 0x0f, 0x46, 0x07, 0xa1, 
	0x83, 0xd2, 0x41, 0x6b, 0xa0, 0xb7, 0xd0, 0xdb, 
	0xe8, 0x6d, 0xf4, 0xb6, 0x7a, 0x5b, 0xbd, 0xad, 
	0xde, 0xd4, 0x6f, 0x6a, 0x37, 0x37, 0x9b, 0x99, 
	0xcd, 0x4e, 0x66, 0xa5, 0xb3, 0xd2, 0x59, 0xeb, 
	0xac, 0xf7, 0xd6, 0x7b, 0xeb, 0xbd, 0xf5, 0x5c, 
	0x7a, 0x2c, 0x3d, 0x16, 0x1e, 0x89, 0x8f, 0xc4,

	/* track 22, size, data contents */
	22,		MAX_DATACMP,
	0xf6, 0x06, 0x04, 0x20, 0x26, 0x39, 0x26, 0x20, 
	0x20, 0x58, 0x41, 0x00, 0x20, 0x35, 0x38, 0x6d, 
	0x34, 0x31, 0x73, 0x30, 0x30, 0x66, 0x20, 0x20, 
	0x7b, 0x03, 0xbd, 0x03, 0xde, 0x83, 0xef, 0x41, 
	0xf7, 0x22, 0x7b, 0x13, 0xbd, 0x0b, 0xde, 0x07, 
	0xef, 0x03, 0xf7, 0x03, 0xfb, 0x03, 0xfd, 0x03, 
	0xfe, 0x03, 0xff, 0x01, 0xff, 0x02, 0x7f, 0x83, 
	0xbf, 0x43, 0xdf, 0x23, 0xef, 0x13, 0xf7, 0x0b, 
	0xfb, 0x87, 0xfd, 0xc1, 0xfe, 0x62, 0x7f, 0x31, 
	0xbf, 0x1a, 0x5f, 0x8f, 0xaf, 0xc5, 0xd7, 0x60, 
	0x6b, 0x32, 0x35, 0x1b, 0x9a, 0x8f, 0xcd, 0xc7, 
	0xe6, 0xe1, 0xf3, 0xf0, 0x79, 0xfa, 0x3c, 0xff, 
	0x9e, 0xff, 0xcf, 0x7f, 0xe7, 0x3d, 0xf3, 0x1c, 
	0x79, 0x8c, 0x3c, 0xc4, 0x1e, 0xe2, 0x0f, 0xf1, 
	0x87, 0xfa, 0x43, 0x7f, 0xa1, 0x3d, 0xd0, 0x9c, 
	0x68, 0x4e, 0x34, 0x27, 0x9a, 0x13, 0xcd, 0x89, 
	0xe6, 0xc6, 0x73, 0x63, 0xb9, 0xb3, 0xdc, 0xdb, 
	0xee, 0x6d, 0xf7, 0xb6, 0x7b, 0xd9, 0xbd, 0x6e, 
	0x5e, 0xb5, 0xaf, 0xda, 0x57, 0x6f, 0xab, 0xb5, 
	0xd5, 0x58, 0x6a, 0x2e, 0x35, 0x97, 0x9a, 0xc9, 
	0xcd, 0x64, 0x66, 0x30, 0x33, 0x18, 0x19, 0x8e, 
	0x0c, 0xc5, 0x86, 0x62, 0x43, 0xb1, 0xa1, 0x5a, 
	0x50, 0x2f, 0xa8, 0x17, 0xd4, 0x8b, 0xea, 0xc5, 
	0xf5, 0x62, 0x7a, 0x33, 0xbd, 0x99, 0xde, 0x4e, 
	0x6f, 0xa7, 0xb7, 0x51, 0xdb, 0x2a, 0x6d, 0x17, 
	0xb6, 0x89, 0xdb, 0xc4, 0x6d, 0x60, 0x36, 0x32, 
	0x1b, 0x99, 0x8d, 0x4e, 0x46, 0x25, 0xa3, 0x92, 
	0x51, 0xcb, 0xa8, 0x67, 0xd4, 0xb3, 0xea, 0x59, 
	0xf5, 0x2c, 0x7a, 0x94, 0x3d, 0xca, 0x1e, 0x67, 
	0x8f, 0x33, 0xc7, 0x1b, 0xe3, 0x0f, 0xf1, 0x05, 
	0xf8, 0x00, 0x7c, 0x00, 0x3e, 0x00, 0x1f, 0x00, 
	0x0f, 0x02, 0x07, 0x83, 0x83, 0x43, 0xc1, 0xa3, 
	0xe0, 0xd3, 0xf0, 0x69, 0xf8, 0xb4, 0x7c, 0x5a, 
	0x3e, 0x2d, 0x9f, 0x96, 0x4f, 0xc9, 0xa7, 0x66, 
	0x53, 0xb1, 0xa9, 0x5a, 0x54, 0xaf, 0xaa, 0x57, 
	0xd5, 0xab, 0xea, 0xd7, 0xf5, 0xeb, 0xfa, 0x77, 
	0xfd, 0xbb, 0xfe, 0x5f, 0xff, 0x2f, 0xff, 0x15, 
	0xff, 0x88, 0x7f, 0x46, 0x3f, 0xa1, 0x9f, 0xd2, 
	0x4f, 0xeb, 0xa7, 0x77, 0xd3, 0xb9, 0xe9, 0xde, 
	0x74, 0xed, 0xba, 0x76, 0x5d, 0xbb, 0xae, 0x5f, 
	0xd7, 0xaf, 0xeb, 0x55, 0xf5, 0x28, 0x7a, 0x96, 
	0x3d, 0xcb, 0x9e, 0xe7, 0xcf, 0xf3, 0xe7, 0xfb, 
	0xf3, 0xff, 0xf9, 0xfd, 0xfc, 0xfc, 0x7e, 0x7e, 
	0x3f, 0xbf, 0x9f, 0xdd, 0xcf, 0xec, 0x67, 0x74, 
	0x33, 0x38, 0x19, 0x9e, 0x0c, 0x4d, 0x86, 0x26, 
	0x43, 0x93, 0xa1, 0x4b, 0xd0, 0x27, 0xe8, 0x93, 
	0xf4, 0xc9, 0xfa, 0x64, 0x7d, 0xb2, 0x3e, 0xdb, 
	0x9f, 0x6d, 0xcf, 0xb4, 0x67, 0xd8, 0x33, 0xee, 
	0x19, 0xf5, 0x8c, 0x78, 0x46, 0xbc, 0x23, 0xde, 
	0x11, 0x6d, 0x88, 0x34, 0x44, 0x9a, 0x22, 0x4d, 
	0x91, 0x26, 0x48, 0x91, 0xa4, 0xc8, 0x52, 0x64, 
	0x29, 0x32, 0x14, 0x9b, 0x8a, 0xcd, 0xc5, 0xe6, 
	0x62, 0x71, 0xb1, 0xb8, 0x58, 0xde, 0x2c, 0x6f, 
	0x96, 0x37, 0xcb, 0x9b, 0xe5, 0x4f, 0xf2, 0xa5, 
	0xf9, 0xd2, 0x7c, 0xeb, 0xbe, 0x75, 0xdf, 0xba, 
	0x6f, 0xdf, 0xb7, 0xed, 0xdb, 0xf4, 0x6d, 0xf8, 
	0x36, 0x7e, 0x1b, 0xbf, 0x8d, 0xdd, 0xc6, 0x6c, 
	0x63, 0xb6, 0x31, 0x59, 0x98, 0x2e, 0x4c, 0x17, 
	0xa6, 0x8b, 0xd3, 0xc5, 0xe9, 0xe0, 0x74, 0x72, 
	0x3a, 0xb9, 0x9d, 0xdc, 0x4e, 0xec, 0x27, 0x76, 
	0x13, 0x39, 0x89, 0x1e, 0x44, 0x0d, 0xa2, 0x06, 
	0x51, 0x03, 0xa8, 0x03, 0xd4, 0x81, 0xea, 0x40, 
	0x75, 0xa0, 0x3a, 0x52, 0x1d, 0x29, 0x8e, 0x96, 
	0x47, 0xcb, 0xa3, 0xe7, 0xd1, 0x71, 0xe8, 0x3a,

	/* track 23, size, data contents */
	23,		MAX_DATACMP,
	0xa1, 0x10, 0x04, 0x20, 0x01, 0x64, 0x26, 0x20, 
	0x20, 0x59, 0x14, 0x00, 0x20, 0x35, 0x39, 0x6d, 
	0x31, 0x34, 0x73, 0x30, 0x30, 0x66, 0x20, 0x20, 
	0x50, 0x0a, 0x28, 0x05, 0x94, 0x82, 0x4a, 0x41, 
	0xa5, 0x20, 0x52, 0x12, 0x29, 0x09, 0x94, 0x06, 
	0x4a, 0x03, 0xa5, 0x01, 0xd2, 0x02, 0x69, 0x01, 
	0xb4, 0x02, 0x5a, 0x01, 0xad, 0x80, 0x56, 0xc2, 
	0x2b, 0xe1, 0x95, 0x72, 0x4a, 0x3b, 0xa5, 0x9d, 
	0xd2, 0xcc, 0x69, 0x66, 0x34, 0xb1, 0x9a, 0xd8, 
	0x4d, 0xec, 0x26, 0xf4, 0x13, 0x7a, 0x09, 0xbf, 
	0x84, 0xdd, 0xc2, 0xee, 0x61, 0xf7, 0xb0, 0x79, 
	0xd8, 0x3c, 0x6c, 0x1e, 0x36, 0x0f, 0x9b, 0x87, 
	0xcd, 0x41, 0xe6, 0xa2, 0x73, 0xd1, 0xb9, 0xea, 
	0x5c, 0xf7, 0xae, 0x7b, 0xd7, 0x3d, 0xeb, 0x1c, 
	0x75, 0x8c, 0x3a, 0xc4, 0x1d, 0xe2, 0x0e, 0xf3, 
	0x87, 0x79, 0xc3, 0x3e, 0x61, 0x9d, 0xb0, 0x4c, 
	0x58, 0x26, 0x2c, 0x13, 0x96, 0x09, 0xcb, 0x04, 
	0x65, 0x80, 0x32, 0xc2, 0x19, 0x61, 0x8c, 0xb2, 
	0x46, 0x59, 0xa3, 0x2c, 0x51, 0x94, 0x28, 0x48, 
	0x14, 0x24, 0x0a, 0x92, 0x05, 0x49, 0x82, 0xa6, 
	0x41, 0x53, 0xa0, 0x2b, 0xd0, 0x95, 0xe8, 0x4a, 
	0x74, 0x25, 0xba, 0x12, 0x5d, 0x89, 0xae, 0xc6, 
	0x57, 0xe3, 0xab, 0x73, 0xd5, 0x3b, 0xea, 0x9f, 
	0xf5, 0x4f, 0xfa, 0xa5, 0xfd, 0xd2, 0x7e, 0x6b, 
	0xbf, 0xb5, 0xdf, 0xd8, 0x6f, 0x6e, 0x37, 0x35, 
	0x9b, 0x18, 0x4d, 0x8e, 0x26, 0xc5, 0x93, 0x62, 
	0x49, 0x33, 0xa4, 0x9b, 0xd2, 0x4d, 0xe9, 0x26, 
	0x74, 0x91, 0xba, 0x48, 0x5d, 0xa4, 0x2e, 0x50, 
	0x17, 0x28, 0x0b, 0x96, 0x05, 0xc9, 0x82, 0xe6, 
	0x41, 0xf3, 0xa0, 0x7b, 0xd0, 0x3d, 0xe8, 0x1e, 
	0x74, 0x0f, 0xba, 0x87, 0xdd, 0xc3, 0xee, 0xe3, 
	0xf7, 0xf1, 0xfb, 0xfa, 0x7d, 0xff, 0xbe, 0x7d,
	0xdf, 0x3e, 0x6f, 0x9d, 0xb7, 0x4c, 0x5b, 0x24, 
	0x2d, 0x90, 0x16, 0xca, 0x0b, 0x65, 0x85, 0x30, 
	0x42, 0x1a, 0x21, 0x8d, 0x90, 0xc4, 0x48, 0x62, 
	0x24, 0x31, 0x92, 0x98, 0x49, 0x4c, 0x24, 0x24, 
	0x12, 0x12, 0x09, 0x89, 0x84, 0xc6, 0x42, 0xe3, 
	0xa1, 0x71, 0xd0, 0x3a, 0x68, 0x1d, 0xb4, 0x0e, 
	0x5a, 0x07, 0xad, 0x83, 0xd6, 0xc3, 0xeb, 0x61, 
	0xf5, 0x32, 0x7a, 0x9b, 0xbd, 0xcd, 0xde, 0xe4, 
	0x6f, 0xf2, 0x37, 0xfb, 0x9b, 0xff, 0xcd, 0xfd, 
	0xe6, 0xfc, 0x73, 0xfe, 0x39, 0x7d, 0x9c, 0x3c, 
	0x4e, 0x9e, 0x27, 0xcf, 0x93, 0x65, 0xc9, 0x30, 
	0x64, 0x1a, 0x32, 0x0d, 0x99, 0x86, 0x4c, 0x41, 
	0xa6, 0x20, 0x53, 0x90, 0x29, 0xca, 0x14, 0x67, 
	0x8a, 0x33, 0xc5, 0x19, 0xe2, 0x0e, 0x71, 0x87, 
	0xb8, 0xc1, 0xdc, 0x60, 0x6e, 0xb0, 0x37, 0x58, 
	0x1b, 0xae, 0x0d, 0xd5, 0x86, 0xe8, 0x43, 0xf4, 
	0x21, 0xf8, 0x10, 0x7e, 0x08, 0x3f, 0x84, 0x9f, 
	0xc2, 0xcf, 0xe1, 0xe7, 0xf0, 0x71, 0xf8, 0xb8, 
	0x7c, 0xdc, 0x3e, 0x6e, 0x1f, 0xb7, 0x8f, 0x59, 
	0xc7, 0xae, 0x63, 0x55, 0xb1, 0x28, 0x58, 0x16, 
	0x2c, 0x0b, 0x96, 0x85, 0xcb, 0x42, 0x65, 0x23, 
	0xb2, 0x93, 0xd9, 0xc9, 0xec, 0xe6, 0x76, 0x73, 
	0xbb, 0x39, 0xdd, 0x9e, 0x6e, 0x4d, 0xb7, 0xa6, 
	0x5b, 0xd1, 0xad, 0xea, 0x56, 0x77, 0xab, 0xbb, 
	0xd5, 0x5f, 0xea, 0xad, 0xf5, 0x56, 0x7a, 0x29, 
	0xbd, 0x14, 0x5e, 0x88, 0x2f, 0xc4, 0x17, 0xe0, 
	0x0b, 0x72, 0x05, 0x3b, 0x82, 0x1f, 0xc1, 0x0f, 
	0xe0, 0x85, 0xf0, 0x42, 0x78, 0x21, 0xbc, 0x10, 
	0x5e, 0x08, 0x2f, 0x04, 0x17, 0x80, 0x0b, 0x42, 
	0x05, 0xa3, 0x82, 0x53, 0xc1, 0x29, 0xe0, 0x96, 
	0x70, 0xcb, 0xb8, 0x65, 0xdc, 0x32, 0x6e, 0x99, 
	0xb7, 0xcc, 0x5b, 0x64, 0x2d, 0x30, 0x16, 0x1a,

	/* lead out track as terminator */
	CDROM_LEADOUT,	0,
};


/*
 * Hitachi test disc 4 specific information.
 */
struct toc_info hitachi4_toc[] =	/* Hitachi disc 4 table of contents */
{
	/* track,	control,	datamode,	lba */
	1,		4,		1,		0*MBLOCKS,
	2,		0,		255,		270150*MBLOCKS,
	3,		1,		255,		271725*MBLOCKS,
	4,		0,		255,		272400*MBLOCKS,
	5,		0,		255,		276750*MBLOCKS,
	6,		0,		255,		279000*MBLOCKS,
	7,		0,		255,		281250*MBLOCKS,
	8,		0,		255,		283500*MBLOCKS,
	9,		0,		255,		285750*MBLOCKS,
	10,		0,		255,		288000*MBLOCKS,
	11,		0,		255,		290250*MBLOCKS,
	12,		0,		255,		292500*MBLOCKS,
	13,		0,		255,		294750*MBLOCKS,
	14,		0,		255,		297000*MBLOCKS,
	15,		0,		255,		299250*MBLOCKS,
	16,		1,		255,		301500*MBLOCKS,
	17,		1,		255,		303750*MBLOCKS,
	18,		1,		255,		306000*MBLOCKS,
	19,		0,		255,		308250*MBLOCKS,
	20,		0,		255,		328500*MBLOCKS,
	170,		0,		255,		332850*MBLOCKS,
};
struct datacon_info hitachi4_data[] =	/* Hitachi disc 4 tracks and data */
{
	/* track 1, size, data contents */
	1,		MAX_DATACMP,
	0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 
	0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 
	0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 
	0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 
	0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 
	0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 
	0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 
	0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 
	0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 
	0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 
	0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 
	0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 
	0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 
	0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 
	0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 
	0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 
	0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 
	0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 
	0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 
	0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 
	0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 
	0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 
	0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 
	0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 
	0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 
	0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 
	0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 
	0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 
	0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 
	0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 
	0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 
	0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00,
	0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 
	0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 
	0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 
	0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 
	0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 
	0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 
	0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 
	0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 
	0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 
	0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 
	0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 
	0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 
	0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 
	0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 
	0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 
	0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 
	0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 
	0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 
	0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 
	0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 
	0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 
	0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 
	0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 
	0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 
	0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 
	0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 
	0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 
	0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 
	0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 
	0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 
	0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 
	0x00, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02,

	/* lead out track as terminator */
	CDROM_LEADOUT,	0,
};


/*
 * CD Assist specific information.
 */
struct toc_info cdassist_toc[] =	/* CD Assist table of contents */
{
	/* track,	control,	data mode,	lba */
	1,		4,		1,		0*MBLOCKS,
	2,		0,		255,		5420*MBLOCKS,
	170,		0,		255,		55982*MBLOCKS,
};
struct datacon_info cdassist_data[] =	/* CD Assist tracks and data */
{
	/* track 1, size, data contents */
	1,		MAX_DATACMP,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

	/* lead out track as terminator */
	CDROM_LEADOUT,	0,
};


/*
 * PDO specific information.
 */
struct toc_info pdo_toc[] =		/* PDO table of contents */
{
	/* track,	control,	data mode,	lba */
	1,		4,		1,		0*MBLOCKS,
	170,		0,		255,		330225*MBLOCKS,
};
struct datacon_info pdo_data[] =	/* PDO tracks and data */
{
	/* track 1, size, data contents */
	1,		MAX_DATACMP,
	0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x53, 0x45, 
	0x43, 0x54, 0x4f, 0x52, 0x3a, 0x30, 0x30, 0x30, 
	0x30, 0x30, 0x30, 0x20, 0x28, 0x30, 0x30, 0x3a, 
	0x30, 0x32, 0x3a, 0x30, 0x30, 0x29, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

	/* lead out track as terminator */
	CDROM_LEADOUT,	0,
};


/*
 * Information on all supported CDs.
 */
char *cd_name[] =			/* all supported CD types */
{
	"sony2",
	"hitachi4",
	"cdassist",
	"pdo",
	"other",
	0,
};
struct toc_info *cd_tochdr[] =		/* all CD table of contents */
{
	sony2_toc,
	hitachi4_toc,
	cdassist_toc,
	pdo_toc,
	0,
};
struct datacon_info *cd_datacon[] =	/* all CD data contents */
{
	sony2_data,
	hitachi4_data,
	cdassist_data,
	pdo_data,
	0,
};

