/*
 * Intel QV Linux kernel driver
 * Copyright (c) 1999 - 2012, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

/*
 *  Module Name:
 *    linuxdriverpci_i.c
 *
 *  Abstract:
 *    This file contains function implementations for NAL PCI
 *    functions for the Linux Driver portion.  These routines
 *    are called as IOCTL's from the NalResolveIoctl routines.
 *
 *
 * VSS Revision Control Information:
 * ---------------------------------
 *   $Workfile: linuxdriverpci_i.c $
 *   $Date: 2012/01/16 13:19:47 $
 *   $Archive: /QV2.0/nal/src/linux/driver/linuxdriverpci_i.c $
 *   $Revision: 1.20 $
 */

#include <nalcodes.h>
#include <naltypes.h>
#include <os_i.h>
#include <hwbus_t.h>
#include <pci_i.h>
#include "linuxnaldriver.h"
#include <linux/module.h>
#include <linux/pci.h>
#include <asm/byteorder.h>

#ifndef CONFIG_PCI
#error "This driver requires PCI support to be available"
#endif

/***************************************************************************
**
** Name:            _NalReadPciDeviceCount()
**
** Description:     Scans all PCI buses and counts the number of devices.
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       Nothing.
**
** Returns:         Number of devices on the system.
**
** Notes:           This scans the devices once in life of the driver currently. If
**                  drivers are enabled/disabled, this isnt updated. This needs to
**                  change somehow while not scanning this stuff everytime.
**
****************************************************************************/
UINT16
_NalReadPciDeviceCount(
    VOID
    )
{
    /* This is implemented in the library portion of Linux NAL. This function only exists
     * so the generic IOCTL interface doesnt have an unresolved external reference */
    return 0;
}


/***************************************************************************
**
** Name:            _NalReadPciDevice()
**
** Description:     Scans all PCI buses and stores the config info.
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       SlotIds = pointer to allocated memory of SlotIds.
**                  Count   = Number of slots to fill.
**
** Returns:         NAL_STATUS.
**
****************************************************************************/
NAL_STATUS
_NalReadPciDevice(
    OUT NAL_DEVICE_LOCATION* PciLocations,
    IN  UINT16*              Count
    )
{
    /* This is implemented in the library portion of Linux NAL. This function only exists
     * so the generic IOCTL interface doesnt have an unresolved external reference */
    return NAL_NOT_IMPLEMENTED;
}

/***************************************************************************
**
** Name:            _NalFillPciConfigSpec()
**
** Description:     Fills the PCI_DEVICE structure at slot SlotId.
**
** Author:          MVM
**
** Born on Date:    02/28/2002
**
** Arguments:       PciLocation = PCI address (Bus, Dev, Function)
**                  Device      = PCI_DEVICE pointer to fill.
**                  DwordCount  = Number of words from PCI Config to fill
**
** Returns:         NAL_SUCCESS if the specified number of DWORDS were
**                  read into the PCI_DEVICE structure.
**                  NAL_INVALID_PARAMETER if one of the specified
**                  parameters were out of range or NULL.
**
****************************************************************************/
NAL_STATUS
_NalFillPciConfigSpec(
    IN  NAL_DEVICE_LOCATION PciLocation,
    OUT PCI_DEVICE*         Device,
    IN  UINT32              DwordCount
    )
{
    /* This is implemented in the library portion of Linux NAL. This function only exists
     * so the generic IOCTL interface doesnt have an unresolved external reference */
    return NAL_NOT_IMPLEMENTED;
}

/***************************************************************************
**
** Name:            _NalOSReadPciConfig32()
**
** Description:     Reads a specified 32bit value from the PCI config space.
**
** Author:          MVM
**
** Born on Date:    04/10/2001
**
** Arguments:       SlotId        = Slot number (Bus, Dev, Function)
**                  DwordNumber   = One of the 64 DWORDS to read from PCI Config Space.
**                  Value         = Location to store the word read.
**
** Returns:         NAL_STATUS Value & Fills Device pointer.
**
****************************************************************************/
NAL_STATUS
_NalOSReadPciConfig32(
    IN  NAL_DEVICE_LOCATION PciLocation,
    IN  UINT32              DwordNumber,
    OUT UINT32*             Value
    )
{
    struct pci_dev*         LinuxPciDevice = NULL;
    struct pci_bus*         LinuxPciBus    = NULL;
    NAL_STATUS              NalStatus      = NAL_INVALID_PCI_SLOT_ID;
    unsigned int            DeviceFunction = 0;
    int                     Result         = 0;

    /* Use segments for creating a Linux PCi Device, for segemented architecture */
    LinuxPciBus = pci_find_bus(PciLocation.Pci.Segment, PciLocation.Pci.Bus);

    if(LinuxPciBus != NULL)
    {
        /* Set up the DeviceFunction variable */
        DeviceFunction = PCI_DEVFN(PciLocation.Pci.Device, PciLocation.Pci.Function);

        /* Get the pci device from the bus/dev/function */
        LinuxPciDevice = pci_get_slot(LinuxPciBus, DeviceFunction);

        /* Read the DWORD */
        if(LinuxPciDevice != NULL)
        {
            Result = pci_read_config_dword(LinuxPciDevice, DwordNumber*4, (u32*)Value);

            if(Result == 0)
            {

                NalStatus = NAL_SUCCESS;
            }
            else
            {
                NalStatus = NAL_PCI_CAPABILITY_NOT_FOUND;
                *Value = 0;
            }
        }
        pci_dev_put(LinuxPciDevice);
    }
    return NalStatus;
}

/***************************************************************************
**
** Name:            _NalOSWritePciConfig32()
**
** Description:     Writes a specified 32bit value to the PCI config space.
**
** Author:          MVM
**
** Born on Date:    04/05/2001
**
** Arguments:       SlotId     = Slot number (Bus, Dev, Function)
**                  DwordNumber   = One of the 64 DWORDS to read from PCI Config Space.
**                  Data       = Value to write.
**
** Returns:         NAL_STATUS Value
**
****************************************************************************/
NAL_STATUS
_NalOSWritePciConfig32(
    IN  NAL_DEVICE_LOCATION PciLocation,
    IN  UINT32              DwordNumber,
    IN  UINT32              Data
    )
{
    struct pci_dev*         LinuxPciDevice = NULL;
    struct pci_bus*         LinuxPciBus    = NULL;
    NAL_STATUS              NalStatus      = NAL_INVALID_PCI_SLOT_ID;
    unsigned int            DeviceFunction = 0;
    int                     Result         = 0;

    /* Set up the DeviceFunction variable */
    DeviceFunction = PCI_DEVFN(PciLocation.Pci.Device, PciLocation.Pci.Function);

    /* Use segments for creating a Linux PCi Device, for segemented architecture */
    LinuxPciBus = pci_find_bus(PciLocation.Pci.Segment, PciLocation.Pci.Bus);

    if(LinuxPciBus != NULL)
    {
        /* Get the pci device from the bus/dev/function */
        LinuxPciDevice = pci_get_slot(LinuxPciBus, DeviceFunction);

        /* Read the DWORD */
        if(LinuxPciDevice != NULL)
        {
            Result = pci_write_config_dword(LinuxPciDevice, DwordNumber*4, Data);
            if(Result == 0)
            {
                NalStatus = NAL_SUCCESS;
            }
            else
            {
                NalStatus = NAL_PCICONFIG_NOT_AVAILABLE;
            }
        }
        pci_dev_put(LinuxPciDevice);
    }
    return NalStatus;
}

/***************************************************************************
**
** Name:            _NalOSWritePciConfigVariable()
**
** Description:     Write a value 32bits or less depending on the mask value to
**                  the PCI Config Space. Might require more than 1 write.
**
** Author:          MVM
**
** Born on Date:    01/30/2002
**
** Arguments:       SlotId        = Slot number (Bus, Dev, Function)
**                  DwordNumber   = One of the 64 DWORDS to read from PCI Config Space.
**                  ByteMask      = Mask value to mask on the Data value.
**                  Data          = Value to write.
**
** Returns:         NAL_STATUS Value
**
****************************************************************************/
NAL_STATUS
_NalOSWritePciConfigVariable(
    IN  NAL_DEVICE_LOCATION PciLocation,
    IN  UINT32              DwordNumber,
    IN  UINT8               ByteMask,
    IN  UINT32              Data
    )
{
    struct pci_dev*         LinuxPciDevice = NULL;
    struct pci_bus*         LinuxPcibus    = NULL;
    NAL_STATUS              NalStatus      = NAL_INVALID_PCI_SLOT_ID;
    unsigned int            DeviceFunction = 0;
    int                     Result         = 0;
    UINT16                  Data16         = 0;
    UINT8                   Data8          = 0;

    /* Set up the DeviceFunction variable */
    DeviceFunction = PCI_DEVFN(PciLocation.Pci.Device, PciLocation.Pci.Function);


    /* Read the DWORD */
    do
    {
        /* Use segments for creating a Linux PCi Device, for segemented architecture */
        LinuxPcibus = pci_find_bus(PciLocation.Pci.Segment, PciLocation.Pci.Bus);
        if(LinuxPcibus == NULL)
        {
            break;
        }

        /* Get the pci device from the bus/dev/function */
        LinuxPciDevice = pci_get_slot(LinuxPcibus, DeviceFunction);

        if(LinuxPciDevice == NULL)
        {
            break;
        }
        /* There are several cases to handle here. First case is ByteMask wants the
           entire value written */
        if(ByteMask == PCI_CONFIG_BM_ENTIRE_DWORD)
        {
            Result = pci_write_config_dword(LinuxPciDevice, DwordNumber*4, Data);
            break;
        }

        /* The next case to write the two right-most bytes */
        if(ByteMask == PCI_CONFIG_BM_LOWORD)
        {
            Data16 = LOWORD(Data);
            Result = pci_write_config_word(LinuxPciDevice, DwordNumber*4, Data);
            break;
        }

        /* The next case to write the two left-most bytes */
        if(ByteMask == PCI_CONFIG_BM_HIWORD)
        {
            Data16 = HIWORD(Data);
            Result = pci_write_config_word(LinuxPciDevice, (DwordNumber*4 + 2), Data);
            break;
        }

        /* If still here, then we are going to write each byte at a time */
        if(ByteMask & PCI_CONFIG_BM_BYTE1)
        {
            Data8 = (UINT8)Data;
            Result = pci_write_config_byte(LinuxPciDevice, (DwordNumber*4), Data8);
        }

        if(ByteMask & PCI_CONFIG_BM_BYTE2)
        {
            Data8 = (UINT8)(Data >> 8);
            Result = pci_write_config_byte(LinuxPciDevice, (DwordNumber*4 + 1), Data8);
        }

        if(ByteMask & PCI_CONFIG_BM_BYTE3)
        {
            Data8 = (UINT8)(Data >> 16);
            Result = pci_write_config_byte(LinuxPciDevice, (DwordNumber*4 + 2), Data8);
       }

        if(ByteMask & PCI_CONFIG_BM_BYTE4)
        {
            Data8 = (UINT8)(Data >> 24);
            Result = pci_write_config_byte(LinuxPciDevice, (DwordNumber*4 + 3), Data8);
        }
    }
    while(FALSE);
    if(Result == 0)
    {
        NalStatus = NAL_SUCCESS;
    }
    else
    {
        NalStatus = NAL_PCICONFIG_NOT_AVAILABLE;
    }
    return NalStatus;
}

/***************************************************************************
**
** Name:            _NalEnablePciDevice()
**
** Description:     Enable the PCI slot.
**
** Author:          MVM
**
** Born on Date:    03/27/2003
**
** Arguments:       DeviceLocation = Slot to enable.
**
** Returns:         NAL_STATUS Value
**
****************************************************************************/
NAL_STATUS
_NalEnablePciDevice(
    IN      NAL_DEVICE_LOCATION     DeviceLocation
    )
{
    struct pci_dev*         LinuxPciDevice = NULL;
    struct pci_bus*         LinuxPciBus    = NULL;
    unsigned int            DeviceFunction = 0;
    int                     Result         = 0;
    NAL_STATUS              Status         = NAL_SUCCESS;

    /* Set up the DeviceFunction variable */
    DeviceFunction = PCI_DEVFN(DeviceLocation.Pci.Device, DeviceLocation.Pci.Function);

    /* Use segments for creating a Linux PCi Device, for segemented architecture */
    LinuxPciBus = pci_find_bus(DeviceLocation.Pci.Segment, DeviceLocation.Pci.Bus);

    if(LinuxPciBus != NULL)
    {
        /* Get the pci device from the bus/dev/function */
        LinuxPciDevice = pci_get_slot(LinuxPciBus, DeviceFunction);

        /* Enable the device now */
        Result = pci_enable_device(LinuxPciDevice);

        if(Result != 0)
        {
            Status = NAL_PCICONFIG_NOT_AVAILABLE;
        }
    }
    return Status;
}

/***************************************************************************
**
** Name:            _NalOSWritePciExConfig32()
**
** Description:     Writes a specified 32bit value to the PCI config space.
**
** Author:          AS
**
** Born on Date:    05/15/2009
**
** Arguments:       Location     = (Bus, Dev, Function)
**                  DwordNumber  = One of the 64 DWORDS to read from PCI Config Space.
**                  Data         = Value to write.
**
** Returns:         NAL_STATUS Value
**
****************************************************************************/
NAL_STATUS
_NalOSWritePciExConfig32(
    IN  NAL_DEVICE_LOCATION PciLocation,
    IN  UINT32              DwordNumber,
    IN  UINT32              Data
    )
{
    return _NalOSWritePciConfig32(PciLocation, DwordNumber, Data);
}

/***************************************************************************
**
** Name:            _NalOSReadPciExConfig32()
**
** Description:     Reads a specified 32bit value to the PCI config space.
**
** Author:          AS
**
** Born on Date:    05/15/2009
**
** Arguments:       Location     = (Bus, Dev, Function)
**                  DwordNumber  = One of the 64 DWORDS to read from PCI Config Space.
**                  Value         = Value to read in.
**
** Returns:         NAL_STATUS Value
**
****************************************************************************/
NAL_STATUS
_NalOSReadPciExConfig32(
    IN  NAL_DEVICE_LOCATION PciLocation,
    IN  UINT32              DwordNumber,
    IN  UINT32*             Value
    )
{
   return _NalOSReadPciConfig32(PciLocation, DwordNumber, Value);
}

/***************************************************************************
**
** Name:            _NalOsReadPciExByte()
**
** Description:     Reads an  8bit value from the PCI config space, starting at a byteoffset.
**
** Author:          AS
**
** Born on Date:    12/06/2004
**
** Arguments:       DeviceLocation = The device location.
**                  ByteIndex      = Starting offset to read from PCI Config Space.
**                  NoOfBytes      = Number of Bytes to read.
**                  Value          = Space provided by the user to return the read config space
**
** Returns:         NAL_INVALID_PARAMETER = Invalid paramter.
**                  NAL_SUCCESS           = Operation was successful
**
****************************************************************************/
NAL_STATUS
_NalOsReadPciExByte(
    IN  NAL_DEVICE_LOCATION  DeviceLocation,
    IN  UINT32               ByteIndex,
    OUT UINT8*               Value
    )
{
    struct pci_dev*         LinuxPciDevice  = NULL;
    struct pci_bus*         LinuxPciBus     = NULL;
    NAL_STATUS              Status          = NAL_INVALID_PCI_SLOT_ID;
    unsigned int            DeviceFunction  = 0;

    printk(KERN_DEBUG "in the driver Pci Ex config function byte index %d\n", ByteIndex);

    /* Use segments for creating a Linux PCi Device, for segemented architecture */
    LinuxPciBus = pci_find_bus(DeviceLocation.Pci.Segment, DeviceLocation.Pci.Bus);

    if((LinuxPciBus != NULL) && (ByteIndex < 4096))
    {
        /* Set up the DeviceFunction variable */
        DeviceFunction = PCI_DEVFN(DeviceLocation.Pci.Device, DeviceLocation.Pci.Function);

        /* Get the pci device from the bus/dev/function */
        LinuxPciDevice = pci_get_slot(LinuxPciBus, DeviceFunction);

        /* Read the DWORD */
        if(LinuxPciDevice != NULL)
        {
            pci_read_config_byte(LinuxPciDevice, ByteIndex, Value);
            Status = NAL_SUCCESS;
        }
    }

    return Status;
}

/***************************************************************************
**
** Name:            _NalOsWritePciExByte()
**
** Description:     Writes a byte value to the PCI config space, starting at a byteoffset.
**
** Author:          AS
**
** Born on Date:    05/15/2009
**
** Arguments:       DeviceLocation = The device location.
**                  ByteIndex      = Starting offset to read from PCI Config Space.
**                  NoOfBytes      = Number of Bytes to read.
**                  Value          = Space provided by the user to return the read config space
**
** Returns:         NAL_INVALID_PARAMETER = Invalid paramter.
**                  NAL_SUCCESS           = Operation was successful
**
****************************************************************************/
NAL_STATUS
_NalOsWritePciExByte(
    IN  NAL_DEVICE_LOCATION  DeviceLocation,
    IN  UINT32               ByteOffset,
    IN  UINT8                Data
    )
{
    struct pci_dev*         LinuxPciDevice  = NULL;
    struct pci_bus*         LinuxPciBus     = NULL;
    NAL_STATUS              Status          = NAL_INVALID_PCI_SLOT_ID;
    unsigned int            DeviceFunction  = 0;

    /* Use segments for creating a Linux PCi Device, for segemented architecture */
    LinuxPciBus = pci_find_bus(DeviceLocation.Pci.Segment, DeviceLocation.Pci.Bus);

    if((LinuxPciBus != NULL) && (ByteOffset < 4096))
    {
        /* Set up the DeviceFunction variable */
        DeviceFunction = PCI_DEVFN(DeviceLocation.Pci.Device, DeviceLocation.Pci.Function);

        /* Get the pci device from the bus/dev/function */
        LinuxPciDevice = pci_get_slot(LinuxPciBus, DeviceFunction);

        /* Read the DWORD */
        if(LinuxPciDevice != NULL)
        {
            pci_write_config_byte(LinuxPciDevice, ByteOffset, Data);
            Status = NAL_SUCCESS;
        }
    }

    return Status;
}

