/**********        Intel Confidential       ***********/
//----------------------------------------------------------------------------
//                    INTEL CORPORATION PROPRIETARY INFORMATION
//      This software is supplied under the terms of a license agreement or
//      nondisclosure agreement with Intel Corporation and may not be copied
//      or disclosed except in accordance with the terms of that agreement.
//        Copyright (c) 2008-2010 Intel Corporation.  All Rights Reserved.
//
//  File:       $ipmi.cpp
//  Contents:   Sample code for an Intel SPS Firmware update application
//
//----------------------------------------------------------------------------

#include "stdafx.h"


/*++
******************************************************
***      Private variables & declarations
******************************************************
--*/ 

#define IPMI_TRANSACTION_SLOTS  20
#define IPMI_NUM_RETRY			5
#define IPMI_RESPONSE_TIMEOUT	250 // ms
#define IPMI_RETRY_WAIT_TIME	50



typedef struct
{
	IpmiDataType	Frame;
	IpmiDataType	BridgedFrame;
	clock_t			TimeStart;
	long long		handle;

} IpmiTransactionType;



typedef struct 
{
	int				DestAddr;
	int				SourceAddr;
	int				BridgeAddr;
	int				BridgeEnable;
	int				BridgeChannelId;

	unsigned int	SentFrames;
	unsigned int	RcvFrames;


	IpmiTransactionType Transactions[IPMI_TRANSACTION_SLOTS];
	unsigned int					TransactionHead;
	unsigned int					TransactionTail;
	int					TransactionSize;

	
	IpmiDataType		InFrame;
	IpmiDataType		InBridgedFrame;

	int SeqNr;

} IpmiStackType;

static IpmiStackType  stack;

/*++
 *****************************************************************************

 Routine Description:
				Setup of IPMI stack
 Arguments:
				WindowSize		- number of the "Send Message" commands to be sent in a single transaction
				DestAddr		- destination address ( remote device)
				SourceAddr		- source address (our address)
				BridgeAddr		- IPMI bridge address
				BridgeEnable	- if nonzero, then messages will be sent via bridge ( using "Send Message" /"Get Message" commands )
				BridgeChannelId	- channel id for the messages to be sent via bridge into
 Returns:
				-

 *****************************************************************************
 --*/ 
void IpmiSetupTransaction(
	int WindowSize,
	int DestAddr,
	int SourceAddr,
	int BridgeAddr,
	int BridgeEnable,
	int BridgeChannelId)
{
	stack.DestAddr     = DestAddr;
	stack.SourceAddr   = SourceAddr;
	stack.BridgeAddr   = BridgeAddr;
	stack.BridgeEnable    = BridgeEnable;
	stack.BridgeChannelId = BridgeChannelId;
	stack.SeqNr = 0;

	memset(&stack.Transactions,0,sizeof(stack.Transactions));
	stack.TransactionHead = 0;
	stack.TransactionTail = 0;
	stack.TransactionSize = (WindowSize > 0) ? WindowSize : 1;
}


/*++
 *****************************************************************************

 Routine Description:
				Prepares IPMI frame
 Arguments:
				DestAddr	- frame destination address
				SourceAddr	- frame source address
				NetFn		- frame net function
				DestLun		- frame destination Lun
				SourceLun	- frame source lun
				Command		- frame command
				pSendData0	- frame data ( first part )
				DataLength0 - frame data length ( first part )
				pSendData1	- frame data ( second part )
				DataLength1	- frame data length ( second part )
				pFrame		- pointer to frame structure 	
 Returns:
    			0 - Success
				All other values - failure

 *****************************************************************************
 --*/ 
static int PrepareIpmiFrame(
	int           DestAddr,
	int           SourceAddr,
	int           NetFn,
	int           DestLun,
	int           SourceLun,
	int           Command,
	char*         pSendData0,
	unsigned int  DataLength0,
	char*         pSendData1,
	unsigned int  DataLength1,
	IpmiDataType* pFrame)
{
   int ii,jj;

   if(DataLength1+DataLength0 > sizeof(pFrame->Data))
   {
	   return -1;
   }

   /* Logical frame */
   pFrame->DestAddress = (unsigned char)DestAddr;
   pFrame->NetFn_DestLun = (unsigned char)((NetFn << 2) | (DestLun & 0x03));
   pFrame->HdrChksum = (unsigned char)256 - (pFrame->DestAddress + pFrame->NetFn_DestLun);
   pFrame->SourceAddress = (unsigned char)SourceAddr;
   pFrame->SeqNumber_SourceLun	= (unsigned char)((stack.SeqNr++ & 0x3F)<<2) | (SourceLun & 0x03);
   pFrame->Command = (unsigned char)Command;

   pFrame->Chksum = pFrame->SourceAddress + pFrame->SeqNumber_SourceLun + pFrame->Command;
   pFrame->DataLength = 0;
   for(ii =0; ii < (int)DataLength0;ii++)
   {
	  pFrame->Data[ii] = pSendData0[ii];
	  pFrame->Chksum += pSendData0[ii];
   }
   for(jj =0; jj < (int)DataLength1;jj++)
   {
	  pFrame->Data[ii+jj] = pSendData1[jj];		
	  pFrame->Chksum += pSendData1[jj];
   }
   pFrame->DataLength = DataLength0 + DataLength1;	

   pFrame->Chksum = (unsigned char)256 - pFrame->Chksum;
	
   /* Raw format frame */
   pFrame->RawFrame[0] = pFrame->DestAddress;
   pFrame->RawFrame[1] = pFrame->NetFn_DestLun;
   pFrame->RawFrame[2] = pFrame->HdrChksum;
   pFrame->RawFrame[3] = pFrame->SourceAddress;
   pFrame->RawFrame[4] = pFrame->SeqNumber_SourceLun;
   pFrame->RawFrame[5] = pFrame->Command;

   if((int)pFrame->DataLength >= (sizeof(pFrame->Data) / sizeof(pFrame->Data[0])))
   {
       return -1;
   }

   for(ii =0; ii < (int)pFrame->DataLength ;ii++)
   {
       pFrame->RawFrame[6+ ii] = pFrame->Data[ii];	
   }
   pFrame->RawFrame[6+ ii] = pFrame->Chksum;
   pFrame->RawFrameLength = 6+ ii + 1;

   pFrame->RawFormatDecoded	= FRAME_DECODED;

   return 0;
}



/*++
 *****************************************************************************

 Routine Description:
			Converts frame raw format into logical format
 Arguments:
			pFrame - pointer to frame structure
 Returns:
    		0 - Success
			All other values - failure


 *****************************************************************************
 --*/ 

static int DecomposeMessage(IpmiDataType * pFrame)
{
   int ii;

   if( (pFrame->RawFormatDecoded	& FRAME_DECODED) != 0)
   {
		// Already processed
		return 0;
   }

	
   if( (pFrame->RawFrameLength < 7) || (pFrame->RawFrameLength > sizeof(pFrame->RawFrame)) )
   {
	   //Wrong length ??
	   return -1;
   }
	
   pFrame->DestAddress		= pFrame->RawFrame[0];
   pFrame->NetFn_DestLun	= pFrame->RawFrame[1];
   pFrame->HdrChksum		= pFrame->RawFrame[2];
   pFrame->SourceAddress    = pFrame->RawFrame[3];
   pFrame->SeqNumber_SourceLun = pFrame->RawFrame[4];
   pFrame->Command			= pFrame->RawFrame[5];

   pFrame->DataLength = pFrame->RawFrameLength - 6 -1/*crc*/;
	
   for(ii =0; ii < (int)pFrame->DataLength;ii++)
   {
		pFrame->Data[ii] = pFrame->RawFrame[6+ii];
   }
   pFrame->Chksum = pFrame->RawFrame[6+ii];	
   pFrame->RawFormatDecoded	|= FRAME_DECODED;

   return 0;	
}

/*++
 *****************************************************************************

 Routine Description:
			Check message validity ( checksums)
 Arguments:
			pFrame - pointer to frame structure
 Returns:
    		0 - Success
			All other values - failure

 *****************************************************************************
 --*/ 

static int CheckMessage(IpmiDataType * pFrame)
{
    unsigned char crc;
    int ii;

    if( (pFrame->RawFormatDecoded & FRAME_NOCHCKSUM) != 0)
    {
		// Already verified
		return 0;
    }
   
	crc = (unsigned char)256 - (pFrame->DestAddress  + pFrame->NetFn_DestLun);
	if(crc != pFrame->HdrChksum)
	{

		return -1;
	}

    crc = pFrame->SourceAddress + pFrame->SeqNumber_SourceLun + pFrame->Command;
    for(ii =0; ii < (int)pFrame->DataLength;ii++)
    {	    		
	   crc+= pFrame->Data[ii];
    }
    crc = (unsigned char)256 - crc;
	if(crc != pFrame->Chksum)
	{
		return -2;
	}

	if(pFrame->DataLength > sizeof(pFrame->Data))
	{
		return -3;
	}

	return 0;

}

/*++
 *****************************************************************************

 Routine Description:
				Checks if two IPMI messages form matching request-response
 Arguments:
				request  - pointer to request frame
				response - pointer to response frame
 Returns:
				0	- match
				-1	- no match

 *****************************************************************************
 --*/ 

static int MatchMessage(IpmiDataType * request,IpmiDataType * response)
{
	if( (request->SeqNumber_SourceLun>>2)  == (response->SeqNumber_SourceLun>>2) )
	{
		// Match !!
		return 0;
	}
	else
	{
		return -1;
	}
}

/*++
 *****************************************************************************

 Routine Description:
				Prepares and send IPMI frame  
 Arguments:	
				pTransaction - pointer to IPMI stack stransaction structure
				NetFn		- frame net function
				Command		- frame command
				pSendData	- frame data 
				DataLength  - frame data length 
				SourceLun	- frame source lun
				DestLun		- frame destination Lun
 Returns:
    			0 - Success
				All other values - failure  

 *****************************************************************************
 --*/ 
static int SendFrame(
	IpmiTransactionType* pTransaction,
	int                  NetFn,
	int                  Command,
	char*                pSendData,
	unsigned int         DataLength,
	unsigned int         SrcLun,
	unsigned int         DestLun)
{
	char ExtraData[10];
	int  ExtraDataLength;
	int  result = -1;


	if(stack.BridgeEnable)
	{
		SrcLun = 0x02; //SMS Lun must be used
	}
	if(PrepareIpmiFrame(stack.DestAddr,stack.SourceAddr,NetFn,DestLun,SrcLun,Command,pSendData,DataLength,NULL,0,&pTransaction->Frame) != 0)
	{
		// Fatal
		return result ;
	}

	if (stack.BridgeEnable)
	{
		//  Encapsulate the command into bridged command	
		//  Use 'Send Message' frame
		ExtraData[0]	= stack.BridgeChannelId; /* No tracking, configured channel */
		ExtraDataLength = 1;
		if (PrepareIpmiFrame(stack.BridgeAddr, stack.SourceAddr, IPMI_NETFN_APPL, DestLun, SrcLun,
			                 IPMI_CMD_SEND_MESSAGE, ExtraData,ExtraDataLength, (char*)pTransaction->Frame.RawFrame,
			                 pTransaction->Frame.RawFrameLength, &pTransaction->BridgedFrame) != 0)
		{
			// Fatal
			return result;
		}
		result = transport.SendRequest(&pTransaction->BridgedFrame);
		//Sleep(100);
	}
	else
	{
		// Message to ME are sent directly 
		result = transport.SendRequest(&pTransaction->Frame);
	}
		
	return result;
}

/*++
 *****************************************************************************

 Routine Description:
				Retrives IPMI message response from physical communication channel
 Arguments:
				pTransaction - pointer to IPMI stack stransaction structure
				timeout - timeout waiting for response
 Returns:
  

 *****************************************************************************
 --*/ 
static int GetFrame(IpmiTransactionType * pTransaction,int timeout)
{
	clock_t					now = clock();
	clock_t					running;
	unsigned int			timespan;
	int						result;

	if(stack.BridgeEnable)
	{
		if(transport.GetBridgedResponse != NULL)
		{
			//  The driver polls itself 'attention' bits and extracts message queue itself
			//  We need to get the message from its queue ( no extra 'GetMessage' is needed )

			if(transport.GetBridgedResponse(&pTransaction->Frame,timeout)!=0)
			{
				return -1;
			}	
			pTransaction->Frame.RawFrame[0] = stack.SourceAddr;
		}
		else
		{
			//  Use 'Get Message' frame to retrive the bridged answer
			// 

			while(1)
			{
				bool isResp = 1;

				if (transport.IsResponse != NULL)
				{
					result = transport.IsResponse(&isResp);
					if(result != 0)
					{
						return -1;
					}
				}

				if (isResp)
				{
					if(PrepareIpmiFrame(stack.BridgeAddr,stack.SourceAddr,IPMI_NETFN_APPL,0,0,IPMI_CMD_GET_MESSAGE,NULL,0,NULL,0,&pTransaction->BridgedFrame) != 0)
					{
						// Fatal
						return -1 ;
					}
					result = transport.SendRequest(&pTransaction->BridgedFrame);

					if(result == 0)
					{
						if(transport.GetResponse(&pTransaction->BridgedFrame,timeout)!=0)
						{
							return -1;
						}

						// Got something
						if(DecomposeMessage(&pTransaction->BridgedFrame)!= 0)
						{
							// Wrong contents
							continue;
						}

						if(CheckMessage(&pTransaction->BridgedFrame)!= 0)
						{
							// Wrong checksums
							continue;
						}

						if(pTransaction->BridgedFrame.DataLength >=3)
						{
						
							pTransaction->Frame.RawFrame[0] = stack.SourceAddr;
							pTransaction->Frame.RawFrameLength = pTransaction->BridgedFrame.DataLength  - 2  + 1;
							memcpy(&pTransaction->Frame.RawFrame[1],&pTransaction->BridgedFrame.Data[2],pTransaction->Frame.RawFrameLength-1);
							pTransaction->Frame.RawFormatDecoded = 0;
							break;
						}
					}
				}

				running = clock();

				timespan = (unsigned int)(1000*((float)(running - now)/CLOCKS_PER_SEC));

				if( timespan > IPMI_RESPONSE_TIMEOUT)
				{
					return -1;
				}
			}
		}
	}
	else
	{
		if(transport.GetResponse(&pTransaction->Frame,timeout)!=0)
		{
			return -1;
		}
	}

	// Decompose final message

	if(DecomposeMessage(&pTransaction->Frame)!= 0)
	{
		// Wrong contents
		return -1;
	}

	if(CheckMessage(&pTransaction->Frame)!= 0)
	{
		// Wrong checksums
		return -1;
	}

	return 0;
}

/*++
 *****************************************************************************

 Routine Description:
				High level routine for cleaning out IPMI asynchronous responses.
				Sends requests for asynchronous responses until all of them are received.

 Arguments:
				NetFn		- frame net function
 Returns:
				0			- Success
				-1			- failure

 *****************************************************************************
--*/
int IpmiCleanResponses(int * msgCnt)
{
	IpmiTransactionType		TransactionIn;

	if(stack.BridgeEnable)
	{
		while(1)
		{
			if(GetFrame(&TransactionIn,IPMI_RESPONSE_TIMEOUT) != 0)
			{
				return 0;
			}
			else
			{
				(*msgCnt)++;
			}
		}
	}
	else
	{
		return 0;
	}

	return -1;
}


/*++
 *****************************************************************************

 Routine Description:
				High level routine for sending IPMI requests.
				Sends requests, waits for response ( returns completion code only)

 Arguments:
				NetFn		- frame net function
				Command		- frame command
				pSendData	- frame data 
				DataLength  - frame data length 
				SourceLun	- frame source lun
				DestLun		- frame destination Lun  
 Returns:
				-1 - failure
				All other - response frame completion code

 *****************************************************************************
 --*/
int IpmiSendSimpleRequest(
	int          NetFn,
	int          Command,
	char*        pSendData,
	unsigned int DataLength,
	unsigned int SrcLun,
	unsigned int DestLun)
{
	int				retry = IPMI_NUM_RETRY;
	int				result = -1;
	int				timeout;
	clock_t			now;
	clock_t			running;
	unsigned int	timespan;
	bool           message_found;

	IpmiTransactionType		TransactionIn;
	IpmiTransactionType		TransactionOut;
	
	TransactionOut.handle = -1;
	while(retry > 0)
	{
		if(retry != IPMI_NUM_RETRY)
		{
			Sleep(IPMI_RETRY_WAIT_TIME);
		}
		retry--;

		if(SendFrame(&TransactionOut,NetFn,Command,pSendData,DataLength,SrcLun,DestLun) !=0)
		{
			continue;
		}

		timeout = IPMI_RESPONSE_TIMEOUT/5;
		now = clock();

		message_found = false;
		while(1)
		{

			running = clock();
			timespan = (1000*(running - now))/CLOCKS_PER_SEC;

			if(timespan > IPMI_RESPONSE_TIMEOUT)
			{
				break;
			}

			if(GetFrame(&TransactionIn,IPMI_RESPONSE_TIMEOUT) == 0)
			{
				if(MatchMessage(&TransactionOut.Frame,&TransactionIn.Frame)== 0)
				{		
					if(TransactionIn.Frame.DataLength > 0)
					{
						result = (int)TransactionIn.Frame.Data[0];
						message_found = true;
					}
					else
					{
						result = -1;
					}
					break;
				}
			}
		}

		if(message_found)
		{
			break;
		}

	}

	return result;
}

/*++
 *****************************************************************************

 Routine Description:
  				High level routine for sending IPMI requests.
				Sends requests, does not wait for response.
 Arguments:
  				NetFn		- frame net function
				Command		- frame command
				pSendData	- frame data 
				DataLength  - frame data length 
				SourceLun	- frame source lun
				DestLun		- frame destination Lun 
 Returns:
  				-1 - failure
				All other values - Message handle. Matching response will be given the same handle value.

 *****************************************************************************
 --*/ 
int IpmiSendRequest(int NetFn, int Command, char * pSendData, unsigned int DataLength, long long handle, unsigned int SrcLun, unsigned int DestLun)
{
	IpmiTransactionType * pTransaction;
	unsigned int					  TmpHead; 	
	int					  retry = IPMI_NUM_RETRY;
	int					  result = -1;

	TmpHead = (stack.TransactionHead + 1)%IPMI_TRANSACTION_SLOTS;

	if(TmpHead == stack.TransactionTail)
	{
		//Not enough  space in buffers to hold another message
		return -1;
	}	

	pTransaction = &stack.Transactions[stack.TransactionHead];
	pTransaction->handle = handle;
	pTransaction->TimeStart= clock();

	while(retry > 0)
	{
		if(retry != IPMI_NUM_RETRY)
		{
			Sleep(IPMI_RETRY_WAIT_TIME);
		}
		retry--;
		if(SendFrame(pTransaction,NetFn,Command,pSendData,DataLength,SrcLun,DestLun) !=0)
		{
			continue;
		}
		else
		{
			stack.TransactionHead = TmpHead;
			result = 0;
			break;
		
		}
	}

	return result;
}

/*++
 *****************************************************************************

 Routine Description:
    			High level routine for retriving IPMI responses for requests sent with IpmiSendRequest().
 Arguments:
				handle - pointer to handle value of original request to which response has been matched
 Returns:
				-1 - failure ( handle value set to INT_MIN )
				All other values - completion code in response with given *handle value

 *****************************************************************************
 --*/ 
int   IpmiGetResponse(long long *handle, int *pDataLength, unsigned char** pData)
{
	int						result =-1;
	unsigned int						TmpTail;
	IpmiTransactionType		TransactionIn;
	clock_t					now = clock();
	clock_t					running;
	int						ii,jj;
	unsigned int			timespan;

	//clear returned data
	if(pData != NULL)
	{
		delete [] *pData;
		*pData = new unsigned char [DATAINDEX];
	}

	if(pDataLength != NULL)
	{
		*pDataLength=0;
	}

	if(stack.TransactionTail == stack.TransactionHead)
	{
		// Request without response ??
		*handle = _I64_MIN;
		return -1;
	}

	timespan = (unsigned int)(1000*((float)(now - stack.Transactions[stack.TransactionTail].TimeStart)/CLOCKS_PER_SEC));

	// Look for timed-out transactions
	if( timespan > (unsigned int)(stack.TransactionSize * IPMI_RESPONSE_TIMEOUT) )
	{
		*handle = stack.Transactions[stack.TransactionTail].handle;
		stack.TransactionTail = (stack.TransactionTail + 1)%IPMI_TRANSACTION_SLOTS;
		return -1;
	}

	while(1)
	{
		running = clock();

		timespan = (unsigned int)(1000*((float)(running - now)/CLOCKS_PER_SEC));

		if( timespan > IPMI_RESPONSE_TIMEOUT)
		{
			break;
		}

		if(GetFrame(&TransactionIn,IPMI_RESPONSE_TIMEOUT) == 0)
		{
			TmpTail = stack.TransactionTail;

			while(TmpTail != stack.TransactionHead)
			{
				// Match request with response
				if(MatchMessage(&TransactionIn.Frame,&stack.Transactions[TmpTail].Frame)!= 0)
				{
					// No match

					TmpTail  = (TmpTail  + 1)%IPMI_TRANSACTION_SLOTS;
					continue;
				}
				else
				{
					// Found !!
					*handle = stack.Transactions[TmpTail].handle;

					if(TmpTail == stack.TransactionTail )
					{
						// Should be always the case 
					}
					else
					{
						// Malformed case: responses in mixed order
						ii = TmpTail;
						while(1)
						{
							if(ii > 0)
							{
								jj = (ii - 1)%IPMI_TRANSACTION_SLOTS;
							}
							else
							{
								jj = IPMI_TRANSACTION_SLOTS - 1;
							}

							stack.Transactions[ii] = stack.Transactions[jj];						
							if(jj == stack.TransactionTail)
							{
								break;
							}
							ii = jj;
						}
					}

					result = (unsigned int)TransactionIn.Frame.Data[0];
					if(pData != NULL)
					{
						memcpy(* pData, TransactionIn.Frame.Data, sizeof(TransactionIn.Frame.Data));
					}
					if(pDataLength != NULL)
					{
						memcpy(pDataLength, &TransactionIn.Frame.DataLength, sizeof(TransactionIn.Frame.DataLength));
					}
					stack.TransactionTail  = (stack.TransactionTail  + 1)%IPMI_TRANSACTION_SLOTS;

					return result;
				}
			}

			// Response not matched to requests - drop it
		}
	}

	// Nothing ? then we return oldest transaction - at this time timed-out
	if(stack.TransactionTail != stack.TransactionHead)
	{
		*handle = stack.Transactions[stack.TransactionTail].handle;
		stack.TransactionTail  = (stack.TransactionTail  + 1)%IPMI_TRANSACTION_SLOTS;
		return -1;

	}
	else
	{	
		// Request without response ??
		*handle = _I64_MIN;
		return -1;
	}
}

/*++
 *****************************************************************************

 Routine Description:
    			High level routine for removing already acknowledge frames after receiving good response for frame with higher sequence.
 Arguments:
				handle - pointer to handle value of acknowledge request with higher sequency
 Returns:
				Number of removed frames
 *****************************************************************************
 --*/ 
int   IpmiRemoveAcknowledgeRequest(long long *handle)
{
	int RemovedFrameCounter = 0;

	while(1)
	{
		if((stack.Transactions[stack.TransactionTail].handle > *handle) ||
			(stack.TransactionHead == stack.TransactionTail))
		{
			return RemovedFrameCounter;
		}
		stack.TransactionTail = (stack.TransactionTail + 1) % IPMI_TRANSACTION_SLOTS;
		RemovedFrameCounter++;
	}
}

/*++
 *****************************************************************************

 Routine Description:
    			High level routine for removing frames that didn't have chance for acknowledge 
				- previous had unacceptable complition code.
 Arguments:
				handle - pointer to handle value of acknowledge request with higher sequency
 Returns:
				Number of removed frames
 *****************************************************************************
 --*/ 
int   IpmiDropRequest(long long *handle)
{
	int RemovedFrameCounter = 0;
	unsigned int TmpTail, ii, jj;
	
	TmpTail = stack.TransactionTail;

	while(TmpTail != stack.TransactionHead)
	{
		if(stack.Transactions[TmpTail].handle  >= *handle)
		{
			// Malformed case: responses in mixed order
			ii = TmpTail;
			while(1)
			{
				if(ii > 0)
				{
					jj = (ii - 1)%IPMI_TRANSACTION_SLOTS;
				}
				else
				{
					jj = IPMI_TRANSACTION_SLOTS - 1;
				}
				stack.Transactions[ii] = stack.Transactions[jj];						
				if(jj == stack.TransactionTail)
				{
					
					break;
				}
				ii = jj;
			}
			stack.TransactionTail = (stack.TransactionTail + 1) % IPMI_TRANSACTION_SLOTS;
			RemovedFrameCounter++;
		}
		TmpTail = (TmpTail + 1) % IPMI_TRANSACTION_SLOTS;
	}
	return RemovedFrameCounter;
}
