/**********        Intel Confidential       ***********/
//----------------------------------------------------------------------------
//                    INTEL CORPORATION PROPRIETARY INFORMATION
//      This software is supplied under the terms of a license agreement or
//      nondisclosure agreement with Intel Corporation and may not be copied
//      or disclosed except in accordance with the terms of that agreement.
//        Copyright (c) 2008-2010 Intel Corporation.  All Rights Reserved.
//
//  File:       $spsloader.cpp 
//  Contents:   Sample code for an Intel SPS Firmware update application
//
//----------------------------------------------------------------------------

#include "stdafx.h"
#include "image\me_region.h"


/*****************************************************************************
 * Definitions
 *****************************************************************************/
#define MAX_CONSEQUITVE_ERROR_COUNT 4
#ifdef INTEL_DRIVER
   #define WRITE_LEN_PACKAGE           55 // limited by Intel IPMI driver
#else
   #define WRITE_LEN_PACKAGE           68 // limited by IPMI specification
#endif
#define WRITE_LEN_PACKAGE_MAX       (sizeof(((IpmiDataType*)0)->Data)-16)
C_ASSERT(WRITE_LEN_PACKAGE <= WRITE_LEN_PACKAGE_MAX); // assure it fits in buffers

#define MIN_PARAMS_NUMBER           3
#define MAX_PARAMS_NUMBER           7



/*****************************************************************************
 * Variables
 *****************************************************************************/
SpsUpdateParametersType SpsUpdateParameters;


/*****************************************************************************
 * Functions
 *****************************************************************************/
/*++
 *****************************************************************************

 Routine Description:
				Performs ME Update
 Arguments:
  
 Returns:

 *****************************************************************************
 --*/ 
int main(int argc, char * argv[])
{
	int             handle = -1;
	long            FileSize; 
	long            CurrentFilePointer;
	int	          Length;
	int             Sequence;
	int             LastSequenceOk;
	int             LastValidSequence = 0;
	int             ReceivedSequence;
	int             ConsequitveErrorCount;
	int	          Response;
	int             DataLength;
	int             WriteLengthPackage;
	char          * endptr;
	unsigned char * buffer = NULL;
	unsigned char   params[IPMI_BUFFER_LEN];
	unsigned char * pTransData = new unsigned char [DATAINDEX];
	bool            ForceDowngrade = false;
    int             ArgNo = 0;

	int	          MsgSent = 0;
	int			    MsgReceived = 0;
	int			  MsgCleaned = 0;
    int             NofRepeatedWindows = 0;
	int			    LastPercentage = -1;
	clock_t		    start,startWhole,stop,LastAckTimestamp;
   long long       id;
   int             OrigImageNo = 0;
   PlatformIdType  PlatformId = (PlatformIdType)0;
	
	memset(&SpsUpdateParameters, 0, sizeof(SpsUpdateParameters));
   SpsUpdateParameters.WindowSize      = DEFAULT_WINDOW_SIZE;
   SpsUpdateParameters.BrigdeChannelId = DEFAULT_CHANNEL_ID;
   WriteLengthPackage                  = WRITE_LEN_PACKAGE;

   #ifdef _DEBUG
   SpsPrintf("[INFO] Sample Update Tool, compiled " __DATE__ " " __TIME__ ".\n");
   #endif
   #ifdef INTEL_DRIVER
   SpsPrintf("[INFO] Using Intel IPMI driver.\n");
   #else
   SpsPrintf("[INFO] Using Microsoft IPMI driver.\n");
   #endif

   /*
	 * Process input arguments
	 */
	if ((argc < MIN_PARAMS_NUMBER) || (argc > MAX_PARAMS_NUMBER))
	{
		SpsPrintf("Usage : SampleUpdateTool.exe MeAddr File [IpmiWindowSize] [BridgeChannelId] [ImgBytesInFrame] [-f]\n");
		SpsPrintf("Default IpmiWindowSize is %d (safe setting).\n", DEFAULT_WINDOW_SIZE);
		SpsPrintf("Default BridgeChannelId is %d.\n", DEFAULT_CHANNEL_ID);
		SpsPrintf("Default ImgBytesInFrame is %d (max is %d).\n", WRITE_LEN_PACKAGE, WRITE_LEN_PACKAGE_MAX);
		SpsPrintf("-f (force downgrade) can appear anywhere after mandatory parameters (default is false)\n");
		delete [] pTransData;
		return -1;
	}

	startWhole = clock();

	ArgNo = 1;
	SpsUpdateParameters.DestAddr = strtol(argv[ArgNo++], &endptr, 0);
	char * inputFile = argv[ArgNo++];
	if(strlen(inputFile)<sizeof(SpsUpdateParameters.InputFile)/sizeof(char))
	strcpy_s(SpsUpdateParameters.InputFile, inputFile);
	else 
	{
		SpsPrintf("[Error] Too long input file name");
		delete [] pTransData;
		return 1;
	}

   if ((argc > ArgNo) && (strstr("-f", argv[ArgNo])))
   {
      ForceDowngrade = true;
      ArgNo++;
   }
      
   if (argc > ArgNo)
	{
		// Overwrite default setting
		SpsUpdateParameters.WindowSize = strtol(argv[ArgNo++], &endptr, 0);
		if (SpsUpdateParameters.WindowSize > MAX_WINDOW_SIZE)
		{
			SpsPrintf("[Error] Given IpmiWindowSize is too big! (max is %d)\n", MAX_WINDOW_SIZE);
			delete [] pTransData;
			return -1;
		}
   }

   if ((argc > ArgNo) && (strstr("-f", argv[ArgNo])))
   {
      ForceDowngrade = true;
      ArgNo++;
   }
     
   if (argc > ArgNo)
	{
		// Overwrite default setting
		SpsUpdateParameters.BrigdeChannelId = strtol(argv[ArgNo++], &endptr, 0);
	}

   if ((argc > ArgNo) && (strstr("-f", argv[ArgNo])))
   {
      ForceDowngrade = true;
      ArgNo++;
   }

   if (argc > ArgNo)
	{
		// Overwrite default setting
		WriteLengthPackage = strtol(argv[ArgNo++], &endptr, 0);
      if (WriteLengthPackage > WRITE_LEN_PACKAGE_MAX)
      {
         SpsPrintf("[Error] Given ImgBytesInFrame is too big! (max is %d)\n", WRITE_LEN_PACKAGE_MAX);
			delete [] pTransData;
		   return -1;
      }
	}

   if (argc > ArgNo)
	{
      if (strstr("-f", argv[ArgNo]))
      {
         ForceDowngrade = true;
         ArgNo++;
      }
      else
      {
         SpsPrintf("[Error] Invalid parameters - wrong flag? multiple \"-f\" flag?\n");
			delete [] pTransData;
		   return -1;
      }
   }

   // print configuration
   if (false == ForceDowngrade)
   {
      SpsPrintf("[INFO] Downgrade disallowed (lack of \"-f\" switch).\n");
   }
   else
   {
      SpsPrintf("[INFO] Downgrade allowed (by \"-f\" switch).\n");
   }
	SpsPrintf("[INFO] ME addres=0x%02x IpmiWindowSize=%d BridgeChannelId=%d ImgBytesInFrame=%d\n"
             "File=%s\n",
	          SpsUpdateParameters.DestAddr, SpsUpdateParameters.WindowSize,
             SpsUpdateParameters.BrigdeChannelId, WriteLengthPackage, SpsUpdateParameters.InputFile);

   /*
	 * Actual Update process
	 */
	do
	{
		// Read the input file 
	
		if (_sopen_s(&handle, SpsUpdateParameters.InputFile, _O_RDONLY|_O_BINARY, _SH_DENYNO, 0) != 0)
		{
			SpsPrintf("[Error] Cannot open input file %s\n", SpsUpdateParameters.InputFile);
			delete [] pTransData;
			return -1;
		}
		FileSize = _filelength( handle );
		SpsPrintf("[OK] File opened %s (%ld bytes)\n", SpsUpdateParameters.InputFile, FileSize);

		try
		{
			buffer = new unsigned char[FileSize];
		}
		catch(...)
		{
		}

		if (buffer == NULL)
		{
			SpsPrintf("[Error] Cannot allocate file buffer\n");
			break;
		}
		Length = _read(handle,buffer,FileSize);
		if ((Length  == -1) || (Length != FileSize))
		{
			SpsPrintf("[Error] Fatal error while reading file\n");
			delete [] pTransData;
			return -1;
		}

		// Validate the image
		if (ValidateMeModule(buffer, Length, &PlatformId) != 0)
		{
			SpsPrintf("[Error] Fatal error while validating ME image\n");
			delete [] pTransData;
			return -1;
		}
      SpsPrintf("[OK] ME image verification\n");

      // Configure the interface
	   transport.ConfigureInterface(&SpsUpdateParameters);

		// Configure IPMI transport layer
		// Ping BMC - send 'GetDeviceId' to the BMC (to the bridge)
		if (SpsUpdateParameters.BridgeEnabled != 0)
		{
			// Configure transport layer 
			if (transport.StartInterface(SpsUpdateParameters.BridgeAddr) != 0)
			{
				SpsPrintf("[Error] Fatal error while starting communication interface\n");
				delete [] pTransData;
				return -1;
			}

			IpmiSetupTransaction(SpsUpdateParameters.WindowSize, SpsUpdateParameters.BridgeAddr, SpsUpdateParameters.SourceAddr, 0, 0, 0);
         Response = IpmiSendSimpleRequest(IPMI_NETFN_APPL, IPMI_CMD_GET_DEVICE_ID, NULL, 0);
			if (IPMI_CC_ACK != Response)
			{
				SpsPrintf("[Error] while sending 'Get Device Id' message to the bridge (0x%X)\n", Response);
				break;
			}
         SpsPrintf("[OK] Sending 'Get Device Id' to the bridge\n");

         // Setup bridging
			IpmiSetupTransaction(SpsUpdateParameters.WindowSize,
								SpsUpdateParameters.DestAddr,
								SpsUpdateParameters.SourceAddr,
								SpsUpdateParameters.BridgeAddr,
								1,
								SpsUpdateParameters.BrigdeChannelId);
		}
		else
		{
			// Configure transport layer 
			if(transport.StartInterface(SpsUpdateParameters.SourceAddr)!=0)
			{
				SpsPrintf( "[Error] Fatal error while starting communication interface\n");
				delete [] pTransData;
				return -1;
			}
			// No bridging
			IpmiSetupTransaction(SpsUpdateParameters.WindowSize, SpsUpdateParameters.DestAddr, SpsUpdateParameters.SourceAddr, 0, 0, 0);
		}

		Response = IpmiCleanResponses(&MsgCleaned);
		if (0 == Response)
		{
			SpsPrintf("[OK] Cleaning out pre-existing async messages: %d\n", MsgCleaned);
		}
		else
		{
			SpsPrintf("[Error] while cleaning out pre-existing async messages: %d\n", MsgCleaned);
			break;
		}

      // Ping ME - send message 'GetDeviceId' to ME (may be bridged)
      id = 0;
      Response = IpmiSendRequest(IPMI_NETFN_APPL, IPMI_CMD_GET_DEVICE_ID, NULL, 0, id);
		if (0 != Response)
		{
			SpsPrintf("[Error] while sending 'Get Device Id' message to ME (0x%X)\n", Response);
			break;
		}
      Response = IpmiGetResponse(&id, &DataLength, &pTransData);
		if (IPMI_CC_ACK != Response)
		{
			SpsPrintf("[Error] while getting 'Get Device Id' message from ME (0x%X)\n", Response);
			break;
		}
		// Check if response length is valid for GetDeviceID command
      if (sizeof(IPMI_GET_DEVICE_ID_RESP) != DataLength)
		{
			SpsPrintf("[ERROR] Invalid 'Get Device Id' message response length from ME (%d)\n", DataLength);
			break;
		}
		{
         /*
			// Check NM FW version (Boot Revision major byte)
			if (((IPMI_CMD_GET_DEVICE_ID_RESP*)pTransData)->FwIfaceVer == IPMI_IFACE_SPEC)
         */
         const IPMI_GET_DEVICE_ID_RESP * const pDevId = (IPMI_GET_DEVICE_ID_RESP*)pTransData;
			if (IPMI_IFACE_SPEC != pDevId->IpmiVersion)
			{
				SpsPrintf("[ERROR] Sending 'Get Device Id' message to ME - Node Manager with unknown IPMI version 0x%2X\n", pDevId->IpmiVersion);
				break;
			}

         unsigned char Major  = pDevId->FwMajorRev_InUPDATE;
         unsigned char Minor  = (pDevId->FwMinorRev >> 4) & 0x0F;
         unsigned char Hotfix = (pDevId->FwMinorRev >> 0) & 0x0F;
         unsigned char BuildH = (pDevId->BuildNo >>  4) & 0x0F;
         unsigned char BuildM = (pDevId->BuildNo >>  0) & 0x0F;
         unsigned char BuildL = (pDevId->BuildNo >> 12) & 0x0F;
         unsigned short ProdId = pDevId->ProdId;
         OrigImageNo = pDevId->ImageNo;
         SpsPrintf("[OK] Sending 'Get Device Id' message to ME - %d.%X.%X.%X%X%X (OPR%X)\n", Major, Minor, Hotfix, BuildH, BuildM, BuildL, OrigImageNo);

         // check platform compatibility
         if ((PROD_ID_BROMOLOW != ProdId) &&
             (PROD_ID_ROMLEY   != ProdId) &&
			 (PROD_ID_DENLOW   != ProdId) &&
			 (PROD_ID_GRANTLEY != ProdId))
         {
            SpsPrintf("[Error] Unsupported platform\n");
				break;
         }
         if ((PROD_ID_BROMOLOW == ProdId) &&
             (PLATFORM_ID_IN_MANIFEST_NAME_BROMOLOW != PlatformId))
         {
            SpsPrintf("[Error] Given image is not compatible with Bromolow platform\n");
				break;
         }
         if ((PROD_ID_ROMLEY == ProdId) &&
             (PLATFORM_ID_IN_MANIFEST_NAME_ROMLEY != PlatformId))
         {
            SpsPrintf("[Error] Given image is not compatible with Romley platform\n");
				break;
         }
		 if ((PROD_ID_DENLOW == ProdId) &&
             (PLATFORM_ID_IN_MANIFEST_NAME_DENLOW != PlatformId))
         {
            SpsPrintf("[Error] Given image is not compatible with Denlow platform\n");
				break;
         }
         if ((PROD_ID_GRANTLEY == ProdId) &&
             (PLATFORM_ID_IN_MANIFEST_NAME_GRANTLEY != PlatformId))
         {
            SpsPrintf("[Error] Given image is not compatible with Grantley platform\n");
				break;
         }

         // validate on chip ME version with file ME version and downgrade capabilities
         if (false == ValidateMeVersionWithFile(buffer, Length, Major, Minor, Hotfix, BuildH*100+BuildM*10+BuildL*1))
         {
            if (false == ForceDowngrade)
            {
               SpsPrintf("[ERROR] ME firmware version later than in the image file, downgrade available only with force flag\n");
				   break;
            }
            SpsPrintf("[OK] FORCING downgrade of the ME firmware\n");
         }
      }
		
		Response = IpmiSendSimpleRequest(IPMI_NETFN_OEMSPEC, IPMI_CMD_UPDATE_UPDATE_STATUS, NULL, 0);
		if (IPMI_CC_INVALID_CMD == Response)
      {
         SpsPrintf("[ERROR] Online Update not supported - single image configuration? Recovery booted?\n");
         break;
      }
		if (IPMI_CC_NOT_IN_PRESENT_STATE == Response)
      {
			SpsPrintf("[INFO] Waiting for ME to enable Online Update\n");
         Sleep(5000);
      }
      // If there is any other problem, subsequent commands will fail.
         

      /* Start Update */
		start = clock();

		// Prepare for Update
      Response = IpmiSendSimpleRequest(IPMI_NETFN_OEMSPEC, IPMI_CMD_UPDATE_PREPARE, NULL, 0);
      if (IPMI_CC_NODE_BUSY == Response)
      {
			SpsPrintf("[INFO] Node busy - repeating 'Online Update Prepare For Update'\n");
         Sleep(1000);
         Response = IpmiSendSimpleRequest(IPMI_NETFN_OEMSPEC, IPMI_CMD_UPDATE_PREPARE, NULL, 0);
      }
		if (IPMI_CC_ACK != Response)
		{
			SpsPrintf("[Error] while executing 'Online Update Prepare For Update' (0x%X)\n", Response);
			break;
		}
      SpsPrintf("[OK] Sending 'Online Update Prepare For Update'\n");

		// Open Area
		DataLength = 0;
		params[DataLength++] = 0x01; /* Operational code */
		params[DataLength++] = 0x00; /* Reserved */

      Response = IpmiSendSimpleRequest(IPMI_NETFN_OEMSPEC, IPMI_CMD_UPDATE_OPEN, (char*)params, DataLength);
		if (IPMI_CC_ACK != Response)
		{
			SpsPrintf("[Error] while executing 'Online Update Open Area' (0x%X)\n", Response);
			break;
		}
      SpsPrintf("[OK] Sending 'Online Update Open Area'\n");


		// Write Area
		LastSequenceOk		  = -1;
		ConsequitveErrorCount = 0;
      LastAckTimestamp    = clock();
		MsgSent				  = 0;
		MsgReceived			  = 0;
		LastValidSequence = (FileSize - 1) / WriteLengthPackage;

		for (Sequence = 0; LastValidSequence != LastSequenceOk; Sequence++)      
		{     
         //
         // Control of Sequence while waiting for responses for frames from last window
         // Length halted on 0
         //
         if (Sequence > LastValidSequence + 1)
         {
            Sequence = LastValidSequence + 1;            
         }

			CurrentFilePointer = Sequence * WriteLengthPackage;

			if(CurrentFilePointer + WriteLengthPackage < FileSize)
			{
				Length = WriteLengthPackage;
			}
			else
			{
				if(CurrentFilePointer < FileSize)
				{
					Length  = FileSize - CurrentFilePointer;
				}
				else
				{
					Length  = 0;
				}
			}
	
			if (Length > 0)
			{
				DataLength = 0;
				params[DataLength++] = (unsigned char)(Sequence & 0xff);				
				memcpy(&params[DataLength],&buffer[CurrentFilePointer],Length); 
	
				id = ((long long)MsgSent << 32) | Sequence;

				Response = IpmiSendRequest(IPMI_NETFN_OEMSPEC, IPMI_CMD_UPDATE_WRITE, (char *)params, DataLength + Length, id);
            if (0 != Response)
				{
					// Error while sending is fatal 
					SpsPrintf("\n[Error] while sending 'Online Update Write Area' (0x%X)\n", Response);
					goto OuterSpace;
				}
				MsgSent++;

				if (Sequence - LastSequenceOk < (int)SpsUpdateParameters.WindowSize)
				{
					continue;
				}
			}
	
			Response = IpmiGetResponse(&id, NULL, NULL);
			if (IPMI_CC_ACK != Response)
			{			
				MsgReceived++;	
				ConsequitveErrorCount++;
				if (ConsequitveErrorCount > MAX_CONSEQUITVE_ERROR_COUNT)
				{
					SpsPrintf("\n[Error] while executing 'Online Update Write Area' (0x%X)\n", Response);
					goto OuterSpace;
				}

				// If Node Busy don't wait for further responses
				if (IPMI_CC_NODE_BUSY == Response)
				{
					MsgReceived += IpmiDropRequest(&id);
               // Nody Busy happens when erasing flash with large page size (64kB on N25Q128 - may take up to 3s, 700ms typically)
               if ((clock() - LastAckTimestamp) < (int)(3.1*CLOCKS_PER_SEC))
               {
                  ConsequitveErrorCount--;
                  Sleep(100);
               }
				}

				/* Wait until queue of messages is cleared - read all responses before sending new requests */
				while (MsgReceived < MsgSent)
				{
					Response = IpmiGetResponse(&id, NULL, NULL);
					if (IPMI_CC_ACK == Response)
					{
						ReceivedSequence = (int)id;
						// Small chance
						if(LastSequenceOk < ReceivedSequence)
						{
							MsgReceived += IpmiRemoveAcknowledgeRequest(&id);
							LastSequenceOk = ReceivedSequence;
							ConsequitveErrorCount = 0;
						}
					}
					else
               if (IPMI_CC_NODE_BUSY == Response)
					{
						// If Node Busy don't wait for further responses
						MsgReceived += IpmiDropRequest(&id);			
					}
					MsgReceived++;
				}

				// Restart the sequence
				Sequence = LastSequenceOk;
            NofRepeatedWindows++;
				continue;
			}
			else
			{
				ReceivedSequence	    = (int)id;
            LastAckTimestamp      = clock();
				ConsequitveErrorCount = 0;
				MsgReceived++;
				if (LastSequenceOk < ReceivedSequence)
				{
					MsgReceived += IpmiRemoveAcknowledgeRequest(&id);
					LastSequenceOk = ReceivedSequence;				
				}
				else
				{
					/* 
						Ignore, most likely frames did not come in order 
						Works as long as ACK for higher sequence number means that there is guarantee for ACK for all sequence number below.
						True for Update but not in general case.
					*/
				}
				if (LastPercentage != (100 * Sequence)/(LastValidSequence))
				{
					LastPercentage = (100 * Sequence)/(LastValidSequence);
					SpsPrintf("[OK] Sending 'Online Update Write Area' - %d%%\r", LastPercentage);
				}

				if(LastValidSequence == LastSequenceOk)
				{
					// All done
					break;
				}
			}
		}

		SpsPrintf("\n");

		// Close Area
		DataLength           = 0;
		params[DataLength++] = (unsigned char)((FileSize>> 0) & 0xff);
		params[DataLength++] = (unsigned char)((FileSize>> 8) & 0xff);
		params[DataLength++] = (unsigned char)((FileSize>>16) & 0xff);
		params[DataLength++] = (unsigned char)((FileSize>>24) & 0xff);
		params[DataLength++] = Crc8(0, buffer, FileSize);
		params[DataLength++] = 0;
      stop = clock();
      Response = IpmiSendSimpleRequest(IPMI_NETFN_OEMSPEC, IPMI_CMD_UPDATE_CLOSE, (char*)params, DataLength);
      // 6s timeout is worst case page erase time (4s on 64kB page parts) plus CRC computing (including read of whole image from flash)
      while ((IPMI_CC_NODE_BUSY == Response) &&
             ((clock()-stop) <= 6*CLOCKS_PER_SEC))
      {
			SpsPrintf("[INFO] Node busy - repeating 'Online Update Close Area'\n");
         Sleep(300);
         Response = IpmiSendSimpleRequest(IPMI_NETFN_OEMSPEC, IPMI_CMD_UPDATE_CLOSE, (char*)params, DataLength);
      }
		if (IPMI_CC_ACK != Response)
		{
			SpsPrintf("[Error] while executing 'Online Update Close Area' (0x%X)\n", Response);
			break;
		}
      SpsPrintf("[OK] Sending 'Online Update Close Area'\n");

		// Update Area
		DataLength = 0;
		params[DataLength++] = 0x01 /*Normal update*/;
		params[DataLength++] = 0x00;
      Response = IpmiSendSimpleRequest(IPMI_NETFN_OEMSPEC, IPMI_CMD_UPDATE_REGISTER, (char*)params, DataLength);
		if (IPMI_CC_ACK != Response)
		{
			SpsPrintf("[Error] while executing 'Online Update Register Update' (0x%X)\n", Response);
			break;
		}
      SpsPrintf("[OK] Sending 'Online Update Register Update'\n");

		stop = clock();
		SpsPrintf("[OK] Update time: %d seconds\n",(stop - start)/CLOCKS_PER_SEC );

		// Cold reset
		Response = IpmiSendSimpleRequest(IPMI_NETFN_APPL, IPMI_CMD_COLD_RESET, NULL, 0);
		if (IPMI_CC_ACK != Response)
		{
			SpsPrintf("[Error] while sending 'Cold reset' message (0x%X)\n", Response);
			break;
		}
      SpsPrintf("[OK] Sending 'Cold reset'\n");

      // Ping ME - send message 'GetDeviceId' to ME
      SpsPrintf("[INFO] Waiting for ME to start before sending 'Get Device Id'...\n");
      Sleep(3000);
      id = 0;
      Response = IpmiSendRequest(IPMI_NETFN_APPL, IPMI_CMD_GET_DEVICE_ID, NULL, 0, id);
		if (0 != Response)
	   {
		   SpsPrintf("[Warning] Problem while sending 'Get Device Id' message to ME (0x%X)\n", Response);
	   }
      if (IpmiGetResponse(&id, &DataLength, &pTransData) != IPMI_CC_ACK)
		{
		   SpsPrintf("[INFO] Repeating 'Get Device Id'\n");
         Sleep(5000);
         id++;
         Response = IpmiSendRequest(IPMI_NETFN_APPL, IPMI_CMD_GET_DEVICE_ID, NULL, 0, id);
         if (0 != Response)
         {
            SpsPrintf("[Warning] Problem while sending 'Get Device Id' message to ME (0x%X)\n", Response);
            break;
         }
         Response = IpmiGetResponse(&id, &DataLength, &pTransData);
         if (IPMI_CC_ACK != Response)
         {
			   SpsPrintf("[Warning] Problem while getting 'Get Device Id' message from ME (0x%X)\n", Response);
            break;
         }
		}
      // Check if response length is valid for GetDeviceID command
      if (sizeof(IPMI_GET_DEVICE_ID_RESP) != DataLength)
		{
			SpsPrintf("[Warning] Invalid 'Get Device Id' message response length (%d)\n", DataLength);
			break;
		}
      {
         const IPMI_GET_DEVICE_ID_RESP * const pDevId = (IPMI_GET_DEVICE_ID_RESP*)pTransData;
         const int ImageNo = pDevId->ImageNo;
         SpsPrintf("[OK] Sending 'Get Device Id' message to ME - %d.%X.%X.%X%X%X (OPR%X)\n", pDevId->FwMajorRev_InUPDATE,
                   (pDevId->FwMinorRev >> 4) & 0x0F, (pDevId->FwMinorRev >> 0) & 0x0F,
                   (pDevId->BuildNo    >> 4) & 0x0F, (pDevId->BuildNo    >> 0) & 0x0F,
                   (pDevId->BuildNo    >>12) & 0x0F, pDevId->ImageNo);
         if (OrigImageNo == ImageNo)
         {
            SpsPrintf("[Warning] Problem with image switching\n");
         }
      }
   } while(0);


OuterSpace:


	delete [] pTransData;
	if(handle != -1)
	{
		_close( handle );
	}

	if(buffer != NULL)
	{
		delete buffer;
	}


	stop = clock();
	SpsPrintf("[INFO] Total time                         : %d seconds.\n", (stop - startWhole) / CLOCKS_PER_SEC);
	SpsPrintf("[INFO] Total number of frames to send     : %d.\n", LastValidSequence + 1);
	SpsPrintf("[INFO] Total number of frames sent        : %d.\n", MsgSent);
	SpsPrintf("[INFO] Total number of frames received    : %d.\n", MsgReceived);
	SpsPrintf("[INFO] Number of times window was repeated: %d.\n", NofRepeatedWindows);

   return 0;
} // main()


void SpsSpeedTest(void)
{
	int		ii;
	int		jj;
	long long	id;
	clock_t		start,stop;

	SpsPrintf("[INFO] Ping pong test\n)");

	for(ii = 0; ii < 20; ii++)
	{
		start = clock();

		if (IPMI_CC_ACK != IpmiSendSimpleRequest(IPMI_NETFN_APPL, IPMI_CMD_GET_DEVICE_ID, NULL, 0))
		{
			stop = clock();
			SpsPrintf("[INFO] Ping ( get device id) : %2.3f seconds\n", ((float)(stop - start)) / CLOCKS_PER_SEC);
		}
		else
		{
			SpsPrintf("[INFO] Ping ( get device id) failure\n)");
		}
	}
	SpsPrintf("[INFO] Burst test\n)");

#define BURST_CNT 3

	for(ii = 0; ii < 20; ii++)
	{
		SpsPrintf("[INFO] Burst test - sending %d 'Get Device Id messages'\n)",BURST_CNT);
		start = clock();

		for(jj = 0; jj < BURST_CNT;jj++)
		{
			IpmiSendRequest(IPMI_NETFN_APPL, IPMI_CMD_GET_DEVICE_ID, NULL, 0, ii*jj);
		}
					
		while(1)
		{
			if (IpmiGetResponse(&id, NULL, NULL) != 0)
			{
				stop = clock();
				SpsPrintf("[INFO] Response: %d seconds\n",((float)(stop - start))/CLOCKS_PER_SEC );
			}
			stop = clock();
			if(((float)(stop - start))/CLOCKS_PER_SEC > BURST_CNT * 2.0 /*s*/ )
			{
				break;
			}
		}
	}
}
