"""
REST Rule generator
"""

__author__ = 'VMware, Inc.'
__copyright__ = 'Copyright (c) 2015 VMware, Inc.  All rights reserved.'

import logging
import six

from werkzeug.routing import Rule

from com.vmware.vapi.metadata.metamodel_client import Type
from vmware.vapi.settings import config

logger = logging.getLogger(__name__)


class MappingRule(object):
    """
    Base class for all the mapping rules. This will contain
    the common helper functions for all the rules.
    """
    def __init__(self):
        """
        Initialize MappingRule
        """
        self._rest_prefix = config.cfg.get('rest', 'prefix')
        if not self._rest_prefix.endswith('/'):
            self._rest_prefix = '%s/' % self._rest_prefix

    def _generate_service_base_url(self, service_id):
        """
        Generate base url for a particular service

        :type  service_id: :class:`str`
        :param service_id: Identifier of the service.
        :rtype: :class:`str`
        :return: base url for all the HTTP REST URLs for a given service.
        """
        suffix = service_id.replace('_', '-').replace('.', '/').lower()
        return '%s%s' % (self._rest_prefix, suffix)

    @staticmethod
    def _get_id_suffix(operation_info):
        """
        Generate suffix using the ID parameters

        :type  operation_info: :class:`com.vmware.vapi.metadata.metamodel_client.OperationInfo`
        :param operation_info: Metamodel operation information
        :rtype: :class:`str` or `None`
        :return: string that can be used in the URL to represent an identifier,
            if there is no identifier, None is returned
        """
        for param_name, param_info in six.iteritems(operation_info):
            if param_info.type.category == Type.Category.BUILTIN:
                if param_info.type.builtin_type == Type.BuiltinType.ID:
                    # TODO: Handle composite identifiers
                    return '/<string:%s>' % param_name
        # No ID parameter
        return ''


class ListMappingRule(MappingRule):
    """
    Mapping rule that handles 'list' operations in the API
    and generates HTTP GET.

    Operations matched:
    list() -> GET /svc
    """
    def __init__(self):
        """
        Initialize ListMappingRule
        """
        MappingRule.__init__(self)

    @staticmethod
    def match(operation_id):
        """
        Check if the given operation matches the criteria for this
        mapping rule.

        :type  operation_id: :class:`str`
        :param operation_id: Operation identifier
        :rtype: :class:`bool`
        :return: True, if the given operation matches the criteria
            for this mapping rule, False, otherwise.
        """
        return (True if operation_id == 'list' else False)

    def url(self, service_id, operation_info):  # pylint: disable=W0613
        """
        Generate the URL for the given operation

        :type  service_id: :class:`str`
        :param service_id: Service identifier
        :type  operation_info: :class:`dict` of :class:`str` and
            :class:`com.vmware.vapi.metadata.metamodel_client.FieldInfo`
        :param operation_info: Metamodel metadata for the operation
        :rtype: :class:`tuple` of :class:`str` and :class:`str`
        :return: Tuple that has URL and the HTTP method for the
            given operation.
        """
        service_url = self._generate_service_base_url(service_id)
        return (service_url, 'GET')


class PostMappingRule(MappingRule):
    """
    Mapping rule that handles 'create' operations in the API
    and generates HTTP POST.

    Operations matched:
    create() -> POST /svc
    create(...) -> POST /svc + body
    """
    def __init__(self):
        """
        Initialize PostMappingRule
        """
        MappingRule.__init__(self)

    @staticmethod
    def match(operation_id):
        """
        Check if the given operation matches the criteria for this
        mapping rule.

        :type  operation_id: :class:`str`
        :param operation_id: Operation identifier
        :rtype: :class:`bool`
        :return: True, if the given operation matches the criteria
            for this mapping rule, False, otherwise.
        """
        return (True if operation_id == 'create' else False)

    def url(self, service_id, operation_info):  # pylint: disable=W0613
        """
        Generate the URL for the given operation

        :type  service_id: :class:`str`
        :param service_id: Service identifier
        :type  operation_info: :class:`dict` of :class:`str` and
            :class:`com.vmware.vapi.metadata.metamodel_client.FieldInfo`
        :param operation_info: Metamodel metadata for the operation
        :rtype: :class:`tuple` of :class:`str` and :class:`str`
        :return: Tuple that has URL and the HTTP method for the
            given operation.
        """
        service_url = self._generate_service_base_url(service_id)
        return (service_url, 'POST')


class DeleteMappingRule(MappingRule):
    """
    Mapping rule that handles 'delete' operations in the API
    and generates HTTP DELETE.

    Operations matched:
    delete(ID id) -> DELETE /svc/<id>
    """
    def __init__(self):
        """
        Initialize DeleteMappingRule
        """
        MappingRule.__init__(self)

    @staticmethod
    def match(operation_id):
        """
        Check if the given operation matches the criteria for this
        mapping rule.

        :type  operation_id: :class:`str`
        :param operation_id: Operation identifier
        :rtype: :class:`bool`
        :return: True, if the given operation matches the criteria
            for this mapping rule, False, otherwise.
        """
        return (True if operation_id == 'delete' else False)

    def url(self, service_id, operation_info):
        """
        Generate the URL for the given operation

        :type  service_id: :class:`str`
        :param service_id: Service identifier
        :type  operation_info: :class:`dict` of :class:`str` and
            :class:`com.vmware.vapi.metadata.metamodel_client.FieldInfo`
        :param operation_info: Metamodel metadata for the operation
        :rtype: :class:`tuple` of :class:`str` and :class:`str`
        :return: Tuple that has URL and the HTTP method for the
            given operation.
        """
        service_url = self._generate_service_base_url(service_id)
        id_suffix = self._get_id_suffix(operation_info)
        if id_suffix:
            return (service_url + id_suffix, 'DELETE')
        else:
            return (service_url, 'POST')


class GetMappingRule(MappingRule):
    """
    Mapping rule that handles 'get' operations in the API
    and generates HTTP GET.

    Operations matched:
    get(ID id) -> GET /svc/<id>
    """
    def __init__(self):
        """
        Initialize GetMappingRule
        """
        MappingRule.__init__(self)

    @staticmethod
    def match(operation_id):
        """
        Check if the given operation matches the criteria for this
        mapping rule.

        :type  operation_id: :class:`str`
        :param operation_id: Operation identifier
        :rtype: :class:`bool`
        :return: True, if the given operation matches the criteria
            for this mapping rule, False, otherwise.
        """
        return (True if operation_id == 'get' else False)

    def url(self, service_id, operation_info):
        """
        Generate the URL for the given operation

        :type  service_id: :class:`str`
        :param service_id: Service identifier
        :type  operation_info: :class:`dict` of :class:`str` and
            :class:`com.vmware.vapi.metadata.metamodel_client.FieldInfo`
        :param operation_info: Metamodel metadata for the operation
        :rtype: :class:`tuple` of :class:`str` and :class:`str`
        :return: Tuple that has URL and the HTTP method for the
            given operation.
        """
        service_url = self._generate_service_base_url(service_id)
        id_suffix = self._get_id_suffix(operation_info)
        if id_suffix:
            return (service_url + id_suffix, 'GET')
        else:
            return (service_url, 'POST')


class PatchMappingRule(MappingRule):
    """
    Mapping rule that handles 'update' operations in the API
    and generates HTTP PATCH.

    Operations matched:
    update(ID id) -> PATCH /svc/<id>
    """
    def __init__(self):
        """
        Initialize PatchMappingRule
        """
        MappingRule.__init__(self)

    @staticmethod
    def match(operation_id):
        """
        Check if the given operation matches the criteria for this
        mapping rule.

        :type  operation_id: :class:`str`
        :param operation_id: Operation identifier
        :rtype: :class:`bool`
        :return: True, if the given operation matches the criteria
            for this mapping rule, False, otherwise.
        """
        return (True if operation_id == 'update' else False)

    def url(self, service_id, operation_info):
        """
        Generate the URL for the given operation

        :type  service_id: :class:`str`
        :param service_id: Service identifier
        :type  operation_info: :class:`dict` of :class:`str` and
            :class:`com.vmware.vapi.metadata.metamodel_client.FieldInfo`
        :param operation_info: Metamodel metadata for the operation
        :rtype: :class:`tuple` of :class:`str` and :class:`str`
        :return: Tuple that has URL and the HTTP method for the
            given operation.
        """
        service_url = self._generate_service_base_url(service_id)
        id_suffix = self._get_id_suffix(operation_info)
        if id_suffix:
            return (service_url + id_suffix, 'PATCH')
        else:
            return (service_url, 'POST')


class PutMappingRule(MappingRule):
    """
    Mapping rule that handles 'set' operations in the API
    and generates HTTP PUT.

    Operations matched:
    set(ID id) -> PUT /svc/<id>
    """
    def __init__(self):
        """
        Initialize PutMappingRule
        """
        MappingRule.__init__(self)

    @staticmethod
    def match(operation_id):
        """
        Check if the given operation matches the criteria for this
        mapping rule.

        :type  operation_id: :class:`str`
        :param operation_id: Operation identifier
        :rtype: :class:`bool`
        :return: True, if the given operation matches the criteria
            for this mapping rule, False, otherwise.
        """
        return (True if operation_id == 'set' else False)

    def url(self, service_id, operation_info):
        """
        Generate the URL for the given operation

        :type  service_id: :class:`str`
        :param service_id: Service identifier
        :type  operation_info: :class:`dict` of :class:`str` and
            :class:`com.vmware.vapi.metadata.metamodel_client.FieldInfo`
        :param operation_info: Metamodel metadata for the operation
        :rtype: :class:`tuple` of :class:`str` and :class:`str`
        :return: Tuple that has URL and the HTTP method for the
            given operation.
        """
        service_url = self._generate_service_base_url(service_id)
        id_suffix = self._get_id_suffix(operation_info)
        if id_suffix:
            return (service_url + id_suffix, 'PUT')
        else:
            return (service_url, 'POST')


class PostActionMappingRule(MappingRule):
    """
    Mapping rule that handles non-crud operations in the API
    and generates HTTP POST.

    Operations matched:
    custom() -> POST /svc?~action=custom
    custom(ID id) -> POST /svc/<id>?~action=custom
    custom(...) -> POST /svc?~action=custom + body
    custom(ID id, ...) -> POST /svc/<id>?~action=custom + body
    """
    _crud_ops = ['create', 'get', 'list', 'update', 'set', 'delete']

    def __init__(self):
        """
        Initialize PostActionMappingRule
        """
        MappingRule.__init__(self)

    @staticmethod
    def match(operation_id):
        """
        Check if the given operation matches the criteria for this
        mapping rule.

        :type  operation_id: :class:`str`
        :param operation_id: Operation identifier
        :rtype: :class:`bool`
        :return: True, if the given operation matches the criteria
            for this mapping rule, False, otherwise.
        """
        return (True if operation_id not in PostActionMappingRule._crud_ops
                else False)

    def url(self, service_id, operation_info):
        """
        Generate the URL for the given operation

        :type  service_id: :class:`str`
        :param service_id: Service identifier
        :type  operation_info: :class:`dict` of :class:`str` and
            :class:`com.vmware.vapi.metadata.metamodel_client.FieldInfo`
        :param operation_info: Metamodel metadata for the operation
        :rtype: :class:`tuple` of :class:`str` and :class:`str`
        :return: Tuple that has URL and the HTTP method for the
            given operation.
        """
        service_url = self._generate_service_base_url(service_id)
        id_suffix = self._get_id_suffix(operation_info)
        return (service_url + id_suffix, 'POST')


class RoutingRuleGenerator(object):
    """
    Generate the routing rules based on vAPI metamodel metadata.
    """
    def __init__(self, metadata):
        """
        Initialize RoutingRuleGenerator

        :type  metadata: :class:`vmware.vapi.server.rest_handler.MetadataStore`
        :param metadata: Object that contains the metamodel metadata of
            all the services.
        """
        self._metadata = metadata
        self._mapping_rules = [
            ListMappingRule(),
            PostMappingRule(),
            DeleteMappingRule(),
            GetMappingRule(),
            PatchMappingRule(),
            PutMappingRule(),
            PostActionMappingRule(),
        ]

    @property
    def rest_rules(self):
        """
        HTTP REST rules

        :rtype: :class:`werkzeug.routing.Map`
        :return: Map instance that has all the available HTTP REST rules
        """
        rules = []
        for service_id, service_info in six.iteritems(self._metadata.service_map):
            for operation_id, operation_info in six.iteritems(service_info):
                for mapping_rule in self._mapping_rules:
                    if mapping_rule.match(operation_id):
                        service_url, http_method = mapping_rule.url(
                            service_id, operation_info)
                        rules.append(Rule(service_url,
                                          endpoint=service_id,
                                          methods=[http_method]))
        return rules
