#!/bin/bash

# ____________________________________________________________________________
#
# Copyright (c) 2016-2023 VMware, Inc. All rights reserved.
# ____________________________________________________________________________
#

# ____________________________________________________________________________
#
# Error handling
#
set -e

trap 'previous_command=$BASH_COMMAND; error_line=$LINENO' DEBUG
trap 'echo -e "
FAILURE: $previous_command command on line $error_line failed with exit code $?.
"' ERR

# ____________________________________________________________________________
#
# Executables
#
VECS_CLI=/usr/lib/vmware-vmafd/bin/vecs-cli
VMAFD_CLI=/usr/lib/vmware-vmafd/bin/vmafd-cli

# ____________________________________________________________________________
#
# VECS coordinates
#

# Trusted certificate store name
TRUSTED_STORE=TRUSTED_ROOTS

# Machine coordinates
MACHINE_STORE=MACHINE_SSL_CERT
MACHINE_ALIAS=__MACHINE_CERT

# vSphere Client coordinates
NGC_STORE="vsphere-webclient"
NGC_ALIAS="vsphere-webclient"

# ____________________________________________________________________________
#
# File names
#
KEY_STORE_FILE=store.jks
DS_CFG=ds.properties
FSS_FILE=vsphereFeatures.cfg
TECH_PREVIEW_FILE=techPreview.cfg
UI_FSS_FILE=uiFeatures.cfg
CFG_ZIP_FILE=ui-cfg.zip

# ____________________________________________________________________________
#
# File paths
#
FSS_CFG_DIR=$VMWARE_CFG_DIR/vsphereFeatures
FSS_SRC_PATH=$FSS_CFG_DIR/$FSS_FILE
TECH_PREVIEW_SRC_PATH=$FSS_CFG_DIR/$TECH_PREVIEW_FILE

CLIENT_CFG_DIR=$VMWARE_CFG_DIR/vsphere-ui
UI_FSS_SRC_PATH=$CLIENT_CFG_DIR/$UI_FSS_FILE
DS_PROP_SRC_PATH=$CLIENT_CFG_DIR/config/$DS_CFG
DS_PROP_SRC_LEGACY_PATH=$VMWARE_CFG_DIR/vsphere-client/config/$DS_CFG

DEST_DIR=ui-cfg
MACHINE_KEY_DEST_PATH=$DEST_DIR/machine-ssl-cert-private-key.pem
KEYSTORE_DEST_PATH=$DEST_DIR/$KEY_STORE_FILE
CLIENT_CERT_DEST_PATH=$DEST_DIR/vsphere-webclient-cert.txt
CLIENT_KEY_DEST_PATH=$DEST_DIR/vsphere-webclient-key.txt
CLIENT_P12_DEST_PATH=$DEST_DIR/vsphere-webclient.p12
FSS_DEST_PATH=$DEST_DIR/vsphereFeatures/$FSS_FILE
TECH_PREVIEW_DEST_PATH=$DEST_DIR/vsphereFeatures/$TECH_PREVIEW_FILE
CLIENT_CFG_DEST_PATH=$DEST_DIR/vsphere-client/webclient.properties
UI_FSS_DEST_PATH=$DEST_DIR/vsphere-ui/$UI_FSS_FILE
DS_CFG_DEST_PATH=$DEST_DIR/vsphere-ui/config/$DS_CFG

# ____________________________________________________________________________
#
# Dev-box constants
#

# Dev-box Java keystore password Password
KEYSTORE_PASS=vmw@re

# Keystore path
if [ -z "$KEYSTORE"  ] ; then
   KEYSTORE='C:\\ProgramData\\VMware\\vCenterServer\\cfg\\'$KEY_STORE_FILE
fi


# ____________________________________________________________________________
#
# Config generation
#


echo
echo -e "Creating $DEST_DIR/ ..."

rm -Rf $DEST_DIR
mkdir $DEST_DIR


# ____________________________________________________________________________
#

echo
echo -e "Generating $CLIENT_CERT_DEST_PATH ..."

# Extract the vSphere Client certificate from VECS.
$VECS_CLI entry getcert      \
          --store $NGC_STORE \
          --alias $NGC_ALIAS \
          > $CLIENT_CERT_DEST_PATH

# ____________________________________________________________________________
#

echo
echo -e "Generating $CLIENT_KEY_DEST_PATH ..."

# Extract the vSphere Client private key from VECS.
$VECS_CLI entry getkey       \
          --store $NGC_STORE \
          --alias $NGC_ALIAS \
          > $CLIENT_KEY_DEST_PATH

# ____________________________________________________________________________
#

echo
echo -e "Generating $MACHINE_KEY_DEST_PATH ..."

# Extract the machine private key from VECS. This is the key used by the
# reverse proxy. You can use it for debug purposes with Wireshark to sniff all
# server communication.
$VECS_CLI entry getkey           \
          --store $MACHINE_STORE \
          --alias $MACHINE_ALIAS \
          > $MACHINE_KEY_DEST_PATH

# ____________________________________________________________________________
#

echo
echo -e "Generating $CLIENT_P12_DEST_PATH ..."

openssl pkcs12                       \
        -export                      \
        -in $CLIENT_CERT_DEST_PATH   \
        -inkey $CLIENT_KEY_DEST_PATH \
        -name $NGC_ALIAS             \
        -out $CLIENT_P12_DEST_PATH   \
        -password pass:$KEYSTORE_PASS

# ____________________________________________________________________________
#

echo
echo -e "Generating $KEYSTORE_DEST_PATH ..."

keytool -importkeystore                    \
        -srckeystore $CLIENT_P12_DEST_PATH \
        -srcstoretype PKCS12               \
        -srcstorepass $KEYSTORE_PASS                \
        -destkeystore $KEYSTORE_DEST_PATH  \
        -deststorepass $KEYSTORE_PASS

# Extract the certificate from the TRUSTED_ROOTS VECS store.
$VECS_CLI entry list                    \
          --store $TRUSTED_STORE        \
   | awk '/Alias :/{ print $3 }' \
      | while read line;
         do
            $VECS_CLI entry getcert          \
                      --store $TRUSTED_STORE \
                      --alias $line          \
               | keytool -importcert                   \
                         -keystore $KEYSTORE_DEST_PATH \
                         -storepass $KEYSTORE_PASS              \
                         -trustcacerts                 \
                         -noprompt                     \
                         -alias $line
         done

# ____________________________________________________________________________
#

if [ -f "$FSS_SRC_PATH" ] ; then

   mkdir -p $FSS_DEST_PATH
   rmdir $FSS_DEST_PATH

   echo
   echo -e "Generating $FSS_DEST_PATH ..."
   cp $FSS_SRC_PATH $FSS_DEST_PATH
fi

# ____________________________________________________________________________
#

if [ -f "$TECH_PREVIEW_SRC_PATH" ] ; then

   mkdir -p $TECH_PREVIEW_DEST_PATH
   rmdir $TECH_PREVIEW_DEST_PATH

   echo
   echo -e "Generating $TECH_PREVIEW_DEST_PATH ..."
   cp $TECH_PREVIEW_SRC_PATH $TECH_PREVIEW_DEST_PATH
fi

# ____________________________________________________________________________
#

if [ -f "$UI_FSS_SRC_PATH" ] ; then

   mkdir -p $UI_FSS_DEST_PATH
   rmdir $UI_FSS_DEST_PATH

   echo
   echo -e "Generating $UI_FSS_DEST_PATH ..."

   cp $UI_FSS_SRC_PATH $UI_FSS_DEST_PATH
fi

# ____________________________________________________________________________
#

echo
echo -e "Generating $CLIENT_CFG_DEST_PATH ..."

mkdir -p $CLIENT_CFG_DEST_PATH
rmdir $CLIENT_CFG_DEST_PATH

DNSNAME=`keytool -printcert -sslserver localhost \
            | awk '/DNSName:/ { print $2 }'`
if [ -z "$DNSNAME" ]
then
   echo ERROR: vCSA DNS name not found. You will need to the value for fix cm.url in $PROPFILE. 1>&2
   DNSNAME=MISSING_DNS_NAME
fi

# Extract the URL to the lookup service.
VMAFD_LS_URL=`$VMAFD_CLI get-ls-location --server-name localhost`
if [ -z "$VMAFD_LS_URL" ]
then
   echo ERROR: Lookup Service URL not found. You will need to fix ls.url in $PROPFILE. 1>&2
   VMAFD_LS_URL=MISSING_LS_URL
fi

VMAFD_NODE_ID=` \
   ${VMAFD_CLI} get-ldu --server-name localhost | tr -d '\n'`

VMAFD_MACHINE_ID=` \
   ${VMAFD_CLI} get-machine-id --server-name localhost | tr -d '\n'`

VMAFD_LOCAL_SSO_DOMAIN_ID=` \
   ${VMAFD_CLI} get-domain-id --server-name localhost | tr -d '\n'`
if [ $? -ne 0 ] ; then
   echo -e "vmafd-cli does not support hybrid linked mode. Skipping"
   VMAFD_LOCAL_SSO_DOMAIN_ID=""
   VMAFD_LOCAL_SSO_DOMAIN_NAME=""
else
   VMAFD_LOCAL_SSO_DOMAIN_NAME=` \
      ${VMAFD_CLI} get-domain-name --server-name localhost | tr -d '\n'`
fi



VCENTER_HOST_ID=`cat ${VMWARE_CFG_DIR}/install-defaults/vmdir.ldu-guid`


cat << EOF > $CLIENT_CFG_DEST_PATH
# ____________________________________________________________________________
#
# Generated webclient.properties for dev environments.
#
# Copy this file to the right location along with the generated
# $KEY_STORE_FILE (see path below).
#
# The keystore.jks.*, ls.url, and cm.url settings allow you to connect your
# local vSphere Client server to your vCenter Server Appliance.
# ____________________________________________________________________________
#

# ____________________________________________________________________________
#
# BEGIN: DO NOT EDIT THESE SETTINGS

# Lookup Service URL with the DNS name of the VCSA deployment.
ls.url=$VMAFD_LS_URL

# ComponentManager URL with the DNS name of your VCSA host.
cm.url=https://$DNSNAME/cm/sdk/

afd.disabled=true

localhost.id=$VMAFD_MACHINE_ID
localhost.ldu-guid=$VMAFD_NODE_ID
sso.localDomainId=$VMAFD_LOCAL_SSO_DOMAIN_ID
sso.localDomainName=$VMAFD_LOCAL_SSO_DOMAIN_NAME

large.inventory.mode=true
show.allusers.tasks=true
dataservice.mode=IS
use.unstable.vmodl=false
vapi.hostid=$VCENTER_HOST_ID
browser.context.menu=true

# Allow the redeployment of bundles through the server/pickup directory.
local.development=true
pickup.deployer=true

# END: DO NOT EDIT THESE SETTINGS
# ____________________________________________________________________________
#


# Set the correct value of keystore.jks.path based on your dev OS.
# Mac OS: /var/lib/vmware/vsphere-client/$KEY_STORE_FILE
# Windows: C:/ProgramData/VMware/vCenterServer/cfg/$KEY_STORE_FILE
keystore.jks.path=$KEYSTORE

# Password for the local JKS keystore. Do not change.
keystore.jks.password=$KEYSTORE_PASS


# ____________________________________________________________________________
#
# Additional flags here...
#

EOF

# ____________________________________________________________________________
#

echo
echo -e "Generating $DS_CFG_DEST_PATH ..."
echo

mkdir -p $DS_CFG_DEST_PATH
rmdir $DS_CFG_DEST_PATH

if [ -f $DS_PROP_SRC_PATH ] ; then
   cp $DS_PROP_SRC_PATH $DS_CFG_DEST_PATH
else
   cp $DS_PROP_SRC_LEGACY_PATH $DS_CFG_DEST_PATH
fi

cat << EOF >> $DS_CFG_DEST_PATH

solutionUser.keyStorePath=$KEYSTORE
solutionUser.keyStorePassword=$KEYSTORE_PASS
lookupService=$VMAFD_LS_URL
sidecar.useForOutgoingConnections=false
mode=dev

EOF

echo
echo -e "Zipping configuration to $CFG_ZIP_FILE ..."
echo

pushd $DEST_DIR
   zip -r $CFG_ZIP_FILE .
   mv -f $CFG_ZIP_FILE ..
popd

# ____________________________________________________________________________
#

echo
echo -e "Cleaning up $DEST_DIR/ ..."
echo

rm -Rf $DEST_DIR

# ____________________________________________________________________________
#

echo
echo -e "SUCCESS"
echo
