/*
 * ******************************************************
 * Copyright (c) VMware, Inc. 2010.  All Rights Reserved.
 * ******************************************************
 *
 * DISCLAIMER. THIS PROGRAM IS PROVIDED TO YOU "AS IS" WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, WHETHER ORAL OR WRITTEN,
 * EXPRESS OR IMPLIED. THE AUTHOR SPECIFICALLY DISCLAIMS ANY IMPLIED
 * WARRANTIES OR CONDITIONS OF MERCHANTABILITY, SATISFACTORY QUALITY,
 * NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE.
 */

using System;
using System.Collections.Generic;
using System.Text;
using System.Web.Services.Protocols;
using AppUtil;
using Vim25Api;

namespace LicenseManager
{
    public class LicenseManager
    {
        private static AppUtil.AppUtil cb = null;
        private static ManagedObjectReference licMgr = null;
        private static ManagedObjectReference licAssignmentMgr = null;

        private void useLicenseManager()
        {
            String action = cb.get_option("action");

            if (action.Equals("browse"))
            {
                Console.WriteLine("License Information");
                var licenses = GetAssignedLicenses();
                DisplayLicensesInfo(licenses);
            }
            else if (action.Equals("setkey"))
            {
                Console.WriteLine("Set License Key");
                SetLicenseKey();
            }
            else if (action.Equals("featureinfo"))
            {
                Console.WriteLine("License Features");
                var licenses = GetAssignedLicenses();
                DisplayLicensesFeatures(licenses);
            }
            else
            {
                Console.WriteLine(
                    "Valid action values are browse, setkey, or featureinfo.");
            }
        }

        private LicenseAssignmentManagerLicenseAssignment[] GetAssignedLicenses()
        {
            var licenseAssignments = cb.getConnection()._service.QueryAssignedLicenses(
                new QueryAssignedLicensesRequest(licAssignmentMgr, null)).returnval;
            if (licenseAssignments == null)
            {
                throw new Exception(
                    "Could not get license information from server.");
            }
            return licenseAssignments;
        }

        public void DisplayLicensesInfo(
            LicenseAssignmentManagerLicenseAssignment[] licenseAssignments)
        {
            foreach (var licenseAssignment in licenseAssignments)
            {
                Console.WriteLine(
                    "\n Name of the license: " + licenseAssignment.assignedLicense.name +
                    "\n License Key:  " + licenseAssignment.assignedLicense.licenseKey +
                    "\n Edition Key: " + licenseAssignment.assignedLicense.editionKey +
                    "\n EntityID: " + licenseAssignment.entityId + "\n\n");
            }
        }

        private void SetLicenseKey()
        {
            var licenseKey = cb.get_option("key");
            var serviceContent = cb.getConnection().ServiceContent;
            var VimPortType = cb.getConnection().Service;
            var apitype = serviceContent.about.apiType;

            if (apitype.Equals("VirtualCenter",
                StringComparison.CurrentCultureIgnoreCase))
            {
                String entity = serviceContent.about.instanceUuid;
                VimPortType.UpdateAssignedLicense(licAssignmentMgr,
                    entity, licenseKey, null);
                Console.WriteLine("License key set for VC server");
            }
            else if (apitype.Equals("HostAgent"))
            {
                VimPortType.DecodeLicense(licMgr, licenseKey);
                VimPortType.UpdateLicense(new UpdateLicenseRequest(licMgr, licenseKey, null));
                Console.WriteLine("License key set for ESX server");
            }
        }

        private void DisplayLicensesFeatures(
            LicenseAssignmentManagerLicenseAssignment[] licenseAssignments)
        {
            string featureName = cb.get_option("feature");
            var licenseFeatures = new Dictionary<string, List<KeyValue>>();
            bool found = false;

            // search for user provided feature name
            foreach (var licenseAssignment in licenseAssignments)
            {
                if (licenseAssignment.assignedLicense != null &&
                    licenseAssignment.assignedLicense.properties != null)
                {
                    var licFeatures = new List<KeyValue>();
                    foreach (var property in licenseAssignment.assignedLicense.properties)
                    {
                        if (property.key.Equals("feature",
                            StringComparison.CurrentCultureIgnoreCase))
                        {
                            var feature = property.value as Vim25Api.KeyValue;
                            if (feature == null)
                            {
                                throw new Exception(
                                    "Could not read the current feature of current license.");
                            }
                            if (feature.key.Equals(featureName,
                                StringComparison.CurrentCultureIgnoreCase))
                            {
                                found = true;
                                var featureInfo = string.Format(
                                    "License Name: {0}, Feature Name: {1}, Description: {2}",
                                    licenseAssignment.assignedLicense.name, feature.key, feature.value);
                                Console.WriteLine(featureInfo);
                            }
                            licFeatures.Add(feature);
                        }
                    }
                    licenseFeatures.Add(licenseAssignment.assignedLicense.name,
                                licFeatures);
                }
            }

            // if not found, display available features
            if (!found)
            {
                Console.WriteLine("Could not find feature " + featureName);
                if (licenseFeatures.Keys.Count > 0)
                {
                    Console.WriteLine("Available features are: ");
                    foreach (var licenseFeature in licenseFeatures)
                    {
                        foreach (var kv in licenseFeature.Value)
                        {
                            var featureInfo = string.Format(
                                "License Name: {0}, Feature Name: {1}, Description: {2}",
                                        licenseFeature.Key, kv.key, kv.value);
                            Console.WriteLine(featureInfo);
                        }
                    }
                }
            }
        }

        private Boolean customValidation()
        {
            Boolean flag = true;
            String action = cb.get_option("action");
            if (action.Equals("setkey"))
            {
                if (!cb.option_is_set("key"))
                {
                    Console.WriteLine("For action setkey --key" +
                                       "is mandatory argument");
                    flag = false;
                }
            }
            else if (action.Equals("featureinfo"))
            {
                if (!cb.option_is_set("feature"))
                {
                    Console.WriteLine("For action featureinfo --feature is " +
                                       "mandatory argument");
                    flag = false;
                }
            }
            return flag;
        }

        private static OptionSpec[] constructOptions()
        {
            OptionSpec[] useroptions = new OptionSpec[3];
            useroptions[0] = new OptionSpec("action", "String", 1,
                                            "[browse|setkey|featureinfo]"
                                            , null);
            useroptions[1] = new OptionSpec("key", "String", 0,
                                            "License key string",
                                            null);
            useroptions[2] = new OptionSpec("feature", "String", 0,
                                            "Name of the feature",
                                            null);
            return useroptions;
        }

        public static void Main(String[] args)
        {
            LicenseManager app = new LicenseManager();
            cb = AppUtil.AppUtil.initialize("LicenseManager", LicenseManager.constructOptions(), args);
            Boolean valid = app.customValidation();
            if (valid)
            {
                cb.connect();
                licMgr = cb.getConnection()._sic.licenseManager;
                licAssignmentMgr = cb.getServiceUtil().GetMoRefProp(licMgr,
                    "licenseAssignmentManager");
                app.useLicenseManager();
                cb.disConnect();
            }
            Console.WriteLine("Press enter to exit: ");
            Console.Read();
        }
    }
}
