﻿/*
 * ******************************************************
 * Copyright (c) VMware, Inc. 2010.  All Rights Reserved.
 * ******************************************************
 *
 * DISCLAIMER. THIS PROGRAM IS PROVIDED TO YOU "AS IS" WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, WHETHER ORAL OR WRITTEN,
 * EXPRESS OR IMPLIED. THE AUTHOR SPECIFICALLY DISCLAIMS ANY IMPLIED
 * WARRANTIES OR CONDITIONS OF MERCHANTABILITY, SATISFACTORY QUALITY,
 * NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE.
 */

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Vim25Api;
using AppUtil;
namespace XVMotion
{
    ///<summary>
    ///This sample is used to migrate/relocate VM to another host and datastore
    ///using the drs placement recommendations
    ///</summary>
    ///<param name="vmname">Required: Name of the virtual machine</param>
    ///<param name="clustername">Required: Name of the target cluster</param>
    ///<remarks>
    ///Relocate VM
    ///--url [webserviceurl]
    ///--username [username] --password [password] --vmname [myVM] --clustername [cluster]
    ///</remarks>
    class XVMotion
    {
        private static AppUtil.AppUtil cb = null;
        public void PlaceVM()
        {
            String vmname = cb.get_option("vmname");
            String clustername = cb.get_option("clustername");
            PlacementResult placementResult;
            PlacementSpec placementSpec = new PlacementSpec();
            ManagedObjectReference vmMOR = null;
            ManagedObjectReference clusterMOR = GetMOR(clustername, "ClusterComputeResource", null);
            if (clusterMOR != null)
            {
                vmMOR = GetMOR(vmname, "VirtualMachine", null);
                if (vmMOR != null)
                {
                    placementSpec.vm = vmMOR;
                    placementResult = cb._connection._service.PlaceVm(clusterMOR, placementSpec);
                    PlacementAction placementAction = GetPlacementAction(placementResult);
                    if (placementAction != null)
                    {
                        VirtualMachineRelocateSpec relocateSpec =
                   placementAction.relocateSpec;
                        //API to perform XVMotion using DRS recommendations
                        ManagedObjectReference taskMOR =
                       cb._connection._service.RelocateVM_Task(vmMOR, relocateSpec, VirtualMachineMovePriority.defaultPriority);
                        String res = cb.getServiceUtil().WaitForTask(taskMOR);
                        if (res.Equals("sucess"))
                        {
                            Console.WriteLine("Relocation done successfully ");
                        }
                        else
                        {
                            Console.WriteLine("Error::  Relocation failed");
                        }
                    }
                    else
                    {
                        Console.WriteLine("Recommendations are not correct");
                    }
                }
                else
                {
                    Console.WriteLine("VM  " + vmname + " not found");
                }
            }
            else
            {
                Console.WriteLine("Cluster " + clustername + " not found");
            }
        }
        /// <summary>
        /// This method is to return  the first valid PlacementAction out of the DRS recommendations.
        /// </summary>
        /// <param name="placementResult">PlacementResult</param>
        /// <returns>PlacementAction</returns>
        public PlacementAction GetPlacementAction(PlacementResult placementResult)
        {
            ClusterRecommendation[] recommendatons = placementResult.recommendations;
            PlacementAction placementAction = null;
            int size = recommendatons.Length;
            bool actionOk = false;
            if (size > 0)
            {
                Console.WriteLine("Total number of recommendations are " + size);
                Console.WriteLine("Processing the xvmotion placement recommendations out of the recommendations received");
                foreach (ClusterRecommendation recommendation in recommendatons)
                {
                    if (recommendation.reason.Equals("xvmotionPlacement"))
                    {
                        ClusterAction[] actions = recommendation.action;
                        foreach (ClusterAction action in actions)
                        {
                            if (action is PlacementAction)
                            {
                                placementAction = (PlacementAction)action;
                                break;
                            }
                        }
                        if (placementAction != null)
                        {
                            if (placementAction.vm == null || placementAction.targetHost == null)
                            {
                                Console.WriteLine("Placement Action doesnot have a vm and targethost set");
                            }
                            else
                            {
                                if (placementAction.relocateSpec != null)
                                {
                                    actionOk = CheckRelocateSpec(placementAction.relocateSpec);
                                    if (actionOk)
                                        break;
                                    else
                                        placementAction = null;
                                }
                            }
                        }
                        else
                        {
                            Console.WriteLine("Recommendation doesnot have a placement action");
                        }
                    }
                }
            }
            else
            {
                Console.WriteLine("No recommendations by DRS");
            }
            return placementAction;
        }


        /// <summary>
        /// This method is to validate the RelocateSpec.
        /// </summary>
        /// <param name="relocateSpec">VirtualMachineRelocateSpec</param>
        /// <returns>bool</returns>
        bool CheckRelocateSpec(VirtualMachineRelocateSpec relocateSpec)
        {
            bool check = false;

            if (relocateSpec.host != null)
            {
                if (relocateSpec.pool != null)
                {
                    if (relocateSpec.datastore != null)
                    {
                        check = true;
                    }
                    else
                    {
                        Console.WriteLine("RelocateSpec doesnot have a datastore");
                    }
                }
                else
                {
                    Console.WriteLine("RelocateSpec doesnot have a resource pool");
                }
            }
            else
            {
                Console.WriteLine("RelocateSpec doesnot have a host");
            }
            return check;
        }
        private ManagedObjectReference GetMOR(String name, String type, ManagedObjectReference root)
        {
            ManagedObjectReference nameMOR
            = (ManagedObjectReference)cb.getServiceUtil().GetDecendentMoRef(root, type, name);
            if (nameMOR == null)
            {
                Console.WriteLine("Error:: " + name + " not found");
                return null;
            }
            else
            {
                return nameMOR;
            }
        }
        public static OptionSpec[] constructOptions()
        {
            OptionSpec[] useroptions = new OptionSpec[2];
            useroptions[0] = new OptionSpec("vmname", "String", 1,
                                            "Name of the virtual machine to be migrated"
                                            , null);
            useroptions[1] = new OptionSpec("clustername", "String", 1,
                                            "Target cluster name",
                                            null);
            return useroptions;
        }

        static void Main(string[] args)
        {
            XVMotion app = new XVMotion();
            cb = AppUtil.AppUtil.initialize("XVMotion", XVMotion.constructOptions(), args);
            try
            {
                cb.connect();
                app.PlaceVM();
                cb.disConnect();
            }
            catch (Exception e)
            {
				Console.WriteLine(e.Message.ToString());
            }
            Console.WriteLine("Press any key to exit: ");
            Console.Read();
        }
    }
}