/*
 * ******************************************************
 * Copyright VMware, Inc. 2019.  All Rights Reserved.
 * ******************************************************
 *
 * DISCLAIMER. THIS PROGRAM IS PROVIDED TO YOU "AS IS" WITHOUT
 * WARRANTIES OR CONDITIONS # OF ANY KIND, WHETHER ORAL OR WRITTEN,
 * EXPRESS OR IMPLIED. THE AUTHOR SPECIFICALLY # DISCLAIMS ANY IMPLIED
 * WARRANTIES OR CONDITIONS OF MERCHANTABILITY, SATISFACTORY # QUALITY,
 * NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE.
 */
package com.vmware.fcd;

import java.util.Arrays;

import com.vmware.common.annotations.Action;
import com.vmware.common.annotations.Option;
import com.vmware.common.annotations.Sample;
import com.vmware.connection.ConnectedVimServiceBase;
import com.vmware.fcd.helpers.FcdHelper;
import com.vmware.fcd.helpers.FcdVslmHelper;
import com.vmware.vslm.VslmPortType;

/**
 * <pre>
 * FcdControlFlagsOperations
 *
 * This sample executes control flag related
 * operations on a given VStorageObject from vslm:
 *
 * 1. Set control flags on VStorageObject.
 * 2. Clear control flags on VStorageObject.
 *
 * Pre-requisite :        1. Existing VStorageObject ID
 * <b>Parameters:</b>
 * url                    [required] : url of the web service
 * username               [required] : username for the authentication
 * password               [required] : password for the authentication
 * vstorageobjectid       [required] : Uuid of the vstorageobject
 * controlflags           [required] : Set control flags on VStorageObject.
 *
 * <b>Command Line:</b>
 * run.bat com.vmware.fcd.FcdControlFlagsOperations --url [webserviceurl]
 * --username [username] --password [password]
 * --vstorageobjectid [vstorageobjectid] --controlflags [controlflags]
 * </pre>
 */

@Sample(name = "fcd-controlflagsoperations", description = "This sample"
		+ " sets and clears control flags on VStorageObject from vslm.")
public class FcdControlFlagsOperations extends ConnectedVimServiceBase {

	private String vStorageObjectId;
	private String controlFlags;

	/**
	 * @param vStorageObjectId
	 *            the vStorageObjectId to set
	 */
	@Option(name = "vstorageobjectid", required = true, description = "Uuid of the vstorageobject.")
	public void setVStorageObjectId(String vStorageObjectId) {
		this.vStorageObjectId = vStorageObjectId;
	}

	/**
	 * @param controlFlags
	 *            the controlFlags to set
	 */
	@Option(name = "controlflags", required = true, description = "Control flags enum array to be set "
			+ "on the VStorageObject.")
	public void setControlFlags(String controlFlags) {
		this.controlFlags = controlFlags;
	}

	/**
	 * This method : Sets and clears control flags on VStorageObject from vslm.
	 *
	 * @throws Exception
	 */
	void controlFlagOperations() throws Exception {
		FcdVslmHelper vslmHelper = new FcdVslmHelper(connection);
		VslmPortType vslmPort = vslmHelper.getVslmPort();

		// Retrieve a vStorageObject:
		System.out.println("Operation: Setting control flags on VStorageObject from vslm : " + controlFlags);
		// Set control flags on VStorageObject.
		vslmPort.vslmSetVStorageObjectControlFlags(vslmHelper.getVStorageObjMgr(), FcdHelper.makeId(vStorageObjectId),
				Arrays.asList(controlFlags));
		System.out.printf("Control flag [flag = %s ] set for vStorageObject :" + " %n [ Uuid = %s ]%n from vslm.",
				controlFlags, vStorageObjectId);

		// Clear control flags on VStorageObject.
		System.out.println("Operation: Clearing control flags on VStorageObject from vslm.");
		vslmPort.vslmClearVStorageObjectControlFlags(vslmHelper.getVStorageObjMgr(), FcdHelper.makeId(vStorageObjectId),
				Arrays.asList(controlFlags));
		System.out.printf("Control flag [flag = %s] cleared from vStorageObject :" + " %n [ Uuid = %s ] %n from vslm.",
				controlFlags, vStorageObjectId);
	}

	@Action
	public void run() throws Exception {
		System.out.println("Invoking control flag operations on VStorageObject from VSLM ::");
		controlFlagOperations();
	}
}
