package com.vmware.fcd;
/*
 * ******************************************************

 * Copyright VMware, Inc. 2019.  All Rights Reserved.
 * ******************************************************
 *
 * DISCLAIMER. THIS PROGRAM IS PROVIDED TO YOU "AS IS" WITHOUT
 * WARRANTIES OR CONDITIONS # OF ANY KIND, WHETHER ORAL OR WRITTEN,
 * EXPRESS OR IMPLIED. THE AUTHOR SPECIFICALLY # DISCLAIMS ANY IMPLIED
 * WARRANTIES OR CONDITIONS OF MERCHANTABILITY, SATISFACTORY # QUALITY,
 * NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE.
 */

import java.util.HashMap;
import java.util.Map;

import com.vmware.common.annotations.Action;
import com.vmware.common.annotations.Before;
import com.vmware.common.annotations.Option;
import com.vmware.common.annotations.Sample;
import com.vmware.connection.ConnectedVimServiceBase;
import com.vmware.fcd.helpers.FcdUtil;
import com.vmware.fcd.helpers.FcdUtil.DiskProvisioningTypes;
import com.vmware.fcd.helpers.FcdVslmHelper;
import com.vmware.vim25.InvalidCollectorVersionFaultMsg;
import com.vmware.vim25.InvalidPropertyFaultMsg;
import com.vmware.vim25.LocalizedMethodFault;
import com.vmware.vim25.ManagedObjectReference;
import com.vmware.vim25.RuntimeFaultFaultMsg;
import com.vmware.vim25.TaskInfoState;
import com.vmware.vim25.VStorageObject;
import com.vmware.vim25.VslmCreateSpec;
import com.vmware.vslm.VslmPortType;
import com.vmware.vslm.VslmTaskInfo;

/**
 * <pre>
 * FcdCreate
 *
 * This sample creates a virtual storage object
 *
 * <b>Parameters:</b>
 * url                    [required] : url of the web service
 * username               [required] : username for the authentication
 * password               [required] : password for the authentication
 * vstorageobjectname     [required] : Name of the disk
 * datastorename          [required] : Name of the datastore
 * vstorageobjectsizeinMB [required] : Size of the disk(in MB)
 * provisioningtype       [optional] : Type of provisioning for the disk
 *                                     [thin | eagerZeroedThick |
 *                                     lazyZeroedThick | virtualMode |
 *                                     physicalMode]
 * devicename             [optional] : Canonical name of the LUN to use for disk types
 *
 * <b>Command Line:</b>
 * Create a disk (virtual storage object) given vstorageobject name, datastore
 * name and vstorageobject size
 * run.bat com.vmware.fcd.FcdCreateFromVslm --url [webserviceurl]
 * --username [username] --password [password]
 * --vstorageobjectname [vstorageobjectname] --datastorename [datastorename]
 * --vstorageobject [vstorageobject]
 *
 * Create a disk (virtual storage object) given vstorageobject name, datastore
 * names, vstorageobject size and provisioning type of vstorageobject
 * run.bat com.vmware.fcd.FcdCreateFromVslm --url [webserviceurl]
 * --username [username] --password [password]
 * --vstorageobjectname [vstorageobjectname] --datastorename [datastorename]
 * --vstorageobject [vstorageobject] --provisioningtype [provisioningtype]
 * --devicename [devicename]
 *
 * Ex: devicename :: vmhba0:0:0:0
 * </pre>
 */
@Sample(name = "fcd-create", description = "This sample creates a virtual storage object(FCD) from vslm endpoint")
public class FcdCreateFromVslm extends ConnectedVimServiceBase {

	private String vStorageObjectName;
	private String datastoreName;
	private long vStorageObjectSizeInMB;
	private String provisioningType = "thin"; // Default.
	private String deviceName;
	private final Map<String, DiskProvisioningTypes> provisioningTypeHashMap = new HashMap<String, DiskProvisioningTypes>();

	/**
	 * @param vStorageObjectName the vStorageObjectName to set
	 */
	@Option(name = "vstorageobjectname", required = true, description = "Name of the Disk.")
	public void setVStorageObjectName(String vStorageObjectName) {
		this.vStorageObjectName = vStorageObjectName;
	}

	/**
	 * @param datastoreName the datastoreName to set
	 */
	@Option(name = "datastorename", required = true, description = "Name of datastore.")
	public void setDatastoreName(String datastoreName) {
		this.datastoreName = datastoreName;
	}

	/**
	 * @param vStorageObjectSizeInMB the vStorageObjectSizeInMB to set
	 */
	@Option(name = "vstorageobjectsizeinMB", required = true, description = "Size of the Disk (in MB)")
	public void setVStorageObjectSize(long vStorageObjectSizeInMB) {
		this.vStorageObjectSizeInMB = vStorageObjectSizeInMB;
	}

	/**
	 * @param provisioningType the provisioningType to set
	 */
	@Option(name = "provisioningtype", required = false, description = "Provisioning Type of the Disk.\n [thin | "
			+ "eagerZeroedThick | lazyZeroedThick | virtualMode | physicalMode]")
	public void setProvisioningType(String provisioningType) {
		this.provisioningType = provisioningType;
	}

	/**
	 * @param deviceName the deviceName to set
	 */
	@Option(name = "devicename", required = false, description = "Canonical name of the LUN to use for RDM"
			+ " provisioning type.")
	public void setDeviceName(String deviceName) {
		this.deviceName = deviceName;
	}

	@Before
	public void init() {
		provisioningTypeHashMap.put("thin", DiskProvisioningTypes.THIN);
		provisioningTypeHashMap.put("eagerzeroedthick", DiskProvisioningTypes.EAGER_ZEROED_THICK);
		provisioningTypeHashMap.put("lazyzeroedthick", DiskProvisioningTypes.LAZY_ZEROED_THICK);
		provisioningTypeHashMap.put("virtualmode", DiskProvisioningTypes.VIRTUAL_MODE);
		provisioningTypeHashMap.put("physicalmode", DiskProvisioningTypes.PHYSICAL_MODE);
	}

	/**
	 * This method returns a boolean value specifying whether the Task is succeeded
	 * or failed.
	 *
	 * @param task ManagedObjectReference representing the Task.
	 * @return boolean value representing the Task result.
	 * @throws InvalidCollectorVersionFaultMsg
	 *
	 * @throws RuntimeFaultFaultMsg
	 * @throws InvalidPropertyFaultMsg
	 */
	boolean getTaskResultAfterDone(ManagedObjectReference task)
			throws InvalidPropertyFaultMsg, RuntimeFaultFaultMsg, InvalidCollectorVersionFaultMsg {

		boolean retVal = false;

		// info has a property - state for state of the task
		Object[] result = waitForValues.wait(task, new String[] { "info.state", "info.error" },
				new String[] { "state" },
				new Object[][] { new Object[] { TaskInfoState.SUCCESS, TaskInfoState.ERROR } });

		if (result[0].equals(TaskInfoState.SUCCESS)) {
			retVal = true;
		}
		if (result[1] instanceof LocalizedMethodFault) {
			throw new RuntimeException(((LocalizedMethodFault) result[1]).getLocalizedMessage());
		}
		return retVal;
	}

	/**
	 * Creates the virtual storage object.
	 *
	 * @throws           com.vmware.vslm.FileFaultFaultMsg
	 * @throws           com.vmware.vslm.InvalidDatastoreFaultMsg
	 * @throws           com.vmware.vslm.NotFoundFaultMsg
	 * @throws           com.vmware.vslm.RuntimeFaultFaultMsg
	 * @throws Exception the exception
	 */
	void createVStorageObject() throws Exception {
		VStorageObject createdVStorageObject = null;
		// Get vStorageObjectManager Mor.
		ManagedObjectReference morVStrObjManager = connection.getServiceContent().getVStorageObjectManager();
		// Init provisioning types:
		DiskProvisioningTypes diskProvisioningType = provisioningTypeHashMap.get(provisioningType.trim().toLowerCase());

		if (diskProvisioningType == null) {
			throw new RuntimeException("The input provisioning Type is not valid.");
		}
		// Get all the input Mor's required for creating VStorageObject.
		ManagedObjectReference morDatastore = getMOREFs.inContainerByType(serviceContent.getRootFolder(), "Datastore")
				.get(datastoreName);

		if (morDatastore == null) {
			throw new RuntimeException("The datastore name is not valid.");
		}

		// Create a create spec for VStorageObject
		VslmCreateSpec vslmCreateSpec = FcdUtil.generateVslmCreateSpec(morDatastore, diskProvisioningType, deviceName,
				vStorageObjectName, vStorageObjectSizeInMB);
		FcdVslmHelper vslmHelper = new FcdVslmHelper(connection);
		VslmPortType vslmPort = vslmHelper.getVslmPort();
		System.out.println("Operation: Creating a vStorageObject");
		ManagedObjectReference taskMor = vslmPort.vslmCreateDiskTask(vslmHelper.getVStorageObjMgr(), vslmCreateSpec);
		Boolean isCreateDiskSucceded = vslmHelper.waitForTask(taskMor);
		if (isCreateDiskSucceded) {
			System.out.println("Create disk task is succeded");
			VslmTaskInfo taskInfo = vslmPort.vslmQueryInfo(taskMor);
			createdVStorageObject = (VStorageObject) taskInfo.getResult();
			System.out.printf(
					"Success: Created vStorageObject :  %n [ Name = %s ]"
							+ " %n [ Uuid = %s ] %n [ DatastorePath = %s ]%n",
					createdVStorageObject.getConfig().getName(), createdVStorageObject.getConfig().getId().getId(),
					FcdUtil.getFcdFilePath(createdVStorageObject));
		}

		// Retrieve all the properties of a virtual storage objects based on the
		// Uuid of the vStorageObject obtained from the list call.
		System.out.println("Operation: Retrieve the  createdVStorageObjects in datastore.");
		VStorageObject retrievedVStrObj = vslmPort.vslmRetrieveVStorageObject(vslmHelper.getVStorageObjMgr(),
				createdVStorageObject.getConfig().getId());
		if (retrievedVStrObj.getConfig().getId().getId().equals(createdVStorageObject.getConfig().getId().getId())) {
			System.out.printf("Success: Retrieved vStorageObject :: %n [ %s ]%n",
					retrievedVStrObj.getConfig().getId().getId());
		} else {
			String msg = "Error: Created VStorageObject [ " + createdVStorageObject.getConfig().getId().getId()
					+ "] and retrieved VStorageObject are different.";
			throw new RuntimeException(msg);
		}
	}

	@Action
	public void run() throws Exception {
		init();
		createVStorageObject();
	}

}
