/*
 * ******************************************************
 * Copyright VMware, Inc. 2019.  All Rights Reserved.
 * ******************************************************
 *
 * DISCLAIMER. THIS PROGRAM IS PROVIDED TO YOU "AS IS" WITHOUT
 * WARRANTIES OR CONDITIONS # OF ANY KIND, WHETHER ORAL OR WRITTEN,
 * EXPRESS OR IMPLIED. THE AUTHOR SPECIFICALLY # DISCLAIMS ANY IMPLIED
 * WARRANTIES OR CONDITIONS OF MERCHANTABILITY, SATISFACTORY # QUALITY,
 * NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE.
 */
package com.vmware.fcd;

import java.util.Arrays;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.vmware.common.annotations.Action;
import com.vmware.common.annotations.Option;
import com.vmware.common.annotations.Sample;
import com.vmware.connection.ConnectedVimServiceBase;
import com.vmware.fcd.helpers.FcdHelper;
import com.vmware.fcd.helpers.FcdVslmHelper;
import com.vmware.vim25.ManagedObjectReference;
import com.vmware.vim25.VStorageObject;
import com.vmware.vslm.VslmPortType;
import com.vmware.vslm.VslmVsoVStorageObjectAssociations;
import com.vmware.vslm.VslmVsoVStorageObjectAssociationsVmDiskAssociation;

/**
 * <pre>
 * LegacyDiskAndAssociationOperations
 *
 * This sample executes below operations on a given VStorageObject
 * from vslm:
 *
 * 1. Register a given legacy disk as FCD.
 * 2. Attach a given virtual storage object to the given virtual machine.
 * 3. Retrieve vm associations for each virtual storage object in the query.
 *
 * Pre-requisite :        1. Existing VStorageObject ID
 *                        2. Existing vm name
 *
 * <b>Parameters:</b>
 * url                    [required] : url of the web service
 * username               [required] : username for the authentication
 * password               [required] : password for the authentication
 * vstorageobjectid       [required] : Uuid of the disk
 * legacydiskfilename     [required] : File name of the legacy disk
 * datacentername         [required] : Name of the datacenter
 * vmname                 [required] : Name of the virtual machine. A minimum
 *                                     virtual machine version of 'vmx-13' is
 *                                     required for the attach operation to
 *                                     succeed. vm can be present in any folder,
 *                                     host or datastore.
 * fcdname                [optional] : Name of the newly created first class
 *                                     disk object.
 * controllerkey          [optional] : Device Key of the controller the disk
 *                                     will connect to. It can be unset if
 *                                     there is only one controller (SCSI or
 *                                     SATA) with the available slot in the
 *                                     virtual machine. If there are multiple
 *                                     SCSI or SATA controllers available, user
 *                                     must specify the controller
 * unitnumber             [optional] : Unit number of the virtual machine
 *
 * <b>Command Line:</b>
 * Register a given legacy disk as First class disk.
 * run.bat com.vmware.fcd.FcdLegacyDiskAndAssociationOperations --url [webserviceurl]
 * --username [username] --password [password] --vstorageobjectid [vstorageobjectid]
 * --legacydiskfilename [legacydiskfilename] --vmname [vmname]
 *
 * Ex: --legacydiskfilename "[sharedVmfs-0] VM_NAME/VM_NAME.vmdk"
 * </pre>
 */
@Sample(name = "legacy-disk-and-association", description = "This sample registers"
		+ " a given legacy disk as FCD, attach FCD to VM and retrieve FCD associations.")
public class FcdLegacyDiskAndAssociationOperations extends ConnectedVimServiceBase {
	private String vStorageObjectId;
	private String legacyDiskFileName;
	private String dataCenterName;
	private String fcdName;
	private String vmName;
	private Integer controllerKey;
	private Integer unitNumber;

	/**
	 * @param vStorageObjectId
	 *            the vStorageObjectId to set
	 */
	@Option(name = "vstorageobjectid", required = true, description = "Uuid of the vstorageobject to  be attached  to vm.")
	public void setVStorageObjectId(String vStorageObjectId) {
		this.vStorageObjectId = vStorageObjectId;
	}

	/**
	 * @param legacyDiskFileName
	 *            the legacyDiskFileName to set
	 */
	@Option(name = "legacydiskfilename", required = true, description = "File name of the legacy disk.")
	public void setLegacyDiskFileName(String legacyDiskFileName) {
		this.legacyDiskFileName = legacyDiskFileName;
	}

	/**
	 * @param dataCenterName
	 *            the dataCenterName to set
	 */
	@Option(name = "datacentername", required = true, description = "Name of the datacenter")
	public void setDataCenterName(String dataCenterName) {
		this.dataCenterName = dataCenterName;
	}

	/**
	 * @param fcdName
	 *            the fcdName to set
	 */
	@Option(name = "fcdname", required = false, description = "File name of the legacy disk.")
	public void setFcdName(String fcdName) {
		this.fcdName = fcdName;
	}

	/**
	 * @param vmName
	 *            the vmName to set
	 */
	@Option(name = "vmname", required = true, description = "Name of virtual machine.")
	public void setVmName(String vmName) {
		this.vmName = vmName;
	}

	/**
	 * @param controllerKey
	 *            the controllerKey to set
	 */
	@Option(name = "controllerkey", required = false, description = "Device Key of the controller.")
	public void setControllerKey(String controllerKey) {
		this.controllerKey = Integer.parseInt(controllerKey);
	}

	/**
	 * @param unitNumber
	 *            the unitNumber to set
	 */
	@Option(name = "unitnumber", required = false, description = "Unit number of the virtual machine.")
	public void setUnitNumber(String unitNumber) {
		this.unitNumber = Integer.parseInt(unitNumber);
	}

	/**
	 * This method: 1. Registers a given legacy disk as FCD. 2. Attaches a given
	 * virtual storage object to the given virtual machine. 3. Retrieves vm
	 * associations for each virtual storage object in the query.
	 *
	 * @throws Exception
	 */
	void legacyDiskAndAssociationOperations() throws Exception {
		// Generate the httpUrl from diskPath which the vc recognizes.
		String legacyDiskPathForVc = getDiskPathForVc(legacyDiskFileName);

		FcdVslmHelper vslmHelper = new FcdVslmHelper(connection);
		VslmPortType vslmPort = vslmHelper.getVslmPort();

		// Register the disk as FirstClassDisk.
		System.out.println("Operation: Register a legacy disk as FCD with" + " disk Path :: " + legacyDiskPathForVc);
		VStorageObject registeredVStrObj = vslmPort.vslmRegisterDisk(vslmHelper.getVStorageObjMgr(),
				legacyDiskPathForVc, fcdName);

		System.out.printf("Success: Registered Disk(now a vStorageObject) :" + " [Uuid = %s ] with Name [ %s ]%n",
				registeredVStrObj.getConfig().getId().getId(), registeredVStrObj.getConfig().getName());

		// Get Virtual Machine Mor.
		ManagedObjectReference morPropCol = connection.getServiceContent().getPropertyCollector();
		ManagedObjectReference morVm = getMOREFs.vmByVMname(vmName, morPropCol);
		System.out.println("VM MOR : " + morVm.getValue());

		// Retrieve a vStorageObject:
		System.out.println("Operation: Retrieving a vStorageObject");
		VStorageObject retrievedVStrObj = vslmPort.vslmRetrieveVStorageObject(vslmHelper.getVStorageObjMgr(),
				FcdHelper.makeId(vStorageObjectId));
		System.out.println("Success: Retrieved vStorageObject ::" + retrievedVStrObj.getConfig().getId().getId());
		System.out.println("Operation:  Attaching a given vStorageObject to" + " the given virtualMachine.");
		ManagedObjectReference taskMor = vslmPort.vslmAttachDiskTask(vslmHelper.getVStorageObjMgr(),
				retrievedVStrObj.getConfig().getId(), morVm, controllerKey, unitNumber);
		Boolean isAttachDiskSucceded = vslmHelper.waitForTask(taskMor);
		if (isAttachDiskSucceded) {
			System.out.printf(
					"Success: Attached vStorageObjectId : " + "[ Name = %s ," + " Id = %s ] to VM [ Name = %s ]%n",
					retrievedVStrObj.getConfig().getName(), retrievedVStrObj.getConfig().getId().getId(), vmName);
		} else {
			String message = "Error: Attaching [ " + retrievedVStrObj.getConfig().getId().getId() + "] vStorageObject";
			throw new RuntimeException(message);
		}

		// Retrieve a vStorageObject based on the given vStorageObjectId and
		// verify virtualMachine is reflected as a new consumer in the
		// retrievedVStorageObject when a virtualMachine is reconfigured to ADD
		// a FCD.
		System.out.println("Operation: Retrieve the vStorageObjects in datastore.");
		VStorageObject retrievedVStrObjWithConsumer = vslmPort
				.vslmRetrieveVStorageObject(vslmHelper.getVStorageObjMgr(), retrievedVStrObj.getConfig().getId());
		if (retrievedVStrObjWithConsumer.getConfig().getId().getId()
				.equals(retrievedVStrObj.getConfig().getId().getId())) {
			if (retrievedVStrObjWithConsumer.getConfig().getConsumerId().get(0) != null) {
				System.out.printf(
						"Success: Retrieved vStorageObject :: %n [ Uuid = %s ] is associated with consumer"
								+ " [ ConsumerId = %s ]%n",
						retrievedVStrObjWithConsumer.getConfig().getId().getId(),
						retrievedVStrObjWithConsumer.getConfig().getConsumerId().get(0).getId());
			} else {
				String message = "Error: Given vStorageObject [ " + retrievedVStrObj.getConfig().getId().getId()
						+ "] does not have a consumer attached to it.";
				throw new RuntimeException(message);
			}
		} else {
			String message = "Error: Given vStorageObject [ " + retrievedVStrObj.getConfig().getId().getId()
					+ "] and retrieved VStorageObject are different.";
			throw new RuntimeException(message);
		}

		System.out.println("Operation : Retrieve vStorage object association ::");
		List<VslmVsoVStorageObjectAssociations> vStorageObjectAssociations = vslmPort
				.vslmRetrieveVStorageObjectAssociations(vslmHelper.getVStorageObjMgr(),
						Arrays.asList(retrievedVStrObj.getConfig().getId()));

		System.out.printf(
				"RetrieveVStorageObjectAssociations returned association list of size : [ %s ]"
						+ " for vStorageObject : [ %s ] from vslm.%n",
				vStorageObjectAssociations.size(), retrievedVStrObj.getConfig().getId());

		for (VslmVsoVStorageObjectAssociations vStorageObjectAssociation : vStorageObjectAssociations) {
			for (VslmVsoVStorageObjectAssociationsVmDiskAssociation vmDiskAssociation : vStorageObjectAssociation
					.getVmDiskAssociation()) {
				System.out.printf("VirtualMachine Key :: [ %s ]" + "and Disk Key :: [ %s ]",
						vmDiskAssociation.getVmId(), vmDiskAssociation.getDiskKey());
			}
		}
	}

	/**
	 * Util method to get the diskPath recognized by vc for a given disk.
	 *
	 * @param vStorageObject
	 * @return filePath of vStorageObject
	 */
	private String getDiskPathForVc(String fileNameOfDisk) {
		// Ex: vmdkLocation is :: [sharedVmfs-0] TestVm_3PYN/TestVm_3PYN.vmdk.
		String regex1 = "\\[(.*)\\]\\s(.*)/(.*\\.vmdk)";
		String ds = null;
		String vmFolder = null;
		String vmdk = null;
		if (Pattern.matches(regex1, fileNameOfDisk)) {
			System.out.println("Info: FileName Pattern matches required pattern.");
			Pattern diskNamePattern = Pattern.compile(regex1);
			Matcher fileNameMatcher = diskNamePattern.matcher(fileNameOfDisk);
			if (fileNameMatcher.find()) {
				ds = fileNameMatcher.group(1);
				vmFolder = fileNameMatcher.group(2);
				vmdk = fileNameMatcher.group(3);
			}
		}
		/*
		 * diskPath format as recognized by VC:
		 * https://VCIP/folder/PathToVmdkInsideDatastore
		 * ?dcPath=<DataCenterName>&dsName=DatastoreName
		 *
		 * Ex: diskpath = https://VC_NAME/folder/VM_NAME/VM_NAME.vmdk
		 * ?dcPath=vcqaDC&dsName=sharedVmfs-0
		 */

		String diskPath = "https://" + connection.getHost() + "/" + "folder/" + vmFolder + "/" + vmdk + "?dcPath="
				+ dataCenterName + "&dsName=" + ds;
		return diskPath;
	}

	@Action
	public void run() throws Exception {
		System.out.println(
				"Registering Legacy disk as FCD, attaching FCD to VM and retrieving FCD associations from VSLM ::");
		legacyDiskAndAssociationOperations();
	}
}
