/*
 * ******************************************************
 * Copyright VMware, Inc. 2019.  All Rights Reserved.
 * ******************************************************
 *
 * DISCLAIMER. THIS PROGRAM IS PROVIDED TO YOU "AS IS" WITHOUT
 * WARRANTIES OR CONDITIONS # OF ANY KIND, WHETHER ORAL OR WRITTEN,
 * EXPRESS OR IMPLIED. THE AUTHOR SPECIFICALLY # DISCLAIMS ANY IMPLIED
 * WARRANTIES OR CONDITIONS OF MERCHANTABILITY, SATISFACTORY # QUALITY,
 * NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE.
 */
package com.vmware.fcd;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.vmware.common.annotations.Action;
import com.vmware.common.annotations.Before;
import com.vmware.common.annotations.Option;
import com.vmware.common.annotations.Sample;
import com.vmware.connection.ConnectedVimServiceBase;
import com.vmware.fcd.helpers.FcdVslmHelper;
import com.vmware.vim25.BaseConfigInfoBackingInfo;
import com.vmware.vim25.BaseConfigInfoDiskFileBackingInfoProvisioningType;
import com.vmware.vim25.BaseConfigInfoFileBackingInfo;
import com.vmware.vim25.ID;
import com.vmware.vim25.KeyValue;
import com.vmware.vim25.ManagedObjectReference;
import com.vmware.vim25.VStorageObject;
import com.vmware.vim25.VStorageObjectSnapshotInfo;
import com.vmware.vim25.VStorageObjectSnapshotInfoVStorageObjectSnapshot;
import com.vmware.vim25.VirtualDiskCompatibilityMode;
import com.vmware.vim25.VslmCreateSpec;
import com.vmware.vim25.VslmCreateSpecBackingSpec;
import com.vmware.vim25.VslmCreateSpecDiskFileBackingSpec;
import com.vmware.vim25.VslmCreateSpecRawDiskMappingBackingSpec;
import com.vmware.vslm.VslmPortType;
import com.vmware.vslm.VslmTaskInfo;

/**
 * <pre>
 * FcdMetadataOperations
 *
 * This sample executes metadata related
 * operations on a given VStorageObject from vslm:
 *
 * 1. Create VStorageObject with metadata.
 * 2. Retrieve metadata key-value pairs from a virtual storage object.
 * 3. Retrieve the metadata value by key from a virtual storage object.
 * 4. Updates metadata key-value pairs to a virtual storage object.
 *
 * Pre-requisite :        1. Existing VStorageObject name
 *
 * <b>Parameters:</b>
 * url                    [required] : url of the web service
 * username               [required] : username for the authentication
 * password               [required] : password for the authentication
 * vstorageobjectname     [required] : name of the vstorageobject to be created
 * datastoreName          [required] : Name of the datastore
 * vstorageobjectsizeinMB [required] : Size of the disk(in MB)
 * provisioningtype       [optional] : Type of provisioning for the disk
 *                                     [thin | eagerZeroedThick |
 *                                     lazyZeroedThick | virtualMode |
 *                                     physicalMode]
 * devicename             [optional] : Canonical name of the LUN to use for disk types
 *
 * <b>Command Line:</b>
 * run.bat com.vmware.fcd.FcdMetadataOperations --url [webserviceurl]
 * --username [username] --password [password]
 * --vstorageobjectid [vstorageobjectid] --datastoreName [datastoreName]
 * --vstorageobjectsizeinMB  [vstorageobjectsizeinMB ]
 * </pre>
 */

@Sample(name = "fcd-metatdataoperations", description = "This sample creates FCD with metadata,"
		+ " retrieves metadata, retrieves metadata value and update metadata from vslm.")
public class FcdMetadataOperations extends ConnectedVimServiceBase {
	private String vStorageObjectName;
	private String datastoreName;
	private long vStorageObjectSizeInMB;
	private String provisioningType = "thin"; // Default.
	private String deviceName;
	private final Map<String, DiskProvisioningTypes> provisioningTypeHashMap = new HashMap<String, DiskProvisioningTypes>();
	private String description = "snapshot description";

	private static enum DiskProvisioningTypes {
		THIN("thin"), EAGER_ZEROED_THICK("eagerZeroedThick"), LAZY_ZEROED_THICK("lazyZeroedThick"), VIRTUAL_MODE(
				"virtualMode"), PHYSICAL_MODE("physicalMode");

		private final String value;

		private DiskProvisioningTypes(String value) {
			this.value = value;
		}

		public String value() {
			return value;
		}
	}

	/**
	 * @param vStorageObjectName
	 *            the vStorageObjectName to set
	 */
	@Option(name = "vstorageobjectname", required = true, description = "Name of the Disk.")
	public void setVStorageObjectName(String vStorageObjectName) {
		this.vStorageObjectName = vStorageObjectName;
	}

	/**
	 * @param datastoreName
	 *            the datastoreName to set
	 */
	@Option(name = "datastorename", required = true, description = "Name of datastore.")
	public void setDatastoreName(String datastoreName) {
		this.datastoreName = datastoreName;
	}

	/**
	 * @param vStorageObjectSizeInMB
	 *            the vStorageObjectSizeInMB to set
	 */
	@Option(name = "vstorageobjectsizeinMB", required = true, description = "Size of the Disk (in MB)")
	public void setVStorageObjectSize(long vStorageObjectSizeInMB) {
		this.vStorageObjectSizeInMB = vStorageObjectSizeInMB;
	}

	/**
	 * @param provisioningType
	 *            the provisioningType to set
	 */
	@Option(name = "provisioningtype", required = false, description = "Provisioning Type of the Disk.\n [thin | "
			+ "eagerZeroedThick | lazyZeroedThick | virtualMode | physicalMode]")
	public void setProvisioningType(String provisioningType) {
		this.provisioningType = provisioningType;
	}

	/**
	 * @param deviceName
	 *            the deviceName to set
	 */
	@Option(name = "devicename", required = false, description = "Canonical name of the LUN to use for RDM"
			+ " provisioning type.")
	public void setDeviceName(String deviceName) {
		this.deviceName = deviceName;
	}

	@Before
	public void init() {
		provisioningTypeHashMap.put("thin", DiskProvisioningTypes.THIN);
		provisioningTypeHashMap.put("eagerzeroedthick", DiskProvisioningTypes.EAGER_ZEROED_THICK);
		provisioningTypeHashMap.put("lazyzeroedthick", DiskProvisioningTypes.LAZY_ZEROED_THICK);
		provisioningTypeHashMap.put("virtualmode", DiskProvisioningTypes.VIRTUAL_MODE);
		provisioningTypeHashMap.put("physicalmode", DiskProvisioningTypes.PHYSICAL_MODE);
	}

	/**
	 * This method :
	 * 1. Creates VStorageObject with metadata.
	 * 2. Retrieves metadata key-value pairs from a virtual storage object.
	 * 3. Retrieves the metadata value by key from a virtual storage object.
	 * 4. Updates metadata KV pairs to a virtual storage object.
	 *
	 * @throws Exception
	 */
	void metadataOperations() throws Exception {
		FcdVslmHelper vslmHelper = new FcdVslmHelper(connection);
		VslmPortType vslmPort = vslmHelper.getVslmPort();

		// Init provisioning types:
		DiskProvisioningTypes diskProvisioningType = provisioningTypeHashMap.get(provisioningType.trim().toLowerCase());

		if (diskProvisioningType == null) {
			throw new RuntimeException("The input provisioning Type is not valid.");
		}
		// Get all the input Mor's required for creating VStorageObject.
		ManagedObjectReference morDatastore = getMOREFs.inContainerByType(serviceContent.getRootFolder(), "Datastore")
				.get(datastoreName);

		if (morDatastore == null) {
			throw new RuntimeException("The datastore name is not valid.");
		}

		// Create a create spec for VStorageObject
		VslmCreateSpec vslmCreateSpec = generateVslmCreateSpec(morDatastore, diskProvisioningType);
		System.out.println("Operation: Creating a vStorageObject with metadata");
		ManagedObjectReference taskMorCreate = vslmPort.vslmCreateDiskTask(vslmHelper.getVStorageObjMgr(),
				vslmCreateSpec);
		VStorageObject createdVStorageObject = null;
		Boolean isCreateDiskWithMetadataSucceded = vslmHelper.waitForTask(taskMorCreate);
		if (isCreateDiskWithMetadataSucceded) {
			System.out.println("Create disk with metadata task has succeded");
			VslmTaskInfo taskInfo = vslmPort.vslmQueryInfo(taskMorCreate);
			createdVStorageObject = (VStorageObject) taskInfo.getResult();
			System.out.printf(
					"Success: Created vStorageObject :  %n [ Name = %s ]"
							+ " %n [ Uuid = %s ] %n [ DatastorePath = %s ]%n",
					createdVStorageObject.getConfig().getName(), createdVStorageObject.getConfig().getId().getId(),
					getFcdFilePath(createdVStorageObject));
		} else {
			String message = "Error: Creating [ " + vStorageObjectName + "] vStorageObject";
			throw new RuntimeException(message);
		}

		// Retrieve all the properties of a virtual storage objects based on the
		// Uuid of the vStorageObject obtained from the list call.
		System.out.println("Operation: Retrieve the  createdVStorageObjects in datastore.");
		VStorageObject retrievedVStrObj = vslmPort.vslmRetrieveVStorageObject(vslmHelper.getVStorageObjMgr(),
				createdVStorageObject.getConfig().getId());
		if (retrievedVStrObj.getConfig().getId().getId().equals(createdVStorageObject.getConfig().getId().getId())) {
			System.out.printf("Success: Retrieved vStorageObject :: %n [ %s ]%n",
					retrievedVStrObj.getConfig().getId().getId());
		} else {
			String message = "Error: Created VStorageObject [ " + createdVStorageObject.getConfig().getId().getId()
					+ "] and retrieved VStorageObject are different.";
			throw new RuntimeException(message);
		}

		// Create snapshot of vStorageObject. This snapshot is required to retrieve fcd
		// metadata.
		System.out.println("Operation: Creating snapshot of given vStorageObject from vslm.");
		ManagedObjectReference taskMorSnap = vslmPort.vslmCreateSnapshotTask(vslmHelper.getVStorageObjMgr(),
				createdVStorageObject.getConfig().getId(), description);
		ID snapshotId = null;
		Boolean isSnapshotDiskSucceded = vslmHelper.waitForTask(taskMorSnap);
		if (isSnapshotDiskSucceded) {
			System.out.println("snapshot disk task is succeded");
			VslmTaskInfo taskInfo = vslmPort.vslmQueryInfo(taskMorSnap);
			snapshotId = (ID) taskInfo.getResult();
			System.out.printf("Success: Created snapshot : [ Id = %s ] of vStorageObject : [ UUID = %s ] from vslm.",
					snapshotId.getId(), createdVStorageObject.getConfig().getId().getId());
		} else {
			String message = "Error: Creating vStorageObject [ " + createdVStorageObject.getConfig().getId()
					+ "] snapshot from vslm.";
			throw new RuntimeException(message);
		}

		// Retrieve all the snapshots of vStorageObject and
		// verify if created snapshot is present in returned list
		System.out.println("Operation: Retrieve VStorageObject snapshots from vslm.");
		VStorageObjectSnapshotInfo retrievedSnapshotInfo = vslmPort
				.vslmRetrieveSnapshotInfo(vslmHelper.getVStorageObjMgr(), createdVStorageObject.getConfig().getId());
		if (retrievedSnapshotInfo.getSnapshots().size() > 0
				&& isSnapshotIdInSnapshotList(retrievedSnapshotInfo, snapshotId)) {
			System.out.printf("Success: Retrieved vStorageObject Snapshot :: [ %s ] from vslm.", snapshotId.getId());
		} else {
			String message = "Error: Retrieving VStorageObject Snapshot [ " + snapshotId.getId() + " ] from vslm.";
			throw new RuntimeException(message);
		}

		// Retrieving metadata to verify create fcd metadata
		System.out.println("Operation: Retrieving metadata of a given VStorageObject from vslm.");
		List<KeyValue> metadataList = vslmPort.vslmRetrieveVStorageObjectMetadata(vslmHelper.getVStorageObjMgr(),
				createdVStorageObject.getConfig().getId(), snapshotId, null);
		if (metadataList != null && !metadataList.isEmpty()) {
			System.out.printf("Success: Retrieved metadata of vStorageObject ::" + " vStorageObjectId %n [ %s ]%n",
					createdVStorageObject.getConfig().getId().getId());
		} else {
			String message = "Error: Retrieved metadata of" + "vStorageObject [ "
					+ createdVStorageObject.getConfig().getId().getId() + "] from vslm.";
			throw new RuntimeException(message);
		}
		// Print metadata key value returned by retrieveVStorageObjectMetadata
		System.out.printf("metadata key value returned by retrieveVStorageObjectMetadata: %n");
		for (KeyValue metadata : metadataList) {
			System.out.printf("Metadata key [ " + metadata.getKey() + " ] %n ");
			System.out.printf("Metadata value [ " + metadata.getValue() + " ] %n ");
		}

		// Retrieving metadata value for a metadata key
		System.out.println("Operation: Retrieving metadata value of " + "a given VStorageObject from vslm.");
		String metaDataValue = vslmPort.vslmRetrieveVStorageObjectMetadataValue(vslmHelper.getVStorageObjMgr(),
				createdVStorageObject.getConfig().getId(), snapshotId, metadataList.get(0).getKey());
		if (metaDataValue != null) {
			System.out.printf(
					"Success: Retrieved metadata value of vStorageObject ::" + " vStorageObjectId %n [ %s ]%n",
					createdVStorageObject.getConfig().getId().getId());
		} else {
			String message = "Error: Retrieved metadata value of" + "vStorageObject [ "
					+ createdVStorageObject.getConfig().getId().getId() + "] from vslm.";
			throw new RuntimeException(message);
		}
		// Print metadata key value returned by retrieveVStorageObjectMetadata
		System.out.printf("metadata value returned by retrieveVStorageObjectMetadataValue: %n");
		System.out.printf("Metadata value [ " + metaDataValue + " ] %n ");

		// Updating metadata key-value
		String updatedMetadataValue = metadataList.get(0).getValue() + "-updated";
		KeyValue metadataKeyValue = new KeyValue();
		metadataKeyValue.setKey(metadataList.get(0).getKey());
		metadataKeyValue.setValue(updatedMetadataValue);
		List<KeyValue> metadataListForUpdate = new ArrayList<KeyValue>();
		metadataListForUpdate.add(metadataKeyValue);
		System.out.println("Operation: Updating metadata value of" + " a given VStorageObject from vslm.");
		ManagedObjectReference taskMorUpdate = vslmPort.vslmUpdateVStorageObjectMetadataTask(
				vslmHelper.getVStorageObjMgr(), createdVStorageObject.getConfig().getId(), metadataListForUpdate, null);
		Boolean isUpdateMetadataSucceded = vslmHelper.waitForTask(taskMorUpdate);
		if (isUpdateMetadataSucceded) {
			System.out.printf("Success: Updating metadata  of vStorageObject ::" + " vStorageObjectId %n [ %s ]%n",
					createdVStorageObject.getConfig().getId().getId());
		} else {
			String message = "Error: Updating metadata of " + "vStorageObject [ "
					+ createdVStorageObject.getConfig().getId().getId() + "] from vslm.";
			throw new RuntimeException(message);
		}
		// Retrieving metadata to verify update fcd metadata
		System.out.println("Operation: Retrieving metadata of a given VStorageObject from vslm.");
		List<KeyValue> retrievedMetadataList = vslmPort.vslmRetrieveVStorageObjectMetadata(
				vslmHelper.getVStorageObjMgr(), createdVStorageObject.getConfig().getId(), null, null);
		if (retrievedMetadataList != null) {
			System.out.println("retrievedMetadataList : " + retrievedMetadataList);
			System.out.printf("Success: Retrieved metadata of vStorageObject ::" + " vStorageObjectId %n [ %s ]%n",
					createdVStorageObject.getConfig().getId().getId());
		} else {
			String message = "Error: Retrieved metadata of vStorageObject [ " + createdVStorageObject.getConfig().getId().getId()
					+ "] from vslm.";
			throw new RuntimeException(message);
		}
		System.out.println("Verify list of metadata key-values updated properly from vslm ::");
		for (KeyValue kv : retrievedMetadataList) {
			if (kv.getKey().equals(metadataList.get(0).getKey()) && kv.getValue().equals(updatedMetadataValue)) {
				System.out.println("VStorage object metadata updated succesfully from vslm :");
				System.out.println("updated metadata key : " + kv.getKey());
				System.out.println("updated metadata value : " + kv.getValue());
				break;
			}
		}
	}

	/**
	 * Verify if Snapshot Id is included in retrievedSnapshotInfo
	 *
	 * @param retrievedSnapshotInfo
	 *            VStorageObjectSnapshotInfo containing snapshot list of
	 *            VStorageObject
	 * @param snapshotId
	 *            Snapshot Id of VStorageObject
	 * @return true if retrievedSnapshotInfo contains snapshotId details
	 *
	 */
	public static boolean isSnapshotIdInSnapshotList(VStorageObjectSnapshotInfo retrievedSnapshotInfo, ID snapshotId) {
		List<String> snapshotIdList = new ArrayList<String>();
		for (VStorageObjectSnapshotInfoVStorageObjectSnapshot snapshotDetail : retrievedSnapshotInfo.getSnapshots()) {
			snapshotIdList.add(snapshotDetail.getId().getId());
		}
		if (snapshotIdList.contains(snapshotId.getId())) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * This method constructs a VslmCreateSpec for the vStorageObject
	 *
	 * @param dsMor
	 *            The ManagedObjectReferece of the datastore
	 * @param provisioningType
	 *            The provisioningType of the disk
	 * @return VslmCreateSpec
	 * @throws IllegalArgumentException
	 */
	public VslmCreateSpec generateVslmCreateSpec(ManagedObjectReference dsMor, DiskProvisioningTypes provisioningType)
			throws IllegalArgumentException {
		System.out.println("Info :: Creating VslmCreateSpec with dsMor: " + dsMor.getValue() + " provisioningType:"
				+ provisioningType.toString());
		VslmCreateSpecBackingSpec vslmCreateSpecBackingSpec;
		if (provisioningType != DiskProvisioningTypes.VIRTUAL_MODE
				&& provisioningType != DiskProvisioningTypes.PHYSICAL_MODE) {
			VslmCreateSpecDiskFileBackingSpec diskFileBackingSpec = new VslmCreateSpecDiskFileBackingSpec();
			diskFileBackingSpec.setDatastore(dsMor);
			diskFileBackingSpec.setProvisioningType(
					BaseConfigInfoDiskFileBackingInfoProvisioningType.valueOf(provisioningType.toString()).value());
			vslmCreateSpecBackingSpec = diskFileBackingSpec;
		} else {
			if (deviceName == null || deviceName.isEmpty()) {
				throw new IllegalArgumentException(
						"The devicename is mandatory for specified disktype [ " + provisioningType.value() + " ]");
			}
			VslmCreateSpecRawDiskMappingBackingSpec rdmBackingSpec = new VslmCreateSpecRawDiskMappingBackingSpec();
			rdmBackingSpec.setDatastore(dsMor);
			rdmBackingSpec
					.setCompatibilityMode(VirtualDiskCompatibilityMode.valueOf(provisioningType.toString()).value());
			rdmBackingSpec.setLunUuid(deviceName);
			vslmCreateSpecBackingSpec = rdmBackingSpec;
		}
		VslmCreateSpec createSpec = new VslmCreateSpec();
		createSpec.setBackingSpec(vslmCreateSpecBackingSpec);
		createSpec.setName(vStorageObjectName);
		createSpec.setCapacityInMB(vStorageObjectSizeInMB);
		createSpec.getMetadata().add(generateMetadataPair(1).get(0));
		return createSpec;
	}

	/**
	 * This method creates metadata key-value pairs
	 *
	 * @param numberOfMetadataToCreate
	 * @return
	 */
	public List<KeyValue> generateMetadataPair(int numberOfMetadataToCreate) {
		List<KeyValue> metadataList = new ArrayList<>();

		for (int i = 0; i < numberOfMetadataToCreate; i++) {
			KeyValue keyValue = new KeyValue();
			keyValue.setKey("SampleKey-" + i);
			keyValue.setValue("SampleValue-" + i);
			metadataList.add(keyValue);
		}
		return metadataList;
	}

	/**
	 * Util method to get the FilePath of a given VStorageObject.
	 *
	 * @param vStorageObject
	 *            The vStorageObject whose path is to be found.
	 * @return filePath of vStorageObject
	 */
	public static String getFcdFilePath(VStorageObject vStorageObject) {
		BaseConfigInfoBackingInfo backingInfo = vStorageObject.getConfig().getBacking();
		if (backingInfo instanceof BaseConfigInfoFileBackingInfo) {
			BaseConfigInfoFileBackingInfo fileBackingInfo = (BaseConfigInfoFileBackingInfo) backingInfo;
			return fileBackingInfo.getFilePath();
		}
		return null;
	}

	@Action
	public void run() throws Exception {
		System.out.println("Invoking metadata operations from VSLM ::");
		init();
		metadataOperations();
	}
}
