/*
 * ******************************************************
 * Copyright VMware, Inc. 2019.  All Rights Reserved.
 * ******************************************************
 *
 * DISCLAIMER. THIS PROGRAM IS PROVIDED TO YOU "AS IS" WITHOUT
 * WARRANTIES OR CONDITIONS # OF ANY KIND, WHETHER ORAL OR WRITTEN,
 * EXPRESS OR IMPLIED. THE AUTHOR SPECIFICALLY # DISCLAIMS ANY IMPLIED
 * WARRANTIES OR CONDITIONS OF MERCHANTABILITY, SATISFACTORY # QUALITY,
 * NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE.
 */
package com.vmware.fcd;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.vmware.common.annotations.Action;
import com.vmware.common.annotations.Before;
import com.vmware.common.annotations.Option;
import com.vmware.common.annotations.Sample;
import com.vmware.connection.ConnectedVimServiceBase;
import com.vmware.fcd.helpers.FcdHelper;
import com.vmware.fcd.helpers.FcdVslmHelper;
import com.vmware.vim25.BaseConfigInfoDiskFileBackingInfoProvisioningType;
import com.vmware.vim25.HostScsiDisk;
import com.vmware.vim25.ManagedObjectReference;
import com.vmware.vim25.VStorageObject;
import com.vmware.vim25.VirtualDiskCompatibilityMode;
import com.vmware.vim25.VslmCloneSpec;
import com.vmware.vim25.VslmCreateSpecBackingSpec;
import com.vmware.vim25.VslmCreateSpecDiskFileBackingSpec;
import com.vmware.vim25.VslmCreateSpecRawDiskMappingBackingSpec;
import com.vmware.vim25.VslmRelocateSpec;
import com.vmware.vslm.VslmDatastoreSyncStatus;
import com.vmware.vslm.VslmPortType;
import com.vmware.vslm.VslmTaskInfo;

/**
 * <pre>
 * FcdMigrationOperations
 *
 * This sample executes migration related
 * operations on a given VStorageObject from vslm:
 *
 * 1. Clone a virtual storage object.
 * 2. Relocate a virtual storage object.
 * 3. Query synchronization status of the global catalog.
 * 4. Query the synchronization state of the Global Catalog
 *    for a specified datastore.
 *
 * Pre-requisite :        1. Existing VStorageObject ID
 *
 * <b>Parameters:</b>
 * url                    [required] : url of the web service
 * username               [required] : username for the authentication
 * password               [required] : password for the authentication
 * vstorageobjectid       [required] : Uuid of the vstorageobject
 * destDatastoreName      [required] : Name of destination datastore
 * cloneddiskname         [optional] : Name of cloned vStorageObject
 * provisioningtype       [optional] : Type of provisioning for the disk
 *                                     [thin | eagerZeroedThick |
 *                                     lazyZeroedThick | virtualMode |
 *                                     physicalMode]
 * hostscsidisk           [optional] : Host Scsi disk to clone vStorageObject
 *
 * <b>Command Line:</b>
 * run.bat com.vmware.fcd.FcdMigrationOperations --url [webserviceurl]
 * --username [username] --password [password]
 * --vstorageobjectid [vstorageobjectid] --destdatastoredame [datastorename]
 * --cloneddiskname [cloneddiskname]
 * </pre>
 */

@Sample(name = "fcd-migrationoperation", description = "This sample clones and relocates a virtual storage object from vslm.")
public class FcdMigrationOperations extends ConnectedVimServiceBase {
	private String vStorageObjectId;
	private String destDatastoreName;
	private String clonedDiskName;
	private String provisioningType = "thin";
	private HostScsiDisk hostScsiDisk = null;
	private final Map<String, DiskProvisioningTypes> provisioningTypeHashMap = new HashMap<String, DiskProvisioningTypes>();

	private static enum DiskProvisioningTypes {
		THIN("thin"), EAGER_ZEROED_THICK("eagerZeroedThick"), LAZY_ZEROED_THICK("lazyZeroedThick"), VIRTUAL_MODE(
				"virtualMode"), PHYSICAL_MODE("physicalMode");

		private final String value;

		private DiskProvisioningTypes(String value) {
			this.value = value;
		}

		public String value() {
			return value;
		}
	}

	@Before
	public void init() {
		provisioningTypeHashMap.put("thin", DiskProvisioningTypes.THIN);
		provisioningTypeHashMap.put("eagerzeroedthick", DiskProvisioningTypes.EAGER_ZEROED_THICK);
		provisioningTypeHashMap.put("lazyzeroedthick", DiskProvisioningTypes.LAZY_ZEROED_THICK);
		provisioningTypeHashMap.put("virtualmode", DiskProvisioningTypes.VIRTUAL_MODE);
		provisioningTypeHashMap.put("physicalmode", DiskProvisioningTypes.PHYSICAL_MODE);
	}

	/**
	 * @param vStorageObjectId
	 *            the vStorageObjectId to set
	 */
	@Option(name = "vstorageobjectid", required = true, description = "Uuid of the vstorageobject.")
	public void setVStorageObjectId(String vStorageObjectId) {
		this.vStorageObjectId = vStorageObjectId;
	}

	/**
	 * @param destDatastoreName
	 *            the destDatastoreName to set
	 */
	@Option(name = "destdatastoredame", required = true, description = "Name of destination datastore.")
	public void setDestDatastoreName(String destDatastoreName) {
		this.destDatastoreName = destDatastoreName;
	}

	/**
	 * @param clonedDiskName
	 *            the clonedDiskName to set
	 */
	@Option(name = "cloneddiskname", required = true, description = "Name of the cloned Disk.")
	public void setVStorageObjectName(String clonedDiskName) {
		this.clonedDiskName = clonedDiskName;
	}

	/**
	 * @param provisioningType
	 *            the provisioningType to set
	 */
	@Option(name = "provisioningtype", required = false, description = "Provisioning Type of the Disk.\n [thin | "
			+ "eagerZeroedThick | lazyZeroedThick | virtualMode | physicalMode]")
	public void setProvisioningType(String provisioningType) {
		this.provisioningType = provisioningType;
	}

	/**
	 * @param hostScsiDisk
	 *            the vStorageObjectId to set
	 */
	@Option(name = "hostscsidisk", required = false, description = "Uuid of the scsi disk.")
	public void setHostScsiDisk(HostScsiDisk hostScsiDisk) {
		this.hostScsiDisk = hostScsiDisk;
	}

	/**
	 * This method is required to :
	 * 1. Clone a virtual storage object.
     * 2. Relocate a virtual storage object.
     * 3. Query synchronization status of the global catalog.
     * 4. Query the synchronization state of the Global Catalog
     *    for a specified datastore.
	 *
	 * @throws Exception
	 */
	void cloneRelocateOperations() throws Exception {
		FcdVslmHelper vslmHelper = new FcdVslmHelper(connection);
		VslmPortType vslmPort = vslmHelper.getVslmPort();

		// Init provisioning types:
		DiskProvisioningTypes diskProvisioningType = provisioningTypeHashMap.get(provisioningType.trim().toLowerCase());

		if (diskProvisioningType == null) {
			throw new RuntimeException("The input provisioning Type is not valid.");
		}

		ManagedObjectReference morDestDatastore = getMOREFs
				.inContainerByType(serviceContent.getRootFolder(), "Datastore").get(destDatastoreName);

		if (morDestDatastore == null) {
			throw new RuntimeException("The datastore name is not valid.");
		}

		// Create a cloneSpec for VStorageObject
		VslmCloneSpec vslmCloneSpec = generateVslmCloneSpec(morDestDatastore, diskProvisioningType);

		System.out.println("Operation: Cloning a vStorageObject from vslm.");
		ManagedObjectReference taskMorClone = vslmPort.vslmCloneVStorageObjectTask(vslmHelper.getVStorageObjMgr(),
				FcdHelper.makeId(vStorageObjectId), vslmCloneSpec);
		Boolean isCloneDiskSucceded = vslmHelper.waitForTask(taskMorClone);
		VStorageObject clonedVStorageObject = null;
		if (isCloneDiskSucceded) {
			VslmTaskInfo taskInfo = vslmPort.vslmQueryInfo(taskMorClone);
			clonedVStorageObject = (VStorageObject) taskInfo.getResult();
			System.out.printf(
					"Success: vStorageObject : [ Uuid = %s ]  "
							+ " cloned : [ Uuid = %s ] to [ datastore = %s ] from vslm.%n",
					vStorageObjectId, clonedVStorageObject.getConfig().getId().getId(), morDestDatastore);
		} else {
			String message = "Error: Cloning [ " + FcdHelper.makeId(vStorageObjectId) + "] vStorageObject from vslm.";
			throw new RuntimeException(message);
		}

		// Retrieve all the properties of a virtual storage objects based on the
		// Uuid of the vStorageObject obtained from the list call.
		System.out.println("Operation: Retrieve the clonedVStorageObject in datastore from vslm.");
		VStorageObject retrievedVStrObjAfterClone = vslmPort.vslmRetrieveVStorageObject(vslmHelper.getVStorageObjMgr(),
				clonedVStorageObject.getConfig().getId());
		if (retrievedVStrObjAfterClone.getConfig().getId().getId()
				.equals(clonedVStorageObject.getConfig().getId().getId())) {
			System.out.printf("Success: Retrieved vStorageObject :: [ %s ] from vslm.",
					retrievedVStrObjAfterClone.getConfig().getId().getId());
		} else {
			String message = "Error: Cloned VStorageObject [ " + clonedVStorageObject.getConfig().getId().getId()
					+ " ] and retrieved VStorageObject are different from vslm.";
			throw new RuntimeException(message);
		}

		// Create a relocate spec for VStorageObject
		VslmRelocateSpec vslmRelocateSpec = generateVslmRelocateSpec(morDestDatastore, diskProvisioningType);

		System.out.println("Operation: Relocating a vStorageObject from vslm.");
		ManagedObjectReference taskMorRelocate = vslmPort.vslmRelocateVStorageObjectTask(vslmHelper.getVStorageObjMgr(),
				FcdHelper.makeId(vStorageObjectId), vslmRelocateSpec);

		Boolean isRelocateDiskSucceded = vslmHelper.waitForTask(taskMorRelocate);
		VStorageObject relocatedVStorageObject = null;

		if (isRelocateDiskSucceded) {
			VslmTaskInfo taskInfo = vslmPort.vslmQueryInfo(taskMorRelocate);
			relocatedVStorageObject = (VStorageObject) taskInfo.getResult();
			System.out.printf(
					"Success: Relocated vStorageObject :  %n [ Name = %s ]" + " %n [ Uuid = %s ] %n to"
							+ " destination datastore [ datastore = %s ].%n",
					relocatedVStorageObject.getConfig().getName(), relocatedVStorageObject.getConfig().getId().getId(),
					morDestDatastore.getValue());
		} else {
			String message = "Error: Relocating [ " + FcdHelper.makeId(vStorageObjectId)
					+ "] vStorageObject to [ datastore = %s ] from vslm." + morDestDatastore;
			throw new RuntimeException(message);
		}

		// Retrieve all the properties of a virtual storage objects based on the
		// Uuid of the vStorageObject obtained from the list call.
		System.out.println("Operation: Retrieve the  createdVStorageObjects in datastore from vslm.");
		VStorageObject retrievedVStrObjAfterRelocation = vslmPort.vslmRetrieveVStorageObject(
				vslmHelper.getVStorageObjMgr(), relocatedVStorageObject.getConfig().getId());
		if (retrievedVStrObjAfterRelocation.getConfig().getId().getId()
				.equals(relocatedVStorageObject.getConfig().getId().getId())) {
			System.out.printf("Success: Retrieved vStorageObject :: [ %s ] from vslm.",
					retrievedVStrObjAfterRelocation.getConfig().getId().getId());
		} else {
			String message = "Error: Created VStorageObject [ " + relocatedVStorageObject.getConfig().getId().getId()
					+ " ] and retrieved VStorageObject are different from vslm.";
			throw new RuntimeException(message);
		}

		// query synchronization status of the global catalog.
		System.out.println("Operation:  Query synchronization status of the global catalog from vslm.");
		List<VslmDatastoreSyncStatus> syncStatus = vslmPort
				.vslmQueryGlobalCatalogSyncStatus(vslmHelper.getVStorageObjMgr());
		if (!syncStatus.isEmpty()) {
			System.out.println("Success: Retrieved synchronization status for the below datastores : ");
			for (VslmDatastoreSyncStatus status : syncStatus) {
				System.out.println("Datastore URL : " + status.getDatastoreURL());
			}
		}
		else {
			String message = "Error : Failed to retrieve synchronization status of the global catalog.";
			throw new RuntimeException(message);
		}

		// query the synchronization state of the Global Catalog for a specified datastore.
		System.out.println("Operation:  Query synchronization status of the global catalog"
				+ " for a specified datastore from vslm.");
		VslmDatastoreSyncStatus syncStatusForDatastore = vslmPort.vslmQueryGlobalCatalogSyncStatusForDatastore(
				vslmHelper.getVStorageObjMgr(), syncStatus.get(0).getDatastoreURL());
		if (syncStatusForDatastore != null) {
			System.out.println("Success: Retrieved synchronization status for datastore : ");
			System.out.println("Datastore URL : " + syncStatus.get(0).getDatastoreURL());
		}
		else {
			String message = "Error : Failed to retrieve synchronization status of the"
					+ " global catalog for a specified datastore.";
			throw new RuntimeException(message);
		}
	}

	/**
	 * This method constructs a VslmCloneSpec for the vStorageObject
	 *
	 * @param dsMor
	 *            The ManagedObjectReferece of the datastore
	 * @param provisioningType
	 *            The provisioningType of the disk
	 *
	 * @return VslmCloneSpec
	 *
	 * @throws IllegalArgumentException
	 */
	public VslmCloneSpec generateVslmCloneSpec(ManagedObjectReference dsMor, DiskProvisioningTypes provisioningType)
			throws IllegalArgumentException {
		System.out.println("Info :: Creating VslmCloneSpec with dsMor: " + dsMor.getValue() + " provisioningType: "
				+ provisioningType.toString());
		VslmCreateSpecBackingSpec backingSpec;
		if (provisioningType != DiskProvisioningTypes.VIRTUAL_MODE
				&& provisioningType != DiskProvisioningTypes.PHYSICAL_MODE) {
			VslmCreateSpecDiskFileBackingSpec diskFileBackingSpec = new VslmCreateSpecDiskFileBackingSpec();
			diskFileBackingSpec.setDatastore(dsMor);
			diskFileBackingSpec.setProvisioningType(
					BaseConfigInfoDiskFileBackingInfoProvisioningType.valueOf(provisioningType.toString()).value());
			backingSpec = diskFileBackingSpec;
		} else {
			VslmCreateSpecRawDiskMappingBackingSpec rdmBackingSpec = new VslmCreateSpecRawDiskMappingBackingSpec();
			rdmBackingSpec.setDatastore(dsMor);
			rdmBackingSpec
					.setCompatibilityMode(VirtualDiskCompatibilityMode.valueOf(provisioningType.toString()).value());
			rdmBackingSpec.setLunUuid(hostScsiDisk.getUuid());
			backingSpec = rdmBackingSpec;
		}
		VslmCloneSpec cloneSpec = new VslmCloneSpec();
		cloneSpec.setBackingSpec(backingSpec);
		cloneSpec.setName(clonedDiskName);
		return cloneSpec;
	}

	/**
	 * This method constructs a VslmRelocateSpec for the vStorageObject
	 *
	 * @param dsMor
	 *            The ManagedObjectReferece of the datastore
	 * @param provisioningType
	 *            The provisioningType of the disk
	 * @return VslmCreateSpec
	 * @throws IllegalArgumentException
	 */
	public VslmRelocateSpec generateVslmRelocateSpec(ManagedObjectReference dsMor,
			DiskProvisioningTypes provisioningType) throws IllegalArgumentException {
		System.out.println("Info :: Creating VslmRelocateSpec with dsMor: " + dsMor.getValue() + " provisioningType:"
				+ provisioningType.toString());
		VslmCreateSpecBackingSpec backingSpec;

		if (provisioningType != DiskProvisioningTypes.VIRTUAL_MODE
				&& provisioningType != DiskProvisioningTypes.PHYSICAL_MODE) {
			VslmCreateSpecDiskFileBackingSpec diskFileBackingSpec = new VslmCreateSpecDiskFileBackingSpec();
			diskFileBackingSpec.setDatastore(dsMor);
			diskFileBackingSpec.setProvisioningType(
					BaseConfigInfoDiskFileBackingInfoProvisioningType.valueOf(provisioningType.toString()).value());
			backingSpec = diskFileBackingSpec;
		} else {
			VslmCreateSpecRawDiskMappingBackingSpec rdmBackingSpec = new VslmCreateSpecRawDiskMappingBackingSpec();
			rdmBackingSpec.setDatastore(dsMor);
			rdmBackingSpec
					.setCompatibilityMode(VirtualDiskCompatibilityMode.valueOf(provisioningType.toString()).value());
			rdmBackingSpec.setLunUuid(hostScsiDisk.getUuid());
			backingSpec = rdmBackingSpec;
		}
		VslmRelocateSpec relocateSpec = new VslmRelocateSpec();
		relocateSpec.setBackingSpec(backingSpec);
		return relocateSpec;
	}

	@Action
	public void run() throws Exception {
		System.out.println("Cloning and Relocating vStorage Object from VSLM ::");
		init();
		cloneRelocateOperations();
	}
}
