/*
 * ******************************************************
 * Copyright VMware, Inc. 2019.  All Rights Reserved.
 * ******************************************************
 *
 * DISCLAIMER. THIS PROGRAM IS PROVIDED TO YOU "AS IS" WITHOUT
 * WARRANTIES OR CONDITIONS # OF ANY KIND, WHETHER ORAL OR WRITTEN,
 * EXPRESS OR IMPLIED. THE AUTHOR SPECIFICALLY # DISCLAIMS ANY IMPLIED
 * WARRANTIES OR CONDITIONS OF MERCHANTABILITY, SATISFACTORY # QUALITY,
 * NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE.
 */
package com.vmware.fcd;

import com.vmware.common.annotations.Action;
import com.vmware.common.annotations.Option;
import com.vmware.common.annotations.Sample;
import com.vmware.connection.ConnectedVimServiceBase;
import com.vmware.fcd.helpers.FcdVslmHelper;
import com.vmware.vim25.ManagedObjectReference;
import com.vmware.vslm.VslmPortType;

/**
 * <pre>
 * FcdReconcileDatastoreInventoryOperations
 *
 * This sample executes reconcile related
 * operations on a given VStorageObject from vslm:
 *
 * 1. Reconcile the datastore inventory info of virtual storage objects.
 * 2. Schedule reconcile of the inventory info of
 *    virtual storage objects on one of the hosts that is connected
 *    with the datastore
 *
 * <b>Parameters:</b>
 * url                    [required] : url of the web service
 * username               [required] : username for the authentication
 * password               [required] : password for the authentication
 * datastorename          [required] : Name of the datastore which contains
 *                                     the virtual storage object
 *
 * <b>Command Line:</b>
 * Reconcile the datastore inventory info of virtual storage objects.
 * run.bat com.vmware.fcd.FcdReconcileDatastoreInventoryOperations --url [webserviceurl]
 * --username [username] --password [password] --datastorename [datastorename]
 * </pre>
 */
@Sample(name = "fcd-reconciledatastoreinventoryoperations", description = "This sample reconcile"
		+ " the datastore inventory info of virtual storage objects and"
		+ " schedules reconcile of the inventory info of virtual storage objects on"
		+ " one of the hosts that is connected with the datastore from vslm.")
public class FcdReconcileDatastoreInventoryOperations extends ConnectedVimServiceBase {

	private String datastoreName;

	/**
	 * @param datastoreName
	 *            the datastoreName to set
	 */
	@Option(name = "datastorename", required = true, description = "Name of datastore that needs to be reconciled from vslm.")
	public void setDatastoreName(String datastoreName) {
		this.datastoreName = datastoreName;
	}

	/**
	 * Reconcile the datastore inventory info of virtual storage objects and
	 * schedule reconcile of the inventory info of virtual storage objects on one of
	 * the hosts that is connected with the datastore.
	 *
	 * @throws Exception
	 */
	void reconcileDSInventoryOperation() throws Exception {
		FcdVslmHelper vslmHelper = new FcdVslmHelper(connection);
		VslmPortType vslmPort = vslmHelper.getVslmPort();

		// Get all the input Mor's required for retrieving VStorageObject.
		ManagedObjectReference morDatastore = getMOREFs.inContainerByType(serviceContent.getRootFolder(), "Datastore")
				.get(datastoreName);

		// Retrieve a vStorageObject:
		System.out.println("Operation: Scheduleing reconcile of the inventory from vslm");
		vslmPort.vslmScheduleReconcileDatastoreInventory(vslmHelper.getVStorageObjMgr(), morDatastore);

		System.out.printf(
				"Scheduled reconcile of the inventory info of virtual storage objects on one of the hosts that is "
						+ "connected with the datastore: [ %s ] from vslm.",
				morDatastore);

		// Reconcile the datastore inventory info of virtual storage objects.
		System.out.println("Operation: Reconciling the datastore inventory info of virtual storage objects from vslm.");
		ManagedObjectReference taskMor = vslmPort.vslmReconcileDatastoreInventoryTask(vslmHelper.getVStorageObjMgr(),
				morDatastore);
		Boolean isReconcileDatastoreInventorySucceded = vslmHelper.waitForTask(taskMor);
		if (isReconcileDatastoreInventorySucceded) {
			System.out.printf("Success: Reconciled datastore : [ %s ] from vslm", morDatastore);
		} else {
			String message = "Error: Reconciling datastore [ " + morDatastore + " ] from vslm.";
			throw new RuntimeException(message);
		}
	}

	@Action
	public void run() throws Exception {
		System.out.println("Invoking reconcile datastore inventory operations on VStorageObject from VSLM :: ::");
		reconcileDSInventoryOperation();
	}
}
