/*
 * ******************************************************
 * Copyright VMware, Inc. 2019.  All Rights Reserved.
 * ******************************************************
 *
 * DISCLAIMER. THIS PROGRAM IS PROVIDED TO YOU "AS IS" WITHOUT
 * WARRANTIES OR CONDITIONS # OF ANY KIND, WHETHER ORAL OR WRITTEN,
 * EXPRESS OR IMPLIED. THE AUTHOR SPECIFICALLY # DISCLAIMS ANY IMPLIED
 * WARRANTIES OR CONDITIONS OF MERCHANTABILITY, SATISFACTORY # QUALITY,
 * NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE.
 */
package com.vmware.fcd;

import com.vmware.common.annotations.Action;
import com.vmware.common.annotations.Option;
import com.vmware.common.annotations.Sample;
import com.vmware.connection.ConnectedVimServiceBase;
import com.vmware.fcd.helpers.FcdHelper;
import com.vmware.fcd.helpers.FcdVslmHelper;
import com.vmware.vim25.BaseConfigInfoBackingInfo;
import com.vmware.vim25.BaseConfigInfoDiskFileBackingInfo;
import com.vmware.vim25.BaseConfigInfoRawDiskMappingBackingInfo;
import com.vmware.vim25.ManagedObjectReference;
import com.vmware.vim25.VStorageObject;
import com.vmware.vslm.VslmPortType;

/**
 * <pre>
 * FcdRenameExtendInflateDeleteOperations
 *
 * This sample executes below operations on a given VStorageObject
 * from vslm:
 *
 * 1. Rename a given virtual storage object.
 * 2. Extend a virtual storage object capacity.
 * 3. Inflate a sparse or thin-provisioned virtual disk up to the full size.
 * 4. Delete a given virtual storage object.
 *
 * Pre-requisite :        1. Existing VStorageObject ID
 *
 * <b>Parameters:</b>
 * url                    [required] : url of the web service
 * username               [required] : username for the authentication
 * password               [required] : password for the authentication
 * vstorageobjectid       [required] : Uuid of the vstorageobject
 * newvstorageobjectname  [required] : The new name for the virtual storage object.
 * newcapacityinmb        [required] : The new capacity of the virtual disk in MB,
 *                                     which should be greater than the original
 *                                     disk size.
 *
 * <b>Command Line:</b>
 * Rename a virtual storage object.
 * run.bat com.vmware.fcd.FcdRenameExtendInflateDeleteOperations --url [webserviceurl]
 * --username [username] --password [password] --vstorageobjectid [vstorageobjectid]
 * --newvstorageobjectname [newvstorageobjectname] --newcapacityinmb [newcapacityinmb]
 * </pre>
 */

@Sample(name = "fcd-renameextendinflatedeleteoperations",
        description = "This sample renamnes an FCD,"
		+ "extends capacity of FCD, inflates a sparse or "
        + "thin-provisioned FCD up to the full size and deletes FCD")
public class FcdRenameExtendInflateDeleteOperations extends ConnectedVimServiceBase {

	private String vStorageObjectId;
	private String newVStorageObjectName;
	private long newCapacityInMB;

	/**
	 * @param vStorageObjectId
	 *            the vStorageObjectId to set
	 */
	@Option(name = "vstorageobjectid", required = true, description = "Uuid of the vstorageobject.")
	public void setVStorageObjectId(String vStorageObjectId) {
		this.vStorageObjectId = vStorageObjectId;
	}

	/**
	 * @param newVStorageObjectName
	 *            the newVStorageObjectName to set
	 */
	@Option(name = "newvstorageobjectname", required = true, description = "The new name for the virtual storage object.")
	public void setNewVStorageObjectName(String newVStorageObjectName) {
		this.newVStorageObjectName = newVStorageObjectName;
	}

	/**
	 * @param newCapacityInMB
	 *            the newCapacityInMB to set.
	 */
	@Option(name = "newcapacityinmb", required = true,
			description = "The new capacity of the virtual disk in MB,which should be greater than the original disk size.")
	public void setNewCapacityInMB(long newCapacityInMB) {
		this.newCapacityInMB = newCapacityInMB;
	}

	/**
	 * This method renames an FCD, extends capacity of FCD, inflates a sparse or
	 * thin-provisioned FCD up to the full size and deletes FCD
	 *
	 *
	 * @throws Exception
	 */
	void renameExtendInflateDeleteOperations() throws Exception {
		FcdVslmHelper vslmHelper = new FcdVslmHelper(connection);
		VslmPortType vslmPort = vslmHelper.getVslmPort();
        // Retrieves vStorageObject before renaming and verify
		System.out.println(
				"Operation: Retrieve vStorageObject after revertVStorageObject and before renaming from datastore from vslm.");
		VStorageObject retrievedVStrObjBeforeRename = vslmPort
				.vslmRetrieveVStorageObject(vslmHelper.getVStorageObjMgr(), FcdHelper.makeId(vStorageObjectId));
		if (retrievedVStrObjBeforeRename != null) {
			System.out.printf("Success: Retrieved vStorageObject :: %n [ UUid = %s ]" + " %n [ name = %s ]%n ",
					retrievedVStrObjBeforeRename.getConfig().getId().getId(),
					retrievedVStrObjBeforeRename.getConfig().getName());
		} else {
			String message = "Error: Retrieving VStorageObject [ " + FcdHelper.makeId(vStorageObjectId);
			throw new RuntimeException(message);
		}

		// Rename a vStorageObject:
		System.out.println("Operation: Renaming the given vStorageObject from vslm.");
		vslmPort.vslmRenameVStorageObject(vslmHelper.getVStorageObjMgr(), FcdHelper.makeId(vStorageObjectId),
				newVStorageObjectName);

		// Retrieve vStorageObject after renaming
		System.out.println("Operation: Retrieve the vStorageObject after renaming" + " from datastore from vslm.");
		VStorageObject retrievedVStrObjAfterRename = vslmPort.vslmRetrieveVStorageObject(vslmHelper.getVStorageObjMgr(),
				FcdHelper.makeId(vStorageObjectId));
		if (retrievedVStrObjAfterRename != null) {
			System.out.printf("Success: Retrieved vStorageObject :: %n [ UUid = %s ]" + "%n [ name = %s ] %n",
					retrievedVStrObjAfterRename.getConfig().getId().getId(),
					retrievedVStrObjAfterRename.getConfig().getName());
		} else {
			String message = "Error: Retrieving VStorageObject [ " + FcdHelper.makeId(vStorageObjectId);
			throw new RuntimeException(message);
		}

		// Verify rename vStorageObject
		if (retrievedVStrObjAfterRename.getConfig().getName().equals(newVStorageObjectName)) {
			System.out.printf("Success: Renamed vStorageObject :: [ %s ] from vslm",
					retrievedVStrObjAfterRename.getConfig().getId().getId());
		} else {
			String message = "Error: VStorageObject [ " + retrievedVStrObjAfterRename.getConfig().getId().getId()
					+ "] rename to [ " + newVStorageObjectName + " ] from vslm";
			throw new RuntimeException(message);
		}

		// Extend capacity of VStorageObject
		System.out.println("Operation: Extending capacity of vStorageObject from vslm.");
		ManagedObjectReference taskMor = vslmPort.vslmExtendDiskTask(vslmHelper.getVStorageObjMgr(),
				FcdHelper.makeId(vStorageObjectId), newCapacityInMB);
		Boolean isExtendDiskSucceeded = vslmHelper.waitForTask(taskMor);
		if (isExtendDiskSucceeded) {
			System.out.printf("Success: Extended vStorageObject : %n [ Uuid = %s ]" + "%n [ NewCapacity = %s ]%n",
					vStorageObjectId, newCapacityInMB);
		} else {
			String message = "Error: Extending vStorageObject [ " + FcdHelper.makeId(vStorageObjectId) + "] from vslm.";
			throw new RuntimeException(message);
		}

		// Retrieve all the properties of a virtual storage objects based on the
		// Uuid of the vStorageObject and verify the new capacity
		System.out.println("Operation: Retrieve the extendedVStorageObject from datastore from vslm.");
		VStorageObject retrievedVStrObjAfterExtend = vslmPort.vslmRetrieveVStorageObject(vslmHelper.getVStorageObjMgr(),
				FcdHelper.makeId(vStorageObjectId));
		if (retrievedVStrObjAfterExtend != null) {
			System.out.printf("Success: Retrieved vStorageObject :: [ %s ] from vslm.%n",
					retrievedVStrObjAfterExtend.getConfig().getId().getId());
		} else {
			String message = "Error: Retrieving VStorageObject [ " + vStorageObjectId + " ] from vslm.";
			throw new RuntimeException(message);
		}

		// Verify capacity of vStorageObject
		if (retrievedVStrObjAfterExtend.getConfig().getCapacityInMB() == newCapacityInMB) {
			System.out.println("Success: Extend vStorageObject capacity  :: [ "
					+ retrievedVStrObjAfterExtend.getConfig().getId().getId() + " ] from vslm.");
		} else {
			String message = "Error: VStorageObject [ " + vStorageObjectId + " ] capcity extend failed from vslm.";
			throw new RuntimeException(message);
		}

		// Retrieve vStorageObject before inflate disk
		VStorageObject vStorageObjectBeforeInflate = vslmPort.vslmRetrieveVStorageObject(vslmHelper.getVStorageObjMgr(),
				FcdHelper.makeId(vStorageObjectId));

		if (vStorageObjectBeforeInflate != null) {
			System.out.printf("Success: Retrieved vStorageObject before inflate" + " :: %n [ %s ]%n from vslm.%n",
					vStorageObjectId);
		} else {
			String message = "Error: Retrieving VStorageObject [ " + vStorageObjectId + " ] before inflate";
			throw new RuntimeException(message);
		}

		// Inflate a sparse or thin-provisioned virtual disk up to the full size.
		System.out.println(
				"Operation: Inflate a sparse or thin-provisioned" + " virtual disk up to the full size from vslm.");
		ManagedObjectReference taskMorInflate = vslmPort.vslmInflateDiskTask(vslmHelper.getVStorageObjMgr(),
				FcdHelper.makeId(vStorageObjectId));
		Boolean isInflateDiskSucceeded = vslmHelper.waitForTask(taskMorInflate);
		if (isInflateDiskSucceeded) {
			System.out.printf("Success: Inflated vStorageObject : [ Uuid = %s ] from vslm.%n", vStorageObjectId);
		} else {
			String message = "Error: Inflating vStorageObject [ " + vStorageObjectId + " ] from vslm.";
			throw new RuntimeException(message);
		}

		// Retrieve vStorageObject after inflate disk
		VStorageObject vStorageObjectAfterInflate = vslmPort.vslmRetrieveVStorageObject(vslmHelper.getVStorageObjMgr(),
				FcdHelper.makeId(vStorageObjectId));

		if (vStorageObjectAfterInflate != null) {
			System.out.printf("Success: Retrieved vStorageObject after inflate :: [ %s ] from vslm.%n",
					vStorageObjectId);
		} else {
			String message = "Error: Retrieving VStorageObject [ " + vStorageObjectId + " ] after inflate from vslm.";
			throw new RuntimeException(message);
		}

		// Print provisioning type before and after inflate
		String diskProvisioningTypeBefore = getProvisioningType(vStorageObjectBeforeInflate);
		String diskProvisioningTypeAfter = getProvisioningType(vStorageObjectAfterInflate);
		System.out.printf("Provisioning type before inflate is : [ %s ] %n", diskProvisioningTypeBefore);
		System.out.printf("Provisioning type after inflate is : [ %s ] %n", diskProvisioningTypeAfter);

		// Verify disk path after and before inflate
		String filePathBeforeInflate = FcdHelper.getFcdFilePath(vStorageObjectBeforeInflate);
		if (filePathBeforeInflate == null) {
			String message = "Error: File backing doesn't present for source vStorageObject [ " + vStorageObjectId + " ]";
			throw new RuntimeException(message);
		}
		String filePathAfterInflate = FcdHelper.getFcdFilePath(vStorageObjectAfterInflate);
		if (filePathAfterInflate == null) {
			String message = "Error: File backing doesn't present for vStorageObject [ " + vStorageObjectId
					+ " ] after inflate";
			throw new RuntimeException(message);
		}
		if (!filePathAfterInflate.equals(filePathBeforeInflate)) {
			String message = "Error: File path changed for vStorageObject [ " + vStorageObjectId + " ] after inflate";
			throw new RuntimeException(message);
		} else {
			System.out.println("File Path before and after inflate is same.");
		}

		// Verify capacity before and after inflate
		long capacityBeforeInflate = vStorageObjectBeforeInflate.getConfig().getCapacityInMB();
		long capacityAFterInflate = vStorageObjectAfterInflate.getConfig().getCapacityInMB();
		if (capacityBeforeInflate != capacityAFterInflate) {
			String message = "Error: Disk size changed for vStorageObject [ " + vStorageObjectId + " ] after inflate";
			throw new RuntimeException(message);
		} else {
			System.out.println("Disk size before and after inflate is same.");
		}

		System.out.println("Operation: Deleting a vStorageObject");
		ManagedObjectReference taskMorDelete = vslmPort.vslmDeleteVStorageObjectTask(vslmHelper.getVStorageObjMgr(),
				FcdHelper.makeId(vStorageObjectId));
		Boolean isDeleteDiskSucceded = vslmHelper.waitForTask(taskMorDelete);
		if (isDeleteDiskSucceded) {
			System.out.println("Delete disk task is succeded");
			System.out.printf("Success: Deleted vStorageObject :  %n [ Uuid = %s ]%n", vStorageObjectId);
		} else {
			String message = "Error: Deleting [ " + vStorageObjectId + "] vStorageObject";
			throw new RuntimeException(message);
		}
	}

	/**
	 * Method to return provisioning type of a VStorageObject.
	 *
	 * @param vStorageObject
	 * @return
	 */
	private String getProvisioningType(VStorageObject vStorageObject) {
		String provType = null;
		BaseConfigInfoBackingInfo backingInfo = vStorageObject.getConfig().getBacking();
		if (backingInfo instanceof BaseConfigInfoDiskFileBackingInfo) {
			BaseConfigInfoDiskFileBackingInfo dfbinfo = (BaseConfigInfoDiskFileBackingInfo) backingInfo;
			provType = dfbinfo.getProvisioningType().toString();
		} else if (backingInfo instanceof BaseConfigInfoRawDiskMappingBackingInfo) {
			BaseConfigInfoRawDiskMappingBackingInfo rdmBackinginfo = (BaseConfigInfoRawDiskMappingBackingInfo) backingInfo;
			provType = rdmBackinginfo.getCompatibilityMode();
		}
		return provType;
	}

	@Action
	public void run() throws Exception {
		System.out.println("Invoking rename, extend, inflate and delete operations on VStorageObject from VSLM :: ::");
		renameExtendInflateDeleteOperations();
	}
}
