/*
 * ******************************************************
 * Copyright VMware, Inc. 2019.  All Rights Reserved.
 * ******************************************************
 *
 * DISCLAIMER. THIS PROGRAM IS PROVIDED TO YOU "AS IS" WITHOUT
 * WARRANTIES OR CONDITIONS # OF ANY KIND, WHETHER ORAL OR WRITTEN,
 * EXPRESS OR IMPLIED. THE AUTHOR SPECIFICALLY # DISCLAIMS ANY IMPLIED
 * WARRANTIES OR CONDITIONS OF MERCHANTABILITY, SATISFACTORY # QUALITY,
 * NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE.
 */
package com.vmware.fcd;

import java.util.Arrays;
import java.util.List;

import com.vmware.common.annotations.Action;
import com.vmware.common.annotations.Option;
import com.vmware.common.annotations.Sample;
import com.vmware.connection.ConnectedVimServiceBase;
import com.vmware.fcd.helpers.FcdHelper;
import com.vmware.fcd.helpers.FcdVslmHelper;
import com.vmware.vim25.VStorageObject;
import com.vmware.vim25.VStorageObjectStateInfo;
import com.vmware.vslm.VslmPortType;
import com.vmware.vslm.VslmVsoVStorageObjectQueryResult;
import com.vmware.vslm.VslmVsoVStorageObjectQuerySpec;
import com.vmware.vslm.VslmVsoVStorageObjectResult;

/**
 * <pre>
 * FcdRetrieveOperations
 *
 * This sample executes below retrieve related operations on
 * a snapshot of a given VStorageObject from vslm.:
 *
 * 1. Retrieve virtual storage object.
 * 2. Retrieve multiple virtual storage objects.
 * 3. Retrieve a virtual storage object state.
 * 4. List all virtual storage objects or by matching queries.
 *
 * Pre-requisite :        1. Existing VStorageObject ID
 *
 * <b>Parameters:</b>
 * url                    [required] : url of the web service
 * username               [required] : username for the authentication
 * password               [required] : password for the authentication
 * vstorageobjectid       [required] : Uuid of the vstorageobject
 * maxresult              [required] : Maximum number of virtual storage object IDs to return.
 * queryfield             [required] : The searchable field.
 * queryoperator          [required] : The operator to compare with the searchable field.
 * queryvalue             [required] : The value to compare with the searchable field.
 *
 * <b>Command Line:</b>
 * run.bat com.vmware.fcd.FcdRetrieveOperations --url [webserviceurl]
 * --username [username] --password [password]
 * --vstorageobjectid [vstorageobjectid] --maxresult [maxresult]
 * --queryfield [queryfield] --queryvalue [queryvalue] --queryoperator [queryoperator]
 * </pre>
 */

@Sample(name = "fcd-retrieveoperations", description = "This sample retrieves a virtual storage object,"
		+ "  retrieves virtual storage object state and lists all virtual storage object from vslm.")
public class FcdRetrieveOperations extends ConnectedVimServiceBase {
	private String vStorageObjectId;
	private String maxResult;
	private String queryField;
	private String queryOperator;
	private String queryValue;


	/**
	 * @param vStorageObjectId
	 *            the vStorageObjectId to set
	 */
	@Option(name = "vstorageobjectid", required = true, description = "Uuid of the vstorageobject.")
	public void setVStorageObjectId(String vStorageObjectId) {
		this.vStorageObjectId = vStorageObjectId;
	}

	/**
	 * @param maxResult
	 *            Maximum number of virtual storage object IDs to return.
	 */
	@Option(name = "maxresult", required = true, description = "Maximum number of virtual storage object IDs to return.")
	public void setMaxResult(String maxResult) {
		this.maxResult = maxResult;
	}

	/**
	 * @param queryField
	 *            This indicates the searchable field
	 */
	@Option(name = "queryfield", required = true, description = "This indicates the searchable field.")
	public void setQueryField(String queryField) {
		this.queryField = queryField;
	}

	/**
	 * @param queryOperator
	 *            the operator to compare the searchable field
	 */
	@Option(name = "queryoperator", required = true, description = "This field specifies the operator to compare the searchable field.")
	public void setQueryOperator(String queryOperator) {
		this.queryOperator = queryOperator;
	}

	/**
	 * @param queryValue
	 *            This field specifies the value to be compared with the searchable
	 *            field
	 */
	@Option(name = "queryvalue", required = true, description = "This field specifies the operator to compare the searchable field.")
	public void setQueryValue(String queryValue) {
		this.queryValue = queryValue;
	}

	/**
	 * This method retrieves a virtual storage object,
	 * retrieves multiple virtual storage objects,
	 * retrieves virtual storage object state and lists
	 * all virtual storage object from vslm.
	 *
	 * @throws Exception
	 */
	void retrieveOperations() throws Exception {
		FcdVslmHelper vslmHelper = new FcdVslmHelper(connection);
		VslmPortType vslmPort = vslmHelper.getVslmPort();
		// Retrieve vStorageObject
		System.out.println("Operation: Retrieve the vStorageObject from vslm");
		VStorageObject retrievedVStrObj = vslmPort.vslmRetrieveVStorageObject(vslmHelper.getVStorageObjMgr(),
				FcdHelper.makeId(vStorageObjectId));
		if (retrievedVStrObj != null) {
			System.out.printf(
					"Success: Retrieved vStorageObject :: %n [ UUid = %s ]%n"
							+ " vStorageObjectName %n [ Name = %s ]%n CapacityInMB %n [ Capacity = %s ]%n",
					retrievedVStrObj.getConfig().getId().getId(), retrievedVStrObj.getConfig().getName(),
					retrievedVStrObj.getConfig().getCapacityInMB());
		} else {
			String msg = "Error: Retrieving VStorageObject [ "
		        + vStorageObjectId + " ] from vslm.";
			throw new RuntimeException(msg);
		}
		// Retrieve vStorageObjects
		System.out.println("Operation: Retrieve the vStorageObjects from vslm");
		List<VslmVsoVStorageObjectResult> retrievedVStrObjs = vslmPort.vslmRetrieveVStorageObjects(
				vslmHelper.getVStorageObjMgr(), Arrays.asList(FcdHelper.makeId(vStorageObjectId)));
		if (retrievedVStrObjs != null) {
			System.out.printf(
					"Success: Retrieved vStorageObjects :: %n [ UUid = %s ]%n"
							+ " vStorageObjectName %n [ Name = %s ]%n CapacityInMB %n [ Capacity = %s ]%n",
							retrievedVStrObjs.get(0).getId().getId(), retrievedVStrObjs.get(0).getName(),
							retrievedVStrObjs.get(0).getCapacityInMB());
		} else {
			String msg = "Error: Retrieving VStorageObjects [ "
		        + vStorageObjectId + " ] from vslm.";
			throw new RuntimeException(msg);
		}
		// Retrieve vStorageObject state
		System.out.println("Operation: Retrieve the vStorageObject state from " + "datastore from vslm.");
		VStorageObjectStateInfo stateInfo = vslmPort.vslmRetrieveVStorageObjectState(vslmHelper.getVStorageObjMgr(),
				FcdHelper.makeId(vStorageObjectId));

		if (stateInfo != null) {
			System.out.printf(
					"Success: Retrieved state info [stateInfo = %s ] of vStorageObject ::"
			        + " [ %s ] from vslm.",
					stateInfo.isTentative(), vStorageObjectId);
		} else {
			String msg = "Error: Retrieving state info for VStorageObject [ "
					+ vStorageObjectId + " ] from vslm.";
			throw new RuntimeException(msg);
		}

		System.out.println("Operation: lists all virtual storage object from vslm.");
		VslmVsoVStorageObjectQuerySpec querySpec = new VslmVsoVStorageObjectQuerySpec();
		querySpec.setQueryField(queryField);
		querySpec.setQueryOperator(queryOperator);
		querySpec.getQueryValue().add(queryValue);
		VslmVsoVStorageObjectQueryResult queryResult = null;
		// If queries are provided for particular vstorage object
		if (queryField != null && queryOperator != null && queryValue != null) {
			queryResult = vslmPort.vslmListVStorageObjectForSpec(vslmHelper.getVStorageObjMgr(),
					Arrays.asList(querySpec), Integer.parseInt(maxResult));
		}
		// If no queries are provided
		else {
			queryResult = vslmPort.vslmListVStorageObjectForSpec(vslmHelper.getVStorageObjMgr(), null,
					Integer.parseInt(maxResult));
		}

		if (queryResult != null) {
			System.out.println("Success:  List virtual storage objects :");
		} else {
			String msg = "Error: List virtual storage objects : ";
			throw new RuntimeException(msg);
		}
		System.out.println("List all vstorage objects IDs : ");
		for (VslmVsoVStorageObjectResult results : queryResult.getQueryResults()) {
			System.out.println("vstorage object id :" + results.getId().getId());
		}
	}

	@Action
	public void run() throws Exception {
		System.out.println("Invoking retrieve operations on virtual storage object from VSLM ::");
		retrieveOperations();
	}
}
