/*
 * ******************************************************
 * Copyright VMware, Inc. 2019.  All Rights Reserved.
 * ******************************************************
 *
 * DISCLAIMER. THIS PROGRAM IS PROVIDED TO YOU "AS IS" WITHOUT
 * WARRANTIES OR CONDITIONS # OF ANY KIND, WHETHER ORAL OR WRITTEN,
 * EXPRESS OR IMPLIED. THE AUTHOR SPECIFICALLY # DISCLAIMS ANY IMPLIED
 * WARRANTIES OR CONDITIONS OF MERCHANTABILITY, SATISFACTORY # QUALITY,
 * NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE.
 */

package com.vmware.fcd;

import java.util.ArrayList;
import java.util.List;

import com.vmware.common.annotations.Action;
import com.vmware.common.annotations.Option;
import com.vmware.common.annotations.Sample;
import com.vmware.connection.ConnectedVimServiceBase;
import com.vmware.fcd.helpers.FcdHelper;
import com.vmware.fcd.helpers.FcdVslmHelper;
import com.vmware.vim25.DiskChangeInfo;
import com.vmware.vim25.ID;
import com.vmware.vim25.ManagedObjectReference;
import com.vmware.vim25.VStorageObject;
import com.vmware.vim25.VStorageObjectSnapshotDetails;
import com.vmware.vim25.VStorageObjectSnapshotInfo;
import com.vmware.vim25.VStorageObjectSnapshotInfoVStorageObjectSnapshot;
import com.vmware.vim25.VirtualMachineProfileSpec;
import com.vmware.vslm.VslmPortType;
import com.vmware.vslm.VslmTaskInfo;

/**
 * <pre>
 * FcdSnapshotOperations
 *
 * This sample executes below snapshot related operation on
 * a snapshot of a given VStorageObject from vslm :
 *
 * 1. Creates snapshot of a given VStorageObject.
 * 2. Retrieves Snapshot Info of a given VStorageObject.
 * 3. Retrieves Snapshot Details of a given VStorageObject.
 * 4. Query disk changed areas of a given VStorageObject.
 * 5. Creates vStorageObject from snapshot.
 * 6. Reverts VStorageObject to a given snapshot.
 * 7. Deletes snapshot  of a given VStorageObject.
 *
 * Pre-requisite :        Existing VStorageObject ID
 * <b>Parameters:</b>
 * url                    [required] : url of the web service
 * username               [required] : username for the authentication
 * password               [required] : password for the authentication
 * vstorageobjectid       [required] : Uuid of the vstorageobject
 * description            [required] : A short description to be associated
 *                                     with the snapshot.
 * fcdname                [required] : FCD name is required to create
 *                                     disk from snapshot.
 * pbmprofileid           [optional] : SPBM Profile requirement on the
 *                                     new virtual storage object.
 * devicegroupid          [optional] : Id of the replication device group.
 * faultdomainid          [optional] : Id of the fault domain to which the group belongs.
 * datastorepath          [optional] : Relative location in the specified datastore where disk needs to be created.
 *
 * <b>Command Line:</b>
 * run.bat com.vmware.fcd.FcdSnapshotOperations --url [webserviceurl]
 * --username [username] --password [password]
 * --vstorageobjectid [vstorageobjectid] --description [description]
 * </pre>
 */

@Sample(name = "fcd-snapshotoperations",
        description = "This sample executes snapshot related operation on a snapshot of a given VStorageObject.")
public class FcdSnapshotOperations extends ConnectedVimServiceBase {
	private String vStorageObjectId;
	private String description;
	private String fcdName;
	private String pbmProfileId = null; // Default.
	private String deviceGroupId = null; // Default.
	private String faultDomainId = null; // Default.
	private String datastorePath = null; // Default.
	private String controlFlag = "enableChangedBlockTracking";

	/**
	 * @param vStorageObjectId
	 *            the vStorageObjectId to set
	 */
	@Option(name = "vstorageobjectid", required = true, description = "Uuid of the vstorageobject.")
	public void setVStorageObjectId(String vStorageObjectId) {
		this.vStorageObjectId = vStorageObjectId;
	}

	/**
	 * @param description
	 *            the description to set
	 */
	@Option(name = "description", required = true, description = "A short description to be associated with the snapshot.")
	public void setDescription(String description) {
		this.description = description;
	}

	/**
	 * @param fcdName
	 *            the fcdName to set
	 */
	@Option(name = "fcdname", required = true, description = "A user friendly name to be"
			+ "associated with the new disk.")
	public void setVStorageObjectName(String fcdName) {
		this.fcdName = fcdName;
	}

	/**
	 * @param pbmProfileId
	 *            the pbmProfileId to set
	 */
	@Option(name = "pbmprofileid", required = false, description = "SPBM Profile requirement on the"
			+ "new virtual storage object.")
	public void setProfileSpec(String pbmProfileId) {
		this.pbmProfileId = pbmProfileId;
	}

	/**
	 * @param deviceGroupId
	 *            the deviceGroupId to set
	 */
	@Option(name = "devicegroupid", required = false, description = "Id of the replication device group.")
	public void setDeviceGroupId(String deviceGroupId) {
		this.deviceGroupId = deviceGroupId;
	}

	/**
	 * @param faultDomainId
	 *            the faultDomainId to set
	 */
	@Option(name = "faultdomainid", required = false, description = "ID of the fault domain to"
			+ " which the group belongs.")
	public void setFaultDomainId(String faultDomainId) {
		this.faultDomainId = faultDomainId;
	}

	/**
	 * @param datastorePath
	 *            the path to set
	 */
	@Option(name = "datastorepath", required = false, description = "Relative location in the specified datastore"
			+ "where disk needs to be created.")
	public void setDatastorePath(String datastorePath) {
		this.datastorePath = datastorePath;
	}

	/**
	 * This method :
	 * 1. Creates snapshot of a given VStorageObject.
	 * 2. Retrieves Snapshot Info of a given VStorageObject.
	 * 3. Retrieves Snapshot Details of a given VStorageObject.
	 * 4. Query disk changed areas of a given VStorageObject.
	 * 5. Creates vStorageObject from snapshot.
	 * 6. Reverts VStorageObject to a given snapshot.
	 * 7. Deletes snapshot  of a given VStorageObject.
	 *
	 * @throws Exception
	 */
	void snapshotOperations() throws Exception {
		FcdVslmHelper vslmHelper = new FcdVslmHelper(connection);
		VslmPortType vslmPort = vslmHelper.getVslmPort();

	    List<String> controlFlags = new ArrayList<String>();
	    controlFlags.add(controlFlag);
	    //set control flag on vstorage object to enable change block tracking.
	    vslmPort.vslmSetVStorageObjectControlFlags(
            vslmHelper.getVStorageObjMgr(), FcdHelper.makeId(vStorageObjectId), controlFlags);
		VStorageObject vStrObj = vslmPort.vslmRetrieveVStorageObject(vslmHelper.getVStorageObjMgr(),
				FcdHelper.makeId(vStorageObjectId));
		// Create snapshot of vStorageObject
		System.out.println("Operation: Creating snapshot of given vStorageObject from vslm.");
		ManagedObjectReference taskMor = vslmPort.vslmCreateSnapshotTask(vslmHelper.getVStorageObjMgr(),
				FcdHelper.makeId(vStorageObjectId), description);
		ID snapshotId = null;
		Boolean isSnapshotDiskSucceded = vslmHelper.waitForTask(taskMor);
		if (isSnapshotDiskSucceded) {
			System.out.println("snapshot disk task is succeded");
			VslmTaskInfo taskInfo = vslmPort.vslmQueryInfo(taskMor);
			snapshotId = (ID) taskInfo.getResult();
			System.out.printf("Success: Created snapshot : [ Id = %s ] of vStorageObject : [ UUID = %s ] from vslm.",
					snapshotId.getId(), vStorageObjectId);
		} else {
			String message = "Error: Creating vStorageObject [ " + vStorageObjectId + "] snapshot from vslm.";
			throw new RuntimeException(message);
		}
		// Retrieve all the snapshots of vStorageObject and
		// verify if created snapshot is present in returned list
		System.out.println("Operation: Retrieve VStorageObject snapshot info from vslm.");
		VStorageObjectSnapshotInfo retrievedSnapshotInfo = vslmPort
				.vslmRetrieveSnapshotInfo(vslmHelper.getVStorageObjMgr(), FcdHelper.makeId(vStorageObjectId));
		if (retrievedSnapshotInfo.getSnapshots().size() > 0
				&& isSnapshotIdInSnapshotList(retrievedSnapshotInfo, snapshotId)) {
			System.out.printf("Success: Retrieved vStorageObject Snapshot :: [ %s ] from vslm.", snapshotId.getId());
		} else {
			String message = "Error: Retrieving VStorageObject Snapshot [ " + snapshotId.getId() + " ] from vslm.";
			throw new RuntimeException(message);
		}

		// Retrieve snapshot details
		System.out.println("Operation: Retrieve VStorageObject snapshot details from vslm.");
		VStorageObjectSnapshotDetails  snapshotDetails  = vslmPort
				.vslmRetrieveSnapshotDetails(vslmHelper.getVStorageObjMgr(),
						FcdHelper.makeId(vStorageObjectId), snapshotId);
		if (snapshotDetails != null) {
			String expectedSnapshotPath = FcdHelper.getFcdFilePath(vStrObj);
			String actualSnapshotPath = snapshotDetails.getPath();
			if (expectedSnapshotPath.equals(actualSnapshotPath)) {
				System.out.printf("Success: Retrieved vStorageObject Snapshot details"
						+ " of snapshot id :: [ %s ], snapshot path :: [ %s ] from vslm.",
						snapshotId.getId(), actualSnapshotPath);
			}
			else {
				System.out.printf("Error: Retrieved vStorageObject Snapshot details"
						+ " of snapshot id :: [ %s ], actual snapshot path :: [ %s ]"
						+ " doesn't match with expected snapshot path :: [ %s ] from vslm.",
						snapshotId.getId(), actualSnapshotPath, expectedSnapshotPath);
			}
		}
		else {
			String message = "Error: Retrieving VStorageObject Snapshot Details [ " + snapshotId.getId() + " ] from vslm.";
			throw new RuntimeException(message);
		}

		//query a list of areas of a virtual disk that have been modified since a well-defined point in the past.
		System.out.println("Operation: query disk changed areas from vslm.");
		String changeId = snapshotDetails.getChangedBlockTrackingId();
		Long startOffset = 0L;
		System.out.println("Retrieved change id :" + changeId);
		DiskChangeInfo diskChangeInfo = vslmPort.vslmQueryChangedDiskAreas(
				vslmHelper.getVStorageObjMgr(), FcdHelper.makeId(vStorageObjectId),
				snapshotId, startOffset, changeId);
		if (diskChangeInfo != null) {
			System.out.printf("Success: Query disk change area of vStorageObject ::" + " vStorageObjectId %n [ %s ]%n",
					vStorageObjectId);
		} else {
			String message = "Error: Query disk change area of " + "vStorageObject [ "
					+ FcdHelper.makeId(vStorageObjectId) + "] from vslm.";
			throw new RuntimeException(message);
		}

		List<VirtualMachineProfileSpec> profileSpec = null;
		if (pbmProfileId != null) {
			profileSpec = FcdHelper.generateVirtualMachineProfileSpec(pbmProfileId, deviceGroupId, faultDomainId);
		}

		// Create vStorageObject from snapshot
		System.out.println("Operation: Creating snapshot of given vStorageObject from vslm.");
		ManagedObjectReference taskMorCreateDiskFromSnap = vslmPort.vslmCreateDiskFromSnapshotTask(
				vslmHelper.getVStorageObjMgr(), FcdHelper.makeId(vStorageObjectId), snapshotId, fcdName, profileSpec,
				null, datastorePath);
		VStorageObject vStorageObjectCreatedFromSnapshot = null;
		Boolean isCreateDiskFromSnapshotSucceded = vslmHelper.waitForTask(taskMorCreateDiskFromSnap);
		if (isCreateDiskFromSnapshotSucceded) {
			VslmTaskInfo taskInfo = vslmPort.vslmQueryInfo(taskMorCreateDiskFromSnap);
			vStorageObjectCreatedFromSnapshot = (VStorageObject) taskInfo.getResult();
			System.out.printf(
					"Success: Created vStorageObject from snapshot : %n [ SnapshotId = %s ]"
							+ " of vstorageObject : %n [ UUid = %s ] " + " with name :  %n [ fcd name = %s ] "
							+ " %n [ Uuid = %s ] %n from vslm.",
					snapshotId, vStorageObjectId, vStorageObjectCreatedFromSnapshot.getConfig().getName(),
					vStorageObjectCreatedFromSnapshot.getConfig().getId().getId());
		} else {
			String message = "Error: Creating vStorageObject from snapshot : [ " + snapshotId.getId()
					+ "] of vStorageObject : [ " + vStorageObjectId + " ] from vslm.";
			throw new RuntimeException(message);
		}

		// Retrieve all the properties of a newly createdvirtual storage objects
		// based on the Uuid of the vStorageObject obtained from the list call.
		System.out.println("Operation: Retrieve the vStorageObject created from snapshot in datastore from vslm.");
		VStorageObject retrievedVStrObj = vslmPort.vslmRetrieveVStorageObject(vslmHelper.getVStorageObjMgr(),
				vStorageObjectCreatedFromSnapshot.getConfig().getId());
		if (retrievedVStrObj.getConfig().getId().getId()
				.equals(vStorageObjectCreatedFromSnapshot.getConfig().getId().getId())) {
			System.out.printf("Success: Retrieved vStorageObject :: %n [ %s ]%n from vslm.",
					retrievedVStrObj.getConfig().getId().getId());
		} else {
			String message = "Error: Created VStorageObject [ "
					+ vStorageObjectCreatedFromSnapshot.getConfig().getId().getId()
					+ "] and retrieved VStorageObject are different from vslm";
			throw new RuntimeException(message);
		}

		// Reverts to a given snapshot of a VStorageObject
		System.out.println("Operation: Reverting to a given snapshot" + " of a VStorageObject from vslm.");
		ManagedObjectReference taskMorRevert = vslmPort.vslmRevertVStorageObjectTask(vslmHelper.getVStorageObjMgr(),
				FcdHelper.makeId(vStorageObjectId), snapshotId);
		Boolean isRevertDiskSucceded = vslmHelper.waitForTask(taskMorRevert);
		if (isRevertDiskSucceded) {
			System.out.printf("Success: Reverted vStorageObject : [ UUID = %s ]"
					+ " to snapshot :  [ SnapshotId = %s ] from vslm.%n", vStorageObjectId, snapshotId.getId());
		} else {
			String message = "Error: Reverting vStorageObject [ " + vStorageObjectId + "] to snapshot [ "
					+ snapshotId.getId() + " ] from vslm.";
			throw new RuntimeException(message);
		}

		// Delete snapshot

		System.out.println("Operation: Deleting snapshot of given vStorageObject from vslm.");
		ManagedObjectReference taskMorDeleteSnap = vslmPort.vslmDeleteSnapshotTask(vslmHelper.getVStorageObjMgr(),
				FcdHelper.makeId(vStorageObjectId), snapshotId);
		Boolean isDeleteSnapshotSucceded = vslmHelper.waitForTask(taskMorDeleteSnap);
		if (isDeleteSnapshotSucceded) {
			System.out.printf("Success: Deleted snapshot : [ SnapshotId = %s ] "
					+ "of vStorageObject :  [ UUID = %s ] from vslm.", snapshotId.getId(), vStorageObjectId);
		} else {
			String message = "Error: Deleting [ " + vStorageObjectId + "] vStorageObject snapshot [ " + snapshotId.getId()
					+ " ] from vslm.%n";
			throw new RuntimeException(message);
		}

		// Retrieve all the properties of a virtual storage objects based on the
		// Uuid of the vStorageObject obtained from the list call.
		System.out.println("Operation: Retrieve VStorageObject snapshot from vslm.");
		VStorageObjectSnapshotInfo retrievedSnapshotInfoAfterDelete = vslmPort
				.vslmRetrieveSnapshotInfo(vslmHelper.getVStorageObjMgr(), FcdHelper.makeId(vStorageObjectId));
		if (!isSnapshotIdInSnapshotList(retrievedSnapshotInfoAfterDelete, snapshotId)) {
			System.out.printf("Success: Deleted vStorageObject Snapshot :: [ %s ]"
					+ " is not included in retrieved snapshot list from vslm.", snapshotId.getId());
		} else {
			String message = "Error: Deleted vStorageObject Snapshot [ " + snapshotId.getId() + " ] is included in"
					+ "retrieved snapshot list from vslm.";
			throw new RuntimeException(message);
		}
	}

	/**
	 * Verifies if Snapshot Id is included in retrievedSnapshotInfo
	 *
	 * @param retrievedSnapshotInfo
	 *            VStorageObjectSnapshotInfo containing snapshot list of
	 *            VStorageObject
	 * @param snapshotId
	 *            Snapshot Id of VStorageObject
	 * @return true if retrievedSnapshotInfo contains snapshotId details
	 *
	 */
	public static boolean isSnapshotIdInSnapshotList(VStorageObjectSnapshotInfo retrievedSnapshotInfo, ID snapshotId) {
		List<String> snapshotIdList = new ArrayList<String>();
		for (VStorageObjectSnapshotInfoVStorageObjectSnapshot snapshotDetail : retrievedSnapshotInfo.getSnapshots()) {
			snapshotIdList.add(snapshotDetail.getId().getId());
		}
		if (snapshotIdList.contains(snapshotId.getId())) {
			return true;
		} else {
			return false;
		}
	}

	@Action
	public void run() throws Exception {
		System.out.println("Invoking Snapshot related APIs from VSLM ::");
		snapshotOperations();
	}
}
