/*
 * ******************************************************
 * Copyright VMware, Inc. 2019.  All Rights Reserved.
 * ******************************************************
 *
 * DISCLAIMER. THIS PROGRAM IS PROVIDED TO YOU "AS IS" WITHOUT
 * WARRANTIES OR CONDITIONS # OF ANY KIND, WHETHER ORAL OR WRITTEN,
 * EXPRESS OR IMPLIED. THE AUTHOR SPECIFICALLY # DISCLAIMS ANY IMPLIED
 * WARRANTIES OR CONDITIONS OF MERCHANTABILITY, SATISFACTORY # QUALITY,
 * NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE.
 */
package com.vmware.fcd;

import java.util.ArrayList;
import java.util.List;

import com.vmware.common.annotations.Action;
import com.vmware.common.annotations.Option;
import com.vmware.common.annotations.Sample;
import com.vmware.connection.ConnectedVimServiceBase;
import com.vmware.fcd.helpers.FcdHelper;
import com.vmware.fcd.helpers.FcdVslmHelper;
import com.vmware.vim25.ID;
import com.vmware.vim25.VslmTagEntry;
import com.vmware.vslm.VslmPortType;

/**
 * <pre>
 * FcdTagOperations
 *
 * This sample executes below tag related
 * operations on a given VStorageObject from vslm:
 *
 * 1. Attach a tag to a virtual storage object.
 * 2. List all tags attached to virtual storage object.
 * 3. List all virtual storage objects attached to the tag.
 * 4. Detach a tag from a virtual storage object.
 *
 * Pre-requisite :        1. Existing VStorageObject name
 *                        2. Existing tag name
 *                        3. Existing category name
 * <b>Parameters:</b>
 * url                    [required] : url of the web service
 * username               [required] : username for the authentication
 * password               [required] : password for the authentication
 * vstorageobjectid       [required] : Uuid of the vstorageobject
 * category               [required] : The category to which the tag belongs.
 * tag                    [required] : The tag which has to be associated
 *                                     with the virtual storage object.
 *
 * <b>Command Line:</b>
 * run.bat com.vmware.fcd.FcdTagOperations --url [webserviceurl]
 * --username [username] --password [password]
 * --vstorageobjectid [vstorageobjectid] --category [category] --tag [tag]
 * </pre>
 */

@Sample(name = "fcd-tagoperations", description = "This sample attaches"
		+ " tag to FCD, lists tags attached to FCD, list FCDs attached"
		+ " to tag and detaches tags from FCD .")
public class FcdTagOperations extends ConnectedVimServiceBase {
	private String vStorageObjectId;
	private String category;
	private String tag;

	/**
	 * @param vStorageObjectId
	 *            the vStorageObjectId to set
	 */
	@Option(name = "vStorageObjectId", required = true, description = "The ID of the virtual storage object.")
	public void setVStorageObjectId(String vStorageObjectId) {
		this.vStorageObjectId = vStorageObjectId;
	}

	/**
	 * @param category
	 *            the category to set
	 */
	@Option(name = "category", required = true, description = "The category to which the tag belongs.")
	public void setCategory(String category) {
		this.category = category;
	}

	/**
	 * @param tag
	 *            the tag to set
	 */
	@Option(name = "tag", required = true, description = "The tag to be queried.")
	public void setTag(String tag) {
		this.tag = tag;
	}

	/**
	 * This method attaches tag to FCD, lists tags attached to FCD, lists FCDs
	 * attached to tag and detaches tags from FCD .
	 *
	 * @throws Exception
	 */

	void taggingOperations() throws Exception {
		FcdVslmHelper vslmHelper = new FcdVslmHelper(connection);
		VslmPortType vslmPort = vslmHelper.getVslmPort();

		// Attach a tag to a virtual storage object.
		System.out.println("Operation: Attach a tag to a virtual storage object from vslm.");
		vslmPort.vslmAttachTagToVStorageObject(vslmHelper.getVStorageObjMgr(), FcdHelper.makeId(vStorageObjectId),
				category, tag);

		// Verify attachTagToVStorageObject operation by invoking
		// listVStorageObjectsAttachedToTag
		System.out.println("Operation: List all vStorageObjects attached to a tag from vslm.");
		List<ID> retrievedVStrObjIdList = vslmPort.vslmListVStorageObjectsAttachedToTag(vslmHelper.getVStorageObjMgr(),
				category, tag);
		List<String> retrivedFcdIdList = new ArrayList<>();
		for (ID retrievedId : retrievedVStrObjIdList) {
			retrivedFcdIdList.add(retrievedId.getId());
		}
		if (!retrievedVStrObjIdList.isEmpty() && retrivedFcdIdList.contains(vStorageObjectId)) {
			System.out.printf(
					"vStorageObject list returned by" + " listVStorageObjectsAttachedToTag"
							+ " contains vstorageObject :: [ %s ] from vslm.%n",
					retrievedVStrObjIdList.size(), vStorageObjectId);
		} else {
			String message = "Error: VStorageObject [ " + vStorageObjectId + " ] is not present in the list returned by"
					+ " listVStorageObjectsAttachedToTag for category " + category + " and tag " + tag + " from vslm.";
			throw new RuntimeException(message);
		}
		// Verify attachTagToVStorageObject by invoking listTagsAttachedToVStorageObject
		System.out.println("Operation: List all tags attached to a vStorageObject from vslm.");
		List<VslmTagEntry> retrievedTagList = vslmPort.vslmListTagsAttachedToVStorageObject(
				vslmHelper.getVStorageObjMgr(), FcdHelper.makeId(vStorageObjectId));
		if (!retrievedTagList.isEmpty()) {
			List<String> categoryList = new ArrayList<>();
			List<String> tagList = new ArrayList<>();
			for (VslmTagEntry retrievedTag : retrievedTagList) {
				categoryList.add(retrievedTag.getParentCategoryName());
				tagList.add(retrievedTag.getTagName());
			}
			if (categoryList.contains(category) && tagList.contains(tag)) {
				System.out.printf(
						"Tag list of size [ %s ] returned by listTagsAttachedToVStorageObject"
								+ " for vstorageObject :: [ %s ] contains category :: [ %s ]" + " and tag :: [ %s ]%n",
						retrievedVStrObjIdList.size(), vStorageObjectId, category, tag);
			}
		} else {
			String message = "Error: Category [ " + category + " ] and tag [ " + tag
					+ " ] is not present in the list returned by" + " listTagsAttachedToVStorageObject"
					+ " for vStorageObject [ " + vStorageObjectId + " ] from vslm.";
			throw new RuntimeException(message);
		}

		// Detaching a tag from a virtual storage object.
		System.out.println("Operation: Detach a tag from a virtual storage object.");
		vslmPort.vslmDetachTagFromVStorageObject(vslmHelper.getVStorageObjMgr(), FcdHelper.makeId(vStorageObjectId),
				category, tag);

		// Verify detachTagFromVStorageObject API by invoking
		// listVStorageObjectsAttachedToTag
		System.out.println("Operation: List all vStorageObject attached to tag from vslm.");
		List<ID> retrievedVStrObjIdListAfterDeletion = vslmPort
				.vslmListVStorageObjectsAttachedToTag(vslmHelper.getVStorageObjMgr(), category, tag);
		if (!retrievedVStrObjIdListAfterDeletion.contains(FcdHelper.makeId(vStorageObjectId))) {
			System.out.printf("vStorageObject list returned by" + " listVStorageObjectsAttachedToTag "
					+ " doesn't contain vstorageObject :: [ %s ].%n", vStorageObjectId);
		} else {
			String message = "Error: VStorageObject [ " + vStorageObjectId + " ] is still present in the list returned by"
					+ " listVStorageObjectsAttachedToTag for category " + category + " and tag " + tag + " from vslm.";
			throw new RuntimeException(message);
		}

		// Verify detachTagFromVStorageObject API by invoking
		// listTagsAttachedToVStorageObject
		System.out.println("Operation: List all tags attached to vStorageObject from vslm.");
		List<VslmTagEntry> retrievedTagListAfterDeletion = vslmPort.vslmListTagsAttachedToVStorageObject(
				vslmHelper.getVStorageObjMgr(), FcdHelper.makeId(vStorageObjectId));
		List<String> categoryListAfterDeletion = new ArrayList<>();
		List<String> tagListAfterDeletion = new ArrayList<>();
		if (!retrievedTagListAfterDeletion.isEmpty()) {
			for (VslmTagEntry retrievedTag : retrievedTagListAfterDeletion) {
				categoryListAfterDeletion.add(retrievedTag.getParentCategoryName());
				tagListAfterDeletion.add(retrievedTag.getTagName());
			}
		}
		if (retrievedTagListAfterDeletion.isEmpty()
				|| !(categoryListAfterDeletion.contains(category) && tagListAfterDeletion.contains(tag))) {
			System.out.printf(
					"Tag list returned by listTagsAttachedToVStorageObject" + " for vstorageObject :: [ %s ]"
							+ " doesn't contain category :: [ %s ]" + " and tag :: [ %s ].%n",
					vStorageObjectId, category, tag);
		}
	}

	@Action
	public void run() throws Exception {
		System.out.println("Invoking Tagging related APIs from VSLM ::");
		taggingOperations();
	}
}
