/*
 * ******************************************************
 * Copyright VMware, Inc. 2019.  All Rights Reserved.
 * ******************************************************
 *
 * DISCLAIMER. THIS PROGRAM IS PROVIDED TO YOU "AS IS" WITHOUT
 * WARRANTIES OR CONDITIONS # OF ANY KIND, WHETHER ORAL OR WRITTEN,
 * EXPRESS OR IMPLIED. THE AUTHOR SPECIFICALLY # DISCLAIMS ANY IMPLIED
 * WARRANTIES OR CONDITIONS OF MERCHANTABILITY, SATISFACTORY # QUALITY,
 * NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE.
 */

package com.vmware.fcd;

import java.util.List;

import com.vmware.common.annotations.Action;
import com.vmware.common.annotations.Option;
import com.vmware.common.annotations.Sample;
import com.vmware.connection.ConnectedVimServiceBase;
import com.vmware.fcd.helpers.FcdHelper;
import com.vmware.fcd.helpers.FcdVslmHelper;
import com.vmware.vim25.ManagedObjectReference;
import com.vmware.vim25.VirtualMachineProfileSpec;
import com.vmware.vim25.VslmInfrastructureObjectPolicy;
import com.vmware.vim25.VslmInfrastructureObjectPolicySpec;
import com.vmware.vslm.VslmPortType;

/**
 * <pre>
 * FcdUpdateAndRetrieveInfrastructurePolicyOperations
 *
 * This sample updates vStorageObject policy,
 * updates and retrieves virtual storage infrastructure object
 * SBPM policy on given datastore from vslm.
 *
 * Pre-requisite :        1. Existing VStorageObject ID
 *                        2. Existing SPBM profile ID
 * <b>Parameters:</b>
 * url                    [required] : url of the web service
 * username               [required] : username for the authentication
 * password               [required] : password for the authentication
 * vstorageobjectid       [required] : Uuid of the vstorageobject
 * pbmprofileid           [required] : SPBM Profile requirement on the
 *                                     new virtual storage object.
 * datastorename          [required] : Datastore on which policy needs to be retrieved.
 *                                     It only supports VSAN datastore.
 * devicegroupid          [optional] : Id of the replication device group.
 * faultdomainid          [optional] : Id of the fault domain to which the group belongs.
 *
 * <b>Command Line:</b>
 * Update vStorageObject policy,Update andRetrieve virtual storage
 * infrastructure object SBPM policy on given datastore.
 * run.bat com.vmware.fcd.FcdUpdateAndRetrieveInfrastructurePolicyOperations --url [webserviceurl]
 * --username [username] --password [password] --vstorageobjectid [vstorageobjectid] --pbmprofileid [pbmprofileid]
 * --datastorename [datastorename]
 * </pre>
 */
@Sample(name = "fcd-updateandretrieveinfrastructurepolicyoperations", description = "This sample updates vStorageObject policy,"
		+ " updates and retrieves virtual storage infrastructure object SBPM policy on given datastore.")
public class FcdUpdateAndRetrieveInfrastructurePolicyOperations extends ConnectedVimServiceBase {
	private String vStorageObjectId;
	private String pbmProfileId;
	private String datastoreName;
	private String deviceGroupId = null; // Default.
	private String faultDomainId = null; // Default.

	/**
	 * @param vStorageObjectId
	 *            the vStorageObjectId to set
	 */
	@Option(name = "vstorageobjectid", required = true, description = "Uuid of the vstorageobject.")
	public void setVStorageObjectId(String vStorageObjectId) {
		this.vStorageObjectId = vStorageObjectId;
	}

	/**
	 * @param pbmProfileId
	 *            the pbmProfileId to set
	 */
	@Option(name = "pbmprofileid", required = true, description = "SPBM Profile requirement on the "
			+ "new virtual storage object.")
	public void setProfileSpec(String pbmProfileId) {
		this.pbmProfileId = pbmProfileId;
	}

	/**
	 * @param datastoreName
	 *            the datastoreName to set
	 */
	@Option(name = "datastorename", required = true,
			description = "Name of datastore on which datastore policy needs to be assigned.It only supports VSAN datastore.")
	public void setDatastoreName(String datastoreName) {
		this.datastoreName = datastoreName;
	}

	/**
	 * @param deviceGroupId
	 *            the deviceGroupId to set for newly created vStorageObject from
	 *            snapshot
	 */
	@Option(name = "devicegroupid", required = false, description = "Id of the replication device group.")
	public void setDeviceGroupId(String deviceGroupId) {
		this.deviceGroupId = deviceGroupId;
	}

	/**
	 * @param faultDomainId
	 *            the faultDomainId to set for newly created vStorageObject from
	 *            snapshot
	 */
	@Option(name = "faultdomainid", required = false, description = "ID of the fault domain to"
			+ " which the group belongs.")
	public void setFaultDomainId(String faultDomainId) {
		this.faultDomainId = faultDomainId;
	}

	/**
	 *
	 * This method updates vStorageObject policy,
	 * updates and retrieves virtual storage infrastructure object SBPM
	 * policy on given datastore.
	 *
	 * @throws Exception
	 */
	void infraObjectPolicyOperations() throws Exception {

		FcdVslmHelper vslmHelper = new FcdVslmHelper(connection);
		VslmPortType vslmPort = vslmHelper.getVslmPort();

		List<VirtualMachineProfileSpec> profileSpec = null;
		if (pbmProfileId != null) {
			profileSpec = FcdHelper.generateVirtualMachineProfileSpec(pbmProfileId, deviceGroupId, faultDomainId);
		}
		// update VStorageObject Policy
		System.out.println("Operation: Updating virtual storage object policy from vslm : ");
		ManagedObjectReference taskMorUpdateVstrObjPolicy = vslmPort.vslmUpdateVstorageObjectPolicyTask(
				vslmHelper.getVStorageObjMgr(), FcdHelper.makeId(vStorageObjectId), profileSpec);
		Boolean isUpdateVstrObjPolSucceeded = vslmHelper.waitForTask(taskMorUpdateVstrObjPolicy);
		if (isUpdateVstrObjPolSucceeded) {
			System.out.printf(
					"Success : Vstorage object : [ Name = %s ]" + " updated with profile : [ ProfileId = %s ].%n",
					vStorageObjectId, pbmProfileId);
		} else {
			System.out.printf("Error : Failed to update Vstorage object : [ Name = %s ]"
					+ " with profile : [ ProfileId = %s ].%n", vStorageObjectId, pbmProfileId);
		}
		// update VStorageObject Infrastructure Object Policy:
		System.out.println(
				"Operation: Updating virtual storage infrastructure" + " object SBPM policy on given from vslm.");
		// Get all the input Mor's required for retrieving VStorageObject.
		ManagedObjectReference morDatastore = getMOREFs.inContainerByType(serviceContent.getRootFolder(), "Datastore")
				.get(datastoreName);

		VslmInfrastructureObjectPolicySpec spec = new VslmInfrastructureObjectPolicySpec();
		spec.setDatastore(morDatastore);
		spec.getProfile().addAll(profileSpec);

		vslmPort.vslmUpdateVStorageInfrastructureObjectPolicyTask(vslmHelper.getVStorageObjMgr(), spec);
		System.out.printf("Success: Updated infrastructure policy for datastore : [ datastore = %s ] from vslm.%n",
				morDatastore.getValue());
		// Retrieve VStorageObject Infrastructure Object Policy:
		System.out.println(
				"Operation: Retrieving virtual storage infrastructure" + " object SBPM policy on given from vslm.");
		List<VslmInfrastructureObjectPolicy> retrieveVStorageObjectInfraPolicyList = vslmPort
				.vslmRetrieveVStorageInfrastructureObjectPolicy(vslmHelper.getVStorageObjMgr(), morDatastore);

		System.out.printf("Success: Retrieved infrastructure policy for datastore : [ datastore = %s ] from vslm.%n",
				morDatastore.getValue());
		for (VslmInfrastructureObjectPolicy infraPolicy : retrieveVStorageObjectInfraPolicyList) {
			System.out.printf(
					"Infrastructure object : [ Name = %s ]" + " associated with profile : [ ProfileId = %s ].%n",
					infraPolicy.getName(), infraPolicy.getProfileId());
		}
	}

	@Action
	public void run() throws Exception {
		System.out.println("Invoking infrastructure object SBPM policy related APIs from VSLM ::");
		infraObjectPolicyOperations();
	}
}
