package com.vmware.fcd.helpers;

import java.util.ArrayList;
import java.util.List;

import com.vmware.vim25.BaseConfigInfoBackingInfo;
import com.vmware.vim25.BaseConfigInfoDiskFileBackingInfoProvisioningType;
import com.vmware.vim25.BaseConfigInfoFileBackingInfo;
import com.vmware.vim25.ID;
import com.vmware.vim25.ManagedObjectReference;
import com.vmware.vim25.VStorageObject;
import com.vmware.vim25.VirtualDiskCompatibilityMode;
import com.vmware.vim25.VslmCreateSpec;
import com.vmware.vim25.VslmCreateSpecBackingSpec;
import com.vmware.vim25.VslmCreateSpecDiskFileBackingSpec;
import com.vmware.vim25.VslmCreateSpecRawDiskMappingBackingSpec;

public class FcdUtil {
	public static enum DiskProvisioningTypes {
		THIN("thin"), EAGER_ZEROED_THICK("eagerZeroedThick"), LAZY_ZEROED_THICK("lazyZeroedThick"),
		VIRTUAL_MODE("virtualMode"), PHYSICAL_MODE("physicalMode");

		private final String value;

		private DiskProvisioningTypes(String value) {
			this.value = value;
		}

		public String value() {
			return value;
		}
	}

	/**
	 * This method constructs a VslmCreateSpec for the vStorageObject
	 *
	 * @param dsMor              The ManagedObjectReferece of the datastore
	 * @param provisioningType   The provisioningType of the disk
	 * @param deviceName         Device name
	 * @param vStorageObjectName Given FCD name
	 * @param vStorageObjectSizeInMB FCD size
	 * @return VslmCreateSpec
	 * @throws IllegalArgumentException
	 */
	public static VslmCreateSpec generateVslmCreateSpec(ManagedObjectReference dsMor,
			DiskProvisioningTypes provisioningType, String deviceName, String vStorageObjectName,
			long vStorageObjectSizeInMB) throws IllegalArgumentException {
		System.out.println("Info :: Creating VslmCreateSpec with dsMor: " + dsMor.getValue() + " provisioningType:"
				+ provisioningType.toString());
		VslmCreateSpecBackingSpec vslmCreateSpecBackingSpec;
		if (provisioningType != DiskProvisioningTypes.VIRTUAL_MODE
				&& provisioningType != DiskProvisioningTypes.PHYSICAL_MODE) {
			VslmCreateSpecDiskFileBackingSpec diskFileBackingSpec = new VslmCreateSpecDiskFileBackingSpec();
			diskFileBackingSpec.setDatastore(dsMor);
			diskFileBackingSpec.setProvisioningType(
					BaseConfigInfoDiskFileBackingInfoProvisioningType.valueOf(provisioningType.toString()).value());
			vslmCreateSpecBackingSpec = diskFileBackingSpec;
		} else {
			if (deviceName == null || deviceName.isEmpty()) {
				throw new IllegalArgumentException(
						"The devicename is mandatory for specified disktype [ " + provisioningType.value() + " ]");
			}
			VslmCreateSpecRawDiskMappingBackingSpec rdmBackingSpec = new VslmCreateSpecRawDiskMappingBackingSpec();
			rdmBackingSpec.setDatastore(dsMor);
			rdmBackingSpec
					.setCompatibilityMode(VirtualDiskCompatibilityMode.valueOf(provisioningType.toString()).value());
			rdmBackingSpec.setLunUuid(deviceName);
			vslmCreateSpecBackingSpec = rdmBackingSpec;
		}
		VslmCreateSpec createSpec = new VslmCreateSpec();
		createSpec.setBackingSpec(vslmCreateSpecBackingSpec);
		createSpec.setName(vStorageObjectName);
		createSpec.setCapacityInMB(vStorageObjectSizeInMB);
		return createSpec;
	}

	/**
	 * Verifies if Fcd Id as string List is included in fcdIDList
	 *
	 * @param fcdStrIdList List of FCD IDs as String
	 * @param fcdIdList    List of FCD IDs as ID
	 * @return true if fcdIDListontains all the fcds in fcdStrIDList
	 */
	public static boolean isFcdIdInFcdList(List<String> fcdStrIdList, List<ID> fcdIdList) {
		List<String> fcdIDAsStringList = new ArrayList<String>();
		for (ID fcdID : fcdIdList) {
			fcdIDAsStringList.add(fcdID.getId());
		}
		if (fcdIDAsStringList.containsAll(fcdStrIdList)) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * Util method to get the FilePath of a given VStorageObject.
	 *
	 * @param vStorageObject The vStorageObject whose path is to be found.
	 * @return filePath of vStorageObject
	 */
	public static String getFcdFilePath(VStorageObject vStorageObject) {
		BaseConfigInfoBackingInfo backingInfo = vStorageObject.getConfig().getBacking();
		if (backingInfo instanceof BaseConfigInfoFileBackingInfo) {
			BaseConfigInfoFileBackingInfo fileBackingInfo = (BaseConfigInfoFileBackingInfo) backingInfo;
			return fileBackingInfo.getFilePath();
		}
		return null;
	}
}
