package com.vmware.fcd.helpers;

import com.vmware.connection.Connection;
import com.vmware.vim25.ManagedObjectReference;
import com.vmware.vslm.RuntimeFaultFaultMsg;
import com.vmware.vslm.VslmPortType;
import com.vmware.vslm.VslmServiceInstanceContent;
import com.vmware.vslm.VslmTaskInfo;
import com.vmware.vslm.VslmTaskInfoState;

/*
 * This helper class can be used to fetch vslm port
 * which can intern be used to invode any api exposed
 * from vslm endpoint
 */
public class FcdVslmHelper {
	private Connection connection;
	private VslmServiceInstanceContent vslmServiceInstanceContent;
	private VslmPortType vslmPort;
	private ManagedObjectReference vStorageObjMgrMor;
	private VslmConnection vslmConnection;
	public static final int VSLM_TASK_SLEEP_INTERVAL = 5000;

	public FcdVslmHelper(Connection connection) throws Exception {
		this.connection = connection;
		this.vslmConnection = new VslmConnection(connection.getUrl(), connection.getUsername(),
				connection.getPassword());
		this.vslmPort = vslmConnection.getVslmPort();
		this.vslmServiceInstanceContent = getVslmServiceInstance();
		this.vStorageObjMgrMor = getVslmServiceInstance().getVStorageObjectManager();
	}

	public ManagedObjectReference getVStorageObjMgr() {
		return vStorageObjMgrMor;
	}

	public VslmPortType getVslmPort() {
		return vslmPort;
	}

	/*
	 * Utility method to get VslmstorageObjectManager
	 *
	 * @return ManagedObjectReference
	 */
	public static ManagedObjectReference getVslmMor() {
		ManagedObjectReference si = new ManagedObjectReference();
		si.setType("VslmServiceInstance");
		si.setValue("ServiceInstance");
		return si;
	}

	/*
	 * Utility method to get VslmServiceInstance
	 *
	 * @return ManagedObjectReference
	 */
	public VslmServiceInstanceContent getVslmServiceInstance() throws RuntimeFaultFaultMsg {
		return vslmPort.retrieveContent(getVslmMor());
	}

	/**
	 * Wait for a vslm task to complete.
	 *
	 * @param taskMor
	 * @return true if task succeeded, false otherwise.
	 * @throws Exception
	 */
	public boolean waitForTask(ManagedObjectReference taskMor) throws Exception {
		boolean taskSucceeded = false;
		int maxWaitForUpdateAttempts = 12;
		VslmTaskInfo vslmTaskInfo = vslmPort.vslmQueryInfo(taskMor);
		System.out.println("Task " + vslmTaskInfo.getKey() + " started at time " + vslmTaskInfo.getStartTime());
		int attempt = 0;
		while (attempt < maxWaitForUpdateAttempts) {
			vslmTaskInfo = vslmPort.vslmQueryInfo(taskMor);
			if (vslmTaskInfo.getState().value().equals(VslmTaskInfoState.SUCCESS.value())) {
				taskSucceeded = true;
				break;
			} else if (vslmTaskInfo.getState().value().equals(VslmTaskInfoState.ERROR.value())) {
				break;
			} else {
				++attempt;
				System.out.println("Vslm task status is: " + vslmTaskInfo.getState() + ", Waiting for "
						+ VSLM_TASK_SLEEP_INTERVAL + " mili secs for task to complete...");
				Thread.sleep(VSLM_TASK_SLEEP_INTERVAL);
			}
		}
		if (!taskSucceeded && (vslmTaskInfo.getState().value().equals(VslmTaskInfoState.RUNNING.value())
				|| vslmTaskInfo.getState().value().equals(VslmTaskInfoState.QUEUED.value()))) {
			int waitTime = VSLM_TASK_SLEEP_INTERVAL * attempt / 1000;
			System.out.println("VslmTaskInfo is still in " + vslmTaskInfo.getState() + " state even after waiting for "
					+ waitTime + " secs");
		}
		return taskSucceeded;
	}
}
