/*
 * *******************************************************
 * Copyright VMware, Inc. 2019-2022.  All Rights Reserved.
 * *******************************************************
 *
 * DISCLAIMER. THIS PROGRAM IS PROVIDED TO YOU "AS IS" WITHOUT
 * WARRANTIES OR CONDITIONS # OF ANY KIND, WHETHER ORAL OR WRITTEN,
 * EXPRESS OR IMPLIED. THE AUTHOR SPECIFICALLY # DISCLAIMS ANY IMPLIED
 * WARRANTIES OR CONDITIONS OF MERCHANTABILITY, SATISFACTORY # QUALITY,
 * NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE.
 */

package com.vmware.fcd;

import com.vmware.common.annotations.Action;
import com.vmware.common.annotations.Option;
import com.vmware.common.annotations.Sample;
import com.vmware.connection.ConnectedVimServiceBase;
import com.vmware.fcd.helpers.FcdHelper;
import com.vmware.vim25.FileFaultFaultMsg;
import com.vmware.vim25.InvalidCollectorVersionFaultMsg;
import com.vmware.vim25.InvalidDatastoreFaultMsg;
import com.vmware.vim25.InvalidPropertyFaultMsg;
import com.vmware.vim25.InvalidStateFaultMsg;
import com.vmware.vim25.LocalizedMethodFault;
import com.vmware.vim25.ManagedObjectReference;
import com.vmware.vim25.NotFoundFaultMsg;
import com.vmware.vim25.RuntimeFaultFaultMsg;
import com.vmware.vim25.TaskInProgressFaultMsg;
import com.vmware.vim25.TaskInfoState;
import com.vmware.vim25.VStorageObject;

/**
 * <pre>
 * FcdExtend
 *
 * This sample extends a virtual storage object capacity
 *
 * <b>Parameters:</b>
 * url                    [required] : url of the web service
 * username               [required] : username for the authentication
 * password               [required] : password for the authentication
 * vStorageObjectId       [required] : Id of the disk
 * datastorename          [required] : Name of the datastore
 * newCapacityInMB        [required] : The new capacity of the virtual disk in MB.
 *
 * <b>Command Line:</b>
 * Extend the capacity of a virtual disk to the new capacity
 * run.bat com.vmware.vm.FcdExtend --url [webserviceurl]
 * --username [username] --password [password]
 * --vStorageObjectId [vStorageObjectId] --datastorename [datastorename]
 * --newcapacityinmb [newcapacityinmb]
 * </pre>
 */
@Sample(name = "fcd-extend", description = "This sample extends a virtual"
        + " storage object(FCD).")
public class FcdExtend extends ConnectedVimServiceBase {

    private String vStorageObjectId;
    private String datastoreName;
	private long newCapacityInMB;

    /**
     * @param vStorageObjectId the vStorageObjectId to set
     */
    @Option(name = "vstorageobjectid",
            required = true,
            description = "Uuid of the vstorageobject.")
    public void setVStorageObjectId(String vStorageObjectId) {
        this.vStorageObjectId = vStorageObjectId;
    }

    /**
     * @param datastoreName the datastoreName to set
     */
    @Option(name = "datastorename",
            required = true,
            description = "Name of datastore.")
    public void setDatastoreName(String datastoreName) {
        this.datastoreName = datastoreName;
    }

    /**
     * @param newCapacityInMB the newCapacityInMB to set.
     */
    @Option(name="newcapacityinmb",
            required = true,
            description = "The new capacity of the virtual disk in MB.")
    public void setNewCapacityInMB(long newCapacityInMB) {
        this.newCapacityInMB = newCapacityInMB;
    }

    /**
     * This method returns a boolean value specifying whether the Task is
     * succeeded or failed.
     *
     * @param task
     *            ManagedObjectReference representing the Task.
     * @return boolean value representing the Task result.
     * @throws InvalidCollectorVersionFaultMsg
     * @throws RuntimeFaultFaultMsg
     * @throws InvalidPropertyFaultMsg
     */
    boolean getTaskResultAfterDone(ManagedObjectReference task)
            throws InvalidPropertyFaultMsg, RuntimeFaultFaultMsg,
            InvalidCollectorVersionFaultMsg {

        boolean retVal = false;

        // info has a property - state for state of the task
        Object[] result = waitForValues.wait(task, new String[] { "info.state",
                "info.error" }, new String[] { "state" },
                new Object[][] { new Object[] { TaskInfoState.SUCCESS,
                        TaskInfoState.ERROR } });

        if (result[0].equals(TaskInfoState.SUCCESS)) {
            retVal = true;
        }
        if (result[1] instanceof LocalizedMethodFault) {
            throw new RuntimeException(
                    ((LocalizedMethodFault) result[1]).getLocalizedMessage());
        }
        return retVal;
    }

    /**
     * Extend the capacity of a virtual disk to the new capacity
     *
     * @throws InvalidDatastoreFaultMsg
     * @throws RuntimeFaultFaultMsg
     * @throws InvalidPropertyFaultMsg
     * @throws InvalidCollectorVersionFaultMsg
     * @throws NotFoundFaultMsg
     * @throws TaskInProgressFaultMsg
     * @throws InvalidStateFaultMsg
     * @throws FileFaultFaultMsg
     */
	void extendvStorageObjectCapacity() throws
             InvalidDatastoreFaultMsg, RuntimeFaultFaultMsg,
             InvalidPropertyFaultMsg, InvalidCollectorVersionFaultMsg,
             NotFoundFaultMsg, FileFaultFaultMsg,
             InvalidStateFaultMsg, TaskInProgressFaultMsg {

		  // Get vStorageObjectManager Mor.
        ManagedObjectReference morVStrObjManager = connection
                .getServiceContent().getVStorageObjectManager();

        // Get all the input Mor's required for retrieving VStorageObject.
        ManagedObjectReference morDatastore = getMOREFs.inContainerByType(
                serviceContent.getRootFolder(), "Datastore")
                .get(datastoreName);

        if(morDatastore == null){
            throw new RuntimeException("The datastore name is not valid.");
        }

		// Extend capacity of VStorageObject
		System.out.println("Operation: Extending capacity of vStorageObject from vc.");
		ManagedObjectReference extendTaskMor = vimPort.extendDiskTask(morVStrObjManager,
				FcdHelper.makeId(vStorageObjectId), morDatastore, newCapacityInMB);

		if (getTaskResultAfterDone(extendTaskMor)) {
            System.out.printf(
                    "Success: Extended vStorageObject : %n [ Uuid = %s ]"
                    + "%n [ NewCapacity = %s ]%n",
                    vStorageObjectId, newCapacityInMB);
        } else {
            String msg = "Error: Extending vStorageObject [ "
                    + FcdHelper.makeId(vStorageObjectId) + "] from vc.";
            throw new RuntimeException(msg);
        }

        // Retrieve all the properties of a virtual storage objects based on the
        // Uuid of the vStorageObject and verify the new capacity
        System.out.println("Operation: Retrieve the extendedVStorageObject from"
                + " datastore from vc.");
        VStorageObject retrievedVStrObj = vimPort.retrieveVStorageObject(
                morVStrObjManager,FcdHelper.makeId(vStorageObjectId),
                morDatastore, null);
		if (retrievedVStrObj != null) {
            System.out.printf(
                    "Success: Retrieved vStorageObject :: [ %s ] from vc.%n",
                    retrievedVStrObj.getConfig().getId().getId());
        } else {
            String msg = "Error: Retrieving VStorageObject [ "
                    + vStorageObjectId + " ] from vc.";
            throw new RuntimeException(msg);
        }

        // Verify capacity of vStorageObject
        if (retrievedVStrObj.getConfig().getCapacityInMB() == newCapacityInMB) {
			System.out.println(
					"Success: Extend vStorageObject capacity  :: [ "
					+ retrievedVStrObj.getConfig().getId().getId()
					+ " ] from vc.");
        } else {
            String msg = "Error: VStorageObject [ "
                    + vStorageObjectId
                    + " ] capcity extend failed from vc.";
            throw new RuntimeException(msg);
        }
	}

	@Action
	public void run() throws InvalidDatastoreFaultMsg,
			RuntimeFaultFaultMsg, InvalidPropertyFaultMsg,
			InvalidCollectorVersionFaultMsg, NotFoundFaultMsg,
			FileFaultFaultMsg, InvalidStateFaultMsg, TaskInProgressFaultMsg {
		extendvStorageObjectCapacity();
	}
}
