/*
 * *******************************************************
 * Copyright VMware, Inc. 2019-2022.  All Rights Reserved.
 * *******************************************************
 *
 * DISCLAIMER. THIS PROGRAM IS PROVIDED TO YOU "AS IS" WITHOUT
 * WARRANTIES OR CONDITIONS # OF ANY KIND, WHETHER ORAL OR WRITTEN,
 * EXPRESS OR IMPLIED. THE AUTHOR SPECIFICALLY # DISCLAIMS ANY IMPLIED
 * WARRANTIES OR CONDITIONS OF MERCHANTABILITY, SATISFACTORY # QUALITY,
 * NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE.
 */

package com.vmware.fcd;

import com.vmware.common.annotations.Action;
import com.vmware.common.annotations.Option;
import com.vmware.common.annotations.Sample;
import com.vmware.connection.ConnectedVimServiceBase;
import com.vmware.fcd.helpers.FcdHelper;
import com.vmware.vim25.BaseConfigInfoBackingInfo;
import com.vmware.vim25.BaseConfigInfoDiskFileBackingInfo;
import com.vmware.vim25.BaseConfigInfoRawDiskMappingBackingInfo;
import com.vmware.vim25.FileFaultFaultMsg;
import com.vmware.vim25.InvalidCollectorVersionFaultMsg;
import com.vmware.vim25.InvalidDatastoreFaultMsg;
import com.vmware.vim25.InvalidPropertyFaultMsg;
import com.vmware.vim25.InvalidStateFaultMsg;
import com.vmware.vim25.LocalizedMethodFault;
import com.vmware.vim25.ManagedObjectReference;
import com.vmware.vim25.NotFoundFaultMsg;
import com.vmware.vim25.RuntimeFaultFaultMsg;
import com.vmware.vim25.TaskInProgressFaultMsg;
import com.vmware.vim25.TaskInfoState;
import com.vmware.vim25.VStorageObject;

/**
 * <pre>
 * FcdDelete
 *
 * This sample inflates a sparse or thin-provisioned virtual disk up to the full size.
 *
 * <b>Parameters:</b>
 * url                    [required] : url of the web service
 * username               [required] : username for the authentication
 * password               [required] : password for the authentication
 * vstorageobjectid       [required] : Uuid of the vstorageobject
 * datastorename          [required] : Name of the datastore which contains
 *                                     the virtual storage object
 *
 * <b>Command Line:</b>
 * Inflate a sparse or thin-provisioned virtual disk up to the full size.
 * run.bat com.vmware.vm.FcdInflateDisk --url [webserviceurl]
 * --username [username] --password [password]
 * --vstorageobjectid [vstorageobjectid] --datastorename [datastorename]
 * </pre>
 */
@Sample(name = "fcd-inflatedisk", description = "This sample inflates a sparse"
		+ " or thin-provisioned virtual disk up to the full size.")
public class FcdInflateDisk extends ConnectedVimServiceBase {

    private String vStorageObjectId;
    private String datastoreName;

    /**
     * @param vStorageObjectId the vStorageObjectId to set
     */
    @Option(name = "vstorageobjectid",
            required = true,
            description = "Uuid of the vstorageobject.")
    public void setVStorageObjectId(String vStorageObjectId) {
        this.vStorageObjectId = vStorageObjectId;
    }

    /**
     * @param datastoreName the datastoreName to set
     */
    @Option(name = "datastorename",
            required = true,
            description = "Name of datastore containing the  vstorageobject.")
    public void setDatastoreName(String datastoreName) {
        this.datastoreName = datastoreName;
    }

    /**
     * This method returns a boolean value specifying whether the Task is
     * succeeded or failed.
     *
     * @param task
     *            ManagedObjectReference representing the Task.
     * @return boolean value representing the Task result.
     * @throws InvalidCollectorVersionFaultMsg
     *
     * @throws RuntimeFaultFaultMsg
     * @throws InvalidPropertyFaultMsg
     */
    boolean getTaskResultAfterDone(ManagedObjectReference task)
            throws InvalidPropertyFaultMsg, RuntimeFaultFaultMsg,
            InvalidCollectorVersionFaultMsg {

        boolean retVal = false;

        // info has a property - state for state of the task
        Object[] result = waitForValues.wait(task, new String[] { "info.state",
                "info.error" }, new String[] { "state" },
                new Object[][] { new Object[] { TaskInfoState.SUCCESS,
                        TaskInfoState.ERROR } });

        if (result[0].equals(TaskInfoState.SUCCESS)) {
            retVal = true;
        }
        if (result[1] instanceof LocalizedMethodFault) {
            throw new RuntimeException(
                    ((LocalizedMethodFault) result[1]).getLocalizedMessage());
        }
        return retVal;
    }

    /**
     * Inflate a sparse or thin-provisioned virtual disk up to the full size.
     *
     * @throws RuntimeFaultFaultMsg
     * @throws InvalidDatastoreFaultMsg
     * @throws FileFaultFaultMsg
     * @throws InvalidCollectorVersionFaultMsg
     * @throws InvalidPropertyFaultMsg
     * @throws TaskInProgressFaultMsg
     * @throws NotFoundFaultMsg
     * @throws InvalidStateFaultMsg
     */
    void inflateVStorageObject() throws InvalidPropertyFaultMsg,
            RuntimeFaultFaultMsg, FileFaultFaultMsg, InvalidDatastoreFaultMsg,
            InvalidStateFaultMsg, NotFoundFaultMsg, TaskInProgressFaultMsg,
            InvalidCollectorVersionFaultMsg {
        // Get vStorageObjectManager Mor.
        ManagedObjectReference morVStrObjManager = connection
                .getServiceContent().getVStorageObjectManager();

        // Get all the input Mor's required for retrieving VStorageObject.
        ManagedObjectReference morDatastore = getMOREFs.inContainerByType(
                serviceContent.getRootFolder(), "Datastore")
                .get(datastoreName);

        if(morDatastore == null){
            throw new RuntimeException("The datastore name is not valid.");
        }

		// Retrieve vStorageObject before inflate disk
		VStorageObject vStorageObjectBeforeInflate = vimPort
				.retrieveVStorageObject(morVStrObjManager,
				FcdHelper.makeId(vStorageObjectId), morDatastore, null);

		if (vStorageObjectBeforeInflate != null) {
            System.out.printf(
                    "Success: Retrieved vStorageObject before inflate"
                    + " :: [ %s ] from vc.%n", vStorageObjectId);
        } else {
            String msg = "Error: Retrieving VStorageObject [ "
                    + vStorageObjectId + " ] before inflate";
            throw new RuntimeException(msg);
        }

        // Inflate a sparse or thin-provisioned virtual disk up to the full size.
		System.out.println("Operation: Inflate a sparse or thin-provisioned"
        + " virtual disk up to the full size from vc.");
        ManagedObjectReference inflateDiskTaskMor = vimPort.inflateDiskTask(
                morVStrObjManager, FcdHelper.makeId(vStorageObjectId),
                morDatastore);
        if (getTaskResultAfterDone(inflateDiskTaskMor)) {
            System.out.printf(
                    "Success: Inflated vStorageObject : [ Uuid = %s ]"
                    + " from vc.%n", vStorageObjectId);
        } else {
            String msg = "Error: Inflating vStorageObject [ "
					+ vStorageObjectId + " ] from vc.";
            throw new RuntimeException(msg);
        }

        // Retrieve vStorageObject after inflate disk
		VStorageObject inflatedDisk = vimPort
				.retrieveVStorageObject(morVStrObjManager,
						FcdHelper.makeId(vStorageObjectId), morDatastore, null);

		if (inflatedDisk != null) {
            System.out.printf(
                    "Success: Retrieved vStorageObject after inflate "
                    + ":: [ %s ] from vc.%n", vStorageObjectId);
        } else {
            String msg = "Error: Retrieving VStorageObject [ "
                    + vStorageObjectId + " ] after inflate from vc.";
            throw new RuntimeException(msg);
        }
    }

    @Action
    public void run() throws InvalidPropertyFaultMsg, RuntimeFaultFaultMsg,
            FileFaultFaultMsg, InvalidDatastoreFaultMsg, InvalidStateFaultMsg,
            NotFoundFaultMsg, TaskInProgressFaultMsg, InvalidCollectorVersionFaultMsg {
        inflateVStorageObject();
    }
}
