/*
 * *******************************************************
 * Copyright VMware, Inc. 2019-2022.  All Rights Reserved.
 * *******************************************************
 *
 * DISCLAIMER. THIS PROGRAM IS PROVIDED TO YOU "AS IS" WITHOUT
 * WARRANTIES OR CONDITIONS # OF ANY KIND, WHETHER ORAL OR WRITTEN,
 * EXPRESS OR IMPLIED. THE AUTHOR SPECIFICALLY # DISCLAIMS ANY IMPLIED
 * WARRANTIES OR CONDITIONS OF MERCHANTABILITY, SATISFACTORY # QUALITY,
 * NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE.
 */

package com.vmware.fcd;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.vmware.common.annotations.Action;
import com.vmware.common.annotations.Before;
import com.vmware.common.annotations.Option;
import com.vmware.common.annotations.Sample;
import com.vmware.connection.ConnectedVimServiceBase;
import com.vmware.fcd.helpers.FcdHelper;
import com.vmware.vim25.BaseConfigInfoDiskFileBackingInfoProvisioningType;
import com.vmware.vim25.FileFaultFaultMsg;
import com.vmware.vim25.HostScsiDisk;
import com.vmware.vim25.ID;
import com.vmware.vim25.InvalidCollectorVersionFaultMsg;
import com.vmware.vim25.InvalidDatastoreFaultMsg;
import com.vmware.vim25.InvalidPropertyFaultMsg;
import com.vmware.vim25.InvalidStateFaultMsg;
import com.vmware.vim25.LocalizedMethodFault;
import com.vmware.vim25.ManagedObjectReference;
import com.vmware.vim25.NotFoundFaultMsg;
import com.vmware.vim25.RuntimeFaultFaultMsg;
import com.vmware.vim25.TaskInfoState;
import com.vmware.vim25.VStorageObject;
import com.vmware.vim25.VirtualDiskCompatibilityMode;
import com.vmware.vim25.VslmCreateSpecBackingSpec;
import com.vmware.vim25.VslmCreateSpecDiskFileBackingSpec;
import com.vmware.vim25.VslmCreateSpecRawDiskMappingBackingSpec;
import com.vmware.vim25.VslmRelocateSpec;

/**
 * <pre>
 * FcdRelocate
 *
 * This sample relocates a virtual storage object.
 *
 * <b>Parameters:</b>
 * url                    [required] : url of the web service
 * username               [required] : username for the authentication
 * password               [required] : password for the authentication
 * vstorageobjectid       [required] : Uuid of the vstorageobject
 * datastorename          [required] : Name of the datastore which contains
 *                                     the virtual storage object
 * destDatastoreName      [required] : Name of destination datastore
 * provisioningtype       [optional] : Type of provisioning for the disk
 *                                     [thin | eagerZeroedThick |
 *                                     lazyZeroedThick | virtualMode |
 *                                     physicalMode]
 * hostscsidisk           [optional] : Host Scsi disk to clone vStorageObject
 *
 * <b>Command Line:</b>
 * Relocate a virtual storage object.
 * run.bat com.vmware.vm.FcdRelocate --url [webserviceurl]
 * --username [username] --password [password]
 * --vstorageobjectid [vstorageobjectid] --datastorename [datastorename]
 * </pre>
 */
@Sample(name = "fcd-relocate", description = "This sample relocates"
		+ " a virtual storage object.")
public class FcdRelocate extends ConnectedVimServiceBase {

    private String vStorageObjectId;
    private String datastoreName;
    private String destDatastoreName;
    private String provisioningType = "thin"; // Default
	private HostScsiDisk hostScsiDisk = null; // Default
    private final Map<String, DiskProvisioningTypes> provisioningTypeHashMap =
            new HashMap<String, DiskProvisioningTypes>();

    private static enum DiskProvisioningTypes {
        THIN("thin"),
        EAGER_ZEROED_THICK("eagerZeroedThick"),
        LAZY_ZEROED_THICK("lazyZeroedThick"),
        VIRTUAL_MODE("virtualMode"),
        PHYSICAL_MODE("physicalMode");

        private final String value;

        private DiskProvisioningTypes(String value) {
            this.value = value;
        }

        public String value() {
            return value;
        }
    }

    @Before
    public void init() {
        provisioningTypeHashMap.put("thin", DiskProvisioningTypes.THIN);
        provisioningTypeHashMap.put("eagerzeroedthick",
                DiskProvisioningTypes.EAGER_ZEROED_THICK);
        provisioningTypeHashMap.put("lazyzeroedthick",
                DiskProvisioningTypes.LAZY_ZEROED_THICK);
        provisioningTypeHashMap.put("virtualmode",
                DiskProvisioningTypes.VIRTUAL_MODE);
        provisioningTypeHashMap.put("physicalmode",
                DiskProvisioningTypes.PHYSICAL_MODE);
    }

    /**
     * @param vStorageObjectId the vStorageObjectId to set
     */
    @Option(name = "vstorageobjectid",
            required = true,
            description = "Uuid of the vstorageobject.")
    public void setVStorageObjectId(String vStorageObjectId) {
        this.vStorageObjectId = vStorageObjectId;
    }

    /**
     * @param datastoreName the datastoreName to set
     */
    @Option(name = "datastorename",
            required = true,
            description = "Name of destination datastore for relocation.")
    public void setDatastoreName(String datastoreName) {
        this.datastoreName = datastoreName;
    }

    /**
     * @param destDatastoreName the destDatastoreName to set
     */
    @Option(name = "destdatastoredame",
            required = true,
            description = "Name of destination datastore.")
    public void setDestDatastoreName(String destDatastoreName) {
        this.destDatastoreName = destDatastoreName;
    }

    /**
     * @param provisioningType the provisioningType to set
     */
    @Option(name = "provisioningtype",
            required = false,
            description = "Provisioning Type of the Disk.\n [thin | " +
            "eagerZeroedThick | lazyZeroedThick | virtualMode | physicalMode]")
    public void setProvisioningType(String provisioningType) {
        this.provisioningType = provisioningType;
    }

    /**
     * @param hostScsiDisk the hostScsiDisk to set
     */
    @Option(name = "hostScsiDisk",
            required = false,
            description = "Uuid of the vstorageobject.")
    public void setHostScsiDisk(HostScsiDisk hostScsiDisk) {
        this.hostScsiDisk = hostScsiDisk;
    }

    /**
     * This method returns a boolean value specifying whether the Task is
     * succeeded or failed.
     *
     * @param task
     *            ManagedObjectReference representing the Task.
     * @return boolean value representing the Task result.
     * @throws InvalidCollectorVersionFaultMsg
     *
     * @throws RuntimeFaultFaultMsg
     * @throws InvalidPropertyFaultMsg
     */
    boolean getTaskResultAfterDone(ManagedObjectReference task)
            throws InvalidPropertyFaultMsg, RuntimeFaultFaultMsg,
            InvalidCollectorVersionFaultMsg {

        boolean retVal = false;

        // info has a property - state for state of the task
        Object[] result = waitForValues.wait(task, new String[] { "info.state",
                "info.error" }, new String[] { "state" },
                new Object[][] { new Object[] { TaskInfoState.SUCCESS,
                        TaskInfoState.ERROR } });

        if (result[0].equals(TaskInfoState.SUCCESS)) {
            retVal = true;
        }
        if (result[1] instanceof LocalizedMethodFault) {
            throw new RuntimeException(
                    ((LocalizedMethodFault) result[1]).getLocalizedMessage());
        }
        return retVal;
    }

    /**
     * Relocate a virtual storage object.
     *
     * @throws RuntimeFaultFaultMsg
     * @throws InvalidDatastoreFaultMsg
     * @throws FileFaultFaultMsg
     * @throws InvalidCollectorVersionFaultMsg
     * @throws InvalidPropertyFaultMsg
     * @throws NotFoundFaultMsg
     * @throws InvalidStateFaultMsg
     */
    void relocateVStorageObject() throws InvalidPropertyFaultMsg,
            RuntimeFaultFaultMsg, FileFaultFaultMsg, InvalidDatastoreFaultMsg,
            InvalidStateFaultMsg, NotFoundFaultMsg,
            InvalidCollectorVersionFaultMsg {
        // Get vStorageObjectManager Mor.
        ManagedObjectReference morVStrObjManager = connection
                .getServiceContent().getVStorageObjectManager();

        // Init provisioning types:
        DiskProvisioningTypes diskProvisioningType = provisioningTypeHashMap
                .get(provisioningType.trim().toLowerCase());

        if(diskProvisioningType == null){
            throw new RuntimeException("The input provisioning Type is not valid.");
        }
        // Get all the input Mor's required for creating VStorageObject.
        ManagedObjectReference morDatastore = getMOREFs.inContainerByType(
                serviceContent.getRootFolder(), "Datastore").get(datastoreName);

        if(morDatastore == null){
            throw new RuntimeException("The datastore name is not valid.");
        }

        ManagedObjectReference morDestDatastore = getMOREFs.inContainerByType(
                serviceContent.getRootFolder(), "Datastore").get(destDatastoreName);

        if(morDestDatastore == null){
            throw new RuntimeException("The datastore name is not valid.");
        }

        // Create a relocate spec for VStorageObject
        VslmRelocateSpec vslmRelocateSpec = generateVslmRelocateSpec(morDestDatastore,
                diskProvisioningType);

        System.out.println("Operation: Relocating a vStorageObject from vc.");
        ManagedObjectReference relocateDiskTaskMor = vimPort.relocateVStorageObjectTask(
				morVStrObjManager, FcdHelper.makeId(vStorageObjectId),
				morDatastore, vslmRelocateSpec);
        VStorageObject relocatedVStorageObject = null;
        if (getTaskResultAfterDone(relocateDiskTaskMor)) {
			relocatedVStorageObject = (VStorageObject) getMOREFs.entityProps(
					relocateDiskTaskMor, new String[] { "info.result" }).get(
					"info.result");
            System.out.printf(
                    "Success: Relocated vStorageObject :  %n [ Name = %s ]"
                            + " %n [ Uuid = %s ] from source datastore"
                            + " %n [ datastore = %s ] %n to"
                            + " destination datastore [ datastore = %s ].%n",
                    relocatedVStorageObject.getConfig().getName(),
                    relocatedVStorageObject.getConfig().getId().getId(),
                    morDatastore.getValue(), morDestDatastore.getValue());
        } else {
            String msg = "Error: Relocating [ " + FcdHelper.makeId(vStorageObjectId)
                    + "] vStorageObject to [ datastore = %s ] from vc." + morDatastore;
            throw new RuntimeException(msg);
        }

        // Retrieve a list of all the virtual storage objects in given datastore
        // and verify if the created vStorageObject is present in the returned list.
        System.out.println("Operation: List all vStorageObjects in datastore from vc.");
        List<ID> listOfVStrObj = vimPort.listVStorageObject(morVStrObjManager,
				morDestDatastore);

        if (FcdHelper.isFcdIdInFcdList(Arrays.asList(relocatedVStorageObject.getConfig()
                .getId().getId()), listOfVStrObj)) {
            System.out.printf(
                    "Success: listVStorageObject contains the created"
                            + " vStorageObjectId : [ %s ] from vc.%n",
                    relocatedVStorageObject.getConfig().getId().getId());
        } else {
            String msg = "Error: Created VStorageObject [ "
                    + relocatedVStorageObject.getConfig().getId().getId()
                    + "] is not present in the returned list from vc.";
            throw new RuntimeException(msg);
        }

        // Retrieve all the properties of a virtual storage objects based on the
        // Uuid of the vStorageObject obtained from the list call.
        System.out.println("Operation: Retrieve the  createdVStorageObjects in"
                + " datastore from vc.");
        VStorageObject retrievedVStrObj = vimPort.retrieveVStorageObject(
                morVStrObjManager, relocatedVStorageObject.getConfig().getId(),
                morDestDatastore, null);
        if (retrievedVStrObj.getConfig().getId().getId()
                .equals(relocatedVStorageObject.getConfig().getId().getId())) {
            System.out.printf(
                    "Success: Retrieved vStorageObject :: [ %s ] from vc.",
                    retrievedVStrObj.getConfig().getId().getId());
        } else {
            String msg = "Error: Created VStorageObject [ "
                    + relocatedVStorageObject.getConfig().getId().getId()
                    + " ] and retrieved VStorageObject are different from vc.";
            throw new RuntimeException(msg);
        }
    }

    /**
     * This method constructs a VslmCloneSpec for the vStorageObject
     *
     * @param dsMor The ManagedObjectReferece of the datastore
     * @param provisioningType The provisioningType of the disk
     * @return VslmCreateSpec
     * @throws IllegalArgumentException
     */
    public VslmRelocateSpec generateVslmRelocateSpec(ManagedObjectReference dsMor,
            DiskProvisioningTypes provisioningType)
            throws IllegalArgumentException {
        System.out.println("Info :: Creating VslmCloneSpec with dsMor: "
                + dsMor.getValue() + " provisioningType:"
                + provisioningType.toString());
        VslmCreateSpecBackingSpec backingSpec;

        if (provisioningType != DiskProvisioningTypes.VIRTUAL_MODE
                && provisioningType != DiskProvisioningTypes.PHYSICAL_MODE) {
			VslmCreateSpecDiskFileBackingSpec diskFileBackingSpec =
                    new VslmCreateSpecDiskFileBackingSpec();
            diskFileBackingSpec.setDatastore(dsMor);
            diskFileBackingSpec.setProvisioningType(
                    BaseConfigInfoDiskFileBackingInfoProvisioningType
                            .valueOf(provisioningType.toString()).value());
            backingSpec = diskFileBackingSpec;
        } else {
            VslmCreateSpecRawDiskMappingBackingSpec rdmBackingSpec =
                    new VslmCreateSpecRawDiskMappingBackingSpec();
            rdmBackingSpec.setDatastore(dsMor);
            rdmBackingSpec.setCompatibilityMode(VirtualDiskCompatibilityMode
                    .valueOf(provisioningType.toString()).value());
            rdmBackingSpec.setLunUuid(hostScsiDisk.getUuid());
            backingSpec = rdmBackingSpec;
        }
        VslmRelocateSpec relocateSpec = new VslmRelocateSpec();
        relocateSpec.setBackingSpec(backingSpec);
        return relocateSpec;
    }

    @Action
    public void run() throws InvalidPropertyFaultMsg, RuntimeFaultFaultMsg,
            FileFaultFaultMsg, InvalidDatastoreFaultMsg, InvalidStateFaultMsg,
            NotFoundFaultMsg, InvalidCollectorVersionFaultMsg {
		init();
        relocateVStorageObject();
    }
}
