/*
 * *******************************************************
 * Copyright VMware, Inc. 2019-2022.  All Rights Reserved.
 * *******************************************************
 *
 * DISCLAIMER. THIS PROGRAM IS PROVIDED TO YOU "AS IS" WITHOUT
 * WARRANTIES OR CONDITIONS # OF ANY KIND, WHETHER ORAL OR WRITTEN,
 * EXPRESS OR IMPLIED. THE AUTHOR SPECIFICALLY # DISCLAIMS ANY IMPLIED
 * WARRANTIES OR CONDITIONS OF MERCHANTABILITY, SATISFACTORY # QUALITY,
 * NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE.
 */

package com.vmware.fcd;

import com.vmware.common.annotations.Action;
import com.vmware.common.annotations.Option;
import com.vmware.common.annotations.Sample;
import com.vmware.connection.ConnectedVimServiceBase;
import com.vmware.fcd.helpers.FcdHelper;
import com.vmware.vim25.FileFaultFaultMsg;
import com.vmware.vim25.InvalidDatastoreFaultMsg;
import com.vmware.vim25.InvalidPropertyFaultMsg;
import com.vmware.vim25.ManagedObjectReference;
import com.vmware.vim25.NotFoundFaultMsg;
import com.vmware.vim25.RuntimeFaultFaultMsg;
import com.vmware.vim25.VStorageObject;

/**
 * <pre>
 * FcdRename
 *
 * This sample renames a given virtual storage object.
 *
 * <b>Parameters:</b>
 * url                    [required] : url of the web service
 * username               [required] : username for the authentication
 * password               [required] : password for the authentication
 * vstorageobjectid       [required] : Uuid of the vstorageobject
 * datastorename          [required] : Name of the datastore which contains
 *                                     the virtual storage object
 * newvstorageobjectname  [required] : The new name for the virtual storage object.
 *
 * <b>Command Line:</b>
 * Rename a virtual storage object.
 * run.bat com.vmware.vm.Rename --url [webserviceurl]
 * --username [username] --password [password]
 * --vstorageobjectid [vstorageobjectid] --datastorename [datastorename]
 * --newvstorageobjectname [newvstorageobjectname]
 * </pre>
 */
@Sample(name = "fcd-rename", description = "This sample renames a virtual"
        + " storage object.")
public class FcdRename extends ConnectedVimServiceBase {

    private String vStorageObjectId;
    private String datastoreName;
    private String newVStorageObjectName;

    /**
     * @param vStorageObjectId the vStorageObjectId to set
     */
    @Option(name = "vstorageobjectid",
            required = true,
            description = "Uuid of the vstorageobject.")
    public void setVStorageObjectId(String vStorageObjectId) {
        this.vStorageObjectId = vStorageObjectId;
    }

    /**
     * @param datastoreName the datastoreName to set
     */
    @Option(name = "datastorename",
            required = true,
            description = "Name of datastore containing the  vstorageobject.")
    public void setDatastoreName(String datastoreName) {
        this.datastoreName = datastoreName;
    }

    /**
     * @param newVStorageObjectName the newVStorageObjectName to set
     */
    @Option(name = "newvstorageobjectname",
            required = true,
            description = "The new name for the virtual storage object.")
    public void setNewVStorageObjectName(String newVStorageObjectName) {
        this.newVStorageObjectName = newVStorageObjectName;
    }

    /**
     * Rename a virtual storage object.
     *
     * @throws RuntimeFaultFaultMsg
     * @throws InvalidDatastoreFaultMsg
     * @throws FileFaultFaultMsg
     * @throws InvalidPropertyFaultMsg
     * @throws NotFoundFaultMsg
     */
    void renameVStorageObject() throws InvalidPropertyFaultMsg,
            RuntimeFaultFaultMsg, FileFaultFaultMsg, InvalidDatastoreFaultMsg,
            NotFoundFaultMsg {
        // Get vStorageObjectManager Mor.
        ManagedObjectReference morVStrObjManager = connection
                .getServiceContent().getVStorageObjectManager();

        // Get all the input Mor's required for retrieving VStorageObject.
        ManagedObjectReference morDatastore = getMOREFs.inContainerByType(
                serviceContent.getRootFolder(), "Datastore")
                .get(datastoreName);

        if(morDatastore == null){
            throw new RuntimeException("The datastore name is not valid.");
        }

        // Retrieve vStorageObject before renaming
        System.out.println("Operation: Retrieve vStorageObject before renaming"
				+ " from datastore from vc.");
        VStorageObject retrievedVStrObjBeforeRename = vimPort.retrieveVStorageObject(
                morVStrObjManager, FcdHelper.makeId(vStorageObjectId),
                morDatastore, null);
		if (retrievedVStrObjBeforeRename != null) {
            System.out.printf(
                    "Success: Retrieved vStorageObject :: %n [ UUid = %s ]"
                    + " %n [ name = %s ]%n ",
                    retrievedVStrObjBeforeRename.getConfig().getId().getId(),
                    retrievedVStrObjBeforeRename.getConfig().getName());
        } else {
            String msg = "Error: Retrieving VStorageObject [ "
                    + FcdHelper.makeId(vStorageObjectId);
            throw new RuntimeException(msg);
        }

        // Raname a vStorageObject:
        System.out.println("Operation: Renaming the given vStorageObject from vc.");
        vimPort.renameVStorageObject(morVStrObjManager,
				FcdHelper.makeId(vStorageObjectId),
				morDatastore, newVStorageObjectName);

        // Retrieve vStorageObject after renaming
        System.out.println("Operation: Retrieve the vStorageObject after renaming"
				+ " from datastore from vc.");
        VStorageObject retrievedVStrObjAfterRename = vimPort.retrieveVStorageObject(
                morVStrObjManager, FcdHelper.makeId(vStorageObjectId),
                morDatastore, null);
		if (retrievedVStrObjAfterRename != null) {
            System.out.printf(
                    "Success: Retrieved vStorageObject :: %n [ UUid = %s ]"
                    + "%n [ name = %s ] %n",
                    retrievedVStrObjAfterRename.getConfig().getId().getId(),
                    retrievedVStrObjAfterRename.getConfig().getName());
        } else {
            String msg = "Error: Retrieving VStorageObject [ "
                    + FcdHelper.makeId(vStorageObjectId);
            throw new RuntimeException(msg);
        }

		// Verify rename vStorageObject
		if (retrievedVStrObjAfterRename.getConfig().getName().equals(newVStorageObjectName)) {
            System.out.printf(
                    "Success: Renamed vStorageObject :: [ %s ] from vc",
                    retrievedVStrObjAfterRename.getConfig().getId().getId());
        } else {
            String msg = "Error: VStorageObject [ "
                    + retrievedVStrObjAfterRename.getConfig().getId().getId()
                    + "] rename to [ " + newVStorageObjectName + " ] from vc";
            throw new RuntimeException(msg);
		}
    }

    @Action
    public void run() throws InvalidPropertyFaultMsg, RuntimeFaultFaultMsg,
            FileFaultFaultMsg, InvalidDatastoreFaultMsg,
            NotFoundFaultMsg {
        renameVStorageObject();
    }
}
