/*
 * ******************************************************
 * Copyright VMware, Inc. 2010-2020.  All Rights Reserved.
 * ******************************************************
 *
 * DISCLAIMER. THIS PROGRAM IS PROVIDED TO YOU "AS IS" WITHOUT
 * WARRANTIES OR CONDITIONS # OF ANY KIND, WHETHER ORAL OR WRITTEN,
 * EXPRESS OR IMPLIED. THE AUTHOR SPECIFICALLY # DISCLAIMS ANY IMPLIED
 * WARRANTIES OR CONDITIONS OF MERCHANTABILITY, SATISFACTORY # QUALITY,
 * NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE.
 */

package com.vmware.host;

import com.vmware.common.annotations.Action;
import com.vmware.common.annotations.Option;
import com.vmware.common.annotations.Sample;
import com.vmware.connection.ConnectedVimServiceBase;
import com.vmware.vim25.*;

import java.lang.reflect.InvocationTargetException;
import java.util.Map;

/**
 * <pre>
 * AddVirtualNic
 *
 * This sample is used to add a Virtual Nic to a PortGroup or a DVPortGroup
 *
 * <b>Parameters:</b>
 * url              [required] : url of the web service
 * username         [required] : username for the authentication
 * password         [required] : password for the authentication
 * portgroupname    [required] : Name of the port group
 * ipaddress        [optional] : ipaddress for the nic, if not set DHCP
 *                               will be in affect for the nic
 * hostname         [required] : Name of the host
 *
 * <b>Command Line:</b>
 * Add VirtualNic to a PortGroup on a Virtual Switch
 * run.bat com.vmware.host.AddVirtualNic --url [webserviceurl]
 * --username [username] --password  [password]
 * --hostname [hostname]
 * --portgroupname [myportgroup] --ipaddress [AAA.AAA.AAA.AAA]
 * </pre>
 */
@Sample(name = "add-virtual-nic", description = "This sample is used to add a Virtual Nic to a PortGroup")
public class AddVirtualNic extends ConnectedVimServiceBase {
    private ManagedObjectReference rootFolder;
    private ManagedObjectReference propCollectorRef;

    String hostname;
    String portgroupname;
    String ipaddress;

    @Option(name = "portgroupname", required = true, description = "Name of the port group")
    public void setPortgroupname(String portgroupname) {
        this.portgroupname = portgroupname;
    }

    @Option(
            name = "ipaddress",
            required = false,
            description = "ipaddress for the nic, if not set DHCP will be in affect for the nic"
    )
    public void setIpaddress(String ipaddress) {
        this.ipaddress = ipaddress;
    }

    @Option(name = "hostname", required = false, description = "Name of the host")
    public void setHostname(String hostname) {
        this.hostname = hostname;
    }

    void init() {
        propCollectorRef = serviceContent.getPropertyCollector();
        rootFolder = serviceContent.getRootFolder();
    }

    HostVirtualNicSpec createVirtualNicSpecification(ManagedObjectReference portGroupMor) throws InvalidPropertyFaultMsg, RuntimeFaultFaultMsg {
        HostIpConfig hipconfig = new HostIpConfig();
        if (ipaddress != null && !ipaddress.isEmpty()) {
            hipconfig.setDhcp(Boolean.FALSE);
            hipconfig.setIpAddress(ipaddress);
            hipconfig.setSubnetMask("255.255.255.0");
        } else {
            hipconfig.setDhcp(Boolean.TRUE);
        }
        HostVirtualNicSpec hvnicspec = new HostVirtualNicSpec();
        hvnicspec.setIp(hipconfig);

        if (portGroupMor.getType().equals("DistributedVirtualPortgroup")) {
            Map<String, Object> result = null;
            result = getMOREFs.entityProps(portGroupMor,
                    new String[]{"config.distributedVirtualSwitch", "key"});

            ManagedObjectReference dvsMor = (ManagedObjectReference) result.get("config.distributedVirtualSwitch");
            String dvsUuid = (String) getMOREFs.entityProps(dvsMor, new String[] {"uuid"}).get("uuid");

            DistributedVirtualSwitchPortConnection dvsConn = new DistributedVirtualSwitchPortConnection();
            dvsConn.setPortgroupKey((String) result.get("key"));
            dvsConn.setSwitchUuid(dvsUuid);
            hvnicspec.setDistributedVirtualPort(dvsConn);
        }
        return hvnicspec;

    }

    void addVirtualNIC() throws HostConfigFaultFaultMsg, AlreadyExistsFaultMsg, InvalidStateFaultMsg, InvalidPropertyFaultMsg, RuntimeFaultFaultMsg {
        ManagedObjectReference hostmor = null;
        Map<String, ManagedObjectReference> hostResults = getMOREFs.inFolderByType(
                serviceContent.getRootFolder(), "HostSystem", new RetrieveOptions());
        hostmor = hostResults.get(hostname);

        if (hostmor == null) {
            System.out.println("Host "+ hostname +" not found");
            return;

        }
        ArrayOfManagedObjectReference Arr = (ArrayOfManagedObjectReference) getMOREFs.entityProps(hostmor,
                new String[] {"network"}).get("network");
        ManagedObjectReference targetPortgroupMor = null;
        for (ManagedObjectReference portGroup : Arr.getManagedObjectReference()) {
            if (portgroupname.equals(getMOREFs.entityProps(portGroup,
                    new String[] {"name"}).get("name"))) {
                targetPortgroupMor = portGroup;
            }
        }
        if (targetPortgroupMor == null) {
            System.out.println("Portgroup "+ portgroupname + " not found");
            return;
        }

        HostConfigManager configMgr = (HostConfigManager) getMOREFs.entityProps(hostmor,
                new String[] { "configManager" }).get("configManager");
        ManagedObjectReference nwSystem = configMgr.getNetworkSystem();
        HostPortGroupSpec portgrp = new HostPortGroupSpec();
        portgrp.setName(portgroupname);
        HostVirtualNicSpec vNicSpec = createVirtualNicSpecification(targetPortgroupMor);
        String vnic = null;
        if (targetPortgroupMor.getType().equals("DistributedVirtualPortgroup")) {
             vnic = vimPort.addVirtualNic(nwSystem, "", vNicSpec);
        } else {
             vnic = vimPort.addVirtualNic(nwSystem, portgroupname, vNicSpec);
        }
        System.out.println("Successful in creating nic : " + vnic
                + " with PortGroup :" + portgroupname);
    }

    @Action
    public void run() throws RuntimeFaultFaultMsg, AlreadyExistsFaultMsg, InvalidStateFaultMsg, InvocationTargetException, InvalidPropertyFaultMsg, NoSuchMethodException, IllegalAccessException, HostConfigFaultFaultMsg {
        init();
        addVirtualNIC();
    }
}
