/*
 * ******************************************************
 * Copyright VMware, Inc. 2010-2012,2019,2022.  All Rights Reserved.
 * ******************************************************
 *
 * DISCLAIMER. THIS PROGRAM IS PROVIDED TO YOU "AS IS" WITHOUT
 * WARRANTIES OR CONDITIONS # OF ANY KIND, WHETHER ORAL OR WRITTEN,
 * EXPRESS OR IMPLIED. THE AUTHOR SPECIFICALLY # DISCLAIMS ANY IMPLIED
 * WARRANTIES OR CONDITIONS OF MERCHANTABILITY, SATISFACTORY # QUALITY,
 * NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE.
 */

package com.vmware.vm;

import com.vmware.common.annotations.Action;
import com.vmware.common.annotations.Option;
import com.vmware.common.annotations.Sample;
import com.vmware.connection.ConnectedVimServiceBase;
import com.vmware.vim25.*;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;

/**
 * <pre>
 * VMClone
 *
 * This sample makes a template of an existing VM , change target network
 * deploy multiple instances of this template onto a datacenter
 *
 * <b>Parameters:</b>
 * url             [required] : url of the web service
 * username        [required] : username for the authentication
 * password        [required] : password for the authentication
 * datacentername  [required] : name of Datacenter
 * vmpath          [required] : inventory path of the VM
 * clonename       [required] : name of the clone
 * hostname        [optional] : name of the host
 * targetnetworkname [optional] : name of the target network
 *
 * <b>Command Line:</b>
 * java com.vmware.vm.VMClone --url [webserviceurl]
 * --username [username] --password [password]
 * --datacentername [DatacenterName]"
 * --vmpath [vmPath] --clonename [CloneName] --host [hostname] 
 * --targetnetworkname [Target Network Name]
 * </pre>
 */
@Sample(
        name = "vm-clone",
        description = "This sample makes a template of an " +
                "existing VM and deploy multiple instances of this " +
                "template onto a datacenter"
)
public class VMClone extends ConnectedVimServiceBase {
    private ManagedObjectReference propCollectorRef;

    private String dataCenterName;
    private String vmPathName;
    private String cloneName;
    private String targetNetworkName;
    private String hostName;

    @Option(name = "datacentername", description = "name of Datacenter")
    public void setDataCenterName(String dataCenterName) {
        this.dataCenterName = dataCenterName;
    }

    @Option(name = "vmpath", description = "inventory path of the VM")
    public void setVmPathName(String vmPathName) {
        this.vmPathName = vmPathName;
    }

    @Option(name = "clonename", description = "name of the clone")
    public void setCloneName(String cloneName) {
        this.cloneName = cloneName;
    }

    @Option(name = "hostname", required =false,
            description = "name of target host")
    public void setHostName(String hostName) {
        this.hostName = hostName;
    }

    @Option(name = "targetnetworkname",required = false,
            description = "name of the target network "
            + "Options: Standard|Distributed|Opaque Network")
    public void setTargetNetworkName(String targetNetworkName) {
        this.targetNetworkName = targetNetworkName;
    }


    /**
     * This method returns a boolean value specifying whether the Task is
     * succeeded or failed.
     *
     * @param task ManagedObjectReference representing the Task.
     * @return boolean value representing the Task result.
     * @throws InvalidCollectorVersionFaultMsg
     *
     * @throws RuntimeFaultFaultMsg
     * @throws InvalidPropertyFaultMsg
     */
    boolean getTaskResultAfterDone(ManagedObjectReference task)
            throws InvalidPropertyFaultMsg, RuntimeFaultFaultMsg,
            InvalidCollectorVersionFaultMsg {

        boolean retVal = false;

        // info has a property - state for state of the task
        Object[] result =
                waitForValues.wait(task, new String[]{"info.state", "info.error"},
                        new String[]{"state"}, new Object[][]{new Object[]{
                        TaskInfoState.SUCCESS, TaskInfoState.ERROR}});

        if (result[0].equals(TaskInfoState.SUCCESS)) {
            retVal = true;
        }
        if (result[1] instanceof LocalizedMethodFault) {
            throw new RuntimeException(
                    ((LocalizedMethodFault) result[1]).getLocalizedMessage());
        }
        return retVal;
    }

    /**
     * This method returns the ConfigTarget for a HostSystem.
     *
     * @param computeResMor A MoRef to the ComputeResource used by the HostSystem
     * @param hostMor       A MoRef to the HostSystem
     * @return Instance of ConfigTarget for the supplied
     *         HostSystem/ComputeResource
     * @throws Exception When no ConfigTarget can be found
     */
    ConfigTarget getConfigTargetForHost(ManagedObjectReference hostMor)
            throws RuntimeFaultFaultMsg, InvalidPropertyFaultMsg {
        ManagedObjectReference crmor =
                (ManagedObjectReference) getMOREFs.entityProps(hostMor,
                        new String[]{"parent"}).get("parent");
        if (crmor == null) {
            System.out.println("No Compute Resource Found On Specified Host");
        }
        ManagedObjectReference envBrowseMor =
                (ManagedObjectReference) getMOREFs.entityProps(crmor,
                        new String[]{"environmentBrowser"}).get(
                        "environmentBrowser");
        ConfigTarget configTarget =
                vimPort.queryConfigTarget(envBrowseMor, hostMor);
        if (configTarget == null) {
            throw new RuntimeException("No ConfigTarget found in ComputeResource");
        }
        return configTarget;
    }

    /**
     * Return the list of any available networks on ESX which will host the VM
     *
     * @return
     * @throws RuntimeFaultFaultMsg
     * @throws InvalidPropertyFaultMsg
     */
    public HashMap<String,VirtualDeviceBackingInfo> getAvailableHostNetworkDevice(ConfigTarget configTarget)
            throws InvalidPropertyFaultMsg, RuntimeFaultFaultMsg {
        HashMap<String, VirtualDeviceBackingInfo> deviceBckg =
                new HashMap<String,VirtualDeviceBackingInfo>();
        for (VirtualMachineNetworkInfo info : configTarget.getNetwork()) {
            VirtualEthernetCardNetworkBackingInfo networkDeviceBckg = new
                    VirtualEthernetCardNetworkBackingInfo();
            networkDeviceBckg.setDeviceName(info.getNetwork().getName());
            deviceBckg.put(info.getNetwork().getName(),networkDeviceBckg);
        }

        for (DistributedVirtualPortgroupInfo info : configTarget.getDistributedVirtualPortgroup()) {
            VirtualEthernetCardDistributedVirtualPortBackingInfo dvsDeviceBckg = new
                    VirtualEthernetCardDistributedVirtualPortBackingInfo();
            dvsDeviceBckg.setPort(new DistributedVirtualSwitchPortConnection());
            dvsDeviceBckg.getPort().setPortgroupKey(info.getPortgroupKey());
            dvsDeviceBckg.getPort().setSwitchUuid(info.getSwitchUuid());
            deviceBckg.put(info.getPortgroupName(),dvsDeviceBckg);
        }

        for (OpaqueNetworkTargetInfo info : configTarget.getOpaqueNetwork()) {
            OpaqueNetworkSummary opaqueNetworkSummary =
                    (OpaqueNetworkSummary) getMOREFs.entityProps(info.getNetwork().getNetwork(),
                            new String[] {"summary"}).get("summary");
            VirtualEthernetCardOpaqueNetworkBackingInfo opaqueNetworkDeviceBckg = new
                    VirtualEthernetCardOpaqueNetworkBackingInfo();
            opaqueNetworkDeviceBckg.setOpaqueNetworkId(opaqueNetworkSummary.getOpaqueNetworkId());
            opaqueNetworkDeviceBckg.setOpaqueNetworkType(opaqueNetworkSummary.getOpaqueNetworkType());
            deviceBckg.put(info.getNetwork().getName(),opaqueNetworkDeviceBckg);
        }
        return deviceBckg;
    }

    /** Config Specification to change network of a virtual machine
     * @param vmRef
     * @param targetNetworkName
     * @return VirtualDeviceConfigSpec
     * @throws InvalidPropertyFaultMsg
     * @throws RuntimeFaultFaultMsg
     * @throws Exception
     */
    VirtualDeviceConfigSpec changeVmNicSpec(ManagedObjectReference vmRef, String targetNetworkName, ManagedObjectReference hostRef)
            throws RuntimeFaultFaultMsg, InvalidPropertyFaultMsg {
        List<VirtualDevice> listvd = null;
        listvd = ((ArrayOfVirtualDevice) getMOREFs
                .entityProps(vmRef,
                        new String[] { "config.hardware.device" }).get(
                                "config.hardware.device")).getVirtualDevice();
        VirtualEthernetCard nic = new VirtualPCNet32();
        VirtualDeviceConfigSpec nicSpec = new VirtualDeviceConfigSpec();
        ConfigTarget configTarget =
                getConfigTargetForHost(hostRef);
        HashMap<String, VirtualDeviceBackingInfo> availableNetworks =
                getAvailableHostNetworkDevice(configTarget);
        if (availableNetworks.containsKey(targetNetworkName)) {
            for (VirtualDevice virtualDevice : listvd) {
                if(virtualDevice.getDeviceInfo().getLabel().contains("Network adapter")) {
                    nic=(VirtualEthernetCard) virtualDevice;
                    nicSpec.setOperation(VirtualDeviceConfigSpecOperation.EDIT);
                    nic.setBacking(availableNetworks.get(targetNetworkName));
                    nicSpec.setDevice(nic);
                }
            }
        }
        return nicSpec;
    }

    void cloneVM() throws RuntimeFaultFaultMsg, InvalidPropertyFaultMsg,
    InvalidCollectorVersionFaultMsg, CustomizationFaultFaultMsg, TaskInProgressFaultMsg,
    VmConfigFaultFaultMsg, InsufficientResourcesFaultFaultMsg, InvalidDatastoreFaultMsg, FileFaultFaultMsg,
    MigrationFaultFaultMsg, InvalidStateFaultMsg, InvocationTargetException, NoSuchMethodException, IllegalAccessException {
        // Find the Datacenter reference by using findByInventoryPath().
        ManagedObjectReference datacenterRef =
                vimPort.findByInventoryPath(serviceContent.getSearchIndex(),
                        dataCenterName);
        if (datacenterRef == null) {
            System.out.printf("The specified datacenter [ %s ]is not found %n",
                    dataCenterName);
            return;
        }
        // Find the virtual machine folder for this datacenter.
        ManagedObjectReference vmFolderRef =
                (ManagedObjectReference) getDynamicProperty(datacenterRef,
                        "vmFolder");
        if (vmFolderRef == null) {
            System.out.println("The virtual machine is not found");
            return;
        }
        ManagedObjectReference vmRef =
                vimPort.findByInventoryPath(serviceContent.getSearchIndex(),
                        vmPathName);
        if (vmRef == null) {
            System.out.printf("The VMPath specified [ %s ] is not found %n",
                    vmPathName);
            return;
        }

        VirtualMachineCloneSpec cloneSpec = new VirtualMachineCloneSpec();
        VirtualMachineRelocateSpec relocSpec = new VirtualMachineRelocateSpec();
        if(targetNetworkName!=null && hostName!=null) {
            ManagedObjectReference hostRef = getMOREFs.inContainerByType(datacenterRef, "HostSystem").get(
                    hostName);
            if (hostRef == null) {
                System.out.println("Host " + hostName + " not found");
                return;
            }
         VirtualDeviceConfigSpec nicSpec = new VirtualDeviceConfigSpec();
         nicSpec = changeVmNicSpec(vmRef, targetNetworkName,hostRef);
         relocSpec.getDeviceChange().add(nicSpec);
        }

        cloneSpec.setLocation(relocSpec);
        cloneSpec.setPowerOn(false);
        cloneSpec.setTemplate(false);

        // If the source virtual machine has a virtual TPM device, specify the
        // clone policy for the TPM secrets here.
        cloneSpec.setTpmProvisionPolicy(VirtualMachineCloneSpecTpmProvisionPolicy.REPLACE.value());

        System.out.printf("Cloning Virtual Machine [%s] to clone name [%s] %n",
                vmPathName.substring(vmPathName.lastIndexOf("/") + 1), cloneName);
        ManagedObjectReference cloneTask =
                vimPort.cloneVMTask(vmRef, vmFolderRef, cloneName, cloneSpec);
        if (getTaskResultAfterDone(cloneTask)) {
            System.out
                    .printf(
                            "Successfully cloned Virtual Machine [%s] to clone name [%s] %n",
                            vmPathName.substring(vmPathName.lastIndexOf("/") + 1),
                            cloneName);
        } else {
            System.out.printf(
                    "Failure Cloning Virtual Machine [%s] to clone name [%s] %n",
                    vmPathName.substring(vmPathName.lastIndexOf("/") + 1),
                    cloneName);
        }
    }

    /**
     * Uses the new RetrievePropertiesEx method to emulate the now deprecated
     * RetrieveProperties method
     *
     * @param listpfs
     * @return list of object content
     * @throws Exception
     */
    List<ObjectContent> retrievePropertiesAllObjects(
            List<PropertyFilterSpec> listpfs) throws RuntimeFaultFaultMsg, InvalidPropertyFaultMsg {

        RetrieveOptions propObjectRetrieveOpts = new RetrieveOptions();

        List<ObjectContent> listobjcontent = new ArrayList<ObjectContent>();

        RetrieveResult rslts =
                vimPort.retrievePropertiesEx(propCollectorRef, listpfs,
                        propObjectRetrieveOpts);
        if (rslts != null && rslts.getObjects() != null
                && !rslts.getObjects().isEmpty()) {
            listobjcontent.addAll(rslts.getObjects());
        }
        String token = null;
        if (rslts != null && rslts.getToken() != null) {
            token = rslts.getToken();
        }
        while (token != null && !token.isEmpty()) {
            rslts =
                    vimPort.continueRetrievePropertiesEx(propCollectorRef, token);
            token = null;
            if (rslts != null) {
                token = rslts.getToken();
                if (rslts.getObjects() != null && !rslts.getObjects().isEmpty()) {
                    listobjcontent.addAll(rslts.getObjects());
                }
            }
        }
        return listobjcontent;
    }

    Object getDynamicProperty(ManagedObjectReference mor,
                              String propertyName) throws NoSuchMethodException, InvocationTargetException, IllegalAccessException, RuntimeFaultFaultMsg, InvalidPropertyFaultMsg {
        ObjectContent[] objContent =
                getObjectProperties(mor, new String[]{propertyName});

        Object propertyValue = null;
        if (objContent != null) {
            List<DynamicProperty> listdp = objContent[0].getPropSet();
            if (listdp != null) {
                /*
                * Check the dynamic property for ArrayOfXXX object
                */
                Object dynamicPropertyVal = listdp.get(0).getVal();
                String dynamicPropertyName =
                        dynamicPropertyVal.getClass().getName();
                if (dynamicPropertyName.indexOf("ArrayOf") != -1) {
                    String methodName =
                            dynamicPropertyName.substring(
                                    dynamicPropertyName.indexOf("ArrayOf")
                                            + "ArrayOf".length(),
                                    dynamicPropertyName.length());
                    /*
                    * If object is ArrayOfXXX object, then get the XXX[] by
                    * invoking getXXX() on the object.
                    * For Ex:
                    * ArrayOfManagedObjectReference.getManagedObjectReference()
                    * returns ManagedObjectReference[] array.
                    */
                    if (methodExists(dynamicPropertyVal, "get" + methodName, null)) {
                        methodName = "get" + methodName;
                    } else {
                        /*
                        * Construct methodName for ArrayOf primitive types
                        * Ex: For ArrayOfInt, methodName is get_int
                        */
                        methodName = "get_" + methodName.toLowerCase();
                    }
                    Method getMorMethod =
                            dynamicPropertyVal.getClass().getDeclaredMethod(
                                    methodName, (Class[]) null);
                    propertyValue =
                            getMorMethod.invoke(dynamicPropertyVal, (Object[]) null);
                } else if (dynamicPropertyVal.getClass().isArray()) {
                    /*
                    * Handle the case of an unwrapped array being deserialized.
                    */
                    propertyValue = dynamicPropertyVal;
                } else {
                    propertyValue = dynamicPropertyVal;
                }
            }
        }
        return propertyValue;
    }

    /**
     * Retrieve contents for a single object based on the property collector
     * registered with the service.
     *
     * @param mobj       Managed Object Reference to get contents for
     * @param properties names of properties of object to retrieve
     * @return retrieved object contents
     */
    ObjectContent[] getObjectProperties(
            ManagedObjectReference mobj, String[] properties) throws RuntimeFaultFaultMsg, InvalidPropertyFaultMsg {
        if (mobj == null) {
            return null;
        }

        PropertyFilterSpec spec = new PropertyFilterSpec();
        spec.getPropSet().add(new PropertySpec());
        if ((properties == null || properties.length == 0)) {
            spec.getPropSet().get(0).setAll(Boolean.TRUE);
        } else {
            spec.getPropSet().get(0).setAll(Boolean.FALSE);
        }
        spec.getPropSet().get(0).setType(mobj.getType());
        spec.getPropSet().get(0).getPathSet().addAll(Arrays.asList(properties));
        spec.getObjectSet().add(new ObjectSpec());
        spec.getObjectSet().get(0).setObj(mobj);
        spec.getObjectSet().get(0).setSkip(Boolean.FALSE);
        List<PropertyFilterSpec> listpfs = new ArrayList<PropertyFilterSpec>(1);
        listpfs.add(spec);
        List<ObjectContent> listobjcont = retrievePropertiesAllObjects(listpfs);
        return listobjcont.toArray(new ObjectContent[listobjcont.size()]);
    }

    /**
     * Determines of a method 'methodName' exists for the Object 'obj'.
     *
     * @param obj            The Object to check
     * @param methodName     The method name
     * @param parameterTypes Array of Class objects for the parameter types
     * @return true if the method exists, false otherwise
     */
    @SuppressWarnings("rawtypes")
    boolean methodExists(Object obj, String methodName,
                         Class[] parameterTypes) throws NoSuchMethodException {
        boolean exists = false;
        Method method = obj.getClass().getMethod(methodName, parameterTypes);
        if (method != null) {
            exists = true;
        }
        return exists;
    }

    @Action
    public void run() throws RuntimeFaultFaultMsg, TaskInProgressFaultMsg, VmConfigFaultFaultMsg, InvalidDatastoreFaultMsg, FileFaultFaultMsg, NoSuchMethodException, MigrationFaultFaultMsg, InvalidStateFaultMsg, InvalidCollectorVersionFaultMsg, IllegalAccessException, CustomizationFaultFaultMsg, InsufficientResourcesFaultFaultMsg, InvocationTargetException, InvalidPropertyFaultMsg {
        propCollectorRef = serviceContent.getPropertyCollector();
        cloneVM();
    }
}