@echo off
setlocal enabledelayedexpansion
goto start
__________________________________________________

CVE-2021-44228 - Prevent log4j parameter expansion
Horizon Enterprise Windows components
VMware, Inc. 2022
__________________________________________________

:help
echo:
echo Command line interface for Horizon Windows Log4j Mitigation script.
echo:
echo:
echo Usage:
echo     Reporting mode:
echo         Horizon_Windows_Log4j_Mitigation.bat: Executes the script in brief Reporting mode.
echo:
echo         Horizon_Windows_Log4j_Mitigation.bat /verbose: Executes in extended Reporting mode.
echo:
echo:
echo     Resolution mode:
echo         Horizon_Windows_Log4j_Mitigation.bat /resolve: Executes the script in Resolution mode with minimal output.
echo:
echo         Horizon_Windows_Log4j_Mitigation.bat /resolve /verbose: Executes the script in Resolution mode with extended output.
echo:
echo         Horizon_Windows_Log4j_Mitigation.bat /resolve /force: Executes the script in strict Resolution mode with minimal output.
echo:
echo         Horizon_Windows_Log4j_Mitigation.bat /resolve /force /verbose: Executes the script in strict Resolution mode with extended output.
echo:
echo:
echo     Restoration mode:
echo             Horizon_Windows_Log4j_Mitigation.bat /restore: Executes the script in Restoration mode with minimal output.
echo:
echo             Horizon_Windows_Log4j_Mitigation.bat /restore /verbose: Executes the script in Restoration mode with extended output.
echo:
echo:
echo     Verification mode:
echo             Horizon_Windows_Log4j_Mitigation.bat /checkversion^=version: Executes the script in Verification mode with minimal output.
echo:
echo             Horizon_Windows_Log4j_Mitigation.bat /checkversion^=version /verbose: Executes the script in Verification mode with extended output.
echo:
echo     Ex: Horizon_Windows_Log4j_Mitigation.bat /checkversion^=2.17.1 /verbose: Checks if the log4j libraries present under install location matches the provided version.

goto :EOF

:start

set force=0
set verbose=0
set resolve=0
set restore=0
set checkversion=0
set version=0
set patched=1

:again
if not "%1" == "" (
	
	set valid=0
	
	if "%1"=="/help" (
		goto help
	)
	if "%1"=="/force" (
		set force=1
		set valid=1
	)
	if "%1"=="/verbose" (
		set verbose=1
		set valid=1
	)
	
	if "%1"=="/checkversion" (
		set checkversion=1
		set valid=1
		set version=%2
		shift

		if "!version!" == "" (
			echo No version provided.
			goto help
		)
		if not "!version:~0,1!"=="1" (
			if not "!version:~0,1!"=="2" (
				if not "!version:~0,1!"=="3" set valid=0
			)
		)
		if !valid!==0 ( 
			echo Invalid version provided: !version!.
			goto help
		)
	)

	if "%1"=="/restore" (
		set valid=1
		set restore=1
	)

	if "%1"=="/resolve" (
		set valid=1
		set resolve=1
	)
	if not !valid!==1 (
		echo Invalid switch provided: %1
		goto help
	)
	shift
    goto again
)

Rem Resolve switch cannot be used with Restore and CheckVersion switches.
if !resolve!==1 (
	if !restore!==1 (
		echo Invalid command usage. /resolve and /restore switches cannot be used together.
		goto help
	)
	if !checkversion!==1 (
		echo Invalid command  usage. /resolve and /checkversion switches cannot be used together.
		goto help
	)
)

Rem Restore switch cannot be used with Resolve, Force and CheckVersion switches.
if !restore!==1 (
	if !checkversion!==1 (
		echo Invalid command  usage. /restore and /checkversion switches cannot be used together.
		goto help
	)
	if !force!==1 (
		echo Invalid command  usage. /restore and /force switches cannot be used together.
		goto help
	)
)
Rem Checkversion switch cannot be used with Resolve, Restore and Force switches.
if !checkversion!==1 (
	if !force!==1 (
		echo Invalid command  usage. /checkversion and /force switches cannot be used together.
		goto help
	)
)

set options=JVMOptions
set agentinstall=AgentInstallPath
set serverinstall=ServerInstallPath
set killflag=-Dlog4j2.formatMsgNoLookups=true
set classLoc=org/apache/logging/log4j/core/lookup/JndiLookup.class
set classLocForFind="org\\apache\\logging\\log4j\\core\\lookup\\JndiLookup.class"
set vdmpath=HKLM\Software\VMware, Inc.\VMware VDM
set restorefile=\..\script.restore
set "vulnerable="
set agent=0
set "needsrestart="
set iscs=0
set servicerunning=0

Rem Check the registry entries to identify component type
set sigpath=HKLM\Software\VMware, Inc.\VMware VDM\plugins\wsnm\TomcatService
for /f "delims=" %%q in ('reg.exe query "%sigpath%" /v Filename 2^> nul') do set sigval=%%q
if not "%sigval%"=="" goto serverfix
set sigpath=HKLM\Software\VMware, Inc.\VMware VDM\plugins\wsnm\TunnelService
for /f "delims=" %%q in ('reg.exe query "%sigpath%" /v Filename 2^> nul') do set sigval=%%q
if not "%sigval%"=="" goto tunnelfix
set sigpath=HKLM\Software\VMware, Inc.\VMware VDM\plugins\wssm\desktop
for /f "delims=" %%q in ('reg.exe query "%sigpath%" /v Filename 2^> nul') do set sigval=%%q
if not "%sigval%"=="" goto agentfix
echo Wrong machine type.
pause
goto :EOF

:agentfix
echo Component Detected: Agent. Please ensure that following services are stopped before executing this script.
echo * VMware Horizon View Agent
echo * VMware vRealize Operations for Horizon Desktop Agent (if exists)
set agent=1
echo:
set jvmregpath=HKLM\Software\VMware, Inc.\VMware VDM\Node Manager\JVM

if !resolve!==1 (
	setx /M LOG4J_FORMAT_MSG_NO_LOOKUPS true >nul 2>&1
	for /f "tokens=2*" %%q in ('reg.exe query "%jvmregpath%" /v %options%') do set svcval=%%r
	echo !svcval!|find "%killflag%" >nul
	if errorlevel 1 (
		if !verbose!==1 echo Updating registry options for Agent
		reg add "%jvmregpath%" /v %options% /d "!svcval! %killflag%" /f
		set "needsrestart=y"
	)
)
for /f "tokens=2*" %%q in ('reg.exe query "%vdmpath%" /v %agentinstall%') do set installloc=%%r
goto commonfix

:tunnelfix
echo Component Detected: Security Server. Please ensure that following services are stopped before executing this script.
echo * VMware Horizon View Security Server
echo:
set svcpath=HKLM\Software\VMware, Inc.\VMware VDM\plugins\wsnm\TunnelService\Params
if !resolve!==1 (
	for /f "tokens=2*" %%q in ('reg.exe query "%svcpath%" /v %options%') do set svcval=%%r
	echo !svcval!|find "%killflag%" >nul
	if errorlevel 1 (
		if !verbose!==1 echo Updating registry options for *TunnelService*
		reg add "%svcpath%" /v %options% /d "!svcval! %killflag%" /f
		set "needsrestart=y"
	)
)
for /f "tokens=2*" %%q in ('reg.exe query "%vdmpath%" /v %serverinstall%') do set installloc=%%r
goto commonfix

:serverfix
echo Component Detected: Connection Server.
echo:
for /f "tokens=2*" %%q in ('reg.exe query "!vdmpath!" /v %serverinstall%') do set installloc=%%r

Rem Check if compromise is detected
set fileLoc="%installloc%appblastgateway\lib\absg-worker.js"
for /f "tokens=*" %%i in ('findstr "child_process" !fileLoc!') do (
	if not "%%i"=="" (
		echo DANGER: This system is compromised. Take down the system and engage Incident Response Team.
		goto :EOF
	)
)
echo  Please ensure that following services are stopped before executing this script.
echo * VMware Horizon View Connection Server
echo:
set iscs=1

if !resolve!==1 (
	for %%s in (MessageBusService TunnelService TomcatService) do (
		set svcpath=HKLM\Software\VMware, Inc.\VMware VDM\plugins\wsnm\%%s\Params
		for /f "tokens=2*" %%q in ('reg.exe query "!svcpath!" /v %options%') do set svcval=%%r
		echo !svcval!|find "%killflag%" >nul
		if errorlevel 1 (
			if !verbose!==1 echo Updating registry options for *%%s*
			reg add "!svcpath!" /v %options% /d "!svcval! %killflag%" /f
			set "needsrestart=y"
		)
	)
)
goto commonfix

Rem Common code for all the components
:commonfix
set zipexe="%installloc%\DCT\7za.exe" 
set pattern=%installloc%\log4j-core*.jar
set vlsipattern=%installloc%\view-vlsi-cli.jar

if !restore!==1 (
	set patterns[0]="%pattern%"
	set patterns[1]="%vlsipattern%"

	if !verbose!==1 echo:
	
	Rem If it is not CS, no need to check for view-vlsi-cli.jar
	if %iscs%==1 (
		set end=1
	) else (
		set end=0
	)
	for /l %%n in (0,1,!end!) do (
		for /f "delims=" %%p in ('dir /s/b !patterns[%%n]!') do (
			call :LINEBREAK
			if not exist "%%p%restorefile%" ( 
				if !verbose!==1 echo Restore file does not exist for "%%p"
			) else (

				if %agent%==1 (
					set service=WSNM
				) else (
					set service=wsbroker
				)

				Rem Restores the jar files if the corresponding services are not running and deletes the restore file.
				for /f "tokens=3 delims=: " %%H in ('sc query "!service!" ^| findstr " STATE"') do (
					if "%%H"=="STOPPED" (
						set "needsrestart=y"
						if !verbose!==1 echo Restoring JAR file "%%p"
						echo F|xcopy /Q /Y /F "%%p%restorefile%" "%%p" >nul 2>&1
						del /Q /F "%%p%restorefile%"
						if !verbose!==1 echo Restored JAR file "%%p"
					) else (
						set running=1
						echo Services are still running, cannot restore the libraries. Please stop the service and run again in restoration mode.
						goto :EOF
					)
				)
			)
		)
	)
	
	Rem If HTML access is installed, restore the portal.war file in CS when the services are not running and delete the portal folder and restore file
	if not !running!==1 (
		if %iscs%==1 (
			set portalrestore=script.restore
			pushd %installloc%broker\webapps
			call :LINEBREAK
			if EXIST portal.war (
				if not EXIST !portalrestore! (

					if !verbose!==1 echo Restore file does not exist for "%installloc%broker\webapps\portal.war"
				) else (
					if !verbose!==1 echo Restoring WAR file "%installloc%broker\webapps\portal.war"
					del /Q /F portal.war >nul 2>&1
					echo F|xcopy /Q /Y /F "!portalrestore!" "portal.war" >nul 2>&1
					del /Q /F "!portalrestore!"
					if !verbose!==1 echo Restored WAR file "%installloc%broker\webapps\portal.war"
					rmdir /S/Q portal >nul 2>&1
				)
			)
				popd
		)
	)
	
	goto reboot
)

set vulnerableversion=0
set compareVersion=1
set patterns[0]="%pattern%"
set patterns[1]="%vlsipattern%"

Rem If it is not CS, no need to check for view-vlsi-cli.jar
if %iscs%==1 (
	set end=1
) else (
	set end=0
)

Rem If it is a log4j-core jar, fetches the version from MANIFEST.MF and identifies if it is vulnerable or not.
Rem If checkversion flag is set it just compares if the available version matches the provided version or not.
:retry
for /l %%n in (0,1,!end!) do (
	for /f "delims=" %%x in ('dir /s /b !patterns[%%n]!') do (

		if %%n==1 (
			set compareVersion=0
		) else (
			set compareVersion=1
		)
		
		Rem Regardless of vulnerable class' existence, comapre versions if checkversion flag is set.
		if !checkversion!==1 (
			call :processfile %%x
		) else (
			Rem check for vulnerability only if affected class is present in the jar.
			%zipexe% l "%%x" > output.tmp
			for /f "tokens=*" %%i in ('findstr %classLocForFind% output.tmp') do (
				if not %%i == "" (
					call :processfile %%x
				)
			)
		)
	)
)

Rem If HTML access is installed, create a restore file for portal.war, resolve the jar inside it when the services are not running.
if !resolve!==1 (
	if not !servicerunning!==1 (
		if %iscs%==1 (
			set portalrestore=script.restore
			pushd %installloc%broker\webapps
			if exist portal.war (
			
				if not EXIST portal (
					%zipexe% x -bso0 portal.war -oportal
					set patterns[0]="%installloc%\broker\webapps\portal\log4j-core*.jar
					set end=0
					popd
					goto retry
				) else (

					if not EXIST !portalrestore! (
						call :LINEBREAK
						if !verbose!==1 echo Found library containing log4j classes - "%installloc%broker\webapps\portal.war", checking for vulnerability...
						
						if EXIST portal\WEB-INF\lib\script.restore (
							set resolveportal=1
						) else (
							if EXIST portal (					
								if !verbose!==1 echo Library - "%installloc%broker\webapps\portal.war" is not vulnerable
								set resolveportal=0
							) else (
								set resolveportal=1
							)
						)

						if !resolveportal!==1 (
							if !verbose!==1 (
								if defined vulnerable (
									echo Library - "%installloc%broker\webapps\portal.war" is vulnerable
									echo:
								) else (
									if !force!==1 (
										echo Library - "%installloc%broker\webapps\portal.war" is not vulnerable
										echo:
										echo Force mode enabled, the affected class will be deleted.
									)
								)
								echo Deleting the affected class from "%installloc%broker\webapps\portal.war"
							)

							echo F|xcopy /Q /Y /F "portal.war" "!portalrestore!" >nul 2>&1
							
							pushd portal
							%zipexe% u -y -tzip portal.war -u- -up0q1x1y2z1^^!..\portalnew.war
							%zipexe% d "..\portalnew.war" "WEB-INF/lib/script.restore"
							%zipexe% d "..\portalnew.war" "META-INF/war-tracker"
							popd
							
							del /Q /F portal.war
							rename portalnew.war portal.war
						)
					)
				)
			)
			popd
		)
	)
)

del /Q /F MANIFEST.MF >nul 2>&1
del /Q /F output.tmp >nul 2>&1

if not defined vulnerable (
	
	if !deleteclass!==1 (
		goto reboot
	)
	
	call :LINEBREAK
	
	Rem if checkversion flag is set, we don't need to print vulnerability status.
	if !checkversion!==1 (
		if not !patched!==1 echo The system is not patched with the provided Log4j version: !version!
		if !patched!==1 echo The system is fully patched with the provided Log4j version: !version!
		goto :EOF
	)
	
	echo No vulnerable Log4j libraries detected.
	if defined needsrestart goto restart
	goto :EOF
	
) else (
	if !resolve!==1 (
		if %servicerunning%==1 (
			Rem Print the below message only when services are running but registry edits got applied.
			if defined needsrestart (
				echo:
				echo Workaround applied partially. Please stop the service and run again in resolution mode to apply full workaround.
				echo:
			)
			goto :EOF
		)
		goto reboot
	) else (
		call :LINEBREAK
		echo Vulnerable Log4j libraries detected.
		goto :EOF
	)
)

:reboot
if defined needsrestart goto restart
goto :EOF

:restart
call :LINEBREAK
echo Completed. Please restart the services for the changes to take effect.
goto :EOF

:LINEBREAK
if !verbose!==1 (
	echo:
	echo ==========================================================================================================================================================
	echo:
)
exit /b 0
	
:processfile
for %%F in ("%1") do (
	call :LINEBREAK
	
	if !verbose!==1  (
		if !checkversion!==1 (
			echo Found library containing log4j classes - "%%x", checking if it matches provided version...
		) else (			
			echo Found library containing log4j classes - "%%x", checking for vulnerability...
		)
	)
	
	if !compareVersion!==1 (
		%zipexe% e -bso0 -aoa "%%x" META-INF\MANIFEST.MF
		for /f "usebackq tokens=1,2 delims=: " %%i  in ("MANIFEST.MF") do (
			echo "%%i" | find /i "Implementation-Version" >nul 2>&1 && (
				if !verbose!==1 echo Detected log4j version "%%j"
				if !checkversion!==1 (
					if not "!version!"=="%%j" (
						set patched=0
						if !verbose!==1 echo Detected version: "%%j" does not match the provided version: "!version!" for "%%x"
					) else (
						if !verbose!==1 echo Detected version: "%%j" matches the provided version: "!version!" for "%%x" 
					)
				) else (
					for /f "tokens=1-3 delims=." %%A in ("%%j") do (
						set major=%%A
						set minor=%%B
						
						if !major! EQU 2 (
							if !minor! LSS 16 (
								set vulnerableversion=1
							) else (
								set vulnerableversion=0
							)
						) else (
							set vulnerableversion=0
						)
					)	
				)									
			)
		)
	) else (
		Rem Can't directly identify version for view-vlsi-cli.jar. If the system is patched so far, then view-vlsi-cli.jar is guaranteed to be patched
		if !checkversion!==1 (
			if !patched!==1 (
				if !verbose!==1 echo Detected version: "!version!" matches the provided version: "!version!" for "%%x"
			) else (
				if !verbose!==1 echo Detected version does not match the provided version: "!version!" for "%%x"
			)
		)
	)
	Rem If checkversion flag is not set, check for vulnerability and resolve them if resolve flag is set.
	if not !checkversion!==1 (
		
		if !vulnerableversion!==1 (
			set "vulnerable=y"
			if !verbose!==1 echo Library - "%%x" is vulnerable
			set deleteclass=1
			if !verbose!==1 echo:
		) else (
			if !verbose!==1 echo Library - "%%x" is not vulnerable
			set deleteclass=0
			if !verbose!==1 echo:
		)
		
		Rem Creates a restore file and resolves the vulnerability if the services are not running.
		if !resolve!==1 (
			if !force!==1 (
				set deleteclass=1
				if !verbose!==1 echo Force mode enabled, the affected class will be deleted.
			)
			
			if !deleteclass!==1 (
			
				if %agent%==1 (
						set service=WSNM
				) else (
					set service=wsbroker
				)

				for /f "tokens=3 delims=: " %%H in ('sc query "!service!" ^| findstr " STATE"') do (
					if "%%H"=="STOPPED" (

						if !verbose!==1 echo Deleting the affected class from "%%x"
						echo F|xcopy /Q /Y /F "%%x" "%%x%restorefile%" >nul 2>&1
						if !verbose!==1 (
							%zipexe% d "%%x" "%classLoc%"
						) else (
							%zipexe% d -bso0 "%%x" "%classLoc%"
						)
						set "needsrestart=y"
						if %ERRORLEVEL% NEQ 0 echo Some error occurred while removing the affected class.
					) else (
						set servicerunning=1
						echo Services are still running, cannot delete the affected class from "%%x".
					)
				)
			)
		)
	)
)