﻿#
#  Copyright (C) 2015, 2016, 2017, 2018, 2019, 2020 VMware Inc. All rights reserved.
#
#  Permission is hereby granted, free of charge, to any person obtaining a copy
#  of the software in this file (the "Software"), to deal in the Software 
#  without restriction, including without limitation the rights to use, copy, 
#  modify, merge, publish, distribute, sublicense, and/or sell copies of the 
#  Software, and to permit persons to whom the Software is furnished to do so, 
#  subject to the following conditions:
#  
#  The above copyright notice and this permission notice shall be included in 
#  all copies or substantial portions of the Software.
#  
#  The names "VMware" and "VMware, Inc." must not be used to endorse or promote 
#  products derived from the Software without the prior written permission of 
#  VMware, Inc.
#  
#  Products derived from the Software may not be called "VMware", nor may 
#  "VMware" appear in their name, without the prior written permission of 
#  VMware, Inc.
#  
#  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
#  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
#  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL 
#  VMWARE,INC. BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
#  IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
#  CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#

param(
      [Parameter(Mandatory=$false)] [string] $awAPIServerPwd,
	  [Parameter(Mandatory=$false)] [string] $awTunnelGatewayAPIServerPwd,
	  [Parameter(Mandatory=$false)] [string] $awTunnelProxyAPIServerPwd,
	  [Parameter(Mandatory=$false)] [string] $awCGAPIServerPwd,
	  [Parameter(Mandatory=$false)] [string] $awSEGAPIServerPwd
	  )
#
# Function to parse token values from a .ini configuration file
#

function ImportIni {
	param ($file)

	$ini = @{}
	switch -regex -file $file
	{
            "^\s*#" {
                continue
            }
    		"^\[(.+)\]$" {
        		$section = $matches[1]
        		$ini[$section] = @{}
    		}
    		"([A-Za-z0-9#_]+)=(.+)" {
        		$name,$value = $matches[1..2]
        		$ini[$section][$name] = $value.Trim()
    		}
	}
	$ini
}

#
# Function to write an error message in red with black background
#

function WriteErrorString {
	param ($string)
	write-host $string -foregroundcolor red -backgroundcolor black
}

#
# Function to prompt the user for an UAG VM name and validate the input
#

function GetAPName {
	$valid=0
	while (! $valid) {
		$apName = Read-host "Enter a name for this VM"
		if (($apName.length -lt 1) -Or ($apName.length -gt 32)) { 
			WriteErrorString "Error: Virtual machine name must be between 1 and 32 characters in length"
		} else {
			$valid=1
		}
	}
	$apName 
}

#
# Function to decrypt an encrypted password
#

function ConvertFromSecureToPlain {
    
    param( [Parameter(Mandatory=$true)][System.Security.SecureString] $SecurePassword)
    
    # Create a "password pointer".
    $PasswordPointer = [Runtime.InteropServices.Marshal]::SecureStringToBSTR($SecurePassword)
    
    # Get the plain text version of the password.
    $PlainTextPassword = [Runtime.InteropServices.Marshal]::PtrToStringAuto($PasswordPointer)
    
    # Free the pointer.
    [Runtime.InteropServices.Marshal]::ZeroFreeBSTR($PasswordPointer)
    
    # Return the plain text password.
    $PlainTextPassword
    
}


function ValidateStringIsNumeric {
	param( [Parameter(Mandatory=$true)] $numericString, $fieldName, $iniGroup)

	if ($numericString.length -gt 0) {
		try {
			$number = [int]$numericString
		} catch {
			WriteErrorString "Error: $fieldName in the section $iniGroup is not an integer"
			Exit
		}
	}
}

#
# Function to prompt the user for an UAG root password and validate the input
#

function GetRootPwd {
	param( [Parameter(Mandatory=$true)] $apName)
	$match=0
	while (! $match) {
		$valid=0
		while (! $valid) { 
			$rootPwd = Read-Host -assecurestring "Enter a root password for" $apName
			$rootPwd = ConvertFromSecureToPlain $rootPwd
			if ($rootPwd.length -lt 6) {
				WriteErrorString "Error: Password must contain at least 6 characters"
				Continue
			}
			$valid=1
		}

		$rootPwd2 = Read-Host -assecurestring "Re-enter the root password"
		$rootPwd2 = ConvertFromSecureToPlain $rootPwd2
		if ($rootPwd -ne $rootPwd2) {
			WriteErrorString "Error: re-entered password does not match"
		} else {
			$match=1
		}
	}
	$rootPwd = $rootPwd -replace '"', '\"'
	$rootPwd = $rootPwd -replace "'", "\047"
	$rootPwd
}

#
# Function to prompt the user for an UAG admin password and validate the input
#

function GetAdminPwd {
	param( [Parameter(Mandatory=$true)] $apName)

    write-host "An admin password must be specified if access to the UAG Admin UI and REST API is required"

	$match=0
	while (! $match) {
		$valid=0
		while (! $valid) { 
			$adminPwd = Read-Host -assecurestring "Enter an optional admin password for the Admin UI and REST API management access for" $apName
			$adminPwd = ConvertFromSecureToPlain $adminPwd
			if ($adminPwd.length -eq 0) {
				return
			}

			if ($adminPwd.length -lt 8) {
				WriteErrorString "Error: Password must contain at least 8 characters"
				WriteErrorString "Password must contain at least 8 characters including an upper case letter, a lower case letter, a digit and a special character from !@#$%*()"
				Continue
			}
			if (([regex]"[0-9]").Matches($adminPwd).Count -lt 1 ) {
				WriteErrorString "Error: Password must contain at least 1 numeric digit"
				WriteErrorString "Password must contain at least 8 characters including an upper case letter, a lower case letter, a digit and a special character from !@#$%*()"
				Continue
			}
			if (([regex]"[A-Z]").Matches($adminPwd).Count -lt 1 ) {
				WriteErrorString "Error: Password must contain at least 1 upper case character (A-Z)"
				WriteErrorString "Password must contain at least 8 characters including an upper case letter, a lower case letter, a digit and a special character from !@#$%*()"
				Continue
			}
			if (([regex]"[a-z]").Matches($adminPwd).Count -lt 1 ) {
				WriteErrorString "Error: Password must contain at least 1 lower case character (a-z)"
				WriteErrorString "Password must contain at least 8 characters including an upper case letter, a lower case letter, a digit and a special character from !@#$%*()"
				Continue
			}
			if (([regex]"[!@#$%*()]").Matches($adminPwd).Count -lt 1 ) {
				WriteErrorString "Error: Password must contain at least 1 special character (!@#$%*())"
				WriteErrorString "Password must contain at least 8 characters including an upper case letter, a lower case letter, a digit and a special character from !@#$%*()"
				Continue
			}
			$valid = 1
		}

		$adminPwd2 = Read-Host -assecurestring "Re-enter the admin password"
		$adminPwd2 = ConvertFromSecureToPlain $adminPwd2
		if ($adminPwd -ne $adminPwd2) {
			WriteErrorString "Error: re-entered password does not match"
		} else {
			$match=1
		}
	}
	$adminPwd = $adminPwd -replace '"', '\"'
	$adminPwd = $adminPwd -replace "'", "\047"
	$adminPwd
}

#
# Function to prompt the user for whether to join VMware's Customer Experience Improvement Program (CEIP)
# Default is yes.
#

function GetCeipEnabled {
	param( [Parameter(Mandatory=$true)] $apName)
write-host "Join the VMware Customer Experience Improvement Program?

This setting is supported in UAG versions 3.1 and newer.

VMware’s Customer Experience Improvement Program (CEIP) provides VMware with information that enables VMware to
improve its products and services, to fix problems, and to advise you on how best to deploy and use our products.

As part of the CEIP, VMware collects technical information about your organization's use of VMware products and
services on a regular basis in association with your organization's VMware license key(s). This information does
not personally identify any individual.

Additional information regarding the data collected through CEIP and the purposes for which it is used by VMware
is set forth in the Trust & Assurance Center at http://www.vmware.com/trustvmware/ceip.html.

If you prefer not to participate in VMware's CEIP for UAG 3.1 and newer, you should enter no.

You may join or leave VMware's CEIP for this product at any time. In the UAG Admin UI in System Configuration,
there is a setting 'Join CEIP' which can be set to yes or no and has immediate effect.

To Join the VMware Customer Experience Improvement Program with Unified Access Gateway version 3.1 and newer,
either enter yes or just hit return as the default for this setting is yes."

    $valid=$false
	while (! $valid) { 
	    $yorn = Read-Host "Join CEIP for" $apName "? (default is yes for UAG 3.1 and newer)"
        if (($yorn -eq "yes") -Or ($yorn -eq "")) {
            $ceipEnabled = $true
            $valid=$true
            break
        } elseif ($yorn -eq "no") {
            $ceipEnabled = $false
            $valid=$true
            break
        }
        WriteErrorString 'Error: please enter "yes" or "no", or just hit return for yes.'
    }
    $ceipEnabled
}

function GetTrustedCertificates {
    param($edgeService)
    #Add Trusted certificates entries in json
    $allCerts = "\'trustedCertificates\': [ "
    for($i=1;;$i++)
    {
	    $cert = "trustedCert$i"
	    $cert = $settings.$edgeService.$cert
	    if($cert.length -gt 0)
	    {
		    if (!(Test-path $cert)) {
			    WriteErrorString "Error: PEM Certificate file not found ($cert)"
			    Exit
		    }
		    else
		    {
			    $content = (Get-Content $cert | Out-String) -replace "'", "\\047" -replace "`r`n", "\\n"

			    if ($content -like "*-----BEGIN CERTIFICATE-----*") {
	    			#Write-host "valid cert"
			    } else {
				    WriteErrorString "Error: Invalid certificate file It must contain -----BEGIN CERTIFICATE-----."
				    Exit
			    }
			    $fileName = $cert.SubString($cert.LastIndexof('\')+1)
			    #Write-Host "$fileName"
			    $allCerts += "{ \'name\': \'$fileName\'"
			    $allCerts += ","
			    $allCerts += "\'data\': \'"
			    $allCerts += $content
			    $allCerts += "\'"
			    $allCerts += "},"
		    }
	    }
	    else {
            $allCerts = $allCerts.Substring(0, $allCerts.Length-1)
		    #Write-Host "$($i-1) Certificates Added successfully"
		    break;
	    }
    }
    $allCerts += "]"

    $allCerts
}

function GetHostEntries {
    param($edgeService)
    # Add all host entries into json
    $allHosts = "\'hostEntries\': [ "
    for($i=1;;$i++)
    {
	    $hostEntry = "hostEntry$i"
	    $hostEntry = $settings.$edgeService.$hostEntry
	    if($hostEntry.length -gt 0)
	    {
		    $allHosts += "\'"+$hostEntry+"\',"
	    }
	    else {
            $allHosts = $allHosts.Substring(0, $allHosts.Length-1)
		    #Write-Host "$($i-1) Host entries Added successfully"
		    break;
	    }
    }
    $allHosts += "]"

    $allHosts
}

function GetJWTSettings {
    Param ($settings)

    $icount = 0
    for($i=1;$i -lt 100;$i++)
    {
        $iniGroup = "JWTSettings$i"

        $jwtName = $settings.$iniGroup.name

        if ($jwtName.length -gt 0) {

            if ($icount -eq 0) {
                $jwtSettings = "\'jwtSettingsList\': { \'jwtSettingsList\': [ "
            } else {
                $jwtSettings += ","
            }

            $icount++
            $jwtSettings += "{ \'name\': \'$jwtName\'"

            $jwtIssuer = $settings.$iniGroup.issuer
            if ($jwtIssuer.length -gt 0) {
                $jwtSettings += ","
				$jwtSettings += "\'issuer\': \'$jwtIssuer\'"
            }

            $jcount = 0
            for($j=1;$j -lt 100;$j++)
            {
                $keyFileName = "publicKey$j"
                $keyFileVal = $settings.$iniGroup.$keyFileName
                if($keyFileVal.length -gt 0)
                {
                    if (!(Test-path $keyFileVal)) {
                        WriteErrorString "Error: JWT public key file not found ($keyFileVal) in $iniGroup"
                        Exit
                    }

                    $content = (Get-Content $keyFileVal | Out-String) -replace "'", "\\047" -replace "`r`n", "\\n"

                    if ($content -like "*-----BEGIN PUBLIC KEY-----*") {
                        #Write-host "valid key file"
                    } else {
                        WriteErrorString "Error: Invalid JWT public key file in $iniGroup. It must contain -----BEGIN PUBLIC KEY-----."
                        Exit
                    }
                    $fileName = $keyFileVal.SubString($keyFileVal.LastIndexof('\')+1)

                    if ($fileName -eq "Public_Key_From_URL") {
                        WriteErrorString "Error: A static public key file cannot have the name : Public_Key_From_URL in $iniGroup"
                        Exit
                    }

                    if ($jcount -eq 0) {
						$jwtSettings += ","
                        $jwtSettings += "\'publicKeys\': [ "
                    } Elseif ($jcount -gt 0) {
                        $jwtSettings += ","
                    }

                    $jcount++
                    $jwtSettings += "{ \'name\': \'$fileName\'"
	                $jwtSettings += ","
	                $jwtSettings += "\'data\': \'"
	                $jwtSettings += $content
	                $jwtSettings += "\'"
	                $jwtSettings += "}"
                }
            }
            if ($jcount -gt 0) {
                $jwtSettings += " ]"
            }

            #Adding public key URL
            $jwtPublicKeyUrl = $settings.$iniGroup.url
            if ($jwtPublicKeyUrl.length -gt 0) {
                #
                # Strip the final :443 if specified as that is the default anyway
                #

                if ($jwtPublicKeyUrl.Substring($jwtPublicKeyUrl.length - 4, 4) -eq ":443") {
                    $jwtPublicKeyUrl=$jwtPublicKeyUrl.Substring(0, $jwtPublicKeyUrl.IndexOf(":443"))
                }

                $jwtSettings += ","
                $jwtSettings += "\'publicKeyURLSettings\': { "
                $jwtSettings += "\'url\': \'$jwtPublicKeyUrl\'"

                #Adding thumbprints
                $jwtPublicKeyUrlThumbprints=$settings.$iniGroup.urlThumbprints  -replace ":","="
                if ($jwtPublicKeyUrlThumbprints.length -gt 0) {
                    #
                    # Remove invalid thumbprint characters
                    #
                    $jwtPublicKeyUrlThumbprints = $jwtPublicKeyUrlThumbprints -replace, "[^a-zA-Z0-9,= ]", ""

                    $jwtSettings += ","
                    $jwtSettings += "\'urlThumbprints\': \'$jwtPublicKeyUrlThumbprints\'"
                }

                #Adding trusted certificates
                if ($settings.$iniGroup.trustedCert1.length -gt 0) {
                    $trustedCertificates = GetTrustedCertificates $iniGroup
                    $jwtSettings += ","
                    $jwtSettings += $trustedCertificates
                    $edgeServiceSettingsVIEW += $trustedCertificates
                }


                #Adding public key refresh interval
                if ($settings.$iniGroup.publicKeyRefreshInterval.length -gt 0) {
					try {
						$publicKeyRefreshInterval = [int]$settings.$iniGroup.publicKeyRefreshInterval
					} catch {
						WriteErrorString "Error: publicKeyRefreshInterval in $iniGroup is not an integer"
                        Exit
					}
                    if (($publicKeyRefreshInterval -ne 0) -and (($publicKeyRefreshInterval -lt 10) -or ($publicKeyRefreshInterval -gt 86400))) {
                        WriteErrorString "Error: Public key refresh interval can be either 0 or between 10 secs - 86400 secs (both inclusive) in $iniGroup"
                        Exit
                    }
                    $jwtSettings += ","
                    $jwtSettings += "\'urlResponseRefreshInterval\': \'$publicKeyRefreshInterval\'"
                }

                $jwtSettings += " }"

            } ElseIf ($jcount -eq 0) {
                WriteErrorString "Error: At least one static public key or a public key URL is mandatory in $iniGroup"
                Exit
            }

            if ($icount -gt 0) {
                $jwtSettings += " }"
            }
        }
    }

    if ($icount -gt 0) {
        $jwtSettings += "] }"
    }

    $jwtSettings
}

function GetSAMLServiceProviderMetadata {
    Param ($settings)

    $samlMetadata = "\'serviceProviderMetadataList\': { "
    $samlMetadata += "\'items\': [ "
    $spCount=0

    for($i=1;$i -lt 99;$i++)
    {
	    $spNameLabel = "spName$i"
	    $spName = $settings.SAMLServiceProviderMetadata.$spNameLabel
        $metadataXmlLabel = "metadataXml$i"
        $metadataXml = $settings.SAMLServiceProviderMetadata.$metaDataXmlLabel
	    if($spName.length -gt 0)
	    {
            if ($metaDataXml.length -eq 0) {
			    WriteErrorString "Error: Missing $metaDataXmlLabel"
			    Exit
            }

		    if (!(Test-path $metaDataXml)) {
			    WriteErrorString "Error: SAML Metada file not found ($metaDataXml)"
			    Exit
		    }
			$content = (Get-Content $metaDataXml | Out-String) -replace "'", "\\047" -replace "`r`n", "\\n" -replace """", "\\"""

		    if ($content -like "*urn:oasis:names:tc:SAML:2.0:metadata*") {
    			#Write-host "valid metadata"
		    } else {
			    WriteErrorString "Error: Invalid metadata specified in $metaDataXml"
			    Exit
		    }
    	    if ($spCount -gt 0) {
                $samlMetadata += ", "
            }
			$samlMetadata += "{ \'spName\': \'$spName\'"
			$samlMetadata += ","
			$samlMetadata += "\'metadataXml\': \'"
			$samlMetadata += $content
			$samlMetadata += "\'"
			$samlMetadata += "}"

            $spCount++
	    }

    }
    $samlMetadata += "] }"

    $samlMetadata
}


function GetSAMLIdentityProviderMetadata {
    Param ($settings)

    $sslCertsFile=$settings.SAMLIdentityProviderMetadata.pemCerts

    if ($sslCertsFile.length -gt 0) {

	    if (!(Test-path $sslCertsFile)) {
		    WriteErrorString "Error: [SAMLIdentityProviderMetadata] PEM Certificate file not found ($sslCertsFile)"
		    Exit
	    }

	    $rsaPrivKeyFile=$settings.SAMLIdentityProviderMetadata.pemPrivKey

	    if ($rsaPrivKeyFile.length -eq 0) {
		    WriteErrorString "Error: [SAMLIdentityProviderMetadata] PEM RSA private key file pemPrivKey not specified"
		    Exit
	    }

	    if (!(Test-path $rsaPrivKeyFile)) {
		    WriteErrorString "Error: [SAMLIdentityProviderMetadata]PEM RSA private key file not found ($rsaPrivKeyFile)"
		    Exit
	    }

        #
        # Read the PEM contents and remove any preamble before ----BEGIN
        #

        $sslcerts = (Get-Content $sslCertsFile | Out-String) -replace "'", "\\047" -replace "`r`n", "\\n" -replace """", ""
        $sslcerts = $sslcerts.Substring($sslcerts.IndexOf("-----BEGIN"))

	    if (!($sslcerts -like "*-----BEGIN*")) {
		    WriteErrorString "Error: [SAMLIdentityProviderMetadata] Invalid certs PEM file (pemCerts) specified. It must contain a certificate."
		    Exit
	    }

	    $rsaprivkey = (Get-Content $rsaPrivKeyFile | Out-String) -replace "'", "\\047" -replace "`r`n", "\\n" -replace """", ""
        $rsaprivkey = $rsaprivkey.Substring($rsaprivkey.IndexOf("-----BEGIN"))

	    if ($rsaprivkey -like "*-----BEGIN RSA PRIVATE KEY-----*") {
		    Write-host Deployment will use the specified [SAMLIdentityProviderMetadata] certificate and private key
	    } else {
		    WriteErrorString "Error: [SAMLIdentityProviderMetadata] Invalid private key PEM file (pemPrivKey) specified. It must contain an RSA private key."
		    Exit
	    }
    }

    $samlMetadata="\'identityProviderMetaData\': { "

    #
    # If the signing certificate/key is not specified, we use {} which results in a self-signed cert/key being generated by UAG automatically
    #

    if ($sslcerts.length -gt 0) {
	    $samlMetadata="\'identityProviderMetaData\': { \'privateKeyPem\': \'"
	    $samlMetadata+=$rsaprivkey
	    $samlMetadata+="\', \'certChainPem\': \'"
	    $samlMetadata+=$sslcerts
	    $samlMetadata+="\'"
    }

    $samlMetadata+=" }"

    $samlMetadata
}

function IsPfxPasswordProtected {
    param($sslCertsFilePfx)

    $cert = New-Object System.Security.Cryptography.X509Certificates.X509Certificate2

    try {
        $response = $cert.Import($sslCertsFilePfx, '','DefaultKeySet')
    } catch {
        if ($_.Exception.InnerException.HResult -eq 0x80070056) { # ERROR_INVALID_PASSWORD
            return $true
        }
    }
    return $false
}

function GetPfxPassword {
    param($sslCertsFilePfx, $section)

    $pwd = ""

    if (IsPfxPasswordProtected $sslCertsFilePfx) {

        $cert = New-Object System.Security.Cryptography.X509Certificates.X509Certificate2

        $pfxFilename = Split-Path $sslCertsFilePfx -leaf
    
        while (! $valid) {
            $prompt='Enter the password for the specified [' + $section + '] PFX certificate file '+$pfxFilename+'' 
            $pwd = Read-Host -assecurestring $prompt
            $pwd = ConvertFromSecureToPlain $pwd
            $valid=1

        
            try {
                $response = $cert.Import($sslCertsFilePfx, $pwd,'DefaultKeySet')
            } catch {
                if ($_.Exception.InnerException.HResult -eq 0x80070056) { # ERROR_INVALID_PASSWORD
                    WriteErrorString "Error: Incorrect password - please try again"
                    $valid = 0
                }
            }
        }
    }

    $pwd
}

function isValidPfxFile {
    param($sslCertsFilePfx, $pwd)

    $cert = New-Object System.Security.Cryptography.X509Certificates.X509Certificate2
    
    try {
        $response = $cert.Import($sslCertsFilePfx, $pwd,'DefaultKeySet')
    } catch {
        WriteErrorString "Error: The specified PFX certificate file is invalid ($sslCertsFilePfx)"
        return $false
    }

    if (!$cert.HasPrivateKey) {
        WriteErrorString "Error: The specified PFX Certificate file does not contain a private key ($sslCertsFilePfx)"
        return $false
    }

    return $true
}

#
# Processes normal 443 or Admin 9443 cert. Called as:
# GetCertificateWrapper $setings
# or GetCertificateWrapper $setings "Admin"
#

function GetCertificateWrapper {
    Param ($settings, $admin)

    $section="SSLcert" + $admin

    $sslCertsFile=$settings.$section.pemCerts

    $sslCertsFilePfx=$settings.$section.pfxCerts

    if ($sslCertsFile.length -gt 0) {

	    if (!(Test-path $sslCertsFile)) {
		    WriteErrorString "Error: PEM Certificate file not found ($sslCertsFile)"
		    Exit
	    }

	    $rsaPrivKeyFile=$settings.$section.pemPrivKey

	    if ($rsaPrivKeyFile.length -eq 0) {
		    WriteErrorString "Error: PEM RSA private key file pemPrivKey not specified"
		    Exit
	    }

	    if (!(Test-path $rsaPrivKeyFile)) {
		    WriteErrorString "Error: PEM RSA private key file not found ($rsaPrivKeyFile)"
		    Exit
	    }

        #
        # Read the PEM contents and remove any preamble before ----BEGIN
        #

        $sslcerts = (Get-Content $sslCertsFile | Out-String) -replace "'", "\\047" -replace "`r`n", "\\n" -replace """", ""
        $sslcerts = $sslcerts.Substring($sslcerts.IndexOf("-----BEGIN"))

	    if (!($sslcerts -like "*-----BEGIN*")) {
		    WriteErrorString "Error: Invalid certs PEM file (pemCerts) specified. It must contain a certificate."
		    Exit
	    }

	    $rsaprivkey = (Get-Content $rsaPrivKeyFile | Out-String) -replace "'", "\\047" -replace "`r`n", "\\n" -replace """", ""
        $rsaprivkey = $rsaprivkey.Substring($rsaprivkey.IndexOf("-----BEGIN"))

	    if ($rsaprivkey -like "*-----BEGIN RSA PRIVATE KEY-----*") {
		    Write-host "Deployment will use the specified SSL/TLS server certificate ($section)"
	    } else {
		    WriteErrorString "Error: Invalid private key PEM file (pemPrivKey) specified. It must contain an RSA private key."
		    Exit
	    }
    } elseif ($sslCertsFilePfx.length -gt 0) {
	    if (!(Test-path $sslCertsFilePfx)) {
		    WriteErrorString "Error: PFX Certificate file not found ($sslCertsFilePfx)"
		    Exit
	    }

        $sslCertsFilePfx = Resolve-Path -Path $sslCertsFilePfx

        $pfxPassword = GetPfxPassword $sslCertsFilePfx $section

        if (!(isValidPfxFile $sslCertsFilePfx $pfxPassword)) {
		    Exit
        }

    	$Content = Get-Content -Path $sslCertsFilePfx -Encoding Byte
	    $sslCertsFilePfxB64 = [System.Convert]::ToBase64String($Content)

        $pfxCertAlias=$settings.$section.pfxCertAlias


    } else {
	    Write-host "Deployment will use a self-signed SSL/TLS server certificate ($section)"
    }

    if ($sslcerts.length -gt 0) {
	    $certificateWrapper="\'certificateWrapper" + $admin + "\': { \'privateKeyPem\': \'"
	    $certificateWrapper+=$rsaprivkey
	    $certificateWrapper+="\', \'certChainPem\': \'"
	    $certificateWrapper+=$sslcerts
	    $certificateWrapper+="\' }"
    } elseif ($sslCertsFilePfxB64.length -gt 0) {
	    $certificateWrapper="\'pfxCertStoreWrapper" + $admin + "\': { \'pfxKeystore\': \'"
	    $certificateWrapper+=$sslCertsFilePfxB64
	    $certificateWrapper+="\', \'password\': \'"
	    $certificateWrapper+=$pfxPassword
        if ($pfxCertAlias.length -gt 0) {
	        $certificateWrapper+="\', \'alias\': \'"
	        $certificateWrapper+=$pfxCertAlias
        }
        $certificateWrapper+="\' }"
    }

    $certificateWrapper

}

function GetHorizonCertificatePEM {
    Param ($settings, $HorizonCertificateName)

    $CertificateFile=$settings.Horizon.$HorizonCertificateName

    if ($CertificateFile.length -le 0) {
        return
    }

	if (!(Test-path $CertificateFile)) {
		WriteErrorString "Error: PEM Certificate file not found ($CertificateFile)"
		Exit
	}

	$certificatePEM = (Get-Content $CertificateFile | Out-String) -replace "'", "\\047" -replace "'", "\\047" -replace "`r`n", "\\n"

	if ($certificatePEM -like "*-----BEGIN CERTIFICATE-----*") {
		Write-host Deployment will use the specified [Horizon] $HorizonCertificateName
	} else {
		WriteErrorString "Error: Invalid PEM file ([Horizon] $HorizonCertificateName) specified. It must contain -----BEGIN CERTIFICATE-----."
		Exit
	}

    $certificatePEM
}

#
# Process PFX certificate
#

function GetPFXCertificate {
    param($edgeService)

    $pfxCertsFile=$settings.$edgeService.pfxCerts

    if ($pfxCertsFile.length -gt 0) {
        if (!(Test-path $pfxCertsFile)) {
            WriteErrorString "Error: $edgeService PFX Certificate file not found ($pfxCertsFile)"
            Exit
        }

        $pfxCertsFile = Resolve-Path -Path $pfxCertsFile
        $Content = Get-Content -Path $pfxCertsFile -Encoding Byte
        $pfxCertsFilePfxB64 = [System.Convert]::ToBase64String($Content)
    }

    $pfxCertsFilePfxB64
}

#
# Horizon View settings
#

function GetEdgeServiceSettingsVIEW {
    Param ($settings)

    $proxyDestinationUrl=$settings.Horizon.proxyDestinationUrl

    if ( $proxyDestinationUrl.length -le 0) {
        return
    }

    #
    # Strip the final :443 if specified as that is the default anyway
    #

    if ($proxyDestinationUrl.Substring($proxyDestinationUrl.length - 4, 4) -eq ":443") {
        $proxyDestinationUrl=$proxyDestinationUrl.Substring(0, $proxyDestinationUrl.IndexOf(":443"))
    }
    
    $proxyDestinationUrlThumbprints=$settings.Horizon.proxyDestinationUrlThumbprints  -replace ":","="

    #
    # Remove invalid thumbprint characters
    #

    $proxyDestinationUrlThumbprints = $proxyDestinationUrlThumbprints -replace, "[^a-zA-Z0-9,= ]", ""

    $blastExternalUrl=$settings.Horizon.blastExternalUrl

    $pcoipExternalUrl=$settings.Horizon.pcoipExternalUrl
    if ($pcoipExternalUrl.length -gt 0) {
    	if (([regex]"[.]").Matches($pcoipExternalUrl).Count -ne 3 ) {
    		WriteErrorString "Error: Invalid pcoipExternalUrl value specified ($pcoipExternalUrl). It must contain an IPv4 address."
    		Exit
    	}
    }

    $tunnelExternalUrl=$settings.Horizon.tunnelExternalUrl

    $edgeServiceSettingsVIEW += "{ \'identifier\': \'VIEW\'"
    $edgeServiceSettingsVIEW += ","
    $edgeServiceSettingsVIEW += "\'enabled\': true"
    $edgeServiceSettingsVIEW += ","
    $edgeServiceSettingsVIEW += "\'proxyDestinationUrl\': \'"+$proxyDestinationUrl+"\'"
    if ($proxyDestinationUrlThumbprints.length -gt 0) {
    	$edgeServiceSettingsVIEW += ","
    	$edgeServiceSettingsVIEW += "\'proxyDestinationUrlThumbprints\': \'"+$proxyDestinationUrlThumbprints+"\'"
    }

    if ($pcoipExternalUrl.length -gt 0) {
    	$edgeServiceSettingsVIEW += ","
        $edgeServiceSettingsVIEW += "\'pcoipEnabled\':true"
        if ($settings.Horizon.pcoipDisableLegacyCertificate -eq "true") {
            $edgeServiceSettingsVIEW += ","
            $edgeServiceSettingsVIEW += "\'pcoipDisableLegacyCertificate\': true"
        }
    	$edgeServiceSettingsVIEW += ","
    	$edgeServiceSettingsVIEW += "\'pcoipExternalUrl\': \'"+$pcoipExternalUrl+"\'"
    } else {
    	$edgeServiceSettingsVIEW += ","
    	$edgeServiceSettingsVIEW += "\'pcoipEnabled\':false"
    }

    if ($blastExternalUrl.length -gt 0) {
    	$edgeServiceSettingsVIEW += ","
    	$edgeServiceSettingsVIEW += "\'blastEnabled\':true"
    	$edgeServiceSettingsVIEW += ","
    	$edgeServiceSettingsVIEW += "\'blastExternalUrl\': \'"+$blastExternalUrl+"\'"
        $proxyBlastPemCert = GetHorizonCertificatePEM $settings "proxyBlastPemCert"
        if ($proxyBlastPemCert.length -gt 0) {
            $edgeServiceSettingsVIEW += ","
            $edgeServiceSettingsVIEW += "\'proxyBlastPemCert\': \'"+$proxyBlastPemCert+"\'"
        }
        $proxyBlastSHA1Thumbprint=$settings.Horizon.proxyBlastSHA1Thumbprint
        if ($proxyBlastSHA1Thumbprint.length -gt 0) {
            $edgeServiceSettingsVIEW += ","
            $edgeServiceSettingsVIEW += "\'proxyBlastSHA1Thumbprint\': \'"+$proxyBlastSHA1Thumbprint+"\'"
        }
        $proxyBlastSHA256Thumbprint=$settings.Horizon.proxyBlastSHA256Thumbprint
        if ($proxyBlastSHA256Thumbprint.length -gt 0) {
            $edgeServiceSettingsVIEW += ","
            $edgeServiceSettingsVIEW += "\'proxyBlastSHA256Thumbprint\': \'"+$proxyBlastSHA256Thumbprint+"\'"
        }
    } else {
    	$edgeServiceSettingsVIEW += ","
    	$edgeServiceSettingsVIEW += "\'blastEnabled\':false"
    }

    if ($tunnelExternalUrl.length -gt 0) {
    	$edgeServiceSettingsVIEW += ","
    	$edgeServiceSettingsVIEW += "\'tunnelEnabled\':true"
        $edgeServiceSettingsVIEW += ","
        $edgeServiceSettingsVIEW += "\'tunnelExternalUrl\': \'"+$tunnelExternalUrl+"\'"
        $proxyTunnelPemCert = GetHorizonCertificatePEM $settings "proxyTunnelPemCert"
        if ($proxyTunnelPemCert.length -gt 0) {
            $edgeServiceSettingsVIEW += ","
            $edgeServiceSettingsVIEW += "\'proxyTunnelPemCert\': \'"+$proxyTunnelPemCert+"\'"
        }
        $proxyTunnelSHA1Thumbprint=$settings.Horizon.proxyTunnelSHA1Thumbprint
        if ($proxyTunnelSHA1Thumbprint.length -gt 0) {
            $edgeServiceSettingsVIEW += ","
            $edgeServiceSettingsVIEW += "\'proxyTunnelSHA1Thumbprint\': \'"+$proxyTunnelSHA1Thumbprint+"\'"
        }
        $proxyTunnelSHA256Thumbprint=$settings.Horizon.proxyTunnelSHA256Thumbprint
        if ($proxyTunnelSHA256Thumbprint.length -gt 0) {
            $edgeServiceSettingsVIEW += ","
            $edgeServiceSettingsVIEW += "\'proxyTunnelSHA256Thumbprint\': \'"+$proxyTunnelSHA256Thumbprint+"\'"
        }
    } else {
    	$edgeServiceSettingsVIEW += ","
    	$edgeServiceSettingsVIEW += "\'tunnelEnabled\':false"
    }

    $edgeServiceSettingsVIEW += ","

    if (($settings.Horizon.trustedCert1.length -gt 0) -Or (($settings.Horizon.hostEntry1.length -gt 0))) {

        $trustedCertificates = GetTrustedCertificates "Horizon"
        $edgeServiceSettingsVIEW += $trustedCertificates
        $edgeServiceSettingsVIEW += ","

        $hostEntries = GetHostEntries "Horizon"
        $edgeServiceSettingsVIEW += $hostEntries
        $edgeServiceSettingsVIEW += ","
    }

    if ($settings.Horizon.proxyPattern.length -gt 0) {
        $settings.Horizon.proxyPattern = $settings.Horizon.proxyPattern -replace "\\", "\\\\"
        $edgeServiceSettingsVIEW += "\'proxyPattern\': \'"+$settings.Horizon.proxyPattern+"\'"
    } else {
        $edgeServiceSettingsVIEW += "\'proxyPattern\':\'(/|/view-client(.*)|/portal(.*)|/appblast(.*))\'"
    }

    $authMethods=$settings.Horizon.authMethods
    if ($authMethods.length -gt 0) {
    	$edgeServiceSettingsVIEW += ","
    	$edgeServiceSettingsVIEW += "\'authMethods\': \'"+$authMethods+"\'"
    }

    $samlSP=$settings.Horizon.samlSP
    if ($samlSP.length -gt 0) {
    	$edgeServiceSettingsVIEW += ","
    	$edgeServiceSettingsVIEW += "\'samlSP\': \'"+$samlSP+"\'"
    }

    $windowsSSOEnabled=$settings.Horizon.windowsSSOEnabled
    if ($windowsSSOEnabled.length -gt 0) {
    	$edgeServiceSettingsVIEW += ","
    	$edgeServiceSettingsVIEW += "\'windowsSSOEnabled\': "+$windowsSSOEnabled
    }

    $matchWindowsUserName=$settings.Horizon.matchWindowsUserName
    if ($matchWindowsUserName.length -gt 0) {
    	$edgeServiceSettingsVIEW += ","
    	$edgeServiceSettingsVIEW += "\'matchWindowsUserName\': "+$matchWindowsUserName
    }

    $rewriteOriginHeader=$settings.Horizon.rewriteOriginHeader
    if ($rewriteOriginHeader.length -gt 0) {
        $edgeServiceSettingsVIEW += ","
        $edgeServiceSettingsVIEW += "\'rewriteOriginHeader\': "+$rewriteOriginHeader
    }

    $endpointComplianceCheckProvider=$settings.Horizon.endpointComplianceCheckProvider
    if ($endpointComplianceCheckProvider.length -gt 0) {
    	$edgeServiceSettingsVIEW += ","
    	$edgeServiceSettingsVIEW += "\'devicePolicyServiceProvider\': \'"+$endpointComplianceCheckProvider+"\'"
    }

    if ($settings.Horizon.securityHeaders.length -gt 0) {
        $edgeServiceSettingsVIEW += ","
        $securityHeaders = $settings.Horizon.securityHeaders
        $securityHeaders = $securityHeaders -replace "'", "\\047"
        $edgeServiceSettingsVIEW += "\'securityHeaders\': "+$securityHeaders
    }

    if ($settings.Horizon.radiusClassAttributeList.length -gt 0) {
            $edgeServiceSettingsVIEW += ","
            $edgeServiceSettingsVIEW += "\'radiusClassAttributeList\': \'"+$settings.Horizon.radiusClassAttributeList+"\'"
    }

    if ($settings.Horizon.disclaimerText.length -gt 0) {
        $settings.Horizon.disclaimerText = $settings.Horizon.disclaimerText -replace "\\\\","\" -replace "'","\\047" -replace '\"','\\\"' -replace "\\n","\\n"
        $edgeServiceSettingsVIEW += ","
        $edgeServiceSettingsVIEW += "\'disclaimerText\': \'"+$settings.Horizon.disclaimerText+"\'"
     }

    $csIPMode=$settings.Horizon.proxyDestinationIPSupport
    if ($csIPMode.length -gt 0) {
        if ($csIPMode -ne "IPV4" -And $csIPMode -ne "IPV6" -And $csIPMode -ne "IPV4_IPV6") {
            WriteErrorString "Error: Invalid proxyDestinationIPSupport value specified. It can be IPV4 or IPV6 or IPV4_IPV6"
            Exit
        }
        $edgeServiceSettingsVIEW += ","
        $edgeServiceSettingsVIEW += "\'proxyDestinationIPSupport\': \'"+$csIPMode.ToUpper()+"\'"
    }

    if ($settings.Horizon.redirectHostMappingList.length -gt 0) {
        $edgeServiceSettingsVIEW += ","
        $edgeServiceSettingsVIEW += "\'redirectHostMappingList\': \'"+$settings.Horizon.redirectHostMappingList+"\'"
    }

    if ($settings.Horizon.idpEntityID.length -gt 0) {
        $edgeServiceSettingsVIEW += ","
        $edgeServiceSettingsVIEW += "\'idpEntityID\': \'"+$settings.Horizon.idpEntityID+"\'"
    }

    if ($settings.Horizon.allowedAudiences.length -gt 0) {
        $edgeServiceSettingsVIEW += ","
        $edgeServiceSettingsVIEW += "\'allowedAudiences\': \'"+$settings.Horizon.allowedAudiences+"\'"
    }
	
	$radiusLabelMaxLength = 20
	
	if ($settings.Horizon.radiusUsernameLabel.length -gt 0) {
		ValidateLabelLength radiusUsernameLabel $settings.Horizon.radiusUsernameLabel $radiusLabelMaxLength
        $edgeServiceSettingsVIEW += ","
        $edgeServiceSettingsVIEW += "\'radiusUsernameLabel\': \'"+$settings.Horizon.radiusUsernameLabel+"\'"
    }
	
	if ($settings.Horizon.radiusPasscodeLabel.length -gt 0) {
		ValidateLabelLength radiusPasscodeLabel $settings.Horizon.radiusPasscodeLabel $radiusLabelMaxLength
        $edgeServiceSettingsVIEW += ","
        $edgeServiceSettingsVIEW += "\'radiusPasscodeLabel\': \'"+$settings.Horizon.radiusPasscodeLabel+"\'"
    }

    if ($settings.Horizon.jwtSettings.length -gt 0) {
        $edgeServiceSettingsVIEW += ","
        $edgeServiceSettingsVIEW += "\'jwtSettings\': \'"+$settings.Horizon.jwtSettings+"\'"

        $edgeServiceSettingsVIEW += ","
        $edgeServiceSettingsVIEW += "\'jwtAudiences\': [ "

        $icount = 0
        for ($i=1; $i -lt 100; $i++) {
            $jwtAudienceName = "jwtAudience$i"
            if ($settings.Horizon.$jwtAudienceName.length -gt 0) {

                if ($icount -gt 0) {
                    $edgeServiceSettingsVIEW += ","
                }
                $icount++
                $edgeServiceSettingsVIEW += "\'"+$settings.Horizon.$jwtAudienceName+"\'"
            }
        }
        $edgeServiceSettingsVIEW += " ]"
    }

    if ($settings.Horizon.logoutOnCertRemoval -eq "true") {
        $edgeServiceSettingsVIEW += ","
        $edgeServiceSettingsVIEW += "\'logoutOnCertRemoval\': true"
    }

    if ($settings.Horizon.samlUnauthUsernameAttribute.length -gt 0) {
        $edgeServiceSettingsVIEW += ","
        $edgeServiceSettingsVIEW += "\'samlUnauthUsernameAttribute\': \'"+$settings.Horizon.samlUnauthUsernameAttribute+"\'"
    }

    if ($settings.Horizon.defaultUnauthUsername.length -gt 0) {
        $edgeServiceSettingsVIEW += ","
        $edgeServiceSettingsVIEW += "\'defaultUnauthUsername\': \'"+$settings.Horizon.defaultUnauthUsername+"\'"
    }

    if ($settings.Horizon.gatewayLocation.length -gt 0) {
        $edgeServiceSettingsVIEW += ","
        $edgeServiceSettingsVIEW += "\'gatewayLocation\': \'"+$settings.Horizon.gatewayLocation+"\'"
    }


    $edgeServiceSettingsVIEW += " }"
    
    $edgeServiceSettingsVIEW
}

#
# Web Reverse Proxy settings
#

function GetEdgeServiceSettingsWRP {
    Param ($settings, $id)

    $WebReverseProxy = "WebReverseProxy"+$id

    $proxyDestinationUrl=$settings.$WebReverseProxy.proxyDestinationUrl

    if ($proxyDestinationUrl.length -le 0) {
        return
    }

    $edgeServiceSettingsWRP += "{ \'identifier\': \'WEB_REVERSE_PROXY\'"
    $edgeServiceSettingsWRP += ","
    $edgeServiceSettingsWRP += "\'enabled\': true"
    $edgeServiceSettingsWRP += ","

    $instanceId=$settings.$WebReverseProxy.instanceId

    if (!$instanceId) {
        $instanceId=""
    }

    if (!$id) {
        $id=""
    }

    if ($instanceId.length -eq 0) {
        if ($id.length -eq 0) {
            $id="0"
        }
        $instanceId=$id
    }

    $edgeServiceSettingsWRP += "\'instanceId\': \'"+$instanceId+"\'"
    $edgeServiceSettingsWRP += ","

    if (($settings.$WebReverseProxy.trustedCert1.length -gt 0) -Or (($settings.$WebReverseProxy.hostEntry1.length -gt 0))) {

        $trustedCertificates = GetTrustedCertificates $WebReverseProxy
        $edgeServiceSettingsWRP += $trustedCertificates
        $edgeServiceSettingsWRP += ","

        $hostEntries = GetHostEntries $WebReverseProxy
        $edgeServiceSettingsWRP += $hostEntries
        $edgeServiceSettingsWRP += ","
    }

    $edgeServiceSettingsWRP += "\'proxyDestinationUrl\': \'"+$proxyDestinationUrl+"\'"

    $proxyDestinationUrlThumbprints=$settings.$WebReverseProxy.proxyDestinationUrlThumbprints  -replace ":","="

    if ($proxyDestinationUrlThumbprints.length -gt 0) {

        #
        # Remove invalid thumbprint characters
        #

        $proxyDestinationUrlThumbprints = $proxyDestinationUrlThumbprints -replace, "[^a-zA-Z0-9,= ]", ""

    	$edgeServiceSettingsWRP += ","
    	$edgeServiceSettingsWRP += "\'proxyDestinationUrlThumbprints\': \'"+$proxyDestinationUrlThumbprints+"\'"
    }

    if ($settings.$WebReverseProxy.proxyPattern.length -gt 0) {
        $edgeServiceSettingsWRP += ","
        $settings.$WebReverseProxy.proxyPattern = $settings.$WebReverseProxy.proxyPattern -replace "\\", "\\\\"
        $edgeServiceSettingsWRP += "\'proxyPattern\': \'"+$settings.$WebReverseProxy.proxyPattern+"\'"
    } else {
        WriteErrorString "Error: Missing proxyPattern in [WebReverseProxy]."
    	Exit
    }

    if ($settings.$WebReverseProxy.unSecurePattern.length -gt 0) {
        $edgeServiceSettingsWRP += ","
        $edgeServiceSettingsWRP += "\'unSecurePattern\': \'"+$settings.$WebReverseProxy.unSecurePattern+"\'"
    }
    
    if ($settings.$WebReverseProxy.authCookie.length -gt 0) {
        $edgeServiceSettingsWRP += ","
        $edgeServiceSettingsWRP += "\'authCookie\': \'"+$settings.$WebReverseProxy.authCookie+"\'"
    }

    if ($settings.$WebReverseProxy.loginRedirectURL.length -gt 0) {
        $edgeServiceSettingsWRP += ","
        $edgeServiceSettingsWRP += "\'loginRedirectURL\': \'"+$settings.$WebReverseProxy.loginRedirectURL+"\'"
    }
    
    $authMethods=$settings.$WebReverseProxy.authMethods
    if ($authMethods.length -gt 0) {
    	$edgeServiceSettingsWRP += ","
    	$edgeServiceSettingsWRP += "\'authMethods\': \'"+$authMethods+"\'"
    }

    if ($settings.$WebReverseProxy.proxyHostPattern.length -gt 0) {
        $edgeServiceSettingsWRP += ","
        $edgeServiceSettingsWRP += "\'proxyHostPattern\': \'"+$settings.$WebReverseProxy.proxyHostPattern+"\'"
    }

    if ($settings.$WebReverseProxy.keyTabPrincipalName.length -gt 0) {
        $edgeServiceSettingsWRP += ","
        $edgeServiceSettingsWRP += "\'keyTabPrincipalName\': \'"+$settings.$WebReverseProxy.keyTabPrincipalName+"\'"
    }

    if ($settings.$WebReverseProxy.targetSPN.length -gt 0) {
        $edgeServiceSettingsWRP += ","
        $edgeServiceSettingsWRP += "\'targetSPN\': \'"+$settings.$WebReverseProxy.targetSPN+"\'"
    }

    if ($settings.$WebReverseProxy.idpEntityID.length -gt 0) {
        $edgeServiceSettingsWRP += ","
        $edgeServiceSettingsWRP += "\'idpEntityID\': \'"+$settings.$WebReverseProxy.idpEntityID+"\'"
    }

    if ($settings.$WebReverseProxy.landingPagePath.length -gt 0) {
        $edgeServiceSettingsWRP += ","
        $edgeServiceSettingsWRP += "\'landingPagePath\': \'"+$settings.$WebReverseProxy.landingPagePath+"\'"
    }

    if ($settings.$WebReverseProxy.userNameHeader.length -gt 0) {
        $edgeServiceSettingsWRP += ","
        $edgeServiceSettingsWRP += "\'userNameHeader\': \'"+$settings.$WebReverseProxy.userNameHeader+"\'"
    }

    if ($settings.$WebReverseProxy.wrpAuthConsumeType.length -gt 0) {
        $edgeServiceSettingsWRP += ","
        $edgeServiceSettingsWRP += "\'wrpAuthConsumeType\': \'"+$settings.$WebReverseProxy.wrpAuthConsumeType+"\'"
    }

    if ($settings.$WebReverseProxy.securityHeaders.length -gt 0) {
        $edgeServiceSettingsWRP += ","
        $securityHeaders = $settings.$WebReverseProxy.securityHeaders
        $securityHeaders = $securityHeaders -replace "'", "\\047"
        $edgeServiceSettingsWRP += "\'securityHeaders\': "+$securityHeaders
    }

    if ($settings.$WebReverseProxy.samlAttributeHeaderMap.length -gt 0) {
        $edgeServiceSettingsWRP += ","
        $edgeServiceSettingsWRP += "\'samlAttributeHeaderMap\': "+$settings.$WebReverseProxy.samlAttributeHeaderMap
    }

    if ($settings.$WebReverseProxy.allowedAudiences.length -gt 0) {
        $edgeServiceSettingsWRP += ","
        $edgeServiceSettingsWRP += "\'allowedAudiences\': \'"+$settings.$WebReverseProxy.allowedAudiences+"\'"
    }

    if ($settings.$WebReverseProxy.healthCheckUrl.length -gt 0) {
        $edgeServiceSettingsWRP += ","
        $edgeServiceSettingsWRP += "\'healthCheckUrl\': \'"+$settings.$WebReverseProxy.healthCheckUrl+"\'"
    }

    $edgeServiceSettingsWRP += "}"

    $edgeServiceSettingsWRP
}

#
# Function to prompt the user for an Airwatch password and attempt to validate the input with the
# Airwatch service. If we are not able to validate the password, we allow the user to continue.
# If we positively determine that the password is invalid (Unauthorized), we re-prompt.
#

function GetAirwatchPwd {
    param($apiServerUrl, $apiServerUsername, $organizationGroupCode)

    while (! $valid) {
        if ($organizationGroupCode.length -gt 0) {
            $prompt='Enter the UEM Console '+$apiServerUrl+' password for user "'+$apiServerUsername+'" (group code "'+$organizationGroupCode+'")'
        } else {
            $prompt='Enter the UEM Console '+$apiServerUrl+' password for user "'+$apiServerUsername+'"'
        }
        $pwd = Read-Host -assecurestring $prompt
        if ($pwd.length -eq 0) {
            Continue
        }
        $pwd = ConvertFromSecureToPlain $pwd
        $secpasswd = ConvertTo-SecureString $pwd -AsPlainText -Force
        $cred = New-Object System.Management.Automation.PSCredential ($apiServerUsername, $secpasswd)
        $valid=1
        $uri=$apiServerUrl+"/API/mdm/gateway/configuration?type=VPN&locationgroupcode="+$organizationGroupCode
        try { 
            $response = Invoke-RestMethod -Uri $uri -Method "Post"  -Credential $cred -ContentType "application/json" -Body '{"randomString" : "abc123"}'
        } catch {
            if ($_.Exception.Response.StatusCode -eq "Unauthorized") {
                write-host "Incorrect password"
                $valid=0
            }
        }
    }

    $pwd
}

#
# Airwatch common settings
# 

function GetEdgeServiceSettingsAWCommon {
    Param ($settings, $groupName, $pwd)
    
    $edgeServiceSettingsAWCommon += ","
    $edgeServiceSettingsAWCommon += "\'enabled\': true"
    $edgeServiceSettingsAWCommon += ","
    $edgeServiceSettingsAWCommon += "\'proxyDestinationUrl\': \'https://null\'"
    $edgeServiceSettingsAWCommon += ","
    $apiServerUrl = $settings.$groupName.apiServerUrl
    
    $edgeServiceSettingsAWCommon += "\'apiServerUrl\': \'"+$apiServerUrl+"\'"
    $edgeServiceSettingsAWCommon += ","
    $apiServerUsername = $settings.$groupName.apiServerUsername
    $apiServerUsername = $apiServerUsername -replace '\\', '\\\\'
    $edgeServiceSettingsAWCommon += "\'apiServerUsername\': \'"+$apiServerUsername+"\'"

    if (!$pwd) {
        $pwd = GetAirwatchPwd $settings.$groupName.apiServerUrl $settings.$groupName.apiServerUsername $settings.$groupName.organizationGroupCode
    }
    if ($pwd.length -gt 0) {
        $edgeServiceSettingsAWCommon += ","
        $edgeServiceSettingsAWCommon += "\'apiServerPassword\': \'"+$pwd+"\'"
    }
    $edgeServiceSettingsAWCommon += ","
    $edgeServiceSettingsAWCommon += "\'organizationGroupCode\': \'"+$settings.$groupName.organizationGroupCode+"\'"
    $edgeServiceSettingsAWCommon += ","
    $edgeServiceSettingsAWCommon += "\'airwatchServerHostname\': \'"+$settings.$groupName.airwatchServerHostname+"\'"
    $edgeServiceSettingsAWCommon += ","

    $edgeServiceSettingsAWCommon += "\'airwatchAgentStartUpMode\': \'install\'"
    
    if ($settings.$groupName.reinitializeGatewayProcess.length -gt 0) {
        $edgeServiceSettingsAWCommon += ","
        $edgeServiceSettingsAWCommon += "\'reinitializeGatewayProcess\': \'"+$settings.$groupName.reinitializeGatewayProcess+"\'"
    }

    if ($settings.$groupName.airwatchOutboundProxy -eq "true") {
        $edgeServiceSettingsAWCommon += ","
    	$edgeServiceSettingsAWCommon += "\'airwatchOutboundProxy\': \'true\'"
    }

    if ($settings.$groupName.outboundProxyPort.length -gt 0) {
        $edgeServiceSettingsAWCommon += ","
        $edgeServiceSettingsAWCommon += "\'outboundProxyPort\': \'"+$settings.$groupName.outboundProxyPort+"\'"
    }

    if ($settings.$groupName.outboundProxyHost.length -gt 0) {
        $edgeServiceSettingsAWCommon += ","
        $edgeServiceSettingsAWCommon += "\'outboundProxyHost\': \'"+$settings.$groupName.outboundProxyHost+"\'"
    }

    if ($settings.$groupName.outboundProxyUsername.length -gt 0) {
        $edgeServiceSettingsAWCommon += ","
        $outboundProxyUsername = $settings.$groupName.outboundProxyUsername
        $outboundProxyUsername = $outboundProxyUsername -replace "\\", "\\\\"
        $edgeServiceSettingsAWCommon += "\'outboundProxyUsername\': \'"+$outboundProxyUsername+"\'"

        $pwd = Read-Host -assecurestring "Enter the password for the VMware tunnel outbound proxy server"
        $pwd = ConvertFromSecureToPlain $pwd
        if ($pwd.length -gt 0) {
            $edgeServiceSettingsAWCommon += ","
            $edgeServiceSettingsAWCommon += "\'outboundProxyPassword\': \'"+$pwd+"\'"
        }
    }

    if ($settings.$groupName.ntlmAuthentication.length -gt 0) {
        $edgeServiceSettingsAWCommon += ","
        $edgeServiceSettingsAWCommon += "\'ntlmAuthentication\': \'"+$settings.$groupName.ntlmAuthentication+"\'"
    }
   
    $edgeServiceSettingsAWCommon
}

#
# Airwatch Tunnel Gateway settings
# 

function GetEdgeServiceSettingsAWTGateway {
    Param ($settings, $groupName, $edgeServiceSettingsAWCommon)
        
    $edgeServiceSettingsAWTGateway += "{ \'identifier\': \'TUNNEL_GATEWAY\'"
    $edgeServiceSettingsAWTGateway += ","
    $edgeServiceSettingsAWTGateway += "\'airwatchComponentsInstalled\':\'TUNNEL\'"
    $edgeServiceSettingsAWTGateway += $edgeServiceSettingsAWCommon

    if (($settings.$groupName.trustedCert1.length -gt 0) -Or (($settings.$groupName.hostEntry1.length -gt 0))) {
        $trustedCertificates = GetTrustedCertificates $groupName
        $edgeServiceSettingsAWTGateway += ","
        $edgeServiceSettingsAWTGateway += $trustedCertificates

        $hostEntries = GetHostEntries $groupName
        $edgeServiceSettingsAWTGateway += ","
        $edgeServiceSettingsAWTGateway += $hostEntries
    }
    $edgeServiceSettingsAWTGateway += "}"

    $edgeServiceSettingsAWTGateway 
}

#
# Airwatch Tunnel Proxy settings
# 

function GetEdgeServiceSettingsAWTProxy {
    Param ($settings, $groupName, $edgeServiceSettingsAWCommon)

    $edgeServiceSettingsAWTProxy += "{ \'identifier\': \'TUNNEL_PROXY\'"
    $edgeServiceSettingsAWTProxy += ","
    $edgeServiceSettingsAWTProxy += "\'airwatchComponentsInstalled\':\'TUNNEL\'"
    $edgeServiceSettingsAWTProxy += $edgeServiceSettingsAWCommon

    $pacFile=$settings.$groupName.pacFilePath
    $pacFileUrl=$settings.$groupName.pacFileUrl
    $credentialsFile=$settings.$groupName.credentialFilePath

    ValidateFileWithExtension $pacFile ".pac" "PAC"
    ValidateFileWithExtension $credentialsFile ".xml" "CREDENTIAL"

   if($pacFile.length -gt 0){
       $pacFileContent = Get-Content -Path $pacFile -Encoding Byte
       $encodedPACFile = [System.Convert]::ToBase64String($pacFileContent)
       $edgeServiceSettingsAWTProxy += ","
       $edgeServiceSettingsAWTProxy += "\'pacFile\': \'"+$encodedPACFile+"\'"
   }Elseif($pacFileUrl.length -gt 0){
       $edgeServiceSettingsAWTProxy += ","
       $edgeServiceSettingsAWTProxy += "\'pacFileUrl\': \'"+$pacFileUrl+"\'"
   }
   if(($credentialsFile.length -gt 0) -and (($pacFileUrl.length -gt 0) -or ($pacFile.length -gt 0))){
       $credentialsFileContent = Get-Content -Path $credentialsFile -Encoding Byte
       $encodedCredentialsFile = [System.Convert]::ToBase64String($credentialsFileContent)
       $edgeServiceSettingsAWTProxy += ","
       $edgeServiceSettingsAWTProxy += "\'credentialsFile\': \'"+$encodedCredentialsFile+"\'"
   } Elseif(($credentialsFile.length -gt 0) -and ($pacFileUrl.length -eq 0) -and ($pacFile.length -eq 0)){
       WriteErrorString "Error:Please provide PAC file path or PAC file url and Credentials file."
       Exit
   }

    $edgeServiceSettingsAWTProxy += "}"
    
    $edgeServiceSettingsAWTProxy 
}
    
#
# Airwatch CG settings
# 

function GetEdgeServiceSettingsAWCG {
    Param ($settings, $groupName, $edgeServiceSettingsAWCommon)
    
    $edgeServiceSettingsAWCG += "{ \'identifier\': \'CONTENT_GATEWAY\'"
    $edgeServiceSettingsAWCG += ","
    $edgeServiceSettingsAWCG += "\'airwatchComponentsInstalled\':\'CG\'"
    $edgeServiceSettingsAWCG += $edgeServiceSettingsAWCommon

    if ($settings.$groupName.cgConfigId.length -gt 0) {
        $edgeServiceSettingsAWCG += ","
        $edgeServiceSettingsAWCG += "\'cgConfigurationId\': \'"+$settings.$groupName.cgConfigId+"\'"
    }

    if (($settings.$groupName.trustedCert1.length -gt 0) -Or (($settings.$groupName.hostEntry1.length -gt 0))) {
        $trustedCertificates = GetTrustedCertificates $groupName
        $edgeServiceSettingsAWCG += ","
        $edgeServiceSettingsAWCG += $trustedCertificates

        $hostEntries = GetHostEntries $groupName
        $edgeServiceSettingsAWCG += ","
        $edgeServiceSettingsAWCG += $hostEntries
    }

    $edgeServiceSettingsAWCG += "}"
    
    $edgeServiceSettingsAWCG 
}

#
# Airwatch SEG settings
#

function GetEdgeServiceSettingsAWSEG {
    Param ($settings, $groupName, $edgeServiceSettingsAWCommon)

    $edgeServiceSettingsAWSEG += "{ \'identifier\': \'SEG\'"
    $edgeServiceSettingsAWSEG += ","
    $edgeServiceSettingsAWSEG += "\'airwatchComponentsInstalled\':\'SEG\'"
    $edgeServiceSettingsAWSEG += $edgeServiceSettingsAWCommon

    if ($settings.$groupName.memConfigurationId.length -gt 0) {
        $edgeServiceSettingsAWSEG += ","
        $edgeServiceSettingsAWSEG += "\'memConfigurationId\': \'"+$settings.$groupName.memConfigurationId+"\'"
    }

    if (($settings.$groupName.trustedCert1.length -gt 0) -Or (($settings.$groupName.hostEntry1.length -gt 0))) {
        $trustedCertificates = GetTrustedCertificates $groupName
        $edgeServiceSettingsAWSEG += ","
        $edgeServiceSettingsAWSEG += $trustedCertificates

        $hostEntries = GetHostEntries $groupName
        $edgeServiceSettingsAWSEG += ","
        $edgeServiceSettingsAWSEG += $hostEntries
    }


    if ($settings.$groupName.pfxCerts.length -gt 0) {
        $pfxCertificate = GetPFXCertificate  $groupName
        $edgeServiceSettingsAWSEG += ","
        $edgeServiceSettingsAWSEG += "\'pfxCerts\': \'"+$pfxCertificate+"\'"

        $pfxCertificatePassword = GetPfxPassword $settings.$groupName.pfxCerts  $groupName
        $edgeServiceSettingsAWSEG += ","
        $edgeServiceSettingsAWSEG += "\'pfxCertsPassword\': \'"+$pfxCertificatePassword+"\'"

		$pfxCertAlias=$settings.$groupName.pfxCertAlias
		if ($pfxCertAlias.length -gt 0) {
			$edgeServiceSettingsAWSEG += ","
			$edgeServiceSettingsAWSEG += "\'pfxCertAlias\': \'"+$pfxCertAlias+"\'"
		}
    }

    $edgeServiceSettingsAWSEG += "}"

    $edgeServiceSettingsAWSEG
}

function ValidateFileWithExtension {
	param ($file, $extn, $type)

	if ($file.length -gt 0) {
		if (!(Test-path $file)) {
			WriteErrorString "Error: ($file) file not found"
			Exit
		}else{
			$extension = [IO.Path]::GetExtension($file)
			if ($extension -ne $extn )
			{
				WriteErrorString "Error:Please provide valid $type file with extension ($extn)"
				Exit
			}

		}
	}
}

function GetAuthMethodSettingsCertificate {
    Param ($settings)
 
    $CertificateAuthCertsFile=$settings.CertificateAuth.pemCerts

    if ($CertificateAuthCertsFile.length -le 0) {
        return
    }

	if (!(Test-path $CertificateAuthCertsFile)) {
		WriteErrorString "Error: PEM Certificate file not found ($CertificateAuthCertsFile)"
		Exit
	}

	$CertificateAuthCerts = (Get-Content $CertificateAuthCertsFile | Out-String) -replace "'", "\\047" -replace "`r`n", "\\n"

	if ($CertificateAuthCerts -like "*-----BEGIN CERTIFICATE-----*") {
		Write-host Deployment will use the specified Certificate Auth PEM file
	} else {
		WriteErrorString "Error: Invalid PEM file ([CertificateAuth] pemCerts) specified. It must contain -----BEGIN CERTIFICATE-----."
		Exit
	}

	$authMethodSettingsCertificate += "{ \'name\': \'certificate-auth\'"
	$authMethodSettingsCertificate += ","
	$authMethodSettingsCertificate += "\'enabled\': true"

    if ($settings.CertificateAuth.enableCertRevocation -eq "true") {
        $crlEnabled="false"
        $ocspEnabled="false"
	    if ($settings.CertificateAuth.enableCertCRL -eq "true") {
            $authMethodSettingsCertificate += ","
            $authMethodSettingsCertificate += "\'enableCertCRL\': \'true\'"
            $crlEnabled="true"
        } else {
            # We have to set enableCertCRL as false if not set by user. Else auth broker sets it to true by default and it may fail
            # in runtime if the certificate does not have a CRL!!
            $authMethodSettingsCertificate += ","
            $authMethodSettingsCertificate += "\'enableCertCRL\': \'false\'"
        }

        if ($settings.CertificateAuth.crlLocation.length -gt 0) {
            $authMethodSettingsCertificate += ","
            $authMethodSettingsCertificate += "\'crlLocation\': \'"+$settings.CertificateAuth.crlLocation+"\'"
            $crlEnabled="true"
        }

        # Not used by authbroker anymore. However having this in case script used with older ova's.
        if ($settings.CertificateAuth.crlCacheSize.length -gt 0) {
            $authMethodSettingsCertificate += ","
        	$authMethodSettingsCertificate += "\'crlCacheSize\': \'"+$settings.CertificateAuth.crlCacheSize+"\'"
        }

        if ($settings.CertificateAuth.enableOCSP -eq "true") {
            $ocspEnabled="true"
            $ocspURLSource = GetOCSPUrlSource($settings)

            if ($settings.CertificateAuth.enableOCSPCRLFailover -eq "true")
            {
                if ($crlEnabled -eq "false") {
                    WriteErrorString "CRL should be setup to enable OCSP to CRL failover"
                    Exit
                }
                $authMethodSettingsCertificate += ","
                $authMethodSettingsCertificate += "\'enableOCSPCRLFailover\': \'true\'"
            } else {
                # We have to set enableOCSPCRLFailover as false if not set by user. Else auth broker sets it to true by default and it will fail
                # during configuration or runtime if the certificate does not have a CRL or CRL settings not configured!!
                $authMethodSettingsCertificate += ","
                $authMethodSettingsCertificate += "\'enableOCSPCRLFailover\': \'false\'"
            }

            if ($settings.CertificateAuth.sendOCSPNonce -eq "true")
            {
               $authMethodSettingsCertificate += ","
               $authMethodSettingsCertificate += "\'sendOCSPNonce\': \'true\'"
            }

            $authMethodSettingsCertificate += ","
            $authMethodSettingsCertificate += "\'enableOCSP\': \'true\'"
            $authMethodSettingsCertificate += ","
            $authMethodSettingsCertificate += "\'ocspURLSource\': \'"+$ocspURLSource+"\'"

            if ($settings.CertificateAuth.ocspURL.length -gt 0) {
                $authMethodSettingsCertificate += ","
                $authMethodSettingsCertificate += "\'ocspURL\': \'"+$settings.CertificateAuth.ocspURL+"\'"
            }

        }
        if ($crlEnabled -eq "false" -And $ocspEnabled -eq "false") {
             WriteErrorString "Enable either CRL or OCSP to enable cert revocation check"
             Exit
        }
        $authMethodSettingsCertificate += ","
        $authMethodSettingsCertificate += "\'enableCertRevocation\': \'true\'"
    }
    
	$authMethodSettingsCertificate += ","
	$authMethodSettingsCertificate += "\'caCertificates\': \'"
	$authMethodSettingsCertificate += $CertificateAuthCerts
	$authMethodSettingsCertificate += "\'"
	$authMethodSettingsCertificate += "}"

    $authMethodSettingsCertificate
}

function GetAuthMethodSettingsSecurID {
    Param ($settings)
    $securidServerConfigFile=$settings.SecurIDAuth.serverConfigFile

    if ($securidServerConfigFile.length -le 0) {
        return
    }

	if (!(Test-path $securidServerConfigFile)) {
		WriteErrorString "Error: SecurID config file not found ($securidServerConfigFile)"
		Exit
	}

	$Content = Get-Content -Path $securidServerConfigFile -Encoding Byte
	$securidServerConfigB64 = [System.Convert]::ToBase64String($Content)

	$authMethodSettingsSecurID += "{ \'name\': \'securid-auth\'"
	$authMethodSettingsSecurID += ","
	$authMethodSettingsSecurID += "\'enabled\': true"

	$numIterations=$settings.SecurIDAuth.numIterations
	$authMethodSettingsSecurID += ","
	if ($numIterations.length -gt 0) {
		$authMethodSettingsSecurID += "\'numIterations\':  \'"+$numIterations+"\'"
	} else {
		$authMethodSettingsSecurID += "\'numIterations\': \'5\'"
	}

	$externalHostName=$settings.SecurIDAuth.externalHostName
	$authMethodSettingsSecurID += ","
	if ($externalHostName.length -gt 0) {
		$authMethodSettingsSecurID += "\'externalHostName\':  \'"+$externalHostName+"\'"
	}

	$internalHostName=$settings.SecurIDAuth.internalHostName
	$authMethodSettingsSecurID += ","
	if ($internalHostName.length -gt 0) {
		$authMethodSettingsSecurID += "\'internalHostName\':  \'"+$internalHostName+"\'"
	}

	$authMethodSettingsSecurID += ","
	$authMethodSettingsSecurID += "\'serverConfig\': \'"
	$authMethodSettingsSecurID += $securidServerConfigB64
	$authMethodSettingsSecurID += "\'"
	$authMethodSettingsSecurID += "}"

    $authMethodSettingsSecurID
}

function GetRADIUSSharedSecret {
    param($hostName)

    while (1) {
        $prompt='Enter the RADIUS server shared secret for host '+$hostName 
        $pwd = Read-Host -assecurestring $prompt

        if ($pwd.length -gt 0) {
            $pwd = ConvertFromSecureToPlain $pwd
            Break
        }
    }

    $pwd
}

function GetAuthMethodSettingsRADIUS {
    Param ($settings)

	$hostName=$settings.RADIUSAuth.hostName

	if ($hostName.length -le 0) {
        return
    }

	$authMethodSettingsRADIUS += "{ \'name\': \'radius-auth\'"
	$authMethodSettingsRADIUS += ","
	$authMethodSettingsRADIUS += "\'enabled\': true"
	$authMethodSettingsRADIUS += ","
    $authMethodSettingsRADIUS += "\'hostName\':  \'"+$hostName+"\'"
	$authMethodSettingsRADIUS += ","
	$authMethodSettingsRADIUS += "\'displayName\':  \'RadiusAuthAdapter\'"

    $sharedSecret = GetRADIUSSharedSecret $settings.RADIUSAuth.hostName
    $authMethodSettingsRADIUS += ","
	$authMethodSettingsRADIUS += "\'sharedSecret\':  \'"+$sharedSecret+"\'"

    $authMethodSettingsRADIUS += ","
	if ($settings.RADIUSAuth.authType.length -gt 0) {
        $authType = $settings.RADIUSAuth.authType -replace "MSCHAPv2", "MSCHAP2"
        $authMethodSettingsRADIUS += "\'authType\':  \'"+$authType+"\'"
    } else {
    	$authMethodSettingsRADIUS += "\'authType\':  \'PAP\'"
    }

    $authMethodSettingsRADIUS += ","
	if ($settings.RADIUSAuth.authPort.length -gt 0) {
		$authMethodSettingsRADIUS += "\'authPort\':  \'"+$settings.RADIUSAuth.authPort+"\'"
    } else {
 		$authMethodSettingsRADIUS += "\'authPort\':  \'1812\'"
    }

	$authMethodSettingsRADIUS += ","
	if ($settings.RADIUSAuth.radiusDisplayHint.length -gt 0) {
		$authMethodSettingsRADIUS += "\'radiusDisplayHint\':  \'"+$settings.RADIUSAuth.radiusDisplayHint+"\'"
	} else {
		$authMethodSettingsRADIUS += "\'radiusDisplayHint\': \'two-factor\'"
	}

	$authMethodSettingsRADIUS += ","
	if ($settings.RADIUSAuth.numIterations.length -gt 0) {
		$authMethodSettingsRADIUS += "\'numIterations\':  \'"+$settings.RADIUSAuth.numIterations+"\'"
	} else {
		$authMethodSettingsRADIUS += "\'numIterations\': \'5\'"
	}

	if ($settings.RADIUSAuth.accountingPort.length -gt 0) {
    	$authMethodSettingsRADIUS += ","
		$authMethodSettingsRADIUS += "\'accountingPort\':  \'"+$settings.RADIUSAuth.accountingPort+"\'"
    }

  	$authMethodSettingsRADIUS += ","
	if ($settings.RADIUSAuth.serverTimeout.length -gt 0) {
		$authMethodSettingsRADIUS += "\'serverTimeout\':  \'"+$settings.RADIUSAuth.serverTimeout+"\'"
    } else {
		$authMethodSettingsRADIUS += "\'serverTimeout\':  \'5\'"
    }

	if ($settings.RADIUSAuth.realmPrefix.length -gt 0) {
    	$authMethodSettingsRADIUS += ","
		$authMethodSettingsRADIUS += "\'realmPrefix\':  \'"+$settings.RADIUSAuth.realmPrefix+"\'"
    }

	if ($settings.RADIUSAuth.realmSuffix.length -gt 0) {
    	$authMethodSettingsRADIUS += ","
		$authMethodSettingsRADIUS += "\'realmSuffix\':  \'"+$settings.RADIUSAuth.realmSuffix+"\'"
    }

	$authMethodSettingsRADIUS += ","
	if ($settings.RADIUSAuth.numAttempts.length -gt 0) {
		$authMethodSettingsRADIUS += "\'numAttempts\':  \'"+$settings.RADIUSAuth.numAttempts+"\'"
	} else {
		$authMethodSettingsRADIUS += "\'numAttempts\': \'3\'"
	}

	if ($settings.RADIUSAuth.enableBasicMSCHAPv2Validation_1 -eq "true" ) {
        $authMethodSettingsRADIUS += ","
		$authMethodSettingsRADIUS += "\'enableBasicMSCHAPv2Validation_1\':  true"
    }

	if ($settings.RADIUSAuth.hostName_2.length -gt 0) {
    	$authMethodSettingsRADIUS += ","
		$authMethodSettingsRADIUS += "\'hostName_2\':  \'"+$settings.RADIUSAuth.hostName_2+"\'"

        $authMethodSettingsRADIUS += ","
	    if ($settings.RADIUSAuth.authType_2.length -gt 0) {
            $authType = $settings.RADIUSAuth.authType_2 -replace "MSCHAPv2", "MSCHAP2"
            $authMethodSettingsRADIUS += "\'authType_2\':  \'"+$authType+"\'"
        } else {
    	    $authMethodSettingsRADIUS += "\'authType_2\':  \'PAP\'"
        }

        $authMethodSettingsRADIUS += ","
	    if ($settings.RADIUSAuth.authPort_2.length -gt 0) {
		    $authMethodSettingsRADIUS += "\'authPort_2\':  \'"+$settings.RADIUSAuth.authPort_2+"\'"
        } else {
 		    $authMethodSettingsRADIUS += "\'authPort_2\':  \'1812\'"
        }

        if ($settings.RADIUSAuth.accountingPort_2.length -gt 0) {
    	    $authMethodSettingsRADIUS += ","
		    $authMethodSettingsRADIUS += "\'accountingPort_2\':  \'"+$settings.RADIUSAuth.accountingPort_2+"\'"
        }

	    $authMethodSettingsRADIUS += ","
	    if ($settings.RADIUSAuth.numAttempts_2.length -gt 0) {
		    $authMethodSettingsRADIUS += "\'numAttempts_2\':  \'"+$settings.RADIUSAuth.numAttempts_2+"\'"
	    } else {
		    $authMethodSettingsRADIUS += "\'numAttempts_2\': \'3\'"
	    }

        $sharedSecret_2 = GetRADIUSSharedSecret $settings.RADIUSAuth.hostName_2
   	    $authMethodSettingsRADIUS += ","
	    $authMethodSettingsRADIUS += "\'sharedSecret_2\':  \'"+$sharedSecret_2+"\'"

	    if ($settings.RADIUSAuth.realmPrefix_2.length -gt 0) {
    	    $authMethodSettingsRADIUS += ","
		    $authMethodSettingsRADIUS += "\'realmPrefix_2\':  \'"+$settings.RADIUSAuth.realmPrefix_2+"\'"
        }

	    if ($settings.RADIUSAuth.realmSuffix_2.length -gt 0) {
    	    $authMethodSettingsRADIUS += ","
		    $authMethodSettingsRADIUS += "\'realmSuffix_2\':  \'"+$settings.RADIUSAuth.realmSuffix_2+"\'"
        }

      	$authMethodSettingsRADIUS += ","
    	if ($settings.RADIUSAuth.serverTimeout_2.length -gt 0) {
	    	$authMethodSettingsRADIUS += "\'serverTimeout_2\':  \'"+$settings.RADIUSAuth.serverTimeout_2+"\'"
        } else {
		    $authMethodSettingsRADIUS += "\'serverTimeout_2\':  \'5\'"
        }

        if ($settings.RADIUSAuth.enableBasicMSCHAPv2Validation_2 -eq "true" ) {
            $authMethodSettingsRADIUS += ","
            $authMethodSettingsRADIUS += "\'enableBasicMSCHAPv2Validation_2\':  true"
        }

        $authMethodSettingsRADIUS += ","
        $authMethodSettingsRADIUS += "\'enabledAux\': true"

    }

	$authMethodSettingsRADIUS += "}"

    $authMethodSettingsRADIUS
}

#
# Get UAG system settings
#

function GetSystemSettings {
    Param ($settings)

    $systemSettings = "\'systemSettings\':"
    $systemSettings += "{"
    $systemSettings += "\'locale\': \'en_US\'"

    if ($settings.General.cipherSuites.length -gt 0) {
        $systemSettings += ","
        $systemSettings += "\'cipherSuites\': \'"+$settings.General.cipherSuites+"\'"
    }

    if ($settings.General.ssl30Enabled -eq "true" ) {
        $systemSettings += ","
        $systemSettings += "\'ssl30Enabled\': \'true\'"
    } else {
        $systemSettings += ","
        $systemSettings += "\'ssl30Enabled\': \'false\'"
    }

    if ($settings.General.tls10Enabled -eq "true" ) {
        $systemSettings += ","
        $systemSettings += "\'tls10Enabled\': \'true\'"
    } else {
        $systemSettings += ","
        $systemSettings += "\'tls10Enabled\': \'false\'"
    }

    if ($settings.General.tls11Enabled.length -gt 0 ) {
        $systemSettings += ","
        $systemSettings += "\'tls11Enabled\': \'"+$settings.General.tls11Enabled+"\'"
    } else {
        $systemSettings += ","
        $systemSettings += "\'tls11Enabled\': \'false\'"
    }

    if ($settings.General.tls12Enabled -eq "false" ) {
        $systemSettings += ","
        $systemSettings += "\'tls12Enabled\': \'false\'"
    } else {
        $systemSettings += ","
        $systemSettings += "\'tls12Enabled\': \'true\'"
    }

    if ($settings.General.tls13Enabled.length -gt 0 ) {
         $systemSettings += ","
         $systemSettings += "\'tls13Enabled\': \'"+$settings.General.tls13Enabled+"\'"
    } else {
         $systemSettings += ","
         $systemSettings += "\'tls13Enabled\': \'true\'"
    }

    if ($settings.General.snmpEnabled -eq "true" ) {
        $systemSettings += ","
        $systemSettings += "\'snmpEnabled\': \'true\'"
    }

    if ($settings.General.ntpServers.length -gt 0) {
        $systemSettings += ","
        $systemSettings += "\'ntpServers\': \'"+$settings.General.ntpServers+"\'"
    }

    if ($settings.General.fallBackNtpServers.length -gt 0) {
        $systemSettings += ","
        $systemSettings += "\'fallBackNtpServers\': \'"+$settings.General.fallBackNtpServers+"\'"
    }

    $maxSystemCPUAllowedValue = GetValue General maxSystemCPUAllowed $settings.General.maxSystemCPUAllowed 1 100 percent False
    if ($maxSystemCPUAllowedValue.length -gt 0) {
        $systemSettings += ","
        $systemSettings += "\'maxSystemCPUAllowed\': \'$maxSystemCPUAllowedValue\'"
    }

	$requestTimeoutValue = GetValue General requestTimeoutMsec $settings.General.requestTimeoutMsec 1 300000 millisecs True
	if ($requestTimeoutValue.length -gt 0) {
		$systemSettings += ","
		$systemSettings += "\'requestTimeoutMsec\': \'$requestTimeoutValue\'"
	}

	$bodyReceiveTimeoutValue = GetValue General bodyReceiveTimeoutMsec $settings.General.bodyReceiveTimeoutMsec 1 300000 millisecs True
	if ($bodyReceiveTimeoutValue.length -gt 0) {
		$systemSettings += ","
		$systemSettings += "\'bodyReceiveTimeoutMsec\': \'$bodyReceiveTimeoutValue\'"
	}


 #   if ($settings.General.source -like "*-fips-*") {
 #       $systemSettings += "\'cipherSuites\': \'TLS_RSA_WITH_AES_256_CBC_SHA256,TLS_RSA_WITH_AES_128_CBC_SHA256,TLS_RSA_WITH_AES_256_CBC_SHA,TLS_RSA_WITH_AES_128_CBC_SHA\'"
 #       $systemSettings += ", "
 #       $systemSettings += "\'ssl30Enabled\': false, \'tls10Enabled\': false, \'tls11Enabled\': false, \'tls12Enabled\': true"
 #   } else {
 #       $systemSettings += "\'cipherSuites\': \'TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384,TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA,TLS_RSA_WITH_AES_256_CBC_SHA,TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256,TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA,TLS_RSA_WITH_AES_128_CBC_SHA\'"
 #       $systemSettings += ", "
 #       $systemSettings += "\'ssl30Enabled\': false, \'tls10Enabled\': false, \'tls11Enabled\': true, \'tls12Enabled\': true"
 #   }
 
    if ($settings.WebReverseProxy.proxyDestinationUrl.length -gt 0) {
        $systemSettings += ", "
        $systemSettings += "\'cookiesToBeCached\': \'none\'"
    }
    if ($settings.General.sessionTimeout.length -gt 0) {
        $systemSettings += ","
        $systemSettings += "\'sessionTimeout\': \'"+$settings.General.sessionTimeout+"\'"
    }
	if ($settings.General.maxConnectionsAllowedPerSession.length -gt 0) {
		ValidateStringIsNumeric $settings.General.maxConnectionsAllowedPerSession 'maxConnectionsAllowedPerSession' 'General'
		$systemSettings += ","
		$systemSettings += "\'maxConnectionsAllowedPerSession\': \'"+$settings.General.maxConnectionsAllowedPerSession+"\'"
	}
    if ($settings.General.syslogUrl.length -gt 0) {
        $systemSettings += ","
        $systemSettings += "\'syslogUrl\': \'"+$settings.General.syslogUrl+"\'"
    }
    if ($settings.General.syslogAuditUrl.length -gt 0) {
        $systemSettings += ","
        $systemSettings += "\'syslogAuditUrl\': \'"+$settings.General.syslogAuditUrl+"\'"
    }
    if ($settings.General.adminPasswordExpirationDays.length -gt 0) {
        $systemSettings += ","
        $systemSettings += "\'adminPasswordExpirationDays\': \'"+$settings.General.adminPasswordExpirationDays+"\'"
    }
    if ($settings.General.uagName.length -gt 0) {
        $systemSettings += ","
        $systemSettings += "\'uagName\': \'"+$settings.General.uagName+"\'"
    }
	if ($settings.General.dnsSearch.length -gt 0) {
		$systemSettings += ","
		$systemSettings += "\'dnsSearch\': \'"+$settings.General.dnsSearch+"\'"
	}
	if ($settings.General.adminDisclaimerText.length -gt 0) {
		$settings.General.adminDisclaimerText = $settings.General.adminDisclaimerText -replace "\\\\","\" -replace "'","\\047" -replace '\"','\\\"' -replace "\\n","\\n"
		$systemSettings += ","
		$systemSettings += "\'adminDisclaimerText\': \'"+$settings.General.adminDisclaimerText+"\'"
	}

	$sysLogType=$settings.General.sysLogType
    if ($sysLogType.length -gt 0) {
        if ($sysLogType -ne "UDP" -And $sysLogType -ne "TLS") {
            WriteErrorString "Error: Invalid sysLogType value specified. It can be UDP or TLS"
            Exit
        }
    } else {
        $sysLogType = "UDP"
    }
    $systemSettings += ","
    $systemSettings += "\'sysLogType\': \'"+$sysLogType.ToUpper()+"\'"

    if ($sysLogType -eq "TLS") {
        $sysLogCACert=$settings.General.syslogServerCACertPem
        if ($sysLogCACert.length -eq 0) {
            WriteErrorString "Error: For syslog settings with TLS mode need to provide a CA certificate (syslogServerCACertPem)"
            Exit
        }
        $certContent = GetPEMCertificateContent $sysLogCACert "syslogServerCACertPem"
        $systemSettings += ","
        $systemSettings += "\'syslogServerCACertPem\': \'"+$certContent+"\'"

        $syslogClientCert=$settings.General.syslogClientCertCertPem
        $syslogClientCertKey=$settings.General.syslogClientCertKeyPem

        if (($syslogClientCert.length -gt 0 -And $syslogClientCertKey.length -eq 0) -Or ($syslogClientCert.length -eq 0 -And $syslogClientCertKey.length -gt 0)) {
            WriteErrorString "Error: For syslog settings provide both client certificate (syslogClientCertCertPem) and key (syslogClientCertKeyPem)"
            Exit
        }

        if ($syslogClientCert.length -gt 0) {
            $certContent = GetPEMCertificateContent $syslogClientCert "syslogClientCertCertPem"
            $systemSettings += ","
            $systemSettings += "\'syslogClientCertCertPem\': \'"+$certContent+"\'"

            $rsaprivkey = GetPEMPrivateKeyContent $syslogClientCertKey "syslogClientCertKeyPem"
            $systemSettings += ","
            $systemSettings += "\'syslogClientCertKeyPem\': \'"+$rsaprivkey+"\'"
        }
        $sysLogList = GetTLSSyslogServerList $settings
        $systemSettings += ","
        $systemSettings += $sysLogList
    }

    if (($settings.General.sshEnabled -eq "true") -and ($settings.General.sshKeyAccessEnabled -eq "true")) {
        $sshPublicKeyList = GetSSHPublicKeys $settings
        $systemSettings += ","
        $systemSettings += $sshPublicKeyList
    }
    $systemSettings += "}"

    $systemSettings
}

function GetSSHPublicKeys {
    param($settings)
    $sshPublicKeyList = "\'sshPublicKeys\': [ "
    for($i=1;;$i++)
        {
    	    $sshKey = "sshPublicKey$i"
    	    $sshKey = $settings.General.$sshKey
    	    if($sshKey.length -gt 0)
    	    {
    		    if (!(Test-path $sshKey)) {
    			    WriteErrorString "Error: SSH public key file not found ($sshKey)"
    			    Exit
    		    }
    		    else
    		    {
    			    $content = (Get-Content $sshKey | Out-String) -replace "'", "\\047" -replace "`r`n", "\\n"

    			    if ($content -like "*ssh-rsa*") {
    	    			#Write-host "valid ssh public key"
    			    } else {
    				    WriteErrorString "Error: Invalid ssh public key file. It must contain ssh-rsa"
    				    Exit
    			    }
    			    $fileName = $sshKey.SubString($sshKey.LastIndexof('\')+1)
    			    $sshPublicKeyList += "{ \'name\': \'$fileName\'"
    			    $sshPublicKeyList += ","
    			    $sshPublicKeyList += "\'data\': \'"
    			    $sshPublicKeyList += $content
    			    $sshPublicKeyList += "\'"
    			    $sshPublicKeyList += "},"
    		    }
    	    }
    	    else {
                $sshPublicKeyList = $sshPublicKeyList.Substring(0, $sshPublicKeyList.Length-1)
    		    break;
    	    }
        }
        $sshPublicKeyList += "]"

        $sshPublicKeyList
}

function GetTLSSyslogServerList {
    param($settings)
    $sysLogList = "\'tlsSyslogServerSettings\': [ "
    for($i=1;;$i++)
    {

	    $sysLogServer = "tlsSyslogServerSettings$i"
	    $sysLogServer = $settings.General.$sysLogServer

        if ($sysLogServer.length -gt 0) {
            if ($i -gt 2) {
               WriteErrorString "Only two syslog servers are allowed"
               Exit
            }

            $hostName,$port,$acceptedPeer = $sysLogServer.split('|')
            $sysLogList += "{ \'hostname\': \'$hostName\'"

            if ($port.length -gt 0) {
                $sysLogList += ","
                $sysLogList += "\'port\': \'$port\'"
            }
            if ($acceptedPeer.length -gt 0) {
                $sysLogList += ","
                $sysLogList += "\'acceptedPeer\': \'$acceptedPeer\'"
            }
            $sysLogList += "},"
        } else {
            $sysLogList = $sysLogList.Substring(0, $sysLogList.Length-1)
            break;
        }

	}
    $sysLogList += "]"
    $sysLogList
}

function GetPEMCertificateContent {
    Param ($cert, $field)
    if (!(Test-path $cert)) {
        WriteErrorString "Error: PEM Certificate file not found ($cert)"
        Exit
    }

    $certContent = (Get-Content $cert | Out-String) -replace "'", "\\047" -replace "`r`n", "\\n"

    if ($certContent -like "*-----BEGIN CERTIFICATE-----*") {
        #Write-host "valid cert"
    } else {
        WriteErrorString "Error: Invalid certificate file for $field. It must contain -----BEGIN CERTIFICATE-----."
        Exit
    }
    $certContent
}

function GetPEMPrivateKeyContent {
   Param ($rsaPrivKeyFile, $field)
   if (!(Test-path $rsaPrivKeyFile)) {
   	  WriteErrorString "Error: RSA private key file not found ($rsaPrivKeyFile)"
   	  Exit
   }

    $rsaprivkey = (Get-Content $rsaPrivKeyFile | Out-String) -replace "'", "\\047" -replace "`r`n", "\\n" -replace """", ""
    $rsaprivkey = $rsaprivkey.Substring($rsaprivkey.IndexOf("-----BEGIN"))

    if ($rsaprivkey -like "*-----BEGIN RSA PRIVATE KEY-----*") {
        #Write-Host "valid private key"
    } else {
        WriteErrorString "Error: Invalid private key PEM file specified for $field. It must contain an RSA private key."
        Exit
    }
    $rsaprivkey
}
#
# UAG Edge service settings
#

function GetEdgeServiceSettings {
    Param ($settings)

    $edgeCount = 0

    $edgeServiceSettings = "\'edgeServiceSettingsList\':"
    $edgeServiceSettings += "{ \'edgeServiceSettingsList\': ["

    #
    # Horizon View edge service
    #

    $edgeServiceSettingsVIEW += GetEdgeServiceSettingsVIEW($settings)
    if ($edgeServiceSettingsVIEW.length -gt 0) {
        $edgeServiceSettings += $edgeServiceSettingsVIEW
        $edgeCount++
    }

    #
    # Web Reverse Proxy edge services
    #

    for ($i=0; $i -lt 100; $i++) {

        if ($i -eq 0) {
            $id=""
        } else {
            $id=$i
        }
        $edgeServiceSettingsWRP = ""

        $edgeServiceSettingsWRP += GetEdgeServiceSettingsWRP $settings $id
        if ($edgeServiceSettingsWRP.length -gt 0) {
            if ($edgeCount -gt 0) {
                $edgeServiceSettings += ", "
            }
            $edgeServiceSettings += $edgeServiceSettingsWRP
            $edgeCount++
        }
    }

    if (($settings.AirWatch.tunnelGatewayEnabled -eq "true") -or ($settings.Airwatch.tunnelProxyEnabled -eq "true") -or `
        ($settings.Airwatch.contentGatewayEnabled -eq "true") -or ($settings.Airwatch.secureEmailGatewayEnabled -eq "true")) {

        $groupName = "Airwatch"

        $edgeServiceSettingsAWCommon += GetEdgeServiceSettingsAWCommon $settings $groupName $awAPIServerPwd

        #
        # Airwatch Tunnel Gateway edge service
        #

        if ($settings.AirWatch.tunnelGatewayEnabled -eq "true") {
            $edgeServiceSettingsAWTGateway += GetEdgeServiceSettingsAWTGateway $settings $groupName $edgeServiceSettingsAWCommon

            if ($edgeServiceSettingsAWTGateway.length -gt 0) {
    	        if ($edgeCount -gt 0) {
                    $edgeServiceSettings += ", "
                }
                $edgeServiceSettings += $edgeServiceSettingsAWTGateway
                $edgeCount++
            }
        }

        #
        # Airwatch Tunnel Proxy edge service
        #

        if ($settings.Airwatch.tunnelProxyEnabled -eq "true") {

            $edgeServiceSettingsAWTProxy += GetEdgeServiceSettingsAWTProxy $settings $groupName $edgeServiceSettingsAWCommon
    
            if ($edgeServiceSettingsAWTProxy.length -gt 0) {
    	        if ($edgeCount -gt 0) {
                    $edgeServiceSettings += ", "
                }
                $edgeServiceSettings += $edgeServiceSettingsAWTProxy
                $edgeCount++
            }
        }

        #
        # Airwatch CG edge service
        #

        if ($settings.Airwatch.contentGatewayEnabled -eq "true") {
            $edgeServiceSettingsAWCG += GetEdgeServiceSettingsAWCG $settings $groupName $edgeServiceSettingsAWCommon
    
            if ($edgeServiceSettingsAWCG.length -gt 0) {
    	        if ($edgeCount -gt 0) {
                    $edgeServiceSettings += ", "
                }
                $edgeServiceSettings += $edgeServiceSettingsAWCG
                $edgeCount++
            }
        }

        #
        # Airwatch SEG edge service
        #

        if ($settings.Airwatch.secureEmailGatewayEnabled -eq "true") {
            $edgeServiceSettingsAWCG += GetEdgeServiceSettingsAWSEG $settings $groupName $edgeServiceSettingsAWCommon
            if ($edgeServiceSettingsAWSEG.length -gt 0) {

                if ($edgeCount -gt 0) {
                    $edgeServiceSettings += ", "
                }
                $edgeServiceSettings += $edgeServiceSettingsAWSEG
                $edgeCount++
            }
        }
    }


    if (($settings.AirWatch.tunnelGatewayEnabled -ne "true") -and ($settings.AirwatchTunnelGateway.apiServerUsername.length -gt 0)) {

        $groupName = "AirwatchTunnelGateway"

        $edgeServiceSettingsAWCommon = GetEdgeServiceSettingsAWCommon $settings $groupName $awTunnelGatewayAPIServerPwd

        #
        # Airwatch Tunnel Gateway edge service
        #

        $edgeServiceSettingsAWTGateway += GetEdgeServiceSettingsAWTGateway $settings $groupName $edgeServiceSettingsAWCommon

        if ($edgeServiceSettingsAWTGateway.length -gt 0) {
    	    if ($edgeCount -gt 0) {
                $edgeServiceSettings += ", "
            }
            $edgeServiceSettings += $edgeServiceSettingsAWTGateway
            $edgeCount++
        }
    }

    if (($settings.AirWatch.tunnelProxyEnabled -ne "true") -and ($settings.AirwatchTunnelProxy.apiServerUsername.length -gt 0)) {

        $groupName = "AirwatchTunnelProxy"

        $edgeServiceSettingsAWCommon = GetEdgeServiceSettingsAWCommon $settings $groupName $awTunnelProxyAPIServerPwd
        #
        # Airwatch Tunnel Proxy edge service
        #


        $edgeServiceSettingsAWTProxy += GetEdgeServiceSettingsAWTProxy $settings $groupName $edgeServiceSettingsAWCommon
    
        if ($edgeServiceSettingsAWTProxy.length -gt 0) {
    	    if ($edgeCount -gt 0) {
                $edgeServiceSettings += ", "
            }
            $edgeServiceSettings += $edgeServiceSettingsAWTProxy
            $edgeCount++
        }
    }

    if (($settings.AirWatch.contentGatewayEnabled -ne "true") -and ($settings.AirwatchContentGateway.apiServerUsername.length -gt 0)) {

        $groupName = "AirwatchContentGateway"

        $edgeServiceSettingsAWCommon = GetEdgeServiceSettingsAWCommon $settings $groupName $awCGAPIServerPwd

        #
        # Airwatch CG edge service
        #

        $edgeServiceSettingsAWCG += GetEdgeServiceSettingsAWCG $settings $groupName $edgeServiceSettingsAWCommon
    
        if ($edgeServiceSettingsAWCG.length -gt 0) {
    	    if ($edgeCount -gt 0) {
                $edgeServiceSettings += ", "
            }
            $edgeServiceSettings += $edgeServiceSettingsAWCG
            $edgeCount++
        }
    }


    if (($settings.AirWatch.secureEmailGatewayEnabled -ne "true") -and ($settings.AirwatchSecureEmailGateway.apiServerUsername.length -gt 0)) {

        $groupName = "AirwatchSecureEmailGateway"

        $edgeServiceSettingsAWCommon = GetEdgeServiceSettingsAWCommon $settings $groupName $awSEGAPIServerPwd
		
        #
        # Airwatch SEG edge service
        #

        $edgeServiceSettingsAWSEG += GetEdgeServiceSettingsAWSEG $settings $groupName $edgeServiceSettingsAWCommon

        if ($edgeServiceSettingsAWSEG.length -gt 0) {
            if ($edgeCount -gt 0) {
                $edgeServiceSettings += ", "
            }
            $edgeServiceSettings += $edgeServiceSettingsAWSEG
            $edgeCount++
        }
    }

    $edgeServiceSettings += "] }"

    $edgeServiceSettings
}

#
# Auth Method settings
#

function GetAuthMethodSettings {
    Param ($settings)

    $authMethodSettingsCertificate = GetAuthMethodSettingsCertificate($settings)

    $authMethodSettingsSecurID = GetAuthMethodSettingsSecurID($settings)

    $authMethodSettingsRADIUS = GetAuthMethodSettingsRADIUS($settings)

    $authMethodSettings = "\'authMethodSettingsList\':"
    $authMethodSettings += "{ \'authMethodSettingsList\': ["

    $authCount=0

    if ($authMethodSettingsCertificate.length -gt 0) {
	    $authMethodSettings += $authMethodSettingsCertificate
	    $authCount++
    }

    if ($authMethodSettingsSecurID.length -gt 0) {
	    if ($authCount -gt 0) {
		    $authMethodSettings += ","
	    }
	    $authMethodSettings += $authMethodSettingsSecurID
	    $authCount++
    }

    if ($authMethodSettingsRADIUS.length -gt 0) {
	    if ($authCount -gt 0) {
		    $authMethodSettings += ","
	    }
	    $authMethodSettings += $authMethodSettingsRADIUS
	    $authCount++
    }

    $authMethodSettings += "] }"

    $authMethodSettings
}

function GetKeytabSettings {
    Param ($settings)

    $keyTabCount = 0

    $keytabSettings = "\'kerberosKeyTabSettingsList\':"
    $keytabSettings += "{ \'kerberosKeyTabSettings\': ["

    for($i=1; $i -le 100; $i++) {

        $section = "KerberosKeyTabSettings$i"

        if ($settings.$section.keyTab.length -gt 0) {

            if ($settings.$section.principalName.length -eq 0) {
                WriteErrorString "Error: Invalid .INI file. Missing principalName in [KerberosKeyTabSettings$i]."
                Exit
            }

            $fileName = $settings.$section.keyTab

            if (!(Test-path $filename)) {
                WriteErrorString "Error: keyTab file $fileName in [$section] not found"
                Exit
            }

            $Content = Get-Content -Path $fileName -Encoding Byte
	        $keyTabB64 = [System.Convert]::ToBase64String($Content)

            $principalName = $settings.$section.principalName

            if ($keyTabCount -gt 0) {
                $keytabSettings += ", "
            }

            $keytabSettings += "{"

            $keytabSettings += "\'principalName\': \'$principalName\'"
            $keytabSettings += ", "
            $keytabSettings += "\'keyTab\': \'$keyTabB64\'"
            $keytabSettings += "}"

            $keyTabCount++
        }
    }

    $keytabSettings += "]"
    $keytabSettings += "}"

    $keytabSettings
}

function GetkerberosRealmSettings {
    Param ($settings)

    $realmCount = 0

    $kerberosRealmSettings = "\'kerberosRealmSettingsList\':"
    $kerberosRealmSettings += "{ \'kerberosRealmSettingsList\': ["

    for($i=1; $i -le 100; $i++) {

        $section = "KerberosRealmSettings$i"

        if ($settings.$section.name.length -gt 0) {

            $name = $settings.$section.name

            if ($realmCount -gt 0) {
                $kerberosRealmSettings += ", "
            }

            $kerberosRealmSettings += "{"

            $kerberosRealmSettings += "\'name\': \'$name\'"
            $kerberosRealmSettings += ", "

            $kdcTimeout = $settings.$section.kdcTimeout

            $kerberosRealmSettings += "\'kdcHostNameList\': ["

            $hostCount = 0

            for($j=1; $j -le 100; $j++) {

                $hostLabel = "kdcHostNameList$j"
                $host = $settings.$section.$hostLabel

                if ($host.length -gt 0) {

                    if ($hostCount -gt 0) {
                        $kerberosRealmSettings += ", "
                    }

                $kerberosRealmSettings += "\'$host\'"
                $hostCount++
                }

            }

            $kerberosRealmSettings += "]"

            if ($kdcTimeout.length -gt 0) {
                $kerberosRealmSettings += ", "
                $kerberosRealmSettings += "\'kdcTimeout\': \'$kdcTimeout\'"
            }

            $kerberosRealmSettings += "}"

            $realmCount++
        }
    }

    $kerberosRealmSettings += "]"
    $kerberosRealmSettings += "}"

    $kerberosRealmSettings
}

function GetIDPExternalMetadataSettings {
    Param ($settings)

    $metaDataCount = 0

    $externalMetadataSettings = "\'idPExternalMetadataSettingsList\':"
    $externalMetadataSettings += "{ \'idPExternalMetadataSettingsList\': ["

    for($i=1; $i -le 100; $i++) {

        $section = "IDPExternalMetadata$i"

        if ($settings.$section.metadataXmlFile.length -gt 0) {

            $fileName = $settings.$section.metadataXmlFile

            if (!(Test-path $filename)) {
                WriteErrorString "Error: metadataXmlFile file $fileName in [$section] not found"
                Exit
            }

            $Content = Get-Content -Path $fileName -Encoding Byte
	        $metaDataB64 = [System.Convert]::ToBase64String($Content)

            if ($metaDataCount -gt 0) {
                $externalMetadataSettings += ", "
            }

            $externalMetadataSettings += "{"

            $externalMetadataSettings += "\'metadata\': \'$metaDataB64\'"

			if ($settings.$section.entityID.length -gt 0) {
				$externalMetadataSettings += ","
				$externalMetadataSettings += "\'entityID\':  \'"+$settings.$section.entityID+"\'"
			}

            if ($settings.$section.forceAuthN -eq "true") {
                $externalMetadataSettings += ","
                $externalMetadataSettings += "\'forceAuthN\': true"
            }

            $externalMetadataSettings += "}"

            $metaDataCount++
        }
    }

    $externalMetadataSettings += "]"
    $externalMetadataSettings += "}"

    $externalMetadataSettings
}

function GetLoadBalancerSettings {
    Param ($settings)

    if ($settings.HighAvailability.groupID.length -gt 0) {

        if ($settings.HighAvailability.virtualIPAddress.length -eq 0) {
            WriteErrorString "Error: Invalid .INI file. Missing VirtualIPAddress in [HighAvailability]."
            Exit
        }

        $loadBalancerSettings = "\'loadBalancerSettings\': {"
        $loadBalancerSettings += "\'groupID\':  \'"+$settings.HighAvailability.GroupID+"\'"
        $loadBalancerSettings += ", "
        $loadBalancerSettings += "\'virtualIPAddress\':  \'"+$settings.HighAvailability.VirtualIPAddress+"\'"
        $loadBalancerSettings += ", "
        $loadBalancerSettings += "\'loadBalancerMode\':  \'ONEARM\'"
        $loadBalancerSettings += "}"

    }

    $loadBalancerSettings
}


function GetDevicePolicySettings {
    Param ($settings)

    if ($settings.OpswatEndpointComplianceCheckSettings.clientKey.length -gt 0) {

        $devicePolicySettings = "\'devicePolicySettingsList\':"
        $devicePolicySettings += "{ \'devicePolicySettingsList\': ["
        $devicePolicySettings += "{"

 
        $devicePolicySettings += "\'name\':  \'OPSWAT\'"
        $devicePolicySettings += ", "
        $devicePolicySettings += "\'userName\':  \'"+$settings.OpswatEndpointComplianceCheckSettings.clientKey+"\'"

        if ($settings.OpswatEndpointComplianceCheckSettings.clientSecret.length -gt 0) {
            $devicePolicySettings += ", "
            $devicePolicySettings += "\'password\':  \'"+$settings.OpswatEndpointComplianceCheckSettings.clientSecret+"\'"
        } else {
            WriteErrorString "Error: Invalid .INI file. Missing clientSecret value in [OpswatEndpointComplianceCheckSettings]."
		    Exit
        }

        if ($settings.OpswatEndpointComplianceCheckSettings.hostName.length -gt 0) {
            $devicePolicySettings += ", "
            $devicePolicySettings += "\'hostName\':  \'"+$settings.OpswatEndpointComplianceCheckSettings.hostName+"\'"
        }

        $devicePolicyAllowedStatuses = GetDevicePolicyAllowedStatuses ($settings)

        $complianceCheckIntervals = GetComplianceCheckIntervals OpswatEndpointComplianceCheckSettings $settings

        $hostedResources = GetDevicePolicySettingsHostedResources $settings

        $devicePolicySettings += $devicePolicyAllowedStatuses + $complianceCheckIntervals + $hostedResources + "} ] }"
    }

    $devicePolicySettings

}

function GetDevicePolicySettingsHostedResources {
	Param ($settings)

	$platformTable = GetPlatformTable

    # "hostedResourceMap" JSON section begins
	$hostedResourceSettings = ","
	$hostedResourceSettings += "\'hostedResourceMap\': { "

    # Loop through all platforms present in hash table
	foreach ($entry in $platformTable.GetEnumerator()) {
		$platformNameInINI = $($entry.Name)
		$platformObj = $($entry.Value)

        #The name of the INI section would be OpswatEndpointComplianceCheckSettings-macOS OR OpswatEndpointComplianceCheckSettings-Windows
		$hostedResourceSectionName = "OpswatEndpointComplianceCheckSettings-$platformNameInINI"

		$url = $settings.$hostedResourceSectionName.url

		if ($url.length -gt 0) {
			# This gets the JSON key to be used for the specified platform name from the platformTable and adds that to JSON
			# e.g. - for macOS (name present in INI section), the JSON key is -- "Mac". With this, JSON section for
			# a specific platform begins, e.g. "Mac" : {
			$hostedResourceSettings += "\'" + $platformObj.jsonKey + "\': { "

            # "resourceURLSettings" (for a specific platform, e.g macOS or Windows) JSON section begins
			$hostedResourceSettings += "\'resourceURLSettings\': { "
			#Adding url
			$hostedResourceSettings += "\'url\': \'$url\'"

			#Adding thumbprints
			$urlThumbprints=$settings.$hostedResourceSectionName.urlThumbprints  -replace ":","="
			if ($urlThumbprints.length -gt 0) {
				# Removing invalid thumbprint characters
				$urlThumbprints = $urlThumbprints -replace, "[^a-zA-Z0-9,=*]", ""

				$hostedResourceSettings += ","
				$hostedResourceSettings += "\'urlThumbprints\': \'$urlThumbprints\'"
			}

			#Adding trusted certificates
			if ($settings.$hostedResourceSectionName.trustedCert1.length -gt 0) {
				$trustedCertificates = GetTrustedCertificates $hostedResourceSectionName
				$hostedResourceSettings += ","
				$hostedResourceSettings += $trustedCertificates
			}


			#Adding response refresh interval
			$urlResponseRefreshIntervalValue = GetValue $hostedResourceSectionName urlResponseRefreshInterval $settings.$hostedResourceSectionName.urlResponseRefreshInterval 10 86400 secs True
			if ($urlResponseRefreshIntervalValue.length -gt 0) {
				$hostedResourceSettings += ","
				$hostedResourceSettings += "\'urlResponseRefreshInterval\': \'$urlResponseRefreshIntervalValue\'"
			}
			# "resourceURLSettings" (for a specific platform, e.g macOS or Windows) JSON section ends
			$hostedResourceSettings += " }"

            # "hostedResourceMetadata" (for a specific platform, e.g macOS or Windows) JSON section begins
			$hostedResourceSettings += ","
			$hostedResourceSettings += "\'hostedResourceMetadata\': { "

            # Getting the list of INI fields to be ignored for a specific platform
			$listOfFieldsToBeIgnored = $platformObj.listOfFieldsToBeIgnored

            # Getting the list of mandatory INI fields for a specific platform
            $mandatoryFields = $platformObj.mandatoryFields

            # Adding fields like name, params, executable (if not marked as ignored) in "hostedResourceMetadata" JSON section
            $hostedResourceSettings += addJsonElement $hostedResourceSectionName $mandatoryFields $listOfFieldsToBeIgnored name $settings.$hostedResourceSectionName.name

            $hostedResourceSettings += addJsonElement $hostedResourceSectionName $mandatoryFields $listOfFieldsToBeIgnored params $settings.$hostedResourceSectionName.params

            $hostedResourceSettings += addJsonElement $hostedResourceSectionName $mandatoryFields $listOfFieldsToBeIgnored executable $settings.$hostedResourceSectionName.executable

			# Removing the extra delimiter (comma), if present at the end
			$hostedResourceSettings = removeTrailingDelimiter $hostedResourceSettings

			# "hostedResourceMetadata" JSON section ends
			$hostedResourceSettings += " }"

			# JSON section for a specific platform ends
			$hostedResourceSettings += " }"

			# A new platform section may begin now. Hence putting the delimiter
			$hostedResourceSettings += ","
		}
	}

    # Removing the extra delimiter (comma), if present at the end
	$hostedResourceSettings = removeTrailingDelimiter $hostedResourceSettings

	# "hostedResourceMap" JSON section ends
	$hostedResourceSettings += "}"

	$hostedResourceSettings
}

# Prepares a hashtable with key as platform name as specified in INI section and value as :
# 1) The JSON key string to be used for that platform
# 2) The list of INI fields to be ignored (and hence not fed to settings JSON) in that INI section, e.g  executable is
# only needed for macOS and not for Windows.
function GetPlatformTable {
    $platformTable = @{

        "macOS" = [pscustomobject]@{ jsonKey = "Mac"; listOfFieldsToBeIgnored = @(); mandatoryFields = @("executable") }

        "Windows" = [pscustomobject]@{ jsonKey = "Windows"; listOfFieldsToBeIgnored = @("executable"); mandatoryFields = @() }
    }

    $platformTable
}

function GetComplianceCheckIntervals {
	Param ($iniGroup, $settings)

	$complianceCheckIntervalValue = GetValue $iniGroup complianceCheckInterval $settings.OpswatEndpointComplianceCheckSettings.complianceCheckInterval 5 1440 mins True

	$complianceCheckFastIntervalValue = GetValue $iniGroup complianceCheckFastInterval $settings.OpswatEndpointComplianceCheckSettings.complianceCheckFastInterval 1 1440 mins True

	# If complianceCheckInterval is not specified (i.e., it is zero) and complianceCheckFastInterval is specified and is a non-zero integer OR if both are specified and
	# complianceCheckInterval is less than complianceCheckFastInterval, then throwing an error message

	if (($complianceCheckIntervalValue.length -eq 0 -and $complianceCheckFastIntervalValue.length -gt 0 -and $complianceCheckFastIntervalValue -ne 0) -or
		($complianceCheckIntervalValue.length -gt 0 -and $complianceCheckFastIntervalValue.length -gt 0 -and ($complianceCheckIntervalValue -lt $complianceCheckFastIntervalValue))) {

		WriteErrorString "Error: complianceCheckInterval (if not specified, the value is 0) cannot be less than complianceCheckFastInterval"
        Exit
	}

    if ($complianceCheckIntervalValue.length -gt 0) {
		$complianceCheckIntervals = ", "
		$complianceCheckIntervals += "\'complianceCheckInterval\':  \'"+$complianceCheckIntervalValue+"\'"
	}

    if ($complianceCheckFastIntervalValue.length -gt 0) {
		$complianceCheckIntervals += ", "
		$complianceCheckIntervals += "\'complianceCheckFastInterval\':  \'"+$complianceCheckFastIntervalValue+"\'"
	}
	$complianceCheckIntervals
}

function GetDevicePolicyAllowedStatuses {
	Param ($settings)

	$devicePolicyAllowedStatuses = ", "
	$devicePolicyAllowedStatuses += "\'allowedStatuses\': ["

	#Gets the hash table which maps status bucket names as specified in INI fields to 1) the status to be specified in JSON and 2) the default value of each status
	$deviceStatusTable = GetDevicePolicyStatusTable

	# Loop through all status buckets present in hash table
	foreach ($entry in $deviceStatusTable.GetEnumerator()) {
		$statusKey = $($entry.Name)
		$statusValue = $($entry.Value)

		# Get the specified INI field for the current status
		$statusINIFieldName = $settings.OpswatEndpointComplianceCheckSettings.$statusKey

		# If a value for the current status bucket is specified in INI file (the valid values are true or false, case-insensitive)
		if ($statusINIFieldName.length -gt 0) {

			# If the value of the current status bucket in INI file is true, such as : allowAssessmentPending=true (case-insensitive check) indicating that this status # is allowed to gain access to EUC resources, then adding the corresponding mapped JSON status name to allowedStatuses list
			if ($statusINIFieldName -ieq "true") {
				$allowedStatuses += "\'" + $statusValue.statusNameInJSON + "\'"
				$allowedStatuses += ","
			} Elseif ($statusINIFieldName -ine "false" -And $statusValue.defaultValue -ieq "true") {
				# If the value of the current status bucket in INI file is not false (case-insensitive check) but some invalid text like allowDeviceNotFound=invalid,
				# and the mapped default value of this status is true, then adding the corresponding mapped JSON status name to allowedStatuses list. If the default # value is false, we don't need to do anything
				$allowedStatuses += SpecifyDevicePolicyStatusDefaultValue($statusValue)
			}
		} Elseif ($statusValue.defaultValue -ieq "true") {
			# If the current status bucket is not specified in INI file and the mapped default value of this staus is true, then adding the corresponding mapped JSON # status name to allowedStatuses list. If the default value is false, we don't need to do anything
			$allowedStatuses += SpecifyDevicePolicyStatusDefaultValue($statusValue)
		}
	}

	if ($allowedStatuses -ne $null) {
		$allowedStatuses = $allowedStatuses.Substring(0, $allowedStatuses.Length-1)
	}

	$devicePolicyAllowedStatuses += $allowedStatuses + "] "
	$devicePolicyAllowedStatuses
}

# Prepares a hash table of all possible device policy status buckets with key as the field name, which would be specified in INI file and value as a PS custom object
# containing two members : 1) the name of that status bucket to be specified in JSON, 2) the default value of this status bucket, if not specified in INI file or value
# of this is specified as some invalid text (neither true nor false; case-insensitive).
#
# If we need to add support for a new status bucket in future, all we need to do is just add one more entry in this hash table with INI field name as key and the
# corresponding JSON value and the default value.

function GetDevicePolicyStatusTable {
    $statusTable = @{

        "allowInCompliance" = [pscustomobject]@{ statusNameInJSON = "COMPLIANT"; defaultValue = "true" }

        "allowNotInCompliance" = [pscustomobject]@{ statusNameInJSON = "NON_COMPLIANT"; defaultValue = "false" }

        "allowOutOfLicenseUsage" = [pscustomobject]@{ statusNameInJSON = "OUT_OF_LICENSE_USAGE"; defaultValue = "false" }

        "allowAssessmentPending" = [pscustomobject]@{ statusNameInJSON = "ASSESSMENT_PENDING"; defaultValue = "false" }

        "allowEndpointUnknown" = [pscustomobject]@{ statusNameInJSON = "NOT_FOUND"; defaultValue = "false" }

        "allowOthers" = [pscustomobject]@{ statusNameInJSON = "OTHERS"; defaultValue = "false" }
    }

    $statusTable
}

# It is used to add a device policy status in JSON (provied the default value is true) when that specific status field is not present in INI or present with an invalid value.

function SpecifyDevicePolicyStatusDefaultValue {
	Param ($statusValue)

    $statusWithDefaultValue = "\'" + $statusValue.statusNameInJSON + "\'"
	$statusWithDefaultValue += ","
	$statusWithDefaultValue
}

function GetWorkspaceOneIntelligenceSettings {
	Param ($settings)

	if ($settings.WorkspaceOneIntelligence.workspaceOneIntelligenceUrl.length -eq 0) {
		Return
	}

	if ($settings.WorkspaceOneIntelligence.tokenEndpointUrl.length -eq 0) {
		Return
	}

	if ($settings.WorkspaceOneIntelligence.encodedCredentialsFileContent.length -eq 0) {
		Return
	}

	$fileName = $settings.WorkspaceOneIntelligence.encodedCredentialsFileContent

	if (!(Test-path $filename)) {
		WriteErrorString "Error: Workspace One Intelligence credentials file $fileName in [WorkspaceOneIntelligence] not found"
		Exit
	}


	$workspaceOneIntelligenceSettings = "\'workspaceOneIntelligenceSettings\': {"
	$workspaceOneIntelligenceSettings += "\'enabled\': true"
	$workspaceOneIntelligenceSettings += ", "
	$workspaceOneIntelligenceSettings += "\'tokenEndpointUrl\':  \'" + $settings.WorkspaceOneIntelligence.tokenEndpointUrl + "\'"
	$workspaceOneIntelligenceSettings += ", "
	$workspaceOneIntelligenceSettings += "\'workspaceOneIntelligenceUrl\':  \'" + $settings.WorkspaceOneIntelligence.workspaceOneIntelligenceUrl + "\'"

	#Adding public key refresh interval
	if ($settings.WorkspaceOneIntelligence.updateInterval.length -gt 0) {
		try {
			$updateInterval = [int]$settings.WorkspaceOneIntelligence.updateInterval
		} catch {
			WriteErrorString "Error: updateInterval in [WorkspaceOneIntelligence] is not an integer"
			Exit
		}
		if (($updateInterval -ne 0) -and (($updateInterval -lt 10) -or ($updateInterval -gt 86400))) {
			WriteErrorString "Error: Update interval can be either 0 or between 10 secs - 86400 secs (both inclusive) in [WorkspaceOneIntelligence]"
			Exit
		}
		$workspaceOneIntelligenceSettings += ","
		$workspaceOneIntelligenceSettings += "\'updateInterval\':  " + $settings.WorkspaceOneIntelligence.updateInterval
	}

	$Content = Get-Content -Path $fileName -Encoding Byte
	$credentialsB64 = [System.Convert]::ToBase64String($Content)

	$workspaceOneIntelligenceSettings += ", "
	$workspaceOneIntelligenceSettings += "\'encodedCredentialsFileContent\': \'$credentialsB64\'"


	#Adding thumbprints
	$urlThumbprints=$settings.WorkspaceOneIntelligence.urlThumbprints  -replace ":","="
	if ($urlThumbprints.length -gt 0) {
		#
		# Remove invalid thumbprint characters
		#
		$urlThumbprints = $urlThumbprints -replace, "[^a-zA-Z0-9,= ]", ""

		$workspaceOneIntelligenceSettings += ","
		$workspaceOneIntelligenceSettings += "\'urlThumbprints\': \'$urlThumbprints\'"
	}

	#Adding trusted certificates
	if ($settings.WorkspaceOneIntelligence.trustedCert1.length -gt 0) {
		$trustedCertificates = GetTrustedCertificates WorkspaceOneIntelligence
		$workspaceOneIntelligenceSettings += ","
		$workspaceOneIntelligenceSettings += $trustedCertificates
	}

	$workspaceOneIntelligenceSettings += "}"

	$workspaceOneIntelligenceSettings
}

function GetJSONSettings {
    Param ($settings)

    $settingsJSON = "{"

    $certificateWrapper = GetCertificateWrapper ($settings)
    if ($certificateWrapper.length -gt 0) {
        $settingsJSON += $certificateWrapper
        $settingsJSON += ", "
    }

    $certificateWrapperAdmin = GetCertificateWrapper $settings "Admin"
    if ($certificateWrapperAdmin.length -gt 0) {
        $settingsJSON += $certificateWrapperAdmin
        $settingsJSON += ", "
    }
    
    $systemSettings = GetSystemSettings ($settings)

    $edgeServiceSettings = GetEdgeServiceSettings ($settings)

    $authMethodSettings = GetAuthMethodSettings ($settings)
        
    $samlServiceProviderMetadata = GetSAMLServiceProviderMetadata ($settings)

    $samlIdentityProviderMetadata = GetSAMLIdentityProviderMetadata ($settings)

    $jwtSettings = GetJWTSettings ($settings)
    if ($jwtSettings.length -gt 0) {
        $settingsJSON += $jwtSettings
        $settingsJSON += ", "
    }

    $loadBalancerSettings = GetLoadBalancerSettings ($settings)
    if ($loadBalancerSettings.length -gt 0) {
        $settingsJSON += $loadBalancerSettings
        $settingsJSON += ", "
    }

    $keytabSettings = GetKeytabSettings ($settings)
    if ($keytabSettings.length -gt 0) {
        $settingsJSON += $keytabSettings
        $settingsJSON += ", "
    }

    $kerberosRealmSettings = GetkerberosRealmSettings ($settings)
    if ($kerberosRealmSettings.length -gt 0) {
        $settingsJSON += $kerberosRealmSettings
        $settingsJSON += ", "
    }

    $externalMetadataSettings = GetIDPExternalMetadataSettings ($settings)
    if ($externalMetadataSettings.length -gt 0) {
        $settingsJSON += $externalMetadataSettings
        $settingsJSON += ", "
    }

    $devicePolicySettings = GetDevicePolicySettings ($settings)
    if ($devicePolicySettings.length -gt 0) {
        $settingsJSON += $devicePolicySettings
        $settingsJSON += ", "
    }

	$workspaceOneIntelligenceSettings = GetWorkspaceOneIntelligenceSettings ($settings)
	if ($workspaceOneIntelligenceSettings.length -gt 0) {
		$settingsJSON += $workspaceOneIntelligenceSettings
		$settingsJSON += ", "
	}

    $ocspSigningCertList = GetOCSPSigningCertSettings ($settings)
    if ($ocspSigningCertList.length -gt 0) {
        $settingsJSON += $ocspSigningCertList
        $settingsJSON += ", "
    }

    $settingsJSON += $edgeServiceSettings+", "+$systemSettings+", "+$authMethodSettings+", "+$samlServiceProviderMetadata+", "+$samlIdentityProviderMetadata+"}"

    $settingsJSON = $settingsJSON -replace "'", '"'
    $settingsJSON = $settingsJSON -replace "\\047", "'"

    #If any value in INI file ends with backslash (which comes up in settingsJson as \\"), that needs to be actually included in settingsJson as four backslashes one after the other; the fifth backslash (from the beginning, in the second argument of the function) is used to escape double quote. So, the complete replacement would be \\\\\". If this is not done, the resulting settingsJson will not be well-formed and hence cannot be parsed by UAG's admin. The last 3 characters, (i.e. ,\"), are used so that this gets matched with only those values which end with backslash (these characters signify the beginning of another json key).

    $settingsJSON = $settingsJSON.replace('\\",\"','\\\\\",\"')

    $settingsJSON

}

function AddKVPUnit {
    param($VMName, $key, $value)
    
    #
    # Add Key-Value Pairs for the VM
    #

    #if ($key.Contains("Password")) {
    #    Write-Host "Setting $key=******"
    #} else {
    #    Write-Host "Setting $key=$value"
    #}

    $VmMgmt = gwmi -n "Root\Virtualization\V2" Msvm_VirtualSystemManagementService #Get-WmiObject -Namespace root\virtualization\v2 -Class Msvm_VirtualSystemManagementService
    $Vm = gwmi -n "root\virtualization\v2" Msvm_ComputerSystem|?{$_.ElementName -eq $VMName }  #Get-WmiObject -Namespace root\virtualization\v2 -Class Msvm_ComputerSystem -Filter {ElementName='TEST-APP38'}     # has to be same as $VMName

    $kvpDataItem = ([WMIClass][String]::Format("\\{0}\{1}:{2}", $VmMgmt.ClassPath.Server, $VmMgmt.ClassPath.NamespacePath, "Msvm_KvpExchangeDataItem")).CreateInstance()
    $null=$KvpItem.psobject.properties
    $kvpDataItem.Name = $key
    $kvpDataItem.Data = $value
    $kvpDataItem.Source = 0
    $result = $VmMgmt.AddKvpItems($Vm, $kvpDataItem.PSBase.GetText(1))

    $job = [wmi]$result.Job

    if (!$job) {
        WriteErrorString "Error: Failed to set KVP $key on $VMName"
        Return
    }    

    if ($job) {
        $job.get()
        #write-host $job.jobstate
        #write-host $job.SystemProperties.Count.ToString()
    }

    while($job.jobstate -lt 7) {
	    $job.get()
        Start-Sleep -Seconds 2
    } 

    if ($job.ErrorCode -ne 0) {
        WriteErrorString "Error: Failed to set KVP $key on $VMName (error code $($job.ErrorCode))"
        Return
    }    
    
    if ($job.Status -ne "OK") {
        WriteErrorString "Error: Failed to set KVP $key on $VMName (status $job.Status)"
        Return
    }

    $job
}

function GetKVP {
    param($VMName, $key)

    $VmMgmt = gwmi -n "Root\Virtualization\V2" Msvm_VirtualSystemManagementService #Get-WmiObject -Namespace root\virtualization\v2 -Class Msvm_VirtualSystemManagementService
    $Vm = gwmi -n "root\virtualization\v2" Msvm_ComputerSystem|?{$_.ElementName -eq $VMName }  #Get-WmiObject -Namespace root\virtualization\v2 -Class Msvm_ComputerSystem -Filter {ElementName='TEST-APP38'}     # has to be same as $VMName

    $n = $vm.GetRelated("Msvm_KvpExchangeComponent").GuestIntrinsicExchangeItems
    $n = $vm.GetRelated("Msvm_KvpExchangeComponent").GuestExchangeItems

    $n = $vm.GetRelated("Msvm_KvpExchangeComponent").GetRelated('Msvm_KvpExchangeComponentSettingData').HostExchangeItems

    $n | % {
        $GuestExchangeItemXml = ([XML]$_).SelectSingleNode(`
            "/INSTANCE/PROPERTY[@NAME='Name']/VALUE[child::text()='$key']")

        if ($GuestExchangeItemXml -ne $null)
        {
            $val = $GuestExchangeItemXml.SelectSingleNode( `
                "/INSTANCE/PROPERTY[@NAME='Data']/VALUE/child::text()").Value
                $val
                Return
        }
    }
}

function AddKVP {
    param($VMName, $key, $value)
    $max = 1000
    $len=$value.length
    $index=0
    if ($len -le $max) {
        $job=AddKVPUnit $VMName $key $value
    } else {
        for ($i=0; $i -lt $len; $i += $max) {
            $chunkSize = [Math]::Min($max, ($len - $i))
            $valueChunk=$value.Substring($i, $chunkSize)
            $keyChunk=$key+"."+$index.ToString(0)
            $job=AddKVPUnit $VMName $keyChunk $valueChunk
            $index++
        }
    }
    $job
}

function DeleteKVP {
    param($VMName, $key)

    $VmMgmt = gwmi -n "Root\Virtualization\V2" Msvm_VirtualSystemManagementService
    $Vm = gwmi -n "root\virtualization\v2" Msvm_ComputerSystem|?{$_.ElementName -eq $VMName }

    $kvpDataItem = ([WMIClass][String]::Format("\\{0}\{1}:{2}", $VmMgmt.ClassPath.Server, $VmMgmt.ClassPath.NamespacePath, "Msvm_KvpExchangeDataItem")).CreateInstance()
    $null=$KvpItem.psobject.properties
    $kvpDataItem.Name = $key
    $kvpDataItem.Data = [String]::Empty
    $kvpDataItem.Source = 0
    $result = $VmMgmt.RemoveKvpItems($Vm, $kvpDataItem.PSBase.GetText(1))

    $job = [wmi]$result.Job

    if (!$job) {
        WriteErrorString "Error: Failed to set KVP $key on $VMName"
        Return
    }    

    if ($job) {
        $job.get()
        #write-host $job.jobstate
        #write-host $job.SystemProperties.Count.ToString()
    }

    while($job.jobstate -lt 7) {
	    $job.get()
        Start-Sleep -Seconds 2
    } 

    if ($job.ErrorCode -ne 0) {
        WriteErrorString "Error: Failed to set KVP $key on $VMName (error code $($job.ErrorCode))"
        Return
    }    
    
    $job
}

function DeleteKVPAll {
    param($VMName)

    $VmMgmt = gwmi -n "Root\Virtualization\V2" Msvm_VirtualSystemManagementService
    $Vm = gwmi -n "root\virtualization\v2" Msvm_ComputerSystem|?{$_.ElementName -eq $VMName }

    $hostExchangeItems = $vm.GetRelated("Msvm_KvpExchangeComponent").GetRelated('Msvm_KvpExchangeComponentSettingData').HostExchangeItems

    $hostExchangeItems | % {

    $GuestExchangeItemXml = ([XML]$_).SelectSingleNode(`
        "/INSTANCE/PROPERTY[@NAME='Name']/VALUE")
        $key = $GuestExchangeItemXml.InnerText

        if ($key.length -gt 0) {
            $job = DeleteKVP $VMName $key
        }      
    }
}

function IsVMDeployed {
    param ($VMName, $ipAddress)
    #
    # WE consider the VM to be deployed if we can obtain an IP address from it and the ip address matches what is configured.
    #

    $out=Get-VM $VMName | ?{$_.ReplicationMode -ne "Replica"} | Select -ExpandProperty NetworkAdapters | Select IPAddresses
    if ($ipAddress.length -gt 0) {
        #static IP address
        if (($out.IPAddresses)[0] -eq $ipAddress) {
            return $true
        }
    } else {
        #DHCP
        if  ((($out.IPAddresses)[0]).length -gt 0) {
            $wait = 0
            #sleep for 4 mins (240 secs) to give time for the appliance to be ready
            while ($wait -le 240) {
                Write-Host -NoNewline "."
                $wait += 2
                Start-Sleep -Seconds 2
            }
            return $true
        }
    }

    return $false
}

function IsVMUp {
    param ($VMName, [ref]$ipAddress)

    #
    # WE consider the VM to be up if we can obtain an IP address from it.
    #

    $out=Get-VM $VMName | ?{$_.ReplicationMode -ne "Replica"} | Select -ExpandProperty NetworkAdapters | Select IPAddresses
    if ($out.IPAddresses.Length -gt 0) {
        $ipAddress = ($out.IPAddresses)[0]
        return $true
    }

    return $false
}

function GetNetOptions {
    Param ($settings, $nic)

    $ipModeLabel = "ipMode" + $nic
    $ipMode = $settings.General.$ipModeLabel
    
    $ipLabel = "ip" + $nic
    $ip=$settings.General.$ipLabel

    $netmaskLabel = "netmask" + $nic
    $netmask=$settings.General.$netmaskLabel

    $v6ipLabel = "v6ip" + $nic
    $v6ip=$settings.General.$v6ipLabel

    $v6ipprefixLabel = "v6ipprefix" + $nic
    $v6ipprefix=$settings.General.$v6ipprefixLabel

    #
    # IPv4 address must have a netmask
    #

    if (($ip.length -gt 0) -and ($netmask.length -eq 0)) {
        WriteErrorString "Error: missing value $netmaskLabel."
        Exit
    }

    #
    # IPv6 address must have a prefix
    #

    if (($v6ip.length -gt 0) -and ($v6ipprefix.length -eq 0)) {
        WriteErrorString "Error: missing value $v6ipprefixLabel."
        Exit
    }

    #
    # If ipMode is not specified, assign a default
    #

    if ($ipMode.length -eq 0) {

        $ipMode = "DHCPV4"

        if (($ip.length -gt 0) -and ($v6ip.length -eq 0)) {
            $ipMode = "STATICV4"
        }

        if (($ip.length -eq 0) -and ($v6ip.length -gt 0)) {
            $ipMode = "STATICV6"
        }

        if (($ip.length -gt 0) -and ($v6ip.length -gt 0)) {
            $ipMode = "STATICV4+STATICV6"
        }
    }

    #
    # Assign network properties based on the 11 supported combinations
    #

    switch ($ipMode) {

           
        { ($_ -eq "DHCPV4") -or ($_ -eq "DHCPV4+DHCPV6") -or ($_ -eq "DHCPV4+AUTOV6") -or ($_ -eq "DHCPV6") -or ($_ -eq "AUTOV6") } {

            #
            # No addresses required
            #

            $options = " --prop:ipMode$nic='"+$ipMode+"'"
            $options
            return
        }

        { ($_ -eq "STATICV6") -or ($_ -eq "DHCPV4+STATICV6") } {

            #
            # IPv6 address and prefix required
            #

           if ($v6ip.length -eq 0) {
               WriteErrorString "Error: missing value $v6ipLabel."
               Exit
           }
           $options = " --prop:ipMode$nic='"+$ipMode+"' --prop:v6ip$nic='"+$v6ip+"' --prop:forceIpv6Prefix$nic='"+$v6ipprefix+"'"
           $options
           return
        }

        { ($_ -eq "STATICV4") -or ($_ -eq "STATICV4+DHCPV6") -or ($_ -eq "STATICV4+AUTOV6") } {

            #
            # IPv4 address and netmask required
            #

           if ($ip.length -eq 0) {
               WriteErrorString "Error: missing value $ipLabel."
               Exit
           }
           $options = " --prop:ipMode$nic='"+$ipMode+"' --prop:ip$nic='"+$ip+"' --prop:forceNetmask$nic='"+$netmask+"'"
           $options
           return
        }

        { "STATICV4+STATICV6" } {

            #
            # IPv4 address, netmask, IPv6 address and prefix required
            #

           if ($ip.length -eq 0) {
               WriteErrorString "Error: missing value $ipLabel."
               Exit
           }
           if ($v6ip.length -eq 0) {
               WriteErrorString "Error: missing value $v6ipLabel."
               Exit
           }
           $options = " --prop:ipMode$nic='"+$ipMode+"'  --prop:ip$nic='"+$ip+"' --prop:forceNetmask$nic='"+$netmask+"' --prop:v6ip$nic='"+$v6ip+"' --prop:forceIpv6Prefix$nic='"+$v6ipprefix+"'"
           $options
           return
        }

        #
        # Invalid
        #

        default {
            WriteErrorString "Error: Invalid value ($ipModeLabel=$ipMode)."
		    Exit
      
        }
    }
}

function SetKVPNetOptions {
    Param ($settings, $VMName, $nic)

    $ipModeLabel = "ipMode" + $nic
    $ipMode = $settings.General.$ipModeLabel
    
    $ipLabel = "ip" + $nic
    $ip=$settings.General.$ipLabel

    $netmaskLabel = "netmask" + $nic
    $netmask=$settings.General.$netmaskLabel

    $v6ipLabel = "v6ip" + $nic
    $v6ip=$settings.General.$v6ipLabel

    $v6ipprefixLabel = "v6ipprefix" + $nic
    $v6ipprefix=$settings.General.$v6ipprefixLabel

    #
    # IPv4 address must have a netmask
    #

    if (($ip.length -gt 0) -and ($netmask.length -eq 0)) {
        WriteErrorString "Error: missing value $netmaskLabel."
        Exit
    }

    #
    # IPv6 address must have a prefix
    #

    if (($v6ip.length -gt 0) -and ($v6ipprefix.length -eq 0)) {
        WriteErrorString "Error: missing value $v6ipprefixLabel."
        Exit
    }

    #
    # If ipMode is not specified, assign a default
    #

    if ($ipMode.length -eq 0) {

        $ipMode = "DHCPV4"

        if (($ip.length -gt 0) -and ($v6ip.length -eq 0)) {
            $ipMode = "STATICV4"
        }

        if (($ip.length -eq 0) -and ($v6ip.length -gt 0)) {
            $ipMode = "STATICV6"
        }

        if (($ip.length -gt 0) -and ($v6ip.length -gt 0)) {
            $ipMode = "STATICV4+STATICV6"
        }
    }

    #
    # Assign network properties based on the 11 supported combinations
    #

    switch ($ipMode) {

           
        { ($_ -eq "DHCPV4") -or ($_ -eq "DHCPV4+DHCPV6") -or ($_ -eq "DHCPV4+AUTOV6") -or ($_ -eq "DHCPV6") -or ($_ -eq "AUTOV6") } {

            #
            # No addresses required
            #

            $job=AddKVP $VMName "ipMode$nic" $ipMode

            return
        }

        { ($_ -eq "STATICV6") -or ($_ -eq "DHCPV4+STATICV6") } {

            #
            # IPv6 address and prefix required
            #

           if ($v6ip.length -eq 0) {
               WriteErrorString "Error: missing value $v6ipLabel."
               Exit
           }
           $job=AddKVP $VMName "ipMode$nic" $ipMode
           $job=AddKVP $VMName "v6ip$nic" $v6ip
           $job=AddKVP $VMName "forceIpv6Prefix$nic" $v6ipprefix

           return
        }

        { ($_ -eq "STATICV4") -or ($_ -eq "STATICV4+DHCPV6") -or ($_ -eq "STATICV4+AUTOV6") } {

            #
            # IPv4 address and netmask required
            #

           if ($ip.length -eq 0) {
               WriteErrorString "Error: missing value $ipLabel."
               Exit
           }
           $job=AddKVP $VMName "ipMode$nic" $ipMode
           $job=AddKVP $VMName "ip$nic" $ip
           $job=AddKVP $VMName "forceNetmask$nic" $netmask

           return
        }

        { "STATICV4+STATICV6" } {

            #
            # IPv4 address, netmask, IPv6 address and prefix required
            #

           if ($ip.length -eq 0) {
               WriteErrorString "Error: missing value $ipLabel."
               Exit
           }
           if ($v6ip.length -eq 0) {
               WriteErrorString "Error: missing value $v6ipLabel."
               Exit
           }
           $job=AddKVP $VMName "ipMode$nic" $ipMode
           $job=AddKVP $VMName "ip$nic" $ip
           $job=AddKVP $VMName "forceNetmask$nic" $netmask
           $job=AddKVP $VMName "v6ip$nic" $v6ip
           $job=AddKVP $VMName "forceIpv6Prefix$nic" $v6ipprefix

           return
        }

        #
        # Invalid
        #

        default {
            WriteErrorString "Error: Invalid value ($ipModeLabel=$ipMode)."
		    Exit
      
        }
    }
}
function GetSettingsJSONProperty {
    param($value)
    $max = 65535
    $len=$value.length
    $index=0
	$key="settingsJSON"
	$jsonString=""
	$maxLengthAllowed=16 *$max
    if ($len -le $max) {
        $jsonString=AddSettingJsonUnit $jsonString $key $value
    } Elseif ($len -gt $maxLengthAllowed){
	   WriteErrorString "Provided settings exceeds max allowed settings that can be deployed."
       Exit 1
	}
	else {
        for ($i=0; $i -lt $len; $i += $max) {
            $chunkSize = [Math]::Min($max, ($len - $i))
            $valueChunk=$value.Substring($i, $chunkSize)
            $keyChunk=$key+"-"+$index.ToString(0)
            $jsonString=AddSettingJsonUnit $jsonString $keyChunk $valueChunk
            $index++
        }
    }
    $jsonString
}

function AddSettingJsonUnit {
    param($existingValue, $key, $value)
	$existingValueLen=$existingValue.length
	$jsonString=$existingValue
	if ($existingValueLen -gt 0) {
	   $jsonString += "`r`n"
	}
	$jsonString += "prop:"+$key+"="
	$jsonString += $value

	$jsonString

}

function GetDeploymentSettingOption {
    param ($settings)
    $deploymentOption=$settings.General.deploymentOption

    if (!$deploymentOption) {
    	$deploymentOption="onenic"
    } Elseif ($deploymentOption -eq "onenic-L") {
        $deploymentOption="onenic-large"
    } Elseif ($deploymentOption -eq "twonic-L") {
        $deploymentOption="twonic-large"
    } Elseif ($deploymentOption -eq "threenic-L") {
        $deploymentOption="threenic-large"
    }

    $deploymentOption
}

function GetOCSPUrlSource {
    Param($settings)

    # Use OCSPURLSource if specified by user (or due to ini export!!) then use that value. Else try to deduce with other settings
    if ($settings.CertificateAuth.ocspURLSource.length -gt 0) {
        $ocspURLSource = $settings.CertificateAuth.ocspURLSource
    } Else {
        $useOCSPUrlInCert="false"
        if ($settings.CertificateAuth.useOCSPUrlInCert -eq "true") {
           $useOCSPUrlInCert="true"
        }
        $ocspUrl = $settings.CertificateAuth.ocspUrl

        if ($useOCSPUrlInCert -eq "false" -And $ocspUrl.length -eq 0) {
           WriteErrorString "Either use ocsp url in certificate [useOCSPUrlInCert=true] or set an ocsp url[ocspUrl] to use"
           Exit
        } ElseIf ($useOCSPUrlInCert -eq "true" -And $ocspUrl.length -gt 0) {
           $ocspURLSource="cert_and_config"
        } ElseIf ($useOCSPUrlInCert -eq "true") {
           $ocspURLSource="cert_only_required"
        } Else {
           $ocspURLSource="config_only"
        }
    }

    $ocspURLSource
}

function GetOCSPSigningCertSettings {
   Param($settings)
   $ocspCertCount = 0
   $allCerts = "\'ocspSigningCerts\': [ "
   for($i=1;;$i++)
   {
   	    $cert = "ocspSigningCert$i"
   	    $cert = $settings.OCSPSigningCertificates.$cert
   	    if($cert.length -gt 0)
   	    {
   		    if (!(Test-path $cert)) {
   			    WriteErrorString "Error: PEM Certificate file not found ($cert)"
   			    Exit
   		    }
   		    else
   		    {
   			    $content = (Get-Content $cert | Out-String) -replace "'", "\\047" -replace "`r`n", "\\n"

   			    if ($content -like "*-----BEGIN CERTIFICATE-----*") {

   			    } else {
   				    WriteErrorString "Error: Invalid certificate file It must contain -----BEGIN CERTIFICATE-----."
   				    Exit
   			    }
                $allCerts += "\'"+$content+"\'"+","
   			}
   			$ocspCertCount++
   	    } else {
            $allCerts = $allCerts.Substring(0, $allCerts.Length-1)
            break;
        }
    }
    $allCerts += "]"

   if ($ocspCertCount -gt 0) {
      $ocspSigningCertList = "\'ocspSigningCertList\': { " + $allCerts + " }"
   }

   $ocspSigningCertList
}

function ValidateLabelLength {
    Param ($labelName, $label, $labelMaxLength)
	
	if($label.length -gt $labelMaxLength) {
		WriteErrorString "Error: $labelName cannot have more than $labelMaxLength characters"
   		Exit
	}	
}

# Checks if a field's value is an integer and is either 0 OR falls within the passed range
function GetValue {
    Param ($iniSection, $fieldName, $fieldValue, $fieldMinValue, $fieldMaxValue, $timeUnit, $isZeroAllowed)

	if ($fieldValue.length -gt 0) {
		try {
			$fieldValue = [int]$fieldValue
		} catch {
			WriteErrorString "Error: $fieldName in the section $iniSection is not an integer"
            Exit
		}
		if ($isZeroAllowed -ieq "True") {
		    if (($fieldValue -ne 0) -and (($fieldValue -lt $fieldMinValue) -or ($fieldValue -gt $fieldMaxValue))) {
                WriteErrorString "Error: $fieldName can be either 0 or between $fieldMinValue $timeUnit - $fieldMaxValue $timeUnit (both inclusive) in the section $iniSection"
                Exit
            }
		} Elseif (($fieldValue -lt $fieldMinValue) -or ($fieldValue -gt $fieldMaxValue)) {
            WriteErrorString "Error: $fieldName should be between $fieldMinValue $timeUnit - $fieldMaxValue $timeUnit (both inclusive) in the section $iniSection"
            Exit
		}
	}
	$fieldValue
}

# Adds a JSON key, value pair and returns that string
function addJsonElement {
    Param ($iniSection, $mandatoryFields, $listOfFieldsToBeIgnored, $fieldNameToCheck, $fieldValue)

    $isFieldMandatory = "" + $mandatoryFields.Contains($fieldNameToCheck)
    $isFieldToBeIgnored = "" + $listOfFieldsToBeIgnored.Contains($fieldNameToCheck)

    # If a field's value is not specified and it is marked as mandatory
    if ($fieldValue.length -eq 0 -and $isFieldMandatory -ieq "True") {
		WriteErrorString "Error: $fieldNameToCheck is mandatory to be specified in the section $iniSection"
		Exit
	}

    # If a field's value is specified and it is not marked as ignored
    if ($fieldValue.length -gt 0 -and $isFieldToBeIgnored -ieq "False") {
		$uagSettings = "\'" + $fieldNameToCheck + "\': \'$fieldValue\'"
		$uagSettings += ","
	}

	$uagSettings
}

function removeTrailingDelimiter {
	Param ($uagSettings)

	if ($uagSettings.Substring($uagSettings.Length-1, 1) -eq ",") {
		$uagSettings = $uagSettings.Substring(0, $uagSettings.Length-1)
	}

	$uagSettings
}

# The Windows environments from which PowerShell script is used to deploy UAG on Azure or EC2 may not have ovftool
# installed as it's not a requirement for installation on these hypervisors. Hence, VMware dir may not be present
# in APPDATA dir, if no other VMWare product is installed in that Windows machine. Since, the configured UAG
# settings are temporarily persisted in a file in VMware dir in that machine, presence of VMware dir is a must.
# This function creates that dir, if it is not present. This is not needed for HyperV.
function SetUp {

	$vmwareDir = "${env:APPDATA}\VMware"

	if (!(Test-path $vmwareDir)) {
		Write-host "Creating the directory $vmwareDir, since it is not present"
		New-Item -ItemType Directory -Force -Path $vmwareDir | out-null
		
		if(!(Test-path $vmwareDir)) {
			WriteErrorString "Error: The directory $vmwareDir could not be created."
			exit
		}	
	}
}
