#
#  Copyright (C) 2018, 2019, 2020 VMware Inc. All rights reserved.
#
#  Permission is hereby granted, free of charge, to any person obtaining a copy
#  of the software in this file (the "Software"), to deal in the Software
#  without restriction, including without limitation the rights to use, copy,
#  modify, merge, publish, distribute, sublicense, and/or sell copies of the
#  Software, and to permit persons to whom the Software is furnished to do so,
#  subject to the following conditions:
#
#  The above copyright notice and this permission notice shall be included in
#  all copies or substantial portions of the Software.
#
#  The names "VMware" and "VMware, Inc." must not be used to endorse or promote
#  products derived from the Software without the prior written permission of
#  VMware, Inc.
#
#  Products derived from the Software may not be called "VMware", nor may
#  "VMware" appear in their name, without the prior written permission of
#  VMware, Inc.
#
#  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
#  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
#  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
#  VMWARE,INC. BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
#  IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
#  CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#
#

<#
    .SYNOPSIS
     Sample Powershell script to deploy a VMware UAG virtual appliance to Amazon AWS EC2.
    .EXAMPLE
     .\uagdeployec2.ps1 uag1.ini
#>

param([string]$iniFile = "uag.ini", [string] $rootPwd, [string] $adminPwd, [string] $ceipEnabled,
      [string] $awAPIServerPwd, [string] $awTunnelGatewayAPIServerPwd, [string] $awTunnelProxyAPIServerPwd, [string] $awCGAPIServerPwd, [string] $awSEGAPIServerPwd)

function SetNICDeleteOnTermination {
    Param ($settings, $eth)

    if ($eth.NetworkInterfaceId.length -gt 0) {
        $net=Get-EC2NetworkInterface -NetworkInterfaceId $eth.NetworkInterfaceId
        if ($net.Attachment.AttachmentId.length -gt 0) {
            Edit-EC2NetworkInterfaceAttribute -NetworkInterfaceId $eth.NetworkInterfaceId -Attachment_DeleteOnTermination $true -Attachment_AttachmentId $net.Attachment.AttachmentId
        }
    }
}

function CreateNIC {
    Param ($settings, $nic, $description)

    $privateIPAddress = $settings.AmazonEC2.("privateIPAddress"+$nic)

    if ($privateIPAddress.Length -gt 0) {
        $newnic = New-EC2NetworkInterface -SubnetId $settings.AmazonEC2.("subnetId"+$nic) -Group $settings.AmazonEC2.("securityGroupId"+$nic) -PrivateIPAddress $privateIPAddress
    } else {
        $newnic = New-EC2NetworkInterface -SubnetId $settings.AmazonEC2.("subnetId"+$nic) -Group $settings.AmazonEC2.("securityGroupId"+$nic)
    }

    If([string]::IsNullOrEmpty($newnic)) {
        $msg = $error[0]
        WriteErrorString "Error: Failed to create NIC$nic - $msg"
        Exit
    }

    New-EC2Tag -Region $awsRegion -Tag @( @{ Key = "Name" ; Value = "$uagName-eth$nic"}) -Resource $newnic.NetworkInterfaceId -Force

    $e = Edit-EC2NetworkInterfaceAttribute -NetworkInterfaceId $newnic.NetworkInterfaceId -Description "$uagName-eth$nic ($description)"

    if ($settings.AmazonEC2.("publicIPId"+$nic).length -gt 0) {
        $r = Register-EC2Address -AllocationId $settings.AmazonEC2.("publicIPId"+$nic) -NetworkInterfaceId $newnic.NetworkInterfaceId
        If([string]::IsNullOrEmpty($r)) {
            $msg = $error[0]
            WriteErrorString "Error: Failed to register address - $msg"
            Exit
        }
    }

    $eth = new-object Amazon.EC2.Model.InstanceNetworkInterfaceSpecification
    $eth.NetworkInterfaceId = $newnic.NetworkInterfaceId
    $eth.DeviceIndex = $nic
    $eth.DeleteOnTermination = $false

    $eth
}

function terminateExistingInstance {
    Param ($settings)

    $statePending="0"
    $stateRunning="16"
    $stateTerminated="48"
    $stateStopped="80"

    $uagName=$settings.General.name

    $i=(Get-EC2Instance -Filter @(@{name="tag:Name";value=$uagName};@{name="instance-state-code";values=$statePending,$stateRunning,$stateStopped})).Instances.InstanceId

    if ($i.Count -eq 1) {
        write-host "Terminating existing instance of $uagName"

        $r=Remove-EC2Instance -InstanceId $i -Force

        while ($true) {
            $j=Get-EC2Instance -InstanceId $i
            if ($j.Instances[0].State.Code -eq $stateTerminated) {
                write-host "Existing instance of $uagName terminated successfully"
                break
            }
            sleep 2
        }
    }
}

function ValidateNetworkSettings {
    Param ($settings, $nic)

    $subnetID = $settings.AmazonEC2.("subnetId"+$nic)

    if ($subnetID.length -gt 0) {

        $subnet = Get-EC2Subnet -SubnetId $subnetID
        If([string]::IsNullOrEmpty($subnet)) {
            $msg = $error[0]
            WriteErrorString "Error: [AmazonEC2] subnetID$nic ($subnetID) not found"
            Exit
        }
    } else {
        WriteErrorString "Error: [AmazonEC2] subnetID$nic not specified"
        Exit
    }

    $publicIPId = $settings.AmazonEC2.("publicIPId"+$nic)

    if ($publicIPId.length -gt 0) {

        $ipObj = Get-EC2Address -AllocationId $publicIPId

        If([string]::IsNullOrEmpty($ipObj)) {
            WriteErrorString "Error: [AmazonEC2] publicIPId$nic ($publicIPId) not found"
            Exit
        }

        $publicIP = $ipObj.PublicIp

        if ($ipObj.InstanceId.length -gt 0) {
            WriteErrorString "Error: [AmazonEC2] publicIPId$nic ($publicIPId - $publicIP) is already in use by another instance"
            Exit
        }
    }

    $securityGroupId = $settings.AmazonEC2.("securityGroupId"+$nic)

    if ($securityGroupId.length -gt 0) {
        $sg = Get-EC2SecurityGroup -GroupId $securityGroupId
        If([string]::IsNullOrEmpty($sg)) {
            WriteErrorString "Error: [AmazonEC2] securityGroupId$nic ($securityGroupId) not found"
            Exit
        }

        if ($sg.VpcId -ne $subnet.VpcId) {
            WriteErrorString "Error: [AmazonEC2] securityGroupId$nic ($securityGroupId) is not in the same VPC as the specified subnet"
            Exit
        }
    }
}

#
# Load the dependent PowerShell Module
#

$ScriptPath = $MyInvocation.MyCommand.Path
$ScriptDir  = Split-Path -Parent $ScriptPath
$uagDeployModule=$ScriptDir+"\uagdeploy.psm1"

if (!(Test-path $uagDeployModule)) {
    Write-host "Error: PowerShell Module $uagDeployModule not found." -foregroundcolor red -backgroundcolor black
    Exit
}

import-module $uagDeployModule -Force -ArgumentList $awAPIServerPwd, $awTunnelGatewayAPIServerPwd, $awTunnelProxyAPIServerPwd, $awCGAPIServerPwd, $awSEGAPIServerPwd

if (-not (Get-Module -ListAvailable -Name "AWSPowerShell")) {
    WriteErrorString "Error: Powershell module AWSPowerShell not found. Run the command 'Install-Module -Name AWSPowerShell' and retry"
    Exit
}

Write-host "Unified Access Gateway (UAG) virtual appliance Amazon AWS EC2 deployment script"

if (!(Test-path $iniFile)) {
    WriteErrorString "Error: Configuration file ($iniFile) not found."
    Exit
}

$settings = ImportIni $iniFile

$uagName=$settings.General.name

#
# Assign and validate network settings
#

$deploymentOption=GetDeploymentSettingOption $settings

if ($uagName.length -gt 32) {
    WriteErrorString "Error: Virtual machine name must be no more than 32 characters in length"
    Exit
}

if (!$uagName) {
    WriteErrorString "Error: [General] Name not specified"
    Exit
}

if (!$rootPwd) {
    $rootPwd = GetRootPwd $uagName
}

if (!$adminPwd) {
    $adminPwd = GetAdminPwd $uagName
}

if (!$ceipEnabled) {
    $ceipEnabled = GetCeipEnabled $uagName
}

$settingsJSON=GetJSONSettings $settings

SetUp

$ovfFile = "${env:APPDATA}\VMware\$uagName.cfg"

[IO.File]::WriteAllLines($ovfFile, [string[]]("deploymentOption="+"$deploymentOption"))

$dns=$settings.General.dns
if ($dns.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("DNS="+"$dns"))
}

$defaultGateway=$settings.General.defaultGateway
if ($defaultGateway.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("defaultGateway="+"$defaultGateway"))
}

$v6DefaultGateway=$settings.General.v6DefaultGateway
if ($v6defaultGateway.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("v6defaultGateway="+"$v6defaultGateway"))
}

$forwardrules=$settings.General.forwardrules
if ($forwardrules.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("forwardrules="+"$forwardrules"))
}

$routes0=$settings.General.routes0
if ($routes0.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("routes0="+"$routes0"))
}

$routes1=$settings.General.routes1
if ($routes1.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("routes1="+"$routes1"))
}

$routes2=$settings.General.routes2
if ($routes2.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("routes2="+"$routes2"))
}

$policyRouteGateway0=$settings.General.policyRouteGateway0
if ($policyRouteGateway0.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("policyRouteGateway0="+"$policyRouteGateway0"))
}

$policyRouteGateway1=$settings.General.policyRouteGateway1
if ($policyRouteGateway1.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("policyRouteGateway1="+"$policyRouteGateway1"))
}

$policyRouteGateway2=$settings.General.policyRouteGateway2
if ($policyRouteGateway2.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("policyRouteGateway2="+"$policyRouteGateway2"))
}

if ($ceipEnabled -eq $false) {
    $ovfOptions += " --prop:ceipEnabled='false'"
    [IO.File]::AppendAllLines($ovfFile, [string[]]("ceipEnabled=false"))
}

if ($settings.General.tlsPortSharingEnabled -eq "true") {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("tlsPortSharingEnabled=true"))
}

if ($settings.General.sshEnabled -eq "true") {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("sshEnabled=true"))
}

if ($settings.General.sshPasswordAccessEnabled -eq "false") {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("sshPasswordAccessEnabled=false"))
}

if ($settings.General.sshKeyAccessEnabled -eq "true") {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("sshKeyAccessEnabled=true"))
}

[IO.File]::AppendAllLines($ovfFile, [string[]]("rootPassword="+"$rootPwd"))

if ($adminPwd.length -gt 0) {
    [IO.File]::AppendAllLines($ovfFile, [string[]]("adminPassword="+"$adminPwd"))
}

$awsamiId = $settings.AmazonEC2.amiId

if ($awsamiId.length -eq 0) {
   WriteErrorString "Error: [AmazonEC2] amiId not found"
   Exit
}

$ErrorActionPreference = "SilentlyContinue"

$credentialProfileName = $settings.AmazonEC2.credentialProfileName

if ($credentialProfileName.length -gt 0) {
    $cred = Get-AWSCredential -ProfileName $credentialProfileName
    If([string]::IsNullOrEmpty($cred)) {
        WriteErrorString "Error: [AmazonEC2] credentialProfileName ($credentialProfileName) not found. To set a named credential profile, run the command:"
        WriteErrorString "Set-AWSCredential -AccessKey <accesskey> -SecretKey <secretkey> -StoreAs $credentialProfileName"
        Exit
    }
    $subNets = get-ec2subnet -ProfileName $credentialProfileName
    If([string]::IsNullOrEmpty($subNets)) {
        WriteErrorString "Error: [AmazonEC2] credentialProfileName ($credentialProfileName) is invalid. To set a named credential profile, run the command:"
        WriteErrorString "Set-AWSCredential -AccessKey <accesskey> -SecretKey <secretkey> -StoreAs $credentialProfileName"
        Exit
    }
} else {
    $credentialProfileName = "default"
    $subNets = get-ec2subnet -ProfileName $credentialProfileName
    If([string]::IsNullOrEmpty($subNets)) {
        WriteErrorString "Error: Default credential profile is not set or is invalid. To set a default credential profile, run the command:"
        WriteErrorString "Set-AWSCredential -AccessKey <accesskey> -SecretKey <secretkey> -StoreAs default"
        Exit
    }
}

$awsInstanceType = $settings.AmazonEC2.instanceType
if ($awsInstanceType.length -gt 0) {
    write-host "Deploying $uagName as $awsInstanceType instance type"
} else {
    $awsInstanceType = "c4.large"
}

$awsRegion = $settings.AmazonEC2.region
if ($awsRegion.length -gt 0) {
    $region = Get-EC2Region -Region $awsRegion
    If([string]::IsNullOrEmpty($region)) {
        WriteErrorString "Error: [AmazonEC2] region ($awsRegion) not found"
        $regions = Get-EC2Region
        $regionNames = $regions.RegionName
        WriteErrorString "Specify a region from the following list - $regionNames"
        Exit
    }
    Set-DefaultAWSRegion $awsRegion
} else {
    WriteErrorString "Error: [AmazonEC2] region not specified"
    Exit
}

Initialize-AWSDefaultConfiguration -ProfileName $credentialProfileName -Region $awsRegion

terminateExistingInstance $settings

$ami = Get-EC2Image -ImageId $awsamiId -ErrorAction SilentlyContinue

If([string]::IsNullOrEmpty($ami)) {
    WriteErrorString "Error: AMI image $awsamiId not found in region $awsRegion"
    Exit
}

switch -Wildcard ($deploymentOption) {

    'onenic*' {
        ValidateNetworkSettings $settings "0"
        $eth0 = CreateNIC $settings "0" "Internet, Management and Backend"
        [IO.File]::AppendAllLines($ovfFile, [string[]]("ipMode0=DHCPV4+DHCPV6"))
        $NetworkInterfaces = $eth0
    }
    'twonic*' {
        ValidateNetworkSettings $settings "0"
        ValidateNetworkSettings $settings "1"
        $eth0 = CreateNIC $settings "0" "Internet"
        [IO.File]::AppendAllLines($ovfFile, [string[]]("ipMode0=DHCPV4+DHCPV6"))
        $eth1 = CreateNIC $settings "1" "Management and Backend"
        [IO.File]::AppendAllLines($ovfFile, [string[]]("ipMode1=DHCPV4+DHCPV6"))
        $NetworkInterfaces = $eth0,$eth1
    }
    'threenic*' {
        ValidateNetworkSettings $settings "0"
        ValidateNetworkSettings $settings "1"
        ValidateNetworkSettings $settings "2"
        $eth0 = CreateNIC $settings "0" "Internet"
        [IO.File]::AppendAllLines($ovfFile, [string[]]("ipMode0=DHCPV4+DHCPV6"))
        $eth1 = CreateNIC $settings "1" "Management"
        [IO.File]::AppendAllLines($ovfFile, [string[]]("ipMode1=DHCPV4+DHCPV6"))
        $eth2 = CreateNIC $settings "2" "Backend"
        [IO.File]::AppendAllLines($ovfFile, [string[]]("ipMode2=DHCPV4+DHCPV6"))
        $NetworkInterfaces = $eth0,$eth1,$eth2
    }
    default {
        WriteErrorString "Error: Invalid deploymentOption ($deploymentOption)."
        Exit
    }
}

[IO.File]::AppendAllLines($ovfFile, [string[]]("settingsJSON="+"$settingsJSON"))

$ovfProperties = Get-Content -Raw $ovfFile

$UserData = [System.Convert]::ToBase64String([System.Text.Encoding]::ASCII.GetBytes($ovfProperties))

$InstanceParams = @{
    ImageId = $awsamiId
    NetworkInterface = $NetworkInterfaces
    MinCount = "1"
    MaxCount = "1"
    InstanceType = $awsInstanceType
    UserData = $UserData
}

$i = New-EC2Instance @InstanceParams

If([string]::IsNullOrEmpty($i)) {
    $msg = $error[0]
    WriteErrorString "Error: Failed to deploy UAG - $msg"
    [IO.File]::Delete($ovfFile)
    Exit
}

$instance = $i.Instances[0]
$instanceId = $instance.InstanceId

New-EC2Tag -Region $awsRegion -Tag @( @{ Key = "Name" ; Value = "$uagName"})-Resource $instanceId -Force

SetNICDeleteOnTermination $settings $eth0
SetNICDeleteOnTermination $settings $eth1
SetNICDeleteOnTermination $settings $eth2

write-host "$uagName Deployed successfully to AWS EC2 region $awsRegion - InstanceId=$instanceId"

[IO.File]::Delete($ovfFile)
