/* **********************************************************
 * Copyright 2012-2014, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi;

import java.util.Arrays;

import com.vmware.vapi.internal.util.Validate;

/**
 * This class represents a localizable message.
 *
 * @see MessageFactory
 */
public class Message {
    private String id;
    private String defMsg;
    private String[] args;

    /**
     * Construct a Message with a unique identifier, a ready-to-use default
     * message, in default/server locale (say English) and the args.
     *
     * <p>The message id is usually used to lookup a message template string in
     * some message catalog. However this instance doesn't look it up, neither
     * it guarantees that the id is available in a catalog. More generally this
     * class doesn't attempt to load message catalogs at all.
     *
     * @param id the identifier for the message; must not be <code>null</code>
     * @param defMsg default message with already placed in <code>args</code> in
     *               the default/server locale; must not be <code>null</code>
     * @param args values for the message template placeholders
     */
    public Message(String id, String defMsg, String... args) {
        Validate.notNull(id);
        Validate.notNull(defMsg);

        this.id = id;
        this.defMsg = defMsg;
        // varargs (...) subtleness - explicit "(String[]_) null" can be passed in
        // which sets the member field to null instead of empty array
        this.args = (args == null) ? new String[0]
                                   : args;
    }

    /**
     * @return the message identifier; never <code>null</code>
     */
    public String getId() {
        return id;
    }

    /**
     * @return the message detault representation; never <code>null</code>
     */
    public String getDefaultMessage() {
        return defMsg;
    }

    /**
     * @return the message argument values (for replacement into the message
     *         template's placeholders); maybe be empty array, but never <code>null</code>
     */
    public String[] getArgs() {
        return args;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (!(obj instanceof Message)) {
            return false;
        }
        Message other = (Message) obj;

        return id.equals(other.id) &&
               defMsg.equals(other.defMsg) &&
               Arrays.equals(args, other.args);
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + id.hashCode();
        result = prime * result + Arrays.hashCode(args);
        result = prime * result + defMsg.hashCode();

        return result;
    }

    @Override
    public String toString() {
        return defMsg;
    }
}
