/* **********************************************************
 * Copyright (c) 2012-2013, 2018-2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.bindings;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.vmware.vapi.bindings.client.RetryPolicy;
import com.vmware.vapi.bindings.type.ErrorType;
import com.vmware.vapi.core.ExecutionContext.SecurityContext;
import com.vmware.vapi.internal.util.Validate;

/**
 * This is a helper class for creating stub configurations.
 */
public final class StubConfiguration extends StubConfigurationBase {

    private SecurityContext securityContext;
    private Set<ErrorType> registeredErrorTypes = new HashSet<>();
    private RetryPolicy retryPolicy = null;

    /**
     * Default Constructor.
     */
    public StubConfiguration() {
    }

    /**
     * Constructor.
     *
     * @param securityContext can be null
     */
    public StubConfiguration(SecurityContext securityContext) {
        this.securityContext = securityContext;
    }

    /**
     * @param context the security context that will be used in the stub.
     *                can be null.
     */
    public void setSecurityContext(SecurityContext context) {
        this.securityContext = context;
    }

    @Override
    public SecurityContext getSecurityContext() {
        return securityContext;
    }

    /**
     * Registers a list of <code>ErrorType</code>s to be used by the
     * configured stub for resolving <tt>VMODL2</tt> errors received as
     * result of a method invocation.
     *
     * <p>The types for all <b>standard VMODL2 errors</b> are registered
     * and resolvable by default. This method is useful for registration
     * of additional types for custom errors.
     *
     * @param errors types to register; can not be <code>null</code>;
     *        can not contain <code>null</code> elements
     */
    public void registerErrorTypes(List<ErrorType> errors) {
        Validate.noNullElements(errors);
        registeredErrorTypes.addAll(errors);
    }

    @Override
    public Set<ErrorType> getErrorTypes() {
        return registeredErrorTypes;
    }

    /**
     * @param retryPolicy retry policy for failed invocations; can be
     *                    <code>null</code>
     */
    public void setRetryPolicy(RetryPolicy retryPolicy) {
        this.retryPolicy = retryPolicy;
    }

    @Override
    public RetryPolicy getRetryPolicy() {
        return retryPolicy;
    }
}
