/* **********************************************************
 * Copyright 2012 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.bindings.client;

import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import com.vmware.vapi.std.Progress;

/**
 * Utility which is both an asynchronous callback and a future. Could be used to
 * work in a synchronous fashion with an asynchronous operation.
 */
public class AsyncCallbackFuture<T> extends AsyncCallback<T> implements Future<T> {

    private final Lock lock = new ReentrantLock();
    private final Condition set = lock.newCondition();
    private boolean completed = false;

    private T result;
    private RuntimeException error;

    /**
     * this implementation doesn't support canceling, so it will always return false.
     */
    @Override
    public boolean cancel(boolean mayInterruptIfRunning) {
        return false;
    }

    @Override
    public boolean isCancelled() {
        return false;
    }

    @Override
    public boolean isDone() {
        lock.lock();
        try {
            return completed;
        } finally {
            lock.unlock();
        }
    }

    @Override
    public T get() throws InterruptedException, ExecutionException {
        lock.lock();
        try {
            while (!completed) {
                set.await();
             }
             if (error == null) {
                return result;
             } else {
                 throw new ExecutionException(error);
             }
        } finally {
            lock.unlock();
        }
    }

    @Override
    public T get(long timeout, TimeUnit unit) throws InterruptedException,
            ExecutionException, TimeoutException {
        lock.lock();
        try {
            while (!completed) {
                boolean signalled = set.await(timeout, unit);
                if (!signalled) {
                    throw new TimeoutException();
                }
             }
             if (error == null) {
                return result;
             } else {
                 throw new ExecutionException(error);
             }
        } finally {
            lock.unlock();
        }
    }

    @Override
    public void onProgress(Progress progress) {
    }

    @Override
    public void onResult(T result) {
        lock.lock();
        try {
            this.result = result;
            this.completed = true;
            set.signalAll();
        } finally {
            lock.unlock();
        }
    }

    @Override
    public void onError(RuntimeException error) {
        lock.lock();
        try {
            this.error = error;
            this.completed = true;
            set.signalAll();
        } finally {
            lock.unlock();
        }
    }

}
